#ifndef PINYIN_ENGINE_H
#define PINYIN_ENGINE_H

#include "sunpinyin.h"


class PinyinWord
{
public:
    PinyinWord() {}
    PinyinWord(const std::string& word)
    {
        m_Word = word;
    }
    PinyinWord(const std::string& word, const std::string& pinyin)
    {
        m_Word = word;
        m_Pinyin = pinyin;
    }
    virtual ~PinyinWord(){}

    const std::string& Word() const
    {
        return m_Word;
    }
    const std::string& Pinyin() const
    {
        return m_Pinyin;
    }
    bool operator == (const PinyinWord& b) const
    {
        return (m_Word == b.m_Word && m_Pinyin == b.m_Pinyin);
    }

private:
    std::string m_Word;
    std::string m_Pinyin;
};


class CIMIWinHandler;

class PinyinResult : public CIMIWinHandler
{
    friend class PinyinEngine;

public:
    PinyinResult();
    virtual ~PinyinResult();

    const std::string& Commit() const;
    const std::string& Preedit() const;
    const PinyinWord& WordList(unsigned index) const;
    unsigned Count() const;

private:
    void SetCaret(const char caret);
    unsigned GetandClrUpdateFlag();
    void SetUpdateFlag(unsigned flag);
    unsigned GetCandidateIndex(const std::string& word);
    void Reset();

private:
    virtual void commit(const TWCHAR* wstr);
    virtual void updatePreedit(const IPreeditString* ppd);
    virtual void updateCandidates(const ICandidateList* pcl);

private:
    std::string                 m_CommitString;
    std::string                 m_PreeditString;
    std::vector<PinyinWord>     m_Candidates;
    unsigned                    m_UpdateFlag;
    unsigned                    m_CommitFlag;
    unsigned                    m_PreeditFlag;
    unsigned                    m_CandidateFlag;
    unsigned                    m_CandiPageSize;
    unsigned                    m_CandiPageSizeOld;
    iconv_t                     m_iconv;
};


class CIMIView;
class CHotkeyProfile;

class PinyinEngine
{
public:
    PinyinEngine();
    virtual ~PinyinEngine();

    unsigned Init();
    void Destroy();
    PinyinResult* GetResultPtr();
    static void SetDataDir(const std::string& dataDir);
    static void SetUserDataDir(const std::string& userDataDir);
    static void SetAutoCorrect(bool autoCorrect);
    static void SetFuzzy(bool fuzzy);
    static void SetSegmentation(bool segmentation);
    static void SetPageSize(unsigned size);
    void SetPageKey(unsigned pageUp, unsigned pageDown);

    unsigned PageUp();
    unsigned PageDown();
    unsigned KeyPress(unsigned keyCode);
    unsigned KeyPress(const std::string& key);
    unsigned WordSelect(unsigned index);
    unsigned WordSelect(const std::string& word);
    void Reset();

    // not support yet
    void Prediction(const PinyinWord& word);
    void LearnWord(const PinyinWord& word);
    void DeleteWord(const PinyinWord& word);

private:
    CIMIView            *m_pView;
    PinyinResult        *m_pPinyinResult;
    CHotkeyProfile      *m_pHotkeyProfile;
};

#endif

