/*
 * utils_test_queue.c
 * Common test data queue handling code for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include "main.h"

#include "bt_appl/common_def.h"
#include "utils_async_queue.h"
#include "utils_test_queue.h"
#include "node_test.h"
#include "dbus_error.h"
#include "debug.h"

#ifdef USE_TEST_NODE

/*
 * exported function
 */
void utils_test_queue_initialise(struct test_queue_list *queue_info)
{
	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);

	/* initialise the list */
	queue_info->cfm_tql = NULL;

	DEBUG_FUNC("Exited");
}

static void utils_test_queue_free_test_data(enum node_cfm_number cfm_id,
						gpointer test_data_struct)
{
	DEBUG("freeing id %u CFM test data", cfm_id);

	switch (cfm_id) {
	case ID_AVP_Get_Metadata_Attr_CFM:
	{
		struct avp_get_metadata_attr_cfm_parms_s *cfm_test_data;

		cfm_test_data = test_data_struct;

		/* free the GByte Array */
		g_byte_array_unref(cfm_test_data->data);

		break;
	}
	default:
		DEBUG_ERROR("Failed to free CFM test data for %u", cfm_id);
		break;
	}
}

/*
 * destroy the test queue by first cleaning up entries
 *
 * exported function
 */
void utils_test_queue_destroy(struct test_queue_list *queue_info)
{
	GSList *list_entry;
	struct cfm_test_queue_entry *queue_record = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);

	/* start with the first entry */
	list_entry = queue_info->cfm_tql;

	while (list_entry != NULL)
	{
		/* spin through the list */
		queue_record = list_entry->data;

		/* remove the entry from the list */
		queue_info->cfm_tql = g_slist_delete_link(
							queue_info->cfm_tql,
							list_entry);

		/* free any test data */
		utils_test_queue_free_test_data(queue_record->cfm_id,
					queue_record->test_data_struct);

		/* free the record as no longer needed */
		g_free(queue_record);

		/* move to the new first entry */
		list_entry = queue_info->cfm_tql;
	}
	DEBUG_FUNC("Exited");
}

/*
 * add a CFM record to the test queue list
 *
 * exported function
 */
void utils_test_queue_add_list_entry(struct test_queue_list *queue_info,
						enum node_cfm_number cfm_id,
						const BD_ADDRESS bd_addr,
						gpointer test_data_struct)
{
	struct cfm_test_queue_entry *queue_record = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);
	g_assert(test_data_struct != NULL);
	g_assert(cfm_id < ID_total);

	/* create a record */
	queue_record = (struct cfm_test_queue_entry *)
			g_malloc0(sizeof(struct cfm_test_queue_entry));

	g_assert(queue_record != NULL);

	/* update the record */
	queue_record->queue_info = queue_info;
	queue_record->cfm_id = cfm_id;
	memcpy(queue_record->bd_addr, &bd_addr[0], sizeof(BD_ADDRESS));
	queue_record->test_data_struct = test_data_struct;

	/* append the record to the cfm test queue list */
	queue_info->cfm_tql = g_slist_append(queue_info->cfm_tql,
								queue_record);

	DEBUG_FUNC("Exited");
}

/*
 * finds the CFM ID and DB_ADDRESS from the cfm test queue and returns
 * a pointer to the cfm test data
 *
 * removes the entry from the list
 * returns NULL if not entry has been found
 *
 * the calling function needs to free the gpointer
 *
 * exported function
 */
gpointer utils_test_queue_find_list_entry(
					struct test_queue_list *queue_info,
					enum node_cfm_number cfm_id,
					const BD_ADDRESS bd_addr)
{
	GSList *list_entry;
	struct cfm_test_queue_entry *queue_record = NULL;
	gpointer test_data_struct = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);
	g_assert(cfm_id < ID_total);

	/* start with the first entry */
	list_entry = queue_info->cfm_tql;

	DEBUG("Number of entries = %d", g_slist_length(list_entry));

	while (list_entry != NULL)
	{
		/* spin through the list */
		queue_record = list_entry->data;

		/* compare id and address */
		if ((queue_record->cfm_id == cfm_id) &&
			(queue_record->bd_addr[0] == bd_addr[0]) &&
			(queue_record->bd_addr[1] == bd_addr[1]) &&
			(queue_record->bd_addr[2] == bd_addr[2]) &&
			(queue_record->bd_addr[3] == bd_addr[3]) &&
			(queue_record->bd_addr[4] == bd_addr[4]) &&
			(queue_record->bd_addr[5] == bd_addr[5])) {

			DEBUG("Found matching cfm_id %u and device address",
								cfm_id);

			test_data_struct = queue_record->test_data_struct;

			/* free the record as no longer needed */
			g_free(queue_record);

			/* remove the entry from the list */
			queue_info->cfm_tql = g_slist_delete_link(
							queue_info->cfm_tql,
							list_entry);

			/* quit the while loop */
			list_entry = NULL;
		} else {
			/* move to the next entry */
			list_entry = g_slist_next(list_entry);
		}
	}

	DEBUG_FUNC("Exited");

	return test_data_struct;
}

#endif
