/*
 * utils_async_queue.c
 * Common async queue handling code for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "main.h"

#include "bt_appl/common_def.h"
#include "utils_async_queue.h"
#include "dbus_error.h"
#include "debug.h"

/* CFM information */
struct cfm_info {
	gchar *name;
	guint timeout;
};

/* local global to hold CFM information data for CFM timeout handling */
static struct cfm_info cfm_data[ID_total] =
{
	[ID_Get_Device_Name_CFM] = {
		.name = "BT_APPL_Get_Device_Name_CFM",
		.timeout = 50 },
	[ID_HF_Dial_CFM] = {
		.name = "BT_APPL_HF_Dial_CFM",
		.timeout = 40 },
	[ID_HF_Redial_CFM] = {
		.name = "BT_APPL_HF_Redial_CFM",
		.timeout = 40 },
	[ID_HF_Mem_Dial_CFM] = {
		.name = "BT_APPL_HF_Mem_Dial_CFM",
		.timeout = 40 },
	[ID_HF_Terminate_Call_CFM] = {
		.name = "BT_APPL_HF_Terminate_Call_CFM",
		.timeout = 40 },
	[ID_HF_Terminate_All_CFM] = {
		.name = "BT_APPL_HF_Terminate_All_CFM",
		.timeout = 40 },
	[ID_HF_Audio_Transfer_CFM] = {
		.name = "BT_APPL_HF_Audio_Transfer_CFM",
		.timeout = 40 },
	[ID_HF_Send_DTMF_CFM] = {
		.name = "BT_APPL_HF_Send_DTMF_CFM",
		.timeout = 40 },
	[ID_HF_Incoming_Call_Act_CFM] = {
		.name = "BT_APPL_HF_Incoming_Call_Act_CFM",
		.timeout = 40 },
	[ID_HF_Release_Inactive_Call_CFM] = {
		.name = "BT_APPL_HF_Release_Inactive_Call_CFM",
		.timeout = 40 },
	[ID_HF_Release_Active_Call_CFM] = {
		.name = "BT_APPL_HF_Release_Active_Call_CFM",
		.timeout = 40 },
	[ID_HF_Swap_Call_CFM] = {
		.name = "BT_APPL_HF_Swap_Call_CFM",
		.timeout = 40 },
	[ID_HF_Add_Call_CFM] = {
		.name = "BT_APPL_HF_Add_Call_CFM",
		.timeout = 40 },
	[ID_HF_Release_Spec_Call_CFM] = {
		.name = "BT_APPL_HF_Release_Spec_Call_CFM",
		.timeout = 40 },
	[ID_HF_Hold_Excep_Call_CFM] = {
		.name = "BT_APPL_HF_Hold_Excep_Call_CFM",
		.timeout = 40 },
	[ID_HF_Exp_Call_Trans_CFM] = {
		.name = "BT_APPL_HF_Exp_Call_Trans_CFM",
		.timeout = 40 },
	[ID_HF_Subscriber_CFM] = {
		.name = "BT_APPL_HF_Subscriber_CFM",
		.timeout = 40 },
	[ID_HF_Manufacturer_ID_CFM] = {
		.name = "BT_APPL_HF_Manufacturer_ID_CFM",
		.timeout = 40 },
	[ID_HF_Model_ID_CFM] = {
		.name = "BT_APPL_HF_Model_ID_CFM",
		.timeout = 40 },
	[ID_HF_Revision_ID_CFM] = {
		.name = "BT_APPL_HF_Revision_ID_CFM",
		.timeout = 40 },
	[ID_HF_Voice_Recognition_Activation_CFM] = {
		.name = "BT_APPL_Voice_Recognition_Activation_CFM",
		.timeout = 40 },
	[ID_APPL_SIRI_XAPL_CFM] = {
		.name = "BT_APPL_SIRI_XAPL_CFM",
		.timeout = 40 },
	[ID_APPL_SIRI_NrStat_CFM] = {
		.name = "BT_APPL_SIRI_NrStat_CFM",
		.timeout = 40 },
	[ID_APPL_SIRI_EFM_CFM] = {
		.name = "BT_APPL_SIRI_EFM_CFM",
		.timeout = 40 },
	[ID_HF_Release_Waiting_Call] = {
		.name = "HF_RELEASE_WAITING_CALL",
		.timeout = 40 },
	[ID_AVP_Ctrl_Cmd_CFM] = {
		.name = "BT_APPL_AVP_Ctrl_Cmd_CFM",
		.timeout = 40 },
	[ID_AVP_Get_Capabilities_CFM] = {
		.name = "BT_APPL_AVP_Get_Capabilities_CFM",
		.timeout = 180 },
	[ID_AVP_List_Attr_CFM] = {
		.name = "BT_APPL_AVP_List_Attr_CFM",
		.timeout = 40 },
	[ID_AVP_List_Val_CFM] = {
		.name = "BT_APPL_AVP_List_Val_CFM",
		.timeout = 45 },
	[ID_AVP_Set_Val_CFM] = {
		.name = "BT_APPL_AVP_Set_Val_CFM",
		.timeout = 40 },
	[ID_AVP_Set_Player_CFM] = {
		.name = "BT_APPL_AVP_Set_Player_CFM",
		.timeout = 40 },
	[ID_AVP_Get_Metadata_Attr_CFM] = {
		.name = "BT_APPL_AVP_Get_Metadata_Attr_CFM",
		.timeout = 40 },
	[ID_AVP_Get_Folder_Item_CFM] = {
		.name = "BT_APPL_AVP_Get_Folder_Item_CFM",
		.timeout = 40 },
	[ID_AVP_Get_Media_Player_Item_CFM] = {
		.name = "BT_APPL_AVP_Get_Media_Player_Item_CFM",
		.timeout = 40 },
	[ID_AVP_Change_Path_CFM] = {
		.name = "BT_APPL_AVP_Change_Path_CFM",
		.timeout = 40 },
	[ID_AVP_Play_Item_CFM] = {
		.name = "BT_APPL_AVP_Play_Item_CFM",
		.timeout = 40 },
	[ID_PBDL_Get_Capabilities_CFM] = {
		.name = "BT_APPL_PBDL_Get_Capabilities_CFM",
		.timeout = 195 },
	[ID_PBDL_Prepare_CFM] = {
		.name = "BT_APPL_PBDL_Prepare_CFM",
		.timeout = 195 },
	[ID_PBDL_Start_CFM] = {
		.name = "BT_APPL_PBDL_Start_CFM",
		.timeout = 195 },
	[ID_PBDL_Stop_CFM] = {
		.name = "BT_APPL_PBDL_Stop_CFM",
		.timeout = 40 },
	[ID_PBDL_Complete_CFM] = {
		.name = "BT_APPL_PBDL_Complete_CFM",
		.timeout = 195 },
	[ID_MAP_Get_Capabilities_CFM] = {
		.name = "BT_APPL_MAP_Get_Capabilities_CFM",
		.timeout = 195 },
	[ID_MAP_Start_CFM] = {
		.name = "BT_APPL_MAP_Start_CFM",
		.timeout = 195 },
	[ID_MAP_Stop_CFM] = {
		.name = "BT_APPL_MAP_Stop_CFM",
		.timeout = 195 },
	[ID_MAP_Chg_Instance_CFM] = {
		.name = "BT_APPL_MAP_Chg_Instance_CFM",
		.timeout = 195 },
	[ID_MAP_Update_Inbox_CFM] = {
		.name = "BT_APPL_MAP_Update_Inbox_CFM",
		.timeout = 195 },
	[ID_MAP_List_Folder_CFM] = {
		.name = "BT_APPL_MAP_List_Folder_CFM",
		.timeout = 195 },
	[ID_MAP_List_Msg_CFM] = {
		.name = "BT_APPL_MAP_List_Msg_CFM",
		.timeout = 195 },
	[ID_MAP_Read_Msg_CFM] = {
		.name = "BT_APPL_MAP_Read_Msg_CFM",
		.timeout = 195 },
	[ID_MAP_Upload_CFM] = {
		.name = "BT_APPL_MAP_Upload_CFM",
		.timeout = 195 },
	[ID_MAP_Set_Msg_Status_CFM] = {
		.name = "BT_APPL_MAP_Set_Msg_Status_CFM",
		.timeout = 80 },
	[ID_MAP_Abort_CFM] = {
		.name = "BT_APPL_MAP_Abort_CFM",
		.timeout = 80 },
	[ID_NAP_Create_Device_CFM] = {
		.name = "BT_APPL_Create_Device_CFM",
		.timeout = 195 },
	[ID_NAP_Destroy_Device_CFM] = {
		.name = "BT_APPL_Destroy_Device_CFM",
		.timeout = 195 },
	[ID_validation_test_mode_CFM] = {
		.name = "BT_APPL_TestMode_CFM",
		.timeout = 40 },
	[ID_validation_link_quality_CFM] = {
		.name = "BT_APPL_LinkQuality_CFM",
		.timeout = 40 },
};

/*
 * exported function
 */
void utils_async_queue_initialise(struct async_queue_list *queue_info)
{
	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);

	/* initialise the list */
	queue_info->node_aql = NULL;

	DEBUG_FUNC("Exited");
}

/*
 * destroy the async method queue by first cleaning up entries
 *
 * exported function
 */
void utils_async_queue_destroy(struct async_queue_list *queue_info)
{
	GSList *list_entry;
	struct node_queue_entry *queue_record = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);

	/* start with the first entry */
	list_entry = queue_info->node_aql;

	while (list_entry != NULL)
	{
		/* spin through the list */
		queue_record = list_entry->data;

		/* terminate the CFM timeout */
		g_source_destroy(queue_record->gsource_info);

		DEBUG("context %p", queue_record->context);

		g_assert(queue_record->context != NULL);

		/* remove the entry from the list */
		queue_info->node_aql = g_slist_delete_link(
							queue_info->node_aql,
							list_entry);

		/* free the DBusGMethodInvocation context as no longer needed */
		g_free(queue_record->context);

		/* free the record as no longer needed */
		g_free(queue_record);

		/* move to the new first entry */
		list_entry = queue_info->node_aql;
	}

	DEBUG_FUNC("Exited");
}

/*
 * async queue list timeout callback
 *
 * exported function
 */
gboolean utils_async_queue_list_timeout(gpointer data)
{
	GError *error = NULL;
	DBusGMethodInvocation *context = NULL;
	BTHRESULT result = DAEMON_TIMEOUT_EXPIRED;
	gboolean return_code = TRUE;
	struct node_queue_entry *queue_record =
					(struct node_queue_entry *) data;

	DEBUG_FUNC("Called");

	g_assert(queue_record != NULL);
	g_assert(queue_record->queue_info != NULL);
	g_assert(queue_record->cfm_id < ID_total);

	/* return a daemon error code */
	return_code = report_any_error_to_dbus(result,
					cfm_data[queue_record->cfm_id].name,
					&error);

	/*
	 * find and remove the context from the async queue list
	 * also cancels the timeout and deletes the queue_record
	 */
	context = utils_async_queue_find_list_entry(queue_record->queue_info,
							queue_record->cfm_id);
	queue_record = NULL;
	
	/* an error occurred so abort the method call */
	dbus_g_method_return_error(context, error);
	g_error_free(error);

	DEBUG_FUNC("Exited");

	return return_code;
}

/*
 * add a CFM record to the async method queue
 *
 * exported function
 */
void utils_async_queue_add_list_entry(struct async_queue_list *queue_info,
						DBusGMethodInvocation *context,
						enum node_cfm_number cfm_id)
{
	struct node_queue_entry *queue_record = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);
	g_assert(context != NULL);
	g_assert(cfm_id < ID_total);

	/* create a record */
	queue_record = (struct node_queue_entry *)
			g_malloc0(sizeof(struct node_queue_entry));

	g_assert(queue_record != NULL);

	/* update the record */
	queue_record->queue_info = queue_info;
	queue_record->cfm_id = cfm_id;
	queue_record->context = context;

	DEBUG_HIGH("context %p", queue_record->context);

	/* append the record to the async queue list */
	queue_info->node_aql = g_slist_append(queue_info->node_aql,
								queue_record);

	/* setup the timeout for the CFM */
	queue_record->gsource_info = g_timeout_source_new_seconds(
				cfm_data[queue_record->cfm_id].timeout);

	g_source_set_callback(queue_record->gsource_info,
						utils_async_queue_list_timeout,
						queue_record,
						NULL);

	g_source_attach(queue_record->gsource_info, NULL);

	DEBUG_FUNC("Exited");
}

/*
 * finds the CFM ID from the async queue and returns the D-bus context
 *
 * removes the entry from the list and cancels the timeout
 * returns NULL if not entry has been found
 *
 * exported function
 */
DBusGMethodInvocation *utils_async_queue_find_list_entry(
					struct async_queue_list *queue_info,
					enum node_cfm_number cfm_id)
{
	GSList *list_entry;
	struct node_queue_entry *queue_record = NULL;
	DBusGMethodInvocation *context = NULL;

	DEBUG_FUNC("Called");

	g_assert(queue_info != NULL);
	g_assert(cfm_id < ID_total);

	/* start with the first entry */
	list_entry = queue_info->node_aql;

	DEBUG("Number of entries = %d", g_slist_length(list_entry));

	while (list_entry != NULL)
	{
		/* spin through the list */
		queue_record = list_entry->data;

		if (queue_record->cfm_id == cfm_id) {
			DEBUG("found CFM %d", cfm_id);

			/* extract the context from the record */
			context = queue_record->context;

			DEBUG_HIGH("context %p", queue_record->context);

			DEBUG_HIGH("Cancelling the CFM timeout");
			g_source_destroy(queue_record->gsource_info);

			/* free the record as no longer needed */
			g_free(queue_record);

			/* remove the entry from the list */
			queue_info->node_aql = g_slist_delete_link(
							queue_info->node_aql,
							list_entry);

			/* quit the while loop */
			list_entry = NULL;
		} else {
			/* move to the next entry */
			list_entry = g_slist_next(list_entry);
		}
	}

	DEBUG_FUNC("Exited");

	return context;
}


