/*
 * Agent node handling for the D-bus Bluetooth Daemon.
 * The Bluetooth daemon is a client, not the server, of the agent node.
 *
 * Author: Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include "bt_appl/common_def.h"
#include "bt_appl/api/bt_appl_bt_setting_block.h"

#include "main.h"
#include "debug.h"

/* include auto-generated binding information */
#include "alps-bt-agent-introspection.h"

#include "callbacks.h"
#include "dbus_error.h"
#include "node_agent.h"
#include "handle_spp_service.h"

static DBusGProxy *agent_proxy;

/* Free a linkkey GArray. */
static void g_linkkey_free(GArray * g_linkkey)
{
	g_array_free(g_linkkey, TRUE);
}

/*
 * Convert a linkkey to a GArray. linkkey_len is number of guint8 elements.
 * Returns a pointer to the GArray.  Free the GArray with g_linkkey_free().
 */
static GArray *linkkey_to_g_array(const guint8 *linkkey, guint8 linkkey_len)
{
	GArray *g_linkkey;

	g_linkkey = g_array_sized_new(FALSE, FALSE, sizeof(guint8), linkkey_len);
	g_linkkey = g_array_append_vals(g_linkkey, linkkey, linkkey_len);

	return g_linkkey;
}

/* Free a bd_addr GArray. */
static void g_bd_addr_free(GArray * g_bd_addr)
{
	g_array_free(g_bd_addr, TRUE);
}

/*
 * Convert a BD_ADDRESS to a GArray.
 * Returns a pointer to the GArray.  Free the GArray with g_bd_addr_free().
 */
static GArray *bd_addr_to_g_array(const BD_ADDRESS bd_addr)
{
	GArray *g_bd_addr;

	g_bd_addr = g_array_sized_new(FALSE, FALSE, sizeof(guint8),
				      sizeof(BD_ADDRESS));
	g_bd_addr = g_array_append_vals(g_bd_addr, bd_addr, sizeof(BD_ADDRESS));

	return g_bd_addr;
}

#ifdef DBG_ENABLED
static char *callback_id_to_name(enum alps_callbacks id)
{
	char *method;

	switch (id) {
	case BT_APPL_REQUEST_PIN_IND_CB:
		method = "RequestPin";
		break;
	case BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB:
		method = "SspNumericConfirm";
		break;
	case BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB:
		method = "SspPasskeyNotification";
		break;
	case BT_APPL_PAIRING_COMPLETE_IND_CB:
		method = "PairingComplete";
		break;
	case BT_APPL_LINKKEY_REQUEST_IND_CB:
		method = "LinkkeyRequest";
		break;
	case BT_APPL_CONNECTION_REQUEST_IND_CB:
		method = "ConnectionRequest";
		break;
	case BT_APPL_AUTHENTICATION_IND_CB:
		method = "Authentication";
		break;
	default:
		method = "";
	}

	return method;
}
#endif

typedef struct agent_data_s {
	enum alps_callbacks id;
	BD_ADDRESS bd_addr;
	u8 service;
} agent_data_t;

static void free_agent_data(agent_data_t *agent_data)
{
	g_free(agent_data);
}

/*
 * Log an error returned by one of the ALPS BT_APPL_*_RES routines.
 */
static void report_bt_appl_error(enum alps_callbacks id, BTHRESULT res)
{
	if (res != BT_APPL_SUCCESS) {
		DEBUG_ERROR("%s bt_appl error %d: %s",
			callback_id_to_name(id),
		  	res,
			lookup_bthresult_str(res));
	}
}

/*
 * Log an error reply from an agent method.  This routine frees the GError.
 */
static void report_agent_reply_error(enum alps_callbacks id, GError *error)
{
	if (error != NULL) {
		DEBUG_ERROR("%s agent error %d: %s",
			callback_id_to_name(id),
			error->code, error->message);
		g_error_free(error);
	}
}

static void do_agent_pairing_complete_reply(DBusGProxy *proxy,
					    GError *error, gpointer userdata)
{
	agent_data_t *agent_data = userdata;

	DEBUG_FUNC("Called");
	(void) proxy;

	report_agent_reply_error(agent_data->id, error);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_authentication_reply(DBusGProxy *proxy,
					  char *user_id, GArray *g_password,
					  GError *error, gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	BTHRESULT res;
	(void) proxy;
	DEBUG_FUNC("Called");

	DEBUG("error:%p user_id:%p g_password:%p", error, user_id, g_password);

	if (error || !user_id || !g_password) {
		report_agent_reply_error(agent_data->id, error);
		res = BT_APPL_Authentication_RES(agent_data->bd_addr,
							0,
							(guint8 *) "",
							(guint8 *) "");
	} else {
		guint8 zero = 0;
		/* ensure the password is NULL terminated */
		if ((g_password->len == 0) ||
			(g_password->data[g_password->len - 1] != 0)) {
			g_array_append_val(g_password, zero);
		}
		DEBUG_DUMP_ARRAY(g_password->len, (guint8 *) g_password->data);
		res = BT_APPL_Authentication_RES(agent_data->bd_addr,
						strlen(user_id),
						(guint8 *) user_id,
						(guint8 *) g_password->data);
	}

	/* Note g_password probably not initialised on error so do not free */
	if ((error == NULL) && (g_password != NULL))
		g_array_free(g_password, TRUE);

	report_bt_appl_error(agent_data->id, res);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_connection_request_reply(DBusGProxy *proxy,
					      gboolean accept, GError *error,
					      gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	BTHRESULT res;
	(void) proxy;
	DEBUG_FUNC("Called");

	DEBUG("service %u, accept: %s", agent_data->service,
					accept == TRUE ? "yes":"no");

	if (error) {
		report_agent_reply_error(agent_data->id, error);
		res =
		    BT_APPL_Connection_Request_RES(agent_data->bd_addr,
						   agent_data->service, 0);
	} else
		res =
		    BT_APPL_Connection_Request_RES(agent_data->bd_addr,
						   agent_data->service, accept);

	report_bt_appl_error(agent_data->id, res);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_linkkey_request_reply(DBusGProxy *proxy,
					   guint8 linkkey_type,
					   GArray *g_linkkey,
					   GArray *g_dlinkkey,
					   GError *error,
					   gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	BTHRESULT res;
	(void) proxy;
	DEBUG_FUNC("Called");

	g_assert(agent_data != NULL);

	DEBUG("error %p, g_linkkey %p, g_dlinkkey %p",
						error, g_linkkey, g_dlinkkey);

	if (error || !g_linkkey || !g_linkkey->data ||
		!g_dlinkkey || !g_dlinkkey->data) {

		DEBUG_WARN("No linkkeys are available");

		report_agent_reply_error(agent_data->id, error);
		res = BT_APPL_Linkkey_Request_RES(agent_data->bd_addr,
						  0,
						  0,
						  NULL,
						  0,
						  NULL);
	} else {
#ifdef DBG_ENABLED
		if (btd_debug_level >= BTD_DBG_INFO_LEV) {
			char test_data[100] = {0};
			guint i;

			DEBUG("linkkeytype %d", linkkey_type);

			DEBUG("linkkey_len %d", g_linkkey->len);

			for(i = 0; i < g_linkkey->len; i++) {
	 			g_sprintf(&test_data[i*2], "%02X",
						g_linkkey->data[i]);
			}
			test_data[i*2] = 0;

			DEBUG("linkkey: %s", &test_data[0]);

			DEBUG("dlinkkey_len %d", g_dlinkkey->len);

			for(i = 0; i < g_dlinkkey->len; i++) {
	 			g_sprintf(&test_data[i*2], "%02X",
						g_dlinkkey->data[i]);
			}
			test_data[i*2] = 0;

			DEBUG("dlinkkey: %s", &test_data[0]);
		}
#endif		
		res = BT_APPL_Linkkey_Request_RES(agent_data->bd_addr,
						  linkkey_type,
						  g_linkkey->len,
						  (guint8 *) g_linkkey->data,
						  g_dlinkkey->len,
						  (guint8 *) g_dlinkkey->data);
	}

	/*
	 * Note g_linkkey and g_dlinkkey are not initialised on error.
	 * Freeing the non-NULL pointers caused a segfault
	 */
	if (error == NULL) {
		if (g_linkkey != NULL) {
			g_array_free(g_linkkey, TRUE);
		}
		if (g_dlinkkey != NULL) {
			g_array_free(g_dlinkkey, TRUE);
		}
	}

	report_bt_appl_error(agent_data->id, res);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_ssp_passkey_notification_reply(DBusGProxy *proxy,
						    GError *error,
						    gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	(void) proxy;
	DEBUG_FUNC("Called");

	report_agent_reply_error(agent_data->id, error);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_ssp_numeric_confirm_reply(DBusGProxy *proxy,
					       gboolean accept, GError *error,
					       gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	BTHRESULT res;
	(void) proxy;
	DEBUG_FUNC("Called");

	if (error) {
		report_agent_reply_error(agent_data->id, error);
		res = BT_APPL_SSP_Numeric_Confirm_RES(agent_data->bd_addr, 0);
	} else
		res = BT_APPL_SSP_Numeric_Confirm_RES(agent_data->bd_addr, accept);

	report_bt_appl_error(agent_data->id, res);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

static void do_agent_request_pin_reply(DBusGProxy *proxy, char *pin,
				       GError *error, gpointer userdata)
{
	agent_data_t *agent_data = userdata;
	BTHRESULT res;
	(void) proxy;
	DEBUG_FUNC("Called");

	if (error || !pin) {
		report_agent_reply_error(agent_data->id, error);
		res = BT_APPL_Request_PIN_RES(agent_data->bd_addr, 0, NULL);
	} else
		res = BT_APPL_Request_PIN_RES(agent_data->bd_addr, strlen(pin),
					      (u8 *) pin);

	report_bt_appl_error(agent_data->id, res);
	free_agent_data(agent_data);

	DEBUG_FUNC("Exited");
}

/* handle agent solution for SspNumericConfirmInd */
void do_agent_ssp_numeric_confirm_ind_wrapper(callback_data *data)
{
	DBusGProxyCall *proxy_call;
	gen_ssp_numeric_confirm_ind_t *parms =
				&data->parms.gen_ssp_numeric_confirm_ind_parms;
	GArray *g_bd_addr;
	agent_data_t *agent_data;

	DEBUG_FUNC("Called %d", data->id);

	g_assert(data->id == BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB);

	agent_data = g_new0(agent_data_t, 1);
	g_assert(agent_data != NULL);

	agent_data->id = data->id;
	memcpy(agent_data->bd_addr, parms->bd_addr, sizeof(BD_ADDRESS));

	g_bd_addr = bd_addr_to_g_array(parms->bd_addr);

	proxy_call = com_alps_bt_agent_ssp_numeric_confirm_async(
					agent_proxy,
					g_bd_addr,
					(char *) parms->name,
					parms->numeric_value,
					do_agent_ssp_numeric_confirm_reply,
					agent_data);

	g_bd_addr_free(g_bd_addr);

	DEBUG_FUNC("Exited");
}

/* handle agent solution for ConnectionRequestInd */
void do_agent_connection_request_ind_wrapper(callback_data *data)
{
	DBusGProxyCall *proxy_call;
	gen_connection_request_ind_t *parms =
				&data->parms.gen_connection_request_ind_parms;
	GArray *g_bd_addr;
	agent_data_t *agent_data;

	DEBUG_FUNC("Called %d", data->id);

	g_assert(data->id == BT_APPL_CONNECTION_REQUEST_IND_CB);

	agent_data = g_new0(agent_data_t, 1);
	g_assert(agent_data != NULL);

	agent_data->id = data->id;
	memcpy(agent_data->bd_addr, parms->bd_addr, sizeof(BD_ADDRESS));

	g_bd_addr = bd_addr_to_g_array(parms->bd_addr);

	agent_data->service = parms->service;

	proxy_call = com_alps_bt_agent_connection_request_async(agent_proxy,
					g_bd_addr,
					parms->service,
					do_agent_connection_request_reply,
					agent_data);

	g_bd_addr_free(g_bd_addr);

	DEBUG_FUNC("Exited");
}

void do_agent_callback_wrapper(callback_data *data)
{
	DBusGProxyCall *proxy_call;
	general_callback_parms *parms = &data->parms.general;
	GArray *g_bd_addr, *g_linkkey, *g_dlinkkey;
	agent_data_t *agent_data;

	DEBUG_FUNC("Called %d", data->id);

	agent_data = g_new0(agent_data_t, 1);
	g_assert(agent_data != NULL);

	agent_data->id = data->id;
	memcpy(agent_data->bd_addr, parms->bd_addr, sizeof(BD_ADDRESS));

	g_bd_addr = bd_addr_to_g_array(parms->bd_addr);

	switch (data->id) {
	case BT_APPL_REQUEST_PIN_IND_CB:
		proxy_call =
		    com_alps_bt_agent_request_pin_async(agent_proxy, g_bd_addr,
							(char *)parms->name,
							do_agent_request_pin_reply,
							agent_data);
		break;
	case BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB:
		proxy_call =
		    com_alps_bt_agent_ssp_passkey_notification_async
		    (agent_proxy, g_bd_addr, (char *)parms->name,
		     parms->passkey, do_agent_ssp_passkey_notification_reply,
		     agent_data);
		break;
	case BT_APPL_PAIRING_COMPLETE_IND_CB:
		g_linkkey = linkkey_to_g_array(&parms->pairing_info.aucLinkKey[0],
				ARRAYSIZE(parms->pairing_info.aucLinkKey));
		g_dlinkkey = linkkey_to_g_array(&parms->pairing_info.aucDLinkKey[0],
				ARRAYSIZE(parms->pairing_info.aucDLinkKey));

		proxy_call = com_alps_bt_agent_pairing_complete_async(
				agent_proxy,
				g_bd_addr,
				parms->status,
				parms->pairing_info.ucLinkKeyType,
				g_linkkey,
				g_dlinkkey,
				parms->pairing_info.stCod.usMajorServiceClass,
				parms->pairing_info.stCod.ucMajorDeviceClass,
				parms->pairing_info.stCod.ucMinorDeviceClass,
				do_agent_pairing_complete_reply,
				agent_data);

		g_linkkey_free(g_linkkey);
		g_linkkey_free(g_dlinkkey);
		break;
	case BT_APPL_LINKKEY_REQUEST_IND_CB:
		proxy_call =
		    com_alps_bt_agent_linkkey_request_async(agent_proxy,
							    g_bd_addr,
							    do_agent_linkkey_request_reply,
							    agent_data);
		break;
	case BT_APPL_AUTHENTICATION_IND_CB:
		proxy_call =
		    com_alps_bt_agent_authentication_async(agent_proxy,
							   g_bd_addr,
							   do_agent_authentication_reply,
							   agent_data);
		break;
	default:
		DEBUG_ERROR("Given unrecognized code %d", data->id);
		break;
	}

	g_bd_addr_free(g_bd_addr);

	DEBUG_FUNC("Exited");
}

int create_agent_dbus_proxy(DBusGConnection * bus)
{
	agent_proxy = dbus_g_proxy_new_for_name(bus,
						AGENT_SERVICE_NAME,
						AGENT_SERVICE_OBJECT_PATH,
						AGENT_SERVICE_INTERFACE);

	if (!agent_proxy) {
		DEBUG_ERROR("Failed to create the agent dbus proxy");
		return -1;
	}

	return 0;
}
