/*
 * main file for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <stdlib.h>
#include <execinfo.h>
#include <sched.h>
#include <sys/reboot.h>
#include <sys/stat.h>
#include <unistd.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "callbacks.h"
#include "node_manager.h"
#include "utils_async_queue.h"
#include "utils_test_queue.h"
#include "device_info.h"
#include "node_device.h"
#include "node_adapter.h"
#include "node_agent.h"
#include "node_test.h"
#include "node_validation.h"
#include "handle_phonebook.h"

#include "debug.h"

#include "bt_appl_dummy.h"

#define EXCEPTION_HANDLER_REBOOT_FILE "/opt/bosch/disable_reset.txt"

/************/
/* #defines */
/************/

#define MAX_INIT_RETRIES	(10)

/****************/
/* LOCAL GLOBAL */
/****************/

static gboolean display_version = FALSE;
static gboolean version_check = FALSE;
static gboolean show_settings = FALSE;
static gint audio_thread_priority = 50;	/* default RT priority */
static gint alps_thread_priority = 2;	/* default RT priority */
static gboolean requested_system_reboot_upon_crash = FALSE;
static volatile gboolean system_reboot_upon_crash_allowed = FALSE;
static guint dbus_api_options = 0;
static gint max_init_count = MAX_INIT_RETRIES;

static gint alps_debug_level = 0;

static GOptionEntry cmd_options[] =
{
	[0] = {
		.long_name = "version",
		.short_name = 'v',
		.flags = 0,
		.arg = G_OPTION_ARG_NONE,
		.arg_data = &display_version,
		.description = "Shows the version string",
		.arg_description = NULL},
	[1] = {
		.long_name = "no-ALPS-version-check",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_NONE,
		.arg_data = &version_check,
		.description = "Disables the ALPS libraries version test",
		.arg_description = NULL},
	[2] = {
		.long_name = "show-settings",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_NONE,
		.arg_data = &show_settings,
		.description = "Shows the settings that are in use",
		.arg_description = NULL},
	[3] = {
		.long_name = "audio-thread-priority",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_INT,
		.arg_data = &audio_thread_priority,
		.description = "Sets the real-time priority of the audio thread",
		.arg_description = "specify a high real-time priority"},
	[4] = {
		.long_name = "alps-thread-priority",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_INT,
		.arg_data = &alps_thread_priority,
		.description = "Sets the real-time priority of the ALPS threads",
		.arg_description = "specify a low real-time priority"},
	[5] = {
		.long_name = "enable-system-reboot-upon-crash",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_NONE,
		.arg_data = &requested_system_reboot_upon_crash,
		.description = "Allow a system reboot after a crash is detected",
		.arg_description = NULL},
	[6] = {
		.long_name = "dbus-api",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_INT,
		.arg_data = &dbus_api_options,
		.description = "Select various D-bus API options",
		.arg_description = "bitmap of options"},
	[7] = {
		.long_name = "max-init-count",
		.short_name = 0,
		.flags = 0,
		.arg = G_OPTION_ARG_INT,
		.arg_data = &max_init_count,
		.description = "Max number of initialization attempts",
		.arg_description = NULL},
	{NULL}
};

/* exported function */
guint get_audio_thread_priority(void)
{
	return (guint) audio_thread_priority;
}

/* exported function */
guint get_alps_thread_priority(void)
{
	return (guint) alps_thread_priority;
}

/* exported function */
guint8 get_alps_debug_level(void)
{
	guint8 ret;

	if (alps_debug_level >= 0) {
		if (alps_debug_level < G_MAXUINT8) {
			ret = (guint8) alps_debug_level;
		} else {
			ret = G_MAXUINT8;
		}
	} else {
		ret = 0;
	}

	return ret;
}

/* Main event loop */
static GMainLoop *loop = NULL;

/* TRUE means perform the ALPS libraries version check */
static gboolean alps_vcheck = TRUE;

static void set_alps_version_check(gboolean setting)
{
	alps_vcheck = setting;
}

/* exported function */
gboolean test_alps_version_check(void)
{
	return alps_vcheck;
}

/* exported function */
guint read_dbus_api_options(void)
{
	return dbus_api_options;
}

/* exported function */
gint read_max_init_count(void)
{
	return max_init_count;
}

/* SIGINT signal handler used by gconv testing */
static void sigint_handler(int sig)
{
	DEBUG("Terminating the daemon...");
	(void) sig;
	exit(0);
}

#define MAX_ADDRESSES	(100)
/*
 * General signal handler to manage
 * SIGSEGV for segmentation fault
 * SIGBUS for bus error (alignment fault)
 */
static void gen_sig_handler(int sig)
{
	void *trace[MAX_ADDRESSES];
	size_t size;
	guint16 alps_version;
	guint8 alps_test_version;

	/* how many entries in the backtrace ? */
	size = backtrace(trace, MAX_ADDRESSES);

	DEBUG("Crash signal value is: %d", sig);

	switch (sig) {
	case SIGSEGV:
		DEBUG_ERROR("FATAL ERROR: segmentation fault detected");
		break;
	case SIGBUS:
		DEBUG_ERROR("FATAL ERROR: bus error detected");
		break;
	case SIGABRT:
		DEBUG_ERROR("FATAL ERROR: abort detected");
		break;
	default:
		DEBUG_ERROR("FATAL ERROR: got crash signal %d");
	}

	DEBUG_ERROR("%s", D_BUS_BT_VERSION);
	alps_version = get_alps_stack_version();
	alps_test_version = get_alps_stack_test_version();

	if (alps_test_version == 0) {
		DEBUG_ERROR("ALPS BT stack version is %u.%02u",
				alps_version/256, alps_version%256);
	} else {
		DEBUG_ERROR("ALPS BT stack version is %u.%02u: test build %u",
			alps_version/256, alps_version%256, alps_test_version);
	}

	/* print out the frames to STDERR */
	backtrace_symbols_fd(trace, size, 2);

	/* Is a system reboot after the crash allowed ? */
	if (system_reboot_upon_crash_allowed == TRUE) {
		DEBUG("System reboot upon crash is enabled");
		fprintf(stderr, "BTD rebooting system... (to prevent: touch %s)",
				EXCEPTION_HANDLER_REBOOT_FILE);

		/* try to write the output before doing reboot */
		fflush(NULL);
		usleep(100000l);

		reboot(RB_AUTOBOOT);
	} else {
		DEBUG("System reboot upon crash is disabled");
		exit(1);
	}
}

/*************/
/* MAIN CODE */
/*************/

int main(int argc, char *argv[])
{
	DBusGConnection *bus = NULL;
	GError *error = NULL;
	GOptionContext *context;
	gint max_rt_priority;
	gint min_rt_priority;
	char *loglev;
	struct sigaction sa_sigint;
	struct sigaction sa_sigsegv;
	struct sigaction sa_sigbus;
	struct sigaction sa_sigabrt;
	struct stat buf;

	/* proxy object to the D-bus manager interface */
	DBusGProxy *dbus_manager_proxy = NULL;

	guint result;

	sa_sigint.sa_handler = sigint_handler;
	sigemptyset(&sa_sigint.sa_mask);
	sa_sigint.sa_flags = SA_ONSTACK;

	sa_sigsegv.sa_handler = gen_sig_handler;
	sigemptyset(&sa_sigsegv.sa_mask);
	sa_sigsegv.sa_flags = SA_ONSTACK;

	sa_sigbus.sa_handler = gen_sig_handler;
	sigemptyset(&sa_sigbus.sa_mask);
	sa_sigbus.sa_flags = SA_ONSTACK;

	sa_sigabrt.sa_handler = gen_sig_handler;
	sigemptyset(&sa_sigabrt.sa_mask);
	sa_sigabrt.sa_flags = SA_ONSTACK;

	/* register the signal handler to allow gconv to terminate the daemon */
	if (sigaction(SIGINT, &sa_sigint, NULL) == -1) {
		DEBUG_ERROR("Failed to register the SIGINT handler");
		return -1;
	}

	/* register the signal handler to trap segmentation faults */
	if (sigaction(SIGSEGV, &sa_sigsegv, NULL) == -1) {
		DEBUG_ERROR("Failed to register the SIGSEGV handler");
		return -1;
	}

	/* register the signal handler to trap bus errors */
	if (sigaction(SIGBUS, &sa_sigbus, NULL) == -1) {
		DEBUG_ERROR("Failed to register the SIGBUS handler");
		return -1;
	}

	/* register the signal handler to trap abort (g_assert) errors */
	if (sigaction(SIGABRT, &sa_sigabrt, NULL) == -1) {
		DEBUG_ERROR("Failed to register the SIGABRT handler");
		return -1;
	}

	/*
	 * Get the debug log level from the environment.
	 * First, set the debug log level to the same level as the ALPS
	 * FUSION log level, then set the log level to the level defined
	 * by the BTD_DEBUG_LEVEL environment variable.  This allows us
	 * to set different log levels for the Bluetooth daemon and the ALPS
	 * stack, if desired.
	 */
	loglev = getenv("FUSION_LOG_LEVEL");
	if (loglev) {
		alps_debug_level = atoi(loglev);
		btd_debug_level = alps_debug_level;
	}
	loglev = getenv("BTD_LOG_LEVEL");
	if (loglev)
		btd_debug_level = atoi(loglev);
	if (btd_debug_level < BTD_DBG_NONE_LEV)
		btd_debug_level = BTD_DBG_NONE_LEV;
	else if (btd_debug_level > BTD_DBG_MAX_LEV)
		btd_debug_level = BTD_DBG_MAX_LEV;

	/*
	 * Do not suppress error logging to the IVI error memory mechanism
	 * via STDERR
	 */
	if (btd_debug_level < BTD_DBG_ERR_LEV)
		btd_debug_level = BTD_DBG_ERR_LEV;

	/* process the command line parameters */
	context = g_option_context_new("");
	g_option_context_set_summary(context, "This is the D-bus Bluetooth "
		"Daemon that uses D-bus to control the ALPS Bluetooth stack");
	g_option_context_add_main_entries(context, cmd_options, NULL);
	g_option_context_set_description(context, 
		"Debug logging is controlled via the environmental variables:\n"
		"FUSION_LOG_LEVEL sets the ALPS stack log level, range 0 to 5\n"
		"BTD_LOG_LEVEL sets the BTD log level to be different to the "
		"ALPS stack, range 0 to 5\n\n"
		"Inhibit system reboot upon crash by `touch "
		EXCEPTION_HANDLER_REBOOT_FILE"`\n"
		"By default, system reboot upon crash is disabled\n\n"
		"D-bus API bitmap options are:\n"
		"bit 0 set = use alternative non-agent SspNumericConfirm\n"
		"bit 1 set = use alternative non-agent ConnectionRequest\n");

	/* note -h and --help exits the program */
	if (g_option_context_parse(context, &argc, &argv, &error) == FALSE)
	{
		g_printf("option parsing failed: %s\n", error->message);
		g_error_free(error);
		g_printf("use -h or --help to see the help\n");
		g_option_context_free(context);
		return 1;
	}

	g_option_context_free(context);

	if (version_check == TRUE) {
		g_printf("ALPS libraries version check is disabled\n");
		set_alps_version_check(FALSE);
	}

	if (requested_system_reboot_upon_crash == TRUE) {
		/* check for presence of system reboot inhibitor file */
		if (stat(EXCEPTION_HANDLER_REBOOT_FILE, &buf) == 0) {
			g_printf("BTD system reboot upon crash is inhibited\n");
			system_reboot_upon_crash_allowed = FALSE;
		} else {
			g_printf("BTD system reboot upon crash is enabled\n");
			system_reboot_upon_crash_allowed = TRUE;
		}
	}

	if (show_settings == TRUE) {
		g_printf("The settings in use are:\n");
		g_printf("Audio thread priority is %d\n", audio_thread_priority);
		g_printf("ALPS thread priority is %d\n", alps_thread_priority);
		g_printf("ALPS stack log level is %d\n", alps_debug_level);
		g_printf("BTD log level is %d\n", btd_debug_level);
		g_printf("D-bus API bitmap setting is 0x%02X\n",
				read_dbus_api_options());
		g_printf("Maxium number of initialization attempts is %d\n",
				read_max_init_count());
	}

	if (display_version == TRUE) {
		g_printf("%s\n", D_BUS_BT_VERSION);
		return 1;
	}

	DEBUG("%s starting...", D_BUS_BT_VERSION);

	if (requested_system_reboot_upon_crash == FALSE) {
		DEBUG("System reboot upon crash is disabled");
	}

	if (read_dbus_api_options() & DBUS_USE_ALT_SSP_NUMERIC_CONFIRM) {
		DEBUG("D-bus: Using alternative non-agent SspNumericConfirm");
	}

	if (read_dbus_api_options() & DBUS_USE_ALT_CONNECTION_REQUEST) {
		DEBUG("D-bus: Using alternative non-agent ConnectionRequest");
	}

	max_rt_priority = sched_get_priority_max(SCHED_FIFO);
	g_assert(max_rt_priority >= 0);

	min_rt_priority = sched_get_priority_min(SCHED_FIFO);
	g_assert(min_rt_priority >= 0);

	/* check that the thread priority is in range */
	if ((audio_thread_priority > max_rt_priority) ||
		(audio_thread_priority < min_rt_priority)) {
		DEBUG_ERROR("audio_thread_priority %d is out of "
			"range: min is %d, max is %d",
			audio_thread_priority, min_rt_priority, max_rt_priority);
		return -1;
	}

	/* check that the thread priority is in range */
	if ((alps_thread_priority > max_rt_priority) ||
		(alps_thread_priority < min_rt_priority)) {
		DEBUG_ERROR("alps_thread_priority %d is out of "
			"range: min is %d, max is %d",
			alps_thread_priority, min_rt_priority, max_rt_priority);
		return -1;
	}

	if (read_max_init_count() < 1) {
		DEBUG_ERROR("max-init-count %d is out of range",
			read_max_init_count());
		return -1;
	}

	/* initialise GLib type system */
	g_type_init();
	g_thread_init(NULL);

#ifdef USE_DUMMY_API
	/* initialise the dummy API function handling */

#ifdef USE_TEST_NODE
	dummy_use_all_dummy_api_functions(FALSE);
#else
	dummy_use_all_dummy_api_functions(TRUE);
#endif
#endif

	/* Create the main loop instance, uses default context */
	loop = g_main_loop_new(NULL, FALSE);

	DEBUG("connecting to the System D-bus...");

	/* Connect to the system D-bus */
	bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (bus == NULL) {
		DEBUG_ERROR("Failed to open connection to bus: %s",
				error->message);

		g_error_free(error);
		return -1;
	}

	DEBUG_HIGH("attaching to the D-bus interface entity...");

	/*
	 * need to register our service onto the System bus
	 * so need to use a proxy object to the D-bus interface entity
	 */
	dbus_manager_proxy = dbus_g_proxy_new_for_name_owner(bus,
		"org.freedesktop.DBus"	/* service name */,
		"/org/freedesktop/DBus"	/* object path */,
		"org.freedesktop.DBus"	/* interface */,
		&error);

	if (!dbus_manager_proxy) {
		DEBUG_ERROR("Couldn't connect to the D-bus interface: %s",
				error->message);

		g_error_free(error);
		return -1;
	}

	DEBUG_HIGH("registering our service via the D-bus interface object...");

	/* The function will return FALSE if it sets the GError. */
	if (!dbus_g_proxy_call(dbus_manager_proxy,
			/* Method name to call. */
			"RequestName",
			/* Where to store the GError. */
			&error,
			/*
			 * Parameter type of argument 0. Note that
			 * since we're dealing with GLib/D-Bus
			 * wrappers, you will need to find a suitable
			 * GType instead of using the "native" D-Bus
			 * type codes.
			 */
			G_TYPE_STRING,
			/*
			 * Data of argument 0. In our case, this is
			 * the well-known name Service Name for our server
			 */
			"com.alps.bt",
			/* Parameter type of argument 1. */
			G_TYPE_UINT,
			/*
			 * Data of argument 0. This is the "flags"
			 * argument of the "RequestName" method which
			 * can be use to specify what the bus service
			 * should do when the name already exists on
			 * the bus. We'll go with defaults.
			 */
			0,
			/*
			 * Input arguments are terminated with a
			 * special GType marker.
			 */
			G_TYPE_INVALID,
			/*
			 * Parameter type of return value 0.
			 * For "RequestName" it is UINT32 so we pick
			 * the GType that maps into UINT32 in the
			 * wrappers.
			 */
			G_TYPE_UINT,
			/*
			 * Data of return value 0. These will always
			 * be pointers to the locations where the
			 * proxy can copy the results.
			 */
			&result,
			/* Terminate list of return values. */
			G_TYPE_INVALID)) {
		DEBUG_ERROR("D-Bus.RequestName RPC failed: %s\n",
				error->message);

		g_error_free(error);
		return -1;

		/*
		 * Note that the whole call failed, not "just" the name
		 * registration (we deal with that below). This means that
		 * something bad probably has happened and there's not much
		 * we can do (hence program termination).
		 */
	}

	/* Check the result code of the registration RPC. */
	DEBUG_HIGH("RequestName returned %d", result);
	if (result != 1) {
		DEBUG_ERROR("Failed to register our Service Name on D-bus");
		return -1;

		/*
		 * In this case we could also continue instead of terminating.
		 * We could retry the RPC later. Or "lurk" on the bus waiting
		 * for someone to tell us what to do. If we would be publishing
		 * multiple services and/or interfaces, it even might make sense
		 * to continue with the rest anyway.
		 *
		 * In our simple program, we terminate. Not much left to do for
		 * this poor program if the clients won't be able to find the
		 * Value object using the well-known name.
		 */
	}



	if (manager_object_creation(bus, "/") == FALSE) {
		DEBUG("Failed to create the manager object");
		return -1;
	}


	if (adapter_object_creation(bus, "/com/alps/bt") == FALSE) {
		DEBUG("Failed to create the adapter object");
		return -1;
	}

#ifdef USE_TEST_NODE
	if (test_object_creation(bus, "/com/alps/bt_test") == FALSE) {
		DEBUG("Failed to create the test object");
		return -1;
	}
#endif

#ifdef USE_VALIDATION_NODE
	if (validation_object_creation(bus, "/com/alps/bt_validation") == FALSE) {
		DEBUG("Failed to create the test object");
		return -1;
	}
#endif

	if (create_pbdl_agent_dbus_proxy(bus) == FALSE) {
		DEBUG("Failed to create a proxy to the pbdl agent object");
		return -1;
	}

	/* initialize the agent dbus proxy */
	if (create_agent_dbus_proxy(bus))
	   return -1;

	if (initialize_alps_callbacks())
	   return -1;

	DEBUG("starting the GLib main loop");

	/* Main loop */
	g_main_loop_run(loop);

	DEBUG("finished\n");

	return 0;
}


