/*
 * handle_pnp_info.c
 * PNP Information handling for the D-bus Bluetooth Daemon
 *
 * Author: Vedahari Narasimhan <VedahariNarasimhan.Ganesan@in.bosch.com>
 *
 * 2014 (c) RBEI, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "callbacks.h"
#include "node_adapter.h"
#include "handle_pnp_info.h"
#include "debug.h"

/* ALPS continue flag codes */
#define	CONTINUE_FIRST		(2)
#define	CONTINUE_CONTINUE	(1)
#define	CONTINUE_FINAL		(0)

enum info_ind_state {
	PNP_INFO_IND_IDLE,
	PNP_INFO_IND_GOT_DATA,
	PNP_INFO_IND_TOTAL_STATES
};

/*****************/
/* LOCAL GLOBALS */
/*****************/

static GPtrArray *all_pnp_info_ind = NULL;

static enum info_ind_state pnp_info_ind_state = PNP_INFO_IND_IDLE;

static BD_ADDRESS pnp_bd_addr;

#ifdef DBG_ENABLED
static gchar *pnp_info_ind_state_str[PNP_INFO_IND_TOTAL_STATES] =
{
	[PNP_INFO_IND_IDLE] = "PNP_INFO_IND_IDLE",
	[PNP_INFO_IND_GOT_DATA] = "PNP_INFO_IND_GOT_DATA"
};
#endif

/*
 * exported function
 *
 * state machine to handle the SPP capabilities
 */
void handle_pnp_info_ind(callback_data *data)
{
	gen_pnp_info_attribute_ind_t *parms =
				&data->parms.gen_pnp_info_attribute_ind_parms;
	GValue *value;

	g_assert(data->id == BT_APPL_PNPINFO_IND_CB);

	g_assert(pnp_info_ind_state < PNP_INFO_IND_TOTAL_STATES);

	DEBUG_FUNC("Called, continue_flag %u", parms->continue_flag);
	DEBUG("pnp_info_ind_state %s",
			pnp_info_ind_state_str[pnp_info_ind_state]);

	if ((pnp_info_ind_state == PNP_INFO_IND_GOT_DATA) &&
		(parms->continue_flag == CONTINUE_FIRST)) {

		gpointer free_ptr;

		/*
		 * a final continue_flag failed to appear so abort previous
		 * stored data
		 */
		DEBUG_ERROR("Aborting previously obtained PnP Information");

		g_assert(all_pnp_info_ind != NULL);
		free_ptr = g_ptr_array_free(all_pnp_info_ind, TRUE);
		g_assert(free_ptr == NULL);

		all_pnp_info_ind = NULL;

		pnp_info_ind_state = PNP_INFO_IND_IDLE;
	}

	if (pnp_info_ind_state == PNP_INFO_IND_IDLE) {
		/*
		 * In all continue_flag states
		 * allocate a GPtrArray for the array of structs
		 */
		g_assert(all_pnp_info_ind == NULL);
		all_pnp_info_ind = g_ptr_array_new();
		g_assert(all_pnp_info_ind != NULL);

		/* save BD address to check all calls are for this BD address */
		memcpy(pnp_bd_addr, parms->bd_addr, sizeof(BD_ADDRESS));

		pnp_info_ind_state = PNP_INFO_IND_GOT_DATA;
		//ALPS sometimes sends the PnP details with the continue flag as final.Hence
		//it is necessary to create the structure beforehand
		if(CONTINUE_FINAL==parms->continue_flag)
		{
			/* allocate a GValue struct and fill in the data */
			value = g_new0(GValue, 1);
			g_assert(value != NULL);
			g_value_init(value, DBUS_STRUCT_UINT_UINT_UINT_UINT_UCHAR_UINT_AUCHAR_AUCHAR);
			g_value_take_boxed(value, dbus_g_type_specialized_construct(
					DBUS_STRUCT_UINT_UINT_UINT_UINT_UCHAR_UINT_AUCHAR_AUCHAR));

			dbus_g_type_struct_set(value,
					0, parms->specificationID,
					1, parms->vendorID,
					2, parms->productID,
					3, parms->version,
					4, parms->primaryRecord,
					5, parms->vendorIDSource,
					6, parms->url_length,
					7, parms->url_name,
					G_MAXUINT);

			g_ptr_array_add(all_pnp_info_ind, g_value_get_boxed(value));
			g_free(value);
		}
	}
	else
	{
		guint8 loop;
		/*
		 * check the BD addresses are consistent.
		 * If a mismatch occurs then suggests
		 * simultaneous SPP BT_APPL_PNP_INFO_IND
		 * call sessions are occurring and this
		 * is not supported
		 */
		for (loop = 0; loop < sizeof(BD_ADDRESS); loop++)
		{
			g_assert(pnp_bd_addr[loop] == parms->bd_addr[loop]);
		}
	}

	g_assert(pnp_info_ind_state == PNP_INFO_IND_GOT_DATA);

	switch (parms->continue_flag)
	{
	case CONTINUE_FIRST:
	case CONTINUE_CONTINUE:
		/* allocate a GValue struct and fill in the data */
		value = g_new0(GValue, 1);
		g_assert(value != NULL);

		g_value_init(value, DBUS_STRUCT_UINT_UINT_UINT_UINT_UCHAR_UINT_AUCHAR_AUCHAR);
		g_value_take_boxed(value, dbus_g_type_specialized_construct(
				DBUS_STRUCT_UINT_UINT_UINT_UINT_UCHAR_UINT_AUCHAR_AUCHAR));

		dbus_g_type_struct_set(value,
				0, parms->specificationID,
				1, parms->vendorID,
				2, parms->productID,
				3, parms->version,
				4, parms->primaryRecord,
				5, parms->vendorIDSource,
				6, parms->url_length,
				7, parms->url_name,
				G_MAXUINT);

		g_ptr_array_add(all_pnp_info_ind, g_value_get_boxed(value));
		g_free(value);
		break;
	case CONTINUE_FINAL:
	{
		/***************/
		/* send signal */
		/***************/

		DEBUG("Calling the signal send function");
		emit_pnp_info_ind(&pnp_bd_addr,
				all_pnp_info_ind);

		/* To check for memory leak */
		gpointer free_ptr;
		free_ptr = g_ptr_array_free(all_pnp_info_ind, TRUE);
		g_assert(free_ptr == NULL);
		all_pnp_info_ind = NULL;
		pnp_info_ind_state = PNP_INFO_IND_IDLE;
	}
	break;
	default:
		DEBUG_ERROR("Unhandled continue_flag value: %u",
				parms->continue_flag);
		break;
	}

	// free URL length and name arrays (might not be populated)
	g_array_unref(parms->url_length);
	parms->url_length = NULL;

	g_byte_array_unref(parms->url_name);
	parms->url_name = NULL;

	DEBUG_FUNC("Exited");
}
