/*
 * handle_messaging.c
 * messaging handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2011 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "utils_async_queue.h"
#include "device_info.h"
#include "node_device.h"
#include "callbacks.h"
#include "handle_messaging.h"
#include "debug.h"

/*
 * exported global to hold the msg list state strings
 *
 * Note these strings are not limited to debug messages
 */
gchar *map_list_msg_state_str[MAP_LIST_MSG_TOTAL_STATES] =
{
	[MAP_LIST_MSG_IDLE] = "MAP_LIST_MSG_IDLE",
	[MAP_LIST_MSG_WAIT_DATA] = "MAP_LIST_MSG_WAIT_DATA",
	[MAP_LIST_MSG_GOT_DATA] = "MAP_LIST_MSG_GOT_DATA",
	[MAP_LIST_MSG_SEND_DATA] = "MAP_LIST_MSG_SEND_DATA",
	[MAP_LIST_MSG_COMPLETE_DATA] = "MAP_LIST_MSG_COMPLETE_DATA"
};

/*
 * Allocate the map_list_msg info structure members
 */
static void handle_map_list_msg_info_allocate(MapListMsgInfo *msg_info,
						guint32 max_recs_per_trans)
{
	DEBUG_FUNC("Called");

	g_assert(msg_info != NULL);

	msg_info->recs_trans_count = 0;

	g_assert(msg_info->handle_list == NULL);
	msg_info->handle_list = g_ptr_array_new();
	g_assert(msg_info->handle_list != NULL);

	g_assert(msg_info->subject_list == NULL);
	msg_info->subject_list = g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->subject_list != NULL);

	g_assert(msg_info->date_time_list == NULL);
	msg_info->date_time_list = g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->date_time_list != NULL);

	g_assert(msg_info->sender_name_list == NULL);
	msg_info->sender_name_list = g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->sender_name_list != NULL);

	g_assert(msg_info->sender_address_list == NULL);
	msg_info->sender_address_list = g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->sender_address_list != NULL);

	g_assert(msg_info->replyto_address_list == NULL);
	msg_info->replyto_address_list =
					g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->replyto_address_list != NULL);

	g_assert(msg_info->recipient_name_list == NULL);
	msg_info->recipient_name_list = g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->recipient_name_list != NULL);

	g_assert(msg_info->recipient_address_list == NULL);
	msg_info->recipient_address_list =
					g_new0(gchar *, max_recs_per_trans + 1);
	g_assert(msg_info->recipient_address_list != NULL);

	g_assert(msg_info->type_list == NULL);
	msg_info->type_list = g_byte_array_sized_new(1);
	g_assert(msg_info->type_list != NULL);

	g_assert(msg_info->size_list == NULL);
	msg_info->size_list =
			g_array_sized_new(FALSE, FALSE, sizeof(guint), 1);
	g_assert(msg_info->size_list != NULL);

	g_assert(msg_info->text_list == NULL);
	msg_info->text_list = g_byte_array_sized_new(1);
	g_assert(msg_info->text_list != NULL);

	g_assert(msg_info->reception_status_list == NULL);
	msg_info->reception_status_list = g_byte_array_sized_new(1);
	g_assert(msg_info->reception_status_list != NULL);

	g_assert(msg_info->attach_size_list == NULL);
	msg_info->attach_size_list =
			g_array_sized_new(FALSE, FALSE, sizeof(guint), 1);
	g_assert(msg_info->attach_size_list != NULL);

	g_assert(msg_info->priority_list == NULL);
	msg_info->priority_list = g_byte_array_sized_new(1);
	g_assert(msg_info->priority_list != NULL);

	g_assert(msg_info->read_list == NULL);
	msg_info->read_list = g_byte_array_sized_new(1);
	g_assert(msg_info->read_list != NULL);

	g_assert(msg_info->sent_list == NULL);
	msg_info->sent_list = g_byte_array_sized_new(1);
	g_assert(msg_info->sent_list != NULL);

	g_assert(msg_info->protect_list == NULL);
	msg_info->protect_list = g_byte_array_sized_new(1);
	g_assert(msg_info->protect_list != NULL);

	DEBUG("Exit");
}

/*
 * exported function
 *
 * Free the map_list_msg info structure members
 */
void handle_map_list_msg_info_free(MapListMsgInfo *msg_info,
						guint32 max_recs_per_trans)
{
	gpointer freeing_ptr;
	guint32 rec_loop;
	guint32 num_elements;
	guint32 idx;

	DEBUG_FUNC("Called");

	g_assert(msg_info != NULL);

	if (msg_info->handle_list != NULL) {

		num_elements = msg_info->handle_list->len;

		DEBUG("handle_list: freeing %u items", num_elements);

		for (idx = 0; idx < num_elements; idx++) {
			freeing_ptr =
				g_ptr_array_index(msg_info->handle_list, idx);
			g_byte_array_unref(freeing_ptr);
		}				

		g_ptr_array_unref(msg_info->handle_list);
		msg_info->handle_list = NULL;
	}

	DEBUG("Freeing the string memory...");
	for (rec_loop = 0; rec_loop < max_recs_per_trans; rec_loop++) {
		g_free(msg_info->subject_list[rec_loop]);
		g_free(msg_info->date_time_list[rec_loop]);
		g_free(msg_info->sender_name_list[rec_loop]);
		g_free(msg_info->sender_address_list[rec_loop]);
		g_free(msg_info->replyto_address_list[rec_loop]);
		g_free(msg_info->recipient_name_list[rec_loop]);
		g_free(msg_info->recipient_address_list[rec_loop]);
	}

	DEBUG("Freeing the lists...");
	g_free(msg_info->subject_list);
	msg_info->subject_list = NULL;

	g_free(msg_info->date_time_list);
	msg_info->date_time_list = NULL;

	g_free(msg_info->sender_name_list);
	msg_info->sender_name_list = NULL;

	g_free(msg_info->sender_address_list);
	msg_info->sender_address_list = NULL;

	g_free(msg_info->replyto_address_list);
	msg_info->replyto_address_list = NULL;

	g_free(msg_info->recipient_name_list);
	msg_info->recipient_name_list = NULL;

	g_free(msg_info->recipient_address_list);
	msg_info->recipient_address_list = NULL;

	g_byte_array_unref(msg_info->type_list);
	msg_info->type_list = NULL;

	g_array_unref(msg_info->size_list);
	msg_info->size_list = NULL;

	g_byte_array_unref(msg_info->text_list);
	msg_info->text_list = NULL;

	g_byte_array_unref(msg_info->reception_status_list);
	msg_info->reception_status_list = NULL;

	g_array_unref(msg_info->attach_size_list);
	msg_info->attach_size_list = NULL;

	g_byte_array_unref(msg_info->priority_list);
	msg_info->priority_list = NULL;

	g_byte_array_unref(msg_info->read_list);
	msg_info->read_list = NULL;

	g_byte_array_unref(msg_info->sent_list);
	msg_info->sent_list = NULL;

	g_byte_array_unref(msg_info->protect_list);
	msg_info->protect_list = NULL;

	DEBUG("Exit");
}

/*
 * exported function
 *
 * use this function to switch to the MAP_LIST_MSG_WAIT_DATA from any
 * state. Note data will be lost if the state was MAP_LIST_MSG_GOT_DATA.
 */
void handle_map_list_msg_set_wait_data(DeviceObject *object, gchar *service_name,
						guint32 service_name_len,
						guint32 max_recs_per_trans,
						gboolean use_agent)
{
	DEBUG_FUNC("Called");

	DEBUG("MAP list messaging download state is %s",
			map_list_msg_state_str[object->map_list_msg_state]);

	g_assert(service_name != NULL);

	if (object->map_list_msg_state == MAP_LIST_MSG_GOT_DATA) {
		/* need to throw away any previously stored data */

		DEBUG_ERROR("Was in \'%s\' state for the new download "
			"start, now deleting data from the previous session...",
			map_list_msg_state_str[object->map_list_msg_state]);
	}

	if ((object->map_list_msg_state == MAP_LIST_MSG_GOT_DATA) ||
		(object->map_list_msg_state == MAP_LIST_MSG_WAIT_DATA)) {

		g_assert(object->map_list_msg_info != NULL);

		handle_map_list_msg_info_free(object->map_list_msg_info,
				object->map_list_msg_info->max_recs_per_trans);

		/* free old service name */
		g_free(object->map_list_msg_info->service_name);
		object->map_list_msg_info->service_name = NULL;

	} else {
		/* allocate memory to hold the MAP list msg info */
		g_assert(object->map_list_msg_info == NULL);
		object->map_list_msg_info = g_malloc0(sizeof(MapListMsgInfo));
		g_assert(object->map_list_msg_info != NULL);

		object->map_list_msg_info->service_name = NULL;
	}

	/* select the wait for data state */
	object->map_list_msg_state = MAP_LIST_MSG_WAIT_DATA;

	/* store MAP list msg request start info into the info struct */
	object->map_list_msg_info->max_recs_per_trans = max_recs_per_trans;
	object->map_list_msg_info->use_agent = use_agent;

	DEBUG("Service name is %s", service_name);
	object->map_list_msg_info->service_name = 
			g_strndup((gchar *) service_name, service_name_len);

	/* set the initial conditions for the start of the download */
	handle_map_list_msg_info_allocate(object->map_list_msg_info,
							max_recs_per_trans );

	DEBUG_FUNC("Exited");
}

static void handle_map_list_msg_send_data(DeviceObject *object)
{
	DEBUG_FUNC("Called");

	g_assert(object->map_list_msg_info != NULL);

	DEBUG("MAP list messaging download state is %s",
			map_list_msg_state_str[object->map_list_msg_state]);

	/* Check the download state is valid */
	g_assert((object->map_list_msg_state == MAP_LIST_MSG_SEND_DATA) ||
		(object->map_list_msg_state == MAP_LIST_MSG_COMPLETE_DATA));

	/* transmit the pending records */
	DEBUG("%d records to be sent",
				object->map_list_msg_info->recs_trans_count);

	if (object->map_list_msg_info->use_agent == TRUE) {
		DEBUG("using the agent method");

		DEBUG_ERROR("Agent method not supported");
	} else {
		DEBUG("using the signal solution");

		if( object->map_list_msg_info->recs_trans_count != 0) {
			/* only send the signal when data is available */
			emit_map_list_msg_data(object);
		} else {
			DEBUG("No data to send so do not send signal");
		}
	}

	DEBUG("finished with the MAP list msg info");

	switch (object->map_list_msg_state) {
	case MAP_LIST_MSG_SEND_DATA:
		/* select the wait for data state */
		object->map_list_msg_state = MAP_LIST_MSG_WAIT_DATA;

		/* reset the record counter */
		object->map_list_msg_info->recs_trans_count = 0;

		/* free the MAP list message info */
		handle_map_list_msg_info_free(object->map_list_msg_info,
				object->map_list_msg_info->max_recs_per_trans);

		/* allocate fresh MAP list message info */
		handle_map_list_msg_info_allocate(
				object->map_list_msg_info,
				object->map_list_msg_info->max_recs_per_trans);
		break;
	case MAP_LIST_MSG_COMPLETE_DATA:
		/* select the idle state */
		object->map_list_msg_state = MAP_LIST_MSG_IDLE;

		/* free the MAP list message info */
		handle_map_list_msg_info_free(object->map_list_msg_info,
				object->map_list_msg_info->max_recs_per_trans);

		/* free the MAP list message holding structure */
		g_free(object->map_list_msg_info);
		object->map_list_msg_info = NULL;
		break;
	default:
		DEBUG_ERROR("Unhandled state %d \'%s\'",
			object->map_list_msg_state,
			map_list_msg_state_str[object->map_list_msg_state]);
	}

	DEBUG("MAP list messaging download state is %s",
			map_list_msg_state_str[object->map_list_msg_state]);

	DEBUG_FUNC("Exited");
}

/*
 * exported function
 */
gboolean handle_map_list_msg_data_ind(map_list_msg_data_ind_t *data_info)
{
	DeviceObject *object = NULL;
	guint32 current_rec;
	MapListMsgInfo *msg_info;

	DEBUG_FUNC("Called");

	g_assert(data_info != NULL);
	g_assert(data_info->bd_addr != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(data_info->bd_addr);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the MAP list msg data",
			bd_addr_to_str(data_info->bd_addr));

		/* FIXME: Need to free members of data_info */
		return FALSE;
	}

	DEBUG("MAP list messaging download state is %s",
			map_list_msg_state_str[object->map_list_msg_state]);

	if ((object->map_list_msg_state != MAP_LIST_MSG_GOT_DATA) &&
		(object->map_list_msg_state != MAP_LIST_MSG_WAIT_DATA)) {

		DEBUG_ERROR("Unexpected and unhandled "
			"BT_APPL_MAP_List_Msg_IND received");

		/* FIXME: Need to free members of data_info */
		return FALSE;
	}

	/* Check the download state is valid */
	g_assert((object->map_list_msg_state == MAP_LIST_MSG_GOT_DATA) ||
			(object->map_list_msg_state == MAP_LIST_MSG_WAIT_DATA));

	g_assert(object->map_list_msg_info != NULL);

	/* check that the service name matches the request */
	if (g_strcmp0(data_info->service_name,
			object->map_list_msg_info->service_name) != 0) {
		DEBUG_ERROR("service name mismatched");

		/* FIXME: Need to free members of data_info */
		return FALSE;
	}

	g_free(data_info->service_name);

	/* select the got data state */
	object->map_list_msg_state = MAP_LIST_MSG_GOT_DATA;

	g_assert(object->map_list_msg_info != NULL);

	/***********************/
	/* store the data here */
	/***********************/
	current_rec = object->map_list_msg_info->recs_trans_count;
	msg_info = object->map_list_msg_info;

	g_ptr_array_add(msg_info->handle_list, data_info->handle);

	msg_info->subject_list[current_rec] = data_info->msg_subject;

	msg_info->date_time_list[current_rec] = data_info->msg_date_time;

	msg_info->sender_name_list[current_rec] = data_info->msg_sender_name;

	msg_info->sender_address_list[current_rec] =
						data_info->msg_sender_address;

	msg_info->replyto_address_list[current_rec] =
						data_info->msg_replyto_address;

	msg_info->recipient_name_list[current_rec] =
						data_info->msg_recipient_name;

	msg_info->recipient_address_list[current_rec] =
					data_info->msg_recipient_address;

	msg_info->type_list = g_byte_array_append(msg_info->type_list,
						&data_info->msg_type, 1);

	msg_info->size_list = g_array_append_val(msg_info->size_list,
							data_info->msg_size);

	msg_info->text_list = g_byte_array_append(msg_info->text_list,
						&data_info->msg_text_flag, 1);

	msg_info->reception_status_list =
			g_byte_array_append(msg_info->reception_status_list,
					&data_info->msg_reception_status, 1);

	msg_info->attach_size_list =
				g_array_append_val(msg_info->attach_size_list,
						data_info->msg_attach_size);

	msg_info->priority_list = g_byte_array_append(msg_info->priority_list,
						&data_info->msg_priority, 1);

	msg_info->read_list = g_byte_array_append(msg_info->read_list,
						&data_info->msg_read, 1);

	msg_info->sent_list = g_byte_array_append(msg_info->sent_list,
						&data_info->msg_sent, 1);

	msg_info->protect_list = g_byte_array_append(msg_info->protect_list,
						&data_info->msg_protect, 1);


	/* count up the number of records so far for this D-bus transaction */
	object->map_list_msg_info->recs_trans_count += 1;

	/* check whether the maximum number of records has been reached */
	if (object->map_list_msg_info->recs_trans_count <
				object->map_list_msg_info->max_recs_per_trans) {
		DEBUG("%d records pending to be sent",
				object->map_list_msg_info->recs_trans_count);
	} else {

		/* Reached maximum records so send data */
		object->map_list_msg_state = MAP_LIST_MSG_SEND_DATA;
		
		/* send the data */
		handle_map_list_msg_send_data(object);
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 *
 * This will send any pending MAP list message data
 */
gboolean handle_map_list_msg_data_complete(BD_ADDRESS device_address)
{
	DeviceObject* object = NULL;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the MAP list msg data",
			bd_addr_to_str(device_address));

		return FALSE;
	}

	DEBUG("MAP list messaging download state is %s",
			map_list_msg_state_str[object->map_list_msg_state]);

	if ((object->map_list_msg_state != MAP_LIST_MSG_GOT_DATA) &&
		(object->map_list_msg_state != MAP_LIST_MSG_WAIT_DATA)) {

		DEBUG_ERROR("Unexpected and unhandled "
			"BT_APPL_MAP_List_Msg_Comp_IND received");

		return FALSE;
	}

	/* Check the download state is valid */
	g_assert((object->map_list_msg_state == MAP_LIST_MSG_GOT_DATA) ||
			(object->map_list_msg_state == MAP_LIST_MSG_WAIT_DATA));

	/* select the got completed data state */
	object->map_list_msg_state = MAP_LIST_MSG_COMPLETE_DATA;

	g_assert(object->map_list_msg_info != NULL);

	/* send any pending MAP list message info */
	handle_map_list_msg_send_data(object);

	DEBUG_FUNC("Exited");

	return TRUE;
}

