/*
 * handle_avp_indexing.c
 * avp index handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "utils_async_queue.h"
#include "device_info.h"
#include "node_device.h"

#include "handle_avp_indexing.h"
#include "debug.h"

#ifdef DBG_ENABLED

/* AVPRCP v1.4 media player item (MPI) overlay structure */
struct MPI_offsets {
	unsigned int player_id:16;
	unsigned int major_player_type:8;
	unsigned int player_sub_type:32;
	unsigned int play_status:8;
	unsigned char feature_mask[16];
	unsigned int char_set_id:16;
	unsigned int display_name_length:16;
/*	unsigned char display_name[0]; */	/* variable length, avoid LINT */
} __attribute__((packed));

/* FIXME: need endian test */
inline guint16 swap_be16(guint16 value)
{
#if 1
	return ((value & 0xFF00) >> 8) | ((value & 0x00FF) << 8);
#else
	return value;
#endif
}

inline guint32 swap_be32(guint32 value)
{
#if 1
	return ((value & 0xFF000000) >> 24) |
		((value & 0x00FF0000) >> 8) |
		((value & 0x0000FF00) << 8) |
		((value & 0x000000FF) << 24);
#else
	return value;
#endif
}

inline guint16 swap_le16(guint16 value)
{
#if 0
	return ((value & 0xFF00) >> 8) | ((value & 0x00FF) << 8);
#else
	return value;
#endif
}

inline guint32 swap_le32(guint32 value)
{
#if 0
	return ((value & 0xFF000000) >> 24) |
		((value & 0x00FF0000) >> 8) |
		((value & 0x0000FF00) << 8) |
		((value & 0x000000FF) << 24);
#else
	return value;
#endif
}


/*
 * debug state names for AvpGetMediaPlayerItem
 */
static gchar *AvpGetMediaPlayerItem_state_str[AVP_GET_MEDIA_PLAYER_ITEM_TOTAL_STATES] =
{
	[AVP_GET_MEDIA_PLAYER_ITEM_IDLE] = "AVP_GET_MEDIA_PLAYER_ITEM_IDLE",
	[AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA] = "AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA",
	[AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY] = "AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY",
};

/*
 * debug state names for AvpGetFolderItem
 */
static gchar *AvpGetFolderItem_state_str[AVP_GET_FOLDER_ITEM_TOTAL_STATES] =
{
	[AVP_GET_FOLDER_ITEM_IDLE] = "AVP_GET_FOLDER_ITEM_IDLE",
	[AVP_GET_FOLDER_ITEM_WAIT_DATA] = "AVP_GET_FOLDER_ITEM_WAIT_DATA",
	[AVP_GET_FOLDER_ITEM_SEND_REPLY] = "AVP_GET_FOLDER_ITEM_SEND_REPLY",
};

#endif

/*
 * exported function
 *
 * get the network_order of the ALPS AVP data for decoding.
 * Note also helps external python scripts to decode the data.
 * returns
 *     TRUE for network order (big endian)
 *     FALSE for little endian
 */
gboolean handle_avp_get_network_order(void)
{
	/*
	 * ALPS v0.96 and later uses network order (big endian)
	 * this is as per the Bluetooth SIG documents
	 */
	return TRUE;
}
		
/*
 * exported function
 *
 * handle the initial BT_APPL_AVP_Get_Media_Player_Item_CFM data
 */
gboolean handle_avp_get_media_player_item_cfm(BD_ADDRESS device_address,
							guint8 status,
							guint16 num_items)
{
	DeviceObject *object = NULL;
	GetMediaPlayerItemInfo *player_info;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the avp media player item",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	DEBUG("state %u %s", object->get_media_player_item_state,
		AvpGetMediaPlayerItem_state_str[object->get_media_player_item_state]);

	if (object->get_media_player_item_state ==
					AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA) {
		
		DEBUG_ERROR("Deleting previous request that "
			"was never completed");

		g_assert(object->get_media_player_item_info != NULL);
		g_assert(object->get_media_player_item_info->data != NULL);

		/* free the accumulated AVRCP data */
		g_byte_array_unref(object->get_media_player_item_info->data);

		/* free the media player structure */
		g_free(object->get_media_player_item_info);
		object->get_media_player_item_info = NULL;

		/* set to the idle state */
		object->get_media_player_item_state =
					AVP_GET_MEDIA_PLAYER_ITEM_IDLE;
	}

	g_assert(object->get_media_player_item_state ==
					AVP_GET_MEDIA_PLAYER_ITEM_IDLE);

	g_assert(object->get_media_player_item_info == NULL);
	object->get_media_player_item_info =
				g_malloc(sizeof(GetMediaPlayerItemInfo));
	g_assert(object->get_media_player_item_info != NULL);

	/* point to media info */
	player_info = object->get_media_player_item_info;

	/* save the info for later use */
	player_info->status = status;
	player_info->num_items = num_items;
	player_info->got_items = 0;
	player_info->network_order = handle_avp_get_network_order();

	/* initialise the AVRCP data array for later use */
	player_info->data = g_byte_array_new();
	g_assert(player_info->data != NULL);

	if ((status != 0) || (num_items == 0)) {
		/*
		 * no IND is expected as there was an error or no items
		 * are available, therefore need to send empty D-bus reply
		 */
		object->get_media_player_item_state =
					AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY;

		device_object_avp_get_media_player_item_RTN(object);
	} else {
		/* wait for the expected data INDS to arrive */
		object->get_media_player_item_state =
					AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA;
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 *
 * handle the player item data from BT_APPL_AVP_Get_Media_Player_Item_IND
 */
gboolean handle_avp_get_media_player_item_ind(BD_ADDRESS device_address,
							guint8 status,
							GByteArray *data)
{
	DeviceObject *object = NULL;
	GetMediaPlayerItemInfo *player_info;
	guint8 avrcp_player_item_info[3];

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(data != NULL);

	DEBUG("data length %u", data->len);
	DEBUG("status %u", status);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the avp media player item",
			bd_addr_to_str(device_address));

		/* free the data */
		g_byte_array_unref(data);

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	DEBUG("state %u %s", object->get_media_player_item_state,
		AvpGetMediaPlayerItem_state_str[object->get_media_player_item_state]);

	g_assert(object->get_media_player_item_state ==
					AVP_GET_MEDIA_PLAYER_ITEM_WAIT_DATA);

	/* point to media info */
	player_info = object->get_media_player_item_info;

	g_assert(player_info->data != NULL);

	/* spoof the 3 missing octets of the media player item */
	avrcp_player_item_info[0] = 1;	/* Media Player Item */

	if (player_info->network_order == TRUE) {
		/* Insert length using big endian */
		avrcp_player_item_info[1] = data->len/256;	/* BE MSB of len */
		avrcp_player_item_info[2] = data->len%256;	/* BE LSB of len */
	} else {
		/* Insert length using little endian */
		avrcp_player_item_info[2] = data->len/256;	/* LE MSB of len */
		avrcp_player_item_info[1] = data->len%256;	/* LE LSB of len */
	}

	/* insert the missing 3 octets */
	player_info->data = g_byte_array_append(player_info->data,
						&avrcp_player_item_info[0],
						3);

#ifdef DBG_ENABLED
	/* make sure the data array is on a 4 byte boundary */
	g_assert((((guint32) &data->data[0]) & 0x3) == 0);

	if ((btd_debug_level >= BTD_DBG_HIGH_LEV) &&
		(data->len >= sizeof(struct MPI_offsets))) {
		struct MPI_offsets *raw_data =
			(struct MPI_offsets *)(void*) &data->data[0];

		if (player_info->network_order == TRUE) {
			/* big endian */
			DEBUG_HIGH("player_id %u", swap_be16(raw_data->player_id));
			DEBUG_HIGH("major_player_type %u", raw_data->major_player_type);
			DEBUG_HIGH("player_sub_type %u", swap_be32(raw_data->player_sub_type));
			DEBUG_HIGH("play_status %u", raw_data->play_status);
			DEBUG_HIGH("feature_mask is...");
			DEBUG_DUMP_ARRAY(16, &raw_data->feature_mask[0]);
			DEBUG_HIGH("char_set_id %u", swap_be16(raw_data->char_set_id));
			DEBUG_HIGH("display_name_length %u", swap_be16(raw_data->display_name_length));
		} else {
			/* little endian */
			DEBUG_HIGH("player_id %u", swap_le16(raw_data->player_id));
			DEBUG_HIGH("major_player_type %u", raw_data->major_player_type);
			DEBUG_HIGH("player_sub_type %u", swap_le32(raw_data->player_sub_type));
			DEBUG_HIGH("play_status %u", raw_data->play_status);
			DEBUG_HIGH("feature_mask is...");
			DEBUG_DUMP_ARRAY(16, &raw_data->feature_mask[0]);
			DEBUG_HIGH("char_set_id %u", swap_le16(raw_data->char_set_id));
			DEBUG_HIGH("display_name_length %u", swap_le16(raw_data->display_name_length));
		}
	}
#endif

	/* append the IND data to the accumulated raw AVRCP data */
	player_info->data = g_byte_array_append(player_info->data, data->data,
								data->len);

	/* free the data */
	g_byte_array_unref(data);

	/* count this IND */
	player_info->got_items++;

	/* check the number of INDs matches expectations */
	if(player_info->got_items == player_info->num_items) {
		/*
		 * no more IND is expected so send the data
		 * in the D-bus reply
		 */
		object->get_media_player_item_state =
					AVP_GET_MEDIA_PLAYER_ITEM_SEND_REPLY;

		device_object_avp_get_media_player_item_RTN(object);
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 *
 * handle the initial BT_APPL_AVP_Get_Folder_Item_CFM data
 */
gboolean handle_avp_get_folder_item_cfm(BD_ADDRESS device_address,
							guint8 status,
							guint8 scope,
							guint16 num_items)
{
	DeviceObject *object = NULL;
	GetFolderItemInfo *folder_info;

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the avp folder item",
			bd_addr_to_str(device_address));

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	DEBUG("state %u %s", object->get_folder_item_state,
		AvpGetFolderItem_state_str[object->get_folder_item_state]);

	if (object->get_folder_item_state ==
					AVP_GET_FOLDER_ITEM_WAIT_DATA) {
		
		DEBUG_ERROR("Deleting previous request that "
			"was never completed");

		g_assert(object->get_folder_item_info != NULL);
		g_assert(object->get_folder_item_info->data != NULL);

		/* free the accumulated AVRCP data */
		g_byte_array_unref(object->get_folder_item_info->data);

		/* free the folder structure */
		g_free(object->get_folder_item_info);
		object->get_folder_item_info = NULL;

		/* set to the idle state */
		object->get_folder_item_state =
					AVP_GET_FOLDER_ITEM_IDLE;
	}

	g_assert(object->get_folder_item_state == AVP_GET_FOLDER_ITEM_IDLE);

	g_assert(object->get_folder_item_info == NULL);
	object->get_folder_item_info =
				g_malloc(sizeof(GetFolderItemInfo));
	g_assert(object->get_folder_item_info != NULL);

	/* point to folder info */
	folder_info = object->get_folder_item_info;

	/* save the info for later use */
	folder_info->status = status;
	folder_info->scope = scope;
	folder_info->num_items = num_items;
	folder_info->got_items = 0;
	folder_info->network_order = handle_avp_get_network_order();

	/* initialise the AVRCP data array for later use */
	folder_info->data = g_byte_array_new();
	g_assert(folder_info->data != NULL);

	if ((status != 0) || (num_items == 0)) {
		/*
		 * no IND is expected as there was an error or no items
		 * are available, therefore need to send empty D-bus reply
		 */
		object->get_folder_item_state =
					AVP_GET_FOLDER_ITEM_SEND_REPLY;

		device_object_avp_get_folder_item_RTN(object);
	} else {
		/* wait for the expected data INDS to arrive */
		object->get_folder_item_state =
					AVP_GET_FOLDER_ITEM_WAIT_DATA;
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

/*
 * exported function
 *
 * handle the folder info from BT_APPL_AVP_Get_Folder_Item_IND
 * handle the element info BT_APPL_AVP_Media_Element_IND
 */
gboolean handle_avp_get_folder_and_element_item_ind(BD_ADDRESS device_address,
							guint8 status,
							guint8 scope,
							GByteArray *data,
							guint8 item_type)
{
	DeviceObject *object = NULL;
	GetFolderItemInfo *folder_info;
	guint8 avrcp_folder_item_info[3];

	DEBUG_FUNC("Called");

	g_assert(device_address != NULL);
	g_assert(data != NULL);

	DEBUG("data length %u", data->len);
	DEBUG("status %u", status);
	DEBUG("scope %u", scope);

	/* find the device object relating to the device address */
	object = find_device_object_from_list(device_address);

	if (object == NULL) {
		DEBUG_ERROR("Bluetooth device address: %s not found so "
			"cannot process the avp folder item",
			bd_addr_to_str(device_address));

		/* free the data */
		g_byte_array_unref(data);

		DEBUG_FUNC("Exited");
		return FALSE;
	}

	DEBUG("state %u %s", object->get_folder_item_state,
		AvpGetFolderItem_state_str[object->get_folder_item_state]);

	g_assert(object->get_folder_item_state ==
					AVP_GET_FOLDER_ITEM_WAIT_DATA);

	/* point to folder info */
	folder_info = object->get_folder_item_info;

	g_assert(folder_info->data != NULL);

	/* spoof the 3 missing octets of the folder or element item */
	avrcp_folder_item_info[0] = item_type;	/* Folder or element Item */
	if (folder_info->network_order == TRUE) {
		/* Insert length using big endian */
		avrcp_folder_item_info[1] = data->len/256;	/* BE MSB of len */
		avrcp_folder_item_info[2] = data->len%256;	/* BE LSB of len */
	} else {
		/* Insert length using little endian */
		avrcp_folder_item_info[2] = data->len/256;	/* LE MSB of len */
		avrcp_folder_item_info[1] = data->len%256;	/* LE LSB of len */
	}

	/* insert the missing 3 octets */
	folder_info->data = g_byte_array_append(folder_info->data,
						&avrcp_folder_item_info[0],
						3);

	/* append the IND data to the accumulated raw AVRCP data */
	folder_info->data = g_byte_array_append(folder_info->data, data->data,
								data->len);

	/* free the data */
	g_byte_array_unref(data);

	/* count this IND */
	folder_info->got_items++;

	/* check the number of INDs matches expectations */
	if(folder_info->got_items == folder_info->num_items) {
		/*
		 * no more IND is expected so send the data
		 * in the D-bus reply
		 */
		object->get_folder_item_state =
					AVP_GET_FOLDER_ITEM_SEND_REPLY;

		device_object_avp_get_folder_item_RTN(object);
	}

	DEBUG_FUNC("Exited");

	return TRUE;
}

