/*
 * dbus error handling for the D-bus Bluetooth Daemon
 *
 * Author: Dean Jenkins <djenkins@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "bt_appl/common_def.h"

#include "main.h"
#include "debug.h"
#include "dbus_error.h"

gchar *lookup_bthresult_str(BTHRESULT result)
{
	gchar *str;

	switch (result) {
	case BT_APPL_SUCCESS:
		str = "BT_APPL_SUCCESS";
		break;

	case BT_APPL_ERROR_PARAMETERS:
		str = "BT_APPL_ERROR_PARAMETERS";
		break;

	case BT_APPL_ERROR_OPERATION_FAILED:
		str = "BT_APPL_ERROR_OPERATION_FAILED";
		break;

	case BT_APPL_ERROR_UNINITIALIZED:
		str = "BT_APPL_ERROR_UNINITIALIZED";
		break;

	case BT_APPL_ERROR_CALLBACK_FUN_NOT_REGISTERED:
		str = "BT_APPL_ERROR_CALLBACK_FUN_NOT_REGISTERED";
		break;

	case BT_APPL_ERROR_INTERFACE_NOT_AVAILABLE:
		str = "BT_APPL_ERROR_INTERFACE_NOT_AVAILABLE";
		break;

	case BT_APPL_ERROR_PAIRING_ALREADY_STARTED:
		str = "BT_APPL_ERROR_PAIRING_ALREADY_STARTED";
		break;

	case BT_APPL_ERROR_NO_PAIRING_STARTED:
		str = "BT_APPL_ERROR_NO_PAIRING_STARTED";
		break;

	case BT_APPL_ERROR_STATE_ERROR:
		str = "BT_APPL_ERROR_STATE_ERROR";
		break;

	case BT_APPL_ERROR_NO_CORRESPONDING_IND:
		str = "BT_APPL_ERROR_NO_CORRESPONDING_IND";
		break;

	case BT_APPL_ERROR_MAX_CONNECTION_EXISTED:
		str = "BT_APPL_ERROR_MAX_CONNECTION_EXISTED";
		break;

	case BT_APPL_ERROR_CONNECTION_ALREADY_WAIT:
		str = "BT_APPL_ERROR_CONNECTION_ALREADY_WAIT";
		break;

	case BT_APPL_ERROR_NO_CONNECTION_WAIT:
		str = "BT_APPL_ERROR_NO_CONNECTION_WAIT";
		break;

	case BT_APPL_ERROR_CONNECTION_ALREADY_EXISTED:
		str = "BT_APPL_ERROR_CONNECTION_ALREADY_EXISTED";
		break;

	case BT_APPL_ERROR_CONNECTION_NOT_EXIST:
		str = "BT_APPL_ERROR_CONNECTION_NOT_EXIST";
		break;

	case BT_APPL_ERROR_HFP_NOT_CONNECTED:
		str = "BT_APPL_ERROR_HFP_NOT_CONNECTED";
		break;

	case BT_APPL_ERROR_NOT_SUPPORTED:
		str = "BT_APPL_ERROR_NOT_SUPPORTED";
		break;

	case BT_APPL_ERROR_NOT_PREPARED:
		str = "BT_APPL_ERROR_NOT_PREPARED";
		break;

	case BT_APPL_ERROR_MAP_UNKNOWN_CAPABILITY:
		str = "BT_APPL_ERROR_MAP_UNKNOWN_CAPABILITY";
		break;

	case BT_APPL_ERROR_MAP_ALREADY_STARTED:
		str = "BT_APPL_ERROR_SMS_ALREADY_STARTED";
		break;

	case BT_APPL_ERROR_MAP_NOT_STARTED:
		str = "BT_APPL_ERROR_SMS_NOT_STARTED";
		break;

	case BT_APPL_ERROR_MAP_UNKNOWN_SERVICE:
		str = "BT_APPL_ERROR_SMS_UNKNOWN_SERVICE";
		break;

	case BT_APPL_ERROR_MAP_SERVICE_UNACTIVE:
		str = "BT_APPL_ERROR_SMS_SERVICE_UNACTIVE";
		break;

	case BT_APPL_ERROR_MAP_USELESS_DATAFLAG:
		str = "BT_APPL_ERROR_MAP_USELESS_DATAFLAG";
		break;

	case BT_APPL_ERROR_DEVICE_ALREADY_CREATED:
		str = "BT_APPL_ERROR_DEVICE_ALREADY_CREATED";
		break;

	case BT_APPL_ERROR_DEVICE_NOT_EXISTED:
		str = "BT_APPL_ERROR_DEVICE_NOT_EXISTED";
		break;

	case BT_APPL_ERROR_AVP_CUR_VERSION_NOT_SUPPORTED:
		str = "BT_APPL_ERROR_AVP_CUR_VERSION_NOT_SUPPORTED";
		break;

	case BT_APPL_ERROR_AVP_SUSPEND_HAS_EXIST:
		str = "BT_APPL_ERROR_AVP_SUSPEND_HAS_EXIST";
		break;

	case BT_APPL_ERROR_AVP_RESUME_HAS_EXIST:
		str = "BT_APPL_ERROR_AVP_RESUME_HAS_EXIST";
		break;

	case BT_APPL_ERROR_ALREADY_PREPARED:
		str = "BT_APPL_ERROR_ALREADY_PREPARED";
		break;

	case BT_APPL_ERROR_3GPP_ON_USING:
		str = "BT_APPL_ERROR_3GPP_ON_USING";
		break;

	case BT_APPL_ERROR_NOT_GETCAPABILITY:
		str = "BT_APPL_ERROR_NOT_GETCAPABILITY";
		break;

	case BT_APPL_ERROR_SERIAL_DEV_PARAM_NOT_EXIST:
		str = "BT_APPL_ERROR_SERIAL_DEV_PARAM_NOT_EXIST";
		break;

	case BT_APPL_ERROR_UNKNOWN_SERVICE:
		str = "BT_APPL_ERROR_UNKNOWN_SERVICE";
		break;

	case BT_APPL_ERROR_STATE_ERROR_NO_RETRY:
		str = "BT_APPL_ERROR_STATE_ERROR_NO_RETRY";
		break;

	case BT_APPL_ERROR_STATE_ERROR_TEST_MODE:
		str = "BT_APPL_ERROR_STATE_ERROR_TEST_MODE";
		break;

	case BT_APPL_ERROR_STATE_ERROR_RETRY:
		str = "BT_APPL_ERROR_STATE_ERROR_RETRY";
		break;

	case BT_APPL_ERROR_STATE_ERROR_HFP_BUSY:
		str = "BT_APPL_ERROR_STATE_ERROR_HFP_BUSY";
		break;

	case BT_APPL_ERROR_SPP_ON_USING:
		str = "BT_APPL_ERROR_SPP_ON_USING";
		break;

	case BT_APPL_ERROR_PBDL_PREPARING:
		str  = "BT_APPL_ERROR_PBDL_PREPARING";
		break;

	case BT_APPL_ERROR_DISCONNECT_ALL_IS_ONGOING:
		str  = "BT_APPL_ERROR_DISCONNECT_ALL_IS_ONGOING";
		break;

	case BT_APPL_ERROR_PSKEY_FAILURE:
		str  = "BT_APPL_ERROR_PSKEY_FAILURE";
		break;

	case BT_APPL_ERROR_SHUTDOWN_IS_ONGOING:
		str  = "BT_APPL_ERROR_SHUTDOWN_IS_ONGOING";
		break;

	case BT_APPL_ERROR_INQUIRY_ALREADY_EXIST:
		str = "BT_APPL_ERROR_INQUIRY_ALREADY_EXIST";
		break;

	case BT_APPL_ERROR_INQUIRY_NOT_EXIST:
		str = "BT_APPL_ERROR_INQUIRY_NOT_EXIST";
		break;

	case BT_APPL_ERROR_INQUIRY_CANCEL_ON:
		str = "BT_APPL_ERROR_INQUIRY_CANCEL_ON";
		break;

	case BT_APPL_ERROR_INITIATE_PAIRING_ALREADY_STARTED:
		str = "BT_APPL_ERROR_INITIATE_PAIRING_ALREADY_STARTED";
		break;

	case BT_APPL_ERROR_PAIRING_NOT_EXIST:
		str = "BT_APPL_ERROR_PAIRING_NOT_EXIST";
		break;

	case BT_APPL_ERROR_PAIRING_CANCELING:
		str = "BT_APPL_ERROR_PAIRING_CANCELING";
		break;

	case DAEMON_TIMEOUT_EXPIRED:
		str = "DAEMON_TIMEOUT_EXPIRED";
		break;
 
	default:
		str = "UNKNOWN_BT_APPL_CODE";
		break;
	}

	return str;
}


gboolean report_any_error_to_dbus(BTHRESULT result, gchar *dbus_msg, GError **error)
{
	gboolean return_code = TRUE;
	gchar *error_str = lookup_bthresult_str(result);

	g_assert(dbus_msg != NULL);
	g_assert(error != NULL);
	g_assert(*error == NULL);

	DEBUG_FUNC("Called");

	DEBUG_HIGH("result %d", result);

	if (result != BT_APPL_SUCCESS) {
		if(result == DAEMON_TIMEOUT_EXPIRED) {
			/*
			 * Keep track of daemon timeouts by writing
			 * into the IVI error memory via STDERR
			 */
			DEBUG_ERROR("*** %s returned error %d %s ***",
					dbus_msg, result, error_str);
		} else {
			/*
			 * use STDOUT to avoid writing the error code
			 * to the IVI error memory
			 */
			DEBUG_WARN("*** %s returned error %d %s ***",
					dbus_msg, result, error_str);
		}

		g_set_error(error,
			g_quark_from_string("DbusBtDaemon"),
			result,
			"ERROR:%d:\'%s\'\"%s\"", result, error_str, dbus_msg);

		return_code = FALSE;
	}

	DEBUG_FUNC("Exited");

	return return_code;
}
