/*
 * callbacks.c
 * Handle callbacks from the ALPS Bluetooth stack.
 *
 * Author: Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 */

#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <glib-object.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "main.h"

#include "bt_appl/common_def.h"
#include "bt_appl/api/bt_appl_bt_setting_block.h"

#include "callbacks.h"
#include "dbus_error.h"
#include "node_agent.h"
#include "utils_async_queue.h"
#include "utils_test_queue.h"
#include "device_info.h"
#include "node_device.h"
#include "node_adapter.h"
#include "node_test.h"
#include "node_validation.h"
#include "handle_phonebook.h"
#include "handle_avp_indexing.h"
#include "handle_spp_service.h"
#include "handle_pnp_info.h"
#include "handle_messaging.h"

#include "debug.h"

#ifdef DBG_ENABLED
static gchar handle_str[48];

static gchar *handle_to_str(const guint8 *handle, guint8 len)
{
	if (len > 16)
		len = 16;

	handle_str[0] = 0;
	if (len && handle) {
		gint i, j;

		for (i = 0, j=0; i < len; i++) {
			if (i)
				j += g_sprintf(handle_str + j, ":");
			j += g_sprintf(handle_str + j, "%02x", handle[i]);
		}
	}

	return handle_str;
}

static gchar uuid_str[16*2+1];

static gchar *dump_spp_uuid(ClassIDInfoPtr_t uuid_ptr)
{
	guint8 len;
	guint8 uuid_pos;
	guint8 str_pos = 0;

	uuid_str[0] = 0;

	if (uuid_ptr == NULL) {
		return &uuid_str[0];
	}

	len = uuid_ptr->ucUUIDLen;

	if (len > 16) {
		DEBUG("len %u", len);

		len = 16;
		DEBUG("WARNING: force len to be truncatated to 16");
	}

	for (uuid_pos = 0; uuid_pos < len; uuid_pos++) {
		str_pos += g_sprintf(&uuid_str[str_pos], "%02X",
				     uuid_ptr->aucUUID[uuid_pos]);
	}

	return &uuid_str[0];
}
#endif

static GAsyncQueue *callback_queue;
static int pipe_fd[2];
static GIOChannel *pipe_ch;
	/* the GSource id for the pipe */
static guint pipe_source;
	/* the GSource id for the ConnectionComplete timer */
static guint timeout_source;

/***********/
/* GENERAL */
/***********/

/* GetDeviceName */
static void do_get_device_name_cfm_wrapper(callback_data *data)
{
	gen_get_device_name_cfm_t *parms =
				&data->parms.gen_get_device_name_cfm_parms;

	g_assert(data->id == BT_APPL_GET_DEVICE_NAME_CFM_CB);

	device_object_get_device_name_RTN(parms->bd_addr, parms->device_name);
}

/* RoleChange */
static void do_role_change_ind_wrapper(callback_data *data)
{
	gen_role_change_ind_t *parms = &data->parms.gen_role_change_ind_parms;

	g_assert(data->id == BT_APPL_ROLE_CHANGE_IND_CB);

	emit_role_change(parms->bd_addr, parms->status, parms->role);
}

/* ServiceType */
static void do_service_type_ind_wrapper(callback_data *data)
{
	gen_service_type_ind_t *parms = &data->parms.gen_service_type_ind_parms;

	g_assert(data->id == BT_APPL_SERVICE_TYPE_IND_CB);

	emit_service_type(parms->bd_addr, parms->service_type);
}

/* ConnectionComplete */
static void do_connection_complete_ind_wrapper(callback_data *data)
{
	gen_connection_complete_ind_t *parms =
				&data->parms.gen_connection_complete_ind_parms;

	g_assert(data->id == BT_APPL_CONNECTION_COMPLETE_IND_CB);

	if (parms->status == FALSE) {
		/*
		 * We had a successful connection to this device, so
		 * create a DBus device node for it if we don't already
		 * have one.
		 */
		adapter_add_device_to_list(parms->bd_addr);
	}

	emit_connection_complete(parms->bd_addr,
					parms->status,
					parms->service);
}

/* Disconnection */
static void do_disconnection_ind_wrapper(callback_data *data)
{
	gen_disconnection_ind_t *parms =
				&data->parms.gen_disconnection_ind_parms;

	g_assert(data->id == BT_APPL_DISCONNECTION_IND_CB);

	emit_disconnection(parms->bd_addr,
				parms->service,
				parms->conn_loss_type);
}

/* ValidationTestMode */
static void do_testmode_cfm_wrapper(callback_data *data)
{
	gen_testmode_cfm_t *parms = &data->parms.gen_testmode_cfm_parms;

	g_assert(data->id == BT_APPL_TESTMODE_CFM_CB);

	validation_object_validation_test_mode_RTN(parms->status);
}

/* ValidationTestModeComplete */
static void do_testmode_complete_ind_wrapper(callback_data *data)
{
	gen_testmode_complete_ind_t *parms =
				&data->parms.gen_testmode_complete_ind_parms;

	g_assert(data->id == BT_APPL_TESTMODE_COMPLETE_IND_CB);

	emit_validation_test_mode_complete(parms->noti_type, parms->status);
}

/* ValidationLinkQuality */
static void do_linkqualify_cfm_wrapper(callback_data *data)
{
	gen_linkqualify_cfm_t *parms =
				&data->parms.gen_linkqualify_cfm_parms;

	g_assert(data->id == BT_APPL_LINKQUALITY_CFM_CB);

	validation_object_validation_link_quality_RTN(parms->status,
							parms->rssi,
							parms->link_quality);
}

/* SspNumericConfirmInd */
static void do_ssp_numeric_confirm_ind_wrapper(callback_data *data)
{
	gen_ssp_numeric_confirm_ind_t *parms =
				&data->parms.gen_ssp_numeric_confirm_ind_parms;

	g_assert(data->id == BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB);

	if (read_dbus_api_options() & DBUS_USE_ALT_SSP_NUMERIC_CONFIRM) {
		/* Use alternative SSP Numeric Confirm */
		emit_ssp_numeric_confirm_ind(parms->bd_addr,
						parms->name,
						parms->numeric_value);
	} else {
		/* Use the original SSP Numeric Confirm agent method */
		do_agent_ssp_numeric_confirm_ind_wrapper(data);
	}
}

/* ConnectionRequestInd */
static void do_connection_request_ind_wrapper(callback_data *data)
{
	gen_connection_request_ind_t *parms =
				&data->parms.gen_connection_request_ind_parms;

	g_assert(data->id == BT_APPL_CONNECTION_REQUEST_IND_CB);

	if (read_dbus_api_options() & DBUS_USE_ALT_CONNECTION_REQUEST) {
		/* Use alternative Connection Request */
		emit_connection_request_ind(parms->bd_addr,
						parms->service);
	} else {
		/* Use the original Connection Request agent method */
		do_agent_connection_request_ind_wrapper(data);
	}
}


/* InquiryResultInd */
static void do_inquiry_ind_wrapper(callback_data *data)
{
	DEBUG("Inquiry indication wrapper is entered");
	gen_inquiry_ind_t *parms = &data->parms.gen_inquiry_ind_parms;
	g_assert(data->id == BT_APPL_INQUIRY_IND_CB);
	emit_inquiry_ind(parms->bd_addr,
				parms->page_scan_rep_mode,
				parms->cod.major_device_class,
				parms->cod.minor_device_class,
				(guint32) parms->cod.major_service_class,
				(guint32) parms->clock_offset,
				(gint32) parms->rssi,
				parms->eir_data);
}


/* InquiryCompleteInd */
static void do_inquiry_complete_ind_wrapper(callback_data *data)
{
	gen_inquiry_complete_ind_t *parms = &data->parms.gen_inquiry_complete_ind_parms;
	g_assert(data->id == BT_APPL_INQUIRY_COMPLETE_IND_CB);
	emit_inquiry_complete_ind(parms->inquiry_status);
}

/******/
/* HF */
/******/

/* CallStatus */
static void do_hf_call_status_ind_wrapper(callback_data *data)
{
	hf_call_status_ind_t *parms = &data->parms.hf_call_status_ind_parms;

	g_assert(data->id == BT_APPL_CALL_STATUS_IND_CB);

	emit_call_status(parms->bd_addr,
				parms->call_handle,
				parms->total,
				parms->direction,
				parms->state,
				parms->mode,
				parms->number,
				parms->type);
}

/* PhoneStatus */
static void do_hf_phone_status_ind_wrapper(callback_data *data)
{
	hf_phone_status_ind_t *parms = &data->parms.hf_phone_status_ind_parms;

	g_assert(data->id == BT_APPL_PHONE_STATUS_IND_CB);

	emit_phone_status(parms->bd_addr,
				parms->service,
				parms->signal,
				parms->roaming,
				parms->battery,
				parms->operator_name,
				parms->ag_supp_feature,
				parms->AgThreeWayCallSuppFeatures);
}

/* HfDial */
static void do_hf_dial_cfm_wrapper(callback_data *data)
{
	hf_dial_cfm_t *parms = &data->parms.hf_dial_cfm_parms;

	g_assert(data->id == BT_APPL_HF_DIAL_CFM_CB);

	device_object_hf_dial_RTN(parms->bd_addr, parms->status, parms->code);
}

/* HfRedial */
static void do_hf_redial_cfm_wrapper(callback_data *data)
{
	hf_redial_cfm_t *parms = &data->parms.hf_redial_cfm_parms;

	g_assert(data->id == BT_APPL_HF_REDIAL_CFM_CB);

	device_object_hf_redial_RTN(parms->bd_addr, parms->status, parms->code);
}

/* HfMemDial */
static void do_hf_mem_dial_cfm_wrapper(callback_data *data)
{
	hf_mem_dial_cfm_t *parms = &data->parms.hf_mem_dial_cfm_parms;

	g_assert(data->id == BT_APPL_HF_MEM_DIAL_CFM_CB);

	device_object_hf_mem_dial_RTN(parms->bd_addr,
					parms->status,
					parms->code);
}

/* HfTerminateCall */
static void do_hf_terminate_call_cfm_wrapper(callback_data *data)
{
	hf_terminate_call_cfm_t *parms =
				&data->parms.hf_terminate_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_TERMINATE_CALL_CFM_CB);

	device_object_hf_terminate_call_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfTerminateAll */
static void do_hf_terminate_all_cfm_wrapper(callback_data *data)
{
	hf_terminate_all_cfm_t *parms =
				&data->parms.hf_terminate_all_cfm_parms;

	g_assert(data->id == BT_APPL_HF_TERMINATE_ALL_CFM_CB);

	device_object_hf_terminate_all_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfAudioTransfer */
static void do_hf_audio_transfer_cfm_wrapper(callback_data *data)
{
	hf_audio_transfer_cfm_t *parms =
				&data->parms.hf_audio_transfer_cfm_parms;

	g_assert(data->id == BT_APPL_HF_AUDIO_TRANSFER_CFM_CB);

	device_object_hf_audio_transfer_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfSetupAudio */
static void do_hf_setup_audio_ind_wrapper(callback_data *data)
{
	hf_setup_audio_ind_t *parms = &data->parms.hf_setup_audio_ind_parms;

	g_assert(data->id == BT_APPL_HF_SETUP_AUDIO_IND_CB);

	emit_hf_setup_audio(parms->bd_addr);
}

/* HfReleaseAudio */
static void do_hf_release_audio_ind_wrapper(callback_data *data)
{
	hf_release_audio_ind_t *parms = &data->parms.hf_release_audio_ind_parms;

	g_assert(data->id == BT_APPL_HF_RELEASE_AUDIO_IND_CB);

	emit_hf_release_audio(parms->bd_addr);
}

/* HfSendDtmf */
static void do_hf_send_dtmf_cfm_wrapper(callback_data *data)
{
	hf_send_dtmf_cfm_t *parms = &data->parms.hf_send_dtmf_cfm_parms;

	g_assert(data->id == BT_APPL_HF_SEND_DTMF_CFM_CB);

	device_object_hf_send_dtmf_RTN(parms->bd_addr,
					parms->status,
					parms->code);
}

/* HfIncomingCallAct */
static void do_hf_incoming_call_act_cfm_wrapper(callback_data *data)
{
	hf_incoming_call_act_cfm_t *parms =
				&data->parms.hf_incoming_call_act_cfm_parms;

	g_assert(data->id == BT_APPL_HF_INCOMING_CALL_ACT_CFM_CB);

	device_object_hf_incoming_call_act_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfRing */
static void do_hf_ring_ind_wrapper(callback_data *data)
{
	hf_ring_ind_t *parms = &data->parms.hf_ring_ind_parms;

	g_assert(data->id == BT_APPL_HF_RING_IND_CB);

	emit_hf_ring(parms->bd_addr, parms->ring_tone);
}

/* HfReleaseInactiveCall */
static void do_hf_release_inactive_call_cfm_wrapper(callback_data *data)
{
	hf_release_inactive_call_cfm_t *parms =
				&data->parms.hf_release_inactive_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_RELEASE_INACTIVE_CALL_CFM_CB);

	device_object_hf_release_inactive_call_RTN(parms->bd_addr,
							parms->status,
							parms->code);
}

/* HfReleaseActiveCall */
static void do_hf_release_active_call_cfm_wrapper(callback_data *data)
{
	hf_release_active_call_cfm_t *parms =
				&data->parms.hf_release_active_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_RELEASE_ACTIVE_CALL_CFM_CB);

	device_object_hf_release_active_call_RTN(parms->bd_addr,
							parms->status,
							parms->code);
}


/* HfSwapCall */
static void do_hf_swap_call_cfm_wrapper(callback_data *data)
{
	hf_swap_call_cfm_t *parms = &data->parms.hf_swap_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_SWAP_CALL_CFM_CB);

	device_object_hf_swap_call_RTN(parms->bd_addr, 
					parms->status,
					parms->code);
}

/* HfAddCall */
static void do_hf_add_call_cfm_wrapper(callback_data *data)
{
	hf_add_call_cfm_t *parms = &data->parms.hf_add_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_ADD_CALL_CFM_CB);

	device_object_hf_add_call_RTN(parms->bd_addr,
					parms->status,
					parms->code);
}

/* HfReleaseSpecCall */
static void do_hf_release_spec_call_cfm_wrapper(callback_data *data)
{
	hf_release_spec_call_cfm_t *parms =
				&data->parms.hf_release_spec_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_RELEASE_SPEC_CALL_CFM_CB);

	device_object_hf_release_spec_call_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfHoldExcepCall */
static void do_hf_hold_excep_call_cfm_wrapper(callback_data *data)
{
	hf_hold_excep_call_cfm_t *parms =
				&data->parms.hf_hold_excep_call_cfm_parms;

	g_assert(data->id == BT_APPL_HF_HOLD_EXCEP_CALL_CFM_CB);

	device_object_hf_hold_excep_call_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfExpCallTrans */
static void do_hf_exp_call_trans_cfm_wrapper(callback_data *data)
{
	hf_exp_call_trans_cfm_t *parms =
				&data->parms.hf_exp_call_trans_cfm_parms;

	g_assert(data->id == BT_APPL_HF_EXP_CALL_TRANS_CFM_CB);

	device_object_hf_exp_call_trans_RTN(parms->bd_addr,
						parms->status,
						parms->code);
}

/* HfSubscriber */
static void do_hf_subscriber_cfm_wrapper(callback_data *data)
{
	hf_subscriber_cfm_t *parms = &data->parms.hf_subscriber_cfm_parms;

	g_assert(data->id == BT_APPL_HF_SUBSCRIBER_CFM_CB);

	device_object_hf_subscriber_RTN(parms->bd_addr,
						parms->status,
						parms->code,
						parms->subscriber_info);
}

/* HfManufacturerId */
static void do_hf_manufacturer_id_cfm_wrapper(callback_data *data)
{
	hf_manufacturer_id_cfm_t *parms =
				&data->parms.hf_manufacturer_id_cfm_parms;

	g_assert(data->id == BT_APPL_HF_MANUFACTURER_ID_CFM_CB);

	device_object_hf_manufacturer_id_RTN(parms->bd_addr,
						parms->status,
						parms->code,
						parms->manufacturer);
}

/* HfModelId */
static void do_hf_model_id_cfm_wrapper(callback_data *data)
{
	hf_model_id_cfm_t *parms = &data->parms.hf_model_id_cfm_parms;

	g_assert(data->id == BT_APPL_HF_MODEL_ID_CFM_CB);

	device_object_hf_model_id_RTN(parms->bd_addr,
					parms->status,
					parms->code,
					parms->model);
}

/* HfRevisionId */
static void do_hf_revision_id_cfm_wrapper(callback_data *data)
{
	hf_revision_id_cfm_t *parms = &data->parms.hf_revision_id_cfm_parms;

	g_assert(data->id == BT_APPL_HF_REVISION_ID_CFM_CB);

	device_object_hf_revision_id_RTN(parms->bd_addr,
						parms->status,
						parms->code,
						parms->revision);
}

/* HfVoiceRecognitionActivation */
static void do_hf_voice_recognition_activation_cfm_wrapper(callback_data *data)
{
	hf_voice_recognition_activation_cfm_t *parms =
			 &data->parms.hf_voice_recognition_activation_cfm_parms;

	g_assert(data->id == BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_CFM_CB);

	device_object_hf_voice_recognition_activation_RTN(parms->bd_addr,
						parms->activation,
						parms->status);
}

/* HfVoiceRecognitionActivationInd */
static void do_hf_voice_recognition_activation_ind_wrapper(callback_data *data)
{
	hf_voice_recognition_activation_ind_t *parms =
			&data->parms.hf_voice_recognition_activation_ind_parms;

	g_assert(data->id == BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_IND_CB);

	emit_hf_voice_recognition_activation_ind(parms->bd_addr,
							parms->activation);
}

/* applsirixaplcfm */
static void do_hf_siri_xapl_cfm_wrapper(callback_data *data)
{
	DEBUG("do_hf_siri_xapl_cfm_wrapper");

	appl_xapl_siri_cfm_t *parms =
			 &data->parms.appl_xapl_siri_cfm_params;

	g_assert(data->id == BT_APPL_HF_BT_APPL_SIRI_XAPL_CFM_CB);

	device_object_appl_siri_xapl_RTN(parms->bd_addr,
						parms->status,
						parms->parameter);
}

/* applsirixaplInd */
static void do_hf_siri_aplsiri_ind_wrapper(callback_data *data)
{
	DEBUG("do_hf_siri_aplsiri_ind_wrapper");

	appl_xapl_siri_ind_t *parms =
			&data->parms.appl_xapl_siri_ind_params;

	g_assert(data->id == BT_APPL_HF_BT_APPL_SIRI_XAPL_IND_CB);

	emit_siri_aplsiri_ind(parms->bd_addr,
					parms->status);
}

/* sirinrstatcfm */
static void do_hf_siri_nrstat_cfm_wrapper(callback_data *data)
{
	DEBUG("do_hf_siri_nrstat_cfm_wrapper");

	siri_nrstat_cfm_t *parms =
			 &data->parms.siri_nrstat_cfm_params;

	g_assert(data->id == BT_APPL_HF_BT_APPL_SIRI_NrStat_CFM_CB);

	device_object_siri_nrstat_RTN(parms->bd_addr,
						parms->status);
}


/* sirinrstatInd */
static void do_hf_siri_nrstat_ind_wrapper(callback_data *data)
{
	DEBUG("do_hf_siri_nrstat_ind_wrapper");

	siri_nrstat_ind_t *parms =
			&data->parms.siri_nrstat_ind_params;

	g_assert(data->id == BT_APPL_HF_BT_APPL_SIRI_NrStat_IND_CB);

	emit_siri_nrstat_ind(parms->bd_addr,
					parms->ucAplNrStat);
}


/* siriefmcfm */
static void do_hf_siri_efm_cfm_wrapper(callback_data *data)
{
	DEBUG("do_hf_siri_efm_cfm_wrapper");

	siri_efm_cfm_t *parms =
			 &data->parms.siri_efm_cfm_params;

	g_assert(data->id == BT_APPL_HF_BT_APPL_SIRI_EFM_CFM_CB);

	device_object_siri_efm_RTN(parms->bd_addr,
						parms->status);
}

/* HfReleaseWaitingCall */
static void do_hf_release_waiting_call_cfm_wrapper(callback_data *data)
{
	/* only Registered with but not used right now*/
	DEBUG("Only registered not used right now");
	(void) data;
}

// HfSetupAudioWbsInd
static void do_hf_setup_audio_wbs_ind_wrapper(callback_data *data)
{
	hf_setup_audio_wbs_ind_t *parms =
			&data->parms.hf_setup_audio_wbs_ind_parms;
	g_assert(data->id == BT_APPL_HF_SETUP_AUDIO_WBS_IND_CB);
	emit_hf_setup_audio_wbs_ind(parms->bd_addr,parms->audiocodec);
}


/*******/
/* AVP */
/*******/

/* AvpVersion */
static void do_avp_version_ind_wrapper(callback_data *data)
{
	avp_version_ind_t *parms = &data->parms.avp_version_ind_parms;

	g_assert(data->id == BT_APPL_AVP_VERSION_IND_CB);

	emit_avp_version(parms->bd_addr, parms->avrcp_version);
}

/* AvpCtrlCmd */
static void do_avp_ctrl_cmd_cfm_wrapper(callback_data *data)
{
	avp_ctrl_cmd_cfm_t *parms = &data->parms.avp_ctrl_cmd_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_CTRL_CMD_CFM_CB);

	device_object_avp_ctrl_cmd_RTN(parms->bd_addr,
					parms->command,
					parms->status_flag,
					parms->status);
}

/* AvpCtrlCmdInd */
static void do_avp_ctrl_cmd_ind_wrapper(callback_data *data)
{
	avp_ctrl_cmd_ind_t *parms = &data->parms.avp_ctrl_cmd_ind_parms;

	g_assert(data->id == BT_APPL_AVP_CTRL_CMD_IND_CB);

	emit_avp_ctrl_cmd_ind(parms->bd_addr, parms->command);
}

/* AvpGetCapabilities */
static void do_avp_get_capabilities_cfm_wrapper(callback_data *data)
{
	avp_get_capabilities_cfm_t *parms =
				&data->parms.avp_get_capabilities_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_GET_CAPABILITIES_CFM_CB);

	device_object_avp_get_capabilities_RTN(parms->bd_addr,
					parms->status,
					parms->capability_id,
					(guint32) parms->event_supported);
}

/* AvpListAttr */
static void do_avp_list_attr_cfm_wrapper(callback_data *data)
{
	avp_list_attr_cfm_t *parms = &data->parms.avp_list_attr_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_LIST_ATTR_CFM_CB);

	device_object_avp_list_attr_RTN(parms->bd_addr,
					parms->status,
					parms->attributes_bit);
}

/* AvpListVal */
static void do_avp_list_val_cfm_wrapper(callback_data *data)
{
	avp_list_val_cfm_t *parms = &data->parms.avp_list_val_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_LIST_VAL_CFM_CB);

	device_object_avp_list_val_RTN(parms->bd_addr,
					parms->status,
					parms->setting_values,
					parms->current_value);
}

/* AvpSetVal */
static void do_avp_set_val_cfm_wrapper(callback_data *data)
{
	avp_set_val_cfm_t *parms = &data->parms.avp_set_val_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_SET_VAL_CFM_CB);

	device_object_avp_set_val_RTN(parms->bd_addr, parms->status);
}

/* AvpPlayStatus */
static void do_avp_play_status_ind_wrapper(callback_data *data)
{
	avp_play_status_ind_t *parms = &data->parms.avp_play_status_ind_parms;

	g_assert(data->id == BT_APPL_AVP_PLAY_STATUS_IND_CB);

	emit_avp_play_status(parms->bd_addr,
				parms->status,
				parms->song_length,
				parms->song_position,
				parms->play_status);
}

/* AvpNotifySetComplete */
static void do_avp_notify_set_complete_ind_wrapper(callback_data *data)
{
	avp_notify_set_complete_ind_t *parms =
			&data->parms.avp_notify_set_complete_ind_parms;

	g_assert(data->id == BT_APPL_AVP_NOTIFY_SET_COMPLETE_IND_CB);

	emit_avp_notify_set_complete(parms->bd_addr, parms->event_id);
}

/* AvpNotifyChg */
static void do_avp_notify_chg_ind_wrapper(callback_data *data)
{
	avp_notify_chg_ind_t *parms = &data->parms.avp_notify_chg_ind_parms;

	g_assert(data->id == BT_APPL_AVP_NOTIFY_CHG_IND_CB);

	emit_avp_notify_chg(parms->bd_addr,
				parms->event_id,
				handle_avp_get_network_order(),
				parms->data);
}

/* AvpSetPlayer */
static void do_avp_set_player_cfm_wrapper(callback_data *data)
{
	avp_set_player_cfm_t *parms = &data->parms.avp_set_player_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_SET_PLAYER_CFM_CB);

	device_object_avp_set_player_RTN(parms->bd_addr,
						parms->status,
						parms->num_item,
						parms->is_db_aware);
}

/* AvpGetMetadataAttr */
static void do_avp_get_metadata_attr_cfm_wrapper(callback_data *data)
{
	avp_get_metadata_attr_cfm_t *parms =
				&data->parms.avp_get_metadata_attr_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_GET_METADATA_ATTR_CFM_CB);

	device_object_avp_get_metadata_attr_RTN(parms->bd_addr,
						parms->status,
						parms->num_attributes,
						handle_avp_get_network_order(),
						parms->data);
}

/* AvpGetFolderItem */
static void do_avp_get_folder_item_cfm_wrapper(callback_data *data)
{
	avp_get_folder_item_cfm_t *parms =
				&data->parms.avp_get_folder_item_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_GET_FOLDER_ITEM_CFM_CB);

	handle_avp_get_folder_item_cfm(parms->bd_addr,
					parms->status,
					parms->scope,
					parms->num_items);
}

/* AvpGetFolderItem */
static void do_avp_get_folder_item_ind_wrapper(callback_data *data)
{
	avp_get_folder_item_ind_t *parms =
				&data->parms.avp_get_folder_item_ind_parms;

	g_assert(data->id == BT_APPL_AVP_GET_FOLDER_ITEM_IND_CB);

	handle_avp_get_folder_and_element_item_ind(parms->bd_addr,
							parms->status,
							parms->scope,
							parms->data,
							0x02 /* Folder item */);
}

/* AvpGetFolderItem */
static void do_avp_media_element_ind_wrapper(callback_data *data)
{
	avp_media_element_ind_t *parms =
				&data->parms.avp_media_element_ind_parms;

	g_assert(data->id == BT_APPL_AVP_MEDIA_ELEMENT_IND_CB);

	handle_avp_get_folder_and_element_item_ind(parms->bd_addr,
						parms->status,
						parms->scope,
						parms->data,
						0x03 /* Element item */);
}

/* AvpGetMediaPlayerItem */
static void do_avp_get_media_player_item_cfm_wrapper(callback_data *data)
{
	avp_get_media_player_item_cfm_t *parms =
			&data->parms.avp_get_media_player_item_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_CFM_CB);

	handle_avp_get_media_player_item_cfm(parms->bd_addr,
						parms->status,
						parms->num_items);
}

/* AvpGetMediaPlayerItem */
static void do_avp_get_media_player_ind_wrapper(callback_data *data)
{
	avp_get_media_player_ind_t *parms =
				&data->parms.avp_get_media_player_ind_parms;

	g_assert(data->id == BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_IND_CB);

	handle_avp_get_media_player_item_ind(parms->bd_addr,
						parms->status,
						parms->data);
}

/* AvpPlayItem */
static void do_avp_play_item_cfm_wrapper(callback_data *data)
{
	avp_play_item_cfm_t *parms = &data->parms.avp_play_item_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_PLAY_ITEM_CFM_CB);

	device_object_avp_play_item_RTN(parms->bd_addr, parms->status);
}

/* AvpChangePath */
static void do_avp_change_path_cfm_wrapper(callback_data *data)
{
	avp_change_path_cfm_t *parms = &data->parms.avp_change_path_cfm_parms;

	g_assert(data->id == BT_APPL_AVP_CHANGE_PATH_CFM_CB);

	device_object_avp_change_path_RTN(parms->bd_addr,
						parms->status,
						parms->num_item);
}

/* AvpCodecAndBitrate */
static void do_avp_codec_and_bitrate_ind_wrapper(callback_data *data)
{
	avp_codec_and_bitrate_ind_t *parms =
				&data->parms.avp_codec_and_bitrate_ind_parms;

	g_assert(data->id == BT_APPL_AVP_CODEC_AND_BITRATE_IND_CB);

	emit_avp_codec_and_bitrate(parms->bd_addr,
					parms->codec_type,
					parms->bitrate);
}

/********/
/* PBDL */
/********/

/* PbDlGetCapabilities */
static void do_pbdl_get_capabilities_cfm_wrapper(callback_data *data)
{
	pbdl_get_capabilities_cfm_t *parms =
				&data->parms.pbdl_get_capabilities_cfm_parms;

	g_assert(data->id == BT_APPL_PBDL_GET_CAPABILITIES_CFM_CB);

	device_object_pb_dl_get_capabilities_RTN(parms->bd_addr,
							parms->status,
							parms->pb_type);
}

/* PbDlPrepare */
static void do_pbdl_prepare_cfm_wrapper(callback_data *data)
{
	pbdl_prepare_cfm_t *parms = &data->parms.pbdl_prepare_cfm_parms;

	g_assert(data->id == BT_APPL_PBDL_PREPARE_CFM_CB);

	device_object_pb_dl_prepare_RTN(parms->bd_addr,	parms->pb_type);
}

/* PbDlStart */
static void do_pbdl_start_cfm_wrappper(callback_data *data)
{
	pbdl_start_cfm_t *parms = &data->parms.pbdl_start_cfm_parms;

	g_assert(data->id == BT_APPL_PBDL_START_CFM_CB);

	device_object_pb_dl_start_RTN(parms->bd_addr,
					parms->status,
					parms->pb_number,
					parms->new_missed_call);
}

/* PbDlStop */
static void do_pbdl_stop_cfm_wrappper(callback_data *data)
{
	pbdl_stop_cfm_t *parms = &data->parms.pbdl_stop_cfm_parms;

	g_assert(data->id == BT_APPL_PBDL_STOP_CFM_CB);

	device_object_pb_dl_stop_RTN(parms->bd_addr,parms->status);
}

/* PbDlDataSignal */
static void do_pbdl_data_ind_wrapper(callback_data *data)
{
	pbdl_data_ind_t *parms = &data->parms.pbdl_data_ind_parms;

	g_assert(data->id == BT_APPL_PBDL_DATA_IND_CB);

	handle_phonebook_pb_dl_data(parms->bd_addr,
					parms->data_type,
					parms->data);
}

/* PbDlDataSignal */
static void do_pbdl_complete_ind_wrapper(callback_data *data)
{
	pbdl_complete_ind_t *parms = &data->parms.pbdl_complete_ind_parms;

	g_assert(data->id == BT_APPL_PBDL_DATA_COMPLETE_IND_CB);

	handle_phonebook_pb_dl_data_complete(parms->bd_addr,
						parms->status,
						parms->pb_number);
}

/* PbDlDataSignal */
static void do_pbdl_calender_complete_ind_wrapper(callback_data *data)
{
	pbdl_calender_complete_ind_t *parms =
				&data->parms.pbdl_calender_complete_ind_parms;

	g_assert(data->id == BT_APPL_PBDL_CALENDAR_COMPLETE_IND_CB);

	handle_phonebook_pb_dl_calendar_complete(parms->bd_addr, parms->status);
}

/* PbDlComplete */
static void do_pbdl_complete_cfm_wrapper(callback_data *data)
{
	pbdl_complete_cfm_t *parms = &data->parms.pbdl_complete_cfm_parms;

	g_assert(data->id == BT_APPL_PBDL_COMPLETE_CFM_CB);

	device_object_pb_dl_complete_RTN(parms->bd_addr, parms->status);
}

/* PbDlConnectionLost */
static void do_pbdl_con_lost_ind_wrapper(callback_data *data)
{
	pbdl_con_lost_ind_t *parms = &data->parms.pbdl_con_lost_ind_parms;

	g_assert(data->id == BT_APPL_PBDL_CON_LOST_IND_CB);

	emit_pb_dl_connection_lost(parms->bd_addr);
}

/* PbDlStatusChg */
static void do_pbdl_status_chg_ind_wrapper(callback_data *data)
{
	pbdl_status_chg_ind_t *parms = &data->parms.pbdl_status_chg_ind_parms;

	g_assert(data->id == BT_APPL_PBDL_STATUS_CHG_IND_CB);

	emit_pb_dl_status_chg(parms->bd_addr, parms->status);
}

/* MapGetCapabilities */
static void do_map_get_capabilities_cfm_wrapper(callback_data *data)
{
	map_get_capabilities_cfm_t *parms =
				&data->parms.map_get_capabilities_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_GET_CAPABILITIES_CFM_CB);

	device_object_map_get_capabilities_RTN(parms->bd_addr,
						parms->status,
						parms->msg_capability);
}

/* MapStart */
static void do_map_start_cfm_wrapper(callback_data *data)
{
	map_start_cfm_t *parms = &data->parms.map_start_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_START_CFM_CB);

	device_object_map_start_RTN(parms->bd_addr, parms->status);
}

/* MapActiveInstance */
static void do_map_active_instance_ind_wrapper(callback_data *data)
{
	map_active_instance_ind_t *parms =
				&data->parms.map_active_instance_ind_parms;

	g_assert(data->id == BT_APPL_MAP_ACTIVE_INSTANCE_IND_CB);

	emit_map_active_instance(parms->bd_addr,
					parms->sms_name,
					parms->email_name);
}

/* MapNotifyStatus */
static void do_map_notify_status_ind_wrapper(callback_data *data)
{
	map_notify_status_ind_t *parms =
				&data->parms.map_notify_status_ind_parms;

	g_assert(data->id == BT_APPL_MAP_NOTIFY_STATUS_IND_CB);

	emit_map_notify_status(parms->bd_addr, parms->status);
}

/* MapNotifyRegistStatus */
static void do_map_notify_regist_status_ind_wrapper(callback_data *data)
{
	map_notify_regist_status_ind_t *parms =
				&data->parms.map_notify_regist_status_ind_parms;

	g_assert(data->id == BT_APPL_MAP_NOTIFY_REGIST_STATUS_IND_CB);

	emit_map_notify_regist_status(parms->bd_addr,
					parms->service_name,
					parms->status);
}

/* MapStop */
static void do_map_stop_cfm_wrapper(callback_data *data)
{
	map_stop_cfm_t *parms = &data->parms.map_stop_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_STOP_CFM_CB);

	device_object_map_stop_RTN(parms->bd_addr, parms->status);
}

/* MapChgInstance */
static void do_map_chg_instance_cfm_wrapper(callback_data *data)
{
	map_chg_instance_cfm_t *parms = &data->parms.map_chg_instance_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_CHG_INSTANCE_CFM_CB);

	device_object_map_chg_instance_RTN(parms->bd_addr, parms->status);
}

/* MapUpdateInbox */
static void do_map_update_inbox_cfm_wrapper(callback_data *data)
{
	map_update_inbox_cfm_t *parms = &data->parms.map_update_inbox_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_UPDATE_INBOX_CFM_CB);

	device_object_map_update_inbox_RTN(parms->bd_addr, parms->status);
}

/* MapListFolder */
static void do_map_list_folder_cfm_wrapper(callback_data *data)
{
	map_list_folder_cfm_t *parms = &data->parms.map_list_folder_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_FOLDER_CFM_CB);

	device_object_map_list_folder_RTN(parms->bd_addr,
						parms->status,
						parms->accessible,
						parms->folder_num);
}

/* MapListFolder */
static void do_map_list_folder_ind_wrapper(callback_data *data)
{
	map_list_folder_ind_t *parms = &data->parms.map_list_folder_ind_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_FOLDER_IND_CB);

	emit_map_list_folder_data(parms->bd_addr,
					parms->service_name,
					parms->folder_listing);
}

/* MapListFolderComp */
static void do_map_list_folder_comp_ind_wrapper(callback_data *data)
{
	map_list_folder_comp_ind_t *parms =
				&data->parms.map_list_folder_comp_ind_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_FOLDER_COMP_IND_CB);

	emit_map_list_folder_comp(parms->bd_addr,
					parms->service_name,
					parms->total_size,
					parms->status);
}

/* MapListMsg */
static void do_map_list_msg_cfm_wrapper(callback_data *data)
{
	map_list_msg_cfm_t *parms = &data->parms.map_list_msg_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_MSG_CFM_CB);

	device_object_map_list_msg_RTN(parms->bd_addr,
					parms->status,
					parms->message_num);
}

/* MapListMsgData */
static void do_map_list_msg_data_ind_wrapper(callback_data *data)
{
	map_list_msg_data_ind_t *parms =
				&data->parms.map_list_msg_data_ind_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_MSG_IND_CB);

	handle_map_list_msg_data_ind(parms);
}

/* MapListMsgComp */
static void do_map_list_msg_comp_ind_wrapper(callback_data *data)
{
	map_list_msg_comp_ind_t *parms =
				&data->parms.map_list_msg_comp_ind_parms;

	g_assert(data->id == BT_APPL_MAP_LIST_MSG_COMP_IND_CB);

	handle_map_list_msg_data_complete(parms->bd_addr);
	emit_map_list_msg_comp(parms->bd_addr,
				parms->service_name,
				parms->total_size,
				parms->status);
}

/* MapReadMsg */
static void do_map_read_msg_cfm_wrapper(callback_data *data)
{
	map_read_msg_cfm_t *parms = &data->parms.map_read_msg_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_READ_MSG_CFM_CB);

	device_object_map_read_msg_RTN(parms);
}

/* MapReadMsgData */
static void do_map_read_msg_ind_wrapper(callback_data *data)
{
	map_read_msg_ind_t *parms = &data->parms.map_read_msg_ind_parms;

	g_assert(data->id == BT_APPL_MAP_READ_MSG_IND_CB);

	emit_map_read_msg_data(parms->bd_addr,
				parms->service_name,
				parms->data_flag,
				parms->msg_data);
}

/* MapReadMsgComp */
static void do_map_read_msg_comp_ind_wrapper(callback_data *data)
{
	map_read_msg_comp_ind_t *parms =
				&data->parms.map_read_msg_comp_ind_parms;

	g_assert(data->id == BT_APPL_MAP_READ_MSG_COMP_IND_CB);

	emit_map_read_msg_comp(parms->bd_addr, parms->service_name);
}

/* MapSetMsgStatus */
static void do_map_set_msg_status_cfm_wrapper(callback_data *data)
{
	map_set_msg_status_cfm_t *parms =
				&data->parms.map_set_msg_status_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_SET_MSG_STATUS_CFM_CB);

	device_object_map_set_msg_status_RTN(parms->bd_addr, parms->status);
}

/* MapUpload */
static void do_map_upload_cfm_wrapper(callback_data *data)
{
	map_upload_cfm_t *parms = &data->parms.map_upload_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_UPLOAD_CFM_CB);

	device_object_map_upload_RTN(parms->bd_addr, parms->status);
}

/* MapUploadComp */
static void do_map_upload_ind_wrapper(callback_data *data)
{
	map_upload_ind_t *parms = &data->parms.map_upload_ind_parms;

	g_assert(data->id == BT_APPL_MAP_UPLOAD_IND_CB);

	emit_map_upload_comp(parms->bd_addr,
				parms->service_name,
				parms->status);
}

/* MapAbort */
static void do_map_abort_cfm_wrapper(callback_data *data)
{
	map_abort_cfm_t *parms = &data->parms.map_abort_cfm_parms;

	g_assert(data->id == BT_APPL_MAP_ABORT_CFM_CB);

	device_object_map_abort_RTN(parms->bd_addr, parms->status);
}

/* MapNotify */
static void do_map_notify_ind_wrapper(callback_data *data)
{
	map_notify_ind_t *parms = &data->parms.map_notify_ind_parms;

	g_assert(data->id == BT_APPL_MAP_NOTIFY_IND_CB);

	emit_map_notify(parms);
}

/* MapErrorReport */
static void do_map_error_report_ind_wrapper(callback_data *data)
{
	map_error_report_ind_t *parms = &data->parms.map_error_report_ind_parms;

	g_assert(data->id == BT_APPL_MAP_ERROR_REPORT_IND_CB);

	emit_map_error_report(parms->bd_addr,
				parms->service_name,
				parms->operation,
				parms->error_code);
}

/*******/
/* NAP */
/*******/

/* NapCreateDevice */
static void do_nap_create_device_cfm_wrapper(callback_data *data)
{
	nap_create_device_cfm_t *parms =
				&data->parms.nap_create_device_cfm_parms;

	g_assert(data->id == BT_APPL_CREATE_DEVICE_CFM_CB);

	adapter_object_nap_create_device_RTN(parms->bd_addr,
						parms->service,
						parms->status,
						parms->device_name);
}

/* NapDestroyDevice */
static void do_nap_destroy_device_cfm_wrapper(callback_data *data)
{
	nap_destroy_device_cfm_t *parms =
				&data->parms.nap_destroy_device_cfm_parms;

	g_assert(data->id == BT_APPL_DESTROY_DEVICE_CFM_CB);

	adapter_object_nap_destroy_device_RTN(parms->bd_addr,
						parms->service,
						parms->status);
}

/* NapDestroyDeviceInd */
static void do_nap_destroy_device_ind_wrapper(callback_data *data)
{
	nap_destroy_device_ind_t *parms =
				&data->parms.nap_destroy_device_ind_parms;

	g_assert(data->id == BT_APPL_DESTROY_DEVICE_IND_CB);

	emit_nap_destroy_device_ind(parms->bd_addr, parms->service);
}

static do_callback_function callback_functions[NUM_ALPS_CALLBACKS] = {
	/* general callbacks */
	[BT_APPL_GET_DEVICE_NAME_CFM_CB] = do_get_device_name_cfm_wrapper,
	[BT_APPL_ROLE_CHANGE_IND_CB] = do_role_change_ind_wrapper,
	[BT_APPL_SERVICE_TYPE_IND_CB] = do_service_type_ind_wrapper,
	[BT_APPL_CONNECTION_COMPLETE_IND_CB] = do_connection_complete_ind_wrapper,
	[BT_APPL_DISCONNECTION_IND_CB] = do_disconnection_ind_wrapper,
	[BT_APPL_TESTMODE_CFM_CB] = do_testmode_cfm_wrapper,
	[BT_APPL_TESTMODE_COMPLETE_IND_CB] = do_testmode_complete_ind_wrapper,
	[BT_APPL_LINKQUALITY_CFM_CB] = do_linkqualify_cfm_wrapper,
	[BT_APPL_SPP_CAPABILITIES_IND_CB] = handle_spp_capabilities,
	[BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB] = do_ssp_numeric_confirm_ind_wrapper,
	[BT_APPL_CONNECTION_REQUEST_IND_CB] = do_connection_request_ind_wrapper,	
	[BT_APPL_PNPINFO_IND_CB] = handle_pnp_info_ind,
	[BT_APPL_INQUIRY_IND_CB] = do_inquiry_ind_wrapper,
	[BT_APPL_INQUIRY_COMPLETE_IND_CB] = do_inquiry_complete_ind_wrapper,
	/* general agent callbacks */
	[BT_APPL_REQUEST_PIN_IND_CB] = do_agent_callback_wrapper,
	[BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB] = do_agent_callback_wrapper,
	[BT_APPL_PAIRING_COMPLETE_IND_CB] = do_agent_callback_wrapper,
	[BT_APPL_LINKKEY_REQUEST_IND_CB] = do_agent_callback_wrapper,
	[BT_APPL_AUTHENTICATION_IND_CB] = do_agent_callback_wrapper,

	/* hf callbacks */
	[BT_APPL_CALL_STATUS_IND_CB] = do_hf_call_status_ind_wrapper,
	[BT_APPL_PHONE_STATUS_IND_CB] = do_hf_phone_status_ind_wrapper,
	[BT_APPL_HF_DIAL_CFM_CB] = do_hf_dial_cfm_wrapper,
	[BT_APPL_HF_REDIAL_CFM_CB] = do_hf_redial_cfm_wrapper,
	[BT_APPL_HF_MEM_DIAL_CFM_CB] = do_hf_mem_dial_cfm_wrapper,
	[BT_APPL_HF_TERMINATE_CALL_CFM_CB] = do_hf_terminate_call_cfm_wrapper,
	[BT_APPL_HF_TERMINATE_ALL_CFM_CB] = do_hf_terminate_all_cfm_wrapper,
	[BT_APPL_HF_AUDIO_TRANSFER_CFM_CB] = do_hf_audio_transfer_cfm_wrapper,
	[BT_APPL_HF_SETUP_AUDIO_IND_CB] = do_hf_setup_audio_ind_wrapper,
	[BT_APPL_HF_RELEASE_AUDIO_IND_CB] = do_hf_release_audio_ind_wrapper,
	[BT_APPL_HF_SEND_DTMF_CFM_CB] = do_hf_send_dtmf_cfm_wrapper,
	[BT_APPL_HF_INCOMING_CALL_ACT_CFM_CB] = do_hf_incoming_call_act_cfm_wrapper,
	[BT_APPL_HF_RING_IND_CB] = do_hf_ring_ind_wrapper,
	[BT_APPL_HF_RELEASE_INACTIVE_CALL_CFM_CB] = do_hf_release_inactive_call_cfm_wrapper,
	[BT_APPL_HF_RELEASE_ACTIVE_CALL_CFM_CB] = do_hf_release_active_call_cfm_wrapper,
	[BT_APPL_HF_SWAP_CALL_CFM_CB] = do_hf_swap_call_cfm_wrapper,
	[BT_APPL_HF_ADD_CALL_CFM_CB] = do_hf_add_call_cfm_wrapper,
	[BT_APPL_HF_RELEASE_SPEC_CALL_CFM_CB] = do_hf_release_spec_call_cfm_wrapper,
	[BT_APPL_HF_HOLD_EXCEP_CALL_CFM_CB] = do_hf_hold_excep_call_cfm_wrapper,
	[BT_APPL_HF_EXP_CALL_TRANS_CFM_CB] = do_hf_exp_call_trans_cfm_wrapper,
	[BT_APPL_HF_SUBSCRIBER_CFM_CB] = do_hf_subscriber_cfm_wrapper,
	[BT_APPL_HF_MANUFACTURER_ID_CFM_CB] = do_hf_manufacturer_id_cfm_wrapper,
	[BT_APPL_HF_MODEL_ID_CFM_CB] = do_hf_model_id_cfm_wrapper,
	[BT_APPL_HF_REVISION_ID_CFM_CB] = do_hf_revision_id_cfm_wrapper,
	[BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_CFM_CB] = do_hf_voice_recognition_activation_cfm_wrapper,
	[BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_IND_CB] = do_hf_voice_recognition_activation_ind_wrapper,
	[BT_APPL_HF_BT_APPL_SIRI_XAPL_CFM_CB] = do_hf_siri_xapl_cfm_wrapper,
	[BT_APPL_HF_BT_APPL_SIRI_NrStat_CFM_CB] = do_hf_siri_nrstat_cfm_wrapper,
	[BT_APPL_HF_BT_APPL_SIRI_NrStat_IND_CB] = do_hf_siri_nrstat_ind_wrapper,
	[BT_APPL_HF_BT_APPL_SIRI_XAPL_IND_CB] = do_hf_siri_aplsiri_ind_wrapper,
	[BT_APPL_HF_BT_APPL_SIRI_EFM_CFM_CB] = do_hf_siri_efm_cfm_wrapper,
	[BT_APPL_HF_RELEASE_WAITING_CALL_CFM_CB] = do_hf_release_waiting_call_cfm_wrapper,
	[BT_APPL_HF_SETUP_AUDIO_WBS_IND_CB] = do_hf_setup_audio_wbs_ind_wrapper,

	/* avp callbacks */
	[BT_APPL_AVP_VERSION_IND_CB] = do_avp_version_ind_wrapper,
	[BT_APPL_AVP_CTRL_CMD_CFM_CB] = do_avp_ctrl_cmd_cfm_wrapper,
	[BT_APPL_AVP_CTRL_CMD_IND_CB] = do_avp_ctrl_cmd_ind_wrapper,
	[BT_APPL_AVP_GET_CAPABILITIES_CFM_CB] = do_avp_get_capabilities_cfm_wrapper,
	[BT_APPL_AVP_LIST_ATTR_CFM_CB] = do_avp_list_attr_cfm_wrapper,
	[BT_APPL_AVP_LIST_VAL_CFM_CB] = do_avp_list_val_cfm_wrapper,
	[BT_APPL_AVP_SET_VAL_CFM_CB] = do_avp_set_val_cfm_wrapper,
	[BT_APPL_AVP_PLAY_STATUS_IND_CB] = do_avp_play_status_ind_wrapper,
	[BT_APPL_AVP_NOTIFY_SET_COMPLETE_IND_CB] = do_avp_notify_set_complete_ind_wrapper,
	[BT_APPL_AVP_NOTIFY_CHG_IND_CB] = do_avp_notify_chg_ind_wrapper,
	[BT_APPL_AVP_SET_PLAYER_CFM_CB] = do_avp_set_player_cfm_wrapper,
	[BT_APPL_AVP_GET_METADATA_ATTR_CFM_CB] = do_avp_get_metadata_attr_cfm_wrapper,
	[BT_APPL_AVP_GET_FOLDER_ITEM_CFM_CB] = do_avp_get_folder_item_cfm_wrapper,
	[BT_APPL_AVP_GET_FOLDER_ITEM_IND_CB] = do_avp_get_folder_item_ind_wrapper,
	[BT_APPL_AVP_MEDIA_ELEMENT_IND_CB] = do_avp_media_element_ind_wrapper,
	[BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_CFM_CB] = do_avp_get_media_player_item_cfm_wrapper,
	[BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_IND_CB] = do_avp_get_media_player_ind_wrapper,
	[BT_APPL_AVP_PLAY_ITEM_CFM_CB] = do_avp_play_item_cfm_wrapper,
	[BT_APPL_AVP_CHANGE_PATH_CFM_CB] = do_avp_change_path_cfm_wrapper,
	[BT_APPL_AVP_CODEC_AND_BITRATE_IND_CB] = do_avp_codec_and_bitrate_ind_wrapper,

	/* pbdl callbacks */
	[BT_APPL_PBDL_GET_CAPABILITIES_CFM_CB] = do_pbdl_get_capabilities_cfm_wrapper,
	[BT_APPL_PBDL_PREPARE_CFM_CB] = do_pbdl_prepare_cfm_wrapper,
	[BT_APPL_PBDL_START_CFM_CB] = do_pbdl_start_cfm_wrappper,
	[BT_APPL_PBDL_STOP_CFM_CB] = do_pbdl_stop_cfm_wrappper,
	[BT_APPL_PBDL_DATA_IND_CB] = do_pbdl_data_ind_wrapper,
	[BT_APPL_PBDL_DATA_COMPLETE_IND_CB] = do_pbdl_complete_ind_wrapper,
	[BT_APPL_PBDL_CALENDAR_COMPLETE_IND_CB] = do_pbdl_calender_complete_ind_wrapper,
	[BT_APPL_PBDL_COMPLETE_CFM_CB] = do_pbdl_complete_cfm_wrapper,
	[BT_APPL_PBDL_CON_LOST_IND_CB] = do_pbdl_con_lost_ind_wrapper,
	[BT_APPL_PBDL_STATUS_CHG_IND_CB] = do_pbdl_status_chg_ind_wrapper,

	/* map callbacks */
	[BT_APPL_MAP_GET_CAPABILITIES_CFM_CB] = do_map_get_capabilities_cfm_wrapper,
	[BT_APPL_MAP_START_CFM_CB] = do_map_start_cfm_wrapper,
	[BT_APPL_MAP_ACTIVE_INSTANCE_IND_CB] = do_map_active_instance_ind_wrapper,
	[BT_APPL_MAP_NOTIFY_STATUS_IND_CB] = do_map_notify_status_ind_wrapper,
	[BT_APPL_MAP_NOTIFY_REGIST_STATUS_IND_CB] = do_map_notify_regist_status_ind_wrapper,
	[BT_APPL_MAP_STOP_CFM_CB] = do_map_stop_cfm_wrapper,
	[BT_APPL_MAP_CHG_INSTANCE_CFM_CB] = do_map_chg_instance_cfm_wrapper,
	[BT_APPL_MAP_UPDATE_INBOX_CFM_CB] = do_map_update_inbox_cfm_wrapper,
	[BT_APPL_MAP_LIST_FOLDER_CFM_CB] = do_map_list_folder_cfm_wrapper,
	[BT_APPL_MAP_LIST_FOLDER_IND_CB] = do_map_list_folder_ind_wrapper,
	[BT_APPL_MAP_LIST_FOLDER_COMP_IND_CB] = do_map_list_folder_comp_ind_wrapper,
	[BT_APPL_MAP_LIST_MSG_CFM_CB] = do_map_list_msg_cfm_wrapper,
	[BT_APPL_MAP_LIST_MSG_IND_CB] = do_map_list_msg_data_ind_wrapper,
	[BT_APPL_MAP_LIST_MSG_COMP_IND_CB] = do_map_list_msg_comp_ind_wrapper,
	[BT_APPL_MAP_READ_MSG_CFM_CB] = do_map_read_msg_cfm_wrapper,
	[BT_APPL_MAP_READ_MSG_IND_CB] = do_map_read_msg_ind_wrapper,
	[BT_APPL_MAP_READ_MSG_COMP_IND_CB] = do_map_read_msg_comp_ind_wrapper,
	[BT_APPL_MAP_SET_MSG_STATUS_CFM_CB] = do_map_set_msg_status_cfm_wrapper,
	[BT_APPL_MAP_UPLOAD_CFM_CB] = do_map_upload_cfm_wrapper,
	[BT_APPL_MAP_UPLOAD_IND_CB] = do_map_upload_ind_wrapper,
	[BT_APPL_MAP_ABORT_CFM_CB] = do_map_abort_cfm_wrapper,
	[BT_APPL_MAP_NOTIFY_IND_CB] = do_map_notify_ind_wrapper,
	[BT_APPL_MAP_ERROR_REPORT_IND_CB] = do_map_error_report_ind_wrapper,

	/* internet callbacks */
	[BT_APPL_CREATE_DEVICE_CFM_CB] = do_nap_create_device_cfm_wrapper,
	[BT_APPL_DESTROY_DEVICE_CFM_CB] = do_nap_destroy_device_cfm_wrapper,
	[BT_APPL_DESTROY_DEVICE_IND_CB] = do_nap_destroy_device_ind_wrapper
};

static void callback_data_free(callback_data *data)
{
	g_slice_free(callback_data, data);
}

static callback_data *callback_data_alloc(enum alps_callbacks id,
					  enum tagBtApplCallbackIdEnum type)
{
	callback_data *data;

	data = g_slice_new(callback_data);
	if (!data) {
		/* fatal error -- terminate the daemon */
		DEBUG_ERROR("Failed to allocate memory for callback data");
		exit(1);
	}

	data->id = id;
	data->type = type;
	switch (type) {
	case GENERAL_CALLBACK_ID:
		{
			general_callback_parms *parms = &data->parms.general;

			parms->name[0] = 0;
			parms->name_len = 0;
			parms->pairing_info.pucName = parms->name;
		}
		break;

	case HFP_CALLBACK_ID:
		{
		}
		break;

	case AVP_CALLBACK_ID:
		/* add any AVP common parms initialisation here */
		break;

	case PBDL_CALLBACK_ID:
		{
		}
		break;

	case MESSAGE_CALLBACK_ID:
		{
		}
		break;

	case DEVICE_CALLBACK_ID:
		{
		}
		break;

	default:
		break;
	}

	return data;
}

static void start_connection_complete_timer(void);

static gboolean process_alps_callbacks(GIOChannel * source,
				       GIOCondition condition,
				       gpointer userdata)
{
	const int buf_size = 32;
	GIOStatus status;
	gchar buf[buf_size];
	gsize bytes_read = buf_size;
	callback_data *data;
	(void) userdata;

	if (condition & G_IO_HUP) {
		DEBUG_ERROR("Broken pipe");
		return FALSE;
	} else if (condition & G_IO_ERR) {
		DEBUG_ERROR("I/O error reading pipe");
		return FALSE;
	} else if (!(condition & G_IO_IN)) {
		g_message("Nothing to read from pipe");
		return TRUE;
	}

	/* Drain the pipe fifo.  We don't care about the data. */
	while (bytes_read == buf_size) {
		status = g_io_channel_read_chars(source, buf, buf_size,
						 &bytes_read, NULL);
	}

	while ((data = g_async_queue_try_pop(callback_queue)) != NULL) {
		DEBUG("calling callback handler ID %u, MAX %u",
		      data->id, NUM_ALPS_CALLBACKS);

		if ((data->id < NUM_ALPS_CALLBACKS) &&
		    (callback_functions[data->id] != NULL)) {
			/* call the wrapper function pointer */
			callback_functions[data->id](data);

			/*
			 * Implement special case handling for a successful
			 * ConnectionComplete event.  In this case, we want to
			 * briefly inhibit processing of ALPS callbacks in
			 * order to give clients of the dbus bluetooth daemon
			 * an opportunity to install signal handlers before any
			 * signals are emitted by the daemon.
			 */
			if ((data->id == BT_APPL_CONNECTION_COMPLETE_IND_CB)
				&& (data->parms.gen_connection_complete_ind_parms.status == 0))
			{
				DEBUG("Successful ConnectionComplete event for service %d.",
				      data->parms.gen_connection_complete_ind_parms.service);
				start_connection_complete_timer();
				callback_data_free(data);
				pipe_source = 0;
				return FALSE;	/* the pipe source will be removed */
			}

		} else {
			DEBUG_ERROR("Passed unsupported callback ID %d",
					data->id);
		}			

		/* free the data memory associated with the callback */ 
		callback_data_free(data);
	}

	return TRUE;
}

static BTHRESULT write_to_self_pipe(unsigned char val)
{
	int res;

	res = write(pipe_fd[1], &val, 1);

	/*
	 * The pipe should never fill up, but if it does we can ignore the error
	 * since we only care if the pipe has data ready to read or not.
	 */
	if (res < 0 && errno == EAGAIN)
		return 0;
	else if (res < 0)
		return BT_APPL_ERROR_OPERATION_FAILED;
	else
		return 0;
}

/* Remove the watch from the pipe GIOChannel. */
static void remove_pipe_source(void)
{
	if (pipe_source) {
		if (!g_source_remove(pipe_source)) {
			DEBUG_ERROR("Failed to remove pipe source %d.",
				    pipe_source);
		}
		pipe_source = 0;
	}
}

/* Add a watch for the pipe GIOChannel. */
static void attach_pipe_source(void)
{
	if (!pipe_source) {
		pipe_source = g_io_add_watch(pipe_ch,
					     G_IO_IN | G_IO_HUP | G_IO_ERR,
					     process_alps_callbacks, NULL);
	}
	if (!pipe_source) {
		DEBUG_ERROR("Failed to add watch for pipe GIOChannel");
	}
}

/*
 * This is the timeout callback routine for the ConnectionComplete timer.
 * It reinstalls the watch on the pipe GIOChannel to resume processing of
 * ALPS callbacks.
 */
static gboolean connection_complete_timer_callback(gpointer data)
{
	DEBUG("ConnectionComplete timer id %d expired.", timeout_source);
	(void) data;
	timeout_source = 0;

	/* reinstall the watch on the pipe GIOChannel */
	attach_pipe_source();

	/* force the pipe watch routine to execute */
	write_to_self_pipe(BT_APPL_CONNECTION_COMPLETE_IND_CB);

	return FALSE;	/* the timeout source will be destroyed */
}

/*
 * This routine is used to start a 200 millisecond timer after a successful
 * ConnectionComplete event.  The watch on the pipe GIOChannel must be
 * removed, for example by returning FALSE from the watch callback.
 * Processing of ALPS callbacks is inhibited during the delay.  This delay
 * gives clients of the dbus bluetooth daemon a head start to install handlers
 * for dbus signals that could be spontaneously emitted immediately after
 * ConnectionComplete.  This delay mitigates the race condition by giving the
 * clients a head start, but it doesn't eliminate the race condition.
 */
static void start_connection_complete_timer(void)
{
	GSource *source;
	static const guint timeout_ms = 200;

	/* remove an old timer if it is already running */
	if (timeout_source) {
		DEBUG("Removing ConnectionComplete timer id %d.",
		      timeout_source);
		if (!g_source_remove(timeout_source)) {
			DEBUG_ERROR("Failed to remove ConnectionComplete "
				    "timer id %d.", timeout_source);
		}
		timeout_source = 0;
	}

	/* install a timeout source */
	source = g_timeout_source_new(timeout_ms);
	g_source_set_callback(source, connection_complete_timer_callback, NULL, NULL);
	timeout_source = g_source_attach(source, NULL);
	g_source_unref(source);

	DEBUG("Starting %d millisecond ConnectionComplete timer id %d",
	      timeout_ms, timeout_source);
}

/*********************/
/* General Callbacks */
/*********************/

static BTHRESULT BT_APPL_Get_Device_Name_CFM_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 name_len,
						IN const u8 *name)
{
	callback_data *data;
	gen_get_device_name_cfm_t *parms;

	DEBUG("bd_addr %s, name_len %d, name %s",
	      bd_addr_to_str(bd_addr), name_len, name);

	data = callback_data_alloc(BT_APPL_GET_DEVICE_NAME_CFM_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_get_device_name_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->device_name = g_strndup((gchar *) name, name_len);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_GET_DEVICE_NAME_CFM_CB);
}

static BTHRESULT BT_APPL_Role_Change_IND_Callback(IN const BD_ADDRESS bd_addr,
						  IN u8 status, IN u8 role)
{
	callback_data *data;
	gen_role_change_ind_t *parms;

	DEBUG("bd_addr %s, status %d, role %d",
	      bd_addr_to_str(bd_addr), status, role);

	data = callback_data_alloc(BT_APPL_ROLE_CHANGE_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_role_change_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = (gboolean) status;
	parms->role = role;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_ROLE_CHANGE_IND_CB);
}

static BTHRESULT BT_APPL_Request_PIN_IND_Callback(IN const BD_ADDRESS bd_addr,
						  IN u8 name_len,
						  IN const u8 * name)
{
	callback_data *data;
	general_callback_parms *parms;

	DEBUG("bd_addr %s, name_len %d, name %s",
	      bd_addr_to_str(bd_addr), name_len, name);

	data = callback_data_alloc(BT_APPL_REQUEST_PIN_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.general;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	if (name) {
		parms->name_len = name_len;
		g_strlcpy((gchar *) parms->name, (gchar *) name,
			  MAX_NAME_BYTES);
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_REQUEST_PIN_IND_CB);
}

static BTHRESULT BT_APPL_SSP_Numeric_Confirm_IND_Callback(IN const BD_ADDRESS
							  bd_addr,
							  IN u8 name_len,
							  IN const u8 * name,
							  IN u32 numeric_value)
{
	callback_data *data;
	gen_ssp_numeric_confirm_ind_t *parms;

	DEBUG("bd_addr %s, name_len %d, name %s, numeric_value %u",
	      bd_addr_to_str(bd_addr), name_len, name,
	      (unsigned int)numeric_value);

	data = callback_data_alloc(BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_ssp_numeric_confirm_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->name = g_strndup((gchar *) name, name_len);
	parms->numeric_value = numeric_value;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_SSP_NUMERIC_CONFIRM_IND_CB);
}

static BTHRESULT BT_APPL_SSP_Passkey_Notification_IND_Callback(IN const
							       BD_ADDRESS
							       bd_addr,
							       IN u8 name_len,
							       IN const u8 *
							       name,
							       IN u32 passkey)
{
	callback_data *data;
	general_callback_parms *parms;

	DEBUG("bd_addr %s, name_len %d, name %s, passkey %u",
	      bd_addr_to_str(bd_addr), name_len, name,
	      (unsigned int)passkey);

	data = callback_data_alloc(BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.general;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	if (name) {
		parms->name_len = name_len;
		g_strlcpy((gchar *) parms->name, (gchar *) name,
			  MAX_NAME_BYTES);
	}
	parms->passkey = passkey;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_SSP_PASSKEY_NOTIFICATION_IND_CB);
}

static BTHRESULT BT_APPL_Pairing_Complete_IND_Callback(IN const BD_ADDRESS
						       bd_addr, IN u8 status,
						       IN const PairingInfoPtr_t
						       pairing_info)
{
	callback_data *data;
	general_callback_parms *parms;

	DEBUG("bd_addr %s, status %d", bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_PAIRING_COMPLETE_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.general;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	if (pairing_info) {
		parms->pairing_info = *pairing_info;
		if (pairing_info->pucName) {
			parms->name_len = pairing_info->ucNameLen;
			g_strlcpy((gchar *) parms->name,
				  (gchar *) pairing_info->pucName,
				  MAX_NAME_BYTES);
		}
	} else
		memset(&parms->pairing_info, 0, sizeof(parms->pairing_info));
	parms->pairing_info.pucName = parms->name;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PAIRING_COMPLETE_IND_CB);
}

static BTHRESULT BT_APPL_Service_Type_IND_Callback(IN const BD_ADDRESS bd_addr,
						   IN u16 service_type)
{
	callback_data *data;
	gen_service_type_ind_t *parms;

	DEBUG("bd_addr %s, service_type 0x%02X",
	      bd_addr_to_str(bd_addr), service_type);

#ifdef DBG_ENABLED
	if (service_type & 0x0001) DEBUG("HFP supported");
	if (service_type & 0x0002) DEBUG("A2DP supported");
	if (service_type & 0x0004) DEBUG("AVRCP supported");
	if (service_type & 0x0008) DEBUG("PBAP supported");
	if (service_type & 0x0010) DEBUG("MAP supported");
	if (service_type & 0x0020) DEBUG("DUN supported");
	if (service_type & 0x0040) DEBUG("PAN supported");
	if (service_type & 0x0080) DEBUG("SSP supported");
	if (service_type & 0x0100) DEBUG("PNP supported");
	
	if (service_type & 0x0200) DEBUG("DIP supported");
#endif

	data = callback_data_alloc(BT_APPL_SERVICE_TYPE_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_service_type_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_type = (guint32) service_type;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_SERVICE_TYPE_IND_CB);
}

static BTHRESULT BT_APPL_Linkkey_Request_IND_Callback(IN const BD_ADDRESS
						      bd_addr)
{
	callback_data *data;
	general_callback_parms *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data = callback_data_alloc(BT_APPL_LINKKEY_REQUEST_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.general;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_LINKKEY_REQUEST_IND_CB);
}

static BTHRESULT BT_APPL_Connection_Request_IND_Callback(IN const BD_ADDRESS
								bd_addr,
								IN u8 service)
{
	callback_data *data;
	gen_connection_request_ind_t *parms;

	DEBUG("bd_addr %s, service %d", bd_addr_to_str(bd_addr), service);

	data = callback_data_alloc(BT_APPL_CONNECTION_REQUEST_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_connection_request_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service = service;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_CONNECTION_REQUEST_IND_CB);
}

static void BT_APPL_Reset_Event_IND_Callback(void )
{

	DEBUG("Reset indication");
}

static BTHRESULT BT_APPL_Connection_Complete_IND_Callback(IN const BD_ADDRESS
							  bd_addr, IN u8 status,
							  IN u8 service)
{
	callback_data *data;
	gen_connection_complete_ind_t *parms;

	DEBUG("bd_addr %s, status %d, service %d",
	      bd_addr_to_str(bd_addr), status, service);

	data = callback_data_alloc(BT_APPL_CONNECTION_COMPLETE_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_connection_complete_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = (gboolean) status;
	parms->service = service;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_CONNECTION_COMPLETE_IND_CB);
}

static BTHRESULT BT_APPL_Disconnection_IND_Callback(IN const BD_ADDRESS bd_addr,
						    IN u8 service,
						    IN u8 conn_loss_type)
{
	callback_data *data;
	gen_disconnection_ind_t *parms;

	DEBUG("bd_addr %s, service %d, conn_loss_type %d",
	      bd_addr_to_str(bd_addr), service,
	      conn_loss_type);

	data = callback_data_alloc(BT_APPL_DISCONNECTION_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_disconnection_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service = service;
	parms->conn_loss_type = conn_loss_type;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_DISCONNECTION_IND_CB);
}

static BTHRESULT BT_APPL_Authentication_IND_Callback(
						IN const BD_ADDRESS bd_addr)
{
	callback_data *data;
	general_callback_parms *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data = callback_data_alloc(BT_APPL_AUTHENTICATION_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.general;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AUTHENTICATION_IND_CB);
}

static BTHRESULT BT_APPL_TestMode_CFM_Callback(IN u8 status)
{
	callback_data *data;
	gen_testmode_cfm_t *parms;

	DEBUG("status %d", status);

	data = callback_data_alloc(BT_APPL_TESTMODE_CFM_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_testmode_cfm_parms;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_TESTMODE_CFM_CB);
}

static BTHRESULT BT_APPL_TestMode_Complete_IND_Callback(IN u8 noti_type,
								IN u8 status)
{
	callback_data *data;
	gen_testmode_complete_ind_t *parms;

	DEBUG("noti_type %d, status %d", noti_type, status);

	data = callback_data_alloc(BT_APPL_TESTMODE_COMPLETE_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_testmode_complete_ind_parms;
	parms->noti_type = noti_type;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_TESTMODE_COMPLETE_IND_CB);
}

static BTHRESULT BT_APPL_LinkQuality_CFM_Callback(IN u8 status, IN s8 rssi,
							IN u8 link_quality)
{
	callback_data *data;
	gen_linkqualify_cfm_t *parms;

	DEBUG("status %d, rssi %d, link_quality %d", status, rssi,
	      link_quality);

	data = callback_data_alloc(BT_APPL_LINKQUALITY_CFM_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_linkqualify_cfm_parms;
	parms->status = status;
	parms->rssi = rssi;
	parms->link_quality = link_quality;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_LINKQUALITY_CFM_CB);
}

static BTHRESULT BT_APPL_SPP_Capabilities_IND_Callback(
				IN const BD_ADDRESS bd_addr,
				IN SppServiceInfoPtr_t spp_service_info,
				IN u8 continue_flag)
{
	callback_data *data;
	gen_spp_capabilities_ind_t *parms;
	u8 num_uuids;
	u8 num_loop;
	ClassIDInfoPtr_t uuid_ptr;
	u8 service_handle;
	const u8 *service_name;

	g_assert(spp_service_info != NULL);

	/* grab the service info elements */
	num_uuids = spp_service_info->ucNumUUIDs;
	uuid_ptr = spp_service_info->pstUUID;
	service_handle = spp_service_info->ucServiceHandle;
	service_name = spp_service_info->pucServiceName;

	DEBUG("bd_addr %s num_uuids %u uuid_ptr %p service_handle %u "
	      "service_name \'%s\', continue_flag %u",
	      bd_addr_to_str(bd_addr), num_uuids, uuid_ptr, service_handle,
	      service_name, continue_flag);

#ifdef DBG_ENABLED
	if (uuid_ptr != NULL) {
		if (btd_debug_level >= BTD_DBG_INFO_LEV) {
			for (num_loop=0; num_loop < num_uuids; num_loop++) {
				DEBUG("%u: uuid_len %u, uuid 0x%s",
					num_loop, uuid_ptr[num_loop].ucUUIDLen,
					dump_spp_uuid(&uuid_ptr[num_loop]));
			}
		}
	}
#endif

	data = callback_data_alloc(BT_APPL_SPP_CAPABILITIES_IND_CB,
				   GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_spp_capabilities_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	/* create GArray for the UUID index lengths */
	parms->uuid_index = g_array_new(FALSE, FALSE, sizeof(guint8));
	g_assert(parms->uuid_index != NULL);

	/* create GByteArray for the UUID data */
	parms->uuid_data = g_byte_array_new();
	g_assert(parms->uuid_data != NULL);

	if (uuid_ptr != NULL) {
		/* create GByteArrays of the UUIDs */
		for (num_loop=0; num_loop < num_uuids; num_loop++) {

			if (uuid_ptr[num_loop].ucUUIDLen > 16) {
				DEBUG_ERROR("UUID length %u > 16",
					uuid_ptr[num_loop].ucUUIDLen);
			}

			parms->uuid_index =
					g_array_append_val(parms->uuid_index,
						uuid_ptr[num_loop].ucUUIDLen);
			parms->uuid_data = g_byte_array_append(parms->uuid_data,
						&uuid_ptr[num_loop].aucUUID[0],
						uuid_ptr[num_loop].ucUUIDLen);
		}
	}

	parms->service_handle = service_handle;
	parms->service_name = g_strdup((gchar *) service_name);
	parms->continue_flag = continue_flag;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_SPP_CAPABILITIES_IND_CB);
}

//General Callback function for PnpInfoInd
static BTHRESULT BT_APPL_PnPInfo_IND_Callback(
				IN const BD_ADDRESS aucBD_ADDR,
				IN PnPInfo_AttributePtr_t pstPnPInfoAttribute,
				IN u8 ucContinue)
{
	DEBUG("Called");

	//vga4cob DIPIMPL validate all the data that is received.
	callback_data *data;
	gen_pnp_info_attribute_ind_t *parms;
	//Temporary variables to store the value of the length fo different URLs
	u16 url_len = 0xffff;
	data = callback_data_alloc(BT_APPL_PNPINFO_IND_CB,GENERAL_CALLBACK_ID);
	parms=&data->parms.gen_pnp_info_attribute_ind_parms;
	memcpy(parms->bd_addr,aucBD_ADDR,sizeof(BD_ADDRESS));
	if(NULL != pstPnPInfoAttribute)
	{
		DEBUG("The value of m_usSpecificationID  is %d",pstPnPInfoAttribute->m_usSpecificationID);
		DEBUG("The value of vendorID  is %d",pstPnPInfoAttribute->m_usVendorID);
		DEBUG("The value of productID is %d",pstPnPInfoAttribute->m_usProductID);
		DEBUG("The value of version  is %d",pstPnPInfoAttribute->m_usVersion);
		DEBUG("The value of primaryRecord is %d",pstPnPInfoAttribute->m_ucPrimaryRecord);
		DEBUG("The value of vendorIDSource is %d",pstPnPInfoAttribute->m_usVendorIDSource);
		DEBUG("The value of m_usClientExecutableURLLen is %d",pstPnPInfoAttribute->m_stOptionalAttribute.m_usClientExecutableURLLen);
		DEBUG("The value of m_usServiceDescriptionLen is %d",pstPnPInfoAttribute->m_stOptionalAttribute.m_usServiceDescriptionLen);
		DEBUG("The value of m_aucDocumentationURLLen  is %d",pstPnPInfoAttribute->m_stOptionalAttribute.m_usDocumentationURLLen);
		DEBUG("The value of m_usClientExecutableURL is %s",pstPnPInfoAttribute->m_stOptionalAttribute.m_aucClientExecutableURL);
		DEBUG("The value of m_usServiceDescription is %s",pstPnPInfoAttribute->m_stOptionalAttribute.m_aucServiceDescription);
		DEBUG("The value of m_aucDocumentationURL is %s",pstPnPInfoAttribute->m_stOptionalAttribute.m_aucDocumentationURL);
		parms->specificationID = (pstPnPInfoAttribute->m_usSpecificationID)?pstPnPInfoAttribute->m_usSpecificationID:0xffff;
		
		parms->vendorID = (pstPnPInfoAttribute->m_usVendorID)?(pstPnPInfoAttribute->m_usVendorID):0xffff;
		parms->productID = pstPnPInfoAttribute->m_usProductID?pstPnPInfoAttribute->m_usProductID:0xffff;
		parms->version = pstPnPInfoAttribute->m_usVersion?pstPnPInfoAttribute->m_usVersion:0xffff;
		parms->primaryRecord = pstPnPInfoAttribute->m_ucPrimaryRecord;
		parms->vendorIDSource = pstPnPInfoAttribute->m_usVendorIDSource?pstPnPInfoAttribute->m_usVendorIDSource:0xffff;
		//To assign the variables to the Optional Attributes
		//Create a GArray to store the URL lengths in the GArray url_length and url_name
		parms->url_length = g_array_new(FALSE,FALSE,sizeof(guint));
		g_assert(parms->url_length != NULL);
		//ByteArray to store the URLnames
		parms->url_name = g_byte_array_new();
		g_assert(parms->url_name != NULL);

		//Client Executable URL
		url_len = (pstPnPInfoAttribute->m_stOptionalAttribute.m_usClientExecutableURLLen < CLIENT_EXECUTABLE_URL_MAX_LEN)?
					pstPnPInfoAttribute->m_stOptionalAttribute.m_usClientExecutableURLLen:CLIENT_EXECUTABLE_URL_MAX_LEN;
		parms->url_length = g_array_append_val(parms->url_length, url_len);
		parms->url_name = g_byte_array_append(parms->url_name, pstPnPInfoAttribute->m_stOptionalAttribute.m_aucClientExecutableURL,url_len);

		//Service description URL
		url_len = (pstPnPInfoAttribute->m_stOptionalAttribute.m_usServiceDescriptionLen < SERVICE_DESCRIPTION_MAX_LEN)?
					pstPnPInfoAttribute->m_stOptionalAttribute.m_usServiceDescriptionLen:SERVICE_DESCRIPTION_MAX_LEN;
		parms->url_length = g_array_append_val(parms->url_length, url_len);
		parms->url_name = g_byte_array_append(parms->url_name, pstPnPInfoAttribute->m_stOptionalAttribute.m_aucServiceDescription,url_len);

		//Documention URL
		url_len = (pstPnPInfoAttribute->m_stOptionalAttribute.m_usDocumentationURLLen < DOCUMENTATION_URL_MAX_LEN)?
				pstPnPInfoAttribute->m_stOptionalAttribute.m_usDocumentationURLLen:DOCUMENTATION_URL_MAX_LEN;
		parms->url_length = g_array_append_val(parms->url_length, url_len);
		parms->url_name = g_byte_array_append(parms->url_name, pstPnPInfoAttribute->m_stOptionalAttribute.m_aucServiceDescription,url_len);
	}
	else
	{
		DEBUG("The PnPInfo_AttributePtr is not valid. Hence setting all the values to default error value.");
		parms->specificationID=0xffff;
		parms->vendorID=0xffff;
		parms->productID=0xffff;
		parms->version=0xffff;
		parms->primaryRecord=0xff;
		parms->vendorIDSource=0xffff;
		parms->url_length = g_array_append_val(parms->url_length,url_len);
		parms->url_length = g_array_append_val(parms->url_length,url_len);
		parms->url_length = g_array_append_val(parms->url_length,url_len);
	}
	parms->continue_flag=ucContinue;
	g_async_queue_push(callback_queue,data);
	return write_to_self_pipe(BT_APPL_PNPINFO_IND_CB);
}

//General Callback function for InquiryInd
static BTHRESULT BT_APPL_Inquiry_IND_Callback(IN const InquiryResultInfo_t* inquiry_result)
{
	callback_data *data;
	gen_inquiry_ind_t *parms;
	if (inquiry_result == NULL)
		return BT_APPL_ERROR_PARAMETERS;
	data = callback_data_alloc(BT_APPL_INQUIRY_IND_CB,
					GENERAL_CALLBACK_ID);
	parms=&data->parms.gen_inquiry_ind_parms;
	/*Handle allocation*/
	DEBUG("BD_ADDRESS received is %s", bd_addr_to_str(inquiry_result->m_aucDevAddr));
	BD_ADDRESS bd_ad_rev;
	bd_ad_rev[5]=inquiry_result->m_aucDevAddr[0];
	bd_ad_rev[4]=inquiry_result->m_aucDevAddr[1];
	bd_ad_rev[3]=inquiry_result->m_aucDevAddr[2];
	bd_ad_rev[2]=inquiry_result->m_aucDevAddr[3];
	bd_ad_rev[1]=inquiry_result->m_aucDevAddr[4];
	bd_ad_rev[0]=inquiry_result->m_aucDevAddr[5];
	memcpy(parms->bd_addr,bd_ad_rev,sizeof(BD_ADDRESS));
	DEBUG("BD_Address is copied %s",bd_addr_to_str(parms->bd_addr));
	DEBUG("Page scan repetition mode is %d",inquiry_result->m_ucPageScanRepetitionMode);
	parms->page_scan_rep_mode = inquiry_result->m_ucPageScanRepetitionMode;
	DEBUG("Clock offset is %d", inquiry_result->m_usClockOffset);
	parms->clock_offset = inquiry_result->m_usClockOffset;
	DEBUG("RSSI is %d", inquiry_result->m_scRSSI);
	DEBUG("Major Device Class is %d", inquiry_result->m_stCod.ucMajorDeviceClass);
	parms->cod.major_device_class = inquiry_result->m_stCod.ucMajorDeviceClass;
	DEBUG("Minor Device Class is %d",inquiry_result->m_stCod.ucMinorDeviceClass);
	parms->cod.minor_device_class = inquiry_result->m_stCod.ucMinorDeviceClass;
	DEBUG("Major Service Class is %d",inquiry_result->m_stCod.usMajorServiceClass);
	parms->cod.major_service_class = inquiry_result->m_stCod.usMajorServiceClass;
	DEBUG("The length of the EirData is %d",strlen((gchar *)inquiry_result->m_aucEir));
	parms->eir_data =strlen((gchar *) inquiry_result->m_aucEir)?
			 g_strdup((gchar*)inquiry_result->m_aucEir):
			 g_strdup((gchar *) "");
	g_async_queue_push(callback_queue,data);
	return write_to_self_pipe(BT_APPL_INQUIRY_IND_CB);
}

//General Callback function for InquiryCompleteInd
static BTHRESULT BT_APPL_Inquiry_Complete_IND_Callback(u8 inquiry_status)
{
	callback_data *data;
	gen_inquiry_complete_ind_t *parms;
	DEBUG("The inquiry status is %d",inquiry_status);
	data = callback_data_alloc(BT_APPL_INQUIRY_COMPLETE_IND_CB,GENERAL_CALLBACK_ID);
	parms = &data->parms.gen_inquiry_complete_ind_parms;
	parms->inquiry_status = inquiry_status;
	g_async_queue_push(callback_queue,data);
	return write_to_self_pipe(BT_APPL_INQUIRY_COMPLETE_IND_CB);
}
void BT_APPL_General_Place_Holder_A_Callback(void)
{
	DEBUG("General_Place_Holder_A_Callbac");
}

void BT_APPL_General_Place_Holder_B_Callback(void)
{
	DEBUG("General_Place_Holder_B_Callbac");
}

void BT_APPL_General_Place_Holder_C_Callback(void)
{
	DEBUG("General_Place_Holder_C_Callbac");
}

/****************/
/* HF Callbacks */
/****************/

BTHRESULT BT_APPL_Call_Status_IND_Callback(const BD_ADDRESS bd_addr,
						u8 call_handle,
						u8 total,
						CallStatus_t *call_status)
{
	callback_data *data;
	hf_call_status_ind_t *parms;

	DEBUG("bd_addr %s call_handle %u total %u",
	      bd_addr_to_str(bd_addr), call_handle, total);

	data = callback_data_alloc(BT_APPL_CALL_STATUS_IND_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_call_status_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->call_handle = call_handle;
	parms->total = total;
	if (call_status != NULL) {
		parms->direction = call_status->ucDirection;
		parms->state = call_status->ucState;
		parms->mode = call_status->ucMode;
		parms->number = g_strdup((gchar *) call_status->pucNumber);
		parms->type = call_status->ucType;
	} else {
		parms->direction = 0;
		parms->state = 0;
		parms->mode = 0;
		parms->number = NULL;
		parms->type = 0;
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_CALL_STATUS_IND_CB);
}

BTHRESULT BT_APPL_Phone_Status_IND_Callback(const BD_ADDRESS bd_addr,
		PhoneStatus_t *phone_status)
{
	callback_data *data;
	hf_phone_status_ind_t *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data = callback_data_alloc(BT_APPL_PHONE_STATUS_IND_CB,
			HFP_CALLBACK_ID);
	parms = &data->parms.hf_phone_status_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	if (phone_status != NULL) {
		parms->service = phone_status->ucService;
		parms->signal = phone_status->ucSignal;
		parms->roaming = phone_status->ucRoaming;
		parms->battery = phone_status->ucBattery;

		/*
		 * Check operator name is UTF-8 to avoid the D-Bus daemon
		 * rejecting the string causing the D-Bus socket to close
		 * which causes the alps_bt_daemon to terminate via exit().
		 * exitcatcher then reboots the system.
		 */
		if (g_utf8_validate((gchar *) phone_status->pucOperatorName,
				strlen((gchar *) phone_status->pucOperatorName),
				NULL)) {
			DEBUG("Detected UTF-8 Operator Name string");
			parms->operator_name = g_strdup(
					(gchar *) phone_status->pucOperatorName);
		} else {
			DEBUG("Detected non-UTF8 Operator Name string");
			parms->operator_name = g_strdup((gchar *) "");
		}
		parms->ag_supp_feature = phone_status->ulAGsuppfeature;
		parms->AgThreeWayCallSuppFeatures = phone_status->ucAgThreeWayCallSuppFeatures;

	} else {
		parms->service = 0;
		parms->signal = 0;
		parms->roaming = 0;
		parms->battery = 0;
		parms->operator_name = NULL;
		parms->ag_supp_feature = 0;
		parms->AgThreeWayCallSuppFeatures = 0;
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PHONE_STATUS_IND_CB);
}

BTHRESULT BT_APPL_HF_Dial_CFM_Callback(const BD_ADDRESS bd_addr,
					u8 status,
					u8 code)
{
	callback_data *data;
	hf_dial_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_DIAL_CFM_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_dial_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_DIAL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Redial_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_redial_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_REDIAL_CFM_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_redial_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_REDIAL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Mem_Dial_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_mem_dial_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_MEM_DIAL_CFM_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_mem_dial_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_MEM_DIAL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Terminate_Call_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_terminate_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_TERMINATE_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_terminate_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_TERMINATE_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Terminate_All_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_terminate_all_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_TERMINATE_ALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_terminate_all_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_TERMINATE_ALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Audio_Transfer_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_audio_transfer_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_AUDIO_TRANSFER_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_audio_transfer_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_AUDIO_TRANSFER_CFM_CB);
}

BTHRESULT BT_APPL_HF_Setup_Audio_IND_Callback(const BD_ADDRESS bd_addr)
{
	callback_data *data;
	hf_setup_audio_ind_t *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data = callback_data_alloc(BT_APPL_HF_SETUP_AUDIO_IND_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_setup_audio_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_SETUP_AUDIO_IND_CB);
}

BTHRESULT BT_APPL_HF_Release_Audio_IND_Callback(const BD_ADDRESS bd_addr)
{
	callback_data *data;
	hf_release_audio_ind_t *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data = callback_data_alloc(BT_APPL_HF_RELEASE_AUDIO_IND_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_release_audio_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_RELEASE_AUDIO_IND_CB);
}

BTHRESULT BT_APPL_HF_Send_DTMF_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_send_dtmf_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_SEND_DTMF_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_send_dtmf_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_SEND_DTMF_CFM_CB);
}

BTHRESULT BT_APPL_HF_Incoming_Call_Act_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_incoming_call_act_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_INCOMING_CALL_ACT_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_incoming_call_act_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_INCOMING_CALL_ACT_CFM_CB);
}

BTHRESULT BT_APPL_HF_Ring_IND_Callback(const BD_ADDRESS bd_addr, u8 ring_tone)
{
	callback_data *data;
	hf_ring_ind_t *parms;

	DEBUG("bd_addr %s ring_tone %u",
	      bd_addr_to_str(bd_addr), ring_tone);

	data = callback_data_alloc(BT_APPL_HF_RING_IND_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_ring_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->ring_tone = ring_tone;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_RING_IND_CB);
}

BTHRESULT BT_APPL_HF_Release_Inactive_Call_CFM_Callback(
						const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_release_inactive_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_RELEASE_INACTIVE_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_release_inactive_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_RELEASE_INACTIVE_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Release_Active_Call_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_release_active_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_RELEASE_ACTIVE_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_release_active_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_RELEASE_ACTIVE_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Swap_Call_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_swap_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_SWAP_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_swap_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_SWAP_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Add_Call_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code)
{
	callback_data *data;
	hf_add_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_ADD_CALL_CFM_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_add_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_ADD_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Release_Spec_Call_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_release_spec_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_RELEASE_SPEC_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_release_spec_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_RELEASE_SPEC_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Hold_Excep_Call_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_hold_excep_call_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_HOLD_EXCEP_CALL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_hold_excep_call_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_HOLD_EXCEP_CALL_CFM_CB);
}

BTHRESULT BT_APPL_HF_Exp_Call_Trans_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
	callback_data *data;
	hf_exp_call_trans_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_EXP_CALL_TRANS_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_exp_call_trans_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_EXP_CALL_TRANS_CFM_CB);
}

BTHRESULT BT_APPL_HF_Subscriber_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code,
						u8 node_num,
						SubscriberNodePtr_t sub_node)
{
	callback_data *data;
	hf_subscriber_cfm_t *parms;
	guint32 i;

	DEBUG("bd_addr %s status %u code %u node_num %u sub_node %p",
	      bd_addr_to_str(bd_addr), status, code, node_num, sub_node);

	data = callback_data_alloc(BT_APPL_HF_SUBSCRIBER_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_subscriber_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;

	parms->subscriber_info = g_ptr_array_sized_new(node_num);

	for (i = 0; i < node_num; i++) {
		GValue *value;
		gchar *info_number;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_UCHAR_UCHAR_STRING);
		g_value_set_boxed_take_ownership(value,
			dbus_g_type_specialized_construct(DBUS_STRUCT_UCHAR_UCHAR_STRING));

		info_number = g_strndup((gchar *) sub_node[i].pucNumber,
					strlen((gchar *)sub_node[i].pucNumber));

		dbus_g_type_struct_set(value,
			0, sub_node[i].ucService,
			1, sub_node[i].ucType,
			2, info_number,
			G_MAXUINT);

		g_ptr_array_add(parms->subscriber_info, g_value_get_boxed(value));
		g_free(value);
		g_free(info_number);
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_SUBSCRIBER_CFM_CB);
}

BTHRESULT BT_APPL_HF_Manufacturer_ID_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code,
							u8 *manufacturer)
{
	callback_data *data;
	hf_manufacturer_id_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_MANUFACTURER_ID_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_manufacturer_id_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;
	parms->manufacturer = g_strdup((gchar *) manufacturer);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_MANUFACTURER_ID_CFM_CB);
}

BTHRESULT BT_APPL_HF_Model_ID_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code,
						u8 *model)
{
	callback_data *data;
	hf_model_id_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_MODEL_ID_CFM_CB, HFP_CALLBACK_ID);
	parms = &data->parms.hf_model_id_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;
	parms->model = g_strdup((gchar *) model);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_MODEL_ID_CFM_CB);
}

BTHRESULT BT_APPL_HF_Revision_ID_CFM_Callback(const BD_ADDRESS bd_addr,
						u8 status,
						u8 code,
						u8 *revision)
{
	callback_data *data;
	hf_revision_id_cfm_t *parms;

	DEBUG("bd_addr %s status %u code %u",
	      bd_addr_to_str(bd_addr), status, code);

	data = callback_data_alloc(BT_APPL_HF_REVISION_ID_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_revision_id_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->code = code;
	parms->revision = g_strdup((gchar *) revision);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_REVISION_ID_CFM_CB);
}

BTHRESULT BT_APPL_HF_Voice_Recognition_Activation_CFM_Callback(
						const BD_ADDRESS bd_addr,
						u8 activation,
						u8 status)
{
	callback_data *data;
	hf_voice_recognition_activation_cfm_t *parms;

	DEBUG("bd_addr %s activation %u status %u",
	      bd_addr_to_str(bd_addr), activation, status);

	data = callback_data_alloc(BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_voice_recognition_activation_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->activation = activation;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_CFM_CB);
}

BTHRESULT BT_APPL_HF_Voice_Recognition_Activation_IND_Callback(
						const BD_ADDRESS bd_addr,
						u8 activation)
{
	callback_data *data;
	hf_voice_recognition_activation_ind_t *parms;

	DEBUG("bd_addr %s activation %u", bd_addr_to_str(bd_addr), activation);

	data = callback_data_alloc(BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_IND_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.hf_voice_recognition_activation_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->activation = activation;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_VOICE_RECOGNITION_ACTIVATION_IND_CB);
}

BTHRESULT BT_APPL_SIRI_XAPL_CFM_Callback(IN const BD_ADDRESS bd_addr,
		   	   	   	   	   	   	   	   	   	   	 IN u8 ucStatus,
		   	   	   	   	   	   	   	   	   	   	 IN const u8 * pucParameter)
{
	callback_data *data;
	appl_xapl_siri_cfm_t *parms;

	DEBUG("bd_addr %s ucStatus %u pucParameter %u",
	      bd_addr_to_str(bd_addr), ucStatus, pucParameter);

	data = callback_data_alloc(BT_APPL_HF_BT_APPL_SIRI_XAPL_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.appl_xapl_siri_cfm_params;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = ucStatus;
	parms->parameter = g_strdup((gchar *)pucParameter);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_BT_APPL_SIRI_XAPL_CFM_CB);
}

BTHRESULT BT_APPL_SIRI_AplSiri_IND_Callback(IN const BD_ADDRESS bd_addr,
					   IN u8 ucAplSiri)
{
	callback_data *data;
	appl_xapl_siri_ind_t *parms;

	DEBUG("bd_addr %s SIRI availability %u", bd_addr_to_str(bd_addr), ucAplSiri);

	data = callback_data_alloc(BT_APPL_HF_BT_APPL_SIRI_XAPL_IND_CB, HFP_CALLBACK_ID);
	parms = &data->parms.appl_xapl_siri_ind_params;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = ucAplSiri;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_BT_APPL_SIRI_XAPL_IND_CB);
}

BTHRESULT BT_APPL_SIRI_NrStat_CFM_Callback(IN const BD_ADDRESS bd_addr,
		   	   	   	   	   	   	   	   	   	   	 IN u8 ucStatus )
{
	callback_data *data;
	siri_nrstat_cfm_t *parms;

	DEBUG("bd_addr %s ucStatus %u",
	      bd_addr_to_str(bd_addr), ucStatus);

	data = callback_data_alloc(BT_APPL_HF_BT_APPL_SIRI_NrStat_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.siri_nrstat_cfm_params;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = ucStatus;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_BT_APPL_SIRI_NrStat_CFM_CB);
}

BTHRESULT BT_APPL_SIRI_NrStat_IND_Callback(IN const BD_ADDRESS bd_addr,
					   	   	   	   	   IN u8 aplNrStat)
{
	callback_data *data;
	siri_nrstat_ind_t *parms;

	DEBUG("bd_addr %s SIRI availability %s", bd_addr_to_str(bd_addr), aplNrStat);

	data = callback_data_alloc(BT_APPL_HF_BT_APPL_SIRI_NrStat_IND_CB, HFP_CALLBACK_ID);
	parms = &data->parms.siri_nrstat_ind_params;;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->ucAplNrStat = aplNrStat;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_BT_APPL_SIRI_NrStat_IND_CB);
}



BTHRESULT BT_APPL_SIRI_Efm_CFM_Callback(IN const BD_ADDRESS bd_addr,
		   	   	   	   	   	   	   	   	   	   	 IN u8 ucStatus )
{
	callback_data *data;
	siri_efm_cfm_t *parms;

	DEBUG("bd_addr %s ucStatus %u",
	      bd_addr_to_str(bd_addr), ucStatus);

	data = callback_data_alloc(BT_APPL_HF_BT_APPL_SIRI_EFM_CFM_CB,
					HFP_CALLBACK_ID);
	parms = &data->parms.siri_efm_cfm_params;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = ucStatus;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_HF_BT_APPL_SIRI_EFM_CFM_CB);
}

BTHRESULT BT_APPL_HF_Release_Waiting_Call_CFM_Callback(const BD_ADDRESS bd_addr,
							u8 status,
							u8 code)
{
 /* only Registered with but not used right now*/
	DEBUG("BT_APPL_HF_Release_Waiting_Call_CFM_Callback:: Only registered not used right now");
	(void) bd_addr;
	(void) status;
	(void) code;
	return 0;
}

BTHRESULT BT_APPL_HF_Control_Indicator_CFM_Callback(IN const BD_ADDRESS aucBD_ADDR,
        											IN u8 ucStatus,
        											IN u8 ucCode)
{
	// Will be implemented when required
	DEBUG("Called");
	(void) aucBD_ADDR;
	(void) ucStatus;
	(void) ucCode;
	return 0;
}

BTHRESULT BT_APPL_HF_Setup_Audio_WBS_IND_Callback(const BD_ADDRESS bd_addr,
										u8 audiocodec)
{
	DEBUG("Called");
	callback_data *data;
	hf_setup_audio_wbs_ind_t *parms;
	DEBUG("bd_addr is %s audiocodec is %d",bd_addr_to_str(bd_addr),audiocodec);
	data = callback_data_alloc(BT_APPL_HF_SETUP_AUDIO_WBS_IND_CB,HFP_CALLBACK_ID);
	parms = &data->parms.hf_setup_audio_wbs_ind_parms;
	memcpy(parms->bd_addr,bd_addr,sizeof(BD_ADDRESS));
	parms->audiocodec = audiocodec;
	g_async_queue_push(callback_queue,data);
	return write_to_self_pipe(BT_APPL_HF_SETUP_AUDIO_WBS_IND_CB);
}

void BT_APPL_HF_Place_Holder_A_Callback(void)
{
	DEBUG("BT_APPL_HF_Place_Holder_A_Callback");
}

void BT_APPL_HF_Place_Holder_B_Callback(void)
{
	DEBUG("BT_APPL_HF_Place_Holder_B_Callback");
}

void BT_APPL_HF_Place_Holder_C_Callback(void)
{
	DEBUG("BT_APPL_HF_Place_Holder_C_Callback");
}

/*****************/
/* AVP Callbacks */
/*****************/

BTHRESULT BT_APPL_AVP_Version_IND_Callback(IN const BD_ADDRESS bd_addr,
					   IN u8 avrcp_version)
{
	callback_data *data;
	avp_version_ind_t *parms;

#ifdef DBG_ENABLED
	char *version;

	switch (avrcp_version) {
	case 0:
		version = "1.0";
		break;
	case 1:
		version = "1.3";
		break;
	case 2:
		version = "1.4";
		break;
	default:
		version ="??";
		break;
	}
#endif

	DEBUG("bd_addr %s AVRCP version %s", bd_addr_to_str(bd_addr), version);

	data = callback_data_alloc(BT_APPL_AVP_VERSION_IND_CB, AVP_CALLBACK_ID);
	parms = &data->parms.avp_version_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->avrcp_version = avrcp_version;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_VERSION_IND_CB);
}

BTHRESULT BT_APPL_AVP_Ctrl_Cmd_CFM_Callback(IN const BD_ADDRESS bd_addr,
					    IN u8 command,
					    IN u8 status_flag,
					    IN u8 status)
{
	callback_data *data;
	avp_ctrl_cmd_cfm_t *parms;

	DEBUG("bd_addr %s command %u status_flag %u status %u",
	      bd_addr_to_str(bd_addr), command, status_flag, status);

	data = callback_data_alloc(BT_APPL_AVP_CTRL_CMD_CFM_CB, AVP_CALLBACK_ID);
	parms = &data->parms.avp_ctrl_cmd_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->command = command;
	parms->status_flag = status_flag;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_CTRL_CMD_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Ctrl_Cmd_IND_Callback(IN const BD_ADDRESS bd_addr,
					    IN u8 command)
{
	callback_data *data;
	avp_ctrl_cmd_ind_t *parms;

#ifdef DBG_ENABLED
	char *media_command;

	switch (command) {
	case 0x0A:
		media_command = "SUSPEND";
		break;
	case 0x0B:
		media_command = "RESUME";
		break;
	default:
		media_command ="??";
		break;
	}
#endif

	DEBUG("bd_addr %s command: %s", bd_addr_to_str(bd_addr), media_command);

	data = callback_data_alloc(BT_APPL_AVP_CTRL_CMD_IND_CB, AVP_CALLBACK_ID);
	parms = &data->parms.avp_ctrl_cmd_ind_parms;

	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->command = command;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_CTRL_CMD_IND_CB);
}

BTHRESULT BT_APPL_AVP_Get_Capabilities_CFM_Callback(IN const BD_ADDRESS bd_addr,
						    IN u8 status,
						    IN u8 capability_id,
						    IN u16 event_supported)
{
	callback_data *data;
	avp_get_capabilities_cfm_t *parms;

	DEBUG("bd_addr %s status %u capability_id %u event_supported 0x%04X",
	      bd_addr_to_str(bd_addr), status, capability_id,
	      event_supported);

	data = callback_data_alloc(BT_APPL_AVP_GET_CAPABILITIES_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_get_capabilities_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->capability_id = capability_id;
	parms->event_supported = event_supported;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_GET_CAPABILITIES_CFM_CB);
}

BTHRESULT BT_APPL_AVP_List_Attr_CFM_Callback(IN const BD_ADDRESS bd_addr,
					     IN u8 status,
					     IN u8 attributes_bit)
{
	callback_data *data;
	avp_list_attr_cfm_t *parms;

	DEBUG("bd_addr %s status %u capability_id %u ",
	      bd_addr_to_str(bd_addr), status, attributes_bit);

	data = callback_data_alloc(BT_APPL_AVP_LIST_ATTR_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_list_attr_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->attributes_bit = attributes_bit;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_LIST_ATTR_CFM_CB);
}

BTHRESULT BT_APPL_AVP_List_Val_CFM_Callback(IN const BD_ADDRESS bd_addr,
					    IN u8 status,
					    IN u8 num_setting_values,
					    IN u8 setting_values[],
					    IN u8 current_value)
{
	callback_data *data;
	avp_list_val_cfm_t *parms;

	DEBUG("bd_addr %s status %u num_setting_values %u current_value %u",
		bd_addr_to_str(bd_addr), status,
		num_setting_values, current_value);

	data = callback_data_alloc(BT_APPL_AVP_LIST_VAL_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_list_val_cfm_parms;

	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	/* create a new byte array for settings value data */
	parms->setting_values = g_byte_array_new();

	/* copy the settings value data into the GByteArray for storage */
	if (setting_values != NULL) {
		parms->setting_values = g_byte_array_append(
							parms->setting_values,
							setting_values,
							num_setting_values);
	} else if (num_setting_values != 0) {
		DEBUG_ERROR("setting_values is NULL but "
			"num_setting_values is %u",
			num_setting_values);
	}

	parms->current_value = current_value;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_LIST_VAL_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Set_Val_CFM_Callback(IN const BD_ADDRESS bd_addr,
					   IN u8 status)
{
	callback_data *data;
	avp_set_val_cfm_t *parms;

	DEBUG("bd_addr %s status %u",
		bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_AVP_SET_VAL_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_set_val_cfm_parms;

	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_SET_VAL_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Play_Status_IND_Callback(IN const BD_ADDRESS bd_addr,
					       IN u8 status,
					       IN u32 song_length,
					       IN u32 song_position,
					       IN u8 play_status)
{
	callback_data *data;
	avp_play_status_ind_t *parms;

	DEBUG("bd_addr %s status %u song_length %lu "
	      "song_position %lu, play_status %u",
	      bd_addr_to_str(bd_addr), status, song_length, song_position,
	      play_status);

	data = callback_data_alloc(BT_APPL_AVP_PLAY_STATUS_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_play_status_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->song_length = song_length;
	parms->song_position = song_position;
	parms->play_status = play_status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_PLAY_STATUS_IND_CB);
}

BTHRESULT BT_APPL_AVP_Notify_Set_Complete_IND_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u16 event_id)
{
	callback_data *data;
	avp_notify_set_complete_ind_t *parms;

	data = callback_data_alloc(BT_APPL_AVP_NOTIFY_SET_COMPLETE_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_notify_set_complete_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->event_id = (guint32) event_id;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_NOTIFY_SET_COMPLETE_IND_CB);
}

BTHRESULT BT_APPL_AVP_Notify_Chg_IND_Callback(IN const BD_ADDRESS bd_addr,
					      IN u8 event_id,
					      IN u8 size,
					      IN u8 data[])
{
	callback_data *data_info;
	avp_notify_chg_ind_t *parms;

	data_info = callback_data_alloc(BT_APPL_AVP_NOTIFY_CHG_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data_info->parms.avp_notify_chg_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->event_id = event_id;

	/* create a new byte array for this nofify change data */
	parms->data = g_byte_array_new();

	/* copy the attribute data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, size);
	} else if (size != 0) {
		DEBUG_ERROR("data is NULL but size is %u", size);
	}

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_AVP_NOTIFY_CHG_IND_CB);
}

BTHRESULT BT_APPL_AVP_Set_Player_CFM_Callback(IN const BD_ADDRESS bd_addr,
					      IN u8 status,
					      IN u32 num_item,
					      IN u8 is_db_aware)
{
	callback_data *data;
	avp_set_player_cfm_t *parms;

	DEBUG("bd_addr %s status %u num_item %lu, is_db_aware %u",
	      bd_addr_to_str(bd_addr), status, num_item,
	      is_db_aware);

	data = callback_data_alloc(BT_APPL_AVP_SET_PLAYER_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_set_player_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->num_item = num_item;
	parms->is_db_aware = is_db_aware;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_SET_PLAYER_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Get_Metadata_Attr_CFM_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u8 num_attributes,
						IN u16 size,
						IN u8 data[])
{
	callback_data *data_info;
	avp_get_metadata_attr_cfm_t *parms;

	DEBUG("bd_addr %s status %u num_attributes %u size %u",
	      bd_addr_to_str(bd_addr), status, num_attributes,
	      size);

	/* check that the size of the GByteArray is not exceeded */
	g_assert(size < G_MAXUINT);

	data_info = callback_data_alloc(BT_APPL_AVP_GET_METADATA_ATTR_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data_info->parms.avp_get_metadata_attr_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->num_attributes = num_attributes;

	/* create a new byte array for this attribute data */
	parms->data = g_byte_array_new();

	/* copy the attribute data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, size);
	} else if (size != 0) {
		DEBUG_ERROR("data is NULL but size is %u", size);
	}

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_AVP_GET_METADATA_ATTR_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Get_Folder_Item_CFM_Callback(IN const BD_ADDRESS bd_addr,
						   IN u8 status,
                                                   IN u8 scope,
						   IN u16 num_items)
{
	callback_data *data;
	avp_get_folder_item_cfm_t *parms;

	DEBUG("bd_addr %s status %u scope %u num_items %u",
	      bd_addr_to_str(bd_addr), status, scope,
	      num_items);

	data = callback_data_alloc(BT_APPL_AVP_GET_FOLDER_ITEM_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_get_folder_item_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->scope = scope;
	parms->num_items = num_items;

	g_async_queue_push(callback_queue, data);

	DEBUG_FUNC("Exited");

	return write_to_self_pipe(BT_APPL_AVP_GET_FOLDER_ITEM_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Folder_Item_IND_Callback(IN const BD_ADDRESS bd_addr,
					       IN u8 status,
					       IN u8 scope,
					       IN u16 size,
					       IN u8 data[])
{
	callback_data *data_info;
	avp_get_folder_item_ind_t *parms;

	DEBUG("bd_addr %s status %u scope %u size %u",
	      bd_addr_to_str(bd_addr), status, scope, size);

	DEBUG_DUMP_ARRAY(size, &data[0]);

	data_info = callback_data_alloc(BT_APPL_AVP_GET_FOLDER_ITEM_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data_info->parms.avp_get_folder_item_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->scope = scope;

	/* create a new byte array for this folder item data */
	parms->data = g_byte_array_new();

	/* copy the folder data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, size);
	} else if (size != 0) {
		DEBUG_ERROR("data is NULL but size is %u", size);
	}

	g_async_queue_push(callback_queue, data_info);

	DEBUG_FUNC("Exited");

	return write_to_self_pipe(BT_APPL_AVP_GET_FOLDER_ITEM_IND_CB);
}

BTHRESULT BT_APPL_AVP_Media_Element_IND_Callback(IN const BD_ADDRESS bd_addr,
						 IN u8 status,
						 IN u8 scope,
						 IN u16 size,
						 IN u8 data[])
{
	callback_data *data_info;
	avp_media_element_ind_t *parms;

	DEBUG("bd_addr %s status %u scope %u size %u",
	      bd_addr_to_str(bd_addr), status, scope, size);

	DEBUG_DUMP_ARRAY(size, &data[0]);

	data_info = callback_data_alloc(BT_APPL_AVP_MEDIA_ELEMENT_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data_info->parms.avp_media_element_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->scope = scope;

	/* create a new byte array for this folder item data */
	parms->data = g_byte_array_new();

	/* copy the element data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, size);
	} else if (size != 0) {
		DEBUG_ERROR("data is NULL but size is %u", size);
	}

	g_async_queue_push(callback_queue, data_info);

	DEBUG_FUNC("Exited");

	return write_to_self_pipe(BT_APPL_AVP_MEDIA_ELEMENT_IND_CB);
}

BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_CFM_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u16 num_items)
{
	callback_data *data;
	avp_get_media_player_item_cfm_t *parms;

	DEBUG("bd_addr %s status %u num_items %u",
	      bd_addr_to_str(bd_addr), status, num_items);

	data = callback_data_alloc(BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_get_media_player_item_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->num_items = num_items;

	g_async_queue_push(callback_queue, data);

	DEBUG_FUNC("Exited");

	return write_to_self_pipe(BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_IND_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u16 size,
						IN u8 data[])
{
	callback_data *data_info;
	avp_get_media_player_ind_t *parms;

	/*
	 * the ALPS size includes the 3 octet type and len fields so
	 * reduce the size by 3 for the size of the actual payload info
	 */
	if (size >= 3) {
		size -= 3;
	} else {
		DEBUG_ERROR("size too short %u", size);
	}

	DEBUG("bd_addr %s status %u size %u",
	      bd_addr_to_str(bd_addr), status, size);

	DEBUG_DUMP_ARRAY(size, &data[0]);

	data_info = callback_data_alloc(BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data_info->parms.avp_get_media_player_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	/* create a new byte array for this media player item data */
	parms->data = g_byte_array_new();

	/* copy the media player data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, size);
	} else if (size != 0) {
		DEBUG_ERROR("data is NULL but size is %u", size);
	}

	g_async_queue_push(callback_queue, data_info);

	DEBUG_FUNC("Exited");

	return write_to_self_pipe(BT_APPL_AVP_GET_MEDIA_PLAYER_ITEM_IND_CB);
}

BTHRESULT BT_APPL_AVP_Play_Item_CFM_Callback(IN const BD_ADDRESS bd_addr,
					     IN u8 status)
{
	callback_data *data;
	avp_play_item_cfm_t *parms;

	DEBUG("bd_addr %s status %u",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_AVP_PLAY_ITEM_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_play_item_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_PLAY_ITEM_CFM_CB);
}

BTHRESULT BT_APPL_AVP_Change_Path_CFM_Callback(IN const BD_ADDRESS bd_addr,
					       IN u8 status,
					       IN u32 num_item)
{
	callback_data *data;
	avp_change_path_cfm_t *parms;

	DEBUG("bd_addr %s status %u num_items %lu",
	      bd_addr_to_str(bd_addr), status, num_item);

	data = callback_data_alloc(BT_APPL_AVP_CHANGE_PATH_CFM_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_change_path_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->num_item = num_item;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_CHANGE_PATH_CFM_CB);
}

BTHRESULT BT_APPL_Codec_and_Bitrate_IND_Callback(IN const BD_ADDRESS bd_addr,
						 IN u8 codec_type,
						 IN u16 bitrate)
{
	callback_data *data;
	avp_codec_and_bitrate_ind_t *parms;

	DEBUG("bd_addr %s codec_type %u bitrate %u",
	      bd_addr_to_str(bd_addr), codec_type, bitrate);

	data = callback_data_alloc(BT_APPL_AVP_CODEC_AND_BITRATE_IND_CB,
				   AVP_CALLBACK_ID);
	parms = &data->parms.avp_codec_and_bitrate_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->codec_type = codec_type;
	parms->bitrate = (guint32) bitrate;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_AVP_CODEC_AND_BITRATE_IND_CB);
}

void BT_APPL_AVP_Place_Holder_A_Callback(void)
{
	DEBUG("BT_APPL_AVP_Place_Holder_A_Callback");
}

void BT_APPL_AVP_Place_Holder_B_Callback(void)
{
	DEBUG("BT_APPL_AVP_Place_Holder_B_Callback");
}

void BT_APPL_AVP_Place_Holder_C_Callback(void)
{
	DEBUG("BT_APPL_AVP_Place_Holder_C_Callback");
}

/******************/
/* PBDL Callbacks */
/******************/

BTHRESULT BT_APPL_PBDL_Get_Capabilities_CFM_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u8 pb_type)
{
	callback_data *data;
	pbdl_get_capabilities_cfm_t *parms;

	DEBUG("bd_addr %s status %u pb_type 0x%02X bitmap",
	      bd_addr_to_str(bd_addr), status, pb_type);

	data = callback_data_alloc(BT_APPL_PBDL_GET_CAPABILITIES_CFM_CB,
				   PBDL_CALLBACK_ID);
	parms = &data->parms.pbdl_get_capabilities_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->pb_type = pb_type;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PBDL_GET_CAPABILITIES_CFM_CB);
}

BTHRESULT BT_APPL_PBDL_Prepare_CFM_Callback(IN const BD_ADDRESS bd_addr,
					    IN u8 pb_type)
{
	callback_data *data;
	pbdl_prepare_cfm_t *parms;

	DEBUG("bd_addr %s pb_type 0x%02X",
	      bd_addr_to_str(bd_addr), pb_type);

	data = callback_data_alloc(BT_APPL_PBDL_PREPARE_CFM_CB,
				   PBDL_CALLBACK_ID);
	parms = &data->parms.pbdl_prepare_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->pb_type = pb_type;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PBDL_PREPARE_CFM_CB);
}

BTHRESULT BT_APPL_PBDL_Start_CFM_Callback(IN const BD_ADDRESS bd_addr,
					  IN u8 status,
					  IN u16 pb_number,
					  IN u8 new_missed_call)
{
	callback_data *data;
	pbdl_start_cfm_t *parms;

	DEBUG("bd_addr %s status %u pb_number %u new_missed_call %u",
	      bd_addr_to_str(bd_addr), status, pb_number,
	      new_missed_call);

	data = callback_data_alloc(BT_APPL_PBDL_START_CFM_CB,
				   PBDL_CALLBACK_ID);
	parms = &data->parms.pbdl_start_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->pb_number = pb_number;
	parms->new_missed_call = new_missed_call;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PBDL_START_CFM_CB);
}

BTHRESULT BT_APPL_PBDL_Stop_CFM_Callback(IN const BD_ADDRESS bd_addr,
					 IN u8 status)
{
	callback_data *data;
	pbdl_stop_cfm_t *parms;

	DEBUG("bd_addr %s status %u",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_PBDL_STOP_CFM_CB,
				   PBDL_CALLBACK_ID);
	parms = &data->parms.pbdl_stop_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PBDL_STOP_CFM_CB);
}

BTHRESULT BT_APPL_PBDL_Data_IND_Callback(IN const BD_ADDRESS bd_addr,
					 IN u8 data_type,
					 IN u32 data_len,
					 IN u8 *data)
{
	callback_data *data_info;
	pbdl_data_ind_t *parms;

	DEBUG("bd_addr %s data_type %u data_len %lu",
	      bd_addr_to_str(bd_addr), data_type, data_len);

	/* check that the size of the GByteArray is not exceeded */
	g_assert(data_len < G_MAXUINT);

	data_info = callback_data_alloc(BT_APPL_PBDL_DATA_IND_CB,
					PBDL_CALLBACK_ID);
	parms = &data_info->parms.pbdl_data_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->data_type = data_type;

	/* create a new byte array for this phonebook data */
	parms->data = g_byte_array_new();

	/* copy the phonebook data into the GByteArray for storage */
	if (data != NULL) {
		parms->data = g_byte_array_append(parms->data, data, data_len);
	} else if (data_len != 0) {
		DEBUG_ERROR("data is NULL but data_len is %lu", data_len);
	}

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_PBDL_DATA_IND_CB);
}

BTHRESULT BT_APPL_PBDL_Data_Complete_IND_Callback(IN const BD_ADDRESS bd_addr,
						  IN u8 status,
						  IN u16 pb_number)
{
	callback_data *data_info;
	pbdl_complete_ind_t *parms;

	DEBUG("bd_addr %s status %u pb_number %u",
	      bd_addr_to_str(bd_addr), status, pb_number);

	data_info = callback_data_alloc(BT_APPL_PBDL_DATA_COMPLETE_IND_CB,
					PBDL_CALLBACK_ID);
	parms = &data_info->parms.pbdl_complete_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->pb_number = pb_number;

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_PBDL_DATA_COMPLETE_IND_CB);
}

BTHRESULT BT_APPL_PBDL_Calendar_Complete_IND_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 status)
{
	callback_data *data_info;
	pbdl_calender_complete_ind_t *parms;

	DEBUG("bd_addr %s status %u",
	      bd_addr_to_str(bd_addr), status);

	data_info = callback_data_alloc(BT_APPL_PBDL_CALENDAR_COMPLETE_IND_CB,
					PBDL_CALLBACK_ID);
	parms = &data_info->parms.pbdl_calender_complete_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_PBDL_CALENDAR_COMPLETE_IND_CB);
}

BTHRESULT BT_APPL_PBDL_Complete_CFM_Callback(IN const BD_ADDRESS bd_addr,
					     IN u8 status)
{
	callback_data *data;
	pbdl_complete_cfm_t *parms;

	DEBUG("bd_addr %s status %u",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_PBDL_COMPLETE_CFM_CB,
				   PBDL_CALLBACK_ID);
	parms = &data->parms.pbdl_complete_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_PBDL_COMPLETE_CFM_CB);
}

BTHRESULT BT_APPL_PBDL_Con_Lost_IND_Callback(IN const BD_ADDRESS bd_addr)
{
	callback_data *data_info;
	pbdl_con_lost_ind_t *parms;

	DEBUG("bd_addr %s", bd_addr_to_str(bd_addr));

	data_info = callback_data_alloc(BT_APPL_PBDL_CON_LOST_IND_CB,
					PBDL_CALLBACK_ID);
	parms = &data_info->parms.pbdl_con_lost_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_PBDL_CON_LOST_IND_CB);
}

BTHRESULT BT_APPL_PBDL_Status_Chg_IND_Callback(IN const BD_ADDRESS bd_addr,
								IN u8 status)
{
	callback_data *data_info;
	pbdl_status_chg_ind_t *parms;

	DEBUG("bd_addr %s status %u",
	      bd_addr_to_str(bd_addr), status);

	data_info = callback_data_alloc(BT_APPL_PBDL_STATUS_CHG_IND_CB,
					PBDL_CALLBACK_ID);
	parms = &data_info->parms.pbdl_status_chg_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data_info);

	return write_to_self_pipe(BT_APPL_PBDL_STATUS_CHG_IND_CB);
}

void BT_APPL_PBDL_Place_Holder_A_Callback(void)
{
	DEBUG("BT_APPL_PBDL_Place_Holder_A_Callback");
}

void BT_APPL_PBDL_Place_Holder_B_Callback(void)
{
	DEBUG("BT_APPL_PBDL_Place_Holder_B_Callback");
}

void BT_APPL_PBDL_Place_Holder_C_Callback(void)
{
	DEBUG("BT_APPL_PBDL_Place_Holder_C_Callback");
}

/******************/
/* MAP Callbacks  */
/******************/

BTHRESULT BT_APPL_MAP_Get_Capabilities_CFM_Callback(IN const BD_ADDRESS bd_addr,
					IN u8 status,
					IN MsgCapabilityPtr_t msg_capability)
{
	callback_data *data;
	map_get_capabilities_cfm_t *parms;
	gint i;

	g_assert(msg_capability != NULL);

	DEBUG("bd_addr %s status %d ServiceNum %d",
		bd_addr_to_str(bd_addr), status,
		msg_capability->ucServiceNum);

	data = callback_data_alloc(BT_APPL_MAP_GET_CAPABILITIES_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_get_capabilities_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	parms->msg_capability =
		g_ptr_array_sized_new(msg_capability->ucServiceNum);
	for (i = 0; i < msg_capability->ucServiceNum; i++) {
		GValue *value;
		gchar *service_name;

		value = g_new0(GValue, 1);
		g_value_init(value, DBUS_STRUCT_STRING_UCHAR);
		g_value_set_boxed_take_ownership(value,
			dbus_g_type_specialized_construct(DBUS_STRUCT_STRING_UCHAR));
		service_name = g_strndup((gchar *)
			(msg_capability->astServiceInfo[i].pucServiceName),
			msg_capability->astServiceInfo[i].ucServiceNameLen);
		dbus_g_type_struct_set(value,
			0, service_name,
			1, msg_capability->astServiceInfo[i].ucMsgType,
			G_MAXUINT);
		g_ptr_array_add(parms->msg_capability, g_value_get_boxed(value));
		g_free(value);
		g_free(service_name);
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_GET_CAPABILITIES_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Start_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 status)
{
	callback_data *data;
	map_start_cfm_t *parms;

	DEBUG("bd_addr %s status %d",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_MAP_START_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_start_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_START_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Active_Instance_IND_Callback(IN const BD_ADDRESS bd_addr,
							IN u8 sms_name_len,
							IN const u8 *sms_name,
							IN u8 email_name_len,
							IN const u8 *email_name)
{
	callback_data *data;
	map_active_instance_ind_t *parms;

	DEBUG("bd_addr %s sms_name \"%s\" email_name \"%s\"",
	      bd_addr_to_str(bd_addr),
	      sms_name ? (gchar *) sms_name : "",
	      email_name ? (gchar *) email_name : "");

	data = callback_data_alloc(BT_APPL_MAP_ACTIVE_INSTANCE_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_active_instance_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->sms_name = g_strndup((gchar *) sms_name, sms_name_len);
	parms->email_name = g_strndup((gchar *) email_name, email_name_len);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_ACTIVE_INSTANCE_IND_CB);
}

BTHRESULT BT_APPL_MAP_Notify_Status_IND_Callback(IN const BD_ADDRESS bd_addr,
							IN u8 status)
{
	callback_data *data;
	map_notify_status_ind_t *parms;

	DEBUG("bd_addr %s status %d",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_MAP_NOTIFY_STATUS_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_notify_status_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_NOTIFY_STATUS_IND_CB);
}

BTHRESULT BT_APPL_MAP_Notify_Regist_Status_IND_Callback(
						IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status)
{
	callback_data *data;
	map_notify_regist_status_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" service_name_len %u status %u",
		bd_addr_to_str(bd_addr),
		service_name ? (gchar *) service_name : "",
		service_name_len, status);

	data = callback_data_alloc(BT_APPL_MAP_NOTIFY_REGIST_STATUS_IND_CB,
					MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_notify_regist_status_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_NOTIFY_REGIST_STATUS_IND_CB);
}

BTHRESULT BT_APPL_MAP_Stop_CFM_Callback(IN const BD_ADDRESS bd_addr,
					IN u8 status)
{
	callback_data *data;
	map_stop_cfm_t *parms;

	DEBUG("bd_addr %s status %d",
	      bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_MAP_STOP_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_stop_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_STOP_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Chg_Instance_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 status)
{
	callback_data *data;
	map_chg_instance_cfm_t *parms;

	DEBUG("bd_addr %s status %d", bd_addr_to_str(bd_addr), status);

	data = callback_data_alloc(BT_APPL_MAP_CHG_INSTANCE_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_chg_instance_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_CHG_INSTANCE_CFM_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_Update_Inbox_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status)
{
	callback_data *data;
	map_update_inbox_cfm_t *parms;
	(void) service_name_len;
	DEBUG("bd_addr %s status %d service_name \"%s\" service_name_len %d",
	      bd_addr_to_str(bd_addr), status,
	      service_name ? (gchar *) service_name : "",
	      service_name ? strlen((gchar *) service_name) : 0);

	data = callback_data_alloc(BT_APPL_MAP_UPDATE_INBOX_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_update_inbox_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_UPDATE_INBOX_CFM_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_List_Folder_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status,
						IN u8 accessible,
						IN u32 folder_num)
{
	callback_data *data;
	map_list_folder_cfm_t *parms;
	(void) service_name_len;

	DEBUG("bd_addr %s service_name \"%s\" status %u accessible %u "
	      "folder_num %lu", bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", status, accessible,
	      folder_num);

	data = callback_data_alloc(BT_APPL_MAP_LIST_FOLDER_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_folder_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->accessible = accessible;
	parms->folder_num = folder_num;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_FOLDER_CFM_CB);
}

BTHRESULT BT_APPL_MAP_List_Folder_IND_Callback(IN const BD_ADDRESS bd_addr,
				IN u8 service_name_len,
				IN const u8 *service_name,
				IN const FolderListingPtr_t folder_listing)
{
	callback_data *data;
	map_list_folder_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" folder_name \"%s\"",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "",
	      folder_listing ?
	      (folder_listing->pucFolderName ?
	      (gchar *) folder_listing->pucFolderName : "") : "");

	data = callback_data_alloc(BT_APPL_MAP_LIST_FOLDER_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_folder_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);

	parms->folder_listing = g_new0(gchar *, 2);
	if (folder_listing)
		parms->folder_listing[0] =
			g_strndup((gchar *) folder_listing->pucFolderName,
			folder_listing->usFolderNameLen);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_FOLDER_IND_CB);
}

BTHRESULT BT_APPL_MAP_List_Folder_Comp_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u16 total_size,
						IN u8 status)
{
	callback_data *data;
	map_list_folder_comp_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" total_size %u status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", total_size, status);

	data = callback_data_alloc(BT_APPL_MAP_LIST_FOLDER_COMP_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_folder_comp_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->total_size = total_size;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_FOLDER_COMP_IND_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_List_Msg_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status,
	IN u32 message_num)
{
	callback_data *data;
	map_list_msg_cfm_t *parms;
	(void) service_name_len;

	DEBUG("bd_addr %s service_name \"%s\" status %u "
	      "message_num %lu", bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", status,
	      message_num);

	data = callback_data_alloc(BT_APPL_MAP_LIST_MSG_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_msg_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;
	parms->message_num = message_num;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_MSG_CFM_CB);
}

BTHRESULT BT_APPL_MAP_List_Msg_IND_Callback(IN const BD_ADDRESS bd_addr,
					IN u8 service_name_len,
					IN const u8 *service_name,
					IN const MsgListingPtr_t msg_listing)
{
	callback_data *data;
	map_list_msg_data_ind_t *parms;
	(void) service_name_len;
	DEBUG("bd_addr %s service_name \"%s\" handle \"%s\"",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "",
	      handle_to_str(msg_listing->pucHandle, msg_listing->ucHandleLen));

	data = callback_data_alloc(BT_APPL_MAP_LIST_MSG_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_msg_data_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));

	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);

	parms->handle = g_byte_array_new();
	if (msg_listing->pucHandle != NULL) {
		parms->handle = g_byte_array_append(parms->handle,
					msg_listing->pucHandle,
					msg_listing->ucHandleLen);
	} else if (msg_listing->ucHandleLen != 0) {
		DEBUG_ERROR("msg_listing->pucHandle is NULL but "
			"msg_listing->ucHandleLen is %u",
			msg_listing->ucHandleLen);
	}

	parms->msg_subject = g_strndup((gchar *) msg_listing->pucSubject,
		msg_listing->ucSubjectLen);

	parms->msg_date_time = g_strndup((gchar *) msg_listing->pucDatetime,
		msg_listing->ucDatetimeLen);

	parms->msg_sender_name = g_strndup((gchar *) msg_listing->pucSenderName,
		msg_listing->ucSenderNameLen);

	parms->msg_sender_address =
			g_strndup((gchar *) msg_listing->pucSenderAddress,
					msg_listing->ucSenderAddressLen);

	parms->msg_replyto_address =
			g_strndup((gchar *) msg_listing->pucReplytoAddress,
					msg_listing->ucReplytoAddressLen);

	parms->msg_recipient_name =
			g_strndup((gchar *) msg_listing->pucRecipientName,
					msg_listing->ucRecipientNameLen);

	parms->msg_recipient_address =
			g_strndup((gchar *) msg_listing->pucRecipientAddress,
					msg_listing->ucRecipientAddressLen);

	parms->msg_type = msg_listing->ucType;
	parms->msg_size = msg_listing->ulSize;
	parms->msg_text_flag = msg_listing->ucText;
	parms->msg_reception_status = msg_listing->ucReceptionStatus;
	parms->msg_attach_size = msg_listing->ulAttachSize;
	parms->msg_priority = msg_listing->ucPriority;
	parms->msg_read = msg_listing->ucRead;
	parms->msg_sent = msg_listing->ucSent;
	parms->msg_protect = msg_listing->ucProtected;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_MSG_IND_CB);
}

BTHRESULT BT_APPL_MAP_List_Msg_Comp_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u16 total_size,
						IN u8 status)
{
	callback_data *data;
	map_list_msg_comp_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" total_size %u status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", total_size, status);

	data = callback_data_alloc(BT_APPL_MAP_LIST_MSG_COMP_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_list_msg_comp_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->total_size = total_size;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_LIST_MSG_COMP_IND_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_Read_Msg_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status,
						IN const ReadMsgInfoPtr_t ReadMsgInfo)
{
	callback_data *data;
	map_read_msg_cfm_t *parms;
	gint i;
	(void) service_name_len;
	DEBUG("bd_addr %s service_name \"%s\" status %u",
		bd_addr_to_str(bd_addr),
		service_name ? (gchar *) service_name : "",
		status);

	data = callback_data_alloc(BT_APPL_MAP_READ_MSG_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_read_msg_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	/* If status indicates failure then msg_info is a NULL pointer */
	if ((ReadMsgInfo != NULL) && (ReadMsgInfo->pstMsgInfo != NULL)) {
		gchar *name;
		GValue *value;
		gchar *N, *FN, *TEL, *EMAIL;

		parms->read_status = ReadMsgInfo->pstMsgInfo->ucReadStatus;
		parms->type = ReadMsgInfo->pstMsgInfo->ucType;
		parms->folder_name =
			g_strndup((gchar *) ReadMsgInfo->pstMsgInfo->stFolder.pucFolderName,
					ReadMsgInfo->pstMsgInfo->stFolder.usFolderNameLen);
		parms->msg_len = ReadMsgInfo->pstMsgInfo->ulMsgLen;
		parms->language = ReadMsgInfo->pstMsgInfo->ucLanguage;
		parms->encoding = ReadMsgInfo->pstMsgInfo->ucEncoding;

		N = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stOriginator.pucN),
				ReadMsgInfo->pstMsgInfo->stOriginator.ucNLen);
		FN = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stOriginator.pucFN),
				ReadMsgInfo->pstMsgInfo->stOriginator.ucFNLen);
		TEL = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stOriginator.pucTEL),
				ReadMsgInfo->pstMsgInfo->stOriginator.ucTELLen);
		EMAIL = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stOriginator.pucEMAIL),
				ReadMsgInfo->pstMsgInfo->stOriginator.ucEMAILLen);

		DEBUG("Sender N \"%s\" FN \"%s\" TEL \"%s\" EMAIL \"%s\"",
		      N, FN, TEL, EMAIL);

		parms->sender = g_value_array_new(4);

		g_value_array_append(parms->sender, NULL);
		g_value_init(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), G_TYPE_STRING);
		g_value_set_string(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), N);

		g_value_array_append(parms->sender, NULL);
		g_value_init(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), G_TYPE_STRING);
		g_value_set_string(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), FN);

		g_value_array_append(parms->sender, NULL);
		g_value_init(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), G_TYPE_STRING);
		g_value_set_string(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), TEL);

		g_value_array_append(parms->sender, NULL);
		g_value_init(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), G_TYPE_STRING);
		g_value_set_string(g_value_array_get_nth(parms->sender,
			(parms->sender)->n_values - 1), EMAIL);

		g_free(N);
		g_free(FN);
		g_free(TEL);
		g_free(EMAIL);

		parms->recipient = g_ptr_array_sized_new(ReadMsgInfo->pstMsgInfo->ucReceiverNum);
		DEBUG("ReceiverNum %u", ReadMsgInfo->pstMsgInfo->ucReceiverNum);

		for (i = 0; i < ReadMsgInfo->pstMsgInfo->ucReceiverNum; i++) {
			value = g_new0(GValue, 1);
			g_value_init(value, DBUS_STRUCT_STRING_STRING_STRING_STRING_UCHAR);
			g_value_set_boxed_take_ownership(value,
				dbus_g_type_specialized_construct(
				DBUS_STRUCT_STRING_STRING_STRING_STRING_UCHAR));

			N = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stRecipient[i].pucN),
					ReadMsgInfo->pstMsgInfo->stRecipient[i].ucNLen);
			FN = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stRecipient[i].pucFN),
					ReadMsgInfo->pstMsgInfo->stRecipient[i].ucFNLen);
			TEL = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stRecipient[i].pucTEL),
					ReadMsgInfo->pstMsgInfo->stRecipient[i].ucTELLen);
			EMAIL = g_strndup((gchar *) (ReadMsgInfo->pstMsgInfo->stRecipient[i].pucEMAIL),
					ReadMsgInfo->pstMsgInfo->stRecipient[i].ucEMAILLen);

			dbus_g_type_struct_set(value,
					0, N,
					1, FN,
					2, TEL,
					3, EMAIL,
					4, ReadMsgInfo->pstMsgInfo->stRecipient[i].ucBENVLevel,
					G_MAXUINT);

			g_ptr_array_add(parms->recipient,
						g_value_get_boxed(value));

			g_free(value);
			g_free(N);
			g_free(FN);
			g_free(TEL);
			g_free(EMAIL);
		}

		parms->attachment_list =
				g_ptr_array_sized_new(ReadMsgInfo->pstMsgInfo->ucAttachNum);

		DEBUG("AttachNum %u", ReadMsgInfo->pstMsgInfo->ucAttachNum);

		for (i = 0; i < ReadMsgInfo->pstMsgInfo->ucAttachNum; i++) {
			value = g_new0(GValue, 1);
			g_value_init(value, DBUS_STRUCT_STRING_UINT);
			g_value_set_boxed_take_ownership(value,
					dbus_g_type_specialized_construct(
						DBUS_STRUCT_STRING_UINT));

			name = g_strndup(
				(gchar *)(ReadMsgInfo->pstMsgInfo->stAttachmentList[i].ucName),
				ReadMsgInfo->pstMsgInfo->stAttachmentList[i].usNameLen);

			dbus_g_type_struct_set(value,
					0, name,
					1, ReadMsgInfo->pstMsgInfo->stAttachmentList[i].ulSize,
					G_MAXUINT);

			g_ptr_array_add(parms->attachment_list,
						g_value_get_boxed(value));

			g_free(value);
			g_free(name);
		}
	} else {
		DEBUG("handle msg_info being NULL");

		parms->read_status = 0;
		parms->type = 0;
		parms->folder_name = NULL;
		parms->msg_len = 0;
		parms->language = 0;
		parms->encoding = 0;			

		/*
		 * allocate arrays to prevent segmentation faults
		 * in the glib dbus handling in sending the D-bus reply
		 */ 
		parms->sender = g_value_array_new(4);
		parms->recipient = g_ptr_array_new();
		parms->attachment_list = g_ptr_array_new();
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_READ_MSG_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Read_Msg_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 data_flag,
						IN u16 length,
						IN const u8 *msg_data)
{
	callback_data *data;
	map_read_msg_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" data_flag %u length %u"
		" msg_data %p", bd_addr_to_str(bd_addr),
		service_name ? (gchar *) service_name : "",
		data_flag, length, msg_data);

	data = callback_data_alloc(BT_APPL_MAP_READ_MSG_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_read_msg_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->data_flag = data_flag;

	parms->msg_data = g_byte_array_new();
	if (msg_data != NULL) {
		parms->msg_data = g_byte_array_append(parms->msg_data,
							msg_data,
							length);
	} else if (length != 0) {
		DEBUG_ERROR("msg_data is NULL but length is %u", length);
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_READ_MSG_IND_CB);
}

BTHRESULT BT_APPL_MAP_Read_Msg_Comp_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name)
{
	callback_data *data;
	map_read_msg_comp_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\"",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "");

	data = callback_data_alloc(BT_APPL_MAP_READ_MSG_COMP_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_read_msg_comp_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_READ_MSG_COMP_IND_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_Set_Msg_Status_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status)
{
	callback_data *data;
	map_set_msg_status_cfm_t *parms;
	(void) service_name_len;
	DEBUG("bd_addr %s service_name \"%s\" status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", status);

	data = callback_data_alloc(BT_APPL_MAP_SET_MSG_STATUS_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_set_msg_status_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_SET_MSG_STATUS_CFM_CB);
}

/* note service_name is ignored */
BTHRESULT BT_APPL_MAP_Upload_CFM_Callback(IN const BD_ADDRESS bd_addr,
					IN u8 service_name_len,
					IN const u8 *service_name,
					IN u8 status)
{
	callback_data *data;
	map_upload_cfm_t *parms;
	(void) service_name_len;

	DEBUG("bd_addr %s service_name \"%s\" status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", status);

	data = callback_data_alloc(BT_APPL_MAP_UPLOAD_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_upload_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_UPLOAD_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Upload_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status)
{
	callback_data *data;
	map_upload_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "",
	      status);

	data = callback_data_alloc(BT_APPL_MAP_UPLOAD_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_upload_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_UPLOAD_IND_CB);
}

/* note service name is ignored */
BTHRESULT BT_APPL_MAP_Abort_CFM_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 status)
{
	callback_data *data;
	map_abort_cfm_t *parms;
	(void) service_name_len;

	DEBUG("bd_addr %s service_name \"%s\" status %u",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "", status);

	data = callback_data_alloc(BT_APPL_MAP_ABORT_CFM_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_abort_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_ABORT_CFM_CB);
}

BTHRESULT BT_APPL_MAP_Notify_IND_Callback(IN const BD_ADDRESS bd_addr,
					IN u8 service_name_len,
					IN const u8 *service_name,
					IN const NotifyInfoPtr_t notify_info)
{
	callback_data *data;
	map_notify_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\"",
	      bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "");

	data = callback_data_alloc(BT_APPL_MAP_NOTIFY_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_notify_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	if (notify_info) {
		parms->type = notify_info->ucType;
		parms->handle =
			g_byte_array_sized_new(notify_info->ucHandleLen);
		if (notify_info->pucHandle != NULL) {
			parms->handle = g_byte_array_append(parms->handle,
						notify_info->pucHandle,
						notify_info->ucHandleLen);
		} else if (notify_info->ucHandleLen != 0) {
			DEBUG_ERROR("notify_info->pucHandle is NULL "
				"but notify_info->ucHandleLen is %u",
				notify_info->ucHandleLen);
		}
		parms->folder = g_strndup((gchar *) notify_info->pucFolder,
			notify_info->usFolderLen);
		parms->old_folder =
			g_strndup((gchar *) notify_info->pucOldFolder,
			notify_info->usOldFolderLen);
		parms->msg_type = notify_info->ucMsgType;
	}

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_NOTIFY_IND_CB);
}

BTHRESULT BT_APPL_MAP_Error_Report_IND_Callback(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8 *service_name,
						IN u8 operation,
						IN u8 error_code)
{
	callback_data *data;
	map_error_report_ind_t *parms;

	DEBUG("bd_addr %s service_name \"%s\" operation %u "
	      "error_code %u", bd_addr_to_str(bd_addr),
	      service_name ? (gchar *) service_name : "",
	      operation, error_code);

	data = callback_data_alloc(BT_APPL_MAP_ERROR_REPORT_IND_CB,
				   MESSAGE_CALLBACK_ID);
	parms = &data->parms.map_error_report_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service_name =
		g_strndup((gchar *) service_name, service_name_len);
	parms->operation = operation;
	parms->error_code = error_code;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_MAP_ERROR_REPORT_IND_CB);
}

void BT_APPL_MSDL_Place_Holder_A_Callback(void)
{
	DEBUG("BT_APPL_MSDL_Place_Holder_A_Callback");
}

void BT_APPL_MSDL_Place_Holder_B_Callback(void)
{
	DEBUG("BT_APPL_MSDL_Place_Holder_B_Callback");
}

void BT_APPL_MSDL_Place_Holder_C_Callback(void)
{
	DEBUG("BT_APPL_MSDL_Place_Holder_C_Callback");
}

/***********************/
/* Internet Callbacks  */
/***********************/

BTHRESULT BT_APPL_Create_Device_CFM_Callback(IN const BD_ADDRESS bd_addr,
					     IN u8 service,
					     IN u8 status,
					     IN u8 device_name_len,
					     IN u8 *device_name)
{
	callback_data *data;
	nap_create_device_cfm_t *parms;

	DEBUG("bd_addr %s service %u status %u device name \"%s\"",
	      bd_addr_to_str(bd_addr), service, status,
	      device_name != NULL ? (gchar *) device_name : "");

	data = callback_data_alloc(BT_APPL_CREATE_DEVICE_CFM_CB,
							DEVICE_CALLBACK_ID);
	parms = &data->parms.nap_create_device_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service = service;
	parms->status = status;
	parms->device_name = g_strndup((gchar *) device_name, device_name_len);

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_CREATE_DEVICE_CFM_CB);
}

BTHRESULT BT_APPL_Destroy_Device_CFM_Callback(IN const BD_ADDRESS bd_addr,
							     IN u8 service,
							     IN u8 status)
{
	callback_data *data;
	nap_destroy_device_cfm_t *parms;

	DEBUG("bd_addr %s service %u status %u",
	      bd_addr_to_str(bd_addr), service, status);

	data = callback_data_alloc(BT_APPL_DESTROY_DEVICE_CFM_CB,
							DEVICE_CALLBACK_ID);
	parms = &data->parms.nap_destroy_device_cfm_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service = service;
	parms->status = status;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_DESTROY_DEVICE_CFM_CB);
}

BTHRESULT BT_APPL_Destroy_Device_IND_Callback(IN const BD_ADDRESS bd_addr,
							     IN u8 service)
{
	callback_data *data;
	nap_destroy_device_ind_t *parms;

	DEBUG("bd_addr %s service %u",
	      bd_addr_to_str(bd_addr), service);

	data = callback_data_alloc(BT_APPL_DESTROY_DEVICE_IND_CB,
							DEVICE_CALLBACK_ID);
	parms = &data->parms.nap_destroy_device_ind_parms;
	memcpy(parms->bd_addr, bd_addr, sizeof(BD_ADDRESS));
	parms->service = service;

	g_async_queue_push(callback_queue, data);

	return write_to_self_pipe(BT_APPL_DESTROY_DEVICE_IND_CB);
}

void BT_APPL_Device_Place_Holder_A_Callback(void)
{
	DEBUG("BT_APPL_Device_Place_Holder_A_Callback");
}

void BT_APPL_Device_Place_Holder_B_Callback(void)
{
	DEBUG("BT_APPL_Device_Place_Holder_B_Callback");
}

void BT_APPL_Device_Place_Holder_C_Callback(void)
{
	DEBUG("BT_APPL_Device_Place_Holder_C_Callback");
}


/* General Callbacks */
static GeneralCallback_t general_callbacks = {
	BT_APPL_Get_Device_Name_CFM_Callback,
	BT_APPL_Role_Change_IND_Callback,
	BT_APPL_Request_PIN_IND_Callback,
	BT_APPL_SSP_Numeric_Confirm_IND_Callback,
	BT_APPL_SSP_Passkey_Notification_IND_Callback,
	BT_APPL_Pairing_Complete_IND_Callback,
	BT_APPL_Service_Type_IND_Callback,
	BT_APPL_Linkkey_Request_IND_Callback,
	BT_APPL_Connection_Request_IND_Callback,
	BT_APPL_Reset_Event_IND_Callback,
	BT_APPL_Connection_Complete_IND_Callback,
	BT_APPL_Disconnection_IND_Callback,
	BT_APPL_Authentication_IND_Callback,
	BT_APPL_TestMode_CFM_Callback,
	BT_APPL_TestMode_Complete_IND_Callback,
	BT_APPL_LinkQuality_CFM_Callback,
	BT_APPL_SPP_Capabilities_IND_Callback,
	BT_APPL_PnPInfo_IND_Callback,
	BT_APPL_Inquiry_IND_Callback,
	BT_APPL_Inquiry_Complete_IND_Callback,
	BT_APPL_General_Place_Holder_A_Callback,
	BT_APPL_General_Place_Holder_B_Callback,
	BT_APPL_General_Place_Holder_C_Callback,
};

/* HFP Callbacks */
static HFPCallback_t hfp_callbacks = {
	BT_APPL_Call_Status_IND_Callback,
	BT_APPL_Phone_Status_IND_Callback,
	BT_APPL_HF_Dial_CFM_Callback,
	BT_APPL_HF_Redial_CFM_Callback,
	BT_APPL_HF_Mem_Dial_CFM_Callback,
	BT_APPL_HF_Terminate_Call_CFM_Callback,
	BT_APPL_HF_Terminate_All_CFM_Callback,
	BT_APPL_HF_Audio_Transfer_CFM_Callback,
	BT_APPL_HF_Setup_Audio_IND_Callback,
	BT_APPL_HF_Release_Audio_IND_Callback,
	BT_APPL_HF_Send_DTMF_CFM_Callback,
	BT_APPL_HF_Incoming_Call_Act_CFM_Callback,
	BT_APPL_HF_Ring_IND_Callback,
	BT_APPL_HF_Release_Inactive_Call_CFM_Callback,
	BT_APPL_HF_Release_Active_Call_CFM_Callback,
	BT_APPL_HF_Swap_Call_CFM_Callback,
	BT_APPL_HF_Add_Call_CFM_Callback,
	BT_APPL_HF_Release_Spec_Call_CFM_Callback,
	BT_APPL_HF_Hold_Excep_Call_CFM_Callback,
	BT_APPL_HF_Exp_Call_Trans_CFM_Callback,
	BT_APPL_HF_Subscriber_CFM_Callback,
	BT_APPL_HF_Manufacturer_ID_CFM_Callback,
	BT_APPL_HF_Model_ID_CFM_Callback,
	BT_APPL_HF_Revision_ID_CFM_Callback,
	BT_APPL_HF_Voice_Recognition_Activation_CFM_Callback,
	BT_APPL_HF_Voice_Recognition_Activation_IND_Callback,
	BT_APPL_SIRI_XAPL_CFM_Callback,
	BT_APPL_SIRI_NrStat_CFM_Callback,
	BT_APPL_SIRI_NrStat_IND_Callback,
	BT_APPL_SIRI_AplSiri_IND_Callback,
	BT_APPL_SIRI_Efm_CFM_Callback,
	BT_APPL_HF_Release_Waiting_Call_CFM_Callback,
	BT_APPL_HF_Control_Indicator_CFM_Callback,
	BT_APPL_HF_Setup_Audio_WBS_IND_Callback,
	BT_APPL_HF_Place_Holder_A_Callback,
	BT_APPL_HF_Place_Holder_B_Callback,
	BT_APPL_HF_Place_Holder_B_Callback,

};

/* AVP Callbacks */
static AVPCallback_t avp_callbacks = {
	BT_APPL_AVP_Version_IND_Callback,
	BT_APPL_AVP_Ctrl_Cmd_CFM_Callback,
	BT_APPL_AVP_Ctrl_Cmd_IND_Callback,
	BT_APPL_AVP_Get_Capabilities_CFM_Callback,
	BT_APPL_AVP_List_Attr_CFM_Callback,
	BT_APPL_AVP_List_Val_CFM_Callback,
	BT_APPL_AVP_Set_Val_CFM_Callback,
	BT_APPL_AVP_Play_Status_IND_Callback,
	BT_APPL_AVP_Notify_Set_Complete_IND_Callback,
	BT_APPL_AVP_Notify_Chg_IND_Callback,
	BT_APPL_AVP_Set_Player_CFM_Callback,
	BT_APPL_AVP_Get_Metadata_Attr_CFM_Callback,
	BT_APPL_AVP_Get_Folder_Item_CFM_Callback,
	BT_APPL_AVP_Folder_Item_IND_Callback,
	BT_APPL_AVP_Media_Element_IND_Callback,
	BT_APPL_AVP_Get_Media_Player_Item_CFM_Callback,
	BT_APPL_AVP_Get_Media_Player_Item_IND_Callback,
	BT_APPL_AVP_Play_Item_CFM_Callback,
	BT_APPL_AVP_Change_Path_CFM_Callback,
	BT_APPL_Codec_and_Bitrate_IND_Callback,
	BT_APPL_AVP_Place_Holder_A_Callback,
	BT_APPL_AVP_Place_Holder_B_Callback,
	BT_APPL_AVP_Place_Holder_C_Callback,
};

/* PBDL Callbacks */
static PBDLCallback_t pbdl_callbacks = {
	BT_APPL_PBDL_Get_Capabilities_CFM_Callback,
	BT_APPL_PBDL_Prepare_CFM_Callback,
	BT_APPL_PBDL_Start_CFM_Callback,
	BT_APPL_PBDL_Stop_CFM_Callback,
	BT_APPL_PBDL_Data_IND_Callback,
	BT_APPL_PBDL_Data_Complete_IND_Callback,
	BT_APPL_PBDL_Calendar_Complete_IND_Callback,
	BT_APPL_PBDL_Complete_CFM_Callback,
	BT_APPL_PBDL_Con_Lost_IND_Callback,
	BT_APPL_PBDL_Status_Chg_IND_Callback,
	BT_APPL_PBDL_Place_Holder_A_Callback,
	BT_APPL_PBDL_Place_Holder_B_Callback,
	BT_APPL_PBDL_Place_Holder_C_Callback,
};

/* MAP Callbacks */
static MAPCallback_t map_callbacks = {
	BT_APPL_MAP_Get_Capabilities_CFM_Callback,
	BT_APPL_MAP_Start_CFM_Callback,
	BT_APPL_MAP_Active_Instance_IND_Callback,
	BT_APPL_MAP_Notify_Status_IND_Callback,
	BT_APPL_MAP_Notify_Regist_Status_IND_Callback,
	BT_APPL_MAP_Stop_CFM_Callback,
	BT_APPL_MAP_Chg_Instance_CFM_Callback,
	BT_APPL_MAP_Update_Inbox_CFM_Callback,
	BT_APPL_MAP_List_Folder_CFM_Callback,
	BT_APPL_MAP_List_Folder_IND_Callback,
	BT_APPL_MAP_List_Folder_Comp_IND_Callback,
	BT_APPL_MAP_List_Msg_CFM_Callback,
	BT_APPL_MAP_List_Msg_IND_Callback,
	BT_APPL_MAP_List_Msg_Comp_IND_Callback,
	BT_APPL_MAP_Read_Msg_CFM_Callback,
	BT_APPL_MAP_Read_Msg_IND_Callback,
	BT_APPL_MAP_Read_Msg_Comp_IND_Callback,
	BT_APPL_MAP_Set_Msg_Status_CFM_Callback,
	BT_APPL_MAP_Upload_CFM_Callback,
	BT_APPL_MAP_Upload_IND_Callback,
	BT_APPL_MAP_Abort_CFM_Callback,
	BT_APPL_MAP_Notify_IND_Callback,
	BT_APPL_MAP_Error_Report_IND_Callback,
	BT_APPL_MSDL_Place_Holder_A_Callback,
	BT_APPL_MSDL_Place_Holder_B_Callback,
	BT_APPL_MSDL_Place_Holder_C_Callback,
};

/* Internet Callbacks */
static DeviceCallback_t nap_callbacks = {
	BT_APPL_Create_Device_CFM_Callback,
	BT_APPL_Destroy_Device_CFM_Callback,
	BT_APPL_Destroy_Device_IND_Callback,
	BT_APPL_Device_Place_Holder_A_Callback,
	BT_APPL_Device_Place_Holder_A_Callback,
	BT_APPL_Device_Place_Holder_A_Callback,
};

void unregister_alps_callbacks(void)
{
	BT_APPL_Unregister_Callback_REQ(DEVICE_CALLBACK_ID);
	BT_APPL_Unregister_Callback_REQ(MESSAGE_CALLBACK_ID);
	BT_APPL_Unregister_Callback_REQ(PBDL_CALLBACK_ID);
	BT_APPL_Unregister_Callback_REQ(AVP_CALLBACK_ID);
	BT_APPL_Unregister_Callback_REQ(HFP_CALLBACK_ID);
	BT_APPL_Unregister_Callback_REQ(GENERAL_CALLBACK_ID);
}

BTHRESULT register_alps_callbacks(void)
{
	BTHRESULT err;
	CallbackFun_u callbacks_union;

	DEBUG_FUNC("Called");

	/* Register General Callbacks */
	callbacks_union.stGeneralCallback = general_callbacks;
	err = BT_APPL_Register_Callback_REQ(GENERAL_CALLBACK_ID,
						&callbacks_union);

	if (err == BT_APPL_SUCCESS) {
		callbacks_union.stHFPCallback = hfp_callbacks;
		err = BT_APPL_Register_Callback_REQ(HFP_CALLBACK_ID,
							&callbacks_union);
	}

	if (err == BT_APPL_SUCCESS) {
		callbacks_union.stAVPCallback = avp_callbacks;
		err = BT_APPL_Register_Callback_REQ(AVP_CALLBACK_ID,
							&callbacks_union);
	}

	if (err == BT_APPL_SUCCESS) {
		callbacks_union.stPBDLCallback = pbdl_callbacks;
		err = BT_APPL_Register_Callback_REQ(PBDL_CALLBACK_ID,
							&callbacks_union);
	}

	if (err == BT_APPL_SUCCESS) {
		callbacks_union.stMAPCallback = map_callbacks;
		err = BT_APPL_Register_Callback_REQ(MESSAGE_CALLBACK_ID,
							&callbacks_union);
	}

	if (err == BT_APPL_SUCCESS) {
		callbacks_union.stDeviceCallback = nap_callbacks;
		err = BT_APPL_Register_Callback_REQ(DEVICE_CALLBACK_ID,
							&callbacks_union);
	}

	if (err != BT_APPL_SUCCESS) {
		DEBUG_ERROR("BT_APPL_Register_Callback_REQ error %d: %s",
			   err, lookup_bthresult_str(err));
		unregister_alps_callbacks();
	}

	return err;
}

void shutdown_alps_callbacks(void)
{
	unregister_alps_callbacks();

	/* remove the watch for the pipe GIOChannel */
	remove_pipe_source();

	/* free the queue */
	if (callback_queue) {
		g_async_queue_unref(callback_queue);
		callback_queue = NULL;
	}

	/* free the GIOChannel */
	if (pipe_ch) {
		g_io_channel_unref(pipe_ch);
		pipe_ch = NULL;
	}

	/* close the pipe */
	if (pipe_fd[0])
		close(pipe_fd[0]);
	if (pipe_fd[1])
		close(pipe_fd[1]);
	pipe_fd[0] = pipe_fd[1] = 0;
}

/*
 * Initialize the ALPS callbacks.  We do everything here except actually
 * registering the callbacks with the ALPS stack, as that can only be done
 * after the ALPS stack has been initialized.
 */
int initialize_alps_callbacks(void)
{
	/* create the callback queue */
	callback_queue = g_async_queue_new();
	if (callback_queue == NULL) {
		DEBUG_ERROR("Failed to create callback queue");
		return -1;
	}

	/* create the callback pipe */
	if (pipe(&pipe_fd[0]) < 0) {
		DEBUG_ERROR("Failed to open pipe");
		return -1;
	}
	if ((fcntl(pipe_fd[0], F_SETFL, O_NONBLOCK) < 0)
	    || (fcntl(pipe_fd[1], F_SETFL, O_NONBLOCK) < 0)) {
		DEBUG_ERROR("Can't set flag on pipe fd");
		shutdown_alps_callbacks();
		return -1;
	}

	/* bind the read descriptor of the pipe to a GIOChannel */
	pipe_ch = g_io_channel_unix_new(pipe_fd[0]);
	if (pipe_ch == NULL) {
		DEBUG_ERROR("Can't create GIOChannel for pipe");
		shutdown_alps_callbacks();
		return -1;
	}
	if (g_io_channel_set_encoding(pipe_ch, NULL, NULL)
	    != G_IO_STATUS_NORMAL) {
		DEBUG_ERROR("Can't set encoding for pipe GIOChannel");
		shutdown_alps_callbacks();
		return -1;
	}

	/*
	 * Add a watch on the GIOChannel to wakeup the main loop when data is
	 * available to read.
	 */
	attach_pipe_source();

	return 0;
}

#ifdef USE_DUMMY_API
/* TEST CODE */

BTHRESULT BT_APPL_Phone_Status_IND_Callback_DUMMY(const BD_ADDRESS bd_addr,
						PhoneStatus_t *phone_status)
{
	/* call the real one */
	return BT_APPL_Phone_Status_IND_Callback(bd_addr, phone_status);
}

BTHRESULT BT_APPL_HF_Voice_Recognition_Activation_CFM_Callback_DUMMY(
						const BD_ADDRESS bd_addr,
						u8 activation,
						u8 status)
{
	/* call the real one */
	return BT_APPL_HF_Voice_Recognition_Activation_CFM_Callback(bd_addr,
								activation,
								status);
}

BTHRESULT BT_APPL_HF_Voice_Recognition_Activation_IND_Callback_DUMMY(
						const BD_ADDRESS bd_addr,
						u8 activation)
{
	/* call the real one */
	return BT_APPL_HF_Voice_Recognition_Activation_IND_Callback(bd_addr,
								activation);
}

BTHRESULT BT_APPL_Get_Device_Name_CFM_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 name_len,
						IN const u8 *name)
{
	/* call the real one */
	return BT_APPL_Get_Device_Name_CFM_Callback(bd_addr, name_len, name);
}

BTHRESULT BT_APPL_AVP_Get_Metadata_Attr_CFM_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u8 num_attributes,
						IN u16 size,
						IN u8 data[])
{
	/* call the real one */
	return BT_APPL_AVP_Get_Metadata_Attr_CFM_Callback(bd_addr, status,
							  num_attributes,
							  size,
							  data);
}

BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_CFM_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u16 num_items)
{
	/* call the real one */
	return BT_APPL_AVP_Get_Media_Player_Item_CFM_Callback(bd_addr, status,
							       num_items);
}

BTHRESULT BT_APPL_AVP_Get_Media_Player_Item_IND_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u16 size,
						IN u8 data[])
{
	/* call the real one */
	return BT_APPL_AVP_Get_Media_Player_Item_IND_Callback(bd_addr, status,
								size,
								data);
}

BTHRESULT BT_APPL_PBDL_Get_Capabilities_CFM_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u8 type)
{
	/* call the real one */
	return BT_APPL_PBDL_Get_Capabilities_CFM_Callback(bd_addr, status,
								type);
}

BTHRESULT BT_APPL_PBDL_Prepare_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
					    	  IN u8 type)
{
	/* call the real one */
	return BT_APPL_PBDL_Prepare_CFM_Callback(bd_addr, type);
}

BTHRESULT BT_APPL_PBDL_Start_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN u16 pb_number,
						IN u8 new_missed_call)
{
	/* call the real one */
	return BT_APPL_PBDL_Start_CFM_Callback(bd_addr, status, pb_number,
					new_missed_call);
}

BTHRESULT BT_APPL_PBDL_Stop_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
					       IN u8 status)
{
	/* call the real one */
	return BT_APPL_PBDL_Stop_CFM_Callback(bd_addr, status);
}

BTHRESULT BT_APPL_PBDL_Complete_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						   IN u8 status)
{
	/* call the real one */
	return BT_APPL_PBDL_Complete_CFM_Callback(bd_addr, status);
}

BTHRESULT BT_APPL_MAP_Get_Capabilities_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 status,
						IN MsgCapabilityPtr_t msg_capability)
{
	return BT_APPL_MAP_Get_Capabilities_CFM_Callback(bd_addr, status,
							msg_capability);
}

BTHRESULT BT_APPL_MAP_Start_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
					IN u8 status)
{
	return BT_APPL_MAP_Start_CFM_Callback(bd_addr, status);
}

BTHRESULT BT_APPL_MAP_Stop_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
					IN u8 status)
{
	return BT_APPL_MAP_Stop_CFM_Callback(bd_addr, status);
}

BTHRESULT BT_APPL_MAP_Chg_Instance_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
					IN u8 status)
{
	return BT_APPL_MAP_Chg_Instance_CFM_Callback(bd_addr, status);
}

BTHRESULT BT_APPL_MAP_Update_Inbox_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status)
{
	return BT_APPL_MAP_Update_Inbox_CFM_Callback(bd_addr, service_name_len,
						service_name, status);
}

BTHRESULT BT_APPL_MAP_List_Folder_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status,
						IN u8 accessible,
						IN u32 folder_num)
{
	return BT_APPL_MAP_List_Folder_CFM_Callback(bd_addr, service_name_len,
						service_name, status,
						accessible, folder_num);
}

BTHRESULT BT_APPL_MAP_List_Msg_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status,
						IN u32 message_num)
{
	return BT_APPL_MAP_List_Msg_CFM_Callback(bd_addr, service_name_len,
						service_name, status,
						message_num);
}

BTHRESULT BT_APPL_MAP_Read_Msg_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status,
						IN const ReadMsgInfoPtr_t msg_info)
{
	return BT_APPL_MAP_Read_Msg_CFM_Callback(bd_addr, service_name_len,
						service_name, status, msg_info);
}

BTHRESULT BT_APPL_MAP_Set_Msg_Status_CFM_Callback_DUMMY(
						IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status)
{
	return BT_APPL_MAP_Set_Msg_Status_CFM_Callback(bd_addr,
						service_name_len,
						service_name, status);
}

BTHRESULT BT_APPL_MAP_Abort_CFM_Callback_DUMMY(IN const BD_ADDRESS bd_addr,
						IN u8 service_name_len,
						IN const u8* service_name,
						IN u8 status)
{
	return BT_APPL_MAP_Abort_CFM_Callback(bd_addr, service_name_len,
						service_name, status);
}

BTHRESULT BT_APPL_Authentication_IND_Callback_DUMMY(IN const BD_ADDRESS bd_addr)
{
	return BT_APPL_Authentication_IND_Callback(bd_addr);
}

BTHRESULT BT_APPL_TestMode_CFM_Callback_DUMMY(IN u8 status)
{
	/* call the real one */
	return BT_APPL_TestMode_CFM_Callback(status);
}

BTHRESULT BT_APPL_LinkQuality_CFM_Callback_DUMMY(IN u8 status, IN s8 rssi,
							IN u8 link_quality)
{
	/* call the real one */
	return BT_APPL_LinkQuality_CFM_Callback(status, rssi, link_quality);
}

#endif
