/*
 * Copyright (C) 2012 Robert Bosch Car Multimedia GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <environment.h>
#include <asm/arch/system.h>
#include <config.h>
#include <mmc.h>

/* define device tree and kernel addresses in parallel NOR flash
 * according Gen3 Memory Map */
#define	KBYTE (1024)
#define	MBYTE (KBYTE * KBYTE)

#define BOOT_FLASH_DTB_OFFSET (4 * MBYTE)
#define BOOT_FLASH_KERNEL_OFFSET (BOOT_FLASH_DTB_OFFSET + (128 * KBYTE))

#define BOOT_FLASH_DTB_ADDR \
	(CONFIG_SYS_FLASH_BASE + BOOT_FLASH_DTB_OFFSET)
#define BOOT_FLASH_KERNEL_ADDR \
	(CONFIG_SYS_FLASH_BASE + BOOT_FLASH_KERNEL_OFFSET)

#define BOOT_FLASH_CHAIN_2_OFFSET (8 * MBYTE)

#define BOOT_ROOTFS_TEMPLATE "/dev/mmcblk1p%d rootwait ro"
#define BOOT_ROOTFS_CHAIN_1_PARTITION 1
#define BOOT_ROOTFS_CHAIN_2_PARTITION 2

/* define recovery initial RAM FS offsets in eMMC user area according
 * Gen3 Memory Map
 */
/* Addresses */
#define MMC_BLOCK_SIZE (512)
#define RECOVERY_INITRAMFS_OFFSET (16 * KBYTE)
#define RECOVERY_INITRAMFS_CHAIN_2_OFFSET (48 * MBYTE)
/* Number of MMC blocks */
#define RECOVERY_INITRAMFS_1 (RECOVERY_INITRAMFS_OFFSET / MMC_BLOCK_SIZE)
#define RECOVERY_INITRAMFS_2 \
	((RECOVERY_INITRAMFS_OFFSET + RECOVERY_INITRAMFS_CHAIN_2_OFFSET) \
	/ MMC_BLOCK_SIZE)

#define DEV_EMMC 1 /* eMMC is supposed to be device 1, always */

static const env_entry dld_env[] = {
	{"kernelargs", "setenv bootargs console=${console},${baudrate} "\
		"root=/dev/ram0 initrd=${initramfsaddr} "\
		"rdinit=/linuxrc rootwait rw oops=panic panic=1 "\
		"printk.time=1 ${xtargs} consoleblank=0 dnlid=${dnlid} "\
		"usbf_download=1"},
	{"initrd_high", "0xffffffff"},
	{"initramfsaddr", "0x13000000"},
	{"kerneladdr", "0x12000000"},
	{"dldboot", "echo Booting to download ...; "\
		"run kernelargs; "\
		"bootm ${kerneladdr} ${initramfsaddr} ${dtbaddr}"},
	{"bootcmd", "run dldboot"},
	{0, 0}
};

static const env_entry recovery_dld_env[] = {
	{"kernelargs", "setenv bootargs console=${console},${baudrate} "\
		"root=/dev/ram0 initrd=${initramfsaddr} "\
		"rdinit=/linuxrc rootwait rw oops=panic panic=1 "\
		"printk.time=1 ${xtargs} consoleblank=0 dnlid=${dnlid}"},
	{"initrd_high", "0xffffffff"},
	{"initramfsaddr", "0x13000000"},
	{"loadinitramfs", "mmc dev 1; " \
		"mmc read ${initramfsaddr} ${initramfssrc} 0x18000"},
	{"dldboot", "echo Booting to recovery download ...; "\
		"run kernelargs; "\
		"run loadinitramfs; "\
		"run loaddtb; "\
		"run loadkernel; "\
		"bootm ${kerneladdr} ${initramfsaddr} ${dtbaddr}"},
	{"bootcmd", "run dldboot"},
	{0, 0}
};

/* Contains all the environment vars which are */
/* common for all parallel NOR Boards          */
static const env_entry common_parallel_nor_flash[] = {
	{"dtbsize", "0x8000"},
	{"loadkernel", "echo Starting kernel from parallel flash ..."},
	{"loaddtb", "cp.b ${dtbsrc} ${dtbaddr} ${dtbsize}"},
	{"eraseenv", "protect off ${envaddr} +${envsize}; " \
		"erase ${envaddr} +${envsize}; " \
		"protect on ${envaddr} +${envsize}"},
	{"backupenv", "protect off ${envbackupaddr} +${envsize}; " \
		"erase ${envbackupaddr} +${envsize}; " \
		"cp.b ${envaddr} ${envbackupaddr} ${envsize}; " \
		"protect on ${envbackupaddr} +${envsize}"},
	{"restoreenv","protect off ${envaddr} +${envsize}; " \
		"erase ${envaddr} +${envsize}; " \
		"cp.b ${envbackupaddr} ${envaddr} ${envsize}; " \
		"protect on ${envaddr} +${envsize}"},
	{0, 0}
};

/* Contains all the environment vars which are */
/* common for all serial NOR Boards          */
static const env_entry common_serial_nor_flash[] = {
	{"kerneladdr", "0x12000000"},
	{"kernelsize", "0x1f00"},
	{"dtbsize", "0x40"},
	{"loadkernel", "mmc dev 1 ${bootpart};" \
		"mmc read ${kerneladdr} 0x100 ${kernelsize}"},
	{"loaddtb", "mmc dev 1 ${bootpart}; mmc read ${dtbaddr} 0 ${dtbsize}"},
	{"gen3boot", "echo Booting from Serial NOR and eMMC ...; " \
		"run kernelargs; run loadkernel; run loaddtb; " \
		"bootm ${kerneladdr} - ${dtbaddr}"},
	{"eraseenv", "sf probe; sf erase ${envaddr} ${envsize}"},
	{"backupenv", "sf probe; sf erase ${envbackupaddr} ${envsize}; " \
		"sf read ${loadaddr} ${envaddr} ${envsize}; " \
		"sf write ${loadaddr} ${envbackupaddr} ${envsize}"},
	{"restoreenv", "sf probe; sf erase ${envaddr} ${envsize}; " \
		"sf read ${loadaddr} ${envbackupaddr} ${envsize}; " \
		"sf write ${loadaddr} ${envaddr} ${envsize}"},
	{0, 0}
};

static void gen3_warn_env_not_overwritten(const char *env_name)
{
	static int header_printed;

	if (!header_printed) {
		printf("\nNOTICE! " \
			"Existing/saved environment variable(s) not overwritten:\n");
		header_printed = 1;
	}
	printf("    %s\n", env_name);
}

static void gen3_set_env(const char *name, const char *value,
				int overwrite)
{
	const char *former_value = getenv(name);

	/* overwrite existing settings only if forced to */
	if (!former_value || overwrite)
		setenv(name, value);

	/* warn if existing/saved variable is not overwritten */
	if (former_value && !overwrite)
		gen3_warn_env_not_overwritten(name);
}

static void gen3_set_env_multiple(const env_entry *env, int overwrite)
{
	int i;

	for (i = 0; env[i].name; i++)
		gen3_set_env(env[i].name, env[i].value, overwrite);
}

void gen3_set_dynamic_env(void)
{
	char dnlid[8];
	char bootpartsize[8];
	char env_str[12];
	unsigned int env_addr = 0;

	ocram *config = (ocram *)IRAM_BASE_ADDR;
	enum env_type env_dev =
		(enum env_type)((ocram *)IRAM_BASE_ADDR)->board->env_dev;
	env_entry *env = (env_entry *)((ocram *)IRAM_BASE_ADDR)->board->env;

	/* set board specific environment */
	gen3_set_env_multiple(env, 0);

	/* set common environment according boot device */
	switch (env_dev) {
	case ENV_GEN3_DEV_FLASH: /* Parallel NOR */
		gen3_set_env_multiple(common_parallel_nor_flash, 0);
		env_addr = CONFIG_ENV_ADDR;
		break;
	case ENV_GEN3_DEV_SPI_FLASH: /* Serial NOR */
		gen3_set_env_multiple(common_serial_nor_flash, 0);
		env_addr = CONFIG_ENV_OFFSET;
		break;
	default:
		printf("ERROR %s: Unsupported Gen3 boot device!", __func__);
		break;
	}

	/* export BoardID to environment */
	sprintf(dnlid, "%04x", config->boardid);
	gen3_set_env("dnlid", dnlid, 1);

	/* export environment address */
	if (env_addr) {
		snprintf(env_str, sizeof(env_str), "0x%X", env_addr);
		gen3_set_env("envaddr", env_str, 1);
		snprintf(env_str, sizeof(env_str), "0x%X",
			env_addr + CONFIG_ENV_BACKUP_OFFSET);
		gen3_set_env("envbackupaddr", env_str, 1);
	}

	/* get and set boot partition size */
	if (env_dev == ENV_GEN3_DEV_SPI_FLASH) {
		struct mmc *mmc;
		mmc = find_mmc_device(DEV_EMMC);

		if (mmc)
			mmc_init(mmc);
		else {
			printf("no eMMC device at slot %x\n", DEV_EMMC);
			return;
		}

		/*
		 * the first sector of the eMMC boot partition is
		 * used for the device tree, the rest for the kernel.
		 * Therefore boot_part_mult - 1
		 */
		sprintf(bootpartsize, "0x%04x",
			((mmc->boot_part_mult - 1)  * 128 * KBYTE)
			 / MMC_BLOCK_SIZE);
		gen3_set_env("kernelsize", bootpartsize, 1);
	}
}

void gen3_set_download_env(void)
{
	gen3_set_env_multiple(dld_env, 1);
}

void gen3_set_recovery_download_env(void)
{
	gen3_set_env_multiple(recovery_dld_env, 1);
}

/* Set environment boot parameters according boot chain */
void gen3_set_boot_env(int boot_chain)
{
	char tmp_str[sizeof(BOOT_ROOTFS_TEMPLATE)];
	unsigned int kerneladdr, dtbsrc;
	board_config *board = (board_config *)((ocram *)IRAM_BASE_ADDR)->board;

	switch (board->env_dev) {
	case ENV_GEN3_DEV_FLASH: /* Parallel NOR */
		/* Select DTB and Kernel addresses according boot chain */
		if (boot_chain == 1) {
			dtbsrc = BOOT_FLASH_DTB_ADDR;
			kerneladdr = BOOT_FLASH_KERNEL_ADDR;
		} else {
			dtbsrc = BOOT_FLASH_DTB_ADDR +
					BOOT_FLASH_CHAIN_2_OFFSET;
			kerneladdr = BOOT_FLASH_KERNEL_ADDR +
					BOOT_FLASH_CHAIN_2_OFFSET;
		}

		snprintf(tmp_str, sizeof(tmp_str), "0x%08X", dtbsrc);
		gen3_set_env("dtbsrc", tmp_str, 0);
		snprintf(tmp_str, sizeof(tmp_str), "0x%08X", kerneladdr);
		gen3_set_env("kerneladdr", tmp_str, 0);
		break;
	case ENV_GEN3_DEV_SPI_FLASH: /* Serial NOR */
		/* Select eMMC boot partition according boot chain*/
		gen3_set_env("bootpart", (boot_chain == 1) ? "1" : "2", 0);
		break;
	default:
		printf("ERROR %s: Unsupported Gen3 boot device!", __func__);
		break;
	}

	/* set root file system partition according boot chain */
	if (((boot_chain == 1) || board->n_single_rootfs))
		snprintf(tmp_str, sizeof(tmp_str),
			BOOT_ROOTFS_TEMPLATE, BOOT_ROOTFS_CHAIN_1_PARTITION);
	else
		snprintf(tmp_str, sizeof(tmp_str),
			BOOT_ROOTFS_TEMPLATE, BOOT_ROOTFS_CHAIN_2_PARTITION);

	gen3_set_env("mmcroot", tmp_str, 0);

	/* Select offset in eMMC user area for recovery initial RAM FS
	 * according boot chain (in Number of MMC blocks) */
	snprintf(tmp_str, sizeof(tmp_str), "0x%X",
		(boot_chain == 1) ?
			RECOVERY_INITRAMFS_1 : RECOVERY_INITRAMFS_2);
	gen3_set_env("initramfssrc", tmp_str, 1);
}
