/*
 * Copyright (C) 2014 Robert Bosch Car Multimedia GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <environment.h>
#include <malloc.h>
#include <asm/arch/sys_proto.h>
#include <asm/gpio.h>
#include <asm/io.h>

#ifdef CONFIG_CMD_MMC
#include <mmc.h>
#include <fsl_esdhc.h>
#include <asm/arch/clock.h>
#endif

#include "boot_opt.h"
#include "common_config.h"

DECLARE_GLOBAL_DATA_PTR;

int dram_init(void)
{
	gd->ram_size = (unsigned int)((ocram *)IRAM_BASE_ADDR)->board->ram_size;

	return 0;
}

#ifdef CONFIG_USB_EHCI_MX6
int board_ehci_hcd_init(int port)
{
	return 0;
}
#endif

#ifdef CONFIG_CMD_MMC
struct fsl_esdhc_cfg usdhc_cfg[2] = {
	{USDHC1_BASE_ADDR},
	{USDHC3_BASE_ADDR}
};

int board_mmc_getcd(struct mmc *mmc)
{
	struct fsl_esdhc_cfg *cfg = (struct fsl_esdhc_cfg *)mmc->priv;
	int ret;

	if (cfg->esdhc_base == USDHC1_BASE_ADDR) {
		gpio_direction_input(4); /* SD card slot: GPIO1_4 */
		ret = !gpio_get_value(4);
	} else
		ret = 1; /* eMMC: Card is always inserted */

	return ret;
}

int usdhc_gpio_init(bd_t *bis)
{
	s32 status = 0;
	u32 index = 0;

	usdhc_cfg[0].sdhc_clk = mxc_get_clock(MXC_ESDHC_CLK);
	usdhc_cfg[1].sdhc_clk = mxc_get_clock(MXC_ESDHC3_CLK);

	for (index = 0; index < CONFIG_SYS_FSL_USDHC_NUM;
		++index) {
		status |= fsl_esdhc_initialize(bis, &usdhc_cfg[index]);
	}

	return status;
}

int board_mmc_init(bd_t *bis)
{
	if (!usdhc_gpio_init(bis))
		return 0;
	else
		return -1;
}
#endif

extern void usbmx6_ctrl_reset(int port);

int board_early_init_f(void)
{
	dcd_config *dcd = (dcd_config *)((ocram *)IRAM_BASE_ADDR)->board->dcd;

	dcd_init(dcd);

	usbmx6_ctrl_reset(0);
	usbmx6_ctrl_reset(1);

	return 0;
}
int board_init(void)
{
	board_config *board = (board_config *)((ocram *)IRAM_BASE_ADDR)->board;

	/* if defined, call board specific initialization function */
	if (board->board_init)
		board->board_init();

	/* address of boot parameters */
	gd->bd->bi_boot_params = PHYS_SDRAM + 0x100;

	return 0;
}

int board_late_init(void) {

	board_config *board = (board_config *)((ocram *)IRAM_BASE_ADDR)->board;
	int boot_chain = 1;
	int reset_dyn = 0;
	enum enBootOpt boot_opt;
	enum sw_opt_type sw_opt;

	get_sw_opt(&sw_opt);

	/* evaluate boot magic */
	evaluate_boot_option(&boot_opt, &boot_chain);

	switch (sw_opt) {
	case SW_OPT_DOWNLOAD:
		/* force boot to recovery download */
		boot_opt = EN_RECOVERY_DOWNLOAD;
		break;
	case SW_OPT_TEST:
		/* development boot */
		boot_opt = EN_DEVELOPER_BOOT;
		break;
	case SW_OPT_NORMAL:
	case SW_OPT_UNDEF:
		/* keep boot option as set by boot magic evaluation */
		break;
	}

	/* Evaluate reset counter only if not in USB download mode */
	if (get_boot_mode() != BOOT_MODE_SERIAL_DOWNLOADER)
		evaluate_reset_counter(&boot_opt, &reset_dyn);

	print_boot_config(boot_opt, boot_chain);

	/* Initialize the dynamic environment */
	gen3_set_dynamic_env();

	/* Set environment boot parameters according boot chain */
	gen3_set_boot_env(boot_chain);

	/*
	 * If we are in serial downloader boot mode overwrite the
	 * environment to support boot to RAM and download
	 */
	if (get_boot_mode() == BOOT_MODE_SERIAL_DOWNLOADER) {

		/* switch on USB OTG if necessary */
		if (board->board_usb_otg_init)
			board->board_usb_otg_init();

		gen3_set_download_env();
		printf("USB download environment set\n");
	} else if (boot_opt == EN_RECOVERY_DOWNLOAD) {
		if (board->n_keep_env_in_recovery_mode == 1) {
			printf("Recovery download - env left unchanged\n");
		} else {
			/*
			 * If recovery download boot option, overwrite the
			 * environment to boot with recovery download RAM disk
			 */
			gen3_set_recovery_download_env();
			printf("Recovery download environment set\n");
		}
	} else if (start_ram_test()) {
#ifdef CONFIG_CMD_ASIL_MEMORY_TEST
		printf("RAM Test magics are set. Starting test ...\n");
		/* Set bootcmd to 'asil_mem_test' */
		setenv("gen3boot", "asil_mem_test");
#else /* #ifdef CONFIG_CMD_ASIL_MEMORY_TEST */
		printf("ERROR: RAM Test magics are set but\n"
		       "\ttest not available ...\n");
#endif /* #ifdef CONFIG_CMD_ASIL_MEMORY_TEST */
	}

	/*
	 * Append systemd.unit=restore_dyn_partition.target to the kernel
	 * command line if needed
	 */
	if (reset_dyn) {
		char  *str, *buf;
		char *arg = "kernelargs";
		char *command = " systemd.unit=restore_dyn_partition.target";

		str = getenv(arg);
		if (str == NULL)
			goto skip;

		buf = malloc(strlen(str) + strlen(command) + 1);
		strcpy(buf, str);
		strcat(buf, command);
		setenv(arg, buf);
		free(buf);
	}

skip:
	return 0;
}

int checkboard(void)
{
	ocram *config = (ocram *)IRAM_BASE_ADDR;

	printf("Board: %s\n", config->board->name);

	printf("Board ID: 0x%04x (#%d)\n", config->boardid, config->idcount);

	return 0;
}

void board_eth_use_fixed_phy(int *ptr_b_use_fixed_phy)
{
	*ptr_b_use_fixed_phy =
		((ocram *)IRAM_BASE_ADDR)->board->n_use_fixed_phy;
	return;
}

#ifdef CONFIG_ENV_IS_DYNAMIC
char *env_name_spec = "Dynamic";
env_t *env_ptr = (env_t *)CONFIG_ENV_ADDR;

unsigned long flash_init(void)
{
	if (((ocram *)IRAM_BASE_ADDR)->board->env_dev == ENV_GEN3_DEV_FLASH)
		return board_flash_init();

	return 0;
}

int env_init(void)
{
	switch (((ocram *)IRAM_BASE_ADDR)->board->env_dev) {
	case ENV_GEN3_DEV_FLASH:
		env_name_spec = "Flash";
		return env_flash_init();
		break;
	case ENV_GEN3_DEV_SPI_FLASH:
		env_name_spec = "SPI Flash";
		return env_spi_flash_init();
		break;
	case ENV_GEN3_DEV_MMC:
		env_name_spec = "MMC";
		return env_mmc_init();
		break;
	default:
		break;
	}

	return 0;
}

int saveenv(void)
{
	switch (((ocram *)IRAM_BASE_ADDR)->board->env_dev) {
	case ENV_GEN3_DEV_FLASH:
		return env_flash_saveenv();
		break;
	case ENV_GEN3_DEV_SPI_FLASH:
		return env_spi_flash_saveenv();
		break;
	case ENV_GEN3_DEV_MMC:
		return env_mmc_saveenv();
		break;
	default:
		break;
	}

	return 0;
}

void env_relocate_spec(void)
{
	switch (((ocram *)IRAM_BASE_ADDR)->board->env_dev) {
	case ENV_GEN3_DEV_FLASH:
		env_flash_relocate_spec();
		break;
	case ENV_GEN3_DEV_SPI_FLASH:
		env_spi_flash_relocate_spec();
		break;
	case ENV_GEN3_DEV_MMC:
		env_mmc_relocate_spec();
		break;
	default:
		break;
	}
}
#endif /* CONFIG_ENV_IS_DYNAMIC */
