/*
 * Copyright (C) 2012 Mentor Graphics Corp.
 */
/*
 * Copyright 2012 Freescale Semiconductor, Inc.
 */
/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/of_i2c.h>
#include <linux/clk.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <linux/of_gpio.h>

#include <sound/jack.h>
#include "../codecs/wm8962.h"
#include "imx-audmux.h"

#define DAI_NAME_SIZE	32
#define GPIO_HP_DET	(1<<0)
#define GPIO_MIC_DET	(1<<1)

struct imx_wm8962_data {
	struct snd_soc_dai_link dai;
	struct snd_soc_card card;
	char codec_dai_name[DAI_NAME_SIZE];
	char platform_name[DAI_NAME_SIZE];
	struct clk *codec_clk;
	unsigned int clk_frequency;
	int hp_gpio;
	int hp_active_low;
	int mic_gpio;
	int mic_active_low;
	int flag;
	struct snd_pcm_substream *first_stream;
	struct snd_pcm_substream *second_stream;
};

static struct snd_soc_jack imx_hp_jack;
static struct snd_soc_jack_pin imx_hp_jack_pins[] = {
	{
		.pin = "Ext Spk",
		.mask = SND_JACK_HEADPHONE,
	},
};
static struct snd_soc_jack_gpio imx_hp_jack_gpio = {
	.name = "headphone detect",
	.report = SND_JACK_HEADPHONE,
	.debounce_time = 150,
	.invert = 0,
};

static struct snd_soc_jack imx_mic_jack;
static struct snd_soc_jack_pin imx_mic_jack_pins[] = {
	{
		.pin = "DMIC",
		.mask = SND_JACK_MICROPHONE,
	},
};
static struct snd_soc_jack_gpio imx_mic_jack_gpio = {
	.name = "microphone detect",
	.report = SND_JACK_MICROPHONE,
	.debounce_time = 150,
	.invert = 0,
};

static int imx_wm8962_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct device *dev = rtd->card->dev;
	unsigned int sample_rate = 44100;
	unsigned int sample_format = SNDRV_PCM_FMTBIT_S16_LE;
	int ret = 0;
	unsigned int pll_out;
	struct imx_wm8962_data *data = container_of(rtd->card,
						struct imx_wm8962_data, card);
	if (!data->first_stream)
		data->first_stream = substream;
	else
		data->second_stream = substream;

	sample_rate = params_rate(params);
	sample_format = params_format(params);
	if (sample_format == SNDRV_PCM_FORMAT_S24_LE)
		pll_out = sample_rate * 192;
	else
		pll_out = sample_rate * 256;

	ret = snd_soc_dai_set_pll(codec_dai,  WM8962_FLL,
				WM8962_FLL_MCLK, data->clk_frequency,
					  pll_out);

	ret = snd_soc_dai_set_sysclk(codec_dai,
					 WM8962_SYSCLK_FLL,
					 pll_out,
					 SND_SOC_CLOCK_IN);
	if (ret < 0) {
		dev_err(dev, "could not set codec driver clock params\n");
		return ret;
	}
	return 0;
}

static int imx_wm8962_hw_free(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct imx_wm8962_data *data = container_of(rtd->card,
						struct imx_wm8962_data, card);
	int ret;

	if (data->first_stream == substream)
		data->first_stream = data->second_stream;
	data->second_stream = NULL;

	if (!data->first_stream) {
		/*
		 * wm8962 doesn't allow us to continuously setting FLL,
		 * So we set MCLK as sysclk once, which'd remove the limitation.
		 */
		ret = snd_soc_dai_set_sysclk(codec_dai, WM8962_SYSCLK_MCLK,
				0, SND_SOC_CLOCK_IN);
		if (ret < 0) {
			pr_err("Failed to set SYSCLK: %d\n", ret);
			return ret;
		}

		/*
		 * Continuously setting FLL would cause playback distortion.
		 * We can fix it just by mute codec after playback.
		 */
		ret = snd_soc_dai_digital_mute(codec_dai, 1);
		if (ret < 0) {
			pr_err("Failed to set MUTE: %d\n", ret);
			return ret;
		}
	}
	return 0;
}

static int imx_wm8962_startup(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct imx_wm8962_data *data = container_of(rtd->card,
						struct imx_wm8962_data, card);

	if (!codec_dai->active)
		clk_prepare_enable(data->codec_clk);

	return 0;
}

static void imx_wm8962_shutdown(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct imx_wm8962_data *data = container_of(rtd->card,
						struct imx_wm8962_data, card);

	if (!codec_dai->active)
		clk_disable_unprepare(data->codec_clk);

	return;
}


static int imx_wm8962_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	struct imx_wm8962_data *data = container_of(rtd->card,
						struct imx_wm8962_data, card);
	int ret;

	if (gpio_is_valid(data->hp_gpio)) {
			imx_hp_jack_gpio.gpio = data->hp_gpio;
			ret = snd_soc_jack_new(codec, "Ext Spk",
					SND_JACK_LINEOUT, &imx_hp_jack);
			if (ret) {
				pr_err("Jack creation failed\n");
				return ret;
			}
			ret = snd_soc_jack_add_pins(&imx_hp_jack,
						ARRAY_SIZE(imx_hp_jack_pins),
						imx_hp_jack_pins);
			if (ret) {
				pr_err("adding Jack pins failed\n");
				return ret;
			}
			ret = snd_soc_jack_add_gpios(&imx_hp_jack,
						1, &imx_hp_jack_gpio);
			if (ret) {
				pr_err("adding GPIO pins with jack failed\n");
				return ret;
			}
			data->flag |= GPIO_HP_DET;
	}

	if (gpio_is_valid(data->mic_gpio)) {
			imx_mic_jack_gpio.gpio = data->mic_gpio;
			ret = snd_soc_jack_new(codec, "DMIC",
					SND_JACK_MICROPHONE, &imx_mic_jack);
			if (ret) {
				pr_err("Jack creation failed\n");
				return ret;
			}
			ret = snd_soc_jack_add_pins(&imx_mic_jack,
						ARRAY_SIZE(imx_mic_jack_pins),
						imx_mic_jack_pins);
			if (ret) {
				pr_err("adding Jack pins failed\n");
				return ret;
			}
			ret = snd_soc_jack_add_gpios(&imx_mic_jack,
						1, &imx_mic_jack_gpio);
			if (ret) {
				pr_err("adding GPIO pins with jack failed\n");
				return ret;
			}
			data->flag |= GPIO_MIC_DET;
	}

	snd_soc_dapm_sync(&codec->dapm);
	return 0;
}

static const struct snd_soc_dapm_widget imx_wm8962_dapm_widgets[] = {
	SND_SOC_DAPM_HP("Headphone Jack", NULL),
	SND_SOC_DAPM_SPK("Ext Spk", NULL),
	SND_SOC_DAPM_MIC("AMIC", NULL),
	SND_SOC_DAPM_MIC("DMIC", NULL),
};

static struct snd_soc_ops imx_wm8962_ops = {
	.startup = imx_wm8962_startup,
	.shutdown = imx_wm8962_shutdown,
	.hw_params = imx_wm8962_hw_params,
	.hw_free = imx_wm8962_hw_free,
};

static int imx_wm8962_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device_node *ssi_np, *codec_np;
	struct platform_device *ssi_pdev;
	struct i2c_client *codec_dev;
	struct imx_wm8962_data *data;
	int int_port, ext_port;
	int ret;
	int rate;

	ret = of_property_read_u32(np, "mux-int-port", &int_port);
	if (ret) {
		dev_err(&pdev->dev, "mux-int-port missing or invalid\n");
		return ret;
	}
	ret = of_property_read_u32(np, "mux-ext-port", &ext_port);
	if (ret) {
		dev_err(&pdev->dev, "mux-ext-port missing or invalid\n");
		return ret;
	}

	int_port--;
	ext_port--;
	ret = imx_audmux_v2_configure_port(int_port,
			IMX_AUDMUX_V2_PTCR_SYN |
			IMX_AUDMUX_V2_PTCR_TFSDIR |
			IMX_AUDMUX_V2_PTCR_TFSEL(ext_port) |
			IMX_AUDMUX_V2_PTCR_TCLKDIR |
			IMX_AUDMUX_V2_PTCR_TCSEL(ext_port),
			IMX_AUDMUX_V2_PDCR_RXDSEL(ext_port));

	if (ret) {
			dev_err(&pdev->dev,
				"audmux internal port setup failed with Error = %d\n",
				ret);
			return ret;
		}
	ret = imx_audmux_v2_configure_port(ext_port,
				IMX_AUDMUX_V2_PTCR_SYN,
				IMX_AUDMUX_V2_PDCR_RXDSEL(int_port));
	if (ret) {
			dev_err(&pdev->dev,
				"audmux external port setup failed with Error = %d\n",
					ret);
			return ret;
		}

	ssi_np = of_parse_phandle(pdev->dev.of_node, "ssi-controller", 0);
	codec_np = of_parse_phandle(pdev->dev.of_node, "audio-codec", 0);
	if (!ssi_np || !codec_np) {
		dev_err(&pdev->dev, "phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	ssi_pdev = of_find_device_by_node(ssi_np);
	if (!ssi_pdev) {
		dev_err(&pdev->dev, "failed to find SSI platform device\n");
		ret = -EPROBE_DEFER;
		goto fail;
	}
	codec_dev = of_find_i2c_device_by_node(codec_np);
	if (!codec_dev) {
		dev_err(&pdev->dev, "failed to find codec platform device\n");
		ret = -EPROBE_DEFER;
		goto fail;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		ret = -ENOMEM;
		goto fail;
	}

	data->flag = 0;
	data->hp_gpio = of_get_named_gpio_flags(np, "hp-gpio", 0,
				(enum of_gpio_flags *)&data->hp_active_low);
	data->mic_gpio = of_get_named_gpio_flags(np, "mic-gpio", 0,
				(enum of_gpio_flags *)&data->mic_active_low);
	data->first_stream = NULL;
	data->second_stream = NULL;

	data->codec_clk = clk_get(NULL, "cko1");
	if (IS_ERR(data->codec_clk)) {
		dev_err(&pdev->dev, "can't get CLKO clock.\n");
		goto fail;
	}
	clk_prepare_enable(data->codec_clk);
	if (ret) {
		dev_err(&codec_dev->dev,
				"failed to enable codec clk: %d\n", ret);
		goto clk_prepare_fail;
	}
	rate = clk_round_rate(data->codec_clk, 24000000);
	clk_set_rate(data->codec_clk, rate);
	data->clk_frequency = rate;

	data->dai.name = "HiFi";
	data->dai.stream_name = "HiFi";
	data->dai.codec_dai_name = "wm8962";
	data->dai.codec_of_node = codec_np;
	data->dai.cpu_dai_name = dev_name(&ssi_pdev->dev);
	data->dai.platform_name = "imx-pcm-audio";
	data->dai.ops = &imx_wm8962_ops;
	data->dai.init = imx_wm8962_init;
	data->dai.dai_fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_NB_NF |
			    SND_SOC_DAIFMT_CBM_CFM;

	data->card.dev = &pdev->dev;
	ret = snd_soc_of_parse_card_name(&data->card, "model");
	if (ret)
		goto clk_fail;
	ret = snd_soc_of_parse_audio_routing(&data->card, "audio-routing");
	if (ret)
		goto clk_fail;
	data->card.num_links = 1;
	data->card.dai_link = &data->dai;
	data->card.dapm_widgets = imx_wm8962_dapm_widgets;
	data->card.num_dapm_widgets = ARRAY_SIZE(imx_wm8962_dapm_widgets);

	ret = snd_soc_register_card(&data->card);
	if (ret) {
		dev_err(&pdev->dev, "snd_soc_register_card failed (%d)\n", ret);
		goto clk_fail;
	}

	platform_set_drvdata(pdev, data);
	of_node_put(ssi_np);
	of_node_put(codec_np);

	return 0;

clk_fail:
	if (!IS_ERR(data->codec_clk))
		clk_disable_unprepare(data->codec_clk);

clk_prepare_fail:
	if (!IS_ERR(data->codec_clk))
		clk_put(data->codec_clk);

fail:
	if (ssi_np)
		of_node_put(ssi_np);
	if (codec_np)
		of_node_put(codec_np);

	return ret;
}

static int imx_wm8962_remove(struct platform_device *pdev)
{
	struct imx_wm8962_data *data = platform_get_drvdata(pdev);

	if (data->flag & GPIO_HP_DET)
		snd_soc_jack_free_gpios(&imx_hp_jack, 1, &imx_hp_jack_gpio);
	if (data->flag & GPIO_MIC_DET)
		snd_soc_jack_free_gpios(&imx_mic_jack, 1, &imx_mic_jack_gpio);

	if (data->codec_clk) {
		clk_disable_unprepare(data->codec_clk);
		clk_put(data->codec_clk);
	}
	snd_soc_unregister_card(&data->card);

	return 0;
}

static const struct of_device_id imx_wm8962_dt_ids[] = {
	{.compatible = "fsl,imx-audio-wm8962",},
	{ /* sentinel */ }
};

MODULE_DEVICE_TABLE(of, imx_wm8962_dt_ids);

static struct platform_driver imx_wm8962_driver = {
	.driver = {
		   .name = "imx-wm8962",
		   .owner = THIS_MODULE,
		   .of_match_table = imx_wm8962_dt_ids,
		   },
	.probe = imx_wm8962_probe,
	.remove = imx_wm8962_remove,
};

module_platform_driver(imx_wm8962_driver);

/* Module information */
MODULE_DESCRIPTION("ALSA SoC wm8962 Machine Layer Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:imx-wm8962");
