/*
 * fsl_asrc_dma.c  --  iMX6 ASRC ALSA SoC Platform Driver
 *
 * Copyright (C) 2013 Mentor Graphics, Inc. All Rights Reserved.
 * Copyright 2009 Sascha Hauer <s.hauer@pengutronix.de>
 *
 * This code is based on code copyrighted by Freescale,
 * Liam Girdwood, Javier Martin and probably others.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 */
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/dmaengine.h>
#include <linux/types.h>

#include <sound/core.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/dmaengine_pcm.h>

#include <linux/platform_data/dma-imx.h>
#include "fsl_asrc.h"

int fsl_asrc_dma_snd_dmaengine_open(struct imx_pcm_runtime_data *iprtd,
				struct snd_pcm_substream *substream,
	dma_filter_fn filter_fn, void *filter_data);

/* Private data for the platform device*/
struct fsl_asrc_dma_soc_private {
	struct fsl_asrc_core_private *asrc_core_private;
	struct imx_pcm_runtime_data *iprtd[SECTION_CNT];
	/*
	 * For BE in playback, the ASRC pair section is always OUTPUT and for
	 * capture it is always INPUT
	 */
	unsigned int watermark[2];
	unsigned int burstsize[2];
	unsigned int filter_setting[2];
	unsigned int pairmask[2];
};

static bool filter(struct dma_chan *chan, void *param)
{
	if (!imx_dma_is_general_purpose(chan))
		return false;

	chan->private = param;

	return true;
}

static int
fsl_asrc_dma_dmaengine_prepare_and_submit(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_asrc_dma_soc_private *private =
			dev_get_drvdata(rtd->platform->dev);
	struct imx_pcm_runtime_data *iprtd = private->iprtd[substream->stream];
	struct dma_async_tx_descriptor *desc;

	iprtd->offset = 0;
	desc = dmaengine_prep_dma_cyclic(iprtd->dma_chan,
		0xffff,
		FIFO_DEPTH,
		FIFO_DEPTH, DMA_DEV_TO_DEV, 0);

	if (!desc)
		return -ENOMEM;

	desc->callback = NULL;
	desc->callback_param = NULL;
	dmaengine_submit(desc);

	return 0;
}

static int fsl_asrc_dma_snd_pcm_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_asrc_dma_soc_private *private =
			dev_get_drvdata(rtd->platform->dev);
	struct imx_pcm_runtime_data *iprtd = private->iprtd[substream->stream];

	struct dma_slave_config p2p_slave_config;
	enum asrc_section_direction direction;
	struct section_config config;
	int ret;

	direction = BE_STREAM_DIRECTION(substream->stream);

	/* fill in be section config */
	config.rate = params_rate(params);
	config.channels = params_channels(params);
	config.format = params_format(params);

	dev_dbg(rtd->platform->dev,
			"%s rate:%d channels:%d format:%llu params(%p)\n",
			__func__, config.rate,
			config.channels,
			config.format, params);

	ret = fsl_asrc_set_setup(rtd->dai_link->name, direction, &config,
			substream->stream);
	if (ret)
		return ret;

	/* fill in the data for the asrc side */
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		p2p_slave_config.src_addr = iprtd->dma_params_asrc->dma_addr;
		/* ASRC burst size is in frames and the dma burst size is in
		 * 32 bit words for sdma p_2_p */
		p2p_slave_config.src_maxburst =
				iprtd->dma_params_asrc->burstsize *
				config.channels;
	} else {
		p2p_slave_config.dst_addr =
				iprtd->dma_params_asrc->dma_addr;
		p2p_slave_config.dst_maxburst =
				iprtd->dma_params_asrc->burstsize *
				config.channels;
	}

	ret = snd_hwparams_to_dma_slave_config(substream,
			params, &p2p_slave_config);
	if (ret) {
		dev_err(rtd->platform->dev,
			"Unrecognized format, cannot calculate addr width\n");
		return ret;
	}

	p2p_slave_config.direction = DMA_DEV_TO_DEV;
	p2p_slave_config.device_fc = false;

	/* fill in the data for the cpu-dai side */
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		p2p_slave_config.src_addr_width =
				p2p_slave_config.dst_addr_width;
		p2p_slave_config.dst_addr = iprtd->dma_params_dai->dma_addr;
		p2p_slave_config.dst_maxburst =
				iprtd->dma_params_dai->burstsize;
	} else {
		p2p_slave_config.dst_addr_width =
				p2p_slave_config.src_addr_width;
		p2p_slave_config.src_addr = iprtd->dma_params_dai->dma_addr;
		p2p_slave_config.src_maxburst =
				iprtd->dma_params_dai->burstsize;
	}

	ret = dmaengine_slave_config(iprtd->dma_chan, &p2p_slave_config);
	if (ret)
		return ret;

	snd_pcm_set_runtime_buffer(substream, &substream->dma_buffer);

	return 0;
}

static int fsl_asrc_dma_snd_pcm_prepare(struct snd_pcm_substream *substream)
{
	enum asrc_section_direction direction;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	int ret;

	direction = BE_STREAM_DIRECTION(substream->stream);

	ret = fsl_asrc_set_be_clock_reference(rtd->dai_link->name, direction,
			substream->stream);
	if (ret)
		return ret;

	ret = fsl_asrc_prepare(rtd->dai_link->name, direction,
			substream->stream);

	return ret;
}

static int fsl_asrc_dma_snd_open(struct snd_pcm_substream *substream)
{
	struct imx_pcm_runtime_data *iprtd;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_platform *platform = rtd->platform;
	struct fsl_asrc_dma_soc_private *private =
				dev_get_drvdata(rtd->platform->dev);
	struct imx_pcm_dma_params *dma_params_dai, *dma_params_asrc;
	struct imx_dma_data *dma_data;
	unsigned char mask;
	unsigned char hw_period[SECTION_CNT];
	enum asrc_section_direction direction;
	enum filter_settings settings;
	int ret;

	iprtd = kzalloc(sizeof(*iprtd), GFP_KERNEL);
	if (iprtd == NULL)
		return -ENOMEM;

	mask = private->pairmask[substream->stream];

	/*
	 * For BE in playback, the ASRC pair section is always OUTPUT and for
	 * capture it is always INPUT
	 */
	hw_period[SECTION_INPUT] = private->watermark[SNDRV_PCM_STREAM_CAPTURE];
	hw_period[SECTION_OUTPUT] =
				private->watermark[SNDRV_PCM_STREAM_PLAYBACK];

	settings = private->filter_setting[substream->stream];

	direction = BE_STREAM_DIRECTION(substream->stream);

	ret = fsl_asrc_request_pair(private->asrc_core_private,
			rtd->dai_link->name, direction, mask,
			substream->stream);
	if (ret)
		goto fail_request;

	ret = fsl_asrc_set_period(rtd->dai_link->name, direction,
			hw_period,
			substream->stream);
	if (ret)
		goto fail_settings;


	ret = fsl_asrc_set_filters(rtd->dai_link->name, direction, settings,
			substream->stream);
	if (ret)
		goto fail_settings;

	/*
	 * TODO: Update hw params runtime constraints should mainline
	 * begin considering BE hardware constraints
	 */

	/* get dma params */
	dma_params_asrc = fsl_asrc_get_dma_params(rtd->dai_link->name,
			direction,
			substream->stream);

	dev_dbg(platform->dev, "%s dma_params: (0x%p)\n", __func__,
			dma_params_asrc);

	if (dma_params_asrc == NULL) {
		ret = -EINVAL;
		goto fail_settings;
	}
	dma_params_asrc->burstsize = private->burstsize[substream->stream];

	dma_params_dai = snd_soc_dai_get_dma_data(rtd->cpu_dai, substream);

	iprtd->dma_data = kzalloc(sizeof(*dma_data), GFP_KERNEL);
	if (!iprtd->dma_data) {
		ret = -ENOMEM;
		goto fail_settings;
	}

	iprtd->dma_data->peripheral_type = dma_params_asrc->peripheral_type;
	iprtd->dma_data->priority = FSL_ASRC_PCM_DMA_PRIORITY;

	/* keep the source dma event in dma_request */
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		iprtd->dma_data->dma_request = dma_params_asrc->dma;
		iprtd->dma_data->dma_request_p2p = dma_params_dai->dma;
	} else {
		iprtd->dma_data->dma_request = dma_params_dai->dma;
		iprtd->dma_data->dma_request_p2p = dma_params_asrc->dma;
	}
	dev_dbg(platform->dev, "%s dma_data:\n"
			"\t peripheral_type: %d\n"
			"\t priority: %d\n"
			"\t dma irq: %d\n"
			"\t dma irq p2p: %d\n",
			__func__,
			iprtd->dma_data->peripheral_type,
			iprtd->dma_data->priority,
			iprtd->dma_data->dma_request,
			iprtd->dma_data->dma_request_p2p);

	/* create the dma channel for p2p */
	ret = fsl_asrc_dma_snd_dmaengine_open(iprtd, substream,
			filter, iprtd->dma_data);
	if (ret)
		goto fail_dma_data;

	iprtd->dma_params_asrc = dma_params_asrc;
	iprtd->dma_params_dai = dma_params_dai;

	/* set substream runtime private data */
	private->iprtd[substream->stream] = iprtd;

	return 0;

fail_dma_data:
	kfree(iprtd->dma_data);
	iprtd->dma_data = NULL;
fail_settings:
	fsl_asrc_cleanup(rtd->dai_link->name, SECTION_OUTPUT,
			substream->stream);
	fsl_asrc_cleanup(rtd->dai_link->name, SECTION_INPUT,
			substream->stream);
fail_request:
	kfree(iprtd);
	private->iprtd[substream->stream] = NULL;
	return ret;
}

static int
fsl_asrc_dma_dmaengine_request_channel(struct imx_pcm_runtime_data *iprtd,
	dma_filter_fn filter_fn, void *filter_data)
{
	dma_cap_mask_t mask;

	dma_cap_zero(mask);
	dma_cap_set(DMA_SLAVE, mask);
	dma_cap_set(DMA_CYCLIC, mask);
	iprtd->dma_chan = dma_request_channel(mask, filter_fn, filter_data);

	if (!iprtd->dma_chan)
		return -ENXIO;

	return 0;
}

/**
 * snd_dmaengine_pcm_open - Open a dmaengine based PCM substream
 * @substream: PCM substream
 * @filter_fn: Filter function used to request the DMA channel
 * @filter_data: Data passed to the DMA filter function
 *
 * Returns 0 on success, a negative error code otherwise.
 */
int fsl_asrc_dma_snd_dmaengine_open(struct imx_pcm_runtime_data *iprtd,
				struct snd_pcm_substream *substream,
	dma_filter_fn filter_fn, void *filter_data)
{
	int ret;

	ret = snd_pcm_hw_constraint_integer(substream->runtime,
					    SNDRV_PCM_HW_PARAM_PERIODS);
	if (ret < 0)
		return ret;

	ret = fsl_asrc_dma_dmaengine_request_channel(iprtd,
			filter_fn, filter_data);
	if (ret < 0)
		return ret;

	return 0;
}

static int fsl_asrc_dma_snd_close(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_asrc_dma_soc_private *private =
				dev_get_drvdata(rtd->platform->dev);
	struct imx_pcm_runtime_data *iprtd = private->iprtd[substream->stream];
	struct imx_dma_data *dma_data = iprtd->dma_data;
	enum asrc_section_direction direction;

	direction = BE_STREAM_DIRECTION(substream->stream);

	fsl_asrc_cleanup(rtd->dai_link->name, direction, substream->stream);

	if (iprtd->dma_chan) {
		dev_dbg(rtd->platform->dev, "%s free dma_channel\n", __func__);
		dma_release_channel(iprtd->dma_chan);
	}

	kfree(iprtd);
	private->iprtd[substream->stream] = NULL;
	kfree(dma_data);

	return 0;
}

static int
fsl_asrc_dma_snd_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_asrc_dma_soc_private *private =
				dev_get_drvdata(rtd->platform->dev);
	struct imx_pcm_runtime_data *iprtd = private->iprtd[substream->stream];
	enum asrc_section_direction direction;
	int ret;

	direction = BE_STREAM_DIRECTION(substream->stream);

	dev_dbg(rtd->platform->dev, "%s: offset:%ld frames:%ld cmd:%d\n",
			__func__, iprtd->offset,
			bytes_to_frames(substream->runtime, iprtd->offset),
			cmd);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		ret = fsl_asrc_dma_dmaengine_prepare_and_submit(substream);
		if (ret)
			return ret;
		dma_async_issue_pending(iprtd->dma_chan);
		fsl_asrc_start_conversion(rtd->dai_link->name, direction,
				substream->stream);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		fsl_asrc_stop_conversion(rtd->dai_link->name, direction,
				substream->stream);
		dmaengine_terminate_all(iprtd->dma_chan);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int asrc_filter_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.enumerated.item[0] =
			private->filter_setting[dir];

	return 0;
}

static int asrc_filter_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	private->filter_setting[dir] =
			ucontrol->value.enumerated.item[0];

	return 0;
}

static int asrc_watermark_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = private->watermark[dir];

	return 0;
}

static int asrc_watermark_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	int section;
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	/*
	 * For BE in playback, the ASRC pair section is always OUTPUT and for
	 * capture it is always INPUT
	 */
	section = BE_STREAM_DIRECTION(dir);
	private->watermark[dir] = ucontrol->value.integer.value[0];
	private->burstsize[dir] =
		fsl_asrc_adjust_burst_per_watermark(private->burstsize[dir],
						    private->watermark[dir],
						    section);

	return 0;
}

static int asrc_burstsize_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = private->burstsize[dir];

	return 0;
}

static int asrc_burstsize_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	int section;
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	/*
	 * For BE in playback, the ASRC pair section is always OUTPUT and for
	 * capture it is always INPUT
	 */
	section = BE_STREAM_DIRECTION(dir);
	private->burstsize[dir] =
	   fsl_asrc_adjust_burst_per_watermark(ucontrol->value.integer.value[0],
						private->watermark[dir],
						section);
	return 0;
}

static int asrc_pair_mask_get(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] =
			private->pairmask[dir];

	return 0;
}

static int asrc_pair_mask_put(struct snd_kcontrol *kcontrol,
		struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_platform *platform = snd_kcontrol_chip(kcontrol);
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	int dir = kcontrol->id.subdevice;

	private->pairmask[dir] =
			ucontrol->value.integer.value[0];

	return 0;
}

const char const *filter_settings_text[] = {
	"FILT_TABLE",
	"FILT_FORMULA",
	"UP2_UP2",
	"UP2_DIRECT",
	"UP2_DOWN2",
	"DIRECT_UP2",
	"DIRECT_DIRECT",
	"DIRECT_DOWN2",
	"DOWN2_UP2",
	"DOWN2_DIRECT",
	"DOWN2_DOWN2",
	"PASSTHROUGH"
};

static const struct soc_enum filter_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(filter_settings_text),
			filter_settings_text);

static const struct snd_kcontrol_new  no_pcm_pf_kcontrols[] = {
	FSL_ASRC_DAPM_ENUM_EXT_DIRECTIONAL("capture.filter_settings",
			filter_enum, asrc_filter_get,
			asrc_filter_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_DAPM_ENUM_EXT_DIRECTIONAL("playback.filter_settings",
			filter_enum, asrc_filter_get,
			asrc_filter_put,
			SNDRV_PCM_STREAM_PLAYBACK),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("playback.asrc_out_watermark",
			MAX_PERIOD,
			asrc_watermark_get,
			asrc_watermark_put,
			SNDRV_PCM_STREAM_PLAYBACK),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("playback.asrc_out_burstsize",
			MAX_PERIOD,
			asrc_burstsize_get,
			asrc_burstsize_put,
			SNDRV_PCM_STREAM_PLAYBACK),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("capture.asrc_in_watermark",
			MAX_PERIOD,
			asrc_watermark_get,
			asrc_watermark_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("capture.asrc_in_burstsize",
			MAX_PERIOD,
			asrc_burstsize_get,
			asrc_burstsize_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("capture.asrc_pairmask",
			CHANNEL_MASK_MAX,
			asrc_pair_mask_get,
			asrc_pair_mask_put,
			SNDRV_PCM_STREAM_CAPTURE),
	FSL_ASRC_VALUE_INT_DIRECTIONAL("playback.asrc_pairmask",
			CHANNEL_MASK_MAX,
			asrc_pair_mask_get,
			asrc_pair_mask_put,
			SNDRV_PCM_STREAM_PLAYBACK),
};

static int fsl_asrc_dma_snd_probe(struct snd_soc_platform *platform)
{
	struct fsl_asrc_dma_soc_private *private =
			snd_soc_platform_get_drvdata(platform);
	struct snd_soc_card *card = platform->card;
	struct snd_card *snd_card = card->snd_card;
	const char *prefix = NULL;
	int ret = 0, i;
	struct snd_soc_dai_link *link;
	struct snd_soc_pcm_runtime *rtd;

	for (i = 0; i < card->num_rtd; i++) {
		link = &card->dai_link[i];
		rtd = &card->rtd[i];
		prefix = link->stream_name;

		dev_dbg(platform->dev, "link->pn:%s plat->name:%s",
				link->platform_name, platform->name);

		if (rtd->platform == platform)
			ret = fsl_asrc_add_controls(snd_card, platform->dev,
				no_pcm_pf_kcontrols,
				ARRAY_SIZE(no_pcm_pf_kcontrols),
				prefix, platform);
	}

	/* default sound control watermark/burstsize values */
	private->watermark[SNDRV_PCM_STREAM_PLAYBACK] =
	private->watermark[SNDRV_PCM_STREAM_CAPTURE] = FSL_ASRC_WATERMARK;
	private->burstsize[SNDRV_PCM_STREAM_PLAYBACK] =
			      private->watermark[SNDRV_PCM_STREAM_PLAYBACK] + 1;
	private->burstsize[SNDRV_PCM_STREAM_CAPTURE] = (FIFO_DEPTH -
			      private->watermark[SNDRV_PCM_STREAM_CAPTURE]) + 1;

	private->pairmask[SNDRV_PCM_STREAM_PLAYBACK] =
	private->pairmask[SNDRV_PCM_STREAM_CAPTURE] = CHANNEL_MASK_MAX;

	return ret;
}

static struct snd_pcm_ops fsl_asrc_dma_pcm_ops = {
	.open		= fsl_asrc_dma_snd_open,
	.close		= fsl_asrc_dma_snd_close,
	.prepare	= fsl_asrc_dma_snd_pcm_prepare,
	.hw_params	= fsl_asrc_dma_snd_pcm_hw_params,
	.trigger	= fsl_asrc_dma_snd_pcm_trigger,
};

static struct snd_soc_platform_driver fsl_asrc_dma_p2p_platform = {
	.ops		= &fsl_asrc_dma_pcm_ops,
	.probe		= fsl_asrc_dma_snd_probe,
};

static int fsl_asrc_dma_soc_platform_probe(struct platform_device *pdev)
{
	struct fsl_asrc_dma_soc_private *private;
	int ret;
	struct fsl_asrc_core_private **asrc_core_private;

	asrc_core_private =
		(struct fsl_asrc_core_private **)pdev->dev.platform_data;

	if (asrc_core_private && *asrc_core_private)
		dev_info(&pdev->dev, "%p asrc core pointer\n",
				*asrc_core_private);
	else {
		dev_err(&pdev->dev,
				"asrc core ref not passed in from cpu-dai\n");
		return -EINVAL;
	}

	/* allocate platform instance data */
	private = kzalloc(sizeof(struct fsl_asrc_dma_soc_private), GFP_KERNEL);
	if (!private) {
		dev_err(&pdev->dev,
				"could not allocate platform private data\n");
		return -ENOMEM;
	}

	private->asrc_core_private = *asrc_core_private;

	platform_set_drvdata(pdev, private);

	ret = snd_soc_register_platform(&pdev->dev, &fsl_asrc_dma_p2p_platform);
	if (ret) {
		dev_err(&pdev->dev, "failed to register no_pcm soc platform");
		goto error_dev;
	}

	return 0;

error_dev:
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(private);
	return ret;
}

static int fsl_asrc_dma_soc_platform_remove(struct platform_device *pdev)
{
	snd_soc_unregister_platform(&pdev->dev);
	return 0;
}

static struct platform_driver fsl_asrc_dma_pcm_driver = {
	.driver = {
			.name = "fsl-asrc-pcm-audio",
			.owner = THIS_MODULE,
	},
	.probe = fsl_asrc_dma_soc_platform_probe,
	.remove = fsl_asrc_dma_soc_platform_remove,
};

module_platform_driver(fsl_asrc_dma_pcm_driver);

MODULE_AUTHOR("Joshua Frkuska <joshua_frkuska@mentor.com>");
MODULE_ALIAS("platform:fsl-asrc-pcm-audio");
MODULE_LICENSE("GPL v2");
