#!/usr/bin/perl
# Copyright (C) 2013 Mentor Graphics Corp.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation.
#
# gen_clk parses clk-like.c to generate header file with
# all clock devices defined as Macro in it, so that
# Device-Tree can include it to refer to each clock device.
#

use strict;
use warnings;
use Getopt::Long;

my $output_file = "include/dt-bindings/clock/imx6q-clock.h";
my $clk_enum = "mx6q_clks";

my $USAGE = <<"USAGE";
Usage: gen_clk.pl -i <path/clk-imx6q.c> [-n <clks enum name> -o <output file>]

Necessary Options:

-i|I        -- Specify from which clk-like.c file to parse all clock name

Optional options:
-n|N        -- Specify the clocks enum name, by default it's "$clk_enum"
-o|O        -- Specify the output header file, by default it's "$output_file"

USAGE

my $input_file;
my $clk_end = "CLK_END";

GetOptions ("i|I=s" => \$input_file,
	    "n|N=s"   => \$clk_enum,
	    "o|O=s"  => \$output_file) and -f $input_file
  or die $USAGE;

if (!top_of_kernel_tree('.')) {
	print "Must be run from the top-level dir. of a kernel tree\n";
	exit;
}

my $def_name = $output_file;
$def_name =~ s/.*\///g;
$def_name =~ s/\..*//g;
$def_name =~ s/-/_/g;

my $prefix = uc($def_name);
$prefix =~ s/_.*//g;
$prefix .= "_CLK_";

my $header = "/*
 * Copyright 2013 Mentor Graphics, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#ifndef __DTS_".uc($def_name)."_H
#define __DTS_".uc($def_name)."_H

/*
 * Clock definitions listed below should be in sync with the clocks
 * in $clk_enum enum of file $input_file
 *
 * NOTE: this header was automatically generated by
 * kernel scripts: script/gen_clk.pl
 * Please do not manually edit
 */
";
my $footer = "
#endif /* __DTS_".uc($def_name)."_H */
";

open (INFILE, "<$input_file")
	or die "Cant open $input_file\n";
my @lines = <INFILE>;
chomp(@lines);
my $content = join('', @lines);

$content =~ /$clk_enum {([\sa-z0-9_,]+),.*}/;
if (!defined $1) {
	print "Cant find enum $clk_enum in $input_file\n";
	exit;
}
my @clks = split(/,/,$1);

open (OUTFILE, ">$output_file")
	or die "Cant open $output_file\n";

print OUTFILE "$header";

for my $index(0 .. $#clks) {
	my $clk = $clks[$index];
	$clk =~ s/^\s+//;
	$clk = uc($clk);
	printf(OUTFILE "#define %-30s %d\n", $prefix.$clk, $index);
}

printf(OUTFILE "#define %-30s %d\n", $prefix.$clk_end, $#clks + 1);

print OUTFILE "$footer";

close (INFILE);
close (OUTFILE);

sub top_of_kernel_tree {
	my ($root) = @_;

	my @tree_check = (
		"COPYING", "CREDITS", "Kbuild", "MAINTAINERS", "Makefile",
		"README", "Documentation", "arch", "include", "drivers",
		"fs", "init", "ipc", "kernel", "lib", "scripts",
	);

	foreach my $check (@tree_check) {
		if (! -e $root . '/' . $check) {
			return 0;
		}
	}
	return 1;
}
