/*
 * PWM DEMO DRIVER
 *
 * Copyright (C) 2013 Mentor Graphics Inc.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/pwm.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/ctype.h>
#include "pwm-demo.h"

struct pwm_demo_data {
	struct pwm_device	*pwm;
	struct device		*dev;
	int			deactivated;
	void			(*exit)(struct device *);
};

static struct class *pwm_demo_class;

static ssize_t pwm_demo_show_active(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);

	return sprintf(buf, "%d\n", dd->props.active);
}

static ssize_t pwm_demo_store_active(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);
	unsigned long active;

	rc = kstrtoul(buf, 0, &active);
	if (rc)
		return rc;

	rc = -ENXIO;

	mutex_lock(&dd->ops_lock);
	if (dd->ops) {
		if (active > 1)
			rc = -EINVAL;
		else {
			dd->props.active = active;
			pwm_demo_update_status(dd);
			rc = count;
		}
	}
	mutex_unlock(&dd->ops_lock);

	return rc;
}
static ssize_t pwm_demo_show_polarity(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);

	return sprintf(buf, "%d\n", dd->props.polarity);
}

static ssize_t pwm_demo_store_polarity(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);
	unsigned long polarity;

	rc = kstrtoul(buf, 0, &polarity);
	if (rc)
		return rc;

	rc = -ENXIO;

	mutex_lock(&dd->ops_lock);
	if (dd->ops) {
		if (polarity > PWM_POLARITY_INVERSED)
			rc = -EINVAL;
		else {
			dd->props.polarity = polarity;
			pwm_demo_update_status(dd);
			rc = count;
		}
	}
	mutex_unlock(&dd->ops_lock);

	return rc;
}

static ssize_t pwm_demo_show_period(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);

	return sprintf(buf, "%d\n", dd->props.period);
}

static ssize_t pwm_demo_store_period(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);
	unsigned long period;

	rc = kstrtoul(buf, 0, &period);
	if (rc)
		return rc;

	rc = -ENXIO;

	mutex_lock(&dd->ops_lock);
	if (dd->ops) {
		if (period > dd->props.max_period ||
				period < dd->props.duty)
			rc = -EINVAL;
		else {
			dd->props.period = period;
			pwm_demo_update_status(dd);
			rc = count;
		}
	}
	mutex_unlock(&dd->ops_lock);

	return rc;
}

static ssize_t pwm_demo_show_duty(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);

	return sprintf(buf, "%d\n", dd->props.duty);
}

static ssize_t pwm_demo_store_duty(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);
	unsigned long duty;

	rc = kstrtoul(buf, 0, &duty);
	if (rc)
		return rc;

	rc = -ENXIO;

	mutex_lock(&dd->ops_lock);
	if (dd->ops) {
		if (duty > dd->props.period)
			rc = -EINVAL;
		else {
			dd->props.duty = duty;
			pwm_demo_update_status(dd);
			rc = count;
		}
	}
	mutex_unlock(&dd->ops_lock);

	return rc;
}

static int pwm_demo_update(struct pwm_demo_device *dd)
{
	struct pwm_demo_data *pd = pwm_demo_get_data(dd);
	int keep_active = 1;

	pwm_disable(pd->pwm);
	pwm_config(pd->pwm, dd->props.duty, dd->props.period);
	pwm_set_polarity(pd->pwm, dd->props.polarity);

	if ((!dd->props.duty &&
			dd->props.polarity == PWM_POLARITY_NORMAL) ||
			(dd->props.duty == dd->props.period &&
			 dd->props.polarity == PWM_POLARITY_INVERSED))
		keep_active = 0;

	if (keep_active) {
		if ((!dd->props.active && pd->deactivated) ||
				dd->props.active) {
			pwm_enable(pd->pwm);
			dd->props.active = 1;
			pd->deactivated = 0;
		}
	} else {
		dd->props.active = 0;
		pd->deactivated = 1;

	}

	return 0;
}

static void pwm_demo_device_release(struct device *dev)
{
	struct pwm_demo_device *dd = to_pwm_demo_device(dev);
	kfree(dd);
}

struct pwm_demo_device *pwm_demo_device_register(const char *name,
		struct device *parent, void *devdata,
		const struct pwm_demo_ops *ops,
		const struct pwm_demo_properties *props)
{
	struct pwm_demo_device *new_dd;
	int rc;

	pr_debug("pwm_demo_device_register: name=%s\n", name);

	new_dd = kzalloc(sizeof(struct pwm_demo_device), GFP_KERNEL);
	if (!new_dd)
		return ERR_PTR(-ENOMEM);

	mutex_init(&new_dd->ops_lock);
	mutex_init(&new_dd->update_lock);

	new_dd->dev.class = pwm_demo_class;
	new_dd->dev.parent = parent;
	new_dd->dev.release = pwm_demo_device_release;
	dev_set_name(&new_dd->dev, name);
	dev_set_drvdata(&new_dd->dev, devdata);

	/* Set default properties */
	if (props)
		memcpy(&new_dd->props, props,
				sizeof(struct pwm_demo_properties));
	rc = device_register(&new_dd->dev);
	if (rc) {
		kfree(new_dd);
		return ERR_PTR(rc);
	}

	new_dd->ops = ops;

	return new_dd;
}

void pwm_demo_device_unregister(struct pwm_demo_device *dd)
{
	if (!dd)
		return;
	mutex_lock(&dd->ops_lock);
	dd->ops = NULL;
	mutex_unlock(&dd->ops_lock);

	device_unregister(&dd->dev);
}


static const struct pwm_demo_ops dops = {
	.update_status	= pwm_demo_update
};

static struct of_device_id pwm_user_of_match[] = {
	{ .compatible = "pwm-user" },
	{ }
};
MODULE_DEVICE_TABLE(of, pwm_user_of_match);

static int pwm_demo_probe(struct platform_device *pdev)
{
	struct pwm_demo_properties props;
	struct pwm_demo_device *dd;
	struct pwm_demo_data *pd;
	int ret;

	pd = devm_kzalloc(&pdev->dev, sizeof(*pd), GFP_KERNEL);
	if (!pd) {
		dev_err(&pdev->dev, "no memory for state\n");
		ret = -ENOMEM;
		goto err_alloc;
	}
	pd->deactivated = 0;
	pd->dev = &pdev->dev;
	pd->pwm = devm_pwm_get(&pdev->dev, NULL);
	if (IS_ERR(pd->pwm)) {
		dev_err(&pdev->dev, "unable to request pwm for demo\n");
		ret = PTR_ERR(pd->pwm);
		goto err_alloc;
	}

	memset(&props, 0, sizeof(struct pwm_demo_properties));
	props.max_period = pd->pwm->period;
	props.period = pd->pwm->period;
	props.duty = props.period;
	props.polarity = pd->pwm->polarity;
	props.active = 1;

	dd = pwm_demo_device_register(dev_name(&pdev->dev), &pdev->dev, pd,
			&dops, &props);
	if (IS_ERR(dd)) {
		dev_err(&pdev->dev, "failed to register pwm demo device\n");
		ret = PTR_ERR(dd);
		goto err_alloc;
	}

	platform_set_drvdata(pdev, dd);
	pwm_demo_update_status(dd);
	return 0;

err_alloc:
	return ret;
}

static int pwm_demo_remove(struct platform_device *pdev)
{
	struct pwm_demo_device *dd = platform_get_drvdata(pdev);
	struct pwm_demo_data *pd = pwm_demo_get_data(dd);

	pwm_demo_device_unregister(dd);
	pwm_disable(pd->pwm);
	if (pd->exit)
		pd->exit(&pdev->dev);
	return 0;
}

static struct platform_driver pwm_demo_driver = {
	.driver		= {
		.name		= "pwm-demo",
		.owner		= THIS_MODULE,
		.of_match_table = of_match_ptr(pwm_user_of_match),
	},
	.probe		= pwm_demo_probe,
	.remove		= pwm_demo_remove,
};


static struct device_attribute pwm_demo_device_attributes[] = {
	__ATTR(period, 0644, pwm_demo_show_period,
		  pwm_demo_store_period),
	__ATTR(duty, 0644, pwm_demo_show_duty,
			pwm_demo_store_duty),
	__ATTR(polarity, 0644, pwm_demo_show_polarity,
			pwm_demo_store_polarity),
	__ATTR(active, 0644, pwm_demo_show_active,
			pwm_demo_store_active),
	__ATTR_NULL,
};

static void __exit pwm_demo_class_exit(void)
{
	platform_driver_unregister(&pwm_demo_driver);
	class_destroy(pwm_demo_class);
}

static int __init pwm_demo_class_init(void)
{
	pwm_demo_class = class_create(THIS_MODULE, "pwm");
	if (IS_ERR(pwm_demo_class)) {
		pr_warn("Unable to create pwm demo class; errno = %ld\n",
				PTR_ERR(pwm_demo_class));
		return PTR_ERR(pwm_demo_class);
	}
	pwm_demo_class->dev_attrs = pwm_demo_device_attributes;

	return platform_driver_register(&pwm_demo_driver);
}

module_init(pwm_demo_class_init);
module_exit(pwm_demo_class_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Abbas Raza <abbas_raza@mentor.com>");
MODULE_DESCRIPTION("PWM DEMO DRIVER");
MODULE_ALIAS("platform:pwm-demo");
