/*
 * Watchdog driver for Inter-Node-Communication (INC)
 *
 * Copyright (C) 2014 Robert Bosch Car Multimedia GmbH
 *               Martin Kalms <Martin.Kalms@de.bosch.com>
 *
 * This watchdog driver establishes an INC communication link to another
 * communication partner (the external watchdog master) and offers an ioctl
 * interface to trigger the transmission of a keep-alive watchdog message.
 *
 * As an extended security feature the INC watchdog driver refuses to send
 * further keep-alive watchdog messages in case the communication partner has
 * not (yet) confirmed the reception of this message by sending a related
 * response message. The only exception for this rule is the transmission of
 * the initial keep-alive watchdog message, which is send in any case.
 *
 * If the transmission of keep-alive message was refused to be send to the
 * communication partner due to a not yet confirmed previous one, then this
 * request for the transmission of a keep-alive message is memorized and
 * immediately catched up at the time the keep-alive response message is
 * received.
 *
 * Beside the keep-alive trigger for the external INC watchdog, this watchdog
 * driver also triggers the internal IMX2+ watchdog in parallel. The control
 * of the IMX2+ watchdog driver from within the INC watchdog driver might look
 * unusual and unnecessary, as these drivers could also co-exist independently
 * from each other without performing such a kind of cascaded watchdog trigger.
 * Unfortunately two independent watchdog drivers being used in parallel from a
 * user space application doesn't meet the requirements of the systemd, where
 * the watchdog shall be triggered from. Systemd can only trigger one watchdog
 * device at a time. Systemd is only able to trigger two watchdogs in parallel
 * (external and internal watchdog) with a cascaded watchdog trigger like the
 * INC/IMX2+ watchdog driver duo.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of.h>

#include <linux/watchdog.h>

#include <linux/net.h>
#include <linux/in.h>
#include <linux/inc.h>
#include <linux/inc_ports.h>
#include <linux/dgram_service.h>

#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/kthread.h>
#include <linux/ktime.h>
#include <linux/fs.h>

#define DRIVER_NAME                                                    "inc_wdt"

#define INC_WDT_RX_TASK_NAME                                        "inc_wdt_rx"

#define INC_WDT_OF_NODE_NAME                                           "inc_wdt"
#define INC_WDT_OF_INC_NODE_PROPERTY_NAME                             "inc_node"
#define INC_WDT_OF_INC_LOCAL_ADDRESS_PROPERTY_NAME                  "local-addr"
#define INC_WDT_OF_INC_REMOTE_ADDRESS_PROPERTY_NAME                "remote-addr"

#define INC_WDT_RECEIVE_BUFFER_LEN                                             1
#define INC_WDT_DGRAM_MAX			      INC_WDT_RECEIVE_BUFFER_LEN

#define INC_WDT_MSGID_WDGS_WDG_C_WATCHDOG                                   0x62
#define INC_WDT_MSGID_WDG_WDGS_R_WATCHDOG                                   0x63

#define INC_WDT_DGRAM_RECV_TIMEOUT_S                                           2

#define INC_WDT_IMX2_WD_TIMEOUT_S                                              6
#define INC_WDT_IMX2_WD_PRETIMEOUT_S                                           1

#define INC_WDT_IMX2_DEVICE_NAME                             "/dev/watchdog_imx"

struct inc_wdt_data {
	struct timer_list period_timer;
	struct socket *socket;
	struct sk_dgram *dgram;
	struct task_struct *rx_task;
	char recv_buffer[INC_WDT_RECEIVE_BUFFER_LEN];
	bool device_registered;
	bool inc_wdg_response_received;
	bool inc_wdg_command_prevented;
	struct timespec last_sent;
	struct timespec last_received;
};

static DEFINE_MUTEX(inc_wdt_mutex);

static unsigned int inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a; arr[1] = b; arr[2] = c; arr[3] = d;
	return *(unsigned int *) arr;
}

static int inc_wdt_get_inc_addresses(char **local, char **remote)
{
	int errno = 0;
	struct device_node *node = NULL;
	const void *property = NULL;

	node = of_find_node_by_name(NULL, INC_WDT_OF_NODE_NAME);
	if (!node)
		return -ENODEV;

	property = of_get_property(
		node,
		INC_WDT_OF_INC_NODE_PROPERTY_NAME,
		NULL);
	if (!property)
		return -ENODEV;

	node = of_find_node_by_path((char *) property);
	if (!node)
		return -ENODEV;

	*local = (char *) of_get_property(
		node,
		INC_WDT_OF_INC_LOCAL_ADDRESS_PROPERTY_NAME,
		NULL);
	if (!*local)
		return -ENODEV;

	*remote = (char *) of_get_property(
		node,
		INC_WDT_OF_INC_REMOTE_ADDRESS_PROPERTY_NAME,
		NULL);
	if (!remote)
		return -ENODEV;

	return errno;
}

static int inc_wdt_establish_scc_inc_com(struct watchdog_device *wdev)
{
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(wdev);
	int errno = 0;
	struct sockaddr_in local;
	struct sockaddr_in remote;
	struct timeval timeval = {INC_WDT_DGRAM_RECV_TIMEOUT_S, 0};
	char *local_address_str = NULL;
	char *remote_address_str = NULL;

	errno = sock_create_kern(AF_INC, SOCK_STREAM, 0, &driver_data->socket);
	if (errno < 0)
		return errno;

	errno = kernel_setsockopt(
		driver_data->socket,
		SOL_SOCKET,
		SO_RCVTIMEO, (char *)&timeval,
		sizeof(timeval));
	if (errno < 0)
		goto err_setsockopt;

	driver_data->dgram = dgram_init(driver_data->socket, INC_WDT_DGRAM_MAX,
			NULL);
	if (!driver_data->dgram) {
		errno = -EINVAL;
		goto err_dgram_init;
	}

	errno = inc_wdt_get_inc_addresses(
		&local_address_str,
		&remote_address_str);
	if (errno < 0)
		goto err_get_inc_addresses;

	local.sin_family = AF_INET;
	local.sin_addr.s_addr = inet_addr(local_address_str);
	local.sin_port = htons(WDG_PORT);

	errno = kernel_bind(
		driver_data->socket,
		(struct sockaddr *) &local,
		sizeof(local));
	if (errno < 0)
		goto err_kernel_bind;

	remote.sin_family = AF_INET;
	remote.sin_addr.s_addr = inet_addr(remote_address_str);
	remote.sin_port = htons(WDG_PORT);

	errno = kernel_connect(
		driver_data->socket,
		(struct sockaddr *) &remote,
		sizeof(remote),
		0);

	if (errno >= 0)
		return errno;

err_kernel_bind: /* fall trough */
err_get_inc_addresses:

	dgram_exit(driver_data->dgram);

err_dgram_init: /* fall trough */
err_setsockopt:

	kernel_sock_shutdown(driver_data->socket, SHUT_RDWR);

	return errno;
}

static void inc_wdt_release_scc_inc_com(struct watchdog_device *wdev)
{
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(wdev);
	int errno = 0;

	if (driver_data->dgram)
		errno = dgram_exit(driver_data->dgram);

	if (driver_data->socket) {
		errno = kernel_sock_shutdown(driver_data->socket, SHUT_RDWR);

		if (errno >= 0)
			sock_release(driver_data->socket);
	}

	if (errno < 0)
		dev_err(
			wdev->parent,
			"%s failed, errno = %d\n",
			__func__,
			errno);
}

static int inc_wdt_send_wdg_msg(struct watchdog_device *wdev, bool locked)
{
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(wdev);
	int errno = 0;
	char buffer[1];
	buffer[0] = INC_WDT_MSGID_WDGS_WDG_C_WATCHDOG;

	if (!locked)
		mutex_lock(&inc_wdt_mutex);

	if (driver_data->inc_wdg_response_received) {
		errno = dgram_send(driver_data->dgram, buffer, sizeof(buffer));
		if (errno >= 0) {
			driver_data->inc_wdg_command_prevented = false;
			driver_data->inc_wdg_response_received = false;
			ktime_get_ts(&driver_data->last_sent);
		} else {
			dev_err(
				wdev->parent,
				"%s failed, errno = %d\n",
				__func__,
				errno);
		}
	} else {
		driver_data->inc_wdg_command_prevented = true;
		errno = -EBUSY;
	}

	if (!locked)
		mutex_unlock(&inc_wdt_mutex);

	return errno;
}

static int inc_wdt_recv_inc_msg(struct watchdog_device *wdev)
{
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(wdev);
	int errno = 0;

	errno = dgram_recv(
		driver_data->dgram,
		driver_data->recv_buffer,
		INC_WDT_RECEIVE_BUFFER_LEN);
	if ((errno < 0) && (errno != -EAGAIN))
		dev_err(
			wdev->parent,
			"%s failed, errno = %d\n",
			__func__,
			errno);

	return errno;
}

static void inc_wdt_on_inc_msg_received(struct watchdog_device *wdev)
{
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(wdev);

	mutex_lock(&inc_wdt_mutex);

	if (driver_data->recv_buffer[0] == INC_WDT_MSGID_WDG_WDGS_R_WATCHDOG) {
		driver_data->inc_wdg_response_received = true;
		ktime_get_ts(&driver_data->last_received);
		if (driver_data->inc_wdg_command_prevented)
			inc_wdt_send_wdg_msg(wdev, true);
	}

	mutex_unlock(&inc_wdt_mutex);
}

static int inc_wdt_rx_task(void *arg)
{
	struct watchdog_device *wdev = (struct watchdog_device *) arg;

	do {
		if (inc_wdt_recv_inc_msg(wdev) > 0)
			inc_wdt_on_inc_msg_received(wdev);
	} while (!kthread_should_stop());

	return 0;
}

static int inc_wdt_start(struct watchdog_device *wdev)
{
	return 0;
}

static int inc_wdt_stop(struct watchdog_device *wdev)
{
	return 0;
}

static int inc_wdt_set_imx2_wdt_timeout(struct watchdog_device *wdev,
					struct file *filp)
{
	int errno = 0;

	int set = INC_WDT_IMX2_WD_TIMEOUT_S;
	errno = filp->f_op->unlocked_ioctl(
			filp,
			WDIOC_SETTIMEOUT,
			(unsigned long)&set);
	if (errno >= 0)
		dev_info(
			wdev->parent,
			"IMX2+ WDIOC_SETTIMEOUT (%u)\n",
			set);
	else
		return errno;

	set = INC_WDT_IMX2_WD_PRETIMEOUT_S;
	errno = filp->f_op->unlocked_ioctl(
			filp,
			WDIOC_SETPRETIMEOUT,
			(unsigned long)&set);
	if (errno >= 0)
		dev_info(
			wdev->parent,
			"IMX2+ WDIOC_SETPRETIMEOUT (%u)\n",
			set);

	return errno;
}

static int inc_wdt_ping_imx2(struct watchdog_device *wdev)
{
	static struct file *filp = (void *) -ENXIO;
	static int errno = 1;
	mm_segment_t oldfs;

	/*  If this function once lead to an error, immediately return with
	 *  the same. */
	if (errno < 0)
		return errno;

	oldfs = get_fs();
	set_fs(KERNEL_DS);

	/* As the device INC_WDT_IMX2_DEVICE_NAME might become available later
	 * than this INC watchdog device, we repetitive try to open it with
	 * each call of this function, until it is successfull. */
	if (IS_ERR(filp) && (current->fs != NULL)) {
		/* As usual for watchdog devices, they are once opened
		 * and never closed. So there is intentionally no matching
		 * filp_close for the following filp_open. */
		filp = filp_open(INC_WDT_IMX2_DEVICE_NAME, O_WRONLY, 0);
		if (!IS_ERR(filp)) {
			errno = inc_wdt_set_imx2_wdt_timeout(wdev, filp);
			if (errno < 0)
				goto err_out;
		}
	}

	/* If the device INC_WDT_IMX2_DEVICE_NAME was successfully opened,
	 * then trigger the WDIOC_KEEPALIVE IO-control. */
	if (!IS_ERR(filp)) {
		errno = filp->f_op->unlocked_ioctl(
				filp,
				WDIOC_KEEPALIVE,
				0);
	}

err_out:
	set_fs(oldfs);

	if (errno < 0)
		dev_err(
			wdev->parent,
			"%s failed, errno = %d\n",
			__func__,
			errno);
	return errno;
}

static int inc_wdt_ping(struct watchdog_device *wdev)
{
	int errno = 0;

	errno = inc_wdt_ping_imx2(wdev);

	if (errno >= 0)
		errno = inc_wdt_send_wdg_msg(wdev, false);

	return errno;
}

static const struct watchdog_info inc_wdt_info = {
	.options = WDIOF_KEEPALIVEPING | WDIOF_MAGICCLOSE,
	.identity = "INC Watchdog",
};

static const struct watchdog_ops inc_wdt_ops = {
	.owner = THIS_MODULE,
	.start = inc_wdt_start,
	.stop = inc_wdt_stop,
	.ping = inc_wdt_ping,
};

static struct watchdog_device inc_wdt_dev = {
	.info = &inc_wdt_info,
	.ops = &inc_wdt_ops,
	.timeout = 4,
	.min_timeout = 1,
	.max_timeout = 4,
};

static ssize_t show_debug_stats(
	struct device *dev, struct device_attribute *attr, char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct watchdog_device *inc_wdt_dev = platform_get_drvdata(pdev);
	struct inc_wdt_data *driver_data = watchdog_get_drvdata(inc_wdt_dev);
	struct timespec now;
	s64 ns_last_sent;
	s64 ns_last_received;
	s64 ns_now;
	ssize_t ret;

	mutex_lock(&inc_wdt_mutex);
	ktime_get_ts(&now);
	ns_now = timespec_to_ns(&now);
	ns_last_sent = timespec_to_ns(&driver_data->last_sent);
	ns_last_received = timespec_to_ns(&driver_data->last_received);
	ret = scnprintf(buf, PAGE_SIZE, "%lld %lld %lld %d %d\n",
			ns_now, ns_last_sent, ns_last_received,
			driver_data->inc_wdg_command_prevented,
			driver_data->inc_wdg_response_received);
	mutex_unlock(&inc_wdt_mutex);

	return ret;
}

DEVICE_ATTR(debug_stats, S_IRUGO, show_debug_stats, NULL);

static int inc_wdt_probe(struct platform_device *pdev)
{
	int errno = 0;
	struct task_struct *task;
	struct inc_wdt_data *driver_data;

	driver_data = devm_kzalloc(
		&pdev->dev,
		sizeof(*driver_data),
		GFP_KERNEL);
	if (!driver_data) {
		errno = -ENOMEM;
		goto err_alloc;
	}

	driver_data->socket = NULL;
	driver_data->dgram = NULL;
	driver_data->rx_task = NULL;
	driver_data->device_registered = false;
	/* set to true for the initial INC watchdog message transmission */
	driver_data->inc_wdg_response_received = true;
	driver_data->inc_wdg_command_prevented = false;

	watchdog_set_nowayout(&inc_wdt_dev, WATCHDOG_NOWAYOUT);
	watchdog_set_drvdata(&inc_wdt_dev, driver_data);
	inc_wdt_dev.parent = &pdev->dev;
	platform_set_drvdata(pdev, &inc_wdt_dev);

	errno = inc_wdt_establish_scc_inc_com(&inc_wdt_dev);
	if (errno < 0)
		goto err_scc_inc_com;

	task = kthread_run(
		inc_wdt_rx_task,
		&inc_wdt_dev,
		INC_WDT_RX_TASK_NAME);
	if (IS_ERR(task))
		errno = PTR_ERR(task);
	if (errno < 0)
		goto err_thread_run;

	driver_data->rx_task = task;

	errno = watchdog_register_device(&inc_wdt_dev);
	if (errno < 0)
		goto err_watchdog_register;

	driver_data->device_registered = true;

	errno = device_create_file(&pdev->dev, &dev_attr_debug_stats);
	if (errno)
		dev_err(&pdev->dev, "create file debug_stats failed\n");

	dev_info(&pdev->dev,
		"INC Watchdog Timer enabled. (nowayout=%d)\n",
		(inc_wdt_dev.status & WATCHDOG_NOWAYOUT_INIT_STATUS) ? 1 : 0);

	return 0;

err_watchdog_register:
	kthread_stop(driver_data->rx_task);

err_thread_run:
	inc_wdt_release_scc_inc_com(&inc_wdt_dev);

err_scc_inc_com:
	devm_kfree(&pdev->dev, driver_data);

err_alloc:
	platform_set_drvdata(pdev, NULL);

	dev_err(
		&pdev->dev,
		"%s failed, errno = %d\n",
		__func__,
		errno);

	return errno;
}

static int inc_wdt_remove(struct platform_device *pdev)
{
	int errno = 0;
	struct watchdog_device *wdev = NULL;
	struct inc_wdt_data *driver_data = NULL;

	wdev = platform_get_drvdata(pdev);
	if (!wdev) {
		errno = -ENODATA;
		goto err;
	}

	driver_data = watchdog_get_drvdata(wdev);
	if (!driver_data) {
		errno = -ENODATA;
		goto err;
	}

	device_remove_file(&pdev->dev, &dev_attr_debug_stats);
	if (driver_data->device_registered)
		watchdog_unregister_device(&inc_wdt_dev);

	if (driver_data->rx_task)
		errno = kthread_stop(driver_data->rx_task);

	inc_wdt_release_scc_inc_com(wdev);

	watchdog_set_drvdata(wdev, NULL);
	platform_set_drvdata(pdev, wdev);

	if (errno < 0)
		goto err;

	return 0;
err:
	dev_err(
		&pdev->dev,
		"%s failed, errno = %d\n",
		__func__,
		errno);

	return errno;
}

static const struct of_device_id inc_wdt_of_match[] = {
	{ .compatible = "rbcm,inc_wdt", },
	{}
};

static struct platform_driver inc_wdt_driver = {
	.probe = inc_wdt_probe,
	.remove = inc_wdt_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = DRIVER_NAME,
		.of_match_table = inc_wdt_of_match,
	},
};

module_platform_driver(inc_wdt_driver);

MODULE_DESCRIPTION("Watchdog driver for Inter-Node-Communication (INC)");
MODULE_AUTHOR("Martin Kalms <martin.kalms@de.bosch.com>");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:inc_wdt");
