/*
 * linux/drivers/usb/gadget/utbridge_udc.c
 *
 * USB Driver for the Dual Role Unwired Technology USB Bridge/Hub
 *   UDC driver part
 *
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Martin Herzog (mherzog@de.adit-jv.com)
 *        and Andreas Pape (apape@de.adit-jv.com)
 *
 * Based on a reference driver by Sam Yeda (sam@unwiredtechnology.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/usb.h>
#include <linux/usb/gadget.h>

#include "utbridge.h"

#define REQ_DEV		(USB_TYPE_STANDARD | USB_RECIP_DEVICE)
#define REQ_DEV_IN	(REQ_DEV | USB_DIR_IN)
#define REQ_INTF	(USB_TYPE_STANDARD | USB_RECIP_INTERFACE)
#define REQ_INTF_IN	(REQ_INTF | USB_DIR_IN)
#define REQ_EP		(USB_TYPE_STANDARD | USB_RECIP_ENDPOINT)
#define REQ_EP_IN	(REQ_EP | USB_DIR_IN)

#define USB_REQ_UTB_BHOST_DISCONN	0xff
#define REQ_BHOST_DISCONN		0xff
#define WVALUE_BHOST_DISCONN		0xffff
#define WINDEX_BHOST_DISCONN		0x0000

enum utb_ctrl_state {
	REQ_FOR_SETUP,
	DATA_STAGE_IN,
	DATA_STAGE_OUT,
	STATUS_STAGE_IN,
};
static const char *const ctrl_state_name[] = {
	"SETUP     ",
	"DATA_IN   ",
	"DATA_OUT  ",
	"STATUS_IN ",
};

struct utb_core {
	/* GADGET side support */
	struct usb_gadget_driver *driver;
	struct usb_gadget gadget;
	int address;
	u16 devstatus;

	/* USB interface to host */
	struct usb_utbridge *br_dev;

	/* UDC */
	spinlock_t lock;
	unsigned running:1;
	unsigned active:1;

	struct usb_ctrlrequest ctrl_req;
	struct usb_ctrlrequest ctrl_rsp;
	u8 ctrl_sendbuf[USB_MAX_CTRL_PAYLOAD];
	enum utb_ctrl_state ctrl_state;
	enum usb_device_state bhoststate;

#ifdef CONFIG_USB_GADGET_DEBUG_FS
	struct dentry *debugfs_root;
	struct dentry *debugfs_bridgestatus;
#endif
};

#define USE_COPY_ON_EPX
struct utb_request {
	struct list_head queue; /* ep's requests */
	struct usb_request req;
#ifdef USE_COPY_ON_EPX
	char *ubuf;
#endif
};

/*
 *  container macros
 */
#define usb_ep_to_utbridge_ep(e)   container_of(e, struct utb_ep, ep);
#define gadget_to_utbridge_core(g) container_of(g, struct utb_core, gadget);
#define gadget_dev_to_utbridge_core(d) \
				container_of(d, struct utb_core, gadget.dev);
#define usb_request_to_utbridge_request(r) \
				container_of(r, struct utb_request, req);

/*
 * forward declarations
 */
static void utbridge_ep0_urb_callback(struct urb *urb);
static void utbridge_epX_urb_callback(struct urb *urb);
static int utbridge_handle_ep_queue(struct utb_core *core, struct utb_ep *ep);

/*
 * helpers
 */
static inline int enter_in_progress(struct utb_ep *ep, const char *func)
{
	int ret = cmpxchg(&ep->inprogress, 0, 1);
	smp_wmb();	/* inprogress */
	dev_dbg(&ep->br_dev->interface->dev, "%s: ENTER in progress %s (%s)\n",
		ep->ep.name, ret ? "FAIL" : "SUCCESS", func);
	return ret;
}

static inline void clr_in_progress(struct utb_ep *ep, const char *func)
{
	ep->inprogress = 0;
	smp_wmb();	/* inprogress */
	dev_dbg(&ep->br_dev->interface->dev, "%s: CLEAR in progress (%s)\n",
		ep->ep.name, func);
}

static struct utb_ep *ep_from_a_addr(struct utb_core *core, u8 addr)
{
	int i;

	if ((addr & ~USB_DIR_IN) == 0)
		return &core->br_dev->eps[0];
	for (i = 1; i < core->br_dev->max_eps; i++) {
		struct utb_ep *ep = &core->br_dev->eps[i];

		if ((ep->b_desc) && (ep->a_desc->bEndpointAddress == addr))
			return ep;
	}
	return NULL;
}

static struct utb_ep *ep_from_b_addr(struct utb_core *core, u8 addr)
{
	int i;

	if ((addr & ~USB_DIR_IN) == 0)
		return &core->br_dev->eps[0];
	if (!core->active)
		return NULL;
	for (i = 1; i < core->br_dev->max_eps; i++) {
		struct utb_ep *ep = &core->br_dev->eps[i];

		if ((ep->b_desc) && (ep->b_desc->bEndpointAddress == addr))
			return ep;
	}
	return NULL;
}

static void utbridge_bhost_set_state(struct utb_core *core,
		enum usb_device_state state)
{
	if (core->bhoststate == state)
		return;

	core->bhoststate = state;

	switch (state) {
	case USB_STATE_ADDRESS:
		if (core->driver && core->driver->resume) {
			dev_info(&core->br_dev->interface->dev,
				"state: ISSUE RESUME\n");
			core->driver->resume(&core->gadget);
		}
		break;
	case USB_STATE_NOTATTACHED:
		if (core->driver && core->driver->suspend) {
			dev_info(&core->br_dev->interface->dev,
				"state: ISSUE SUSPEND\n");
			core->driver->suspend(&core->gadget);
		}
		break;
	default:
		/* other states not used here (yet) */
		break;
	}
}

static inline void print_ep0_setup(struct utb_core *core,
	struct usb_ctrlrequest *setup, const char *prefix, const char *func)
{
	dev_dbg(&core->br_dev->interface->dev,
		"ep0: %s: %02x.%02x v%04x i%04x l%d %s (%s)\n", prefix,
		setup->bRequestType, setup->bRequest, setup->wValue,
		setup->wIndex, setup->wLength,
		(prefix[1] == 'E') ? ctrl_state_name[core->ctrl_state] : "",
		func);
}

/*
 * Debug filesystem
 */
#ifdef CONFIG_USB_GADGET_DEBUG_FS
#include <linux/debugfs.h>

static int bridgestatus_dbg_show(struct seq_file *s, void *p)
{
	struct utb_request *req;
	int i;
	unsigned long flags;
	struct utb_core *core = s->private;

	for (i = 1; i < core->br_dev->max_eps; i++) {
		struct utb_ep *ep = &core->br_dev->eps[i];

		spin_lock_irqsave(&ep->lock, flags);
		if (!ep->b_desc) {
			spin_unlock_irqrestore(&ep->lock, flags);
			continue;
		}
		seq_printf(s, "B-Host %s queue%s:\n", ep->ep.name,
				ep->inprogress ? " (in progress)" : "");
		list_for_each_entry(req, &ep->queue, queue) {
			seq_printf(s, " request %p (%d)\n", &req->req,
					req->req.length);
		}
		spin_unlock_irqrestore(&ep->lock, flags);
	}

	return 0;
}

static int bridgestatus_dbg_open(struct inode *inode, struct file *file)
{
	return single_open(file, bridgestatus_dbg_show, inode->i_private);
}

static const struct file_operations bridgestatus_dbg_fops = {
	.owner		= THIS_MODULE,
	.open		= bridgestatus_dbg_open,
	.llseek		= seq_lseek,
	.read		= seq_read,
	.release	= single_release,
};

static void utbridge_init_debugfs(struct utb_core *core)
{
	struct dentry *root, *bridgestatus;

	root = debugfs_create_dir(dev_name(&core->br_dev->pdev->dev), NULL);
	if (IS_ERR(root) || !root)
		goto err_root;

	bridgestatus = debugfs_create_file("bridgestatus", S_IRUGO, root, core,
			&bridgestatus_dbg_fops);
	if (!bridgestatus)
		goto err_bridgestatus;

	core->debugfs_root = root;
	core->debugfs_bridgestatus = bridgestatus;
	return;
err_bridgestatus:
	debugfs_remove(root);
err_root:
	dev_err(&core->br_dev->interface->dev, "debugfs is not available\n");
}

static void utbridge_cleanup_debugfs(struct utb_core *core)
{
	debugfs_remove(core->debugfs_bridgestatus);
	debugfs_remove(core->debugfs_root);
	core->debugfs_bridgestatus = NULL;
	core->debugfs_root = NULL;
}

#else
static inline void utbridge_init_debugfs(struct utb_core *core)
{
}

static inline void utbridge_cleanup_debugfs(struct utb_core *core)
{
}
#endif

/*
 * sysfs attributes
 */

static ssize_t show_bhoststate(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct utb_core *core = gadget_dev_to_utbridge_core(dev);

	return sprintf(buf, "%d\n", core->bhoststate);
}
static DEVICE_ATTR(bhoststate, S_IRUGO, show_bhoststate, NULL);

static ssize_t show_function(struct device *dev, struct device_attribute *attr,
		char *buf)
{
	struct utb_core *core = gadget_dev_to_utbridge_core(dev);

	if (!core->driver || !core->driver->function)
		return 0;
	return scnprintf(buf, PAGE_SIZE, "%s\n", core->driver->function);
}
static DEVICE_ATTR(function, S_IRUGO, show_function, NULL);

static struct device_attribute *utbridge_udc_attr[] = {
	&dev_attr_function,
	&dev_attr_bhoststate,
};
#define NUM_OF_ATTRS ARRAY_SIZE(utbridge_udc_attr)

/*
 * urb handling
 */
static int utbridge_submit_urb(struct utb_ep *ep)
{
	int ret;
	struct usb_utbridge *br_dev = ep->br_dev;

	/* Track submitted URB's */
	usb_anchor_urb(ep->urb, &ep->submitted);

	/* do it */
	ret = usb_submit_urb(ep->urb, GFP_ATOMIC);

	if (ret < 0) {
		dev_err(&br_dev->interface->dev,
			"%s: failed submitting urb, error %d",
			ep->ep.name, ret);
		usb_unanchor_urb(ep->urb);
	}

	return ret;
}

static int utbridge_ep0_req_setup(struct utb_core *core)
{
	struct usb_request _req;
	struct utb_ep *ep = &core->br_dev->eps[0];

	dev_dbg(&core->br_dev->interface->dev,
		"ep0: waiting for setup packet...\n");
	core->ctrl_state = REQ_FOR_SETUP;
	core->ctrl_req.bRequestType = VEND_RD_BMREQTYPE;
	core->ctrl_req.bRequest = VEND_RD_BREQ;
	core->ctrl_req.wValue = 0;
	core->ctrl_req.wIndex = 0;
	core->ctrl_req.wLength = sizeof(struct usb_ctrlrequest);

	_req.length = core->ctrl_req.wLength;
	_req.buf = &ep->buffer;
	_req.zero = 0;

	usb_fill_control_urb(ep->urb, core->br_dev->udev,
			usb_rcvctrlpipe(core->br_dev->udev, 0),
			(unsigned char *) &core->ctrl_req, _req.buf,
			_req.length, utbridge_ep0_urb_callback, core);
	return utbridge_submit_urb(ep);
}

static int utbridge_ep0_read(struct utb_core *core,
		struct usb_ctrlrequest *ctlreq, struct usb_request *_req)
{
	struct utb_ep *ep = &core->br_dev->eps[0];

	core->ctrl_req.bRequestType = VEND_RD_BMREQTYPE;
	core->ctrl_req.bRequest = VEND_RD_BREQ;
	core->ctrl_req.wValue = ctlreq->wValue;
	core->ctrl_req.wIndex = ctlreq->wIndex;
	core->ctrl_req.wLength = ctlreq->wLength;

	print_ep0_setup(core, &core->ctrl_req,
				ctrl_state_name[core->ctrl_state], __func__);

	usb_fill_control_urb(ep->urb, core->br_dev->udev,
			usb_rcvctrlpipe(core->br_dev->udev, 0),
			(unsigned char *) &core->ctrl_req, _req->buf,
			_req->length, utbridge_ep0_urb_callback, core);
	return utbridge_submit_urb(ep);
}

static int utbridge_ep0_write(struct utb_core *core,
		struct usb_ctrlrequest *ctlreq, struct usb_request *_req)
{
	struct utb_ep *ep = &core->br_dev->eps[0];

	if ((ctlreq->wIndex & 0x0001) == 0)
		ctlreq->wLength = 0;

	core->ctrl_req.bRequestType = VEND_WR_BMREQTYPE;
	core->ctrl_req.bRequest = VEND_WR_BREQ;
	core->ctrl_req.wValue = ctlreq->wValue;
	core->ctrl_req.wIndex = ctlreq->wIndex;
	core->ctrl_req.wLength = ctlreq->wLength;

	print_ep0_setup(core, &core->ctrl_req,
			ctrl_state_name[core->ctrl_state], __func__);

#ifdef VERBOSE_DEBUG
	print_hex_dump(KERN_DEBUG, "ep0: data: ", DUMP_PREFIX_NONE, 16, 1,
			_req->buf, _req->length, 0);
#endif

	ep->urb->transfer_flags = 0;

	usb_fill_control_urb(ep->urb, core->br_dev->udev,
			usb_sndctrlpipe(core->br_dev->udev, 0),
			(unsigned char *) &core->ctrl_req, _req->buf,
			_req->length, utbridge_ep0_urb_callback, core);
	return utbridge_submit_urb(ep);
}

static int utbridge_epX_rw(struct utb_core *core, struct utb_ep *ep,
		struct usb_request *_req)
{
	unsigned int io_pipe;
	struct usb_utbridge *br_dev = core->br_dev;

	dev_dbg(&br_dev->interface->dev, "%s %s(%d)\n", ep->ep.name,
		usb_endpoint_dir_in(ep->a_desc) ? "read" : "write",
		_req->length);

	ep->urb->transfer_flags = 0;

	/* Fill URB based on endpoint's transfer type */
	switch (usb_endpoint_type(ep->a_desc)) {
	case USB_ENDPOINT_XFER_BULK:
		if (usb_endpoint_dir_in(ep->a_desc))
			io_pipe = usb_rcvbulkpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		else
			io_pipe = usb_sndbulkpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		usb_fill_bulk_urb(ep->urb, br_dev->udev, io_pipe, _req->buf,
				_req->length, utbridge_epX_urb_callback, core);

		if (usb_pipeout(io_pipe) && _req->zero)
			ep->urb->transfer_flags |= URB_ZERO_PACKET;
		break;
	case USB_ENDPOINT_XFER_ISOC:
		/* NO isoc_mult support _req->length <=
		 *				usb_endpoint_maxp(&ep->desc)*/
		if (usb_endpoint_dir_in(ep->a_desc))
			io_pipe = usb_rcvisocpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		else
			io_pipe = usb_sndisocpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		ep->urb->dev = br_dev->udev;
		ep->urb->pipe = io_pipe;
		ep->urb->number_of_packets = 1;
		ep->urb->transfer_buffer = _req->buf;
		ep->urb->transfer_buffer_length = _req->length;
		ep->urb->complete = utbridge_epX_urb_callback;
		ep->urb->context = core;
		ep->urb->interval = 1 << (ep->a_desc->bInterval - 1);
		ep->urb->transfer_flags = URB_ISO_ASAP;

		ep->urb->iso_frame_desc[0].length = _req->length;
		ep->urb->iso_frame_desc[0].offset = 0;
		break;
	case USB_ENDPOINT_XFER_INT:
		if (usb_endpoint_dir_in(ep->a_desc))
			io_pipe = usb_rcvintpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		else
			io_pipe = usb_sndintpipe(br_dev->udev,
						usb_endpoint_num(ep->a_desc));
		usb_fill_int_urb(ep->urb, br_dev->udev, io_pipe, _req->buf,
				_req->length, utbridge_epX_urb_callback, core,
				ep->a_desc->bInterval);
		break;
	default:
		return -EINVAL;
	}

	if (usb_pipein(io_pipe) && _req->short_not_ok)
		ep->urb->transfer_flags |= URB_SHORT_NOT_OK;

	return utbridge_submit_urb(ep);
}

/**
 * utbridge_handle_ctrl_req() - handles all incoming control transfers
 * IRQ context
 * @core: pointer to utbridge_core
 * @setup: pointer to the setup data for a USB device control request
 *
 * Return       0 - if the request was handled
 *        error code on errors
 */
static int utbridge_handle_ctrl_req(struct utb_core *core,
		struct usb_ctrlrequest *setup)
{
	struct utb_ep *ep0 = &core->br_dev->eps[0];
	struct utb_ep *ep2;
	int ret = -ENOSYS;
	unsigned w_index;
	unsigned w_value;
	char *buf;
	int save2fpga = 0;
	struct usb_request req;

	buf = core->ctrl_sendbuf;
	print_ep0_setup(core, setup, ctrl_state_name[0], __func__);

	w_index = le16_to_cpu(setup->wIndex);
	w_value = le16_to_cpu(setup->wValue);
	switch (setup->bRequest) {
	case USB_REQ_SET_ADDRESS:
		if (setup->bRequestType != REQ_DEV)
			break;
		core->address = w_value;
		dev_dbg(&core->br_dev->interface->dev,
			"ep0: set_address = %d\n", w_value);
		utbridge_bhost_set_state(core, USB_STATE_ADDRESS);

		save2fpga = 1;
		ret = 0;
		break;
	case USB_REQ_SET_FEATURE:
		if (setup->bRequestType == REQ_DEV) {
			ret = 0;
			switch (w_value) {
			case USB_DEVICE_REMOTE_WAKEUP:
				break;
			default:
				ret = -EOPNOTSUPP;
			}
			if (ret == 0) {
				core->devstatus |= (1 << w_value);
				dev_warn(&core->br_dev->interface->dev,
					"ep0: set devstatus = 0x%x\n",
					core->devstatus);
			}
		} else if (setup->bRequestType == REQ_EP) {
			/* endpoint halt */
			ep2 = ep_from_b_addr(core, w_index);
			if (!ep2 || usb_endpoint_num(ep2->a_desc) == 0) {
				ret = -EOPNOTSUPP;
				break;
			}
			ep2->halted = 1;
			ret = 0;
			dev_warn(&core->br_dev->interface->dev,
				"%s: set halt\n", ep2->ep.name);
		}
		break;
	case USB_REQ_CLEAR_FEATURE:
		if (setup->bRequestType == REQ_DEV) {
			ret = 0;
			switch (w_value) {
			case USB_DEVICE_REMOTE_WAKEUP:
				w_value = USB_DEVICE_REMOTE_WAKEUP;
				break;
			default:
				ret = -EOPNOTSUPP;
				break;
			}
			if (ret == 0) {
				core->devstatus &= ~(1 << w_value);
				dev_warn(&core->br_dev->interface->dev,
					"ep0: clear devstatus = 0x%x\n",
					core->devstatus);
			}
		} else if (setup->bRequestType == REQ_EP) {
			/* endpoint halt */
			ep2 = ep_from_b_addr(core, w_index);
			if (!ep2) {
				ret = -EOPNOTSUPP;
				break;
			}
			if (!ep2->wedged)
				ep2->halted = 0;
			ret = 0;
			dev_warn(&core->br_dev->interface->dev,
				"%s: clear halt\n", ep2->ep.name);
		}
		break;
	case USB_REQ_GET_STATUS:
		if (setup->bRequestType == REQ_DEV_IN
				|| setup->bRequestType == REQ_INTF_IN
				|| setup->bRequestType == REQ_EP_IN) {
			/*
			 * device: remote wakeup, selfpowered
			 * interface: nothing
			 * endpoint: halt
			 */
			if (setup->wLength > cpu_to_le16(0)) {
				if (setup->bRequestType == REQ_EP_IN) {
					ep2 = ep_from_b_addr(core, w_index);
					if (!ep2) {
						ret = -EOPNOTSUPP;
						break;
					}
					buf[0] = ep2->halted;
				} else if (setup->bRequestType == REQ_DEV_IN) {
					buf[0] = (u8) core->devstatus;
				} else {
					buf[0] = 0;
				}
			}
			if (setup->wLength > cpu_to_le16(1))
				buf[1] = 0;
			if (setup->wLength > cpu_to_le16(2))
				setup->wLength = cpu_to_le16(2);
			ret = 0;
			dev_warn(&core->br_dev->interface->dev,
				"ep0: get status 0x%x%x\n", buf[0], buf[1]);
			core->ctrl_state = DATA_STAGE_IN;
		}
		break;
	case USB_REQ_UTB_BHOST_DISCONN:
		/* B-Host (Relay Device) disconnected */
		if ((setup->bRequestType == REQ_BHOST_DISCONN) &&
				(w_value == WVALUE_BHOST_DISCONN) &&
				(w_index == WINDEX_BHOST_DISCONN)) {
			dev_info(&core->br_dev->interface->dev,
				"B-Host (Relay Device) disconnected\n");
			utbridge_bhost_set_state(core, USB_STATE_NOTATTACHED);

			clr_in_progress(ep0, __func__);

			/* Queue up a control request from B-Host */
			return utbridge_ep0_req_setup(core);
		}
		break;
	}

	if (ret == 0) {
		setup->wIndex = (setup->bRequest << 4) |
				UBS_PROCESS | UBS_RELAY;
		if (save2fpga)
			setup->wIndex |= UBS_STORE;

		req.length = setup->wLength;
		req.buf = buf;
		req.zero = 0;
		dev_dbg(&core->br_dev->interface->dev,
			"ep0: direct response (%s)\n", __func__);
		return utbridge_ep0_write(core, setup, &req);
	} else if (ret == -ENOSYS) {
		dev_dbg(&core->br_dev->interface->dev,
			"ep0: forward message (%s)\n", __func__);

		/* gadget driver handles all other requests.
		 * block until setup() returns; no reentrancy issues etc.
		 */
		if (core->driver && core->driver->setup) {
			clr_in_progress(ep0, __func__);
			spin_unlock(&ep0->lock);
			ret = core->driver->setup(&core->gadget,
							&core->ctrl_rsp);
			spin_lock(&ep0->lock);
		}
	}
	return ret;
}

/*
 * transfer data between URB and usb_request; caller must own lock
 * IRQ context
 */
static int utbridge_urb_transfer(struct utb_core *core, struct urb *urb,
		struct utb_ep *ep)
{
	struct utb_request *req;
	int ret = 0;

	if (!list_empty(&ep->queue)) {
		req = list_entry(ep->queue.next, struct utb_request, queue);
		dev_dbg(&core->br_dev->interface->dev,
			"%s: transfer urb %p (len=%d) for req %p (len=%d) status=%d\n",
			ep->ep.name, &urb, urb->actual_length, &req->req,
			req->req.length, urb->status);

		/* gadget driver completion */
		list_del_init(&req->queue);

		if (usb_pipeisoc(urb->pipe)) {
			dev_dbg(&core->br_dev->interface->dev,
				"%s: iso sof=%d (len=%d) err_cnt=%d status=%d\n",
				ep->ep.name, urb->start_frame,
				urb->iso_frame_desc[0].actual_length,
				urb->error_count,
				urb->iso_frame_desc[0].status);

			req->req.actual = urb->iso_frame_desc[0].actual_length;
			req->req.status = urb->iso_frame_desc[0].status;
		} else {
			req->req.actual = urb->actual_length;
			req->req.status = urb->status;
		}

#ifdef USE_COPY_ON_EPX
		if (ep != &core->br_dev->eps[0]) {
			if (!req->ubuf)
				dev_err(&core->br_dev->interface->dev,
					"%s: possible double completion !!!!\n",
					ep->ep.name);
			BUG_ON(!req->ubuf);
			if (req->req.status == 0) {
				if (usb_pipein(urb->pipe)) {
					dev_dbg(&core->br_dev->interface->dev,
						"%s: copy input data %p->%p (len=%d)\n",
						ep->ep.name, req->req.buf,
						req->ubuf, req->req.actual);
					memcpy(req->ubuf, req->req.buf,
							req->req.actual);
				} else {
					dev_dbg(&core->br_dev->interface->dev,
						"%s: finished output data %p->%p (len=%d)\n",
						ep->ep.name, req->req.buf,
						req->ubuf, req->req.actual);
				}
			}
			req->req.buf = req->ubuf;
			req->ubuf = NULL;
		}
#endif

		spin_unlock(&ep->lock);
		req->req.complete(&ep->ep, &req->req);
		spin_lock(&ep->lock);
	}

	if (ep == &core->br_dev->eps[0]) {
		if (usb_pipeout(urb->pipe))
			/* Queue up a control request from B-Host */
			ret = utbridge_ep0_req_setup(core);
		else if (core->ctrl_state == DATA_STAGE_OUT) {
			struct usb_request _req;
			_req.length = 0;
			_req.buf = NULL;
			_req.zero = 0;
			core->ctrl_rsp.wIndex = ((core->ctrl_rsp.bRequestType &
				USB_TYPE_MASK) << 7) |
				(core->ctrl_rsp.bRequest << 4) |
				UBS_PROCESS | UBS_RELAY;
			core->ctrl_rsp.wLength = 0;

			ret = utbridge_ep0_write(core, &core->ctrl_rsp, &_req);
		}
	}

	return ret;
}

/* IRQ context */
static void utbridge_ep0_urb_callback(struct urb *urb)
{
	struct utb_core *core = urb->context;
	struct usb_utbridge *br_dev = core->br_dev;
	unsigned long flags;
	struct utb_request *req;
	struct utb_ep *ep = &br_dev->eps[0];

	dev_dbg(&br_dev->interface->dev, "callback ep0: buf=%p (xfered=%d)\n",
		urb->transfer_buffer, urb->actual_length);

	spin_lock_irqsave(&core->lock, flags);
	if (!core->running) {
		spin_unlock_irqrestore(&core->lock, flags);
		dev_dbg(&br_dev->interface->dev,
			"callback ep0: UDC not running\n");
		return;
	}
	spin_unlock_irqrestore(&core->lock, flags);

	spin_lock_irqsave(&ep->lock, flags);
	if (WARN_ON(ep->halted)) {
		/* ep0 should not be halted!
		 * Recover from this state to stay operational. */
		dev_err(&br_dev->interface->dev,
			"callback ep0: recover from halted, urb %p\n", urb);
		ep->halted = 0;
	}
	if (urb->unlinked) {
		dev_dbg(&br_dev->interface->dev,
			"callback ep0: unlinked(%d) urb %p status=%d\n",
			urb->unlinked, urb, urb->status);
	} else if (usb_pipein(urb->pipe) &&
				core->ctrl_state != DATA_STAGE_OUT) {
		/* Received control URB without DATA_STAGE_OUT*/
		int value = 1;

		/* paranoia, in case of stale queued data */
		list_for_each_entry(req, &ep->queue, queue)
		{
			list_del_init(&req->queue);
			req->req.status = -EOVERFLOW;
			dev_warn(&br_dev->interface->dev,
				"callback ep0: stale req %p\n", req);

			spin_unlock(&ep->lock);
			req->req.complete(&ep->ep, &req->req);
			spin_lock(&ep->lock);
			goto exit_clear_in_progress;
		}

		if (urb->status == -EPROTO) {
			dev_warn_ratelimited(&br_dev->interface->dev,
				"callback ep0: status -EPROTO\n");
			goto queue_req_setup;
		}

		if (urb->actual_length != sizeof(struct usb_ctrlrequest)) {
			dev_warn(&br_dev->interface->dev,
				"callback ep0: Invalid setup packet len=%d, status=%d\n",
				urb->actual_length, urb->status);
			goto queue_req_setup;
		}

		memcpy(&core->ctrl_rsp, urb->transfer_buffer,
				urb->actual_length);
		if (core->ctrl_rsp.bRequestType & USB_DIR_IN)
			/*
			 * The USB 2.0 spec states that "if wLength is
			 * zero, there is no data transfer phase."
			 * However, testusb #14 seems to actually
			 * expect a data phase even if wLength = 0...
			 */
			core->ctrl_state = DATA_STAGE_IN;
		else {
			if (core->ctrl_rsp.wLength != cpu_to_le16(0))
				/* gadget driver is expected to issue
				 * a read control data request
				 */
				core->ctrl_state = DATA_STAGE_OUT;
			else
				core->ctrl_state = STATUS_STAGE_IN;
		}

		value = utbridge_handle_ctrl_req(core, &core->ctrl_rsp);
		if (value < 0) {
			dev_err(&br_dev->interface->dev,
				"callback ep0: Unhandled Setup(%d): %02x.%02x v%04x i%04x l%d %s\n",
				value, core->ctrl_rsp.bRequestType,
				core->ctrl_rsp.bRequest,
				core->ctrl_rsp.wValue,
				core->ctrl_rsp.wIndex,
				core->ctrl_rsp.wLength,
				ctrl_state_name[core->ctrl_state]);
			goto queue_req_setup;
		}
		goto exit_handler;
	} else {
		/* Transmitted URB */
		if (utbridge_urb_transfer(core, urb, ep) < 0)
			goto exit_clear_in_progress;
	}

	/* don't submit requests after pull_down */
	if (!core->active)
		goto exit_clear_in_progress;

	/* Re-submit any pending request */
	if (utbridge_handle_ep_queue(core, ep) < 0)
		goto exit_clear_in_progress;
	goto exit_handler;

queue_req_setup:
	/* Queue up a control request from B-Host */
	if (utbridge_ep0_req_setup(core) < 0)
		goto exit_clear_in_progress;

exit_handler:
	spin_unlock_irqrestore(&ep->lock, flags);
	return;

exit_clear_in_progress:
	clr_in_progress(ep, __func__);
	spin_unlock_irqrestore(&ep->lock, flags);
	return;
}

/* IRQ context */
static void utbridge_epX_urb_callback(struct urb *urb)
{
	struct utb_core *core = urb->context;
	struct usb_utbridge *br_dev = core->br_dev;
	unsigned long flags;
	struct utb_ep *ep = NULL;

	dev_dbg(&br_dev->interface->dev,
		"callback A-ep%d%s: buf=%p (xfered=%d)\n",
		usb_pipeendpoint(urb->pipe),
		usb_pipein(urb->pipe) ? "in" : "out", urb->transfer_buffer,
		urb->actual_length);

	spin_lock_irqsave(&core->lock, flags);
	if (!core->running) {
		spin_unlock_irqrestore(&core->lock, flags);
		dev_dbg(&br_dev->interface->dev,
			"callback A-ep%d%s: UDC not running\n",
			usb_pipeendpoint(urb->pipe),
			usb_pipein(urb->pipe) ? "in" : "out");
		return;
	}

	/* convert bridge address to gadget endpoint */
	ep = ep_from_a_addr(core, usb_pipeendpoint(urb->pipe) |
			(usb_pipein(urb->pipe) ? USB_DIR_IN : 0));
	spin_unlock_irqrestore(&core->lock, flags);

	if (!ep) {
		/* set_configuration() disagreement */
		dev_warn(&br_dev->interface->dev,
			"callback A-ep%d%s: ep not configured for urb %p\n",
			usb_pipeendpoint(urb->pipe),
			usb_pipein(urb->pipe) ? "in" : "out", urb);
		return;
	}

	spin_lock_irqsave(&ep->lock, flags);
	if (ep->halted) {
		/* NOTE: must not be iso! */
		dev_err(&br_dev->interface->dev,
			"callback %s: ep halted, urb %p\n", ep->ep.name, urb);
		goto exit_clear_in_progress;
	}
	if (urb->unlinked) {
		dev_dbg(&br_dev->interface->dev,
			"callback %s: unlinked(%d) urb %p status=%d\n",
			ep->ep.name, urb->unlinked, urb, urb->status);
	} else {
		/* Transmitted URB */
		if (utbridge_urb_transfer(core, urb, ep) < 0)
			goto exit_clear_in_progress;
	}

	/* don't submit requests after pull_down */
	if (!core->active)
		goto exit_clear_in_progress;

	/* Re-submit any pending request */
	if (utbridge_handle_ep_queue(core, ep) < 0)
		goto exit_clear_in_progress;

	spin_unlock_irqrestore(&ep->lock, flags);
	return;

exit_clear_in_progress:
	clr_in_progress(ep, __func__);
	spin_unlock_irqrestore(&ep->lock, flags);
	return;
}

/*
 * external triggers on starting config change via e.g. sysfs.
 * changeable: bridgeport
 */

bool pre_config_change(struct usb_utbridge *br_dev, bool force)
{
	struct utb_core *core = platform_get_drvdata(br_dev->pdev);

	/* force = true will interrupt data transfer on ep0, this is used when
	 * changing the bridgeport, as the connection will then be interrupted
	 * anyway. */
	if (force && core->active)
		usb_kill_anchored_urbs(&br_dev->eps[0].submitted);
	return core->active;
}

void post_config_change(struct usb_utbridge *br_dev, bool success)
{
	struct utb_core *core = platform_get_drvdata(br_dev->pdev);

	/* Queue up ctrl request from B-Host if bridge mode enabled */
	if (success && core->active)
		utbridge_ep0_req_setup(core);
}

/*
 * USB endpoint ops
 */

/* IRQ and enqueue-callback, called with spinlock held */
static int utbridge_handle_ep_queue(struct utb_core *core, struct utb_ep *ep)
{
	struct utb_request *req;
	int retval = 0;

	if (list_empty(&ep->queue))
		return -ENODATA;
	req = list_entry(ep->queue.next, struct utb_request, queue);

	if (ep == &core->br_dev->eps[0]) {
		core->ctrl_rsp.wLength = req->req.length;
		if (core->ctrl_state == DATA_STAGE_IN
				|| core->ctrl_state == STATUS_STAGE_IN) {

			core->ctrl_rsp.wIndex =
				((core->ctrl_rsp.bRequestType &
				USB_TYPE_MASK) << 7) |
				(core->ctrl_rsp.bRequest << 4) |
				UBS_PROCESS | UBS_RELAY;

			switch (core->ctrl_rsp.bRequest) {
			case USB_REQ_SET_CONFIGURATION:
			case USB_REQ_SET_INTERFACE:
				core->ctrl_rsp.wIndex |= UBS_STORE;
				utbridge_bhost_set_state(core,
							USB_STATE_CONFIGURED);
				break;
			}
			retval = utbridge_ep0_write(core,
					&core->ctrl_rsp, &req->req);

		} else {
			retval = utbridge_ep0_read(core,
					&core->ctrl_rsp, &req->req);
		}

	} else {
#ifdef USE_COPY_ON_EPX
		if (WARN_ON(req->req.length > 65536)) {
			clr_in_progress(ep, __func__);
			return -ENOMEM;
		}
		req->ubuf = req->req.buf;
		req->req.buf = ep->buffer;
		if (usb_endpoint_dir_out(ep->a_desc))
			memcpy(req->req.buf, req->ubuf, req->req.length);
#endif
		retval = utbridge_epX_rw(core, ep, &req->req);
	}

	return retval;
}

static int utbridge_ep_enable(struct usb_ep *_ep,
		const struct usb_endpoint_descriptor *desc)
{
	struct utb_core *core;
	struct utb_ep *ep;
	unsigned max;
	int retval;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);

	if (!_ep || !desc || ep->b_desc || usb_endpoint_num(ep->a_desc) == 0
			|| desc->bDescriptorType != USB_DT_ENDPOINT)
		return -EINVAL;

	if (!core->driver || !core->active)
		return -ESHUTDOWN;

	dev_dbg(&core->br_dev->interface->dev, "%s: enable\n", ep->ep.name);
	/*
	 * For HS/FS devices only bits 0..10 of the wMaxPacketSize represent the
	 * maximum packet size.
	 * For SS devices the wMaxPacketSize is limited by 1024.
	 */
	max = usb_endpoint_maxp(desc) & 0x7ff;

	/* drivers must not request bad settings, since lower levels
	 * (hardware or its drivers) may not check.
	 * There maybe have maxpacket limitations, etc.
	 */
	retval = -EINVAL;
	switch (usb_endpoint_type(desc)) {
	case USB_ENDPOINT_XFER_BULK:
		if (usb_endpoint_type(ep->a_desc) != USB_ENDPOINT_XFER_BULK)
			goto done;
		switch (core->gadget.speed) {
		case USB_SPEED_HIGH:
			if (max == 512)
				break;
			goto done;
		case USB_SPEED_FULL:
			if (max == 8 || max == 16 || max == 32 || max == 64)
				/* we'll fake any legal size */
				break;
			/* save a return statement */
		default:
			goto done;
		}
		break;
	case USB_ENDPOINT_XFER_INT:
		 /* bulk or int are ok */
		if (usb_endpoint_type(ep->a_desc) == USB_ENDPOINT_XFER_ISOC)
			goto done;
		/* real hardware might not handle all packet sizes */
		switch (core->gadget.speed) {
		case USB_SPEED_HIGH:
			if (max <= 1024)
				break;
			/* save a return statement */
		case USB_SPEED_FULL:
			if (max <= 64)
				break;
			/* save a return statement */
		default:
			if (max <= 8)
				break;
			goto done;
		}
		break;
	case USB_ENDPOINT_XFER_ISOC:
		if (usb_endpoint_type(ep->a_desc) != USB_ENDPOINT_XFER_ISOC)
			goto done;
		/* real hardware might not handle all packet sizes */
		switch (core->gadget.speed) {
		case USB_SPEED_HIGH:
			if (max <= 1024)
				break;
			/* save a return statement */
		case USB_SPEED_FULL:
			if (max <= 1023)
				break;
			/* save a return statement */
		default:
			goto done;
		}
		break;
	default:
		/* few chips support control except on ep0 */
		goto done;
	}

	_ep->maxpacket = max;
	ep->b_desc = desc;

	dev_dbg(&core->br_dev->interface->dev, "%s: enabled (maxpacket=%d)\n",
		ep->ep.name, max);

	/* at this point real hardware should be NAKing transfers
	 * to that endpoint, until a buffer is queued to it.
	 */
	ep->halted = ep->wedged = ep->inprogress = 0;
	smp_wmb();	/* inprogress */
	retval = 0;
done:
	return retval;
}

static int utbridge_ep_disable(struct usb_ep *_ep)
{
	struct utb_ep *ep;
	struct utb_core *core;
	unsigned long flags;
	int retval;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);

	if (!_ep || !ep->b_desc || usb_endpoint_num(ep->a_desc) == 0)
		return -EINVAL;

	local_irq_save(flags);
	spin_lock(&ep->lock);
	ep->b_desc = NULL;
	retval = 0;
	spin_unlock(&ep->lock);

	/* Stop any pending URB's */
	usb_unlink_anchored_urbs(&ep->submitted);

	/*return ownership to gadget driver*/
	spin_lock(&ep->lock);

	while (!list_empty(&ep->queue)) {
		struct utb_request *req;

		req = list_entry(ep->queue.next, struct utb_request, queue);
		list_del_init(&req->queue);
		req->req.status = -ESHUTDOWN;

#ifdef USE_COPY_ON_EPX
		if (req->ubuf) {
			req->req.buf = req->ubuf;
			req->ubuf = NULL;
		}
#endif
		spin_unlock(&ep->lock);
		req->req.complete(&ep->ep, &req->req);
		spin_lock(&ep->lock);
	}

	spin_unlock(&ep->lock);
	local_irq_restore(flags);

	dev_dbg(&core->br_dev->interface->dev, "%s: disabled\n", _ep->name);
	return retval;
}

static struct usb_request *utbridge_ep_alloc_request(struct usb_ep *_ep,
		gfp_t mem_flags)
{
	struct utb_ep *ep;
	struct utb_core *core;
	struct utb_request *req;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);

	if (!_ep)
		return NULL;

	req = kzalloc(sizeof(*req), mem_flags);
	if (!req)
		return NULL;
	INIT_LIST_HEAD(&req->queue);

	dev_dbg(&core->br_dev->interface->dev, "%s: alloc_request %p/%p\n",
			_ep->name, &req->req, req);
	return &req->req;
}

static void utbridge_ep_free_request(struct usb_ep *_ep,
		struct usb_request *_req)
{
	struct utb_ep *ep;
	struct utb_core *core;
	struct utb_request *req;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);
	req = usb_request_to_utbridge_request(_req);

	if (WARN_ON(!ep || !_req))
		return;

	dev_dbg(&core->br_dev->interface->dev,
		"%s: free_request %p/%p (buf %p)\n", _ep->name, _req, req,
		_req->buf);

	WARN_ON(!list_empty(&req->queue));
	kfree(req);
}

static int utbridge_ep_queue(struct usb_ep *_ep, struct usb_request *_req,
		gfp_t mem_flags)
{
	unsigned long flags;
	struct utb_ep *ep;
	struct utb_core *core;
	struct utb_request *req;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);
	req = usb_request_to_utbridge_request(_req);

	if (!_req || !list_empty(&req->queue) || !_req->complete)
		return -EINVAL;

	if (!_ep || (!ep->b_desc && usb_endpoint_num(ep->a_desc) != 0))
		return -EINVAL;

	if (!core->driver || !core->active)
		return -ESHUTDOWN;

	dev_dbg(&core->br_dev->interface->dev,
		"%s: queue req %p (len=%d) buf %p\n", _ep->name, _req,
		_req->length, _req->buf);

	_req->status = -EINPROGRESS;
	_req->actual = 0;

	spin_lock_irqsave(&ep->lock, flags);
	list_add_tail(&req->queue, &ep->queue);

	/* (re)start queue handling if not already running */
	if (!enter_in_progress(ep, __func__)) {
		if (utbridge_handle_ep_queue(core, ep) < 0)
			clr_in_progress(ep, __func__);
	}
	spin_unlock_irqrestore(&ep->lock, flags);
	return 0;
}

static int utbridge_ep_dequeue(struct usb_ep *_ep, struct usb_request *_req)
{
	struct utb_ep *ep;
	struct utb_core *core;
	int retval = -EINVAL;
	unsigned long flags;
	struct utb_request *req = NULL;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);

	if (!_ep || !_req)
		return retval;

	if (!core->driver)
		return -ESHUTDOWN;

	local_irq_save(flags);
	spin_lock(&ep->lock);
	list_for_each_entry(req, &ep->queue, queue) {
		if (&req->req == _req) {
			list_del_init(&req->queue);
			_req->status = -ECONNRESET;
			retval = 0;
			break;
		}
	}

#ifdef USE_COPY_ON_EPX
	if ((retval == 0) && (req->ubuf)) {
		usb_unlink_urb(ep->urb);
		req->req.buf = req->ubuf;
		req->ubuf = NULL;
	}
#endif

	spin_unlock(&ep->lock);

	if (retval == 0) {
		dev_dbg(&core->br_dev->interface->dev,
			"%s: dequeued req %p (len=%d) buf %p\n", _ep->name,
			_req, _req->length, _req->buf);
		_req->complete(_ep, _req);
	}
	local_irq_restore(flags);
	return retval;
}

static int ep_set_halt_and_wedge(struct usb_ep *_ep, int value, int wedged)
{
	struct utb_ep *ep;
	struct utb_core *core;

	ep = usb_ep_to_utbridge_ep(_ep);
	core = gadget_to_utbridge_core(ep->gadget);

	if (!_ep || usb_endpoint_num(ep->a_desc) == 0)
		return -EINVAL;
	if (!core->driver)
		return -ESHUTDOWN;
	if (!value)
		ep->halted = ep->wedged = 0;
	else if (ep->b_desc && (ep->b_desc->bEndpointAddress & USB_DIR_IN) &&
			!list_empty(&ep->queue))
		return -EAGAIN;
	else {
		ep->halted = 1;
		if (wedged)
			ep->wedged = 1;
	}
	dev_warn(&core->br_dev->interface->dev, "%s: %s%s\n", _ep->name,
		(value) ? "halted" : "halt and wedge cleared",
		(wedged) ? " and wedged" : "");
	return 0;
}

static int utbridge_ep_set_halt(struct usb_ep *_ep, int value)
{
	return ep_set_halt_and_wedge(_ep, value, 0);
}

static int utbridge_ep_set_wedge(struct usb_ep *_ep)
{
	return ep_set_halt_and_wedge(_ep, 1, 1);
}

static const struct usb_ep_ops utbridge_ep_ops = {
	.enable    = utbridge_ep_enable,
	.disable   = utbridge_ep_disable,
	.alloc_request = utbridge_ep_alloc_request,
	.free_request  = utbridge_ep_free_request,
	.queue     = utbridge_ep_queue,
	.dequeue   = utbridge_ep_dequeue,
	.set_halt  = utbridge_ep_set_halt,
	.set_wedge = utbridge_ep_set_wedge,
};

/*
 * Gadget ops
 */

static int utbridge_udc_get_frame(struct usb_gadget *gadget)
{
	struct timeval tv;

	do_gettimeofday(&tv);
	return tv.tv_usec / 1000;
}

static int utbridge_udc_wakeup(struct usb_gadget *gadget)
{
	struct utb_core *core = gadget_to_utbridge_core(gadget);

	if (!(core->devstatus & ((1 << USB_DEVICE_REMOTE_WAKEUP))))
		return -EINVAL;

	return 0;
}

static int utbridge_udc_set_selfpowered(struct usb_gadget *gadget, int value)
{
	unsigned long flags;
	struct utb_core *core = gadget_to_utbridge_core(gadget);

	spin_lock_irqsave(&core->lock, flags);

	if (value)
		core->devstatus |= (1 << USB_DEVICE_SELF_POWERED);
	else
		core->devstatus &= ~(1 << USB_DEVICE_SELF_POWERED);

	spin_unlock_irqrestore(&core->lock, flags);
	return 0;
}

static int utbridge_udc_pullup(struct usb_gadget *gadget, int value)
{
	unsigned long flags;
	int i;
	struct utb_ep *ep;
	struct utb_core *core = gadget_to_utbridge_core(gadget);

	ep = &core->br_dev->eps[0];

	mutex_lock(&core->br_dev->io_mutex);
	spin_lock_irqsave(&core->lock, flags);
	if (value && core->driver) {
		/* reset epX streams and ep0 maxpacket */
		for (i = 1; i < core->br_dev->max_eps; i++)
			core->br_dev->eps[i].ep.max_streams = 0;
		ep->ep.maxpacket = USB_MAX_CTRL_PAYLOAD;

		dev_dbg(&core->br_dev->interface->dev,
			"speed info -> gadget(%s) and driver(%s)\n",
			usb_speed_string(core->gadget.speed),
			usb_speed_string(core->driver->max_speed));
		core->active = (value != 0);
		spin_unlock_irqrestore(&core->lock, flags);
		mutex_unlock(&core->br_dev->io_mutex);

		spin_lock_irqsave(&ep->lock, flags);
		if (!enter_in_progress(ep, __func__)) {
			/* Queue up a control request from B-Host */
			if (utbridge_ep0_req_setup(core) < 0)
				clr_in_progress(ep, __func__);
		}
		spin_unlock_irqrestore(&ep->lock, flags);
	} else {
		/* Stop any pending Ctrl requests */
		usb_unlink_anchored_urbs(&ep->submitted);

		core->active = (value != 0);
		spin_unlock_irqrestore(&core->lock, flags);
		mutex_unlock(&core->br_dev->io_mutex);

		/*return ownership to gadget driver*/
		local_irq_save(flags);
		spin_lock(&ep->lock);

		while (!list_empty(&ep->queue)) {
			struct utb_request *req;

			req = list_entry(ep->queue.next, struct utb_request,
									queue);
			list_del_init(&req->queue);
			req->req.status = -ESHUTDOWN;

#ifdef USE_COPY_ON_EPX
			if (req->ubuf) {
				req->req.buf = req->ubuf;
				req->ubuf = NULL;
			}
#endif
			spin_unlock(&ep->lock);
			req->req.complete(&ep->ep, &req->req);
			spin_lock(&ep->lock);
		}

		spin_unlock(&ep->lock);
		local_irq_restore(flags);
	}

	dev_dbg(&core->br_dev->interface->dev, "udc pulled %s\n",
		(value) ? "up" : "down");
	return 0;
}

static int utbridge_udc_start(struct usb_gadget *gadget,
		struct usb_gadget_driver *driver)
{
	struct utb_core *core = gadget_to_utbridge_core(gadget);

	if (driver->max_speed == USB_SPEED_UNKNOWN)
		return -EINVAL;

	core->devstatus = 0;

	core->driver = driver;
	core->gadget.dev.driver = &driver->driver;
	dev_dbg(&core->br_dev->interface->dev, "register gadget driver '%s'\n",
		driver->driver.name);
	return 0;
}

static int utbridge_udc_stop(struct usb_gadget *gadget,
		struct usb_gadget_driver *driver)
{
	struct utb_core *core = gadget_to_utbridge_core(gadget);

	core->gadget.dev.driver = NULL;
	core->driver = NULL;

	return 0;
}

static const struct usb_gadget_ops utbridge_gadget_ops = {
	.get_frame = utbridge_udc_get_frame,
	.wakeup    = utbridge_udc_wakeup,
	.set_selfpowered = utbridge_udc_set_selfpowered,
	.pullup    = utbridge_udc_pullup,
	.udc_start = utbridge_udc_start,
	.udc_stop  = utbridge_udc_stop,
};

/*
 * UDC driver
 */

/* gadget structure is part of the devm_* allocated UDC private data */
static void utbridge_gadget_release(struct device *dev)
{
}

static void utbridge_udc_init_eps(struct utb_core *core)
{
	int i, j;
	struct utb_info *info = core->br_dev->info;

	INIT_LIST_HEAD(&core->gadget.ep_list);
	for (i = 0; i < core->br_dev->max_eps; i++) {
		struct utb_ep *ep = &core->br_dev->eps[i];
		ep->ep.ops = &utbridge_ep_ops;
		ep->halted = ep->wedged = ep->inprogress = 0;
		smp_wmb();	/* inprogress */
		spin_lock_init(&ep->lock);
		ep->gadget = &core->gadget;
		INIT_LIST_HEAD(&ep->queue);
		if (i == 0) {
			/* special treatment for ep0 */
			ep->b_desc = &utbridge_ep0_desc;
			ep->ep.maxpacket = USB_MAX_CTRL_PAYLOAD;
			core->gadget.ep0 = &ep->ep;
			ep->ep.name = info->ep_info[0].epname_b;
		 } else {
			ep->b_desc = NULL;
			/* maxpacket is set by epautoconfig() called by
			 * gadget layer */
			ep->ep.maxpacket = ~0;
			ep->ep.maxpacket_limit = ep->size;
			for (j = 1; j < info->ep_info_count; j++)
				if (ep->a_desc->bEndpointAddress ==
						info->ep_info[j].ep_a)
					ep->ep.name = info->ep_info[j].epname_b;
			list_add_tail(&ep->ep.ep_list, &core->gadget.ep_list);
		}
		dev_dbg(&core->br_dev->interface->dev, "ep_list[%d] = %s\n", i,
			ep->ep.name);
	}
	return;
}

static int utbridge_udc_probe(struct platform_device *pdev)
{
	int rc, i;
	struct utb_core *core;
	struct usb_utbridge *br_dev = dev_get_drvdata(pdev->dev.parent);

	core = devm_kzalloc(&br_dev->interface->dev, sizeof(struct utb_core),
								GFP_KERNEL);
	if (!core) {
		dev_err(&br_dev->interface->dev,
			"utbridge_udc alloc failure\n");
		return -ENOMEM;
	}

	spin_lock_init(&core->lock);

	core->bhoststate = USB_STATE_NOTATTACHED;

	/* Setup gadget structure */
	core->gadget.name = gadget_name;
	core->gadget.ops = &utbridge_gadget_ops;
	core->gadget.speed = br_dev->udev->speed;
	core->gadget.max_speed = USB_SPEED_HIGH;

	/* Setup gadget.dev and register with kernel */
	dev_set_name(&core->gadget.dev, "gadget");
	core->gadget.dev.parent = &pdev->dev;
	core->gadget.dev.release = utbridge_gadget_release;
	rc = device_register(&core->gadget.dev);
	if (rc < 0) {
		dev_err(&br_dev->interface->dev,
			"Error registering gadget device\n");
		goto err_reg_dev;
	}
	core->br_dev = br_dev;
	utbridge_udc_init_eps(core);

	rc = usb_add_gadget_udc(&pdev->dev, &core->gadget);
	if (rc < 0)
		goto err_udc;

	/* Create device attributes */
	for (i = 0; i < NUM_OF_ATTRS; i++) {
		rc = device_create_file(&core->gadget.dev,
					utbridge_udc_attr[i]);
		if (rc)
			goto err_dev_attr;
	}
	utbridge_init_debugfs(core);

	core->running = 1;

	platform_set_drvdata(pdev, core);

	return 0;

err_dev_attr:
	while (--i >= 0)
		device_remove_file(&core->gadget.dev, utbridge_udc_attr[i]);
	usb_del_gadget_udc(&core->gadget);
err_udc:
	device_unregister(&core->gadget.dev);
err_reg_dev:
	put_device(&core->gadget.dev);
	return rc;
}

static int utbridge_udc_remove(struct platform_device *pdev)
{
	struct utb_core *core = platform_get_drvdata(pdev);
	int i;
	unsigned long flags;

	spin_lock_irqsave(&core->lock, flags);
	core->running = 0;
	spin_unlock_irqrestore(&core->lock, flags);

	utbridge_cleanup_debugfs(core);
	for (i = 0; i < NUM_OF_ATTRS; i++)
		device_remove_file(&core->gadget.dev, utbridge_udc_attr[i]);

	if (core->driver)
		usb_gadget_unregister_driver(core->driver);
	usb_del_gadget_udc(&core->gadget);
	platform_set_drvdata(pdev, NULL);
	device_unregister(&core->gadget.dev);

	return 0;
}

static int utbridge_udc_suspend(struct platform_device *pdev,
		pm_message_t state)
{
	struct utb_core *core = platform_get_drvdata(pdev);

	if (core->active) {
		utbridge_bhost_set_state(core, USB_STATE_NOTATTACHED);
		utbridge_udc_pullup(&core->gadget, 0);
	}
	return 0;
}

static int utbridge_udc_resume(struct platform_device *pdev)
{
	return 0;
}

struct platform_driver utbridge_udc_driver = {
	.probe    = utbridge_udc_probe,
	.remove   = utbridge_udc_remove,
	.suspend  = utbridge_udc_suspend,
	.resume   = utbridge_udc_resume,
	.driver = {
	   .name  = gadget_name,
	   .owner = THIS_MODULE,
	},
};
