/*
 * linux/drivers/usb/gadget/utbridge_dev.c
 *
 * USB Driver for the Dual Role Unwired Technology USB Bridge/Hub
 *   USB device part
 *
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Martin Herzog (mherzog@de.adit-jv.com)
 *        and Andreas Pape (apape@de.adit-jv.com)
 *
 * Based on a reference driver by Sam Yeda (sam@unwiredtechnology.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/idr.h>
#include <linux/platform_device.h>
#include <linux/usb.h>
#include <linux/usb/gadget.h>

#include "utbridge.h"

#define USB_UNWIRED_VENDOR_ID	0x2996
#define USB_HIGHLINE_PRODUCT_ID	0x0100	/* production */
#define USB_ASIC_101_PRODUCT_ID	0x0101	/* prototype  */
#define USB_ASIC_102_PRODUCT_ID	0x0102	/* production */

#define API_FPGA		BIT(0)
#define API_ASIC		BIT(1)

#define UTB_RD_REQ		VENDOR_SPECIFIC_REQ
#define UTB_WR_REQ		(VENDOR_SPECIFIC_REQ | UBS_PROCESS | UBS_STORE)

#define VEND_REQ_SHIFT		4

#define WVALUE_DN1_HUB		0
#define WVALUE_DN1_BRIDGE	1
#define WVALUE_DN2_HUB		1
#define WVALUE_DN2_BRIDGE	0

enum utb_vend_req {
	UTB_I_FIRMWARE_VERSION = 0,
	UTB_I_THERMAL_GOOD,
	UTB_O_MUX_SELECT_DN1,
	UTB_O_MUX_SELECT_DN2,
	UTB_O_MUX_BRIDGE_DN_SEL,
	UTB_I_DN1_POWER_STATUS,
	UTB_O_DN1_POWER_ENABLE,
	UTB_I_DN2_POWER_STATUS,
	UTB_O_DN2_POWER_ENABLE,
	UTB_I_DN1_OVER_CURRENT,
	UTB_I_DN2_OVER_CURRENT,
	UTB_O_DIMMING_BRIDGE,
	UTB_O_BRIDGE_OVERCURRENT,
	UTB_I_POWER_GOOD,
	UTB_I_HW_IMAGE_VERSION,
	UTB_O_FLUSH_EP_FIFO,
	UTB_I_BHOST_STATUS,
	UTB_O_SELECT_BRIDGE_PORT,
	UTB_O_PORT_POWER,
	UTB_MAX_VEND_REQ
};

enum utb_direction {
	CTRL_WRITE = 0,
	CTRL_READ
};

struct utb_vend_req_tmpl {
	enum utb_vend_req vend_req;	/* vendor request */
	enum utb_direction direction;	/* read or write request */
	u16 size;		/* usb_ctrlrequest->wLength */
	u8 api_support;		/* Hardware versions supporting this request */
};

static const struct utb_vend_req_tmpl utb_vend_req_table[] = {
	/* vend_req, direction, size, api_support */
	{ UTB_I_FIRMWARE_VERSION,   CTRL_READ,  3, API_FPGA | API_ASIC, },
	{ UTB_I_THERMAL_GOOD,       CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_O_MUX_SELECT_DN1,     CTRL_WRITE, 0, API_FPGA,            },
	{ UTB_O_MUX_SELECT_DN2,     CTRL_WRITE, 0, API_FPGA,            },
	{ UTB_O_MUX_BRIDGE_DN_SEL,  CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_I_DN1_POWER_STATUS,   CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_O_DN1_POWER_ENABLE,   CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_I_DN2_POWER_STATUS,   CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_O_DN2_POWER_ENABLE,   CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_I_DN1_OVER_CURRENT,   CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_I_DN2_OVER_CURRENT,   CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_O_DIMMING_BRIDGE,     CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_O_BRIDGE_OVERCURRENT, CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_I_POWER_GOOD,         CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_I_HW_IMAGE_VERSION,   CTRL_READ,  2, API_FPGA | API_ASIC, },
	{ UTB_O_FLUSH_EP_FIFO,      CTRL_WRITE, 0, API_FPGA | API_ASIC, },
	{ UTB_I_BHOST_STATUS,       CTRL_READ,  1, API_FPGA | API_ASIC, },
	{ UTB_O_SELECT_BRIDGE_PORT, CTRL_WRITE, 0,            API_ASIC, },
	{ UTB_O_PORT_POWER,         CTRL_WRITE, 0,            API_ASIC, },
};
static const struct utb_info highline_info = {
	.ep_info = {
		{ 0x00, 0x00, "ep0", },
		{ 0x81, 0x02, "ep2out-bulk", },
		{ 0x02, 0x81, "ep1in-bulk", },
		{ 0x83, 0x04, "ep4out-bulk", },
		{ 0x04, 0x83, "ep3in-bulk", },
		{ 0x06, 0x85, "ep5in-int", },
		{ 0x87, 0x08, "ep8out-iso", },
	},
	.ep_info_count = 7,
	.port_num_offset = 0,
	.max_bridge_port = 2,
	.api_version = API_FPGA,
};
static const struct utb_info asic_101_info = {
	.ep_info = {
		{ 0x00, 0x00, "ep0", },
		{ 0x81, 0x02, "ep2out-bulk", },
		{ 0x02, 0x81, "ep1in-bulk", },
		{ 0x83, 0x04, "ep4out-bulk", },
		{ 0x04, 0x83, "ep3in-bulk", },
		{ 0x06, 0x85, "ep5in-int", },
		{ 0x87, 0x06, "ep6out-iso", },
	},
	.ep_info_count = 7,
	.port_num_offset = 1,
	.max_bridge_port = 4,
	.api_version = API_ASIC,
};
static const struct utb_info asic_102_info = {
	.ep_info = {
		{ 0x00, 0x00, "ep0", },
		{ 0x81, 0x02, "ep2out-bulk", },
		{ 0x02, 0x81, "ep1in-bulk", },
		{ 0x83, 0x04, "ep4out-bulk", },
		{ 0x04, 0x83, "ep3in-bulk", },
		{ 0x06, 0x85, "ep5in-int", },
		{ 0x87, 0x08, "ep8out-iso", },
	},
	.ep_info_count = 7,
	.port_num_offset = 1,
	.max_bridge_port = 4,
	.api_version = API_ASIC,
};

#define PORT_VALID(p, brdev) ((p) > (brdev)->info->port_num_offset && \
				(p) <= (brdev)->info->max_bridge_port)

/*
 * module parameters
 */
static int bridgeport; /* default 0 */
module_param(bridgeport, int, S_IRUGO);
MODULE_PARM_DESC(bridgeport,
	"Port to configure for bridge mode (default 0):0=Off 1..3=Port");

/* table of devices that work with this driver */
static const struct usb_device_id utbridge_table[] = {
	{ USB_DEVICE(USB_UNWIRED_VENDOR_ID, USB_HIGHLINE_PRODUCT_ID),
		.driver_info = (unsigned long)&highline_info, },
	{ USB_DEVICE(USB_UNWIRED_VENDOR_ID, USB_ASIC_101_PRODUCT_ID),
		.driver_info = (unsigned long)&asic_101_info, },
	{ USB_DEVICE(USB_UNWIRED_VENDOR_ID, USB_ASIC_102_PRODUCT_ID),
		.driver_info = (unsigned long)&asic_102_info, },
	{ USB_DEVICE(0x1772, 0x0002),
		.driver_info = (unsigned long)&highline_info, },
	{ } /* Terminating entry */
};
MODULE_DEVICE_TABLE(usb, utbridge_table);

/* container macros */
#define kref_to_utbridge(d)        container_of(d, struct usb_utbridge, kref)

/**
 * get vendor request table index
 *
 * return req_idx - if vendor request is known and supported by device
 *        -EPROTO - if vendor request is not supported by device's API version
 *        -ENOSYS - if vendor request is completely unknown
 */
static int utbridge_get_vend_req_index(struct usb_utbridge *br_dev,
		enum utb_vend_req req)
{
	int req_idx;

	/* check if this vendor request is vaild */
	for (req_idx = 0; req_idx < ARRAY_SIZE(utb_vend_req_table); req_idx++)
		if (utb_vend_req_table[req_idx].vend_req == req)
			break;
	if (req_idx == ARRAY_SIZE(utb_vend_req_table))
		return -ENOSYS;

	if (!(utb_vend_req_table[req].api_support & br_dev->info->api_version))
		return -EPROTO;
	else
		return req_idx;
}

/**
 * If successful, it returns the number of bytes transferred, otherwise a
 * negative error number.
 */
static int utbridge_send_vend_req(struct usb_utbridge *br_dev,
		enum utb_vend_req req, __u16 value, void *data, int dsize)
{
	unsigned int pipe;
	__u8 request;
	__u8 requesttype;
	__u16 index;
	__u16 size;
	int req_idx;

	/* return if vendor request is not supported by device */
	req_idx = utbridge_get_vend_req_index(br_dev, req);
	if (req_idx < 0)
		return req_idx;

	if ((dsize < utb_vend_req_table[req_idx].size) ||
		(!data && utb_vend_req_table[req_idx].size))
		return -EINVAL;

	if (utb_vend_req_table[req_idx].direction == CTRL_READ) {
		pipe = usb_rcvctrlpipe(br_dev->udev, 0);
		request = VEND_RD_BREQ;
		requesttype = VEND_RD_BMREQTYPE;
		index = (UTB_RD_REQ | (req << VEND_REQ_SHIFT));
		size = utb_vend_req_table[req_idx].size;
	} else {
		pipe = usb_sndctrlpipe(br_dev->udev, 0);
		request = VEND_WR_BREQ;
		requesttype = VEND_WR_BMREQTYPE;
		index = (UTB_WR_REQ | (req << VEND_REQ_SHIFT));
		size = utb_vend_req_table[req_idx].size;
	}
	dev_dbg(&br_dev->interface->dev,
		"usb_control_msg(request=%#04x, requesttype=%#04x, value=%#06x, index=%#06x\n",
		request, requesttype, value, index);
	return usb_control_msg(br_dev->udev, pipe, request, requesttype, value,
				index, data, size, CTRL_MSG_TIMEOUT);
}

static void utbridge_print_version(struct usb_utbridge *br_dev)
{
	u8 hw_ver[utb_vend_req_table[UTB_I_HW_IMAGE_VERSION].size];
	u8 fw_ver[utb_vend_req_table[UTB_I_FIRMWARE_VERSION].size];
	int rv;

	rv = utbridge_send_vend_req(br_dev, UTB_I_HW_IMAGE_VERSION, 0, &hw_ver,
		sizeof(hw_ver));
	if (rv < 0) {
		dev_err(&br_dev->interface->dev,
			"Unable to read hardware revision\n");
		return;
	}

	rv = utbridge_send_vend_req(br_dev, UTB_I_FIRMWARE_VERSION, 0, &fw_ver,
		sizeof(fw_ver));
	if (rv < 0) {
		dev_err(&br_dev->interface->dev,
			"Unable to read firmware revision\n");
		return;
	}

	dev_info(&br_dev->interface->dev,
		"P(0x%04x) Driver(%s) Firmware(%hhd.%hhd.%hhd) Hardware(%hhd.%hhd) %s\n",
		br_dev->udev->descriptor.idProduct, DRIVER_VERSION,
		fw_ver[2], fw_ver[1], fw_ver[0], hw_ver[1], hw_ver[0],
		usb_speed_string(br_dev->udev->speed));
}

static int utbridge_epflush_fifo(struct usb_utbridge *br_dev, u16 ep_bits)
{
	int rv;

	rv = utbridge_send_vend_req(br_dev, UTB_O_FLUSH_EP_FIFO, ep_bits, NULL,
		0);
	if (rv < 0)
		dev_err(&br_dev->interface->dev,
			"Unable to flush ep's 0x%02x fifo %d\n", ep_bits, rv);
	return rv;
}

/**
 * Set Downstream Port Power.
 * Ensure there are no control messages in queue prior to calling this function
 */
static int utbridge_port_power(struct usb_utbridge *br_dev, u8 port, bool power)
{
	int rv;

	if (!PORT_VALID(port, br_dev))
		return -ERANGE;

	rv = utbridge_send_vend_req(br_dev, UTB_O_PORT_POWER,
		(((u16)power << 8) |
		 (port - br_dev->info->port_num_offset)), NULL, 0);
	if (rv < 0)
		dev_err(&br_dev->interface->dev,
			"Failed to set port %hhu power to %s\n", port,
			power ? "On" : "Off");
	br_dev->portpower[port] = power;
	return rv;
}

static int utbridge_port_select_fpga(struct usb_utbridge *br_dev, int port)
{
	unsigned dn1, dn2;
	int rv;

	/* Using the legacy vendor request */
	switch (port) {
	case 0: /* Bridge mode disabled */
		dn1 = WVALUE_DN1_HUB;
		dn2 = WVALUE_DN2_HUB;
		break;
	case 1:
		dn1 = WVALUE_DN1_BRIDGE;
		dn2 = WVALUE_DN2_HUB;
		break;
	case 2:
		dn1 = WVALUE_DN1_HUB;
		dn2 = WVALUE_DN2_BRIDGE;
		break;
	default:
		return -EINVAL;
		break;
	}

	if (port) {
		rv = utbridge_send_vend_req(br_dev, UTB_O_MUX_BRIDGE_DN_SEL,
						(port - 1), NULL, 0);
		if (rv < 0)
			return rv;
	} else if (br_dev->connectedport != 0) {
		/* If the bridge was previously connected to a port,
		 * switch to another port to simulate a disconnect.
		 *
		 * If a previous port_select failed, br_dev->connectedport
		 * is < 0 and the state of the mux is unknown. So always do two
		 * switches here, if br_dev->connectedport is != 0.
		 */
		rv = utbridge_send_vend_req(br_dev, UTB_O_MUX_BRIDGE_DN_SEL, 0,
						NULL, 0);
		if (rv < 0)
			return rv;
		rv = utbridge_send_vend_req(br_dev, UTB_O_MUX_BRIDGE_DN_SEL, 1,
						NULL, 0);
		if (rv < 0)
			return rv;
	}

	msleep(20);	/* wait for device to settle */

	/* Setup downstream port mux's*/
	rv = utbridge_send_vend_req(br_dev, UTB_O_MUX_SELECT_DN1, dn1, NULL, 0);
	if (rv < 0)
		return rv;
	rv = utbridge_send_vend_req(br_dev, UTB_O_MUX_SELECT_DN2, dn2, NULL, 0);
	if (rv < 0)
		return rv;
	return port;
}

static int utbridge_port_select_asic(struct usb_utbridge *br_dev, int port)
{
	int portparam, rv;

	/* Using the new vendor request */
	if (port == 0)
		portparam = 0;
	else if (PORT_VALID(port, br_dev))
		portparam = port - br_dev->info->port_num_offset;
	else
		return -ERANGE;

	rv = utbridge_send_vend_req(br_dev, UTB_O_SELECT_BRIDGE_PORT,
			portparam, NULL, 0);

	if (rv < 0)
		return rv;
	return port;
}

/**
 * Set a downstream port in Bridge mode.
 * Ensure there are no control messages in queue prior to calling this function.
 *
 * If successful returns the port Id, -1 on failure.
 */
static int utbridge_port_select(struct usb_utbridge *br_dev, int port)
{
	int rv;

	/* Flush all endpoint fifo */
	rv = utbridge_epflush_fifo(br_dev, 0xff);
	if (rv < 0)
		return rv;

	/* Vendor request SELECT_BRIDGE_PORT supported? */
	if (utbridge_get_vend_req_index(br_dev, UTB_O_SELECT_BRIDGE_PORT) < 0)
		rv = utbridge_port_select_fpga(br_dev, port);
	else
		rv = utbridge_port_select_asic(br_dev, port);

	if (rv < 0)
		dev_err(&br_dev->interface->dev,
			"Selecting DN%d into bridge mode failed\n", port);
	else if (rv > 0)
		dev_info(&br_dev->interface->dev,
			"Downstream port %d in bridge mode\n", port);
	else
		dev_info(&br_dev->interface->dev,
			"All downstream ports in host mode\n");

	br_dev->connectedport = rv;
	return rv;
}

static ssize_t show_bridgeport(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct usb_utbridge *br_dev = dev_get_drvdata(dev);
	int retval;

	mutex_lock(&br_dev->io_mutex);
	retval = snprintf(buf, PAGE_SIZE, "%d\n", br_dev->connectedport);
	mutex_unlock(&br_dev->io_mutex);
	return retval;
}
static ssize_t store_bridgeport(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct usb_utbridge *br_dev = dev_get_drvdata(dev);
	u8 port;
	int retval;

	if (sscanf(buf, "%hhu", &port) < 1)
		return -EINVAL;

	mutex_lock(&br_dev->io_mutex);
	if ((port != 0) && (!PORT_VALID(port, br_dev))) {
		dev_err(&br_dev->interface->dev,
			"Port %d out of range, must be between %d-%d or 0\n",
			port, (br_dev->info->port_num_offset + 1),
			br_dev->info->max_bridge_port);
		retval = -ERANGE;
		goto unlock_return;
	}
	pre_config_change(br_dev, 1);

	retval = utbridge_port_select(br_dev, port);
	if (retval < 0) {
		post_config_change(br_dev, 0);
		goto unlock_return;
	}

	post_config_change(br_dev, port != 0);
	retval = count;

unlock_return:
	mutex_unlock(&br_dev->io_mutex);
	return retval;
}
static DEVICE_ATTR(bridgeport, S_IRUGO|S_IWUSR|S_IWGRP, show_bridgeport,
		store_bridgeport);

/*
 * driver ops
 */

static void utbridge_draw_down(struct usb_utbridge *br_dev)
{
	int time, i;

	for (i = 0; i < br_dev->max_eps; ++i) {
		time = usb_wait_anchor_empty_timeout(&br_dev->eps[i].submitted,
									1000);
		if (!time)
			usb_kill_anchored_urbs(&br_dev->eps[i].submitted);
	}
}

void utbridge_delete(struct kref *kref)
{
	int i;
	struct usb_utbridge *br_dev = kref_to_utbridge(kref);

	for (i = 0; i < br_dev->max_eps; ++i)
		usb_free_urb(br_dev->eps[i].urb);
	/*decrements the reference count of the usb device*/
	usb_put_dev(br_dev->udev);
}

static char *ep_name_from_desc(const struct usb_endpoint_descriptor *desc,
		char *name)
{
	char *type;

	if (usb_endpoint_num(desc) == 0) {
		strcpy(name, "ep0");
		return name;
	}

	switch (usb_endpoint_type(desc)) {
	case USB_ENDPOINT_XFER_BULK:
		type = "bulk";
		break;
	case USB_ENDPOINT_XFER_ISOC:
		type = "iso";
		break;
	case USB_ENDPOINT_XFER_INT:
		type = "int";
		break;
	default:
		type = "ctrl";
		break;
	};

	sprintf(name, "ep%d%s-%s", usb_endpoint_num(desc),
		usb_endpoint_dir_in(desc) ? "in" : "out", type);

	return name;
}

static DEFINE_IDA(ut_ida);

static int utbridge_probe(struct usb_interface *interface,
					const struct usb_device_id *usbid)
{
	struct usb_utbridge *br_dev;
	struct usb_host_interface *iface_desc;
	int i;
	int id, retval = -ENOMEM;
	char epname[14];

	if (!usbid->driver_info) {
		dev_err(&interface->dev, "No driver_info\n");
		return -ENODEV;
	}

	/* allocate memory for our device state and initialize it */
	br_dev = devm_kzalloc(&interface->dev, sizeof(*br_dev), GFP_KERNEL);
	if (!br_dev) {
		dev_err(&interface->dev, "Out of memory\n");
		goto error;
	}
	kref_init(&br_dev->kref);
	mutex_init(&br_dev->io_mutex);

	/*increments the reference count of the usb device*/
	br_dev->udev = usb_get_dev(interface_to_usbdev(interface));
	br_dev->interface = interface;
	br_dev->info = (struct utb_info *)usbid->driver_info;

	/* set up the endpoint information */
	iface_desc = interface->cur_altsetting;
	br_dev->max_eps = iface_desc->desc.bNumEndpoints + 1;
	br_dev->eps = devm_kzalloc(&interface->dev,
			sizeof(struct utb_ep) * br_dev->max_eps, GFP_KERNEL);
	if (!br_dev->eps) {
		dev_err(&br_dev->interface->dev, "malloc utb_ep failed\n");
		goto error;
	}

	for (i = 0; i < br_dev->max_eps; ++i) {
		if (i == 0)
			/* use fake desc to unify behaviour */
			br_dev->eps[i].a_desc = &utbridge_ep0_desc;
		else
			br_dev->eps[i].a_desc = &iface_desc->endpoint[i-1].desc;
		br_dev->eps[i].br_dev = br_dev;

		init_usb_anchor(&br_dev->eps[i].submitted);
		br_dev->eps[i].size = usb_endpoint_maxp(br_dev->eps[i].a_desc);
		br_dev->eps[i].buffer = devm_kzalloc(&interface->dev, 65536,
								GFP_KERNEL);
		if (!br_dev->eps[i].buffer) {
			dev_err(&br_dev->interface->dev,
				"eps[%d]: Could not allocate buffer\n", i);
			goto error;
		}

		br_dev->eps[i].urb = usb_alloc_urb(0, GFP_KERNEL);

		if (!br_dev->eps[i].urb) {
			dev_err(&br_dev->interface->dev,
				"eps[%d]: Could not allocate urb\n", i);
			goto error;
		}

		dev_dbg(&br_dev->interface->dev, "eps[%d]: %s maxpacket %d\n",
			i, ep_name_from_desc(br_dev->eps[i].a_desc, epname),
			(int) br_dev->eps[i].size);
	}

	/* save our data pointer in this interface device */
	usb_set_intfdata(interface, br_dev);

	/* let the user know about this device */
	dev_dbg(&br_dev->interface->dev,
		"UTBridge device(%d-%s) num_alt=%d max_eps=%d\n",
		br_dev->udev->bus->busnum, br_dev->udev->devpath,
		interface->num_altsetting, br_dev->max_eps);

	/* Print Hardware Info*/
	utbridge_print_version(br_dev);

	/* Configure downstream port for bridge mode */
	utbridge_port_select(br_dev, bridgeport);

	/* Ensure all ports are powered */
	for (i = br_dev->info->port_num_offset + 1;
			i <= br_dev->info->max_bridge_port; i++) {
		retval = utbridge_port_power(br_dev, i, 1);
		if (retval < 0)
			goto error;
	}

	id = ida_simple_get(&ut_ida, 0, 0, GFP_KERNEL);
	if (id < 0) {
		dev_err(&br_dev->interface->dev, "Could not get device id\n");
		retval = id;
		goto error;
	}

	br_dev->pdev = platform_device_alloc(gadget_name, id);
	if (!br_dev->pdev) {
		dev_err(&br_dev->interface->dev, "Not able to register %s\n",
			gadget_name);
		retval = -ENOMEM;
		goto err_plat_alloc;
	}
	br_dev->pdev->dev.parent = &br_dev->interface->dev;

	retval = platform_device_add(br_dev->pdev);
	if (retval < 0)
		goto err_add_udc;

	/* Create device attributes */
	retval = device_create_file(&br_dev->interface->dev,
				&dev_attr_bridgeport);
	if (retval)
		goto err_dev_attr_bport;
	return 0;

err_dev_attr_bport:
	platform_device_del(br_dev->pdev);
err_add_udc:
	platform_device_put(br_dev->pdev);
err_plat_alloc:
	ida_simple_remove(&ut_ida, id);
error:
	if (br_dev)
		/* this frees allocated memory */
		kref_put(&br_dev->kref, utbridge_delete);
	return retval;
}

static void utbridge_disconnect(struct usb_interface *intf)
{
	struct usb_utbridge *br_dev = usb_get_intfdata(intf);
	int id = br_dev->pdev->id;

	/* Remove device attributes */
	device_remove_file(&br_dev->interface->dev, &dev_attr_bridgeport);

	usb_set_intfdata(intf, NULL);

	platform_device_unregister(br_dev->pdev);
	ida_simple_remove(&ut_ida, id);

	/* prevent more I/O from starting */
	mutex_lock(&br_dev->io_mutex);
	br_dev->interface = NULL;
	mutex_unlock(&br_dev->io_mutex);
}

static int utbridge_suspend(struct usb_interface *intf, pm_message_t message)
{
	struct usb_utbridge *br_dev = usb_get_intfdata(intf);

	if (!br_dev)
		return 0;
	utbridge_draw_down(br_dev);

	/* Configure all downstream ports for hub mode */
	utbridge_port_select(br_dev, 0);

	return 0;
}

static int utbridge_resume(struct usb_interface *intf)
{
	return 0;
}

static int utbridge_pre_reset(struct usb_interface *intf)
{
	struct usb_utbridge *br_dev = usb_get_intfdata(intf);

	mutex_lock(&br_dev->io_mutex);
	utbridge_draw_down(br_dev);

	return 0;
}

static int utbridge_post_reset(struct usb_interface *intf)
{
	struct usb_utbridge *br_dev = usb_get_intfdata(intf);

	mutex_unlock(&br_dev->io_mutex);

	return 0;
}

static struct usb_driver utbridge_driver = {
	.name       = "utbridge",
	.probe      = utbridge_probe,
	.disconnect = utbridge_disconnect,
	.suspend    = utbridge_suspend,
	.resume     = utbridge_resume,
	.pre_reset  = utbridge_pre_reset,
	.post_reset = utbridge_post_reset,
	.id_table   = utbridge_table,
	.supports_autosuspend = 1,
};

static int __init utbridge_init(void)
{
	int retval;

	pr_info("Initializing UTBridge driver...\n");

	/* register the driver, return usb_register return code if error */
	retval = usb_register(&utbridge_driver);
	if (retval < 0) {
		pr_err("cannot register usb driver.\n");
		return retval;
	}

	retval = platform_driver_register(&utbridge_udc_driver);
	if (retval < 0) {
		pr_err("cannot register platform driver.\n");
		goto err_reg_platform_driver;
	}

	pr_info("UTBridge support registered.\n");
	return 0;

err_reg_platform_driver:
	usb_deregister(&utbridge_driver);
	return retval;
}

static void __exit utbridge_exit(void)
{
	pr_debug("utbridge_exit() called\n");

	/* Deregister the driver
	 * This will cause disconnect() to be called for each
	 * attached unit
	 */
	pr_debug("-- calling platform_driver_unregister()\n");
	platform_driver_unregister(&utbridge_udc_driver);

	pr_debug("-- calling usb_deregister()\n");
	usb_deregister(&utbridge_driver);
}

module_init(utbridge_init);
module_exit(utbridge_exit);

MODULE_AUTHOR("Martin Herzog <mherzog@de.adit-jv.com>");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL v2");
MODULE_VERSION(DRIVER_VERSION);
