/*
 * INC netdevice
 *
 * Copyright (C) 2012 Andreas Pape, Robert Bosch Car Multimedia GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <linux/if_arp.h>
#include <linux/interrupt.h>
#include "inc_netdev.h"
#include "ssi32_prot.h"


#define INCNDEV_ALEN 1




static int incnetdev_open(struct net_device *ndev)
{
	struct inc_ndev *incndev;

	incndev = (struct inc_ndev *)netdev_priv(ndev);
	ssi32_master_start(incndev->port_handle);

	netif_wake_queue(ndev);
	return 0;
}

static int incnetdev_close(struct net_device *ndev)
{
	struct inc_ndev *incndev;

	netif_stop_queue(ndev);

	incndev = (struct inc_ndev *)netdev_priv(ndev);
	ssi32_master_stop(incndev->port_handle);
	return 0;
}

/*allow new packets to be transmitted to netdev*/
void incnetdev_next_tx_packets(struct inc_ndev *incndev)
{
	/* wake queue only if net device is up */
	if (netif_running(incndev->ndev) && netif_oper_up(incndev->ndev))
		netif_wake_queue(incndev->ndev);
}

/*stop transmitting new packets to netdev*/
void incnetdev_stop_tx_packets(struct inc_ndev *incndev)
{
	netif_stop_queue(incndev->ndev);
}


/*err on phys level*/
void incnetdev_carrier_err_ind(struct inc_ndev *incndev)
{
	incndev->ndev->stats.tx_carrier_errors++;
}

/*tx err*/
void incnetdev_tx_err_ind(struct inc_ndev *incndev)
{
	incndev->ndev->stats.tx_errors++;
}


/*rx err*/
void incnetdev_rx_err_ind(struct inc_ndev *incndev)
{
	incndev->ndev->stats.rx_errors++;
}


/*packet was sent/dropped*/
int incnetdev_sent_tx_packet(struct inc_ndev *incndev,
				struct sk_buff *skb, int success)
{
	struct inc_hdr *hdr;

	/*update statistic and free packet*/
	if (success) {
		incndev->ndev->stats.tx_packets++;
		incndev->ndev->stats.tx_bytes += skb->len;
	} else {
		incndev->ndev->stats.tx_dropped++;

		skb_push(skb, sizeof(struct inc_hdr));
		hdr = (struct inc_hdr *) skb->data;
		dev_err(&incndev->ndev->dev,
				"tx packet dropped: LUN 0x%02X->0x%02X node %d",
				hdr->src_lun, hdr->dest_lun, hdr->dest_node);
	}
	dev_kfree_skb_any(skb);
	return 0;
}

static netdev_tx_t incnetdev_xmit(struct sk_buff *skb, struct net_device *ndev)
{
	struct inc_ndev *incndev;
	struct inc_hdr *hdr;
	if (!ndev)
		return -EINVAL;

	incndev = (struct inc_ndev *)netdev_priv(ndev);

	if (skb->protocol != htons(ETH_P_INC)) {
		dev_warn(&ndev->dev, "invalid protocol 0x%04x",
				ntohs(skb->protocol));
		kfree_skb(skb);
		return NETDEV_TX_OK;
	}

	if (skb->len <= sizeof(struct inc_hdr)) {
		hdr = (struct inc_hdr *) skb->data;
		dev_err(&ndev->dev,
				"invalid length %d LUN 0x%02X->0x%02X node %d",
				skb->len, hdr->src_lun, hdr->dest_lun,
				hdr->dest_node);
		kfree_skb(skb);
		return NETDEV_TX_OK;
	}

	/*trigger port*/
	hdr = (struct inc_hdr *)skb->data;
	skb_pull(skb, sizeof(*hdr));
	if (ssi32m_tx(incndev->port_handle, skb, hdr))
		return NETDEV_TX_BUSY;
	else
		return NETDEV_TX_OK;
}


static int incnetdev_set_mac_address(struct net_device *ndev, void *p)
{
	memcpy(ndev->dev_addr, p, ndev->addr_len);
	return 0;
}



static const struct net_device_ops incnetdev_ops = {
	.ndo_open = incnetdev_open,
	.ndo_stop = incnetdev_close,
	.ndo_start_xmit = incnetdev_xmit,
	.ndo_set_mac_address = incnetdev_set_mac_address,
};



int incnetdev_rx_packet_drop(struct inc_ndev *incndev,
			struct sk_buff *skb)
{
	/*don't treat as dropped packet in statistic ->just free*/
	kfree_skb(skb);
	return 0;
}


int incnetdev_rx_packet_reset(struct inc_ndev *incndev,
			struct sk_buff *skb)
{
	/*TODO: goal is to rollback the skb to the same state as after
	 * incnetdev_rx_packet_alloc; is header hidden when trimming to 0? */
	skb_trim(skb, 0);
	return 0;
}



int incnetdev_rx_packet_put(struct inc_ndev *incndev,
			struct sk_buff *skb)
{
	int res;
	int len = skb->len;
	/*make header visible again*/
	skb_push(skb, sizeof(struct inc_hdr));

	/*initialize MAC header to prevent panic on SOCK_PACKET type sockets*/
	skb_reset_mac_header(skb);

	/*pass to socket*/
	res = netif_rx(skb);
	if (!in_interrupt())
		raise_softirq_irqoff(NET_RX_SOFTIRQ);

	/*update statistic*/
	if (!res) {
		incndev->ndev->stats.rx_packets++;
		incndev->ndev->stats.rx_bytes += len;
	} else
		incndev->ndev->stats.rx_dropped++;

	return 0;
}

int incnetdev_rx_packet_cpy(struct inc_ndev *incndev,
		struct sk_buff *skb, unsigned char *buf, unsigned int length)
{
	struct inc_hdr *hdr;

	/*cpy data*/
	if (skb_tailroom(skb) < length) {
		skb_push(skb, sizeof(struct inc_hdr));
		hdr = (struct inc_hdr *) skb->data;
		dev_alert(&incndev->ndev->dev, "SKB overrun: LUN 0x%02X",
				hdr->dest_lun);
		skb_pull(skb, sizeof(struct inc_hdr));
		return -ENOMEM;
	}
	memcpy(skb_put(skb, length), buf, length);
	return 0;
}

unsigned int incnetdev_mtu(struct inc_ndev *incndev)
{
	return incndev->ndev->mtu;
}

struct sk_buff *incnetdev_rx_packet_alloc(struct inc_ndev *incndev,
		struct inc_hdr *hdr, unsigned int length)
{
	struct sk_buff *skb;
	u8 *dst = NULL;

	skb = netdev_alloc_skb(incndev->ndev, length
				+ sizeof(struct inc_hdr));
	if (!skb) {
		incndev->ndev->stats.rx_dropped++;
		dev_alert(&incndev->ndev->dev, "Alloc SKB failed: LUN 0x%02X",
				hdr->dest_lun);
		return NULL;
	}
	skb->protocol = htons(ETH_P_INC);

	/*cpy header*/
	dst = skb_put(skb, sizeof(struct inc_hdr));
	memcpy(dst, hdr, sizeof(struct inc_hdr));

	/*hide header*/
	skb_pull(skb, sizeof(struct inc_hdr));

	return skb;
}

void incnetdev_configure(struct net_device *ndev, __be32 addr)
{
	struct ifreq ir;
	struct sockaddr_in *sin = (void *) &ir.ifr_ifru.ifru_addr;
	mm_segment_t oldfs = get_fs();
	struct net *net = dev_net(ndev);
	int res;

	memset(&ir, 0, sizeof(ir));
	strcpy(ir.ifr_ifrn.ifrn_name, ndev->name);
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = addr;
	sin->sin_port = 0;

	set_fs(get_ds());
	res = devinet_ioctl(net, SIOCSIFADDR, (struct ifreq __user *) &ir);
	set_fs(oldfs);

	if (res) {
		dev_err(&ndev->dev, "SIOCSIFADDR failed on %s\n", ndev->name);
		return;
	}

	ir.ifr_ifru.ifru_flags = ndev->flags | IFF_UP;

	set_fs(get_ds());
	res = devinet_ioctl(net, SIOCSIFFLAGS, (struct ifreq __user *) &ir);
	set_fs(oldfs);

	if (res < 0) {
		dev_err(&ndev->dev, "Failed to activate %s\n", ndev->name);
		return;
	}

	dev_info(&ndev->dev, "%pI4 up\n", &addr);

	return;
}

static void incnetdev_setup(struct net_device *ndev)
{
	struct inc_ndev *incndev;
	incndev = (struct inc_ndev *)netdev_priv(ndev);

	ndev->features = 0;
	ndev->netdev_ops = &incnetdev_ops;
	ndev->hard_header_len = 0;
	ndev->type = ETH_P_INC;
/*	ndev->type = ARPHRD_ETHER;*/
	ndev->flags = IFF_NOARP;
	ndev->tx_queue_len = 100;
	ndev->destructor = free_netdev;
	ndev->addr_len = INCNDEV_ALEN;
}


int inc_netdev_destroy(struct inc_ndev *incndev)
{
	unregister_netdev(incndev->ndev);
	return 0;
}

struct inc_ndev *inc_netdev_create(void *port_handle, char mac,
		unsigned int mtu, char *ifname)
{
	struct net_device *ndev;
	struct inc_ndev *incndev;
	int res;
	char *ifn = ifname ? ifname : "incdev%d";

	ndev = alloc_netdev(sizeof(struct inc_ndev), ifn, incnetdev_setup);
	if (!ndev)
		return NULL;

	netif_stop_queue(ndev);
	incndev = netdev_priv(ndev);
	incndev->ndev = ndev;
	memset(ndev->dev_addr, 0 , sizeof(ndev->dev_addr));
	ndev->dev_addr[0] = mac;

	incndev->port_handle = port_handle;
	ndev->mtu = mtu;

	/* Register network device. */
	res = register_netdev(ndev);
	if (res)
		pr_err("incDEV: Reg. error: %d.\n", res);

	return incndev;
}
