/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_types.h"
#else
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_card.h"
#endif
#include "csr_wifi_ps_q.h"
#include "csr_wifi_ps_circ.h"
#include "csr_wifi_ps_qsig.h"

#include "csr_wifi_ps_if.h"
#include "csr_wifi_ps.h"
#include "csr_wifi_hip_core_trace.h"

#ifdef PS_USPACE
#include "csr_wifi_ps_card.h"
#endif

static const CsrUint32 tokens_per_byte = 100;

static CsrUint32 ps_dwrr_q_peek(card_t *card, CsrWifiHipPacketSchedulerList *list, CsrWifiHipPacketSchedulerPriorityQueue **ac_qp);

/*
 * ---------------------------------------------------------------------------
 *  ps_dwrr_q_active
 *
 *  See if the queue has anything in it.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      q                 Pointer to queue
 *
 *  Returns:
 *      1 for active (has packets), 0 otherwise.
 * ---------------------------------------------------------------------------
 */
static CsrUint32 ps_dwrr_q_active(card_t *card, CsrWifiHipPacketSchedulerList *list)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;

    /* Find a queue with this priority with a packet on it. */
    return ps_dwrr_q_peek(card, list, &ac_q);
}

/*
 * ---------------------------------------------------------------------------
 *  ps_dwrr_q_peek
 *
 *  Find a queue with a packet in it. Return the length of the packet.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      q                 Pointer to queue
 *      sigp              Pointer to signal
 *
 *  Returns:
 *      length of data
 * ---------------------------------------------------------------------------
 */
static CsrUint32 ps_dwrr_q_peek(card_t *card, CsrWifiHipPacketSchedulerList *list, CsrWifiHipPacketSchedulerPriorityQueue **ac_qp)
{
    CsrWifiHipPacketSchedulerQsig *qsig;
    CsrUint32 rc;
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q, *start;

    rc = 0;

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) csrWifiHipPacketSchedulerListGetNext(list);
    start = ac_q;
    while (ac_q)
    {
        qsig = ac_q->common.peek(card, &ac_q->common);
        if (qsig)
        {
            (*ac_qp) = ac_q;
            rc = qsig_get_signal_datalen(qsig);
            if (rc == 0)
            {
                CSR_LOG_TEXT_ERROR((CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS,
                                    "ps_dwrr_q_peek: qsig len==%d!\n", rc));
            }
            break;
        }
        ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) csrWifiHipPacketSchedulerListGetNext(list);
        if (ac_q == start)
        {
            break;
        }
    }
    return rc;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDwrrInitInstance
 *
 *  Initialise a DWRR instance with proportional values for distribution
 *  of tokens. When a queue has no traffic on it, the tokens allocated
 *  to the queue are re-allocated amongst the other active queues.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      window            Available tx window.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS, CSR_RESULT_FAILURE
 * ---------------------------------------------------------------------------
 */
void csrWifiHipPacketSchedulerDwrrInitInstance(CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi)
{
    CsrUint32 q;

    /* Percentages */
    fi->qs[UNIFI_TRAFFIC_Q_BK].pro = 5; /* 0 */
    fi->qs[UNIFI_TRAFFIC_Q_BE].pro = 15; /* 1 */
    fi->qs[UNIFI_TRAFFIC_Q_VI].pro = 30; /* 2 */
    fi->qs[UNIFI_TRAFFIC_Q_VO].pro = 50; /* 3 */

    for (q = 0; q < PS_MAX_FAIRNESS_PRI; q++)
    {
        fi->qs[q].incl = 0;
        fi->qs[q].tokens = 0;
        fi->qs[q].ptr = 0;
        fi->qs[q].max_pkts = 0;
        fi->qs[q].max_bytes = 0;
        fi->qs[q].pkts_sent = 0;
        fi->qs[q].bytes_sent = 0;
        fi->qs[q].missed = 0;
        fi->qs[q].scheduled = 0;
        fi->qs[q].accumulated_tokens = 0;
    }
}

/*
 * ---------------------------------------------------------------------------
 *  ps_dwrr_calc_token_buckets
 *
 *  Distribute the tokens to the participating (active) queues.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      window            Available tx window.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS, CSR_RESULT_FAILURE
 * ---------------------------------------------------------------------------
 */
static void ps_dwrr_calc_token_buckets(CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi, CsrInt32 max_tokens)
{
    CsrUint32 q;
    CsrUint32 pro_total;
    CsrUint32 multiplier;

    /* Calculate the proportional divisor */
    pro_total = 0;
    for (q = 0; q < PS_MAX_FAIRNESS_PRI; q++)
    {
        if (fi->qs[q].incl)
        {
            pro_total += fi->qs[q].pro;
        }
    }

    if (pro_total == 0)
    {
        return;
    }

    /* Using the proportional divisor, determine quantum multiplier */
    for (q = 0; q < PS_MAX_FAIRNESS_PRI; q++)
    {
        if (fi->qs[q].incl)
        {
            /*lint -e414*/
            multiplier = ((fi->qs[q].pro) * tokens_per_byte) / pro_total;
            fi->qs[q].tokens += ((CsrUint32) max_tokens / tokens_per_byte) * multiplier;
            if (fi->qs[q].tokens >= 0)
            {
                fi->qs[q].scheduled++;
            }
            else
            {
                fi->qs[q].missed++;
            }
            fi->qs[q].accumulated_tokens += ((CsrUint32) max_tokens / tokens_per_byte) * multiplier;
        }
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDwrrInitTokenMax
 *
 *  Calculate the maximum token count using the window value. Then determine
 *  the set of ac with traffic pending and distribute the tokens between the
 *  ac according to the configured proportion.
 *
 *  Arguments:
 *      card              Pointer to card structure.
 *      plists            Pointer to ac lists.
 *      pri               DWRR instance (maps to QS priority).
 *      window            Available tx window.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS, CSR_WIFI_HIP_RESULT_NOT_FOUND
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerDwrrInitTokenMax(card_t *card, CsrWifiHipPacketSchedulerList *plists, CsrInt32 pri, CsrUint32 window)
{
    CsrInt32 q;
    CsrInt32 token_max;
    CsrInt32 reset_tokens = 0;
    CsrUint32 last_window = 0;
    CsrWifiHipPacketSchedulerList *lsp;
    CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi;

    fi = &card->fh_buffer_d.packet_scheduler.fairness_algo[pri];
    last_window = fi->last_window;
    fi->last_window = window;

    if (window < last_window)
    {
        reset_tokens = 1;
    }

    fi->active_queues = 0;
    token_max = (window * tokens_per_byte);
    /* Figure out which queues are going to take part. */
    for (q = PS_MAX_FAIRNESS_PRI - 1, lsp = plists + (PS_MAX_FAIRNESS_PRI - 1); q >= 0; q--, lsp--)
    {
        if (reset_tokens)
        {
            fi->qs[q].tokens = 0;
        }
        if (ps_dwrr_q_active(card, lsp))
        {
            fi->qs[q].incl = 1;
            fi->active_queues++;
        }
        else
        {
            fi->qs[q].incl = 0;
        }
    }
    if (!(fi->active_queues))
    {
        return CSR_WIFI_HIP_RESULT_NOT_FOUND;
    }

    if (!reset_tokens)
    {
        /* Subtract carry-over tokens from new window or add back in deficits. */
        for (q = 0; q < PS_MAX_FAIRNESS_PRI; q++)
        {
            if (fi->qs[q].incl && (fi->qs[q].tokens > 0))
            {
                token_max -= fi->qs[q].tokens;
            }
        }
    }
    ps_dwrr_calc_token_buckets(fi, token_max);
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  ps_dwrr_redistribute_tokens
 *
 *  When a queue has no packets pending, redistribute the DWRR tokens.
 *
 *  Arguments:
 *      q              Queue number (index).
 *
 *  Returns:
 *      void
 * ---------------------------------------------------------------------------
 */
static void ps_dwrr_redistribute_tokens(CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi, CsrUint32 q)
{
    fi->qs[q].incl = 0;
    if (fi->qs[q].tokens > 0)
    {
        ps_dwrr_calc_token_buckets(fi, fi->qs[q].tokens);
        fi->qs[q].tokens = 0;
    }
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDwrrScheduleSignal
 *
 *  Deficit Weighted Round Robin scheduling of next packet to send.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      plist           Point to list of queues.
 *      pri             Instance of DWRR (hi or normal priority).
 *      signal_p        Pointer to signal to pass back.
 *      q_p             Pointer to common queue structure to pass back.
 *      window          Available window.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS, CSR_RESULT_FAILURE
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerDwrrScheduleSignal(
    card_t                            *card,
    CsrWifiHipPacketSchedulerList     *plists,
    CsrInt32                           pri,
    CsrWifiHipPacketSchedulerQsig    **qsig_p,
    CsrWifiPacketSchedulerCommonPart **cp_p,
    CsrUint32                          window)
{
    CsrInt32 q;
    CsrUint32 redist_count;
    CsrWifiHipPacketSchedulerList *lsp;
    CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi;

    fi = &card->fh_buffer_d.packet_scheduler.fairness_algo[pri];

    *qsig_p = 0;
    *cp_p = 0;

    /* Continue processing until all queues are inactive (ie. no traffic or no tokens). */
    redist_count = fi->active_queues;
    while (redist_count--)
    {
        for (q = PS_MAX_FAIRNESS_PRI - 1, lsp = plists + (PS_MAX_FAIRNESS_PRI - 1); q >= 0; q--, lsp--)
        {
            CsrUint32 pktlen;
            CsrWifiHipPacketSchedulerPriorityQueue *ac_q = NULL;

            if (fi->qs[q].incl)
            {
                if (fi->qs[q].tokens >= 0)
                {
                    pktlen = ps_dwrr_q_peek(card, lsp, &ac_q);
                    if (pktlen && (pktlen <= window))
                    {
                        (*qsig_p) = ac_q->common.peek(card, &(ac_q->common));
                        (*cp_p) = (CsrWifiPacketSchedulerCommonPart *) ac_q;

                        fi->qs[q].tokens -= (pktlen * tokens_per_byte);
                        if (fi->qs[q].tokens < 0)
                        {
                            fi->active_queues--;
                            fi->qs[q].incl = 0;
                        }
                        return CSR_RESULT_SUCCESS;
                    }
                    else
                    {
                        /* queue is empty - redistribute */
                        fi->qs[q].incl = 0;
                        fi->active_queues--;
                        if (fi->active_queues == 0)
                        {
                            return CSR_RESULT_SUCCESS;
                        }
                        ps_dwrr_redistribute_tokens(fi, q);
                        break;
                    }
                }
            }
        }
    }
    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipPacketSchedulerDwrrScheduleSignalComplete
 *
 *  Deficit Weighted Round Robin scheduling of next packet to send.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      qsig            Pointer to signal.
 *      cp              Pointer to common queue structure.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS, CSR_RESULT_FAILURE
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipPacketSchedulerDwrrScheduleSignalComplete(card_t *card, CsrWifiHipPacketSchedulerQsig *qsig, CsrWifiPacketSchedulerCommonPart *cp)
{
    CsrWifiHipPacketSchedulerPriorityQueue *ac_q;
    CsrWifiHipPacketSchedulerQueueSet *qs;
    CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;

    ac_q = (CsrWifiHipPacketSchedulerPriorityQueue *) cp;
    qs = cp->qs;

    /* Remove the signal from the circular buffer. */
    if (ac_q->common.dequeue(card, &(ac_q->common), qsig) != CSR_RESULT_SUCCESS)
    {
        return CSR_RESULT_SUCCESS;
    }

    CSR_LOG_TEXT_DEBUG((
                           CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Sending UNICAST: VIF=%d AID=%d signal=0x%p len=%d ac=%d\n",
                           card->instance, qs->vif->id, qs->id, (void *) qsig, qsig_get_signal_datalen(qsig), qsig_get_signal_pri(qsig)));

    ac_q->common.qcod += qsig_get_signal_datalen(qsig);
    if (ac_q->common.qcod >= MOD_TO_BYTES(ac_q->common.qmod))
    {
        /* If the AC queue exceeds its QMOD, remove it from selection. */
        CSR_LOG_TEXT_INFO((
                              CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: QMOD (mod=%d cod=%d): Disabling AC %d for VIF %d AID %d\n",
                              card->instance,
                              MOD_TO_BYTES(ac_q->common.qmod), ac_q->common.qcod, ac_q->pri, qs->vif->id, qs->id));
        csrWifiHipPacketSchedulerDisableQueue(&ac_q->common);
    }

    if (qs->burstMaximum)
    {
        qs->burstCurrent++;
        /* If this QS is configured for burst mode, check if we now match the burst. */
        if ((qs->burstCurrent >= qs->burstMaximum) || (qs->total == 0))
        {
            csrWifiHipPacketSchedulerDisableQueueSet(qs);
            qs->burstMaximum = 0;
            CSR_LOG_TEXT_INFO((
                                  CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: BurstMode: Disable QS for VIF %d AID %d\n",
                                  card->instance,  qs->vif->id, qs->id));
            /* Do action here: Set EOSP bit. */
        }
    }
    qs->scod += qsig_get_signal_datalen(qsig);
    if (qs->scod >= MOD_TO_BYTES(qs->smod))
    {
        /* If the QS exceeds its SMOD, remove it (ie. all of its AC queues) from selection. */
        CSR_LOG_TEXT_DEBUG((
                               CsrWifiHipLto, CSR_WIFI_HIP_LTSO_PS, "unifi%d: SMOD (mod=%d cod=%d): Disabling QS for VIF %d AID %d\n",
                               card->instance,
                               MOD_TO_BYTES(qs->smod), qs->scod, qs->vif->id, qs->id));
        csrWifiHipPacketSchedulerDisableQueueSet(qs);
    }
    /*
     * The signal has bulk data. If it has a pending clear q pointer set (which it should),
     * place the signal on the pending clear queue.
     *
     * [ For retransmissions, if implemented, signals are taken from the pending clear queue
     * (selectively, according to f/w signals) and placed on the appropriate rtx queue ready
     * for scheduling. ]
     *
     * The various pending clear queues (control, VIF, QS) identify the set of signals yet
     * to be cleared by f/w. If a VIF, Queue Set or the entire Packet Scheduler is destroyed,
     * the signals in the relevant pending clear queues are purged, ensuring that any slot
     * clear indications received from f/w after the QS or VIF destruction do not result in
     * incorrect access of destroyed Packet Scheduler queues.
     */
    pending_clr_q = cp->pending_clr_q;
    if (pending_clr_q)
    {
        pending_clr_q->common.enqueue(card, &(pending_clr_q->common), qsig);
    }
    else
    {
        CSR_LOG_TEXT_ERROR((
                               CsrWifiHipLto,  CSR_WIFI_HIP_LTSO_PS,  "unifi%d: Unable to add to pending-clr-q (undefined for queue) AC signal=0x%p len=%d\n",
                               card->instance,  (void *) qsig, qsig_get_signal_datalen(qsig)));
    }

    return CSR_RESULT_SUCCESS;
}
