/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#ifndef CSR_WIFI_PS_H__
#define CSR_WIFI_PS_H__

#include "csr_synergy.h"
#include "csr_wifi_hip_defs.h"

#ifdef __cplusplus
extern "C" {
#endif

#define PS_MAX_FAIRNESS_PRI 4

#define PS_MAX_SMOD CSR_WIFI_HIP_PSCHED_MOD
/*
 * The power save SMOD value is estimated to allow enough frames into the chip
 * so that power save throughput can be maintained without consuming too much
 * buffer space.
 */
#define PS_POWER_SAVE_SMOD 4
#define PS_DEFAULT_QMOD CSR_WIFI_HIP_PSCHED_MOD
/*
 * Multicast QMOD is estimated to allow enough frames into the chip to
 * maintain a significant multicast throughput without preventing all unicast
 * throughput.
 */
#define PS_MCAST_BURST_MAX 16
#define PS_MCAST_QMOD CSR_WIFI_HIP_PSCHED_MOD

struct csrWifiHipPacketSchedulerVif;
struct csrWifiHipPacketSchedulerQueueSet;
struct csrWifiHipPacketSchedulerPendingClrQueue;

/* Queue structures */
typedef struct csrWifiPacketSchedulerCommonPart
{
    CsrWifiHipPacketSchedulerListEntry qptr; /* has to be first */
    CsrInt32                           qmod;
    CsrInt32                           qcod;
    struct csrWifiHipPacketSchedulerQsig *(*peek)(struct card *card, struct csrWifiPacketSchedulerCommonPart *cp);
    CsrResult (*dequeue)(struct card *card, struct csrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
    CsrResult (*enqueue)(struct card *card, struct csrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
    void (*free)(struct card *card, struct csrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
    CsrInt32 (*getlen)(struct card *card, struct csrWifiPacketSchedulerCommonPart *cp);
    CsrInt32                                         active;
    struct csrWifiHipPacketSchedulerVif             *vif;
    struct csrWifiHipPacketSchedulerQueueSet        *qs;
    CsrWifiHipPacketSchedulerList                   *list;
    struct csrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q;
    CsrWifiHipPacketSchedulerStats                   stats;
} CsrWifiPacketSchedulerCommonPart;


typedef struct csrWifiHipPacketSchedulerPriorityQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    ps_circ_buffer_t                 pkts;
    CsrInt32                         pri;
    CsrWifiHipPacketSchedulerQState  state;
} CsrWifiHipPacketSchedulerPriorityQueue;

typedef struct csrWifiHipPacketSchedulerRtxQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    CsrWifiHipPacketSchedulerList    pkts;
    CsrInt32                         pri;
} CsrWifiHipPacketSchedulerRtxQueue;

typedef struct csrWifiHipPacketSchedulerCtrlQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    ps_circ_buffer_t                 pkts;
} CsrWifiHipPacketSchedulerCtrlQueue;

typedef struct csrWifiHipPacketSchedulerMgmtQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    ps_circ_buffer_t                 pkts;
} CsrWifiHipPacketSchedulerMgmtQueue;

typedef struct csrWifiHipPacketSchedulerQsMgmtQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    ps_circ_buffer_t                 pkts;
} CsrWifiHipPacketSchedulerQsMgmtQueue;

typedef struct csrWifiHipPacketSchedulerMulticastQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    ps_circ_buffer_t                 pkts;
    CsrInt32                         burstMaximum;
    CsrInt32                         burstCurrent;
    #ifdef PS_USPACE
    CsrWifiHipPacketSchedulerList sent_pkts;
    #endif
} CsrWifiHipPacketSchedulerMulticastQueue;

typedef struct csrWifiHipPacketSchedulerPendingClrQueue
{
    CsrWifiPacketSchedulerCommonPart common;
    CsrWifiHipPacketSchedulerList    pkts;
    CsrInt32                         pri;
} CsrWifiHipPacketSchedulerPendingClrQueue;

typedef enum
{
    QS_STA_POWERSAVE = 1,
    QS_STA_ACTIVE = 2
} CsrWifiHipPacketSchedulerQSState;

typedef struct csrWifiHipPacketSchedulerQueueSet
{
    CsrInt32 id;

    CsrWifiHipPacketSchedulerPriorityQueue   ac[PS_MAX_FAIRNESS_PRI];
    CsrWifiHipPacketSchedulerRtxQueue        retransmit[PS_MAX_FAIRNESS_PRI];
    CsrWifiHipPacketSchedulerQsMgmtQueue     mgmt_q;
    CsrWifiHipPacketSchedulerQsMgmtQueue     auth_q;
    CsrWifiHipPacketSchedulerPendingClrQueue pending_clr_q;
    CsrInt32                                 smod;
    CsrInt32                                 scod;
    CsrInt32                                 burstMaximum;
    CsrInt32                                 burstCurrent;
    CsrInt32                                 total;

    CsrUint32 pri;
    CsrInt32  active;

    struct csrWifiHipPacketSchedulerVif *vif;
    CsrInt32                             max_queue_len;
    CsrWifiHipPacketSchedulerList        hal_queues[PS_MAX_FAIRNESS_PRI];
    CsrWifiHipFlowControlResumeCb        resumeCb;
    CsrWifiHipFlowControlPauseCb         pauseCb;
    void                                *pauseResumeContext;
    CsrInt8                              availability_signalled[PS_MAX_FAIRNESS_PRI];

    CsrWifiHipPacketSchedulerQSAuthQState authq_state;
    CsrWifiHipPacketSchedulerQSACQState   acq_state;
    CsrWifiHipPacketSchedulerQSState      station_state;
    CsrUint32                             station_state_transitions;
} CsrWifiHipPacketSchedulerQueueSet;

/* VIF structure */
#define PS_MAX_STA_PER_AP 32

typedef struct csrWifiHipPacketSchedulerVif
{
    CsrInt32                                 id;
    CsrWifiHipPacketSchedulerQueueSet       *qs[PS_MAX_AID + 1];
    CsrWifiHipPacketSchedulerMgmtQueue       mgmt_q;
    CsrWifiHipPacketSchedulerMulticastQueue  multicast_q;
    CsrWifiHipPacketSchedulerPendingClrQueue pending_clr_q;
    CsrWifiHipPacketSchedulerRtxQueue        rtx_q;
    CsrWifiHipPacketSchedulerDiscardCb       discard_cb;
    CsrWifiHipPacketSchedulerMcastCompleteCb mcast_complete_cb;
    CsrInt32                                 active_qs;
    CsrInt32                                 powersave_qs;
} CsrWifiHipPacketSchedulerVif;

/* Fairness algorithm instance structure */
typedef struct
{
    CsrInt32  incl;
    CsrInt32  tokens;
    CsrInt32  pro;
    CsrInt32  ptr;
    CsrUint32 max_pkts;
    CsrUint32 max_bytes;
    CsrUint32 pkts_sent;
    CsrUint32 bytes_sent;
    CsrUint32 missed;
    CsrUint32 scheduled;
    CsrUint32 accumulated_tokens;
} CsrWifiHipPacketSchedulerFairnessAlgoQueue;

typedef struct
{
    CsrWifiHipPacketSchedulerFairnessAlgoQueue qs[PS_MAX_FAIRNESS_PRI];
    CsrUint32                                  last_window;
    CsrInt32                                   active_queues;
} CsrWifiHipPacketSchedulerFairnessAlgoInstance;

typedef struct priority_scheduler
{
    CsrResult (*schedule_init)(struct card *card, CsrWifiHipPacketSchedulerList *lists, CsrInt32 pri, CsrUint32 window);
    CsrResult (*schedule_signal)(struct card *card, CsrWifiHipPacketSchedulerList *lists, CsrInt32 pri, struct csrWifiHipPacketSchedulerQsig **qsig_p, CsrWifiPacketSchedulerCommonPart **cp_p, CsrUint32 window);
    CsrResult (*schedule_signal_complete)(struct card *card, struct csrWifiHipPacketSchedulerQsig *qsig, CsrWifiPacketSchedulerCommonPart *cp);
    CsrResult (*schedule_complete)(struct card *card);
    CsrWifiHipPacketSchedulerList *lists;
    CsrInt32                       pri;
} CsrWifiHipPacketSchedulerPriorityScheduler;

/* Index into the priority_scheduler array of structures. */
#define PRIORITY_SCHEDULER_SP_INDEX 0
#define PRIORITY_SCHEDULER_SP_RTX_INDEX 1
#define PRIORITY_SCHEDULER_FAIRNESS_HI_INDEX 2
#define PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX 3
#define PRIORITY_SCHEDULER_MAX_INDEX PRIORITY_SCHEDULER_FAIRNESS_NORMAL_INDEX
/* Card structure */
#define PS_MAX_SP_PRI 3
#define PS_MAX_VIFS CSR_WIFI_MAX_INTERFACES

#define PS_QS_PRI_HI_INDEX 0
#define PS_QS_PRI_NORM_INDEX 1
#define PS_MAX_QS_PRI 2

struct packet_scheduler
{
    CsrWifiHipPacketSchedulerList sp[PS_MAX_SP_PRI];
    CsrWifiHipPacketSchedulerList sp_retransmit[PS_MAX_FAIRNESS_PRI];
    CsrWifiHipPacketSchedulerList fairness[PS_MAX_QS_PRI][PS_MAX_FAIRNESS_PRI];

    CsrWifiHipPacketSchedulerVif *vifs[PS_MAX_VIFS];

    CsrWifiHipPacketSchedulerCtrlQueue       ctrl_q;
    CsrWifiHipPacketSchedulerPendingClrQueue pending_clr_q;

    CsrWifiHipPacketSchedulerFairnessAlgoInstance fairness_algo[PS_MAX_QS_PRI];

    CsrInt32                              restock;
    CsrInt32                              initialised;
    CsrMutexHandle                        cfg_mutex;
    CsrInt32                              active_qs;
    CsrInt32                              powersave_qs;
    CsrWifiHipPacketSchedulerList         free_signal_list;
    struct csrWifiHipPacketSchedulerQsig *signal_buf;

    CsrInt32 next_vif;

    struct
    {
        CsrUint32 qsig_empty;
    } stats;
};

#define MAP_VIFID_TO_INDEX(v) (v)
#define MAP_AID_TO_INDEX(a) (a)

#define MOD_MTU 1536
#define MOD_TO_BYTES(m) (m * MOD_MTU)

/* DWRR Scheduler */
extern void csrWifiHipPacketSchedulerDwrrInitInstance(CsrWifiHipPacketSchedulerFairnessAlgoInstance *fi);
extern CsrResult csrWifiHipPacketSchedulerDwrrInitTokenMax(struct card *card, CsrWifiHipPacketSchedulerList *plists, CsrInt32 pri, CsrUint32 window);
extern CsrResult csrWifiHipPacketSchedulerDwrrScheduleSignal(struct card *card, CsrWifiHipPacketSchedulerList *plists, CsrInt32 pri, struct csrWifiHipPacketSchedulerQsig **pkt_p, CsrWifiPacketSchedulerCommonPart **cp_p, CsrUint32 window);
extern CsrResult csrWifiHipPacketSchedulerDwrrScheduleSignalComplete(struct card *card, struct csrWifiHipPacketSchedulerQsig *signal, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerAcScheduleComplete(struct card *card);

/* Strict Priority Scheduler */
extern CsrResult csrWifiHipSpScheduleSignal(struct card *card, CsrWifiHipPacketSchedulerList *plists, CsrInt32 pri_levels, struct csrWifiHipPacketSchedulerQsig **qsig_p, CsrWifiPacketSchedulerCommonPart **cp_p, CsrUint32 window);
extern CsrResult csrWifiHipSpScheduleSignalComplete(struct card *card, struct csrWifiHipPacketSchedulerQsig *qsig, CsrWifiPacketSchedulerCommonPart *cp);

extern CsrWifiHipPacketSchedulerVif *csrWifiHipPacketSchedulerGetVif(struct card *card, CsrInt32 vif_id);
extern CsrWifiHipPacketSchedulerQueueSet *csrWifiHipPacketSchedulerGetQueueSet(struct card *card, CsrInt32 vif_id, CsrInt32 aid);

extern CsrInt32 do_hip_transmit(struct card *card, CsrInt32 window);

extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerGetFreeSignal(struct card *card);
extern void csrWifiHipPacketSchedulerFreeSignal(struct card *card, struct csrWifiHipPacketSchedulerQsig *qsig);
extern CsrInt32 csrWifiHipPacketSchedulerFreeSignalCount(struct card *card);

extern CsrResult csrWifiHipPacketSchedulerEnableQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs);
extern CsrResult csrWifiHipPacketSchedulerDisableQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs);
extern CsrResult csrWifiHipPacketSchedulerEnableRtxQueueSet(CsrWifiHipPacketSchedulerQueueSet *qs);
extern CsrResult csrWifiHipPacketSchedulerEnableQueue(CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerDisableQueue(CsrWifiPacketSchedulerCommonPart *cp);
extern void csrWifiHipPacketSchedulerDistributeSMod(struct card *card);
extern void csrWifiHipPacketSchedulerQueueSetSModHelper(CsrWifiHipPacketSchedulerQueueSet *qs, CsrInt32 smod);
extern CsrResult csrWifiHipPacketSchedulerQueueSetStateSet(struct card *card, CsrInt32 vif_id, CsrInt32 aid, CsrWifiHipPacketSchedulerQSState state);
extern CsrResult csrWifiHipPacketSchedulerNextVifSet(struct card *card, CsrInt32 vif_id, CsrBool *changed);

extern CsrResult csrWifiHipPacketSchedulerPurgeQueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerFreeSlot(struct card *card, struct csrWifiHipPacketSchedulerQsig *qsig);
extern CsrResult csrWifiHipPacketSchedulerRtxSlot(card_t *card, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrResult csrWifiHipPacketSchedulerPendingClrDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerPendingClrPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerPendingClrGetLen(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerPendingClrEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern CsrResult csrWifiHipPacketSchedulerPendingClrPurge(struct card *card, CsrWifiHipPacketSchedulerPendingClrQueue *pending_clr_q);

extern void csrWifiHipPacketSchedulerCommonFree(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrResult csrWifiHipPacketSchedulerCtrlDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerCtrlPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerCtrlGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerCtrlEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrResult csrWifiHipPacketSchedulerMgmtDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerMgmtPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerMgmtGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerMgmtEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrResult csrWifiHipPacketSchedulerMulticastDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerMulticastPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerMulticastGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerMulticastEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern CsrResult csrWifiHipPacketSchedulerQueueMulticastBurstTrigger(struct card *card, CsrInt32 vif_id);

extern CsrResult csrWifiHipPacketSchedulerAcDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerAcPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerAcGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerAcEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern void csrWifiHipPacketSchedulerAcFree(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrResult csrWifiHipPacketSchedulerRtxDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerRtxPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerRtxGetLen(card_t *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerRtxEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern CsrResult csrWifiHipPacketSchedulerRtxPurge(struct card *card, CsrWifiHipPacketSchedulerRtxQueue *rtx_q);

extern CsrResult csrWifiHipPacketSchedulerQsMgmtDequeue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern struct csrWifiHipPacketSchedulerQsig *csrWifiHipPacketSchedulerQsMgmtPeek(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrInt32 csrWifiHipPacketSchedulerQsMgmtGetLen(struct card *card, CsrWifiPacketSchedulerCommonPart *cp);
extern CsrResult csrWifiHipPacketSchedulerQsMgmtEnqueue(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);
extern void csrWifiHipPacketSchedulerQsMgmtEnqueueFree(struct card *card, CsrWifiPacketSchedulerCommonPart *cp, struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrBool csrWifiHipPacketSchedulerCfmRqd(struct csrWifiHipPacketSchedulerQsig *qsig);

extern CsrWifiHipPacketSchedulerPriorityScheduler *csrWifiHipPacketSchedulerGetScheduler(CsrInt32 index);

/*#define TRANSMISSION_CONTROL_OFFSET ((sizeof(CSR_DATAREF) * 2) + sizeof(CSR_VIF_INDEX) + sizeof(CSR_RATE) + sizeof(CSR_CLIENT_TAG) + sizeof(CSR_PRIORITY) + sizeof(CSR_MACADDRESS))*/
#define TRANSMISSION_CONTROL_OFFSET 30
#define GET_TRANSMISSIONCONTROL_FROM_MA_PACKET_REQ(buf) CSR_GET_UINT16_FROM_LITTLE_ENDIAN((buf + TRANSMISSION_CONTROL_OFFSET))
#define SET_TRANSMISSIONCONTROL_FROM_MA_PACKET_REQ(val, buf) CSR_COPY_UINT16_TO_LITTLE_ENDIAN(val, (buf + TRANSMISSION_CONTROL_OFFSET))

#ifdef __cplusplus
}
#endif
#endif
