/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"
/*
 * ***************************************************************************
 *
 *  FILE:     csr_wifi_hip_send.c
 *
 *  PURPOSE:
 *      Code for adding a signal request to the from-host queue.
 *      When the driver bottom-half is run, it will take requests from the
 *      queue and pass them to the UniFi.
 *
 * ***************************************************************************
 */
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_sigs.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip.h"

#include "csr_wifi_ps_if.h"


unifi_TrafficQueue unifi_frame_priority_to_queue(CSR_PRIORITY priority)
{
    switch (priority)
    {
        case CSR_QOS_UP0:
        case CSR_QOS_UP3:
            return UNIFI_TRAFFIC_Q_BE;
        case CSR_QOS_UP1:
        case CSR_QOS_UP2:
            return UNIFI_TRAFFIC_Q_BK;
        case CSR_QOS_UP4:
        case CSR_QOS_UP5:
            return UNIFI_TRAFFIC_Q_VI;
        case CSR_QOS_UP6:
        case CSR_QOS_UP7:
        case CSR_MANAGEMENT:
            return UNIFI_TRAFFIC_Q_VO;
        default:
            return UNIFI_TRAFFIC_Q_BE;
    }
}

CSR_PRIORITY unifi_get_default_downgrade_priority(unifi_TrafficQueue queue)
{
    switch (queue)
    {
        case UNIFI_TRAFFIC_Q_BE:
            return CSR_QOS_UP0;
        case UNIFI_TRAFFIC_Q_BK:
            return CSR_QOS_UP1;
        case UNIFI_TRAFFIC_Q_VI:
            return CSR_QOS_UP5;
        case UNIFI_TRAFFIC_Q_VO:
            return CSR_QOS_UP6;
        default:
            return CSR_QOS_UP0;
    }
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_send_signal
 *
 *    Invokes send_signal() to queue a signal in the command or traffic queue
 *    If sigptr pointer is NULL, it pokes the bh to check if UniFi is responsive.
 *
 *  Arguments:
 *      card        Pointer to card context struct
 *      vifId       The virtual interface ID in packet scheduler
 *      aId         The association ID maps the queue set in packet scheduler
 *      portType    un-controlled or controlled port
 *      frameType   Management or data frame
 *      pktType     Unicast or group addressed frame
 *      sigptr      Pointer to signal from card.
 *      siglen      Size of the signal
 *      bulkdata    Pointer to the bulk data of the signal
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success
 *      CSR_WIFI_HIP_RESULT_NO_SPACE if there were insufficient data slots or no free signal queue entry
 *
 *  Notes:
 *      unifi_send_signal() is used to queue signals, created by the driver,
 *      to the device. Signals are constructed using the UniFi packed structures.
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_send_signal(card_t *card, CsrUint32 vifId, CsrUint32 aId,
                            CsrWifiHipPortType portType,
                            CsrWifiHipFrameType frameType,
                            CsrWifiHipPacketType pktType,
                            const CsrUint8 *sigptr, CsrUint32 siglen,
                            const CsrWifiHipBulkDataParam *bulkdata)
{
    CsrUint16 signal_id;
    CsrResult result = CSR_RESULT_FAILURE;
    CsrUint32 run_bh;
    card_signal_t signal;

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_send_signal (PS): sigptr=%p siglen=%d bulkdata=%p\n",
                           card->instance,  sigptr, siglen, bulkdata));

    /* A NULL signal pointer is a request to check if UniFi is responsive */
    if (sigptr == NULL)
    {
        card->bh_reason_host = 1;
        return unifi_run_bh(card->ospriv);
    }

    /* Make up the card_signal struct */
    CsrMemSet((void *) &signal, 0, sizeof(signal));
    signal.signal_length = (CsrUint16) siglen;
    CsrMemCpy((void *) signal.sigbuf, (void *) sigptr, siglen);
    signal.bulkdata[0].data_length = 0;
    signal.bulkdata[1].data_length = 0;
    if (bulkdata)
    {
        CsrInt32 instance = 0;
        if (bulkdata->d[0].data_length > 0)
        {
            signal.bulkdata[instance] = bulkdata->d[0];
            instance++;
        }
        if (bulkdata->d[1].data_length > 0)
        {
            signal.bulkdata[instance] = bulkdata->d[1];
        }
    }
    run_bh = 1;
    signal_id = GET_SIGNAL_ID(sigptr);


    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_send_signal: Sending signal 0x%02X \n",
                           card->instance, signal_id));

    /*
     * If the signal is a CSR_MA_PACKET_REQUEST ,
     * we send it using the traffic soft queue. Else we use the command soft queue.
     */
    if (signal_id == CSR_MA_PACKET_REQUEST_ID)
    {
        CsrUint16 frame_priority;
        CsrUint32 priority_q;

        if (card->periodic_wake_mode == UNIFI_PERIODIC_WAKE_HOST_ENABLED)
        {
            run_bh = 0;
        }

        /* Map the frame priority to a traffic queue index. */
        frame_priority = GET_PACKED_MA_PACKET_REQUEST_FRAME_PRIORITY(sigptr);
        priority_q = unifi_frame_priority_to_queue((CSR_PRIORITY) frame_priority);

        CSR_LOG_TEXT_DEBUG((
                               CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_send_signal: priority_q=%d\n",
                               card->instance,  priority_q));

        if (CSR_WIFI_HIP_FRAMETYPE_IEEE80211_DATA == frameType)
        {
            if ((CSR_WIFI_HIP_UNICAST_PDU == pktType) || (CSR_WIFI_HIP_PKT_TYPE_RESERVED == pktType))
            {
                if (CSR_WIFI_HIP_PORT_TYPE_UNCONTROLLED == portType)
                {
                    result = CsrWifiHipPacketSchedulerSendQsAuthQueueFrame(card, vifId, aId, &signal);
                    if (result == CSR_RESULT_FAILURE)
                    {
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame() failed\n"
                                           , card->instance));
                    }
                    if (result == CSR_WIFI_PS_RESULT_FULL)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                            "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendQsAuthQueueFrame() success, queue full\n",
                                            card->instance));
                    }
                }
                else
                {
                    result = CsrWifiHipPacketSchedulerSendUnicastFrame(card, vifId, aId, (CsrWifiHipPacketSchedulerACPri) priority_q, &signal);
                    if (result == CSR_RESULT_FAILURE)
                    {
                        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                           "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendUnicastFrame() failed\n"
                                           , card->instance));
                    }
                    if (result == CSR_WIFI_PS_RESULT_FULL)
                    {
                        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                            "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendUnicastFrame() success, queue %d full\n",
                                            card->instance,  priority_q));
                    }
                }
            }
            else if ((CSR_WIFI_HIP_MULTICAST_PDU == pktType) || (CSR_WIFI_HIP_BROADCAST_PDU == pktType))
            {
                result = CsrWifiHipPacketSchedulerSendMulticastFrame(card, vifId, &signal);
                if (result == CSR_RESULT_FAILURE)
                {
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendMulticastFrame() failed\n"
                                       , card->instance));
                }
                if (result == CSR_WIFI_PS_RESULT_FULL)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                        "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendMulticastFrame() success, queue full\n",
                                        card->instance));
                }
            }
        }
        else if (CSR_WIFI_HIP_FRAMETYPE_IEEE80211_MANAGEMENT == frameType)
        {
            if (aId)
            {
                result = CsrWifiHipPacketSchedulerSendQsMgmtFrame(card, vifId, aId, &signal);

                if (result == CSR_RESULT_FAILURE)
                {
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendQsMgmtFrame() failed\n"
                                       , card->instance));
                }
                if (result == CSR_WIFI_PS_RESULT_FULL)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                        "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendQsMgmtFrame() success, queue (aId = %d) full\n",
                                        card->instance,  aId));
                }
            }
            else
            {
                result = CsrWifiHipPacketSchedulerSendMgmtFrame(card, vifId, &signal);

                if (result == CSR_RESULT_FAILURE)
                {
                    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                       "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendMgmtFrame() failed\n"
                                       , card->instance));
                }
                if (result == CSR_WIFI_PS_RESULT_FULL)
                {
                    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                        "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendMgmtFrame() success, queue full\n",
                                        card->instance));
                }
            }
        }
        else
        {
            result = CsrWifiHipPacketSchedulerSendCtrlData(card, &signal);
            if (result == CSR_RESULT_FAILURE)
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                                   "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendCtrlData() failed for reserved frame types\n"
                                   , card->instance));
            }
            if (result == CSR_WIFI_PS_RESULT_FULL)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                                    "unifi%d: unifi_send_signal: CsrWifiHipPacketSchedulerSendCtrlData() success, queue %d full\n",
                                    card->instance,  priority_q));
            }
        }
    }
    else
    {
        if (bulkdata)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_send_signal: \tbulk[0].len=%d bulk[1].len=%d\n",
                                   card->instance,
                                   bulkdata->d[0].data_length, bulkdata->d[1].data_length));
        }
        result = CsrWifiHipPacketSchedulerSendCtrlData(card, &signal);
        if (result == CSR_RESULT_FAILURE)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_send_signal: unifiSendMgmtFrame(vifId=%d) failed for signal 0x%04x\n",
                                   card->instance,  vifId, signal_id));
        }
        if (result == CSR_WIFI_PS_RESULT_FULL)
        {
            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: unifi_send_signal: unifiSendMgmtFrame() success, mgmt queue full\n"
                                  , card->instance));
        }
    }

    if (result != CSR_RESULT_FAILURE)
    {
        result = CSR_RESULT_SUCCESS;
        if (run_bh == 1)
        {
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6, "unifi%d: unifi_send_signal: scheduling bh"
                                   , card->instance));
            card->bh_reason_host = 1;
            if (unifi_run_bh(card->ospriv) != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                       CSR_WIFI_HIP_LOG_DEF, "unifi%d: unifi_send_signal: unifi_run_bh() failed\n",
                                       card->instance));
            }
        }
    }
    return result;
} /* unifi_send_signal() */
