/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

/*
 * ---------------------------------------------------------------------------
 *
 *  FILE:     csr_wifi_hip_card_sdio.h
 *
 *  PURPOSE:
 *      Internal header for Card API for SDIO.
 * ---------------------------------------------------------------------------
 */
#ifndef CSR_WIFI_HIP_CARD_SDIO_H__
#define CSR_WIFI_HIP_CARD_SDIO_H__

#include "csr_synergy.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_unifi_udi.h"
#include "csr_wifi_hip_unifihw.h"
#include "csr_wifi_hip_unifiversion.h"
#ifndef CSR_WIFI_HIP_TA_DISABLE
#include "csr_wifi_hip_ta_sampling.h"
#endif
#include "csr_wifi_hip_xbv.h"
#include "csr_wifi_hip_chiphelper.h"

#include "csr_wifi_ps_q.h"
#include "csr_wifi_ps_circ.h"
#include "csr_wifi_ps_if.h"
#include "csr_wifi_ps.h"
#include "csr_wifi_hip_spinlock.h"

/*
 *
 * Configuration items.
 *
 */

/*
 * When reading signals from UniFi, the host processes pending all signals
 * and then acknowledges them together in a single write to update the
 * to-host-chunks-read location.
 * When there is more than one bulk data transfer (e.g. one received data
 * packet and a request for the payload data of a transmitted packet), the
 * update can be delayed significantly. This ties up resources on chip.
 *
 * To remedy this problem, to-host-chunks-read is updated after processing
 * a signal if TO_HOST_FLUSH_THRESHOLD bytes of bulk data have been
 * transferred since the last update.
 */
#define TO_HOST_FLUSH_THRESHOLD (500 * 5)


/* SDIO Card Common Control Registers */
#define SDIO_CCCR_SDIO_REVISION     (0x00)
#define SDIO_SD_SPEC_REVISION       (0x01)
#define SDIO_IO_ENABLE              (0x02)
#define SDIO_IO_READY               (0x03)
#define SDIO_INT_ENABLE             (0x04)
#define SDIO_INT_PENDING            (0x05)
#define SDIO_IO_ABORT               (0x06)
#define SDIO_BUS_IFACE_CONTROL      (0x07)
#define SDIO_CARD_CAPABILOTY        (0x08)
#define SDIO_COMMON_CIS_POINTER     (0x09)
#define SDIO_BUS_SUSPEND            (0x0C)
#define SDIO_FUNCTION_SELECT        (0x0D)
#define SDIO_EXEC_FLAGS             (0x0E)
#define SDIO_READY_FLAGS            (0x0F)
#define SDIO_FN0_BLOCK_SIZE         (0x10)
#define SDIO_POWER_CONTROL          (0x12)
#define SDIO_VENDOR_START           (0xF0)

#define SDIO_CSR_HOST_WAKEUP        (0xf0)
#define SDIO_CSR_HOST_INT_CLEAR     (0xf1)
#define SDIO_CSR_FROM_HOST_SCRATCH0 (0xf2)
#define SDIO_CSR_FROM_HOST_SCRATCH1 (0xf3)
#define SDIO_CSR_TO_HOST_SCRATCH0   (0xf4)
#define SDIO_CSR_TO_HOST_SCRATCH1   (0xf5)
#define SDIO_CSR_FUNC_EN            (0xf6)
#define SDIO_CSR_CSPI_MODE          (0xf7)
#define SDIO_CSR_CSPI_STATUS        (0xf8)
#define SDIO_CSR_CSPI_PADDING       (0xf9)


#define UNIFI_SD_INT_ENABLE_IENM 0x0001 /* Master INT Enable */

#ifdef CSR_WIFI_DRIVER_HYDRA
/* Offsets of scratch registers.
 * On Hydra chips, these are the only locations in the WLAN SDIO function
 * that can be access via CMD52.
 */
/* Generic SDIO function 8-bit scratch registers, see CS-207880-SP */
#define UF_FROM_H_SCRATCH(n)    (0x10 + (n))
#define UF_TO_H_SCRATCH(n)      (0x20 + (n))

#define UF_IS_FH_SCRATCH(a)     (((a) >= 0x10) && ((a) < 0x18))
#define UF_IS_TH_SCRATCH(a)     (((a) >= 0x20) && ((a) < 0x28))

#define UNIFI_SDIO_CTRL_CFG_R       (UF_TO_H_SCRATCH(0))     /* SDIO config read handle */


/* D-30105: Fix this: The following may be required for rewind */
#define UNIFI_SDIO_HOST_INT_REQ     (UF_FROM_H_SCRATCH(0)) /* Host reqest f/w interrupt */
#define UNIFI_SDIO_CTRL_REWIND_REQ  (UF_FROM_H_SCRATCH(4)) /* CMD53 buffer rewind control */
#define UNIFI_SDIO_CTRL_REWIND_ACK  (UF_TO_H_SCRATCH(4))   /* CMD53 buffer rewind status */

#define UNIFI_SDIO_AUTO_REWIND      (1 << 15)                /* Auto-rewind buffer on r/w */
#endif

#define UNIFI_SDIO_HIP2_CFG_MAGIC   (0x600DF00D)           /* HIP2 config signature */

#define D4_T_2_MAX_BULK_SLOTS_TO    0                      /* mparam_slot_count[0] */
#define D4_T_2_MAX_BULK_SLOTS_FROM  1                      /* mparam_slot_count[1] */

#define CSR_WIFI_HIP_NUM_FH_PUSHED_SEGMENTS  2 /* Default number of from-host push segments */

/*
 * Fixed sized read of config buffer handle
 */
#define SDIO_CONFIG_DATA_RW_SIZE 128
#define SDIO_FH_PUSHED_HANDLES 6
#define SDIO_TH_HANDLES 6
#define SDIO_TH_PULLED_HANDLES 6

typedef struct
{
    CsrUint32 signature;
    CsrUint16 length;

    CsrUint16 version;
    CsrUint16 sub_version;
    CsrUint32 fw_version;
    CsrUint32 patch_version;

    CsrUint16 signal_chunk_size;
    CsrUint16 block_round_size;

    CsrUint16 ack_record_size;
    CsrUint16 ack_buffer_blocks;

    CsrUint16 ctrl_info_size; /* size of control structure in bytes - use to calculate offset of m2 */
    CsrUint16 ctrl_info_blocks; /* number of blocks for control structure */
    CsrUint16 buff_maximum_size_blocks; /* f/w passed value in blocks */
    CsrUint16 push_maximum_size;

    CsrUint16 th_ctrl_round;
    CsrUint16 th_bulk_round;
    CsrUint16 fh_pulled_round;
    CsrUint16 fh_pushed_round;
    CsrUint16 fh_pushed_block_gap;

    CsrUint16 bf_entries;
    CsrUint16 bt_entries;

    CsrUint16 config_response_buffer_h;
    CsrUint16 ctrl_buffer_h;
    CsrUint16 ctrl_re_read_buffer_h;
    CsrUint16 ack_buffer_h;
    CsrUint16 fh_pushed_buffer_h[SDIO_FH_PUSHED_HANDLES];

    CsrUint16 log_buffer_h;
} sdio_config_data_t;

/* Offset of the INIT flag in the config info block. */
#define SDIO_TO_HOST_SIG_PADDING_OFFSET 0x1C


/* Structure for a bulk data transfer command */
typedef struct
{
    CsrUint16 cmd;
    CsrUint16 data_slot;
    CsrUint16 len;
    CsrUint16 slot_offset;
    CsrUint16 pipe_ptr;
} bulk_data_cmd_t;


/* Bulk Data signal command values */
#define SDIO_CMD_SIGNAL                 0x00
#define SDIO_CMD_TO_HOST_TRANSFER       0x01
#define SDIO_CMD_TO_HOST_TRANSFER_ACK   0x02 /*deprecated*/
#define SDIO_CMD_FROM_HOST_TRANSFER     0x03
#define SDIO_CMD_FROM_HOST_TRANSFER_ACK 0x04 /*deprecated*/
#define SDIO_CMD_CLEAR_SLOT             0x05
#define SDIO_CMD_OVERLAY_TRANSFER       0x06
#define SDIO_CMD_OVERLAY_TRANSFER_ACK   0x07 /*deprecated*/
#define SDIO_CMD_FROM_HOST_AND_CLEAR    0x08
#define SDIO_CMD_BUFFER_HEADER          0x09 /*new*/
#define SDIO_CMD_PADDING                0x0f

#define SLOT_DIR_TO_HOST 0x8000


/* Initialise bulkdata slot
 *  params:
 *      CsrWifiHipBulkDataDesc *bulk_data_slot
 */
#define UNIFI_INIT_BULK_DATA(bulk_data_slot)        \
    {                                               \
        (bulk_data_slot)->os_data_ptr = NULL;       \
        (bulk_data_slot)->data_length = 0;          \
        (bulk_data_slot)->os_net_buf_ptr = NULL;    \
        (bulk_data_slot)->net_buf_length = 0;       \
    }

/*
 * Structure to contain a SIGNAL datagram.
 * This is used to build signal queues between the main driver and the
 * i/o thread.
 * The fields are:
 *      sigbuf          Contains the HIP signal is wire-format (i.e. packed,
 *                      little-endian)
 *      bulkdata        Contains a copy of any associated bulk data
 *      signal_length   The size of the signal in the sigbuf
 */
typedef struct card_signal
{
    CsrUint8 sigbuf[UNIFI_PACKED_SIGBUF_SIZE];

    /* Length of the SIGNAL inside sigbuf */
    CsrUint16 signal_length;

    CsrWifiHipBulkDataDesc bulkdata[UNIFI_MAX_DATA_REFERENCES];
} card_signal_t;


#define UNIFI_RESERVED_COMMAND_SLOTS   2

/* Considering approx 500 us per packet giving 0.5 secs */
#define UNIFI_PACKETS_INTERVAL         1000

#define CSR_WIFI_HIP_ROUNDUP(val, r) (((val) + (r - 1)) & ~(r - 1))

/* These are type-safe and don't write incorrect values to the
 * structure. */


enum unifi_host_state
{
    UNIFI_HOST_STATE_AWAKE = 0,
    UNIFI_HOST_STATE_DROWSY = 1,
    UNIFI_HOST_STATE_TORPID = 2
};

struct fh_data_d
{
    CsrUint16 slot;
};

#define UNIFI_HOST_MAX_VIFS 8
#define UNIFI_HOST_MAX_PEER 8
#define UNIFI_HOST_MC_OCTETS ((UNIFI_HOST_MAX_VIFS + 15) / 16)
#define UNIFI_HOST_PS_OCTETS ((UNIFI_HOST_MAX_VIFS * UNIFI_HOST_MAX_PEER + 15) / 16)
#define UNIFI_HOST_RT_OCTETS ((UNIFI_HOST_MAX_VIFS + 15) / 16)
#define UNIFI_MAX_FH_BULK_SLOTS 256
#define UNIFI_MAX_TH_BULK_SLOTS 256

typedef struct
{
    CsrUint16 multicasting;
    CsrUint16 power_saving[UNIFI_HOST_PS_OCTETS];
    CsrUint16 uapsd_trigs[UNIFI_HOST_PS_OCTETS];
    CsrUint16 retransmit[UNIFI_HOST_RT_OCTETS];
    CsrUint16 scheduling;
} hip_signals_t;
#define SIZEOF_HIP_SIGNALS(h) (  \
        sizeof(h.multicasting) + \
        sizeof(h.power_saving) + \
        sizeof(h.uapsd_trigs) + \
        sizeof(h.retransmit) + \
        sizeof(h.scheduling)     \
        )

typedef struct
{
    /*
     * flag to allow host to identify contents as valid and
     * different from last read
     */
    CsrUint16 M1;

    /*
     * host handles for to host data
     */
    CsrUint16 th_handles[SDIO_TH_HANDLES];

    /*
     * host handles for pulled from host data
     */
    CsrUint16 fh_pulled_handles[SDIO_TH_PULLED_HANDLES];

    /*
     * amount to read from th_handle
     */
    CsrUint16 th_blocks;

    /*
     * amount of signals & commands in the to host buffer
     */
    CsrUint16 th_ctrl_blocks;

    /*
     * limit for from host pushed control (signals & commands)
     */
    CsrUint16 fh_pipe_limit;

    /*
     * current pointer for from host pushed data
     */
    CsrUint16 fh_pipe_current;

    /*
     * HIP Signalling
     *
     * Station state, multicast trigger, etc.
     */
    CsrUint16     fh_defer_slot[(UNIFI_MAX_FH_BULK_SLOTS + 15) / 16];
    CsrUint16     fh_clear_slot[(UNIFI_MAX_FH_BULK_SLOTS + 15) / 16];
    CsrUint16     th_clear_slot[(UNIFI_MAX_TH_BULK_SLOTS + 15) / 16];
    hip_signals_t hip_signals;

    /*
     * flag to allow host to identify contents as valid and
     * different from last read.
     *
     * M1 == 0     structure invalid
     * M2 == 0     structure invalid
     * M1 != M2    structure invalid
     * M1 == M2    structure valid
     */
    CsrUint16 M2;
} ctrl_buffer_t;

/*
 * The SIZEOF_CTRL_BUFFER() macro exists in favour of the built-in sizeof()
 * because of portability issues between platforms. The other alternative is
 * to pack the structure so that the structure reflects the encoding, but
 * this causes its own problems with the ARM EABI compiler not liking
 * packed structures embedded within unpacked structures. We don't want to
 * pack all structures in the HIP because this approach will generate
 * inefficient code to access the structure elements. Since the size of
 * the structure is only required at HIP start-up, we will use this macro.
 * It is not inefficient in code terms, but it needs to be maintained
 * to reflect the definition of ctrl_buffer_t.
 */
#define SIZEOF_CTRL_BUFFER(c) (             \
        sizeof(c.M1) + \
        sizeof(c.th_handles) + \
        sizeof(c.fh_pulled_handles) + \
        sizeof(c.th_blocks) + \
        sizeof(c.th_ctrl_blocks) + \
        sizeof(c.fh_pipe_limit) + \
        sizeof(c.fh_pipe_current) + \
        sizeof(c.fh_defer_slot) + \
        sizeof(c.fh_clear_slot) + \
        sizeof(c.th_clear_slot) + \
        SIZEOF_HIP_SIGNALS(c.hip_signals) + \
        sizeof(c.M2)                        \
        )

typedef struct
{
    /*
     * total length of buffer
     */
    CsrUint32 buffer_size;

    CsrUint16 fh_current_ctrl_ptr;
    CsrUint16 fh_current_data_ptr;
    CsrUint16 fh_pipe_current;

    /*
     * size of from host pushed signals, HIP commands
     * and buffer header
     */
    CsrUint16 pushed_ctrl_size;

    /*
     * size of bulk data pushed in this transfer
     */
    CsrUint16 pushed_bulk_data_size;

    /*
     * window pointer - incremented for each write
     * during a single tx cycle
     */
    CsrUint32 winp;
    CsrUint32 max_window;
    CsrUint16 pulled_round_bytes;
    CsrUint16 pushed_round_bytes;

    CsrUint16 last_written;
    CsrUint32 segment_size;
    /*
     * pushed_bulk_data is used to store the slot numbers
     * for bulk data sent in a single transmit cycle
     */
    CsrUint16        fh_pushed_bulk_data_count;
    struct fh_data_d pushed_bulk_data[256];

    CsrUint16 handles[SDIO_FH_PUSHED_HANDLES];
    CsrUint16 num_handles;

    struct packet_scheduler packet_scheduler;

    /*
     * current offsets into "buffer"
     */
    CsrUint8 *buffer_ptr;

    /*
     * The buffer itself. The contents of which are written to the fw
     *
     * 15  ...  12 11      ...       1             0
     * ____________________________________________
     * |  0x9     |     reserved     | more data   | 0x00
     * --------------------------------------------
     * |             buffer size                   | 0x02
     * --------------------------------------------
     * |          pushed control size              | 0x04
     * --------------------------------------------
     * |          pushed bulk data size            | 0x06
     * --------------------------------------------
     * |                                           |  .
     * |            pushed fh signals              |  .
     * |                                           |  .
     * --------------------------------------------  fh_pushed_max_ctrl
     * |                                           |  .
     * |             pushed fh data                |  .
     * |                                           |  .
     * --------------------------------------------  fh_pushed_max_data;
     */
    CsrUint8 *buffer;
} fh_buffer_desc_t;

typedef struct
{
    CsrUint32 winp;
    CsrUint16 num_handles;
    CsrUint16 pipe_ptr;
} fh_pulled_buffer_desc_t;

typedef struct
{
    /*
     * total length of buffer
     */
    CsrUint32 buffer_size;

    CsrUint16 th_ptr;
    CsrUint16 th_cur_ptr;
    CsrUint8 *th_data;
    CsrUint8 *buffer;
    CsrUint16 block_ctr;
    CsrUint16 num_handles;

    CsrUint16 ctrl_round_bytes;
    CsrUint16 bulk_round_bytes;
    /*
     * window pointer - incremented for each write
     * during a single tx cycle
     */
    CsrUint32 winp;

    CsrUint16 mc;
} th_buffer_desc_t;

/* Maximum theoretical from-host window is 64K. Actual
   size is platform (chip) dependent. */
#define UNIFI_FH_BUF_SIZE (1024 * 64)
#define UNIFI_TH_BUF_SIZE (1024 * 64)

typedef struct slot_desc
{
    struct slot_desc                     *next;
    CsrUint16                             slot;
    CsrWifiHipBulkDataDesc                bd;
    struct csrWifiHipPacketSchedulerQsig *qsig;
} slot_desc_t;

/*
 * Structure describing a UniFi SDIO card.
 */
struct card
{
    /*
     * Back pointer for the higher level OS code. This is passed as
     * an argument to callbacks (e.g. for received data and indications).
     */
    void *ospriv;

    /* card number in case of multiple cards.  duplicated in ospriv. */
    CsrInt16 instance;

    /* Info read from Symbol Table during probe */
    CsrUint32     build_id;
    CsrCharString build_id_string[128];

    /* Retrieve from SDIO driver. */
    CsrUint16 chip_id;

    /* Read from GBL_CHIP_VERSION. */
    CsrUint16 chip_version;

    /* From the SDIO driver (probably 1) */
    CsrUint8 function;

    /* This is sused to get the register addresses and things. */
    ChipDescript *helper;

    /*
     * Bit mask of PIOs for the loader to waggle during download.
     * We assume these are connected to LEDs. The main firmware gets
     * the mask from a MIB entry.
     */
    CsrInt32 loader_led_mask;

    /* UDI callback for logging UniFi interactions */
    udi_func_t udi_hook;

    CsrUint32 bh_reason_host;
    CsrUint32 bh_reason_unifi;
    CsrUint32 bh_reason_unifi_int;
    CsrUint32 bh_reason_int_wake_up_only;

    /* SDIO clock speed request from OS layer */
    CsrUint8 request_max_clock;

    /* Last SDIO clock frequency set */
    CsrUint32 sdio_clock_speed;

    /*
     * Current host state (copy of value in IOABORT register and
     * spinlock to protect it.
     */
    enum unifi_host_state host_state;
    enum unifi_host_state prev_host_state;

    enum unifi_low_power_mode     low_power_mode;
    enum unifi_periodic_wake_mode periodic_wake_mode;

    CsrUint32 fh_signaldata_pending; /* running byte count */
    CsrUint32 fh_bulkdata_pending;   /* running byte count */

    CsrWifiHipSpinlock fh_count_lock;
    CsrWifiHipSpinlock fh_interrupt_lock;

    /* Should specify buffer size as a number of signals */
    /*
     * Enough for 10 th and 10 fh data slots:
     *   1 * 10 * 8 =  80
     *   2 * 10 * 8 = 160
     */


    ctrl_buffer_t           ctrl_buffer;
    fh_buffer_desc_t        fh_buffer_d;
    th_buffer_desc_t        th_buffer_d;
    fh_pulled_buffer_desc_t fh_pulled_buffer_d;
    CsrUint8               *ctrl_read_buffer;
    CsrUint32               ctrl_read_buffer_len; /* from config struct */
    CsrUint16               Pm;

    CsrUint16 next_vif;

    CsrUint16 last_power_saving[UNIFI_HOST_PS_OCTETS];
    /*
     * Field to use for the incrementing value to write to the UniFi
     * SHARED_IO_INTERRUPT register.
     * Flag to say we need to generate an interrupt at end of processing.
     */
    CsrUint32 unifi_interrupt_seq;
    CsrUint8  generate_interrupt;

    /* Pointers to the bulk data slots */
    slot_desc_t            *from_host_data;
    CsrWifiHipBulkDataDesc *to_host_data;

    /*
     * SDIO specific fields
     */

    /* Interface pointer for the SDIO library */
    CsrSdioFunction *sdio_if;

    /* Copy of config_data struct from the card */
    sdio_config_data_t   config_data;
    CsrWifiHipCardParams config_params;
    CsrUint32            zero_copy;
    CsrUint8            *ack_buffer;
    CsrUint8            *ack_buffer_ptr;
    CsrUint16            acks_written;

    /* The last value written to the Shared Data Memory Page register */
    CsrUint32 proc_select;
    CsrUint32 dmem_page;
    CsrUint32 pmem_page;

    /* SDIO traffic counters limited to 32 bits for Synergy compatibility */
    CsrUint32 sdio_bytes_read;
    CsrUint32 sdio_bytes_written;

    CsrUint8 memory_resources_allocated;

    /* UniFi SDIO I/O Block size. */
    CsrUint16 sdio_io_block_size;

    /* Pad transfer sizes to SDIO block boundaries */
    CsrBool sdio_io_block_pad;

    /* Read from the XBV */
    struct FWOV fwov;

#ifndef CSR_WIFI_HIP_TA_DISABLE
    /* TA sampling */
    ta_data_t ta_sampling;
#endif

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE_ENABLE
    struct cmd_profile
    {
        CsrUint32 cmd52_count;
        CsrUint32 cmd53_count;
        CsrUint32 tx_count;
        CsrUint32 tx_cfm_count;
        CsrUint32 rx_count;
        CsrUint32 bh_count;
        CsrUint32 process_count;
        CsrUint32 protocol_count;

        CsrUint32 cmd52_f0_r_count;
        CsrUint32 cmd52_f0_w_count;
        CsrUint32 cmd52_r8or16_count;
        CsrUint32 cmd52_w8or16_count;
        CsrUint32 cmd52_r16_count;
        CsrUint32 cmd52_w16_count;
        CsrUint32 cmd52_r32_count;
    } hip_prof;
    struct cmd_profile cmd_prof;
#endif

    /* Interrupt processing mode flags */
    CsrUint32 intmode;

#ifdef UNIFI_DEBUG
    CsrUint8 lsb;
#endif

    /* Historic firmware panic codes */
    CsrUint32 panic_data_phy_addr;
    CsrUint32 panic_data_mac_addr;
    CsrUint16 last_phy_panic_code;
    CsrUint16 last_phy_panic_arg;
    CsrUint16 last_mac_panic_code;
    CsrUint16 last_mac_panic_arg;

    CsrWifiHipHip2Stats hip2Stats;
    slot_desc_t        *freelist;
    CsrUint16           slot_desc_list_freecount;
#ifdef CSR_WIFI_MAKE_FAKE_CMD53_ERRORS
    CsrInt16 fake_error;
#endif
    CsrUint16 rewind_seq;
    /*
    * pending_window_update is set when the "more" flag has been
    * set in the from-host header. The driver must wait for the
    * f/w to signal that more from-host (tx) window is available
    * before anymore sdio writes are committed.
    */
    CsrUint16 pending_window_update;
    /* pending_ctrl_buffer is informational only. It is set when
    * an invalid control buffer is read from unifi and cleared
    * when a valid control buffer is subsequently read.
    */
    CsrUint16 pending_ctrl_buffer;
    /*
    * ack_outstanding is set when to-host (rx) data has been read
    * but no from-host (tx) data has been written. The flag
    * indicates that an "ack" value is written to the ack buffer.
    * If from-host data is written, the f/w accepts the from-host
    * data as an implicit ack for the to-host read.
    */
    CsrUint16 ack_outstanding;

    CsrUint16 tx_pkts_per_cycle;
    CsrUint16 bulk_to_read;
    CsrUint8  pad_buf[128];

    /* This is the result of HIP2 config struct buf_maximum_size_blocks converted
     * up to bytes by multiplying it by block_round_size.
     */
    CsrUint32 buff_maximum_size_bytes;   /* buf_maximum_size_blocks in bytes */
    CsrUint16 packetFilter;
}; /* struct card */


/* Reset types */
enum unifi_reset_type
{
    UNIFI_COLD_RESET = 1,
    UNIFI_WARM_RESET = 2
};

/*
 * unifi_set_host_state() implements signalling for waking UniFi from
 * deep sleep. The host indicates to UniFi that it is in one of three states:
 *   Torpid - host has nothing to send, UniFi can go to sleep.
 *   Drowsy - host has data to send to UniFi. UniFi will respond with an
 *            SDIO interrupt. When hosts responds it moves to Awake.
 *   Awake  - host has data to transfer, UniFi must stay awake.
 *            When host has finished, it moves to Torpid.
 */
CsrResult unifi_set_host_state(card_t *card, enum unifi_host_state state);


CsrResult unifi_set_proc_select(card_t *card, enum unifi_dbg_processors_select select);
CsrInt32 card_read_signal_counts(card_t *card);
CsrWifiHipBulkDataDesc *card_find_data_slot(card_t *card, CsrInt16 slot);


CsrResult unifi_read32(card_t *card, CsrUint32 unifi_addr, CsrUint32 *pdata);
CsrResult unifi_readnz(card_t *card, CsrUint32 unifi_addr,
                       void *pdata, CsrUint16 len);
CsrInt32 unifi_read_shared_count(card_t *card, CsrUint32 addr);

CsrResult unifi_writen(card_t *card, CsrUint32 unifi_addr, void *pdata, CsrUint16 len);

CsrResult unifi_bulk_rw(card_t *card, CsrUint32 handle,
                        void *pdata, CsrUint32 len, CsrUint32 direction);
CsrResult unifi_bulk_rw_noretry(card_t *card, CsrUint32 handle,
                                void *pdata, CsrUint32 len, CsrUint32 direction);
#define UNIFI_SDIO_READ                   0
#define UNIFI_SDIO_READ_CONFIG            1
#define UNIFI_SDIO_WRITE_CONFIG           2
#define UNIFI_SDIO_READ_CONTROL           3
#define UNIFI_SDIO_WRITE_ACK              4
#define UNIFI_SDIO_READ_TO_HOST           5
#define UNIFI_SDIO_WRITE_FROM_HOST_PUSHED 6
#define UNIFI_SDIO_WRITE_FROM_HOST_PULLED 7
#define UNIFI_SDIO_WRITE                  8

CsrResult unifi_read_8_or_16(card_t *card, CsrUint32 unifi_addr, CsrUint8 *pdata);
CsrResult unifi_write_8_or_16(card_t *card, CsrUint32 unifi_addr, CsrUint8 data);
CsrResult unifi_read_direct_8_or_16(card_t *card, CsrUint32 addr, CsrUint8 *pdata);
CsrResult unifi_write_direct_8_or_16(card_t *card, CsrUint32 addr, CsrUint8 data);

CsrResult unifi_read_direct16(card_t *card, CsrUint32 addr, CsrUint16 *pdata);
CsrResult unifi_read_direct32(card_t *card, CsrUint32 addr, CsrUint32 *pdata);
CsrResult unifi_read_directn(card_t *card, CsrUint32 addr, void *pdata, CsrUint16 len);

CsrResult unifi_write_direct16(card_t *card, CsrUint32 addr, CsrUint16 data);
CsrResult unifi_write_directn(card_t *card, CsrUint32 addr, void *pdata, CsrUint16 len);

CsrResult sdio_read_f0(card_t *card, CsrUint32 addr, CsrUint8 *pdata);
CsrResult sdio_write_f0(card_t *card, CsrUint32 addr, CsrUint8 data);

void unifi_read_panic(card_t *card);

#ifdef __cplusplus
}
#endif

#endif /* __CARD_SDIO_H__ */
