/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_hal_priv.h"

#ifdef CSR_WIFI_DRIVER_HYDRA
#define CSR_WIFI_HIP_MAX_INIT_ATTEMPTS 1
#else
#define CSR_WIFI_HIP_MAX_INIT_ATTEMPTS 4
#endif

/*
 * ---------------------------------------------------------------------------
 *  uf_init_hw
 *
 *      Resets hardware, downloads and initialises f/w.
 *      This function will retry to initialise UniFi if unifi_init_card()
 *      returns an error.
 *
 *  Arguments:
 *      priv          Pointer to OS driver structure for the device.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, a CSR error code on failure
 *
 * ---------------------------------------------------------------------------
 */
CsrResult uf_init_hw(CsrWifiHipHalPriv *priv)
{
    CsrInt32 attempts = 0;

    CsrResult r = CSR_RESULT_FAILURE;

    for ( ; ; )
    {
        if (attempts > CSR_WIFI_HIP_MAX_INIT_ATTEMPTS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_init_hw: Failed to initialise UniFi after %d attempts, " "giving up.\n",
                                   priv ? priv->instance : 0,  attempts));
            break;
        }
        attempts++;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,
                            CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_init_hw: Initialising UniFi, attempt %d\n",
                            priv ? priv->instance : 0,  attempts));

        if (priv->fw_init > 0)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_init_hw: f/w init prevented by module parameter\n", priv ? priv->instance : 0));
            break;
        }
        else if (priv->fw_init == 0)
        {
            priv->fw_init++;
        }

        /*
         * Initialise driver core. This will perform a reset of UniFi
         * internals, but not the SDIO CCCR.
         */
        CsrSdioClaim(priv->sdio);
        r = unifi_init_card(priv->card, &priv->cardParamsCache);
        CsrSdioRelease(priv->sdio);

        if (r == CSR_WIFI_HIP_RESULT_NO_DEVICE)
        {
            break;  /* Card removed. Fatal */
        }
        else if (r == CSR_WIFI_HIP_RESULT_INVALID_VALUE)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_init_hw: Firmware file required, but not found or incorrect.\n", priv ? priv->instance : 0));
            break;  /* Fatal */
        }
        else if (r != CSR_RESULT_SUCCESS)
        {
            /* Retryable failure. Reset h/w and try again */
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: uf_init_hw: Failed to initialise UniFi chip.\n", priv ? priv->instance : 0));
            continue;
        }

        /* Get the version information from the core */
        unifi_card_info(priv->card, &priv->card_info);

        return r;
    }

    return r;
}
