/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_unifi_udi.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_util.h"
#include "csr_wifi_firmware_patch.h"
#include "csr_pmem.h"
#include "csr_wifi_hip.h"

/*
 * ---------------------------------------------------------------------------
 *
 *      F/W download. Part of the HIP core API
 *
 * ---------------------------------------------------------------------------
 */


/*
 * ---------------------------------------------------------------------------
 *  unifi_fw_read_start
 *
 *      Returns a structure to be passed in unifi_fw_read().
 *      This structure is an OS specific description of the f/w file.
 *      In the linux implementation it is a buffer with the f/w and its' length.
 *      The HIP driver calls this functions to request for the loader or
 *      the firmware file.
 *      The structure pointer can be freed when unifi_fw_read_stop() is called.
 *
 *  Arguments:
 *      ospriv          Pointer to driver context.
 *      is_fw           Flag to indicate whether it is the f/w or the loader
 *      info            Versions information. Can be used to determine
 *                      the appropriate f/w file to load.
 *
 *  Returns:
 *      Pointer to the structure holding the firmware or NULL if no firmware
 *      was found.
 *
 * ---------------------------------------------------------------------------
 */
void *unifi_fw_read_start(void *ospriv, CsrInt8 is_fw, const card_info_t *info)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;

    func_enter();

    if (is_fw == UNIFI_FW_STA)
    {
        CsrSize length = 0;
        CsrUint8 *firmware = NULL;
        CsrInt32 r;

        /* The firmware will be provided via the raw sdio API; hence, do not
           use the firmware patch acquire functionality */
        if (priv->hipMode == CSR_WIFI_HIP_RAW_SDIO_MODE)
        {
            return &priv->fw_sta;
        }

        if (priv->haveRequestedFirmware == FALSE)
        {
            priv->haveRequestedFirmware = TRUE;
            r = CsrWifiFirmwarePatchAcquire(priv->osLayerContext, info->fw_build, &length, &firmware);
            if (r)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                    "unifi%d: No patch present for ROM %d (0x%02X)\n",
                                    priv->instance, info->fw_build, r));

                priv->fw_sta.dl_data = NULL;
                priv->fw_sta.dl_len = 0;
                priv->fw_sta.fw_desc = NULL;

                func_exit();
                return NULL;
            }

            priv->fw_sta.dl_data = firmware;
            priv->fw_sta.dl_len = (CsrUint32) length;
            priv->fw_sta.fw_desc = NULL;
        }

        func_exit();
        return &priv->fw_sta;
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: downloading firmware... unknown request: %d\n",
                               priv->instance, is_fw));
    }

    func_exit();
    return NULL;
} /* unifi_fw_read_start() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_fw_read_stop
 *
 *      Called when the HIP driver has finished using the loader or
 *      the firmware file.
 *      The firmware buffer may be released now.
 *
 *  Arguments:
 *      ospriv          Pointer to driver context.
 *      dlpriv          The pointer returned by unifi_fw_read_start()
 *
 * ---------------------------------------------------------------------------
 */
void unifi_fw_read_stop(void *ospriv, void *dlpriv)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;
    struct dlpriv *tmp = (struct dlpriv *) dlpriv;

    if (priv->hipMode == CSR_WIFI_HIP_RAW_SDIO_MODE)
    {
        return;
    }

    priv->haveRequestedFirmware = FALSE;

    if (tmp->dl_len)
    {
        CsrPmemFree((CsrUint8 *) tmp->dl_data);
        tmp->dl_data = NULL;
        tmp->dl_len = 0;
        tmp->fw_desc = NULL;
    }
} /* unifi_fw_read_stop() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_fw_open_buffer
 *
 *  Returns a handle for a buffer dynamically allocated by the driver,
 *  e.g. into which a firmware file may have been converted from another format
 *  which is the case with some production test images.
 *
 *  The handle may then be used by unifi_fw_read() to access the contents of
 *  the buffer.
 *
 *  Arguments:
 *      ospriv          Pointer to driver context.
 *      fwbuf           Buffer containing firmware image
 *      len             Length of buffer in bytes
 *
 *  Returns
 *      Handle for buffer, or NULL on error
 * ---------------------------------------------------------------------------
 */
void *unifi_fw_open_buffer(void *ospriv, void *fwbuf, CsrUint32 len)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) ospriv;
    func_enter();

    if (fwbuf == NULL)
    {
        func_exit();
        return NULL;
    }

    priv->fw_conv.dl_data = fwbuf;
    priv->fw_conv.dl_len = len;
    priv->fw_conv.fw_desc = NULL;   /* No OS f/w resource is associated */

    func_exit();
    return &priv->fw_conv;
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_fw_close_buffer
 *
 *  Releases any handle for a buffer dynamically allocated by the driver,
 *  e.g. into which a firmware file may have been converted from another format
 *  which is the case with some production test images.
 *
 *
 *  Arguments:
 *      ospriv          Pointer to driver context.
 *      fwbuf           Buffer containing firmware image
 *
 *  Returns
 *      Handle for buffer, or NULL on error
 * ---------------------------------------------------------------------------
 */
void unifi_fw_close_buffer(void *ospriv, void *fwbuf)
{
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_fw_read
 *
 *      The HIP driver calls this function to ask for a part of the loader or
 *      the firmware file.
 *
 *  Arguments:
 *      ospriv          Pointer to driver context.
 *      arg             The pointer returned by unifi_fw_read_start().
 *      offset          The offset in the file to return from.
 *      buf             A buffer to store the requested data.
 *      len             The size of the buf and the size of the requested data.
 *
 *  Returns
 *      The number of bytes read from the firmware image, or -ve on error
 * ---------------------------------------------------------------------------
 */
CsrInt32 unifi_fw_read(void *ospriv, void *arg, CsrUint32 offset, void *buf, CsrUint32 len)
{
    const struct dlpriv *dlpriv = arg;

    if (offset >= dlpriv->dl_len)
    {
        /* at end of file */
        return 0;
    }

    if ((offset + len) > dlpriv->dl_len)
    {
        /* attempt to read past end of file */
        return -1;
    }

    CsrMemCpy(buf, dlpriv->dl_data + offset, len);

    return len; /*lint -e713 */
} /* unifi_fw_read() */
