/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_hal_priv.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_unifiversion.h"
#include "csr_sched.h"
#include "csr_wifi_hip.h"
#include "csr_pmem.h"
#include "csr_types.h"
#include "csr_framework_ext.h"
#include "csr_wifi_hip_io.h"
#include "csr_wifi_hip_card_sdio.h"
#include "csr_wifi_hip_card.h"
#include "csr_wifi_hip_util.h"
#include "csr_wifi_ps_if.h"

static void bhErrorHandler(CsrWifiHipHalPriv *priv);

#define CSR_WIFI_HIP_BH_THREAD_STACK_SIZE 1024

/* Map configured HIP thread priority to defines in CSR Synergy Framework Extensions */
#if CSR_WIFI_HIP_PRIORITY == 0
#define THREAD_PRIORITY CSR_THREAD_PRIORITY_HIGHEST
#elif CSR_WIFI_HIP_PRIORITY == 1
#define THREAD_PRIORITY CSR_THREAD_PRIORITY_HIGH
#elif CSR_WIFI_HIP_PRIORITY == 2
#define THREAD_PRIORITY CSR_THREAD_PRIORITY_NORMAL
#elif CSR_WIFI_HIP_PRIORITY == 3
#define THREAD_PRIORITY CSR_THREAD_PRIORITY_LOW
#elif CSR_WIFI_HIP_PRIORITY == 4
#define THREAD_PRIORITY CSR_THREAD_PRIORITY_LOWEST
#endif

#define UNIFI_DEFAULT_WAKE_TIMEOUT      1000

static CsrResult initHipSubtask(CsrWifiHipHalPriv *priv)
{
    CsrResult ret;
    CsrResult result = CSR_RESULT_SUCCESS;
    CsrUint16 i;
    CsrWifiHipVifInstance *vif = NULL;

    /* Initialize vifIndexMapping members */
    CsrMemSet(priv->primaryVifIndexMap, 0, (CSR_WIFI_MAX_INTERFACES * sizeof(CsrUint16)));
    CsrMemSet(priv->interfaceTagMap, 0xFF, (CSR_WIFI_HIP_MAX_VIFINDEX * sizeof(CsrUint16)));
    priv->bh_block_io_thread = 0;

    CsrMemSet(priv->debugInfo.last_debug_string, 0, CSR_WIFI_HIP_LAST_DEBUG_STRING_LENGTH);
    CsrMemSet(priv->debugInfo.last_debug_word16, 0, CSR_WIFI_HIP_LAST_DEBUG_WORD16_LENGTH);

    /* Initialise hardware */
    ret = uf_init_hw(priv);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: HIP init: Initialisation of the UniFi chip failed (error code: %u) \n",
                               priv ? priv->instance : 0,  ret));

        result = CSR_RESULT_FAILURE;
        goto exit;
    }

    /* Ensure that firmware is re-requested for each WifiOnReq */
    priv->haveRequestedFirmware = FALSE;

    /* Initialise controlled port and lists */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        vif = (CsrWifiHipVifInstance *) &priv->vif[i];

        vif->interfaceTag = i;
        vif->priv = (void *) priv;

        /* Enable all queues by default */
        vif->staInfo.queueEnabled[0] = 1;
        vif->staInfo.queueEnabled[1] = 1;
        vif->staInfo.queueEnabled[2] = 1;
        vif->staInfo.queueEnabled[3] = 1;

        CsrMemSet(vif->staInfo.sta_mac_addresses.a, 0, ETH_ALEN);

        CsrMemSet(&vif->staInfo.controlled_data_port, 0, sizeof(unifi_port_config_t));
        vif->staInfo.controlled_data_port.entries_in_use = 1;
        vif->staInfo.controlled_data_port.overide_action = UF_DATA_PORT_OVERIDE;
        vif->staInfo.controlled_data_port.port_cfg[0].port_action = CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
        vif->staInfo.controlled_data_port.port_cfg[0].in_use = TRUE;

        CsrMemSet(&vif->staInfo.uncontrolled_data_port, 0, sizeof(unifi_port_config_t));
        vif->staInfo.uncontrolled_data_port.entries_in_use = 1;
        vif->staInfo.uncontrolled_data_port.overide_action = UF_DATA_PORT_OVERIDE;
        vif->staInfo.uncontrolled_data_port.port_cfg[0].port_action = CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
        vif->staInfo.uncontrolled_data_port.port_cfg[0].in_use = TRUE;

        vif->numOfControlledFrames = 0;
        vif->numOfUncontrolledFrames = 0;
        CSR_WIFI_HIP_LIST_INIT(&vif->controlledPortQueue);
        CSR_WIFI_HIP_LIST_INIT(&vif->uncontrolledPortQueue);

        {
            CsrUint8 j;

            for (j = 0; j < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; j++)
            {
                csrWifiHipApStaRecordInitialise(priv, &vif->staPeerInfo[j]);
            }

            for (j = 1; j < CSR_WIFI_HIP_PEER_CONNECTIONS_MAX; j++)
            {
                vif->staInfo.controlled_data_port.port_cfg[j].port_action =
                    CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
                vif->staInfo.uncontrolled_data_port.port_cfg[j].port_action =
                    CSR_WIFI_ROUTER_CTRL_PORT_ACTION_8021X_PORT_CLOSED_DISCARD;
                vif->staInfo.controlled_data_port.port_cfg[j].in_use = FALSE;
                vif->staInfo.uncontrolled_data_port.port_cfg[j].in_use = FALSE;
            }
        }
    }

    /* Enable sdio function */
    CsrSdioClaim(priv->sdio);
    ret = CsrSdioFunctionEnable(priv->sdio);
    CsrSdioRelease(priv->sdio);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: HIP init: Could not enable SDIO function (error code: %u) \n",
                               priv ? priv->instance : 0,  ret));

        result = CSR_RESULT_FAILURE;
        goto exit;
    }


    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2, "unifi%d: HIP init: Successful \n",
                       priv ? priv->instance : 0));
    result = CSR_RESULT_SUCCESS;
    goto exit;


exit:
    return result;
}

static CsrResult initHip(CsrWifiHipHalPriv *priv)
{
    CsrResult ret;

    /* Create BH event group */
    ret = CsrEventCreate(&priv->bhEventHandle);
    if (ret == CSR_RESULT_SUCCESS)
    {
        /* Create Wi-Fi HIP spinlocks */
        ret = CSR_WIFI_HIP_SPINLOCK_CREATE(&priv->peerInfoLock);
        if (ret == CSR_RESULT_SUCCESS)
        {
            ret = CSR_WIFI_HIP_SPINLOCK_CREATE(&priv->vifMapLock);
            if (ret == CSR_RESULT_SUCCESS)
            {
                ret = CsrMutexCreate(&priv->configurationMutex);
                if (ret == CSR_RESULT_SUCCESS)
                {
                    ret = initHipSubtask(priv);
                    if (ret != CSR_RESULT_SUCCESS)
                    {
                        CsrMutexDestroy(&priv->configurationMutex);
                    }
                }
                else
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: initHip: Could not create configuration mutex (error code: 0x%02x) \n",
                                           priv ? priv->instance : 0,  ret));
                }
                if (ret != CSR_RESULT_SUCCESS)
                {
                    CSR_WIFI_HIP_SPINLOCK_DESTROY(&priv->vifMapLock);
                }
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: initHip: Could not create vifMapLock spinlock (error code: 0x%02x) \n",
                                       priv ? priv->instance : 0,  ret));
            }
            if (ret != CSR_RESULT_SUCCESS)
            {
                CSR_WIFI_HIP_SPINLOCK_DESTROY(&priv->peerInfoLock);
            }
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: initHip: Could not create peerInfoLock spinlock (error code: 0x%02x) \n",
                                   priv ? priv->instance : 0,  ret));
        }
        if (ret != CSR_RESULT_SUCCESS)
        {
            CsrEventDestroy(&priv->bhEventHandle);
        }
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: initHip: HIP failed to create event group (error code: 0x%02x)\n",
                               priv ? priv->instance : 0,  ret));
    }
    return ret == CSR_RESULT_SUCCESS ? CSR_RESULT_SUCCESS : CSR_RESULT_FAILURE;
}

#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
/*
 * ---------------------------------------------------------------------------
 * HipSuspendRes has been received, which means we can put unifi in low power mode and go to sleep
 * This function is called by bh thread
 * This means there is no need for locks on the hardware
 * ---------------------------------------------------------------------------
 */
static void suspendResp(CsrWifiHipHalPriv *priv)
{
    /* Set flag to signal that host has been suspended before low power mode */
    priv->suspend = TRUE;

    /* Ensure that we are not in Raw SDIO mode */
    if (priv->hipMode == CSR_WIFI_HIP_WIFI_STACK_MODE)
    {
#ifdef CSR_WIFI_HIP_SUSPEND_WOL
        CsrResult r;
#endif

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Received suspend resp in bh thread", priv->instance));

#ifdef CSR_WIFI_HIP_SUSPEND_WOL
        CsrSdioClaim(priv->sdio);
        r = unifi_force_low_power_mode(priv->card);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: suspendResp: Failed to force low power mode - continue suspend anyway\n", priv->instance));
        }

#ifdef CSR_WIFI_HIP_SUSPEND_WAKE_PIO
        /* For PIO WOL, disable SDIO interrupt */
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: suspendResp: Remove IRQ to enable PIO WOL", priv->instance));

        CsrSdioClaim(priv->sdio);
        r = CsrSdioInterruptDisable(priv->sdio);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: suspendResp: Could not disable interrupt - continue WOL", priv->instance));
        }
#endif
#endif
    }
    CsrSdioSuspendAcknowledge(priv->sdio, CSR_RESULT_SUCCESS);
}

static void suspendInd(CsrWifiHipHalPriv *priv)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Received Suspend indication", priv->instance));

#ifdef CSR_WIFI_HIP_SUSPEND_WOL
    CsrWifiHipSuspendInd(priv->osLayerContext, 0, FALSE, TRUE);
#else
    CsrWifiHipSuspendInd(priv->osLayerContext, 0, TRUE, FALSE);
#endif
}

static void resumeInd(CsrWifiHipHalPriv *priv)
{
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1, "unifi%d: Received Resume indication", priv->instance));
#ifdef CSR_WIFI_HIP_SUSPEND_WOL
#ifdef CSR_WIFI_HIP_SUSPEND_WAKE_PIO
    /* For PIO WOL, re-enable SDIO interrupt */
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: bh resume: Re-enable SDIO IRQ", priv->instance));

    CsrSdioClaim(priv->sdio);
    if (CsrSdioInterruptEnable(priv->sdio) != CSR_RESULT_SUCCESS)
    {
        CsrSdioRelease(priv->sdio);
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: BH could not enable SDIO interrupt - continue WOL", priv->instance));
    }
    else
    {
        CsrSdioRelease(priv->sdio);
    }
#endif
    CsrWifiHipResumeInd(priv->osLayerContext, 0, TRUE);
#else
    CsrWifiHipResumeInd(priv->osLayerContext, 0, FALSE);
#endif
}

#endif /* CSR_WIFI_HIP_SUSPEND_ENABLE */

/*
 * ---------------------------------------------------------------------------
 *  CsrWifiHipBhThreadFunction
 *
 *      All hardware access happens in this thread.
 *      This means there is no need for locks on the hardware and we don't need
 *      to worry about reentrancy with the SDIO library.
 *
 *  Arguments:
 *      arg             Pointer to OS driver structure for the device.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 *      When the bottom half of the driver needs to process signals, events,
 *      or simply the host status (i.e sleep mode), it invokes unifi_run_bh().
 *      Since we need all SDIO transaction to be in a single thread, the
 *      unifi_run_bh() will wake up this thread to process it.
 *
 * ---------------------------------------------------------------------------
 */
void CsrWifiHipBhThreadFunction(void *arg)
{
    CsrWifiHipHalPriv *priv = (CsrWifiHipHalPriv *) arg;
    CsrResult r = CSR_RESULT_SUCCESS;
    CsrUint32 timeout;
    CsrResult result = CSR_RESULT_FAILURE;
    CsrResult hipInitResult;
    CsrUint32 chipId = 0, chipVersion = 0, firmwareBuild = 0, firmwareHip = 0, driverHip = 0;
    CsrCharString *driverBuild = NULL;
    CsrBool bh_errored = FALSE;
    CsrUint16 i;

    /* Loop handling */
    CsrUint32 bhEventBits;
    CsrBool timedEventLoop = FALSE;
    CsrUint16 timeoutInMs = 0;
    CsrTime timeStartWait = 0;

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: CsrWifiHipBhThreadFunction: started priv = 0x%p\n",
                       priv->instance, priv));

    CsrWifiHipPacketSchedulerInit(priv->card);

    /* Do HIP and chip initialisation and send status to SME */
    hipInitResult = initHip(priv);
    if (hipInitResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: CsrWifiHipBhThreadFunction: Initialisation of HIP and chip failed (%d)!\n",
                           priv->instance, hipInitResult));

        priv->eventBits = 0;
        CsrEventDestroy(&priv->eventHandle);

#ifndef CSR_WIFI_DRIVER_HYDRA
        CsrSdioClaim(priv->sdio);
        CsrSdioPowerOff(priv->sdio);
        CsrSdioRelease(priv->sdio);
#endif
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: CsrWifiHipBhThreadFunction: Bh thread signals that it has terminated\n",
                            priv->instance));

        CsrWifiHipPacketSchedulerDestroy(priv->card);

        priv->bhThreadStarted = 0;
        CsrWifiHipWifiOnCfm(priv->osLayerContext, hipInitResult);
        return;
    }

    /* Prepare information which will be send to SME */
    chipId = priv->card_info.chip_id;
    chipVersion = priv->card_info.chip_version;
    firmwareBuild = priv->card_info.fw_build;
    firmwareHip = priv->card_info.fw_hip_version;
    driverBuild = (CsrCharString *) CSR_WIFI_VERSION;
    driverHip = (UNIFI_HIP_MAJOR_VERSION << 8) | UNIFI_HIP_MINOR_VERSION;


    /*
     * Install interrupt handler.
     */
    r = CSR_WIFI_HIP_LINUX_INSTALL_IRQ(priv->sdio);
    if (r != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Installing IRQ failed 0x%02x\n", priv->instance, r));
    }
    /*
     * Enable the SDIO interrupts now that the f/w is running.
     */
    CsrSdioClaim(priv->sdio);
    r = CsrSdioInterruptEnable(priv->sdio);
    CsrSdioRelease(priv->sdio);
    if (r == CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Enabling Unifi SDIO interrupt\n", priv->instance));
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: Enabling IRQ failed\n", priv->instance));
    }

    /* Clear variables - we are entering bh's while loop */
    result = CSR_RESULT_SUCCESS;
    timeout = 0;
    bhEventBits = 0;

    CsrWifiHipWifiOnInd(priv->osLayerContext,
                        hipInitResult,
                        chipId,
                        chipVersion,
                        firmwareBuild,
                        firmwareHip,
                        driverBuild,
                        driverHip,
                        priv->card_info.sdio_block_size);

    priv->wifiOnState = CSR_WIFI_HIP_WIFI_ON_DONE;

    for ( ; ; )
    {
        /* If a wait with timeout is required, prepare the timeout here */
        if (timeout > 0)
        {
            timeStartWait = (CsrTimeGet(NULL) / 1000);
            timeoutInMs = (CsrUint16) timeout;

            /* Ensure that we don't by mistake waits for ever */
            if (timeoutInMs == CSR_EVENT_WAIT_INFINITE)
            {
                timeoutInMs--;
            }

            timedEventLoop = TRUE;
        }
        else
        {
            timeoutInMs = CSR_EVENT_WAIT_INFINITE;
            timedEventLoop = FALSE;
        }


        /* Check if more event bits were set */
        if (bhEventBits == 0)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG7,
                                "unifi%d: bh_thread goes to sleep, %s\n",
                                priv->instance, timedEventLoop ? "waiting on timeout or event" : "waiting on event"));
            result = CsrEventWait(&priv->bhEventHandle, timeoutInMs, &bhEventBits);
        }

        if (result == CSR_RESULT_SUCCESS)
        {
            if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_RUN)
            {
                if (timedEventLoop)
                {
                    CsrTime timeFinishWait;
                    CsrTime timeDifference;

                    timeFinishWait = (CsrTimeGet(NULL) / 1000);

                    timeDifference = timeFinishWait - timeStartWait;
                    if (timeDifference <= timeFinishWait)
                    {
                        /* Calculate the remaining time */
                        if ((timeout - timeDifference) > timeout)
                        {
                            timeout = 0;
                        }
                        else
                        {
                            timeout -= timeDifference;
                        }
                    }
                    else
                    {
                        /* If CsrTimeGet wraps use default timeout instead. */
                        timeout = UNIFI_DEFAULT_WAKE_TIMEOUT;
                    }
                }

                /* Clear event */
                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_BH_RUN;

                /* Run BH to process event */
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG7,
                                    "unifi%d: bh_thread calls unifi_bh()\n", priv->instance));
                CsrSdioClaim(priv->sdio);
                r = unifi_bh(priv->card, &timeout);
                if (r != CSR_RESULT_SUCCESS)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: unifi_bh (signalled) error 0x%02x\n", priv->instance, r));
                    bh_errored = TRUE;
                    bhErrorHandler(priv);
                    CsrSdioRelease(priv->sdio);
                    break;
                }
                CsrSdioRelease(priv->sdio);
            }
#ifdef CSR_WIFI_HIP_SUSPEND_ENABLE
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_RESUME)
            {
                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_BH_RESUME;
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                    "unifi%d: Received resume event\n", priv->instance));

                if (priv->resumeIrq)
                {
                    /* Handle an interrupt that arrived during suspend */
                    unifi_sdio_interrupt_handler(priv->card);
                    priv->resumeIrq = FALSE;
                }
                CsrWifiHipPacketSchedulerResume(priv->card);
                resumeInd(priv);
            }
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_SUSPEND)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                                    "unifi%d: Received suspend event\n", priv->instance));

                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_BH_SUSPEND;
                suspendInd(priv);
            }
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_SUS_RESP)
            {
                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_BH_SUS_RESP;
                suspendResp(priv);
            }
#endif
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_STA_INACT)
            {
                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_STA_INACT;
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                   "unifi%d: bh_thread: Received STA inactivity expiration\n", priv->instance));
                csrWifiHipTimerInactivityHandler(priv);
            }
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BA_TIMEOUT)
            {
                bhEventBits &= ~CSR_WIFI_HIP_EVENT_MASK_BA_TIMEOUT;
                csrWifiHipBaAgingTimeoutHandler(priv);
            }
            else if (bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE)
            {
                priv->bh_thread_terminate = 1;
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                   "unifi%d: bh_thread signalled to exit\n", priv->instance));
                break;
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrEventWait BH returned unhandled bit mask (bit mask: 0x%08X)\n",
                                       priv->instance, bhEventBits));
                bhEventBits = 0;
            }
        }
        else if (result == CSR_FE_RESULT_TIMEOUT)
        {
            bhEventBits = 0;

            if (timedEventLoop)
            {
                timeout = 0;

                /* Run BH to process event */
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG7,
                                    "unifi%d: bh_thread calling unifi_bh()\n", priv->instance));
                CsrSdioClaim(priv->sdio);
                r = unifi_bh(priv->card, &timeout);
                if (r != CSR_RESULT_SUCCESS)
                {
                    bh_errored = TRUE;
                    bhErrorHandler(priv);
                    CsrSdioRelease(priv->sdio);
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                           "unifi%d: unifi_bh (timer) error 0x%02x\n", priv->instance, r));
                    break;
                }
                CsrSdioRelease(priv->sdio);
            }
            else
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrEventWait returned TIMEOUT in infinite CsrEventWait. Framework extensions porting error!\n",
                                       priv->instance));
            }
        }
        else if ((result == CSR_FE_RESULT_INVALID_HANDLE) || (result == CSR_FE_RESULT_INVALID_POINTER))
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrEventWait in BH thread returned error code 0x%02x. BH stops. SME has been informed\n",
                                   priv->instance, result));
            bhErrorHandler(priv);
            break;
        }
        else
        {
            bhEventBits = 0;
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrEventWait in BH thread returned error code 0x%02x. BH thread continues\n",
                                   priv->instance, result));
            if (timedEventLoop)
            {
                timeout = UNIFI_DEFAULT_WAKE_TIMEOUT;
            }
        }
    } /* for loop */


    /* If we haven't received CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE event yet we need to wait for it now */
    if (!(bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE))
    {
        while ((bhEventBits & CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE) !=
               CSR_WIFI_HIP_EVENT_MASK_BH_CLOSE)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                "unifi%d: CsrWifiHipBhThreadFunction: Await BH Close event\n",
                                priv->instance));

            result = CsrEventWait(&priv->bhEventHandle, CSR_EVENT_WAIT_INFINITE, &bhEventBits);
            if (result != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi%d: CsrWaitEvent returned error code %u while waiting for signal to exit \n",
                                       priv->instance, result));
                break;
            }
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                                "unifi%d: CsrWifiHipBhThreadFunction: Got BH Close event 0x%x\n",
                                priv->instance, bhEventBits));
        }
    }

    /* If the BH thread didn't report an error, try to capture panic code
     * for the case where the SME requested shutdown due to a system
     * error (e.g. a timeout).
     */
    if (!bh_errored)
    {
        CsrSdioClaim(priv->sdio);
        (void) unifi_capture_panic(priv->card);
        CsrSdioRelease(priv->sdio);
    }


    /* Decide if the chip shall be powered off during deinit */
    if (csrWifiHipPowerChipOff(priv))
    {
        /* Try to put Unifi into low power mode. */
        CsrSdioClaim(priv->sdio);
        r = unifi_force_low_power_mode(priv->card);
        CsrSdioRelease(priv->sdio);
        if (r != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                               "unifi%d: Failed to force Unifi into low power mode, error code %d\n",
                               priv->instance, r));
        }
    }
    else
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: Chip left powered\n",
                           priv->instance));
    }

    /*
     * Interrupts might be enabled here, (unifi_bh enables them)
     * so we need to disable interrupts before return.
     */
    CsrSdioClaim(priv->sdio);
    (void) CsrSdioInterruptDisable(priv->sdio);
#ifndef CSR_WIFI_DRIVER_HYDRA
    /* Don't disable function when chip remains powered, to preserve context */
    if (csrWifiHipPowerChipOff(priv))
#endif
    {
        (void) CsrSdioFunctionDisable(priv->sdio);
    }

    CsrSdioRelease(priv->sdio);

    /* Free VIFs and Peer records if present */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: CsrWifiHipBhThreadFunction: Trying to reset VIF %u\n",
                            priv->instance, i + 1));
        CsrMemSet(priv->vif[i].staInfo.sta_mac_addresses.a, 0, ETH_ALEN);
#ifdef CSR_NATIVE_LINUX
        csrWifiHipResetVif(priv, &priv->vif[i], FALSE);
#else
        csrWifiHipResetVif(priv, &priv->vif[i], TRUE);
#endif
    }

    CsrWifiHipPacketSchedulerDestroy(priv->card);

    /* Destroy locks */
    CSR_WIFI_HIP_SPINLOCK_DESTROY(&priv->peerInfoLock);
    CSR_WIFI_HIP_SPINLOCK_DESTROY(&priv->vifMapLock);
    CsrMutexDestroy(&priv->configurationMutex);

    priv->bhThreadStarted = 0;

    /* Destroy BH event */
    CsrEventDestroy(&priv->bhEventHandle);

    if (priv->blockingTerminate)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: CsrWifiHipBhThreadFunction: Signal that blocking terminate has completed",
                            priv ? priv->instance : 0));
        (void) CsrEventSet(&priv->eventHandle, CSR_WIFI_HIP_EVENT_MASK_CLOSED_BH);
    }
    else
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                            "unifi%d: CsrWifiHipBhThreadFunction: Signal that non-blocking terminate has completed",
                            priv ? priv->instance : 0));
        CsrWifiHipWifiOffCfm(priv->osLayerContext, CSR_RESULT_SUCCESS);
    }
} /* CsrWifiHipBhThreadFunction() */

/*
 * ---------------------------------------------------------------------------
 *  csrWifiHipBhInit
 *
 *      Starts the bottom half of the driver.
 *      All we need to do here is start the I/O bh thread.
 *
 *  Arguments:
 *      priv            Pointer to OS driver structure for the device.
 *
 *  Returns:
 *      CSR error code.
 *
 * ---------------------------------------------------------------------------
 */
CsrResult csrWifiHipBhInit(CsrWifiHipHalPriv *priv)
{
    CsrResult result;

    if (!priv)
    {
        return CSR_WIFI_HIP_RESULT_INVALID_VALUE;
    }

    if (priv->bhThreadStarted)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBhInit: Tried to start bh thread, but bh thread is already running\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    result = CsrThreadCreate(CsrWifiHipBhThreadFunction,
                             (void *) priv,
                             CSR_WIFI_HIP_BH_THREAD_STACK_SIZE, THREAD_PRIORITY,
                             (CsrCharString *) "UNIFI_BH",
                             &priv->bhThreadHandle);
    if (result != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: csrWifiHipBhInit: Could not create bh thread, error code %d \n",
                               priv ? priv->instance : 0,  result));
        return CSR_RESULT_FAILURE;
    }

    /* Mark thread as started */
    priv->bhThreadStarted = 1;

    /* Used to indicate that BH thread should terminate */
    priv->bh_thread_terminate = 0;

    return CSR_RESULT_SUCCESS;
}

/*
 * ---------------------------------------------------------------------------
 *  bhErrorHandler
 *
 *      This function reports an error to the SME.
 *      Normally, the SME will try to reset the device and go through
 *      the initialisation of the UniFi.
 *
 *  Arguments:
 *      priv            Pointer to OS driver structure for the device.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 * ---------------------------------------------------------------------------
 */
static void bhErrorHandler(CsrWifiHipHalPriv *priv)
{
    /* Block unifi_run_bh() until the error has been handled. */
    priv->bh_block_io_thread = 1;

    /* Notify about the error */
    if (priv->wifiOnState != CSR_WIFI_HIP_WIFI_ON_IN_PROGRESS)
    {
        CsrWifiHipWifiOffInd(priv->osLayerContext, CSR_WIFI_HIP_CONTROL_ERROR);
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: bhErrorHandler: Fatal error has been reported to the SME \n", priv ? priv->instance : 0));
    }
    else
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: bhErrorHandler: Fatal error. WifiOffInd to SME has been Suppressed! \n", priv ? priv->instance : 0));
    }
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_run_bh
 *
 *      The bottom half of the driver calls this function when
 *      it wants to process anything that requires access to unifi.
 *      We need to call unifi_bh() which in this implementation is done
 *      by waking up the I/O thread.
 *
 *  Arguments:
 *      ospriv          Pointer to OS driver structure for the device.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS or a CSR error code
 *
 *  Notes:  This function may be called from interrupt context!
 * ---------------------------------------------------------------------------
 */
CsrResult unifi_run_bh(void *ospriv)
{
    CsrWifiHipHalPriv *priv = ospriv;

    /*
     * If an error has occured, we discard silently all messages from the bh
     * until the error has been processed and the unifi has been reinitialised.
     */
    if (priv->bh_block_io_thread == 1)
    {
        return CSR_RESULT_FAILURE;
    }

    if (priv->bhThreadStarted == 0)
    {
        return CSR_RESULT_FAILURE;
    }

    /* wake up I/O thread */
    (void) CsrEventSet(&priv->bhEventHandle, CSR_WIFI_HIP_EVENT_MASK_BH_RUN);

    return CSR_RESULT_SUCCESS;
}
