/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"


/*
 * Porting notes:
 * The implementation of unifi_receive_event() in Linux is fairly complicated.
 * The linux driver support multiple userspace applications and several
 * build configurations, so the received signals are processed by different
 * processes and multiple times.
 * In a simple implementation, this function needs to deliver:
 * - The MLME-UNITDATA.ind signals to the Rx data plane and to the Traffic
 *   Analysis using unifi_ta_sample().
 * - The MLME-UNITDATA-STATUS.ind signals to the Tx data plane.
 * - All the other signals to the SME using unifi_sys_hip_ind().
 */

#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "os_linux_priv.h"

#ifdef CSR_NATIVE_LINUX
/*
 * ---------------------------------------------------------------------------
 *  send_to_client
 *
 *      Helper for unifi_receive_event.
 *
 *      This function forwards a signal to one client.
 *
 *  Arguments:
 *      priv        Pointer to driver's private data.
 *      client      Pointer to the client structure.
 *      receiver_id The reciever id of the signal.
 *      sigdata     Pointer to the packed signal buffer.
 *      siglen      Length of the packed signal.
 *      bulkdata    Pointer to the signal's bulk data.
 *
 *  Returns:
 *      None.
 *
 * ---------------------------------------------------------------------------
 */
static void send_to_client(os_linux_priv_t *priv, ul_client_t *client,
                           int receiver_id,
                           unsigned char *sigdata, int siglen,
                           const CsrWifiHipBulkDataParam *bulkdata)
{
    if (client && client->event_hook)
    {
        client->event_hook(client, sigdata, siglen, bulkdata, UDI_TO_HOST);
    }
}

#endif

void CsrWifiHipMaPacketInd(void *osLayerContext, CsrUint16 interfaceTag, CsrUint16 receiverProcessId,
                           CsrUint16 sigLength, CsrUint8 *sigData, CsrUint8 *dMac, CsrUint8 *sMac,
                           CsrWifiHipFrameType frameType, CsrUint8 macHeaderOffset, CsrWifiHipBulkDataParam *bulkData,
                           CsrResult receptionResult, CsrInt16 rssi, CsrInt16 snr, CsrUint16 rate)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;
    struct sk_buff *skb;

    func_enter();



    /* Remove the MAC header */
    skb = (struct sk_buff *) bulkData->d[0].os_net_buf_ptr;
    /*
    * Update the skb->tail pointer.
    */
    skb->len = 0;
    skb_put(skb, bulkData->d[0].data_length);

    skb_pull(skb, macHeaderOffset);

    bulkData->d[0].net_buf_length = bulkData->d[0].data_length = skb->len;
    bulkData->d[0].os_data_ptr = skb->data;
    bulkData->d[0].os_net_buf_ptr = skb;

    indicate_rx_skb(os_linux, interfaceTag, dMac, sMac, skb, bulkData, receptionResult, rssi, snr, rate);
    func_exit();
}

void CsrWifiHipPackedSignalInd(void *osLayerContext, CsrUint16 sigLength, CsrUint8 *sigData, CsrWifiHipBulkDataParam *bulkdata)
{
    os_linux_priv_t *os_linux = osLayerContext;
#ifdef CSR_NATIVE_LINUX
    int receiver_id = 0;

    receiver_id = CSR_GET_UINT16_FROM_LITTLE_ENDIAN((sigData) + sizeof(CsrInt16)) & 0xFF00;

    send_to_client(os_linux, os_linux->sme_cli, receiver_id, sigData, sigLength, (const CsrWifiHipBulkDataParam *) bulkdata);
#else
    CsrWifiRouterCtrlHipIndSend(os_linux->sme_synergy_sched_queue,
                                sigLength,
                                sigData,
                                bulkdata->d[0].data_length,
                                bulkdata->d[0].os_data_ptr,
                                bulkdata->d[1].data_length,
                                bulkdata->d[1].os_data_ptr);
#endif

    os_linux_net_data_free_all(os_linux, bulkdata);
}

void CsrWifiHipMaPacketCfm(void *osLayerContext, CsrUint16 interfaceTag,
                           CsrUint16 receiverProcessId, CsrResult transmissionResult,
                           CsrUint16 rate, CsrUint32 hostTag, CsrUint16 sigLength,
                           CsrUint8 *sigData, CsrWifiHipBulkDataParam *bulkdata)
{
    os_linux_priv_t *os_linux = osLayerContext;

#ifdef CSR_NATIVE_LINUX
    /* If a MaPacketCfm is received for anything else than the network stack,
     * route it to the clients */
    if (os_linux->netdev_client->sender_id != receiverProcessId)
    {
        int receiver_id = 0;

        receiver_id = CSR_GET_UINT16_FROM_LITTLE_ENDIAN((sigData) + sizeof(CsrInt16)) & 0xFF00;
        send_to_client(os_linux, os_linux->sme_cli, receiver_id, sigData, sigLength, (const CsrWifiHipBulkDataParam *) bulkdata);
    }
#else
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: ma packet cfm receiver id=0x%04x hostTag=0x%x\n",
                       os_linux->instance, (int) receiverProcessId,
                       hostTag));

    if ((os_linux->netdev_client->sender_id != receiverProcessId) ||
        (hostTag & CSR_WIFI_HIP_M4_MESSAGE))
    {
        netInterface_priv_t *interfacePriv;
        CsrUint8 appRouting = 0;

        if (interfaceTag >= CSR_WIFI_MAX_INTERFACES)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipMaPacketCfm: Invalid interfaceTag %u\n",
                                   os_linux ? os_linux->instance : 0, interfaceTag));
            os_linux_net_data_free_all(os_linux, bulkdata);
            return;
        }

        interfacePriv = os_linux->interfacePriv[interfaceTag];

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: MA-PACKET Confirm (%x, %x)\n",
                           os_linux ? os_linux->instance : 0, hostTag, transmissionResult));

        appRouting = ((hostTag >> 28) & 0xF);

        if (hostTag & CSR_WIFI_HIP_M4_MESSAGE)
        {
#ifdef CSR_SUPPORT_SME
            unsigned long flags;
            CsrBool send_CsrWifiRouterCtrlM4TransmittedInd;
            CsrWifiMacAddress peerMacAddress;

            /*
             * We cannot call SME API holding a spinlock as they allocate memory.
             * We insted cache the decision and call the API after the spinlock is released.
             */
            send_CsrWifiRouterCtrlM4TransmittedInd = FALSE;

            spin_lock_irqsave(&os_linux->m4_lock, flags);
            if (hostTag == interfacePriv->m4_pending_hostTag)
            {
                /* Check if this is a confirm for EAPOL M4 frame and we need to send transmittedInd */
                send_CsrWifiRouterCtrlM4TransmittedInd = TRUE;
                memcpy(peerMacAddress.a, interfacePriv->m4_peer_mac_addr.a, ETH_ALEN);

                /* Clear the pending hostTag */
                interfacePriv->m4_pending_hostTag = CSR_WIFI_EAPOL_M4_NULL_HOST_TAG;
            }
            spin_unlock_irqrestore(&os_linux->m4_lock, flags);

            if (send_CsrWifiRouterCtrlM4TransmittedInd)
            {
                CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                                   "unifi%d: %s: Sending M4 Transmitted Indication\n",
                                   os_linux ? os_linux->instance : 0, __FUNCTION__));

                CsrWifiRouterCtrlM4TransmittedIndSend(os_linux->sme_synergy_sched_queue, 0,
                                                      interfaceTag,
                                                      peerMacAddress,
                                                      transmissionResult);
            }
#endif

            /* Ensure that MaPacketCfms are discarded if they
               originate from the Linux wpa_supplicant */
            if (os_linux->netdev_client->sender_id != receiverProcessId)
            {
                CsrWifiRouterMaPacketCfmSend((CsrUint16) OS_LINUX_MA_PACKET_CFM_EXTRACT_DEST_SCHED_TASK_ID(sigData),
                                             interfaceTag,
                                             transmissionResult,
                                             (hostTag & 0x3fffffff), rate);
            }

            os_linux_net_data_free_all(os_linux, bulkdata);
            func_exit();
            return;
        }
        else if ((appRouting == CSR_WIFI_ROUTER_APP_TYPE_PAL) ||
                 (appRouting == CSR_WIFI_ROUTER_APP_TYPE_NME) ||
                 (appRouting == CSR_WIFI_ROUTER_APP_TYPE_OTHER))
        {
            CsrWifiRouterMaPacketCfmSend((CsrUint16) OS_LINUX_MA_PACKET_CFM_EXTRACT_DEST_SCHED_TASK_ID(sigData),
                                         interfaceTag,
                                         transmissionResult,
                                         (hostTag & 0x3fffffff), rate);

            os_linux_net_data_free_all(os_linux, bulkdata);
            func_exit();
            return;
        }

        CsrWifiRouterCtrlHipIndSend(os_linux->sme_synergy_sched_queue,
                                    sigLength,
                                    sigData,
                                    bulkdata->d[0].data_length,
                                    bulkdata->d[0].os_data_ptr,
                                    bulkdata->d[1].data_length,
                                    bulkdata->d[1].os_data_ptr);
    }
#endif

    os_linux_net_data_free_all(os_linux, bulkdata);
}
