/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "os_linux_priv.h"
#include "unifiio.h"
#include "unifi_os.h"


static void reset_driver_status(os_linux_priv_t *priv);

/*
 * ---------------------------------------------------------------------------
 *  ul_init_clients
 *
 *      Initialise the clients array to empty.
 *
 *  Arguments:
 *      priv            Pointer to device private context struct
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 *      This function needs to be called before priv is stored in
 *      Unifi_instances[].
 * ---------------------------------------------------------------------------
 */
void ul_init_clients(os_linux_priv_t *priv)
{
    int id;
    ul_client_t *ul_clients;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)
    sema_init(&priv->udi_logging_mutex, 1);
#else
    init_MUTEX(&priv->udi_logging_mutex);
#endif
    priv->logging_client = NULL;

    ul_clients = priv->ul_clients;

    for (id = 0; id < MAX_UDI_CLIENTS; id++)
    {
        memset(&ul_clients[id], 0, sizeof(ul_client_t));

        ul_clients[id].client_id = id;
        ul_clients[id].sender_id = UDI_SENDER_ID_BASE + (id << UDI_SENDER_ID_SHIFT);
        ul_clients[id].instance = -1;
        ul_clients[id].event_hook = NULL;

        INIT_LIST_HEAD(&ul_clients[id].udi_log);
        init_waitqueue_head(&ul_clients[id].udi_wq);
        sema_init(&ul_clients[id].udi_sem, 1);

    }
} /* ul_init_clients() */

/*
 * ---------------------------------------------------------------------------
 *  ul_register_client
 *
 *      This function registers a new ul client.
 *
 *  Arguments:
 *      priv            Pointer to device private context struct
 *      configuration   Special configuration for the client.
 *      udi_event_clbk  Callback for receiving event from unifi.
 *
 *  Returns:
 *      0 if a new clients is registered, -1 otherwise.
 * ---------------------------------------------------------------------------
 */
ul_client_t *ul_register_client(os_linux_priv_t *priv, unsigned int configuration,
                                udi_event_t udi_event_clbk)
{
    unsigned char id;
    ul_client_t *ul_clients;

    ul_clients = priv->ul_clients;

    /* check for an unused entry */
    for (id = 0; id < MAX_UDI_CLIENTS; id++)
    {
        if (ul_clients[id].udi_enabled == 0)
        {
            ul_clients[id].instance = priv->instance;
            ul_clients[id].udi_enabled = 1;
            ul_clients[id].configuration = configuration;

            /* Set the event callback. */
            ul_clients[id].event_hook = udi_event_clbk;

            CSR_LOG_TEXT_INFO((
                                  CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: UDI %d (0x%x) registered. configuration = 0x%x\n",
                                  priv ? priv->instance : 0,
                                  id, &ul_clients[id], configuration));
            return &ul_clients[id];
        }
    }
    return NULL;
} /* ul_register_client() */

/*
 * ---------------------------------------------------------------------------
 *  ul_deregister_client
 *
 *      This function deregisters a blocking UDI client.
 *
 *  Arguments:
 *      client      Pointer to the client we deregister.
 *
 *  Returns:
 *      0 if a new clients is deregistered.
 * ---------------------------------------------------------------------------
 */
int ul_deregister_client(ul_client_t *ul_client)
{
    struct list_head *pos, *n;
    udi_log_t *logptr;
    os_linux_priv_t *priv;

    priv = uf_find_instance(ul_client->instance);

    ul_client->instance = -1;
    ul_client->event_hook = NULL;
    ul_client->udi_enabled = 0;
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: UDI (0x%x) deregistered.\n",
                        priv ? priv->instance : 0,  ul_client));

    /* Free anything pending on the udi_log list */
    down(&ul_client->udi_sem);
    list_for_each_safe(pos, n, &ul_client->udi_log)
    {
        logptr = list_entry(pos, udi_log_t, q);
        list_del(pos);
        kfree(logptr);
    }
    up(&ul_client->udi_sem);

    return 0;
} /* ul_deregister_client() */

/*
 * ---------------------------------------------------------------------------
 *  logging_handler
 *
 *      This function is registered with the driver core.
 *      It is called every time a UniFi HIP Signal is sent. It iterates over
 *      the list of processes interested in receiving log events and
 *      delivers the events to them.
 *
 *  Arguments:
 *      ospriv      Pointer to driver's private data.
 *      sigdata     Pointer to the packed signal buffer.
 *      signal_len  Length of the packed signal.
 *      bulkdata    Pointer to the signal's bulk data.
 *      dir         Direction of the signal
 *                  0 = from-host
 *                  1 = to-host
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void logging_handler(void *ospriv,
                     CsrUint8 *sigdata, CsrUint32 signal_len,
                     const CsrWifiHipBulkDataParam *bulkdata,
                     CsrWifiHipLogUdiDirection direction)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) ospriv;
    ul_client_t *client;

    down(&priv->udi_logging_mutex);
    client = priv->logging_client;
    if (client != NULL)
    {
        client->event_hook(client, sigdata, signal_len,
                           bulkdata, direction);
    }
    up(&priv->udi_logging_mutex);
} /* logging_handler() */

/*
 * ---------------------------------------------------------------------------
 *  ul_log_config_ind
 *
 *      This function uses the client's register callback
 *      to indicate configuration information e.g core errors.
 *
 *  Arguments:
 *      priv        Pointer to driver's private data.
 *      conf_param  Pointer to the configuration data.
 *      len         Length of the configuration data.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void ul_log_config_ind(os_linux_priv_t *priv, u8 *conf_param, int len)
{
#ifdef CSR_SUPPORT_SME
    if (priv->smepriv == NULL)
    {
        return;
    }

    /* Terminate due to error or driver exit.
     * WifiOffInd(error) will only be sent from HAL if WifiOn has completed,
     * i.e. it's not necessary to perform a state check here.
     */
    CsrWifiRouterCtrlWifiOffIndSend(priv->sme_synergy_sched_queue, 0, (CsrWifiRouterCtrlControlIndication) (*conf_param));
#else
    CsrWifiHipBulkDataParam bulkdata;

    /*
     * If someone killed unifi_managed before the driver was unloaded
     * the g_drvpriv pointer is going to be NULL. In this case it is
     * safe to assume that there is no client to get the indication.
     */
    if (!priv)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_LOG_DEF,
                           "unifi : uf_sme_event_ind: NULL priv\n"));
        return;
    }

    /* Create a null bulkdata structure. */
    bulkdata.d[0].data_length = 0;
    bulkdata.d[1].data_length = 0;

    sme_native_log_event(priv->sme_cli, conf_param, sizeof(CsrUint8),
                         &bulkdata, UDI_CONFIG_IND);
#endif /* CSR_SUPPORT_SME */
} /* ul_log_config_ind */

unsigned long hip2_stats_tx_realignment = 0;

static int _align_bulk_data_buffers(os_linux_priv_t *priv, CsrUint8 *signal,
                                    CsrWifiHipBulkDataParam *bulkdata)
{
    unsigned int i;

    if ((bulkdata == NULL) || (CSR_WIFI_ALIGN_BYTES == 0))
    {
        return 0;
    }

    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        struct sk_buff *skb;
        /*
        * The following complex casting is in place in order to eliminate 64-bit compilation warning
        * "cast to/from pointer from/to integer of different size"
        */
        CsrUint32 align_offset = (CsrUint32) (long) (bulkdata->d[i].os_data_ptr) & (CSR_WIFI_ALIGN_BYTES - 1);
        if (align_offset)
        {
            skb = (struct sk_buff *) bulkdata->d[i].os_net_buf_ptr;
            if (skb == NULL)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                      CSR_WIFI_HIP_LOG_DEF, "unifi%d: _align_bulk_data_buffers: Align offset found (%d) but skb is NULL!\n",
                                      priv ? priv->instance : 0,
                                      align_offset));
                return -EINVAL;
            }
            if (bulkdata->d[i].data_length == 0)
            {
                CSR_LOG_TEXT_WARNING((CSR_WIFI_HIP_LOG_ID,
                                      CSR_WIFI_HIP_LOG_DEF, "unifi%d: _align_bulk_data_buffers: Align offset found (%d) but length is zero\n",
                                      priv ? priv->instance : 0,
                                      align_offset));
                return CSR_RESULT_SUCCESS;
            }
            CSR_LOG_TEXT_DEBUG((
                                   CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                                   "unifi%d: Align f-h buffer (0x%p) by %d bytes (skb->data: 0x%p)\n",
                                   priv ? priv->instance : 0,
                                   bulkdata->d[i].os_data_ptr, align_offset, skb->data));


            /* Check if there is enough headroom... */
            if (unlikely(skb_headroom(skb) < align_offset))
            {
                struct sk_buff *tmp = skb;

                CSR_LOG_TEXT_DEBUG((
                                       CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,  "unifi%d: Headroom not enough - realloc it\n"
                                       , priv ? priv->instance : 0));
                skb = skb_realloc_headroom(skb, align_offset);
                if (skb == NULL)
                {
                    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: _align_bulk_data_buffers: skb_realloc_headroom failed - signal is dropped\n", priv ? priv->instance : 0));
                    return -EFAULT;
                }
                /* Free the old bulk data only if allocation succeeds */
                kfree_skb(tmp);
                /* Bulkdata needs to point to the new skb */
                bulkdata->d[i].os_net_buf_ptr = (const void *) skb;
                bulkdata->d[i].os_data_ptr = (CsrUint8 *) skb->data;
                hip2_stats_tx_realignment++;
            }
            /* ... before pushing the data to the right alignment offset */
            skb_push(skb, align_offset);

        }
        /* The direction bit is zero for the from-host */
        signal[SIZEOF_SIGNAL_HEADER + (i * SIZEOF_DATAREF) + 1] = align_offset;
    }
    return 0;
} /* _align_bulk_data_buffers() */

/*
 * ---------------------------------------------------------------------------
 *  ul_send_signal_unpacked
 *
 *      This function sends a host formatted signal to unifi.
 *
 *  Arguments:
 *      priv        Pointer to driver's private data.
 *      sigptr      Pointer to the signal.
 *      bulkdata    Pointer to the signal's bulk data.
 *
 *  Returns:
 *      O on success, error code otherwise.
 *
 *  Notes:
 *  The signals have to be sent in the format described in the host interface
 *  specification, i.e wire formatted. Certain clients use the host formatted
 *  structures. The write_pack() transforms the host formatted signal
 *  into the wired formatted signal. The code is in the core, since the signals
 *  are defined therefore binded to the host interface specification.
 * ---------------------------------------------------------------------------
 */
int ul_send_signal_unpacked(os_linux_priv_t *priv, netInterface_priv_t *interfacePriv, CSR_SIGNAL *sigptr,
                            CsrWifiHipBulkDataParam *bulkdata)
{
    CsrUint8 sigbuf[UNIFI_PACKED_SIGBUF_SIZE];
    CsrResult csrResult;
    int r;
    CsrUint32 aId = 0;



    r = _align_bulk_data_buffers(priv, sigbuf, (CsrWifiHipBulkDataParam *) bulkdata);
    if (r)
    {
        return r;
    }

#ifdef CSR_NATIVE_LINUX
    if (interfacePriv->interfaceMode == CSR_WIFI_ROUTER_CTRL_MODE_AP)
    {
        if (sigptr->SignalPrimitiveHeader.SignalId == CSR_MA_PACKET_REQUEST_ID)
        {
            CsrUint8 toDS, fromDS, *dMac;
            CsrUint16 frameControl;
            CsrUint8 i;

            frameControl = CSR_GET_UINT16_FROM_LITTLE_ENDIAN(bulkdata->d[0].os_data_ptr);
            toDS = CSR_WIFI_HIP_IEEE80211_HAS_TO_DS(frameControl);
            fromDS = CSR_WIFI_HIP_IEEE80211_HAS_FROM_DS(frameControl);

            dMac = (CsrUint8 *) bulkdata->d[0].os_data_ptr + 4 + (toDS * 12); /* Address 1 or 3 */
            if (toDS && fromDS)
            {
                CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                                    "unifi%d: udi_send_signal_unpacked: 4 address frame: Not supported\n",
                                    priv ? priv->instance : 0));
            }

            for (i = 0; i < interfacePriv->peer_sta_count; i++)
            {
                if (!memcmp(interfacePriv->peer_sta_info[i].peer_macaddr, dMac, ETH_ALEN))
                {
                    aId = interfacePriv->peer_sta_info[i].association_id;
                    break;
                }
            }
        }
    }
#endif

    csrResult = CsrWifiHipUnpackedSignalReq(priv->hip_handle,
                                            interfacePriv->InterfaceTag,
                                            aId,
                                            sizeof(CSR_SIGNAL),
                                            (CsrUint8 *) sigptr,
                                            (CsrWifiHipBulkDataParam *) bulkdata);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: Failed to transmit CsrWifiHipUnpackedSignalReq\n", priv ? priv->instance : 0));
        return CsrHipResultToStatus(csrResult);
    }

    return 0;
} /* ul_send_signal_unpacked() */

/*
 * ---------------------------------------------------------------------------
 *  reset_driver_status
 *
 *      This function is called from ul_send_signal_raw() when it detects
 *      that the SME has sent a MLME-RESET request.
 *
 *  Arguments:
 *      priv        Pointer to device private struct
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void reset_driver_status(os_linux_priv_t *priv)
{
} /* reset_driver_status() */

/*
 * ---------------------------------------------------------------------------
 *  ul_send_signal_raw
 *
 *      This function sends a wire formatted data signal to unifi.
 *
 *  Arguments:
 *      priv        Pointer to driver's private data.
 *      sigptr      Pointer to the signal.
 *      siglen      Length of the signal.
 *      bulkdata    Pointer to the signal's bulk data.
 *
 *  Returns:
 *      O on success, error code otherwise.
 * ---------------------------------------------------------------------------
 */
int ul_send_signal_raw(os_linux_priv_t *priv, unsigned char *sigptr, int siglen,
                       CsrWifiHipBulkDataParam *bulkdata)
{
    CsrResult csrResult;
    int r;
    CsrUint16 interfaceTag = 0, interfaceIndex = default_vif;

    /*
     * Make sure that the signal is updated with the bulk data
     * alignment for DMA.
     */
    r = _align_bulk_data_buffers(priv, (CsrUint8 *) sigptr, bulkdata);
    if (r)
    {
        return r;
    }

    csrResult = read_vif_from_packed(sigptr, &interfaceIndex);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: Could not find vifIndex - use default interfaceIndex=%u\n", priv ? priv->instance : 0, interfaceIndex));
        interfaceIndex = default_vif;
    }
    interfaceTag = CsrWifiHipInterfaceTagGetReq(priv->hip_handle, interfaceIndex);

    /* Send signal */
    csrResult = CsrWifiHipPackedSignalReq(priv->hip_handle, interfaceTag, siglen, sigptr, (CsrWifiHipBulkDataParam *) bulkdata);
    if (csrResult != CSR_RESULT_SUCCESS)
    {
        os_linux_net_data_free_all(priv, (CsrWifiHipBulkDataParam *) bulkdata);
        return CsrHipResultToStatus(csrResult);
    }

    /*
     * Since this is use by unicli, if we get an MLME reset request
     * we need to initialize a few status parameters
     * that the driver uses to make decisions.
     */
    if (GET_SIGNAL_ID(sigptr) == CSR_MLME_RESET_REQUEST_ID)
    {
        reset_driver_status(priv);
    }

    return 0;
} /* ul_send_signal_raw() */
