/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include <linux/netdevice.h>

#include "os_linux_priv.h"
#include "csr_wifi_hip_unifi.h"
#include "csr_wifi_hip_conversions.h"
#include "csr_wifi_hip_log_text.h"
#include "csr_wifi_hip.h"
#include "unifi_os.h"

static const unsigned char wildcard_address[ETH_ALEN] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};


DECLARE_WAIT_QUEUE_HEAD(Unifi_native_wifion_wq);

int uf_sme_init(os_linux_priv_t *priv)
{
    func_enter();


    func_exit();
    return 0;
} /* uf_sme_init() */

void uf_sme_deinit(os_linux_priv_t *priv)
{

    func_enter();

    /* Free memory allocated for the scan table */
    /*    unifi_clear_scan_table(priv); */

    /* Cancel any pending workqueue tasks */
    flush_workqueue(priv->unifi_workqueue);


    func_exit();
} /* uf_sme_deinit() */

static void native_wifi_off(os_linux_priv_t *priv)
{
    if (priv->hip_handle)
    {
        CsrWifiHipWifiOffReq(TRUE, priv->hip_handle);
    }
    priv->init_progress = UNIFI_INIT_NONE;

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Ensure the Hydra WLAN subsystem is stopped */
    uf_stop_hydra_wlan_service(priv);
#endif
}

int sme_mgt_wifi_on(os_linux_priv_t *priv)
{
    int i;
    CsrResult ret;
    CsrWifiMacAddress macAddr = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
    CsrWifiHipInstanceConfig config;

    if (priv == NULL)
    {
        return -EINVAL;
    }

    if (priv->init_progress != UNIFI_INIT_NONE)
    {
        CsrResult ret;
        CsrUint8 i;

        for (i = 0; i < priv->totalInterfaceCount; i++)
        {
            ret = CsrWifiHipInterfaceDelReq(priv->hip_handle, priv->interfacePriv[i]->InterfaceTag);
            if (ret != CSR_RESULT_SUCCESS)
            {
                CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                       "unifi : sme_mgt_wifi_on: Failed to remove interface %u\n", priv->interfacePriv[i]->InterfaceTag));
            }
        }
        native_wifi_off(priv);
    }

#if defined CSR_WIFI_DRIVER_HYDRA && defined CSR_WIFI_DRIVER_USE_HYDRA_DRIVER
    /* Start the Hydra WLAN subsystem */
    ret = uf_start_hydra_wlan_service(priv);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : sme_mgt_wifi_on: Failed to start Hydra WLAN service\n"));
        return CSR_RESULT_FAILURE;
    }
#endif

    priv->init_progress = UNIFI_INIT_IN_PROGRESS;

    /* Initialize the interface mode to None */
    for (i = 0; i < CSR_WIFI_MAX_INTERFACES; i++)
    {
        priv->interfacePriv[i]->interfaceMode = 0;
    }

    config.hipMode = CSR_WIFI_HIP_WIFI_STACK_MODE;
    config.allocFunc = os_linux_net_data_malloc;
    config.freeFunc = os_linux_net_data_free;
    config.fw_init = fw_init[0];
    config.cardParams = config_params;
    config.flowControlPauseCb = os_linux_flow_control_pause_cb;
    config.flowControlResumeCb = os_linux_flow_control_resume_cb;
    config.cmanrTestMode = FALSE;
    ret = CsrWifiHipWifiOnReq(priv->hip_handle,
                              priv,
                              0,
                              config);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : sme_mgt_wifi_on: Failed to turn Wi-Fi on\n"));
        return CSR_RESULT_FAILURE;
    }

    wait_event(Unifi_native_wifion_wq, priv->init_progress == UNIFI_INIT_FW_DOWNLOADED ||
               priv->init_progress == UNIFI_INIT_NONE);
    if (priv->init_progress == UNIFI_INIT_NONE)
    {
        return CSR_RESULT_FAILURE;
    }


    /* Have to open an interface to be able to do MIB download, reset etc.;
       however avoid registering netdev for now */
    if (CsrWifiHipInterfaceAddReq(priv->hip_handle, 0, macAddr) != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                            "unifi : %s: Failed to add interface \n", __FUNCTION__));
    }

    CsrWifiHipVifModeSetReq(priv->hip_handle, 0, CSR_WIFI_ROUTER_CTRL_MODE_STA, macAddr,
                            0, 0, default_vif);

    return 0;
}

int sme_sys_suspend(os_linux_priv_t *priv)
{
    const int interfaceNum = 0;     /* D-29892 */

    /* Abort any pending requests. */
    uf_abort_mlme(priv);

    /* Allow our mlme request to go through. */
    priv->io_aborted = 0;

    /* Stop the network traffic */
    netif_carrier_off(priv->netdev[interfaceNum]);

    return 0;
} /* sme_sys_suspend() */

int sme_sys_resume(os_linux_priv_t *priv)
{
    return 0;
} /* sme_sys_resume() */

/*
 * ---------------------------------------------------------------------------
 *  sme_native_log_event
 *
 *      Callback function to be registered as the SME event callback.
 *      Copies the signal content into a new udi_log_t struct and adds
 *      it to the read queue for the SME client.
 *
 *  Arguments:
 *      arg             This is the value given to unifi_add_udi_hook, in
 *                      this case a pointer to the client instance.
 *      signal          Pointer to the received signal.
 *      signal_len      Size of the signal structure in bytes.
 *      bulkdata        Pointers to any associated bulk data.
 *      dir             Direction of the signal. Zero means from host,
 *                      non-zero means to host.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void sme_native_log_event(ul_client_t *pcli,
                          const u8 *sig_packed, int sig_len,
                          const CsrWifiHipBulkDataParam *bulkdata,
                          int dir)
{
    os_linux_priv_t *priv;
    udi_log_t *logptr;
    u8 *p;
    int i, r;
    int signal_len;
    int total_len;
    udi_msg_t *msgptr;
    CSR_SIGNAL signal;
    ul_client_t *client = pcli;

    func_enter();

    if (client == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : sme_native_log_event: client has exited\n"));
        return;
    }

    priv = uf_find_instance(client->instance);

    if (!priv)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : invalid priv\n"));
        return;
    }

    /* Just a sanity check */
    if ((sig_packed == NULL) || (sig_len <= 0))
    {
        return;
    }

    /* Get the unpacked signal */
    r = read_unpack_signal(sig_packed, &signal);
    if (r == 0)
    {
        signal_len = SigGetSize(&signal);
    }
    else
    {
        CsrUint16 receiver_id = CSR_GET_UINT16_FROM_LITTLE_ENDIAN((sig_packed) + sizeof(CsrUint16)) & 0xFF00;

        /* The control indications are 1 byte, pass them to client. */
        if (sig_len == 1)
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: Control indication (0x%x) for native SME.\n",
                                priv ? priv->instance : 0,
                                *sig_packed));

            *(u8 *)&signal = *sig_packed;
            signal_len = sig_len;
        }
        else if (receiver_id == 0)
        {
            /*
             * Also "unknown" signals with a ReceiverId of 0 are passed to the client
             * without unpacking. (This is a code size optimisation to allow signals
             * that the driver not interested in to be dropped from the unpack code).
             */
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                                "unifi%d: Signal 0x%.4X with ReceiverId 0 for native SME.\n",
                                priv ? priv->instance : 0,
                                CSR_GET_UINT16_FROM_LITTLE_ENDIAN(sig_packed)));

            *(u8 *)&signal = *sig_packed;
            signal_len = sig_len;
        }
        else
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                                   CSR_WIFI_HIP_LOG_DEF,
                                   "sme_native_log_event - Received unknown signal 0x%.4X.\n",
                                   CSR_GET_UINT16_FROM_LITTLE_ENDIAN(sig_packed)));
            return;
        }
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3,
                        "unifi : sme_native_log_event: signal 0x%.4X for %d\n",
                        signal.SignalPrimitiveHeader.SignalId, client->client_id));

    /* Snoop for the first MLME-RESET.confirm, so the SDIO clock can be raised.
     * It is only safe to raise the clock once the MIB has been transferred,
     * because it contains the clock configuration needed by firmware.
     */
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG6,
                        "sme_native_log_event: init_progress=%d state=%d\n", priv->init_progress, priv->state));
    if ((signal.SignalPrimitiveHeader.SignalId == CSR_MLME_RESET_CONFIRM_ID) &&
        (priv->init_progress == UNIFI_INIT_COMPLETED) &&
        (priv->state != OS_LINUX_STATE_UF_READY))
    {
        CsrBool intmode = FALSE;
        CsrResult ret;

        /* If the MIB has selected f/w scheduled interrupt mode, apply it now
         * but let module param override.
         */
        if (run_bh_once != -1)
        {
            intmode = TRUE;
        }

        /* The HIP post-init will request that the SDIO clock be raised. */
        ret = CsrWifiHipPostInitReq(priv->hip_handle, intmode);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi : CsrWifiHipWifiOnInd: Failed to do CsrWifiHipPostInitReq\n"));
        }

        ret = CsrWifiHipLowPowerModeReq(priv->hip_handle,
                                        CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_DISABLED,
                                        FALSE);
        if (ret != CSR_RESULT_SUCCESS)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi : CsrWifiHipWifiOnInd: Failed to set CsrWifiHipLowPowerModeReq\n"));
        }

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: UniFi ready\n", priv->instance)); /* Native */

        priv->state = OS_LINUX_STATE_UF_READY;
    }

    total_len = signal_len;
    /* Calculate the buffer we need to store signal plus bulk data */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        total_len += bulkdata->d[i].data_length;
    }

    /* Allocate log structure plus actual signal. */
    logptr = (udi_log_t *) kmalloc(sizeof(udi_log_t) + total_len, GFP_KERNEL);

    if (logptr == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "Failed to allocate %d bytes for a UDI log record\n",
                               sizeof(udi_log_t) + total_len));
        return;
    }

    /* Fill in udi_log struct */
    INIT_LIST_HEAD(&logptr->q);
    msgptr = &logptr->msg;
    msgptr->length = sizeof(udi_msg_t) + total_len;
    msgptr->timestamp = jiffies_to_msecs(jiffies);
    msgptr->direction = dir;
    msgptr->signal_length = signal_len;

    /* Copy signal and bulk data to the log */
    p = (u8 *) (msgptr + 1);
    memcpy(p, &signal, signal_len);
    p += signal_len;

    /* Append any bulk data */
    for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++)
    {
        int len = bulkdata->d[i].data_length;

        /*
         * Len here might not be the same as the length in the bulk data slot.
         * The slot length will always be even, but len could be odd.
         */
        if (len > 0)
        {
            if (bulkdata->d[i].os_data_ptr)
            {
                memcpy(p, bulkdata->d[i].os_data_ptr, len);
            }
            else
            {
                memset(p, 0, len);
            }
            p += len;
        }
    }

    /* Add to tail of log queue */
    down(&client->udi_sem);
    list_add_tail(&logptr->q, &client->udi_log);
    up(&client->udi_sem);

    /* Wake any waiting user process */
    wake_up_interruptible(&client->udi_wq);

    func_exit();
} /* sme_native_log_event() */

/*
 * ---------------------------------------------------------------------------
 *  unifi_ta_indicate_protocol
 *
 *      Report that a packet of a particular type has been seen
 *
 *  Arguments:
 *      drv_priv        The device context pointer passed to ta_init.
 *      protocol        The protocol type enum value.
 *      direction       Whether the packet was a tx or rx.
 *      src_addr        The source MAC address from the data packet.
 *
 *  Returns:
 *      None.
 *
 *  Notes:
 *      We defer the actual sending to a background workqueue,
 *      see uf_ta_ind_wq().
 * ---------------------------------------------------------------------------
 */
void unifi_ta_indicate_protocol(void                              *ospriv,
                                CsrWifiRouterCtrlTrafficPacketType packet_type,
                                CsrWifiRouterCtrlProtocolDirection direction,
                                const CsrWifiMacAddress           *src_addr)
{
} /* unifi_ta_indicate_protocol */

/*
 * ---------------------------------------------------------------------------
 * unifi_ta_indicate_sampling
 *
 *      Send the TA sampling information to the SME.
 *
 *  Arguments:
 *      drv_priv        The device context pointer passed to ta_init.
 *      stats           The TA sampling data to send.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void unifi_ta_indicate_sampling(void *ospriv, CsrWifiRouterCtrlTrafficStats *stats)
{
} /* unifi_ta_indicate_sampling() */

/*
 * ---------------------------------------------------------------------------
 * uf_native_process_udi_signal
 *
 *      Process interesting signals from the UDI interface.
 *
 *  Arguments:
 *      pcli            A pointer to the client instance.
 *      signal          Pointer to the received signal.
 *      signal_len      Size of the signal structure in bytes.
 *      bulkdata        Pointers to any associated bulk data.
 *      dir             Direction of the signal. Zero means from host,
 *                      non-zero means to host.
 *
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void uf_native_process_udi_signal(ul_client_t *pcli,
                                  const u8 *packed_signal, int packed_signal_len,
                                  const CsrWifiHipBulkDataParam *bulkdata, int dir)
{
} /* uf_native_process_udi_signal() */



void CsrWifiHipWifiOnInd(void          *osLayerContext,
                         CsrResult      result,
                         CsrUint32      chipId,
                         CsrUint32      chipVersion,
                         CsrUint32      firmwareBuild,
                         CsrUint32      firmwareHip,
                         CsrCharString *routerBuild,
                         CsrUint32      routerHip,
                         CsrUint32      sdioBlockSize)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    CsrResult ret;

    priv->fw_build = firmwareBuild;
    priv->sdio_block_size = sdioBlockSize;

    ret = CsrWifiHipLowPowerModeReq(priv->hip_handle,
                                    CSR_WIFI_ROUTER_CTRL_LOW_POWER_MODE_DISABLED,
                                    FALSE);
    if (ret != CSR_RESULT_SUCCESS)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi : CsrWifiHipWifiOnInd: Failed to set CsrWifiHipLowPowerModeReq\n"));
    }

    priv->init_progress = UNIFI_INIT_FW_DOWNLOADED;
    wake_up(&Unifi_native_wifion_wq);
}

void CsrWifiHipWifiOffCfm(void *osLayerContext, CsrResult result)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    priv->init_progress = UNIFI_INIT_NONE;
    priv->state = OS_LINUX_STATE_ACTIVATED;
}

void CsrWifiHipWifiOnCfm(void *osLayerContext, CsrResult result)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) osLayerContext;
    priv->init_progress = UNIFI_INIT_NONE;
    wake_up(&Unifi_native_wifion_wq);
}

void CsrWifiHipWifiOffInd(void                       *osLayerContext,
                          CsrWifiHipControlIndication controlIndication)
{
    os_linux_priv_t *os_linux = (os_linux_priv_t *) osLayerContext;
    CsrUint16 interfaceTag = 0;
    u8 conf_param = CONFIG_IND_ERROR;

    /* Consider UniFi to be uninitialised */
    os_linux->init_progress = UNIFI_INIT_NONE;

    /* Stop the network traffic */
    for (interfaceTag = 0; interfaceTag < CSR_WIFI_MAX_INTERFACES; interfaceTag++)
    {
        netInterface_priv_t *interfacePriv = os_linux->interfacePriv[interfaceTag];
        if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
            (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
        {
            netif_carrier_off(os_linux->netdev[interfaceTag]);
            interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED;
        }
    }

#ifdef CSR_NATIVE_LINUX
    /* Force any client waiting on an mlme_wait_for_reply() to abort. */
    uf_abort_mlme(os_linux);

    /* Cancel any pending workqueue tasks */
    flush_workqueue(os_linux->unifi_workqueue);
#endif /* CSR_NATIVE_LINUX */


    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: CsrWifiHipWifiOffInd: fatal error is reported to the SME.\n", os_linux ? os_linux->instance : 0));
    /* Notify the clients (SME or unifi_manager) for the error. */
    ul_log_config_ind(os_linux, &conf_param, sizeof(u8));
}

CsrSchedQid CSR_WIFI_ROUTER_IFACEQUEUE;

#if defined(CSR_LOG_ENABLE) && defined(CSR_LOG_INCLUDE_FILE_NAME_AND_LINE_NUMBER)
void CsrSchedMessagePutStringLog(CsrSchedQid          q,
                                 CsrUint16            mi,
                                 void                *mv,
                                 CsrUint32            line,
                                 const CsrCharString *file)
#define CsrSchedMessagePut(q, mi, mv) CsrSchedMessagePutStringLog((q), (mi), (mv), __LINE__, __FILE__)
#else
void CsrSchedMessagePut(CsrSchedQid q,
                        CsrUint16   mi,
                        void       *mv)
#endif
{
}

CsrResult CsrWifiHipInterfaceUnregisterInd(void *synergyContext, CsrUint16 interfaceTag)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) synergyContext;
    netInterface_priv_t *interfacePriv = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: CsrWifiHipInterfaceUnregisterInd: invalid priv\n",
                               priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d \n",
                        priv ? priv->instance : 0, __FUNCTION__, interfaceTag));

    /* Register the netdevs for requested interface. */
    interfacePriv = (netInterface_priv_t *) priv->interfacePriv[interfaceTag];

    if ((interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED) ||
        (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED))
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG3, "unifi%d: unregistering net device %d\n",
                            priv ? priv->instance : 0, interfaceTag));

        UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[interfaceTag]);
        netif_carrier_off(priv->netdev[interfaceTag]);
        interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED_STOPPED;
    }

    return CSR_RESULT_SUCCESS;
}

CsrResult CsrWifiHipInterfaceRegisterInd(void *synergyContext, CsrUint16 interfaceTag, CsrWifiMacAddress macAddress)
{
    os_linux_priv_t *priv = (os_linux_priv_t *) synergyContext;
    netInterface_priv_t *interfacePriv = NULL;

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF, "unifi%d: (%s) : invalid priv\n",
                               priv ? priv->instance : 0, __FUNCTION__));
        return CSR_RESULT_FAILURE;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d:%s:ifTag=%d \n",
                        priv ? priv->instance : 0, __FUNCTION__, interfaceTag));

    /* Store the MAC address in GHAL database */
    CsrWifiHipSetMacReq(priv->hip_handle, &macAddress, interfaceTag);

    interfacePriv = (netInterface_priv_t *) priv->interfacePriv[interfaceTag];

    /* Register the netdevs for requested interface. */
    if (interfacePriv->netdev_state == OS_LINUX_NETDEV_STATE_UNREGISTERED)
    {
        int r;

        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG3,
                            "unifi%d: registering net device %d\n",
                            priv ? priv->instance : 0, interfaceTag));

        /* Store the MAC address in the netdev structure */
        memcpy(priv->netdev[interfaceTag]->dev_addr, macAddress.a, ETH_ALEN);

        r = uf_register_netdev(priv, interfaceTag);
        if (r)
        {
            CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                                   "unifi%d: CsrWifiHipInterfaceRegisterInd: Failed to register the network device.\n",
                                   priv->instance));
            native_wifi_off(priv);

            priv->state = OS_LINUX_STATE_ACTIVATED;
            return CSR_RESULT_FAILURE;
        }
    }
    else
    {
        /* If interface Mac Address changes, then update accordingly */
        if (memcmp(macAddress.a, priv->netdev[interfaceTag]->dev_addr, ETH_ALEN))
        {
            CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                                "unifi%d:%s: Trying to set new Mac address iface=%s \n",
                                priv ? priv->instance : 0, __FUNCTION__, priv->netdev[interfaceTag]->name));
            /* Trying to set new Mac address for the interface */
            if (netif_running(priv->netdev[interfaceTag]))
            {
                UF_NETIF_TX_STOP_ALL_QUEUES(priv->netdev[interfaceTag]);
            }

            netif_carrier_off(priv->netdev[interfaceTag]);
            /* Store the MAC address in the netdev */
            memcpy(priv->netdev[interfaceTag]->dev_addr, macAddress.a, ETH_ALEN);
            netif_carrier_on(priv->netdev[interfaceTag]);
        }
    }

    priv->init_progress = UNIFI_INIT_COMPLETED;
    interfacePriv->netdev_state = OS_LINUX_NETDEV_STATE_REGISTERED;
    CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1,
                           "unifi%d: Interface %d ready\n", interfaceTag, priv->instance));

    return CSR_RESULT_SUCCESS;
}

/*
 * These are stubs in the Native driver
 */
void CsrWifiHipSuspendInd(void *osLayerContext, CsrUint16 clientData, CsrBool hardSuspend, CsrBool d3Suspend)
{
}

void CsrWifiHipResumeInd(void *osLayerContext, CsrUint16 clientData, CsrBool powerMaintained)
{
}
