/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

#include "os_linux_priv.h"


/*
 * This file also contains the implementation of the asyncronous
 * requests to the SME.
 *
 * Before calling an asyncronous SME function, we call sme_init_request()
 * which gets hold of the SME semaphore and updates the request status.
 * The semaphore makes sure that there is only one pending request to
 * the SME at a time.
 *
 * Now we are ready to call the SME function, but only if
 * sme_init_request() has returned 0.
 *
 * When the SME function returns, we need to wait
 * for the reply. This is done in sme_wait_for_reply().
 * If the request times-out, the request status is set to SME_REQUEST_TIMEDOUT
 * and the sme_wait_for_reply() returns.
 *
 * If the SME replies in time, we call sme_complete_request().
 * There we change the request status to SME_REQUEST_RECEIVED. This will
 * wake up the process waiting on sme_wait_for_reply().
 * It is important that we copy the reply data in priv->sme_reply
 * before calling sme_complete_request().
 *
 * Handling the wext requests, we need to block
 * until the SME sends the response to our request.
 * We use the sme_init_request() and sme_wait_for_reply()
 * to implement this behavior in the following functions:
 * sme_mgt_wifi_on()
 * sme_mgt_wifi_off()
 * sme_mgt_scan_full()
 * sme_mgt_scan_results_get_async()
 * sme_mgt_connect()
 * unifi_mgt_media_status_ind()
 * sme_mgt_disconnect()
 * sme_mgt_pmkid()
 * sme_mgt_key()
 * sme_mgt_versions_get()
 * sme_mgt_set_value()
 * sme_mgt_get_value()
 * sme_mgt_set_value_async()
 * sme_mgt_get_value_async()
 * sme_mgt_packet_filter_set()
 * sme_mgt_tspec()
 */


/*
 * Handling the suspend and resume system events, we need to block
 * until the SME sends the response to our indication.
 * We use the sme_init_request() and sme_wait_for_reply()
 * to implement this behavior in the following functions:
 * sme_sys_suspend()
 * sme_sys_resume()
 */

#define UNIFI_SME_MGT_SHORT_TIMEOUT    10000
#define UNIFI_SME_MGT_LONG_TIMEOUT     19000
#define UNIFI_SME_SYS_LONG_TIMEOUT     10000

#ifdef UNIFI_DEBUG
#define sme_wait_for_reply(priv, t)     _sme_wait_for_reply(priv, t, __func__)
#define sme_init_request(priv)          _sme_init_request(priv, __func__)
#define sme_deinit_request(priv)        _sme_deinit_request(priv, __func__)
#else
#define sme_wait_for_reply(priv, t)     _sme_wait_for_reply(priv, t, NULL)
#define sme_init_request(priv)          _sme_init_request(priv, NULL)
#define sme_deinit_request(priv)        _sme_deinit_request(priv, NULL)
#endif

static int _sme_init_request(os_linux_priv_t *priv, const char *func)
{
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : sme_init_request: Invalid priv"));
        return -EIO;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: sme_init_request: wait sem", priv->instance));

    /* Grab the SME semaphore until the reply comes, or timeout */
    if (down_interruptible(&priv->sme_sem))
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_init_request: Failed to get SME semaphore", priv->instance));
        return -EIO;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5, "unifi%d: sme_init_request: got sem: pending", priv->instance));

    priv->sme_reply.request_status = SME_REQUEST_PENDING;
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: sme_init_request: pending %s (s:%d)\n",
                        priv ? priv->instance : 0,
                        (func ? func : ""), priv->sme_reply.request_status));

    return 0;
} /* _sme_init_request() */

#ifdef CSR_SUPPORT_WEXT
/*
 * Prepares for caller to make another request, without releasing the SME semaphore
*/
static int sme_reinit_request(os_linux_priv_t *priv)
{
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi : sme_reinit_request: Invalid priv"));
        return -EIO;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6, "unifi%d: sme_reinit_request", priv->instance));

    priv->sme_reply.request_status = SME_REQUEST_PENDING;
    return 0;
}

#endif

int uf_sme_complete_request(os_linux_priv_t *priv, CsrResult reply_status, const char *func)
{
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_complete_request: Invalid priv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    if (priv->sme_reply.request_status != SME_REQUEST_PENDING)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: sme_complete_request: request not pending %s (s:%d)\n",
                               priv ? priv->instance : 0,
                               (func ? func : ""), priv->sme_reply.request_status));
        return -EINVAL;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG5,
                        "sme_complete_request: completed %s (s:%d)\n",
                        (func ? func : ""), priv->sme_reply.request_status));

    priv->sme_reply.request_status = SME_REQUEST_RECEIVED;
    priv->sme_reply.reply_status = reply_status;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: uf_sme_complete_request: wakeup %s\n",
                        priv ? priv->instance : 0, (func ? func : "")));

    wake_up_interruptible(&priv->sme_request_wq);

    return 0;
} /* uf_sme_complete_request() */

void uf_sme_cancel_request(os_linux_priv_t *priv, CsrResult reply_status)
{
    /* Check for a blocking SME request in progress, and cancel the wait.
     * This should be used when the character device is closed.
     */

    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: sme_deinit_request: Invalid priv\n",
                               priv ? priv->instance : 0));
        return;
    }

    /* If no request is pending, nothing to wake up */
    if (priv->sme_reply.request_status != SME_REQUEST_PENDING)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "sme_cancel_request: no request was pending (s:%d)\n",
                            priv->sme_reply.request_status));
        /* Nothing to do */
        return;
    }
    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "sme_cancel_request: request cancelled (s:%d)\n",
                        priv->sme_reply.request_status));

    /* Wake up the wait with an error status */
    priv->sme_reply.request_status = SME_REQUEST_CANCELLED;
    priv->sme_reply.reply_status = reply_status; /* unimportant since the CANCELLED state will fail the ioctl */

    wake_up_interruptible(&priv->sme_request_wq);
}

static int _sme_wait_for_reply(os_linux_priv_t *priv,
                               unsigned long timeout, const char *func)
{
    long r;

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: sme_wait_for_reply: sleep %s\n",
                        priv ? priv->instance : 0, (func ? func : "")));

    r = wait_event_interruptible_timeout(priv->sme_request_wq,
                                         (priv->sme_reply.request_status != SME_REQUEST_PENDING),
                                         msecs_to_jiffies(timeout));

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "unifi%d: sme_wait_for_reply: awake %s\n",
                        priv ? priv->instance : 0, (func ? func : "")));

    if (r == -ERESTARTSYS)
    {
        /* The thread was killed */
        return r;
    }
    if (priv->sme_reply.request_status == SME_REQUEST_CANCELLED)
    {
        CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                            "Cancelled waiting for SME to reply (%s s:%d, t:%d, r:%d)\n",
                            (func ? func : ""), priv->sme_reply.request_status, timeout, r));
        return -EIO; /* fail the ioctl */
    }
    if ((r == 0) && (priv->sme_reply.request_status != SME_REQUEST_RECEIVED))
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID,
                           CSR_WIFI_HIP_LOG_DEF, "unifi%d: Timeout waiting for SME to reply (%s s:%d, t:%d)\n",
                           priv ? priv->instance : 0,
                           (func ? func : ""), priv->sme_reply.request_status, timeout));
        priv->sme_reply.request_status = SME_REQUEST_TIMEDOUT;
        return -ETIMEDOUT;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG6,
                        "sme_wait_for_reply: %s received (%d)\n",
                        func ? func : "", r));

    return r < 0 ? r : 0;
} /* _sme_wait_for_reply() */

static void _sme_deinit_request(os_linux_priv_t *priv, const char *func)
{
    if (priv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: sme_deinit_request: Invalid priv\n",
                               priv ? priv->instance : 0));
        return;
    }

    /* Release the SME semaphore that was downed in sme_init_request() */
    up(&priv->sme_sem);
} /* _sme_deinit_request() */

#ifdef CSR_SUPPORT_WEXT
int sme_mgt_wifi_on(os_linux_priv_t *priv)
{
    const CsrWifiMacAddress sta_mac_address = {{0, 0, 0, 0, 0, 0}};
#ifdef CSR_SUPPORT_WEXT_AP
    int r;
    CsrResult res;
#endif

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_wifi_on: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    /* Start the SME */
#ifdef CSR_SUPPORT_WEXT_AP
    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }
#endif
    CsrWifiSmeWifiOnReqSend(0, sta_mac_address, 0, NULL);
#ifdef CSR_SUPPORT_WEXT_AP
    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_LONG_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_wifi_on: unifi_mgt_wifi_on_req <-- (r=%d, status=%d)\n",
                        priv ? priv->instance : 0, r, res));
    return convert_sme_error(res);
#else
    return 0;
#endif
} /* sme_mgt_wifi_on() */

int sme_mgt_wifi_off(os_linux_priv_t *priv)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_wifi_off: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    /* Stop the SME */
    CsrWifiSmeWifiOffReqSend(0);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_LONG_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_wifi_off: unifi_mgt_wifi_off_req <-- (r=%d, status=%d)\n",
                        priv ? priv->instance : 0, r, res));
    return convert_sme_error(res);
} /* sme_mgt_wifi_off */

int sme_mgt_key(os_linux_priv_t *priv, CsrUint16 InterfaceTag, CsrWifiSmeKey *sme_key,
                CsrWifiSmeListAction action)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_key: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeKeyReqSend(0, InterfaceTag, action, *sme_key);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    return convert_sme_error(res);
}

int sme_mgt_scan_full(os_linux_priv_t *priv,
                      CsrWifiSsid     *specific_ssid,
                      int              num_channels,
                      unsigned char   *channel_list)
{
    CsrWifiMacAddress bcastAddress = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};
    CsrBool is_active = (num_channels > 0) ? TRUE : FALSE;
    CsrUint16 *channelInfoList;
    CsrUint16 i;
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_scan_full: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,  "unifi%d: sme_mgt_scan_full: -->\n"
                           , priv ? priv->instance : 0));

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    /* If a channel list is provided, do an active scan */
    if (is_active)
    {
        CSR_LOG_TEXT_INFO((
                              CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,
                              "unifi%d: channel list - num_channels: %d, active scan\n",
                              priv ? priv->instance : 0,
                              num_channels));
    }

    channelInfoList = (CsrUint16 *) CsrPmemAlloc(num_channels * sizeof(CsrUint16));
    for (i = 0; i < num_channels; i++)
    {
        /* Convert Channel Number to ChannelInfo */
        channelInfoList[i] = channel_list[i];
    }
    CsrPmemFree(channel_list);

    CsrWifiSmeScanFullReqSend(0,
                              specific_ssid->length ? 1 : 0, /* 0 or 1 SSIDS */
                              specific_ssid,
                              bcastAddress,
                              is_active,
                              CSR_WIFI_SME_BSS_TYPE_ANY_BSS,
                              (CsrUint16) num_channels, channelInfoList,
                              priv->scanInformationElementsLength, priv->scanInformationElements);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_LONG_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_scan_full: <-- (status=%d)\n",
                        priv ? priv->instance : 0, res));
    if (res == CSR_WIFI_RESULT_UNAVAILABLE)
    {
        return 0; /* initial scan already underway */
    }
    else
    {
        return convert_sme_error(res);
    }
}

int sme_mgt_scan_results_get_async(os_linux_priv_t        *priv,
                                   struct iw_request_info *info,
                                   char                   *scan_results,
                                   long                    scan_results_len)
{
    CsrUint16 scan_result_list_count;
    CsrWifiSmeScanResult *scan_result_list;
    CsrWifiSmeScanResult *scan_result;
    int r;
    int i;
    char *current_ev = scan_results;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_scan_results_get_async: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeScanResultsGetReqSend(0);
    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_LONG_TIMEOUT);
    if (r)
    {
        sme_deinit_request(priv);
        return r;
    }

    scan_result_list_count = priv->sme_reply.reply_scan_results_count;
    scan_result_list = priv->sme_reply.reply_scan_results;
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: scan_results: Scan returned %d, numElements=%d\n",
                       priv ? priv->instance : 0,
                       r, scan_result_list_count));

    /* OK, now we have the scan results */
    for (i = 0; i < scan_result_list_count; ++i)
    {
        scan_result = &scan_result_list[i];

        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                           "unifi%d: Scan Result: %.*s\n",
                           priv ? priv->instance : 0,
                           scan_result->ssid.length,
                           scan_result->ssid.ssid));

        /* unifi_translate_scan call was hardcoded to use netdev 0 */
        r = unifi_translate_scan(priv->netdev[0], info,
                                 current_ev,
                                 scan_results + scan_results_len,
                                 scan_result, i + 1);

        if (r < 0)
        {
            CsrPmemFree(scan_result_list);
            priv->sme_reply.reply_scan_results_count = 0;
            priv->sme_reply.reply_scan_results = NULL;
            sme_deinit_request(priv);
            return r;
        }

        current_ev += r;
    }

    /*
     * Free the scan results allocated in unifi_mgt_scan_results_get_cfm()
     * and invalidate the reply_scan_results to avoid re-using
     * the freed pointers.
     */
    CsrPmemFree(scan_result_list);
    priv->sme_reply.reply_scan_results_count = 0;
    priv->sme_reply.reply_scan_results = NULL;

    sme_deinit_request(priv);

    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG2,
                       "unifi%d: scan_results: Scan translated to %d bytes\n",
                       priv ? priv->instance : 0,
                       current_ev - scan_results));
    return current_ev - scan_results;
}

int sme_mgt_connect(os_linux_priv_t *priv, CsrUint16 InterfaceTag)
{
    int disconnect_r;
    int connect_r = 0; /* initialisation is unnecessary, but avoids incorrect gcc complaints */
    CsrResult disconnect_res;
    CsrResult connect_res = 0;  /* initialisation is unnecessary, but avoids incorrect gcc complaints */

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_connect: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG2,  "unifi%d: sme_mgt_connect: %.*s\n",
                          priv ? priv->instance : 0,
                          priv->connection_config.ssid.length,
                          priv->connection_config.ssid.ssid));

    if (sme_init_request(priv))
    {
        return -EIO;
    }

    CsrWifiSmeDisconnectReqSend(0, InterfaceTag);
    disconnect_r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    disconnect_res = priv->sme_reply.reply_status; /* If it subsequently connects ok, we do not care about this status */
    if (CSR_RESULT_SUCCESS == disconnect_r)
    {
        sme_reinit_request(priv);
        CsrWifiSmeConnectReqSend(0, InterfaceTag, priv->connection_config);
        connect_r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
        connect_res = priv->sme_reply.reply_status;
    }
    sme_deinit_request(priv);

    if (CSR_RESULT_SUCCESS != disconnect_r)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_connect: timeout with CsrWifiSmeDisconnectReq", priv->instance));
        return disconnect_r;
    }

    if (CSR_RESULT_SUCCESS != connect_r)
    {
        CSR_LOG_TEXT_ERROR((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_connect: timeout with CsrWifiSmeConnectReq", priv->instance));
        return connect_r;
    }

    /* Don't report disconnect status unless connect has failed */
    if (CSR_RESULT_SUCCESS != connect_res)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: sme_mgt_connect: CsrWifiSmeDisconnectCfm had status %d", priv->instance, disconnect_res));
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: sme_mgt_connect: CsrWifiSmeConnectCfm had SME status %d", priv->instance, connect_res));
    }
    return convert_sme_error(connect_res);
}

int sme_mgt_disconnect(os_linux_priv_t *priv, CsrUint16 InterfaceTag)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_disconnect: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeDisconnectReqSend(0, InterfaceTag);
    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_disconnect: <-- (status=%d)\n",
                        priv ? priv->instance : 0, res));
    return convert_sme_error(res);
}

int sme_mgt_pmkid(os_linux_priv_t     *priv,
                  CsrUint16            InterfaceTag,
                  CsrWifiSmeListAction action,
                  CsrWifiSmePmkidList *pmkid_list)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_pmkid: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmePmkidReqSend(0, InterfaceTag, action,
                           pmkid_list->pmkidsCount, pmkid_list->pmkids);
    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_pmkid: <-- (status=%d)\n",
                        priv ? priv->instance : 0,  res));
    return convert_sme_error(res);
}

#endif /* CSR_SUPPORT_WEXT */

int sme_mgt_mib_set(os_linux_priv_t *priv, CsrUint16 interface_tag, CsrWifiDataBlock *mibData)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_set: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeSetReqSend(0, interface_tag, FALSE, mibData->dataLength, mibData->data);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_mib_set: CsrWifiSmeSetReq <-- (r=%d status=%d)\n",
                        priv ? priv->instance : 0, r, res));
    return convert_sme_error(res);
}

int sme_mgt_mib_get(os_linux_priv_t *priv, CsrUint16 InterfaceTag, CsrWifiDataBlock *mibData)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_sme_config_set: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeGetReqSend(0, InterfaceTag, mibData->dataLength, mibData->data);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    if (r)
    {
        sme_deinit_request(priv);
        return r;
    }

    res = priv->sme_reply.reply_status;
    *mibData = priv->sme_reply.mibData;
    priv->sme_reply.mibData.dataLength = 0;
    priv->sme_reply.mibData.data = NULL;
    sme_deinit_request(priv);

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_mib_get: CsrWifiSmeSmeStaConfigSetReq <-- (r=%d status=%d)\n",
                        priv ? priv->instance : 0, r, res));

    return convert_sme_error(res);
}

int sme_mgt_packet_filter_set(os_linux_priv_t *priv, CsrUint16 InterfaceTag)
{
#ifdef CSR_WIFI_DRIVER_PACKET_FILTERS_UNSAFE
    netInterface_priv_t *interfacePriv = priv->interfacePriv[InterfaceTag];
    CsrWifiIp4Address ipAddress = {{0xFF, 0xFF, 0xFF, 0xFF}};
    CsrWifiDataBlock mibData = {0, NULL};
    CsrUint8 ieLength;
    CsrUint8 *ie;
    CsrUint32 size = 0;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_packet_filter_set: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }
    if (interfacePriv->packet_filters.arp_filter)
    {
        memcpy(ipAddress.a, &interfacePriv->sta_ip_address, sizeof(ipAddress.a));
    }

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG5,
                           "unifi%d: sme_mgt_packet_filter_set: IP address %d.%d.%d.%d\n",
                           priv ? priv->instance : 0,
                           ipAddress.a[0], ipAddress.a[1],
                           ipAddress.a[2], ipAddress.a[3]));

    /* Set the filter address */
    (void) CsrWifiMibEncodeOctet(&mibData, CSR_WIFI_SME_PSID_IPV4_ADDRESS, sizeof(ipAddress.a), ipAddress.a, 0);


    /* create the filter */
    ieLength = sizeof(CsrWifiSmePacketFilterMode) + interfacePriv->packet_filters.tclas_ies_length;
    ie = CsrPmemAlloc(ieLength);

    /* add the mode */
    ie[size++] = interfacePriv->packet_filters.filter_mode;

    /* add the filters */
    CsrMemCpy(&ie[size], interfacePriv->packet_filters.tclas_ies, interfacePriv->packet_filters.tclas_ies_length);
    if (interfacePriv->packet_filters.packet_filter_type == UNIFI_CFG_FILTER_TYPE_ACTIVE_HOST)
    {
        (void) CsrWifiMibEncodeOctet(&mibData, CSR_WIFI_SME_PSID_PACKET_FILTERS_ACTIVE_HOST, ieLength, ie, 0);
    }
    else
    {
        (void) CsrWifiMibEncodeOctet(&mibData, CSR_WIFI_SME_PSID_PACKET_FILTERS_SUSPENDED_HOST, ieLength, ie, 0);
    }

    CsrPmemFree(ie);

    CsrWifiSmeSetReqSend(0, InterfaceTag, FALSE, mibData.dataLength, mibData.data);

    /* The SME API is used here is a non-blocking fashion, so the mibData.data memory needs to be freed */
    if (mibData.data != NULL)
    {
        CsrPmemFree(mibData.data);
    }
#endif
    return 0;
}

int sme_mgt_tspec(os_linux_priv_t *priv, CsrUint16 InterfaceTag, CsrWifiSmeListAction action,
                  CsrUint32 tid, CsrWifiDataBlock *tspec, CsrWifiDataBlock *tclas)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_mgt_tspec: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiSmeTspecReqSend(0, InterfaceTag,
                           action, tid, TRUE, 0,
                           tspec->dataLength, tspec->data,
                           tclas->dataLength, tclas->data);
    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_mgt_tspec: <-- (status=%d)\n",
                        priv ? priv->instance : 0,  res));
    return convert_sme_error(res);
}

int sme_sys_suspend(os_linux_priv_t *priv)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_sys_suspend: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    /* Send fake cold suspend ind to SME, which will cause it to send a wifi off req */
    priv->init_progress_save = priv->init_progress; /* for restoring init_progress after fake resume */
    priv->expected_reply = SME_REPLY_SUSPEND;
    CsrWifiRouterCtrlSuspendIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, FALSE, FALSE);
    r = sme_wait_for_reply(priv, UNIFI_SME_SYS_LONG_TIMEOUT);
    res = priv->sme_reply.reply_status;
    priv->expected_reply = SME_REPLY_NONE;
    sme_deinit_request(priv);
    if (r)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: sme_sys_suspend: SME did not reply\n",
                               priv->instance));
    }

    /* Consider UniFi to be uninitialised */
    priv->init_progress = UNIFI_INIT_NONE;

    CSR_LOG_TEXT_INFO((
                          CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG1,  "unifi%d: sme_sys_suspend: <-- (r=%d status=%d)\n",
                          priv ? priv->instance : 0,  r, res));
    return convert_sme_error(res);
}

int sme_sys_resume(os_linux_priv_t *priv)
{
    int r;
    CsrResult status;
    CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_UDBG1, "unifi%d: sme_sys_resume", priv->instance));

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_sys_resume: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }
    priv->expected_reply = SME_REPLY_RESUME;
    CsrWifiRouterCtrlResumeIndSend(priv->CSR_WIFI_SME_IFACEQUEUE, 0, FALSE);
    r = sme_wait_for_reply(priv, UNIFI_SME_SYS_LONG_TIMEOUT);

    priv->expected_reply = SME_REPLY_NONE;
    status = priv->sme_reply.reply_status; /* need to read this before releasing sema */
    sme_deinit_request(priv);

    if (r)
    {
        CSR_LOG_TEXT_INFO((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                           "unifi%d: resume: SME did not reply, return success anyway\n",
                           priv ? priv->instance : 0));
    }
    else if (status == CSR_RESULT_SUCCESS)
    {
        priv->init_progress = priv->init_progress_save;
    }
    return 0;
}

#ifdef CSR_SUPPORT_WEXT_AP
int sme_ap_stop(os_linux_priv_t *priv, CsrUint16 interface_tag)
{
    int r;
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_ap_stop: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiNmeApStopReqSend(0, interface_tag);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_ap_stop <-- (r=%d status=%d)\n",
                        priv ? priv->instance : 0, r, res));
    return convert_sme_error(res);
}

int sme_ap_start(os_linux_priv_t *priv, CsrUint16 interface_tag,
                 CsrWifiSmeApConfig_t *ap_config)
{
    int r;
    CsrWifiDataBlock buffer = {0, NULL};
    CsrResult res;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_ap_start: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    r = sme_init_request(priv);
    if (r)
    {
        return -EIO;
    }

    CsrWifiNmeApStartReqSend(0, interface_tag, CSR_WIFI_AP_TYPE_LEGACY, FALSE,
                             ap_config->ssid, ap_config->credentials,
                             CSR_WIFI_AP_ACCESS_TYPE_NONE, 0, NULL,
                             buffer.dataLength, buffer.data);

    r = sme_wait_for_reply(priv, UNIFI_SME_MGT_SHORT_TIMEOUT);
    res = priv->sme_reply.reply_status;
    sme_deinit_request(priv);
    if (r)
    {
        return r;
    }

    CSR_LOG_TEXT_DEBUG((CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                        "unifi%d: sme_ap_start <-- (r=%d status=%d)\n",
                        priv ? priv->instance : 0, r, res));
    return convert_sme_error(res);
}

int sme_ap_config(os_linux_priv_t        *priv,
                  CsrUint16               interface_tag,
                  CsrWifiWextApMacConfig *ap_mac_config,
                  CsrWifiWextApConfig    *group_security_config)
{
    int r;
    CsrWifiDataBlock mibData = {0, NULL};
    CsrUint8 *data;

    if (priv->smepriv == NULL)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: sme_ap_config: invalid smepriv\n", priv ? priv->instance : 0));
        return -EIO;
    }

    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_PHY_SUPPORTED_BITMAP, ap_mac_config->phySupportedBitmap, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_BEACON_PERIOD_TU, ap_mac_config->beaconInterval, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_DTIM_PERIOD, ap_mac_config->dtimPeriod, 0);

    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_MAX_CONNECTIONS, priv->ap_config.max_connections, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_BEACONING_CHANNEL_INFO, (priv->ap_config.channel), 0);

    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_MAX_LISTEN_INTERVAL, ap_mac_config->maxListenInterval, 0);

    if (ap_mac_config->ctsProtectionType <= CSR_WIFI_SME_CTS_PROTECTION_AUTOMATIC_NO_OLBC)
    {
        (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_SME_PSID_AP_CTS_PROTECTION, ap_mac_config->ctsProtectionType, 0);
    }
    if (ap_mac_config->wmmEnabled < 2)
    {
        (void) CsrWifiMibEncodeBool(&mibData, CSR_WIFI_SME_PSID_WMM_ENABLED, ap_mac_config->wmmEnabled, 0);
    }

    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_NME_PSID_AP_GROUPKEY_TIMEOUT, group_security_config->apGroupkeyTimeout, 0);
    (void) CsrWifiMibEncodeBool(&mibData, CSR_WIFI_NME_PSID_AP_STRICT_GTK_REKEY, group_security_config->apStrictGtkRekey, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_NME_PSID_AP_GMK_TIMEOUT, group_security_config->apGmkTimeout, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_NME_PSID_RESPONSE_TIMEOUT, group_security_config->apResponseTimeout, 0);
    (void) CsrWifiMibEncodeUint(&mibData, CSR_WIFI_NME_PSID_RETRANS_LIMIT, group_security_config->apRetransLimit, 0);

    if (ap_mac_config->wpsEnabled < 2)
    {
        (void) CsrWifiMibEncodeBool(&mibData, CSR_WIFI_SME_PSID_AP_WPS_ENABLE, ap_mac_config->wpsEnabled, 0);
    }

    data = mibData.data; /* need to free this later */
    r = sme_mgt_mib_set(priv, interface_tag, &mibData);
    CsrPmemFree(data);

    CSR_LOG_TEXT_DEBUG((
                           CSR_WIFI_HIP_LOG_ID,  CSR_WIFI_HIP_UDBG4,
                           "unifi%d: sme_ap_config <-- (r=%d status=%d)\n",
                           priv ? priv->instance : 0,
                           r, priv->sme_reply.reply_status));
    return r;
}

#endif
