/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2013
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_wifi_hip_log_text.h"

/**
 * The HIP lib OS abstraction consists of the implementation
 * of the functions in this file. It is part of the porting exercise.
 */
#include "os_linux_priv.h"
#include "unifi_os.h"


/*
 * ---------------------------------------------------------------------------
 *  os_linux_net_data_malloc
 *
 *      Allocate an OS specific net data buffer of "size" bytes.
 *      The bulk_data_slot.os_data_ptr must be initialised to point
 *      to the buffer allocated. The bulk_data_slot.length must be
 *      initialised to the requested size, zero otherwise.
 *      The bulk_data_slot.os_net_buf_ptr can be initialised to
 *      an OS specific pointer to be used in the unifi_net_data_free().
 *
 *
 *  Arguments:
 *      ospriv              Pointer to device private context struct.
 *      bulk_data_slot      Pointer to the bulk data structure to initialise.
 *      size                Size of the buffer to be allocated.
 *
 *  Returns:
 *      CSR_RESULT_SUCCESS on success, CSR_RESULT_FAILURE otherwise.
 * ---------------------------------------------------------------------------
 */
CsrResult os_linux_net_data_malloc(void *ospriv, CsrWifiHipBulkDataDesc *bulk_data_slot, unsigned int size)
{
    struct sk_buff *skb;
    os_linux_priv_t *priv = (os_linux_priv_t *) ospriv;
    int rounded_length;

    if (priv->sdio_block_size == 0)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF, "unifi%d: os_linux_net_data_malloc: Invalid SDIO block size\n", priv ? priv->instance : 0));
        return CSR_RESULT_FAILURE;
    }

    rounded_length = (size + priv->sdio_block_size - 1) & ~(priv->sdio_block_size - 1);

    /*
     * (ETH_HLEN + 2) bytes tailroom for header manipulation
     * CSR_WIFI_ALIGN_BYTES bytes headroom for alignment manipulation
     */

    /* ASCH-DL_20140821 OIL 147: countermeasure for CSR Nullpointer exception issue. */
    /* skb = dev_alloc_skb(rounded_length + 2 + ETH_HLEN + CSR_WIFI_ALIGN_BYTES); */
    skb = __dev_alloc_skb(rounded_length + 2 + ETH_HLEN + CSR_WIFI_ALIGN_BYTES,GFP_KERNEL);

    if (!skb)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID,
                               CSR_WIFI_HIP_LOG_DEF,
                               "unifi : alloc_skb failed.\n"));
        bulk_data_slot->os_net_buf_ptr = NULL;
        bulk_data_slot->net_buf_length = 0;
        bulk_data_slot->os_data_ptr = NULL;
        bulk_data_slot->data_length = 0;
        return CSR_RESULT_FAILURE;
    }

    bulk_data_slot->os_net_buf_ptr = (const void *) skb;
    bulk_data_slot->net_buf_length = rounded_length + 2 + ETH_HLEN + CSR_WIFI_ALIGN_BYTES;
    bulk_data_slot->os_data_ptr = (CsrUint8 *) skb->data;
    bulk_data_slot->data_length = size;

    return CSR_RESULT_SUCCESS;
} /*  os_linux_net_data_malloc() */

/*
 * ---------------------------------------------------------------------------
 *  os_linux_net_data_free
 *
 *      Free an OS specific net data buffer.
 *      The bulk_data_slot.length must be initialised to 0.
 *
 *
 *  Arguments:
 *      ospriv              Pointer to device private context struct.
 *      bulk_data_slot      Pointer to the bulk data structure that
 *                          holds the data to be freed.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void os_linux_net_data_free(void *ospriv, CsrWifiHipBulkDataDesc *bulk_data_slot)
{
    struct sk_buff *skb;

    skb = (struct sk_buff *) bulk_data_slot->os_net_buf_ptr;
    if (!skb || !bulk_data_slot->data_length)
    {
        CSR_LOG_TEXT_CRITICAL((CSR_WIFI_HIP_LOG_ID, CSR_WIFI_HIP_LOG_DEF,
                               "unifi%d: os_linux_net_data_free: socket buffer pointer is 0x%p (0x%p) or length=%d\n",
                               ospriv ? ((os_linux_priv_t *) ospriv)->instance : 0, skb, bulk_data_slot->os_data_ptr,
                               bulk_data_slot->data_length));
        return;
    }
    dev_kfree_skb(skb);

    bulk_data_slot->net_buf_length = 0;
    bulk_data_slot->data_length = 0;
    bulk_data_slot->os_data_ptr = NULL;
    bulk_data_slot->os_net_buf_ptr = NULL;

}

/*
 * ---------------------------------------------------------------------------
 *  os_linux_net_data_free_all
 *
 *      Free all the OS specific net data buffers provided in a bulkdata set.
 *
 *
 *  Arguments:
 *      ospriv              Pointer to device private context struct.
 *      bulk_data           Pointer to the bulk data structure that
 *                          holds the data to be freed.
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
void os_linux_net_data_free_all(void *ospriv, CsrWifiHipBulkDataParam *bulk_data)
{
    int i;
    struct sk_buff *skb;
    CSR_UNUSED(ospriv);

    for (i = 0; i < CSR_WIFI_HIP_MAX_DATA_REFERENCES; i++)
    {
        skb = (struct sk_buff *) bulk_data->d[i].os_net_buf_ptr;
        if (skb)
        {
            dev_kfree_skb(skb);
        }

        bulk_data->d[i].net_buf_length = 0;
        bulk_data->d[i].data_length = 0;
        bulk_data->d[i].os_data_ptr = NULL;
        bulk_data->d[i].os_net_buf_ptr = NULL;

    }
}

#ifdef ANDROID_TIMESTAMP
static volatile unsigned int printk_cpu = UINT_MAX;
char tbuf[30];

char *print_time(void)
{
    unsigned long long t;
    unsigned long nanosec_rem;

    t = cpu_clock(printk_cpu);
    nanosec_rem = do_div(t, 1000000000);
    sprintf(tbuf, "[%5lu.%06lu] ",
            (unsigned long) t,
            nanosec_rem / 1000);

    return tbuf;
}

#endif


/*
 * ---------------------------------------------------------------------------
 *
 *      Debugging support.
 *
 * ---------------------------------------------------------------------------
 */

#ifdef UNIFI_DEBUG
#endif /* UNIFI_DEBUG */
/* ---------------------------------------------------------------------------
 *                              - End -
 * ------------------------------------------------------------------------- */
