/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2012
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"
#include "csr_types.h"
#include "csr_prim_defs.h"
#include "csr_msgconv.h"
#include "csr_util.h"
#include "csr_pmem.h"
#include "csr_lib.h"

void CsrUint8Des(CsrUint8 *value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = buffer[*offset];
    *offset += sizeof(*value);
}

void CsrUint16Des(CsrUint16 *value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = (buffer[*offset + 0] << 0) |
             (buffer[*offset + 1] << 8);
    *offset += sizeof(*value);
}

void CsrUint32Des(CsrUint32 *value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = (buffer[*offset + 0] << 0) |
             (buffer[*offset + 1] << 8) |
             (buffer[*offset + 2] << 16) |
             (buffer[*offset + 3] << 24);
    *offset += sizeof(*value);
}

void CsrMemCpyDes(void *value, CsrUint8 *buffer, CsrSize *offset, CsrSize length)
{
    CsrMemCpy(value, &buffer[*offset], length);
    *offset += length;
}

void CsrCharStringDes(CsrCharString **value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = CsrStrDup((CsrCharString *) &buffer[*offset]);
    *offset += CsrStrLen(*value) + 1;
}

void CsrUtf8StringDes(CsrUtf8String **value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = (CsrUtf8String *) CsrStrDup((CsrCharString *) &buffer[*offset]);
    *offset += CsrStrLen((CsrCharString *) *value) + 1;
}

void CsrUtf16StringDes(CsrUtf16String **value, CsrUint8 *buffer, CsrSize *offset)
{
    CsrUint32 length, i;

    CsrUint32Des(&length, buffer, offset);

    *value = CsrPmemAlloc(length * sizeof(**value));
    for (i = 0; i < length; i++)
    {
        CsrUint16Des(&(*value)[i], buffer, offset);
    }
}

void CsrSizeDes(CsrSize *value, CsrUint8 *buffer, CsrSize *offset)
{
    *value = (buffer[*offset + 0] << 0) |
             (buffer[*offset + 1] << 8) |
             (buffer[*offset + 2] << 16) |
             (buffer[*offset + 3] << 24);
    *offset += sizeof(*value);
}

void CsrVoidPtrDes(void **value, CsrUint8 *buffer, CsrSize *offset)
{
    CsrSizeDes((CsrSize *) value, buffer, offset);
}

void CsrUint8Ser(CsrUint8 *buffer, CsrSize *offset, CsrUint8 value)
{
    buffer[*offset] = value;
    *offset += sizeof(value);
}

void CsrUint16Ser(CsrUint8 *buffer, CsrSize *offset, CsrUint16 value)
{
    buffer[*offset + 0] = (CsrUint8) ((value >> 0) & 0xFF);
    buffer[*offset + 1] = (CsrUint8) ((value >> 8) & 0xFF);
    *offset += sizeof(value);
}

void CsrUint32Ser(CsrUint8 *buffer, CsrSize *offset, CsrUint32 value)
{
    buffer[*offset + 0] = (CsrUint8) ((value >> 0) & 0xFF);
    buffer[*offset + 1] = (CsrUint8) ((value >> 8) & 0xFF);
    buffer[*offset + 2] = (CsrUint8) ((value >> 16) & 0xFF);
    buffer[*offset + 3] = (CsrUint8) ((value >> 24) & 0xFF);
    *offset += sizeof(value);
}

void CsrMemCpySer(CsrUint8 *buffer, CsrSize *offset, const void *value, CsrSize length)
{
    CsrMemCpy(&buffer[*offset], value, length);
    *offset += length;
}

void CsrCharStringSer(CsrUint8 *buffer, CsrSize *offset, const CsrCharString *value)
{
    if (value)
    {
        CsrSize valueLenZero = CsrStrLen(value) + 1;
        CsrStrNCpy(((CsrCharString *) &buffer[*offset]), value, valueLenZero);
        *offset += valueLenZero;
    }
    else
    {
        CsrUint8Ser(buffer, offset, 0);
    }
}

void CsrUtf8StringSer(CsrUint8 *buffer, CsrSize *offset, const CsrUtf8String *value)
{
    CsrCharStringSer(buffer, offset, (CsrCharString *) value);
}

void CsrUtf16StringSer(CsrUint8 *buffer, CsrSize *offset, const CsrUtf16String *value)
{
    if (value)
    {
        CsrUint32 length = CsrUtf16StrLen(value) + 1;
        CsrUint32 i;

        CsrUint32Ser(buffer, offset, length);

        for (i = 0; i < length; i++)
        {
            CsrUint16Ser(buffer, offset, (CsrUint16) value[i]);
        }
    }
    else
    {
        CsrUint32Ser(buffer, offset, 0);
    }
}

void CsrSizeSer(CsrUint8 *buffer, CsrSize *offset, CsrSize value)
{
    buffer[*offset + 0] = (CsrUint8) ((value >> 0) & 0xFF);
    buffer[*offset + 1] = (CsrUint8) ((value >> 8) & 0xFF);
    buffer[*offset + 2] = (CsrUint8) ((value >> 16) & 0xFF);
    buffer[*offset + 3] = (CsrUint8) ((value >> 24) & 0xFF);
    *offset += sizeof(value);
}

void CsrVoidPtrSer(CsrUint8 *buffer, CsrSize *offset, void *ptr)
{
    CsrSizeSer(buffer, offset, (CsrSize) ptr);
}

CsrUint32 CsrCharStringSerLen(const CsrCharString *str)
{
    if (str)
    {
        return (CsrUint32) (CsrStrLen(str) + sizeof(*str));
    }
    else
    {
        return sizeof(*str);
    }
}

CsrUint32 CsrUtf8StringSerLen(const CsrUtf8String *str)
{
    if (str)
    {
        return (CsrUint32) (CsrStrLen((CsrCharString *) str) + sizeof(*str));
    }
    else
    {
        return sizeof(*str);
    }
}

CsrUint32 CsrUtf16StringSerLen(const CsrUtf16String *str)
{
    if (str)
    {
        /* We always write down the length of the string */
        return sizeof(CsrUint32) + (CsrUtf16StrLen(str) + 1) * sizeof(*str);
    }
    else
    {
        return sizeof(CsrUint32);
    }
}

CsrSize CsrMessageSizeof(void *msg)
{
    return 2;
}

CsrUint8 *CsrMessageSer(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessage *primitive = (CsrMessage *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    return ptr;
}

void *CsrMessageDes(CsrUint8 *buffer, CsrSize length)
{
    CsrMessage *primitive = (CsrMessage *) CsrPmemAlloc(sizeof(CsrMessage));
    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint8Sizeof(void *msg)
{
    return 3;
}

CsrUint8 *CsrMessageCsrUint8Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint8 *primitive = (CsrMessageCsrUint8 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint8Ser(ptr, len, primitive->value);
    return ptr;
}

void *CsrMessageCsrUint8Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint8 *primitive = (CsrMessageCsrUint8 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint8));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint8Des(&primitive->value, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint16Sizeof(void *msg)
{
    return 4;
}

CsrUint8 *CsrMessageCsrUint16Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint16 *primitive = (CsrMessageCsrUint16 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint16Ser(ptr, len, primitive->value);
    return ptr;
}

void *CsrMessageCsrUint16Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint16 *primitive = (CsrMessageCsrUint16 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint16));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint16Des(&primitive->value, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint32Sizeof(void *msg)
{
    return 6;
}

CsrUint8 *CsrMessageCsrUint32Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint32 *primitive = (CsrMessageCsrUint32 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint32Ser(ptr, len, primitive->value);
    return ptr;
}

void *CsrMessageCsrUint32Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint32 *primitive = (CsrMessageCsrUint32 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint32));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint32Des(&primitive->value, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint16CsrUint8Sizeof(void *msg)
{
    return 5;
}

CsrUint8 *CsrMessageCsrUint16CsrUint8Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint16CsrUint8 *primitive = (CsrMessageCsrUint16CsrUint8 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint16Ser(ptr, len, primitive->value1);
    CsrUint8Ser(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint16CsrUint8Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint16CsrUint8 *primitive = (CsrMessageCsrUint16CsrUint8 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint16CsrUint8));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint16Des(&primitive->value1, buffer, &offset);
    CsrUint8Des(&primitive->value2, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint16CsrUint16Sizeof(void *msg)
{
    return 6;
}

CsrUint8 *CsrMessageCsrUint16CsrUint16Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint16CsrUint16 *primitive = (CsrMessageCsrUint16CsrUint16 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint16Ser(ptr, len, primitive->value1);
    CsrUint16Ser(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint16CsrUint16Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint16CsrUint16 *primitive = (CsrMessageCsrUint16CsrUint16 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint16CsrUint16));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint16Des(&primitive->value1, buffer, &offset);
    CsrUint16Des(&primitive->value2, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint16CsrUint32Sizeof(void *msg)
{
    return 8;
}

CsrUint8 *CsrMessageCsrUint16CsrUint32Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint16CsrUint32 *primitive = (CsrMessageCsrUint16CsrUint32 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint16Ser(ptr, len, primitive->value1);
    CsrUint32Ser(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint16CsrUint32Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint16CsrUint32 *primitive = (CsrMessageCsrUint16CsrUint32 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint16CsrUint32));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint16Des(&primitive->value1, buffer, &offset);
    CsrUint32Des(&primitive->value2, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint16CsrCharStringSizeof(void *msg)
{
    CsrMessageCsrUint16CsrCharString *primitive = (CsrMessageCsrUint16CsrCharString *) msg;
    return 4 + CsrStrLen(primitive->value2) + 1;
}

CsrUint8 *CsrMessageCsrUint16CsrCharStringSer(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint16CsrCharString *primitive = (CsrMessageCsrUint16CsrCharString *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint16Ser(ptr, len, primitive->value1);
    CsrCharStringSer(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint16CsrCharStringDes(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint16CsrCharString *primitive = (CsrMessageCsrUint16CsrCharString *) CsrPmemAlloc(sizeof(CsrMessageCsrUint16CsrCharString));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint16Des(&primitive->value1, buffer, &offset);
    CsrCharStringDes(&primitive->value2, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint32CsrUint16Sizeof(void *msg)
{
    return 8;
}

CsrUint8 *CsrMessageCsrUint32CsrUint16Ser(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint32CsrUint16 *primitive = (CsrMessageCsrUint32CsrUint16 *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint32Ser(ptr, len, primitive->value1);
    CsrUint16Ser(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint32CsrUint16Des(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint32CsrUint16 *primitive = (CsrMessageCsrUint32CsrUint16 *) CsrPmemAlloc(sizeof(CsrMessageCsrUint32CsrUint16));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint32Des(&primitive->value1, buffer, &offset);
    CsrUint16Des(&primitive->value2, buffer, &offset);

    return primitive;
}

CsrSize CsrMessageCsrUint32CsrCharStringSizeof(void *msg)
{
    CsrMessageCsrUint32CsrCharString *primitive = (CsrMessageCsrUint32CsrCharString *) msg;
    return 6 + CsrStrLen(primitive->value2) + 1;
}

CsrUint8 *CsrMessageCsrUint32CsrCharStringSer(CsrUint8 *ptr, CsrSize *len, void *msg)
{
    CsrMessageCsrUint32CsrCharString *primitive = (CsrMessageCsrUint32CsrCharString *) msg;
    *len = 0;
    CsrUint16Ser(ptr, len, primitive->type);
    CsrUint32Ser(ptr, len, primitive->value1);
    CsrCharStringSer(ptr, len, primitive->value2);
    return ptr;
}

void *CsrMessageCsrUint32CsrCharStringDes(CsrUint8 *buffer, CsrSize length)
{
    CsrMessageCsrUint32CsrCharString *primitive = (CsrMessageCsrUint32CsrCharString *) CsrPmemAlloc(sizeof(CsrMessageCsrUint32CsrCharString));

    CsrSize offset = 0;
    CsrUint16Des(&primitive->type, buffer, &offset);
    CsrUint32Des(&primitive->value1, buffer, &offset);
    CsrCharStringDes(&primitive->value2, buffer, &offset);

    return primitive;
}
