/*****************************************************************************

        Copyright Cambridge Silicon Radio Limited 2012
        All rights reserved

        Refer to LICENSE.txt included with this source for details
        on the license terms.

*****************************************************************************/

#include "csr_synergy.h"

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/kthread.h>
#include <linux/timer.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 34)
#include <linux/slab.h>
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 19)
#include <linux/freezer.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 27)
#include <asm/semaphore.h>
#else
#include <linux/semaphore.h>
#endif

#include <linux/bitops.h>

#include "csr_types.h"
#include "csr_framework_ext.h"
#include "csr_panic.h"

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrEventCreate
 *
 *  DESCRIPTION
 *      Creates an event and returns a handle to the created event.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS             in case of success
 *          CSR_FE_RESULT_NO_MORE_EVENTS   in case of out of event resources
 *          CSR_FE_RESULT_INVALID_POINTER  in case the eventHandle pointer is invalid
 *----------------------------------------------------------------------------*/
CsrResult CsrEventCreate(CsrEventHandle *eventHandle)
{
    if (eventHandle == NULL)
    {
        return CSR_FE_RESULT_INVALID_HANDLE;
    }

    init_waitqueue_head(&eventHandle->wakeup_q);
    eventHandle->wakeup_flag = 0;

    spin_lock_init(&eventHandle->lock);
    eventHandle->lock_flag = 0;

    return CSR_RESULT_SUCCESS;
}


/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrEventSet
 *
 *  DESCRIPTION
 *      Set an event.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS              in case of success
 *          CSR_FE_RESULT_INVALID_HANDLE       in case the eventHandle is invalid
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrEventSet(CsrEventHandle *eventHandle, CsrUint32 eventBits)
{
    if (eventHandle == NULL)
    {
        return CSR_FE_RESULT_INVALID_HANDLE;
    }

    spin_lock_irqsave(&eventHandle->lock, eventHandle->lock_flag);
    eventHandle->wakeup_flag |= eventBits;
    spin_unlock_irqrestore(&eventHandle->lock, eventHandle->lock_flag);
    wake_up_interruptible(&eventHandle->wakeup_q);

    return CSR_RESULT_SUCCESS;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrEventWait
 *
 *  DESCRIPTION
 *      Wait for the event to be set.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS                 in case of success
 *          CSR_FE_RESULT_TIMEOUT              in case of timeout
 *          CSR_FE_RESULT_INVALID_HANDLE       in case the eventHandle is invalid
 *          CSR_FE_RESULT_INVALID_POINTER      in case the eventBits pointer is invalid
 *----------------------------------------------------------------------------*/
CsrResult CsrEventWait(CsrEventHandle *eventHandle, CsrUint16 timeoutInMs, CsrUint32 *eventBits)
{
    int ret = 0;
    unsigned int timeout = 0;
    CsrResult result;

    if (eventHandle == NULL)
    {
        return CSR_FE_RESULT_INVALID_HANDLE;
    }

    if (eventBits == NULL)
    {
        return CSR_FE_RESULT_INVALID_POINTER;
    }

    spin_lock_irqsave(&eventHandle->lock, eventHandle->lock_flag);
    if ((eventHandle->wakeup_flag == 0) && (timeoutInMs != 0))
    {
        if (timeoutInMs != CSR_EVENT_WAIT_INFINITE)
        {
            timeout = msecs_to_jiffies(timeoutInMs);
            spin_unlock_irqrestore(&eventHandle->lock, eventHandle->lock_flag);
            do
            {
                CsrUint32 start_wait = jiffies;
                ret = wait_event_interruptible_timeout(eventHandle->wakeup_q,
                                                       eventHandle->wakeup_flag != 0,
                                                       timeout);
                if (ret == -ERESTARTSYS)
                {
                    CsrUint32 end_wait = jiffies;
                    CsrUint32 wait_diff = end_wait - start_wait;
                    if (timeout > wait_diff)
                    {
                        timeout -= wait_diff;
                    }
                    else
                    {
                        break;
                    }
                }
            } while (ret == -ERESTARTSYS);

            if (ret > 0)
            {
            }
            else if (ret == 0)
            {
            }
            else
            {
            }
        }
        else
        {
            spin_unlock_irqrestore(&eventHandle->lock, eventHandle->lock_flag);
            do
            {
                ret = wait_event_interruptible(eventHandle->wakeup_q,
                                               eventHandle->wakeup_flag != 0);
                if (ret == -ERESTARTSYS)
                {
                }
            } while (ret == -ERESTARTSYS);

            if (!ret)
            {
            }
            else if (ret != -ERESTARTSYS)
            {
            }
        }

        spin_lock_irqsave(&eventHandle->lock, eventHandle->lock_flag);
    }

    result = (eventHandle->wakeup_flag == 0) ? CSR_FE_RESULT_TIMEOUT : CSR_RESULT_SUCCESS;
    *eventBits = eventHandle->wakeup_flag;
    eventHandle->wakeup_flag = 0;
    spin_unlock_irqrestore(&eventHandle->lock, eventHandle->lock_flag);


    return result;
}

void CsrEventDestroy(CsrEventHandle *eventHandle)
{
    if (eventHandle == NULL)
    {
        return;
    }

}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMutexCreate
 *
 *  DESCRIPTION
 *      Create a mutex and return a handle to the created mutex.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS           in case of success
 *          CSR_FE_RESULT_NO_MORE_MUTEXES   in case of out of mutex resources
 *          CSR_FE_RESULT_INVALID_POINTER   in case the mutexHandle pointer is invalid
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrMutexCreate(CsrMutexHandle *mutexHandle)
{
    if (mutexHandle == NULL)
    {
        return CSR_FE_RESULT_INVALID_POINTER;
    }

    sema_init(mutexHandle, 1);

    return CSR_RESULT_SUCCESS;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMutexDestroy
 *
 *  DESCRIPTION
 *      Destroy the previously created mutex.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrMutexDestroy(CsrMutexHandle *mutexHandle)
{
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMutexLock
 *
 *  DESCRIPTION
 *      Lock the mutex refered to by the provided handle.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS           in case of success
 *          CSR_FE_RESULT_INVALID_HANDLE    in case the mutexHandle is invalid
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrMutexLock(CsrMutexHandle *mutexHandle)
{
    CsrResult rv;

    if (mutexHandle == NULL)
    {
        rv = CSR_FE_RESULT_INVALID_POINTER;
    }
    else if (down_killable(mutexHandle))
    {
        rv = CSR_RESULT_FAILURE;
    }
    else
    {
        rv = CSR_RESULT_SUCCESS;
    }

    return rv;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMutexUnlock
 *
 *  DESCRIPTION
 *      Unlock the mutex refered to by the provided handle.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS           in case of success
 *          CSR_FE_RESULT_INVALID_HANDLE    in case the mutexHandle is invalid
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrMutexUnlock(CsrMutexHandle *mutexHandle)
{
    if (mutexHandle == NULL)
    {
        return CSR_FE_RESULT_INVALID_POINTER;
    }

    up(mutexHandle);

    return CSR_RESULT_SUCCESS;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrThreadCreate
 *
 *  DESCRIPTION
 *      Create thread function and return a handle to the created thread.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS           in case of success
 *          CSR_FE_RESULT_NO_MORE_THREADS   in case of out of thread resources
 *          CSR_FE_RESULT_INVALID_POINTER   in case one of the supplied pointers is invalid
 *          CSR_RESULT_FAILURE           otherwise
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrThreadCreate(void (*threadFunction)(void *pointer), void *pointer,
                          CsrUint32 stackSize, CsrUint16 priority, const CsrCharString *threadName,
                          CsrThreadHandle *threadHandle)
{
    if ((threadFunction == NULL) || (threadHandle == NULL))
    {
        return CSR_FE_RESULT_INVALID_POINTER;
    }

    *threadHandle = kthread_create((int (*)(void *))threadFunction, pointer, threadName);
    if (IS_ERR(*threadHandle))
    {
        *threadHandle = NULL;
        return CSR_FE_RESULT_NO_MORE_THREADS;
    }

    /* Set the priority of the thread */
    if ((priority & CSR_THREAD_POLICY_REALTIME) != 0)
    {
        const int pr[] = {99, 75, 50, 25, 1};
        struct sched_param sched_param;
        sched_param.sched_priority = pr[priority & ~CSR_THREAD_POLICY_REALTIME];
        (void) sched_setscheduler(*threadHandle, SCHED_RR, &sched_param);
    }
    else
    {
        const int pr[] = {-20, -10, 0, 10, 19};
        (void) set_user_nice(*threadHandle, pr[priority]);
    }

    /* Setting the stack size doesn't apply so start the thread now */
    wake_up_process(*threadHandle);

    return CSR_RESULT_SUCCESS;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrThreadSleep
 *
 *  DESCRIPTION
 *      Sleep for a given period.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrThreadSleep(CsrUint16 sleepTimeInMs)
{
    unsigned long t;

    /* Convert t in ms to jiffies and round up */
    t = ((sleepTimeInMs * HZ) + 999) / 1000;
    schedule_timeout_uninterruptible(t);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMemCalloc
 *
 *  DESCRIPTION
 *      Allocate dynamic memory of a given size calculated as the
 *      numberOfElements times the elementSize.
 *
 *  RETURNS
 *      Pointer to allocated memory, or NULL in case of failure.
 *      Allocated memory is zero initialised.
 *
 *----------------------------------------------------------------------------*/
void *CsrMemCalloc(CsrSize numberOfElements, CsrSize elementSize)
{
    void *buf;
    size_t size;

    size = numberOfElements * elementSize;

    buf = kmalloc(size, GFP_KERNEL);
    if (buf != NULL)
    {
        memset(buf, 0, size);
    }

    return buf;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMemAlloc
 *
 *  DESCRIPTION
 *      Allocate dynamic memory of a given size.
 *
 *  RETURNS
 *      Pointer to allocated memory, or NULL in case of failure.
 *      Allocated memory is not initialised.
 *
 *----------------------------------------------------------------------------*/
void *CsrMemAlloc(CsrSize size)
{
    return kmalloc(size, GFP_KERNEL);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMemFree
 *
 *  DESCRIPTION
 *      Free dynamic allocated memory.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrMemFree(void *pointer)
{
    kfree(pointer);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMemAllocDma
 *
 *  DESCRIPTION
 *      Allocate DMA capable dynamic memory of a given size.
 *
 *  RETURNS
 *      Pointer to allocated memory, or NULL in case of failure.
 *      Allocated memory is not initialised.
 *
 *----------------------------------------------------------------------------*/
void *CsrMemAllocDma(CsrSize size)
{
    return kmalloc(size, GFP_KERNEL | GFP_DMA);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrMemFreeDma
 *
 *  DESCRIPTION
 *      Free DMA capable dynamic allocated memory.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrMemFreeDma(void *pointer)
{
    kfree(pointer);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrTimerCreate
 *
 *  DESCRIPTION
 *      Create a timer.
 *
 *  RETURNS
 *      Possible values:
 *          CSR_RESULT_SUCCESS              in case of success
 *          CSR_FE_RESULT_NO_MORE_TIMERS    in case of out of timer resources
 *
 *----------------------------------------------------------------------------*/
CsrResult CsrTimerCreate(void (*expirationFunction)(void *pointer), void *pointer, CsrTimerHandle *timerHandle)
{
    struct timer_list *timer = (struct timer_list *) timerHandle;

    init_timer(timer);

    /* Note: assumes sizeof(unsigned long) == sizeof(void *) - this is common in the kernel */
    timer->function = (void (*)(unsigned long)) expirationFunction;
    timer->data = (unsigned long) pointer;

    return CSR_RESULT_SUCCESS;
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrTimerStart
 *
 *  DESCRIPTION
 *      Start or restart a timer previously created with CsrTimerCreate.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrTimerStart(CsrTimerHandle *timerHandle, CsrUint32 timeoutInUs)
{
    struct timer_list *timer = (struct timer_list *) timerHandle;
    (void) mod_timer(timer, (jiffies + usecs_to_jiffies(timeoutInUs) + 1));
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrTimerStop
 *
 *  DESCRIPTION
 *      Stop a timer previously started with CsrTimerStart. If the timer has
 *      already been stopped, this will have no effect.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrTimerStop(CsrTimerHandle *timerHandle)
{
    struct timer_list *timer = (struct timer_list *) timerHandle;
    (void) del_timer_sync(timer);
}

/*----------------------------------------------------------------------------*
 *  NAME
 *      CsrTimerDestroy
 *
 *  DESCRIPTION
 *      Destroy a timer previously created with CsrTimerCreate.
 *
 *  RETURNS
 *      void
 *
 *----------------------------------------------------------------------------*/
void CsrTimerDestroy(CsrTimerHandle *timerHandle)
{
    struct timer_list *timer = (struct timer_list *) timerHandle;
    (void) del_timer_sync(timer);
    timer->function = NULL;
    timer->data = 0;
}
