/*
 * Copyright (C) 2012-2013 CETITEC GmbH. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/swab.h>

/* #include <linux/iram_alloc.h> */

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/initval.h>
#include <sound/pcm_params.h>

#include "mxc_mlb.h"
#include "mxc_mlb_ext.h"



#define BUF_MAX       (SYNC_BUFFER_DEP*32)
#define STOP_TIMEOUT_MS  50

/* struct to describe the "sound card" */
struct snd_mlb_chip {
	struct snd_card *card;
	struct snd_pcm *pcm[MLB_MAX_SYNC_DEVICES];

	unsigned int c_cur_period[MLB_MAX_SYNC_DEVICES];
	unsigned int p_cur_period[MLB_MAX_SYNC_DEVICES];

	atomic_t stop_requested[MLB_MAX_SYNC_DEVICES];
	int stop_irq_done[MLB_MAX_SYNC_DEVICES];
	wait_queue_head_t stop_wq[MLB_MAX_SYNC_DEVICES];
};

/* ALSA/platform device stuff; see mxc_mlb_syncsnd_init()
 at the bottom of this file */
static int index[SNDRV_CARDS] = SNDRV_DEFAULT_IDX;      /* Index 0-MAX */
static char *id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;       /* ID for this card */
static int enable[SNDRV_CARDS] = SNDRV_DEFAULT_ENABLE;
static struct platform_device *devices[SNDRV_CARDS];

/* helper to convert Big Endian <-> Little Endian */
static inline void swap_bytes(u32 *data, unsigned int bytes)
{
	unsigned int i;
	for (i = 0; i < bytes/4; i++)
		swahb32s(&data[i]);
}

/* This function is called on interrupts (Buffer Done) *
 * as a callback from mlb_rx_isr() in mxc_mlb.c        */
static void capture_rx_done(u32 *rpos, struct snd_pcm_substream *substream)
{
	unsigned long offset;
	unsigned int period_bytes;

	struct snd_mlb_chip *chip;
	struct snd_pcm *pcm;
	struct snd_pcm_runtime *runtime;

	chip = snd_pcm_substream_chip(substream);
	pcm = substream->pcm;
	runtime = substream->runtime;

	period_bytes = frames_to_bytes(runtime, runtime->period_size);
	offset = (chip->c_cur_period[pcm->device] % runtime->periods)
		 * period_bytes;

	/* only big endian from MOST */
	/* dma_area + offset is always 4-byte-aligned;
	   period_bytes is a multiple of 4 */
	if (snd_pcm_format_little_endian(runtime->format))
		swap_bytes((u32 *)(runtime->dma_area + offset), period_bytes);

	/* update current period */
	chip->c_cur_period[pcm->device]++;
	*rpos = chip->c_cur_period[pcm->device] % runtime->periods;

	/* notify ALSA */
	snd_pcm_period_elapsed(substream);
}

/*
 * This function is called on interrupts (Buffer Done)
 * as a callback from ext_start_tx() in mxc_mlb_ext.c
 */
static void playback_tx_done(u32 *phy_addr, struct snd_pcm_substream *substream)
{
	unsigned long offset;
	unsigned int period_bytes;

	struct snd_mlb_chip *chip;
	struct snd_pcm *pcm;
	struct snd_pcm_runtime *runtime;

	chip = snd_pcm_substream_chip(substream);
	pcm = substream->pcm;
	runtime = substream->runtime;

	if (!atomic_read(&chip->stop_requested[pcm->device])) {
		period_bytes = frames_to_bytes(runtime, runtime->period_size);

		/* update current period */
		chip->p_cur_period[pcm->device]++;
		/* notify ALSA */
		snd_pcm_period_elapsed(substream);
		/* NOTE: on last buffer, trigger(stop) will be called from
		 * snd_pcm_period_elapsed() and set chip->stop_requested */

		if (atomic_read(&chip->stop_requested[pcm->device])) {
			/* stop_requested: write zero-buffer */
			pr_debug("playback_tx_done: stopped after not being stopped\n");
			memset(runtime->dma_area, 0, period_bytes);
			*phy_addr = runtime->dma_addr;
		} else {
			/* "regular" playback */
			offset = (chip->p_cur_period[pcm->device]
				% runtime->periods) * period_bytes;

			/* only big endian to MOST */
			/* dma_area + offset is always 4-byte-aligned;
			   period_bytes is a multiple of 4 */
			if (snd_pcm_format_little_endian(runtime->format))
				swap_bytes((u32 *)(runtime->dma_area + offset),
					   period_bytes);

			*phy_addr = runtime->dma_addr + offset;
		}
	} else {
		pr_debug("playback_tx_done: stopped immediately\n");
		/* close channel to avoid beep at the end */
		syncsound_chan_shutdown(pcm->device);
		chip->stop_irq_done[pcm->device] = 1;
		/* wake up waiting hw_free() call */
		wake_up(&chip->stop_wq[pcm->device]);
		snd_pcm_period_elapsed(substream);
	}
}

/* this is called once on start of playback */
static u32 playback_get_buffer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime;
	unsigned int period_bytes;

	runtime = substream->runtime;
	period_bytes = frames_to_bytes(runtime, runtime->period_size);

	if (snd_pcm_format_little_endian(runtime->format))
		swap_bytes((u32 *)(runtime->dma_area), period_bytes);

	/* offset is always 0 here */
	return runtime->dma_addr;
}

/* HW description (playback) for ALSA */
static struct snd_pcm_hardware snd_mlb_playback_hw = {
	.info = (SNDRV_PCM_INFO_MMAP |
		 SNDRV_PCM_INFO_MMAP_VALID |
		 /* SNDRV_PCM_INFO_DOUBLE | */
		 SNDRV_PCM_INFO_INTERLEAVED |
		 SNDRV_PCM_INFO_BLOCK_TRANSFER |
		 SNDRV_PCM_INFO_HALF_DUPLEX),
	.formats          = (SNDRV_PCM_FMTBIT_S16_BE |
			     SNDRV_PCM_FMTBIT_S16_LE),
	.rates            = SNDRV_PCM_RATE_48000,
	.rate_min         = 48000,
	.rate_max         = 48000,
	.channels_min     = 1,
	.channels_max     = 2,
	.buffer_bytes_max = BUF_MAX,
	.period_bytes_min = SYNC_BUFFER_DEP_MIN,
	.period_bytes_max = SYNC_BUFFER_DEP,
	.periods_min      = 1,
	.periods_max      = BUF_MAX / (SYNC_BUFFER_DEP_MIN),
};

/* HW description (capture) for ALSA */
static struct snd_pcm_hardware snd_mlb_capture_hw = {
	.info = (SNDRV_PCM_INFO_MMAP |
		 SNDRV_PCM_INFO_MMAP_VALID |
		 /* SNDRV_PCM_INFO_DOUBLE | */
		 SNDRV_PCM_INFO_INTERLEAVED |
		 SNDRV_PCM_INFO_BLOCK_TRANSFER |
		 SNDRV_PCM_INFO_HALF_DUPLEX),
	.formats =          (SNDRV_PCM_FMTBIT_S16_BE |
			     SNDRV_PCM_FMTBIT_S16_LE),
	.rates =            SNDRV_PCM_RATE_48000,
	.rate_min =         48000,
	.rate_max =         48000,
	.channels_min =     1,
	.channels_max =     2,
	.buffer_bytes_max = SYNC_BUFFER_DEP * BUF_RING_NODES,
	.period_bytes_min = SYNC_BUFFER_DEP_MIN,
	.period_bytes_max = SYNC_BUFFER_DEP,
	.periods_min =      1,
	.periods_max =      BUF_RING_NODES,
};

static int hw_rule_period_bytes_by_channels(struct snd_pcm_hw_params *params,
					    struct snd_pcm_hw_rule *rule)
{
	struct snd_interval *channels =
		hw_param_interval(params, SNDRV_PCM_HW_PARAM_CHANNELS);
	struct snd_interval *period_bytes =
		hw_param_interval(params, SNDRV_PCM_HW_PARAM_PERIOD_BYTES);
	struct snd_interval limit_bytes;

	snd_interval_any(&limit_bytes);

	if (snd_interval_max(channels) == 1) {
		limit_bytes.min = SYNC_BUFFER_DEP_MIN;
		limit_bytes.max = SYNC_BUFFER_DEP_MIN;
		return snd_interval_refine(period_bytes, &limit_bytes);
	}
	if (snd_interval_min(channels) == 2) {
		limit_bytes.min = SYNC_BUFFER_DEP;
		limit_bytes.max = SYNC_BUFFER_DEP;
		return snd_interval_refine(period_bytes, &limit_bytes);
	}
	return 0;
}

static int hw_rule_channels_by_period_bytes(struct snd_pcm_hw_params *params,
					    struct snd_pcm_hw_rule *rule)
{
	struct snd_interval *channels =
		hw_param_interval(params, SNDRV_PCM_HW_PARAM_CHANNELS);
	struct snd_interval *period_bytes =
		hw_param_interval(params, SNDRV_PCM_HW_PARAM_PERIOD_BYTES);
	struct snd_interval limit_channels;

	snd_interval_any(&limit_channels);

	limit_channels.integer = 1;
	if (snd_interval_min(period_bytes) == SYNC_BUFFER_DEP) {
		limit_channels.min = 2;
		limit_channels.max = 2;
		return snd_interval_refine(channels, &limit_channels);
	}
	if (snd_interval_max(period_bytes) == SYNC_BUFFER_DEP_MIN) {
		limit_channels.min = 1;
		limit_channels.max = 1;
		return snd_interval_refine(channels, &limit_channels);
	}
	return 0;
}

/*****************
  PCM operations */


/* Open
*********/
static int snd_mlb_open(struct snd_pcm_substream *substream)
{
	int err;
	struct snd_mlb_chip *chip;
	struct snd_pcm *pcm;
	struct snd_pcm_runtime *runtime;

	if (substream == NULL) {
		pr_err("open: no substream!!");
		return -1;
	}

	pcm = substream->pcm;

	chip = snd_pcm_substream_chip(substream);
	runtime = substream->runtime;

	runtime->private_data = chip;

	/* initialize to 1 so we don't have to wait in hw_free in error case */
	chip->stop_irq_done[pcm->device] = 1;

	/* In multiframe mode, the MLB DIM uses 4*(2^FCNT_VALUE)
	 * frames per sub-buffer (fixed), so the supported period
	 * size depends on the number of channels. */
	err = snd_pcm_hw_rule_add(runtime, 0, SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
				  hw_rule_period_bytes_by_channels,
				  &runtime->hw, SNDRV_PCM_HW_PARAM_CHANNELS,
				  -1);
	if (err < 0)
		return err;

	err = snd_pcm_hw_rule_add(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
				  hw_rule_channels_by_period_bytes,
				  &runtime->hw, SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
				  -1);
	if (err < 0)
		return err;

	snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		runtime->hw = snd_mlb_playback_hw;
		err = syncsound_open(pcm->device, MLB_WRONLY, NULL, NULL);
	} else {
		runtime->hw = snd_mlb_capture_hw;
		err = syncsound_open(pcm->device, MLB_RDONLY,
				&runtime->dma_area, &runtime->dma_addr);
		runtime->dma_bytes = SYNC_BUFFER_DEP * BUF_RING_NODES;
	}

	pr_debug("alsa syncsound: open(%d) -- %d\n", pcm->device, err);

	init_waitqueue_head(&chip->stop_wq[pcm->device]);
	return err;
}


/* Close
 *********/
static int snd_mlb_close(struct snd_pcm_substream *substream)
{
	int err;
	struct snd_pcm *pcm = substream->pcm;

	pr_debug("alsa syncsound: close(%d)\n", pcm->device);

	substream->private_data = NULL;

	err = syncsound_release(pcm->device);
	return err;
}


/* HW_params
 *************/
static int snd_mlb_hw_params(struct snd_pcm_substream  *substream,
				struct snd_pcm_hw_params  *hw_params)
{
	int err;
	struct snd_pcm *pcm = substream->pcm;
	struct snd_pcm_runtime *runtime = substream->runtime;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (!runtime->dma_area) {

			pr_debug("allocating %u bytes of DMA memory for playback\n",
				params_buffer_bytes(hw_params));

			runtime->dma_area = alloc_ext_mem(
				params_buffer_bytes(hw_params),
				&runtime->dma_addr);

			runtime->dma_bytes = params_buffer_bytes(hw_params);
			if (!runtime->dma_area) {
				pr_err("syncsound could not allocate memory\n");
				return -ENOMEM;
			}
		}
	} else {
		/* using memory from mxc_mlb150_do_open() */
	}

	err = register_syncsound_substream(pcm->device, substream);

	pr_debug("alsa syncsound: hw_params(%d, %u ch) done\n",
		 pcm->device, params_channels(hw_params));

	return err;
}


/* HW_free
 ***********/
static int snd_mlb_hw_free(struct snd_pcm_substream *substream)
{
	struct snd_mlb_chip *chip;
	struct snd_pcm_runtime *runtime;
	struct snd_pcm *pcm;

	chip = snd_pcm_substream_chip(substream);
	runtime = substream->runtime;
	pcm = substream->pcm;

	/* do not do anything if we already got called before */
	if (!runtime->dma_area)
		return 0;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* wait for interrupt; woken up from playback_tx_done() */
		wait_event_timeout(chip->stop_wq[pcm->device],
				chip->stop_irq_done[pcm->device] == 1,
				msecs_to_jiffies(STOP_TIMEOUT_MS));

		/* unregister from mxc_mlb driver */
		unregister_syncsound_substream(pcm->device);

		/* free memory (ring buffer) */
		free_ext_mem(runtime->dma_bytes, runtime->dma_area
			, runtime->dma_addr);
	} else {
		/* unregister from mxc_mlb driver */
		unregister_syncsound_substream(pcm->device);
		/* free() will be done by mxc_mlb150_do_release */
	}

	runtime->dma_area = NULL;
	runtime->dma_addr = 0;
	runtime->dma_bytes = 0;

	pr_debug("alsa syncsound: hw_free(%d) done\n", pcm->device);
	return 0;
}


/* Prepare
***********/
static int snd_mlb_prepare(struct snd_pcm_substream *substream)
{
	struct snd_mlb_chip *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm *pcm = substream->pcm;
	struct snd_pcm_runtime *runtime = substream->runtime;

	if (!runtime->dma_area) {
		pr_err("dma_area is not initialized.\n");
		return 0;
	}

	/* reset buffer */
	memset(runtime->dma_area, 0, runtime->dma_bytes);

	/* reset period counter */
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		chip->p_cur_period[pcm->device] = 0;
	else
		chip->c_cur_period[pcm->device] = 0;

	pr_debug("alsa syncsound: prepare(%d) done\n", pcm->device);

	return 0;
}


/* Trigger
 ***********/
static int snd_mlb_capture_trigger(struct snd_pcm_substream *substream, int cmd)
{
	int res;
	struct snd_pcm *pcm = substream->pcm;
	struct snd_pcm_runtime *runtime = substream->runtime;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		/* case SNDRV_PCM_TRIGGER_RESUME: */
		/* start up MLB channel (ioctl helper in mxc_mlb.c) */
		res = syncsound_chan_startup(pcm->device, runtime->channels);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		/* case SNDRV_PCM_TRIGGER_SUSPEND: */
		/* shut down MLB channel (ioctl helper in mxc_mlb.c) */
		res = syncsound_chan_shutdown(pcm->device);
		break;
	default:
		pr_info("This driver does not support cmd %d.\n", cmd);
		res = -EINVAL;
	}
	return res;
}

static int snd_mlb_playback_trigger(struct snd_pcm_substream *substream
					, int cmd)
{
	int res;
	struct snd_mlb_chip *chip;
	struct snd_pcm *pcm;
	struct snd_pcm_runtime *runtime;
	u32 phy_addr;

	chip = snd_pcm_substream_chip(substream);
	pcm = substream->pcm;
	runtime = substream->runtime;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		/*
		 * See SNDRV_PCM_TRIGGER_STOP below and delayed shutdown
		 * in playback_tx_done
		 */

		if (atomic_cmpxchg(&chip->stop_requested[pcm->device],
				   1, 0) == 1) {
			rmb();
			if (!chip->stop_irq_done[pcm->device]) {
				res = -EBUSY;
				break;
			}
		}

		/* start up MLB channel */
		res = syncsound_chan_startup(pcm->device, runtime->channels);
		if (res) {
			pr_err("syncsound_chan_startup failed\n");
			break;
		}

		phy_addr = playback_get_buffer(substream);
		res = syncsound_start_playback(pcm->device, phy_addr);
		pr_debug("alsa syncsound playback: trigger start(%d, %d) done\n"
			, pcm->device, runtime->channels);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		/* shut down MLB channel */
#ifdef BEEP_WORKAROUND
		{
			struct snd_pcm_runtime *runtime = substream->runtime;

			if (!runtime->dma_area) {
				pr_alert("dma_area is NULL\n");
			} else {
				/* reset buffer */
				memset(runtime->dma_area, 0
					, runtime->dma_bytes);
			}
		}
#endif
		if (atomic_cmpxchg(&chip->stop_requested[pcm->device],
				   0, 1) == 0)
			chip->stop_irq_done[pcm->device] = 0;
		/* Don't shut down channel here but in ISR instead so
		 * playback is not truncated
		 */
		res = 0;
		pr_debug("alsa syncsound playback: trigger stop(%d) done\n"
			, pcm->device);
		break;
	default:
		pr_info("This driver does not support cmd %d.\n", cmd);
		res = -EINVAL;
	}
	return res;
}


/* Pointer
***********/
static snd_pcm_uframes_t snd_mlb_pointer
			(struct snd_pcm_substream *substream)
{
	struct snd_mlb_chip *chip = snd_pcm_substream_chip(substream);
	struct snd_pcm *pcm = substream->pcm;
	struct snd_pcm_runtime *runtime = substream->runtime;
	unsigned int cur_period;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		cur_period = chip->p_cur_period[pcm->device];
	else
		cur_period = chip->c_cur_period[pcm->device];

	/* calculate start address of current period in frames */
	return runtime->period_size *
			(cur_period % runtime->periods);
}

/* mmap
***********/
static int snd_mlb_mmap(struct snd_pcm_substream *substream
				, struct vm_area_struct *vma)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	return mmap_ext_mem(vma, runtime->dma_area
				, runtime->dma_addr, runtime->dma_bytes);
}

static struct snd_pcm_ops snd_mlb_playback_ops = {
	.open      = snd_mlb_open,
	.close     = snd_mlb_close,
	.ioctl     = snd_pcm_lib_ioctl,
	.hw_params = snd_mlb_hw_params,
	.hw_free   = snd_mlb_hw_free,
	.prepare   = snd_mlb_prepare,
	.trigger   = snd_mlb_playback_trigger,
	.pointer   = snd_mlb_pointer,
	.mmap      = snd_mlb_mmap,
};

static struct snd_pcm_ops snd_mlb_capture_ops = {
	.open =        snd_mlb_open,
	.close =       snd_mlb_close,
	.ioctl =       snd_pcm_lib_ioctl,
	.hw_params =   snd_mlb_hw_params,
	.hw_free =     snd_mlb_hw_free,
	.prepare =     snd_mlb_prepare,
	.trigger =     snd_mlb_capture_trigger,
	.pointer =     snd_mlb_pointer,
	.mmap    =     snd_mlb_mmap,
};
/****************
  General stuff */

static struct snd_device_ops dev_ops = { NULL };

static int mxc_mlb_syncsnd_probe(struct platform_device *pdev)
{
	int err;
	struct snd_mlb_chip *chip;
	struct snd_pcm *pcm;
	struct snd_card *card = NULL;
	int ret;
	int i;

	int dev = pdev->id;

	ret = dma_set_mask(&pdev->dev, DMA_BIT_MASK(32));
	if (ret) {
		ret = dma_set_coherent_mask(&pdev->dev,
					DMA_BIT_MASK(32));
		if (ret) {
			dev_err(&pdev->dev,
			"No usable DMA configuration, aborting\n");
			return -ENOMEM;
		}
	}

	err = snd_card_create(index[dev], id[dev], THIS_MODULE
				, sizeof(struct snd_mlb_chip), &card);

	if (card == NULL) {
		pr_err("SYNCSND: snd_card_new() failed, canceling probe\n");
		return -ENOMEM;
	}

	chip = card->private_data;
	chip->card = card;

	err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, chip, &dev_ops);
	if (err < 0) {
		pr_err("SYNCSND: snd_device_new() failed, canceling probe\n");
		snd_card_free(card);
		return err;
	}

	strcpy(card->driver, "MLB_Sync_Driver");
	strcpy(card->shortname, "MLB_Sync_Audio");
	strcpy(card->longname,
	"Virtual soundcard over MLB synchronous channels");
	snd_card_set_dev(card, &pdev->dev);

	/* PCM */
	for (i = 0; i < syncsound_get_num_devices(); i++) {
		/* register pcm with 1 playback and 1 capture substreams */
		err = snd_pcm_new(card, card->driver, i, 1, 1, &pcm);
		if (err < 0) {
			pr_err("SYNCSND: snd_pcm_new() %d failed, canceling probe\n",
					i);
			snd_card_free(card);
			return err;
		}
		pcm->private_data = chip;
		chip->pcm[i] = pcm;
		pcm->info_flags = 0;
		sprintf(pcm->name, "MLB_SYNC%d", i);
		snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE,
				&snd_mlb_capture_ops);
		snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK,
				&snd_mlb_playback_ops);
	}

	err = snd_card_register(card);
	if (err < 0) {
		pr_err("SYNCSND: snd_card_register() failed, canceling probe\n");
		snd_card_free(card);
		return err;
	}

	platform_set_drvdata(pdev, card);

	register_syncsound_callbacks(playback_tx_done,
				     capture_rx_done);

	return 0;
}

static int mxc_mlb_syncsnd_remove(struct platform_device *pdev)
{
	pr_debug("SYNCSND: remove() called\n");
	unregister_syncsound_callbacks();
	snd_card_free(platform_get_drvdata(pdev));
	platform_set_drvdata(pdev, NULL);
	return 0;
}

#ifdef CONFIG_PM
static int mxc_mlb_syncsnd_suspend(struct platform_device *pdev
					, pm_message_t state)
{
	/* not implemented */
	return 0;
}

static int mxc_mlb_syncsnd_resume(struct platform_device *pdev)
{
	/* not implemented */
	return 0;
}
#endif

#define SND_MLB_SYNCSND_DRIVER  "snd_mlb_syncsnd"

static struct platform_driver mxc_mlb_syncsnd_driver = {
	.probe = mxc_mlb_syncsnd_probe,
	.remove = mxc_mlb_syncsnd_remove,
#ifdef CONFIG_PM
	.suspend = mxc_mlb_syncsnd_suspend,
	.resume = mxc_mlb_syncsnd_resume,
#endif
	.driver = {
		.name = SND_MLB_SYNCSND_DRIVER},
};

static void snd_mlb_unregister_all(void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(devices); ++i)
		platform_device_unregister(devices[i]);

	platform_driver_unregister(&mxc_mlb_syncsnd_driver);
}

static int __init mxc_mlb_syncsnd_init(void)
{
	int i, cards, err;

	err = platform_driver_register(&mxc_mlb_syncsnd_driver);
	if (err < 0)
		return err;

	cards = 0;
	for (i = 0; i < SNDRV_CARDS; i++) {
		struct platform_device *device;

		if (!enable[i])
			continue;

		device = platform_device_register_simple(SND_MLB_SYNCSND_DRIVER
							, i, NULL, 0);
		if (IS_ERR(device))
			continue;

		if (!platform_get_drvdata(device)) {
			platform_device_unregister(device);
			continue;
		}

		devices[i] = device;
		cards++;
	}

	if (!cards) {
#ifdef MODULE
		pr_err("SYNCSND: Not enabled, not found or device busy.\n");
#endif
		snd_mlb_unregister_all();
		return -ENODEV;
	}
	return 0;
}

static void __exit mxc_mlb_syncsnd_exit(void)
{
	snd_mlb_unregister_all();
}

module_init(mxc_mlb_syncsnd_init);
module_exit(mxc_mlb_syncsnd_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Cetitec GmbH");
