/*
 * Copyright (c) 2010 Sascha Hauer <s.hauer@pengutronix.de>
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <linux/export.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>

#include "ipu-prv.h"

#define ASYNC_SER_WAVE 6

#define DC_DISP_ID_SERIAL	2
#define DC_DISP_ID_ASYNC	3

#define DC_MAP_CONF_PTR(n)	(0x0108 + ((n) & ~0x1) * 2)
#define DC_MAP_CONF_VAL(n)	(0x0144 + ((n) & ~0x1) * 2)
#define DC_MAX_MAPS            24

#define DC_EVT_NF		0
#define DC_EVT_NL		1
#define DC_EVT_EOF		2
#define DC_EVT_NFIELD		3
#define DC_EVT_EOL		4
#define DC_EVT_EOFIELD		5
#define DC_EVT_NEW_ADDR		6
#define DC_EVT_NEW_CHAN		7
#define DC_EVT_NEW_DATA		8

#define DC_EVT_NEW_ADDR_W_0	0
#define DC_EVT_NEW_ADDR_W_1	1
#define DC_EVT_NEW_CHAN_W_0	2
#define DC_EVT_NEW_CHAN_W_1	3
#define DC_EVT_NEW_DATA_W_0	4
#define DC_EVT_NEW_DATA_W_1	5
#define DC_EVT_NEW_ADDR_R_0	6
#define DC_EVT_NEW_ADDR_R_1	7
#define DC_EVT_NEW_CHAN_R_0	8
#define DC_EVT_NEW_CHAN_R_1	9
#define DC_EVT_NEW_DATA_R_0	10
#define DC_EVT_NEW_DATA_R_1	11

#define DC_WR_CH_CONF		0x0
#define DC_WR_CH_ADDR		0x4
#define DC_RL_CH(evt)		(8 + ((evt) & ~0x1) * 2)

#define DC_GEN			0x00d4
#define DC_DISP_CONF1(disp)	(0x00d8 + (disp) * 4)
#define DC_DISP_CONF2(disp)	(0x00e8 + (disp) * 4)
#define DC_STAT			0x01c8

#define WROD(lf)		(0x18 | (lf << 1))

#define DC_WR_CH_CONF_FIELD_MODE		(1 << 9)
#define DC_WR_CH_CONF_PROG_TYPE_OFFSET		5
#define DC_WR_CH_CONF_PROG_TYPE_MASK		(7 << 5)
#define DC_WR_CH_CONF_PROG_DI_ID		(1 << 2)
#define DC_WR_CH_CONF_PROG_DISP_ID_OFFSET	3
#define DC_WR_CH_CONF_PROG_DISP_ID_MASK		(3 << 3)

/*
 * This structure defines how to map each color component of the incoming
 * RGB24 or YUV444 pixels arriving at the DC onto the DI bus:
 *
 * src_mask[] defines which bits of each incoming 8-bit RGB24/YUV444
 *            component are to be selected and forwarded to the DI bus
 *            (as a bit mask).
 *
 * dest_msb[] defines where to place the selected bits of each component
 *            on the DI bus (as the most-significant-bit position).
 *
 * v4l2_fmt   non-zero if this mapping corresponds to a standard
 *            V4L2 pixel format.
 */
struct ipu_dc_if_map {
	u32 src_mask[3];
	u32 dest_msb[3];
	u32 v4l2_fmt;
};

struct ipu_dc_priv;

struct ipu_dc {
	/* The display interface number assigned to this dc channel */
	unsigned int		di;
	void __iomem		*base;
	struct ipu_dc_priv	*priv;
	int			chno;
	bool			in_use;
	struct completion       disable_comp;
};

struct ipu_dc_priv {
	void __iomem		*dc_reg;
	void __iomem		*dc_tmpl_reg;
	struct ipu_soc		*ipu;
	struct device		*dev;
	struct ipu_dc		channels[IPU_DC_NUM_CHANNELS];
	struct ipu_dc_if_map    map[DC_MAX_MAPS];
	int                     next_map;
	struct mutex		mutex;
	int                     use_count;
};

/* some standard maps */
static struct ipu_dc_if_map standard_maps[] = {
	{
		.src_mask = {0xff, 0xff, 0xff},
		.dest_msb = {7, 15, 23},
		.v4l2_fmt = V4L2_PIX_FMT_RGB24,
	}, {
		.src_mask = {0xf8, 0xfc, 0xf8},
		.dest_msb = {4, 10, 15},
		.v4l2_fmt = V4L2_PIX_FMT_RGB565,
	}, {
		.src_mask = {0xfc, 0xfc, 0xfc},
		.dest_msb = {5, 11, 17},
		.v4l2_fmt = V4L2_PIX_FMT_RGB666,
	},
	/* add more here */
};

/* forward references */
static void ipu_dc_map_config(struct ipu_dc_priv *priv, int mapnr,
			      struct ipu_dc_if_map *map);
static void ipu_dc_map_clear(struct ipu_dc_priv *priv, int mapnr);


static void dc_link_event(struct ipu_dc *dc, int event, int addr, int priority)
{
	u32 reg;

	reg = readl(dc->base + DC_RL_CH(event));
	reg &= ~(0xffff << (16 * (event & 0x1)));
	reg |= ((addr << 8) | priority) << (16 * (event & 0x1));
	writel(reg, dc->base + DC_RL_CH(event));
}

static void dc_write_tmpl(struct ipu_dc *dc, int word, u32 opcode, u32 operand,
		int map, int wave, int glue, int sync)
{
	struct ipu_dc_priv *priv = dc->priv;
	u32 reg;
	int stop = 1;

	reg = sync;
	reg |= glue << 4;
	reg |= ++wave << 11;
	reg |= ++map << 15;
	reg |= (operand << 20) & 0xfff00000;
	writel(reg, priv->dc_tmpl_reg + word * 8);

	reg = operand >> 12;
	reg |= opcode << 4;
	reg |= stop << 9;
	writel(reg, priv->dc_tmpl_reg + word * 8 + 4);
}

static inline bool identical_mapping(struct ipu_dc_if_map *map1,
				     struct ipu_dc_if_map *map2)
{
	int i;
	for (i = 0; i < 3; i++) {
		if (map1->src_mask[i] != map2->src_mask[i] ||
		    map1->dest_msb[i] != map2->dest_msb[i])
			return false;
	}

	return true;
}

/* must hold priv->mutex when calling */
static int ipu_dc_new_map(struct ipu_dc_priv *priv,
			  struct ipu_dc_if_map *map)
{
	int i, mapnr;

	/* first search for an existing map that matches */
	for (i = 0; i < priv->next_map; i++) {
		if (identical_mapping(&priv->map[i], map))
			break;
	}
	if (i < priv->next_map)
		return i;

	if (priv->next_map >= DC_MAX_MAPS) {
		dev_err(priv->dev, "IPU_DISP: No map space left\n");
		return -ENOSPC;
	}

	mapnr = priv->next_map++;

	priv->map[mapnr] = *map;

	ipu_dc_map_clear(priv, mapnr);
	ipu_dc_map_config(priv, mapnr, map);

	return mapnr;
}

static int ipu_dc_get_map(struct ipu_dc_priv *priv, u32 v4l2_fmt, u32 *map)
{
	int i, ret = -EINVAL;

	mutex_lock(&priv->mutex);

	if (map) {
		/* create a new map */
		struct ipu_dc_if_map new_map;

		for (i = 0; i < 3; i++) {
			new_map.dest_msb[i] = map[2 * i];
			new_map.src_mask[i] = map[2 * i + 1];
		}
		new_map.v4l2_fmt = v4l2_fmt;

		ret = ipu_dc_new_map(priv, &new_map);
	} else if (v4l2_fmt) {
		/* otherwise search for a standard map */
		for (i = 0; i < priv->next_map; i++) {
			if (priv->map[i].v4l2_fmt == v4l2_fmt)
				break;
		}

		ret = (i < priv->next_map) ? i : -EINVAL;
	}

	mutex_unlock(&priv->mutex);
	return ret;
}

#define SYNC_WAVE 0

int ipu_dc_init_sync(struct ipu_dc *dc, struct ipu_di *di, bool interlaced,
		     u32 pixel_fmt, u32 *map, u32 width)
{
	struct ipu_dc_priv *priv = dc->priv;
	u32 reg = 0;
	int mapnr;

	dc->di = ipu_di_get_num(di);

	mapnr = ipu_dc_get_map(priv, pixel_fmt, map);
	if (mapnr < 0) {
		dev_dbg(priv->dev, "IPU_DISP: No MAP\n");
		return mapnr;
	}

	if (interlaced) {
		dc_link_event(dc, DC_EVT_NL, 0, 3);
		dc_link_event(dc, DC_EVT_EOL, 0, 2);
		dc_link_event(dc, DC_EVT_NEW_DATA, 0, 1);

		/* Init template microcode */
		dc_write_tmpl(dc, 0, WROD(0), 0, mapnr, SYNC_WAVE, 0, 8);
	} else {
		if (dc->di) {
			dc_link_event(dc, DC_EVT_NL, 2, 3);
			dc_link_event(dc, DC_EVT_EOL, 3, 2);
			dc_link_event(dc, DC_EVT_NEW_DATA, 4, 1);
			/* Init template microcode */
			dc_write_tmpl(dc, 2, WROD(0), 0, mapnr,
				      SYNC_WAVE, 8, 5);
			dc_write_tmpl(dc, 3, WROD(0), 0, mapnr,
				      SYNC_WAVE, 4, 5);
			dc_write_tmpl(dc, 4, WROD(0), 0, mapnr,
				      SYNC_WAVE, 0, 5);
		} else {
			dc_link_event(dc, DC_EVT_NL, 5, 3);
			dc_link_event(dc, DC_EVT_EOL, 6, 2);
			dc_link_event(dc, DC_EVT_NEW_DATA, 7, 1);
			/* Init template microcode */
			dc_write_tmpl(dc, 5, WROD(0), 0, mapnr,
				      SYNC_WAVE, 8, 5);
			dc_write_tmpl(dc, 6, WROD(0), 0, mapnr,
				      SYNC_WAVE, 4, 5);
			dc_write_tmpl(dc, 7, WROD(0), 0, mapnr,
				      SYNC_WAVE, 0, 5);
		}
	}
	dc_link_event(dc, DC_EVT_NF, 0, 0);
	dc_link_event(dc, DC_EVT_NFIELD, 0, 0);
	dc_link_event(dc, DC_EVT_EOF, 0, 0);
	dc_link_event(dc, DC_EVT_EOFIELD, 0, 0);
	dc_link_event(dc, DC_EVT_NEW_CHAN, 0, 0);
	dc_link_event(dc, DC_EVT_NEW_ADDR, 0, 0);

	reg = 0x2;
	reg |= dc->di << DC_WR_CH_CONF_PROG_DISP_ID_OFFSET;
	reg |= dc->di << 2;
	if (interlaced)
		reg |= DC_WR_CH_CONF_FIELD_MODE;

	writel(reg, dc->base + DC_WR_CH_CONF);

	writel(0x00000000, dc->base + DC_WR_CH_ADDR);

	writel(0x00000084, priv->dc_reg + DC_GEN);

	writel(width, priv->dc_reg + DC_DISP_CONF2(dc->di));

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_dc_init_sync);

void ipu_dc_init_async(struct ipu_dc *dc, struct ipu_di *di, bool interlaced)
{
	struct ipu_dc_priv *priv = dc->priv;
	u32 reg = 0;

	dc->di = ipu_di_get_num(di);

	dc_link_event(dc, DC_EVT_NEW_DATA_W_0, 0x64, 1);
	dc_link_event(dc, DC_EVT_NEW_DATA_W_1, 0x64, 1);

	reg = 0x3;
	reg |= DC_DISP_ID_SERIAL << DC_WR_CH_CONF_PROG_DISP_ID_OFFSET;
	writel(reg, dc->base + DC_WR_CH_CONF);

	writel(0x00000000, dc->base + DC_WR_CH_ADDR);

	writel(0x00000084, priv->dc_reg + DC_GEN);
}
EXPORT_SYMBOL_GPL(ipu_dc_init_async);

void ipu_dc_uninit(struct ipu_dc *dc)
{
	if (dc->chno == IPU_DC_CHANNEL_SYNC ||
	    dc->chno == IPU_DC_CHANNEL_DP_SYNC) {
		dc_link_event(dc, DC_EVT_NL, 0, 0);
		dc_link_event(dc, DC_EVT_EOL, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_DATA, 0, 0);
		dc_link_event(dc, DC_EVT_NF, 0, 0);
		dc_link_event(dc, DC_EVT_NFIELD, 0, 0);
		dc_link_event(dc, DC_EVT_EOF, 0, 0);
		dc_link_event(dc, DC_EVT_EOFIELD, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_CHAN, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_ADDR, 0, 0);
	} else if (dc->chno == IPU_DC_CHANNEL_ASYNC ||
		   dc->chno == IPU_DC_CHANNEL_DP_ASYNC) {
		dc_link_event(dc, DC_EVT_NEW_ADDR_W_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_ADDR_W_1, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_CHAN_W_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_CHAN_W_1, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_DATA_W_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_DATA_W_1, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_ADDR_R_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_ADDR_R_1, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_CHAN_R_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_CHAN_R_1, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_DATA_R_0, 0, 0);
		dc_link_event(dc, DC_EVT_NEW_DATA_R_1, 0, 0);
	}
}
EXPORT_SYMBOL_GPL(ipu_dc_uninit);

void ipu_dc_enable_channel(struct ipu_dc *dc)
{
	int di;
	u32 reg;

	di = dc->di;

	reg = readl(dc->base + DC_WR_CH_CONF);
	reg |= 4 << DC_WR_CH_CONF_PROG_TYPE_OFFSET;
	writel(reg, dc->base + DC_WR_CH_CONF);
}
EXPORT_SYMBOL_GPL(ipu_dc_enable_channel);

static irqreturn_t ipu_dc_irq_handler(int irq, void *dev_id)
{
	struct ipu_dc *dc = dev_id;
	struct ipu_dc_priv *priv = dc->priv;
	u32 val;

	/* disable the DC channel */
	val = readl(dc->base + DC_WR_CH_CONF);
	val &= ~DC_WR_CH_CONF_PROG_TYPE_MASK;
	writel(val, dc->base + DC_WR_CH_CONF);

	val = ipu_cm_read(priv->ipu, IPU_DISP_GEN);
	if (dc->di == 1)
		val &= ~IPU_DI1_COUNTER_RELEASE;
	else
		val &= ~IPU_DI0_COUNTER_RELEASE;
	ipu_cm_write(priv->ipu, val, IPU_DISP_GEN);

	complete(&dc->disable_comp);
	return IRQ_HANDLED;
}

void ipu_dc_disable_channel(struct ipu_dc *dc)
{
	struct ipu_dc_priv *priv = dc->priv;
	enum ipu_gp_irq gp_irq;
	unsigned int timeout;
	int irq, ret;

	if (dc->chno == IPU_DC_CHANNEL_SYNC)
		gp_irq = IPU_IRQ_DC_FC_1;
	else if (dc->chno == IPU_DC_CHANNEL_DP_SYNC)
		gp_irq = IPU_IRQ_DP_SF_END;
	else
		return;

	init_completion(&dc->disable_comp);

	irq = ipu_gp_irq(priv->ipu, gp_irq);
	ret = devm_request_irq(priv->dev, irq, ipu_dc_irq_handler, 0,
			       "ipu_dc", dc);
	if (ret < 0) {
		dev_err(priv->dev, "failed to req DC irq %d\n", irq);
		return;
	}

	/* wait for the interrupt */
	timeout = msecs_to_jiffies(50);
	ret = wait_for_completion_timeout(&dc->disable_comp, timeout);
	devm_free_irq(priv->dev, irq, dc);

	if (!ret)
		dev_err(priv->dev, "DC stop timeout!\n");
	else
		dev_dbg(priv->dev, "DC stopped in %d msec\n",
			jiffies_to_msecs(timeout - ret));
}
EXPORT_SYMBOL_GPL(ipu_dc_disable_channel);

static void ipu_dc_map_config(struct ipu_dc_priv *priv, int mapnr,
			      struct ipu_dc_if_map *map)
{
	int i, ptr;
	u32 reg;

	for (i = 0; i < 3; i++) {
		ptr = mapnr * 3 + i;

		reg = readl(priv->dc_reg + DC_MAP_CONF_VAL(ptr));
		reg &= ~(0xffff << (16 * (ptr & 0x1)));
		reg |= (((map->dest_msb[i] << 8) | map->src_mask[i]) <<
			(16 * (ptr & 0x1)));
		writel(reg, priv->dc_reg + DC_MAP_CONF_VAL(ptr));

		reg = readl(priv->dc_reg + DC_MAP_CONF_PTR(mapnr));
		reg &= ~(0x1f << ((16 * (mapnr & 0x1)) + (5 * i)));
		reg |= (ptr << ((16 * (mapnr & 0x1)) + (5 * i)));
		writel(reg, priv->dc_reg + DC_MAP_CONF_PTR(mapnr));
	}
}

static void ipu_dc_map_clear(struct ipu_dc_priv *priv, int mapnr)
{
	u32 reg = readl(priv->dc_reg + DC_MAP_CONF_PTR(mapnr));
	writel(reg & ~(0xffff << (16 * (mapnr & 0x1))),
	       priv->dc_reg + DC_MAP_CONF_PTR(mapnr));
}

void ipu_dc_enable(struct ipu_dc *dc)
{
	struct ipu_dc_priv *priv = dc->priv;

	mutex_lock(&priv->mutex);

	if (!priv->use_count) {
		dev_dbg(priv->dev, "DC enable\n");
		ipu_module_enable(priv->ipu, IPU_CONF_DC_EN);
	}

	priv->use_count++;

	mutex_unlock(&priv->mutex);
}
EXPORT_SYMBOL_GPL(ipu_dc_enable);

void ipu_dc_disable(struct ipu_dc *dc)
{
	struct ipu_dc_priv *priv = dc->priv;

	mutex_lock(&priv->mutex);

	priv->use_count--;
	if (!priv->use_count) {
		dev_dbg(priv->dev, "DC disable\n");
		ipu_module_disable(priv->ipu, IPU_CONF_DC_EN);
	}

	if (priv->use_count < 0)
		priv->use_count = 0;

	mutex_unlock(&priv->mutex);
}
EXPORT_SYMBOL_GPL(ipu_dc_disable);

struct ipu_dc *ipu_dc_get(struct ipu_soc *ipu, int channel)
{
	struct ipu_dc_priv *priv = ipu->dc_priv;
	struct ipu_dc *dc;

	if (channel >= IPU_DC_NUM_CHANNELS)
		return NULL;

	dc = &priv->channels[channel];

	mutex_lock(&priv->mutex);

	if (dc->in_use) {
		mutex_unlock(&priv->mutex);
		return ERR_PTR(-EBUSY);
	}

	dc->in_use = 1;

	mutex_unlock(&priv->mutex);

	return dc;
}
EXPORT_SYMBOL_GPL(ipu_dc_get);

void ipu_dc_put(struct ipu_dc *dc)
{
	struct ipu_dc_priv *priv = dc->priv;

	mutex_lock(&priv->mutex);
	dc->in_use = 0;
	mutex_unlock(&priv->mutex);
}
EXPORT_SYMBOL_GPL(ipu_dc_put);

int ipu_dc_get_num(struct ipu_dc *dc)
{
	return dc->chno;
}
EXPORT_SYMBOL_GPL(ipu_dc_get_num);

int ipu_dc_init(struct ipu_soc *ipu, struct device *dev,
		unsigned long base, unsigned long template_base)
{
	struct ipu_dc_priv *priv;
	static int channel_offsets[] = { 0, 0x1c, 0x38, 0x54, 0x58, 0x5c,
		0x78, 0, 0x94, 0xb4};
	int i;

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	mutex_init(&priv->mutex);

	priv->dev = dev;
	priv->ipu = ipu;
	priv->dc_reg = devm_ioremap(dev, base, PAGE_SIZE);
	priv->dc_tmpl_reg = devm_ioremap(dev, template_base, PAGE_SIZE);
	if (!priv->dc_reg || !priv->dc_tmpl_reg)
		return -ENOMEM;

	for (i = 0; i < IPU_DC_NUM_CHANNELS; i++) {
		priv->channels[i].chno = i;
		priv->channels[i].priv = priv;
		priv->channels[i].base = priv->dc_reg + channel_offsets[i];
	}

	ipu->dc_priv = priv;

	dev_dbg(dev, "DC base: 0x%08lx template base: 0x%08lx\n",
			base, template_base);

	/* add the standard maps */
	mutex_lock(&priv->mutex);
	for (i = 0; i < ARRAY_SIZE(standard_maps); i++)
		ipu_dc_new_map(priv, &standard_maps[i]);
	mutex_unlock(&priv->mutex);

	return 0;
}

void ipu_dc_exit(struct ipu_soc *ipu)
{
}
