/*
 * Copyright (c) 2010 Sascha Hauer <s.hauer@pengutronix.de>
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc.
 * Copyright (C) 2012 Mentor Graphics Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */
#include <linux/module.h>
#include <linux/export.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/spinlock.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/clk.h>
#include <linux/list.h>
#include <linux/irq.h>
#include <linux/of_device.h>
#include <uapi/linux/v4l2-mediabus.h>
#include <asm/mach/irq.h>

#include "ipu-prv.h"

int ipu_get_num(struct ipu_soc *ipu)
{
	return ipu->id;
}
EXPORT_SYMBOL_GPL(ipu_get_num);

void ipu_srm_dp_sync_update(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_cm_read(ipu, IPU_SRM_PRI2);
	val |= 0x8;
	ipu_cm_write(ipu, val, IPU_SRM_PRI2);

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_srm_dp_sync_update);

enum ipu_color_space ipu_pixelformat_to_colorspace(u32 pixelformat)
{
	switch (pixelformat) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YVYU:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
	case V4L2_PIX_FMT_NV16:
	case V4L2_PIX_FMT_NV61:
		return IPUV3_COLORSPACE_YUV;
	case V4L2_PIX_FMT_RGB32:
	case V4L2_PIX_FMT_BGR32:
	case V4L2_PIX_FMT_RGB24:
	case V4L2_PIX_FMT_BGR24:
	case V4L2_PIX_FMT_RGB565:
		return IPUV3_COLORSPACE_RGB;
	default:
		return IPUV3_COLORSPACE_UNKNOWN;
	}
}
EXPORT_SYMBOL_GPL(ipu_pixelformat_to_colorspace);

enum ipu_color_space ipu_mbus_code_to_colorspace(u32 mbus_code)
{
	switch (mbus_code & 0xf000) {
	case 0x1000:
		return IPUV3_COLORSPACE_RGB;
	case 0x2000:
		return IPUV3_COLORSPACE_YUV;
	default:
		return IPUV3_COLORSPACE_UNKNOWN;
	}
}
EXPORT_SYMBOL_GPL(ipu_mbus_code_to_colorspace);

u32 ipu_mbus_code_to_fourcc(u32 mbus_code)
{
	switch (mbus_code) {
	case V4L2_MBUS_FMT_RGB565_2X8_BE:
		return V4L2_PIX_FMT_RGB565X;
	case V4L2_MBUS_FMT_RGB565_2X8_LE:
		return V4L2_PIX_FMT_RGB565;
	case V4L2_MBUS_FMT_RGB444_2X8_PADHI_LE:
		return V4L2_PIX_FMT_RGB444;
	case V4L2_MBUS_FMT_RGB555_2X8_PADHI_BE:
		return V4L2_PIX_FMT_RGB555X;
	case V4L2_MBUS_FMT_RGB555_2X8_PADHI_LE:
		return V4L2_PIX_FMT_RGB555;
	case V4L2_MBUS_FMT_UYVY8_2X8:
	case V4L2_MBUS_FMT_UYVY8_1X16:
		return V4L2_PIX_FMT_UYVY;
	case V4L2_MBUS_FMT_YUYV8_2X8:
	case V4L2_MBUS_FMT_YUYV8_1X16:
		return V4L2_PIX_FMT_YUYV;
	case V4L2_MBUS_FMT_SBGGR8_1X8:
		return V4L2_PIX_FMT_SBGGR8;
	case V4L2_MBUS_FMT_SGBRG8_1X8:
		return V4L2_PIX_FMT_SGBRG8;
	case V4L2_MBUS_FMT_SGRBG8_1X8:
		return V4L2_PIX_FMT_SGRBG8;
	case V4L2_MBUS_FMT_SRGGB8_1X8:
		return V4L2_PIX_FMT_SRGGB8;
	case V4L2_MBUS_FMT_SBGGR10_1X10:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADHI_LE:
	case V4L2_MBUS_FMT_SBGGR10_2X8_PADLO_LE:
		return V4L2_PIX_FMT_SBGGR10;
	case V4L2_MBUS_FMT_SBGGR10_DPCM8_1X8:
		return V4L2_PIX_FMT_SBGGR10DPCM8;
	case V4L2_MBUS_FMT_SGBRG10_DPCM8_1X8:
		return V4L2_PIX_FMT_SGBRG10DPCM8;
	case V4L2_MBUS_FMT_SGBRG10_1X10:
		return V4L2_PIX_FMT_SGBRG10;
	case V4L2_MBUS_FMT_SGRBG10_DPCM8_1X8:
		return V4L2_PIX_FMT_SGRBG10DPCM8;
	case V4L2_MBUS_FMT_SGRBG10_1X10:
		return V4L2_PIX_FMT_SGRBG10;
	case V4L2_MBUS_FMT_SRGGB10_DPCM8_1X8:
		return V4L2_PIX_FMT_SRGGB10DPCM8;
	case V4L2_MBUS_FMT_SRGGB10_1X10:
		return V4L2_PIX_FMT_SRGGB10;
	case V4L2_MBUS_FMT_SBGGR12_1X12:
		return V4L2_PIX_FMT_SBGGR12;
	case V4L2_MBUS_FMT_SGBRG12_1X12:
		return V4L2_PIX_FMT_SGBRG12;
	case V4L2_MBUS_FMT_SGRBG12_1X12:
		return V4L2_PIX_FMT_SGRBG12;
	case V4L2_MBUS_FMT_SRGGB12_1X12:
		return V4L2_PIX_FMT_SRGGB12;
	case V4L2_MBUS_FMT_JPEG_1X8:
		return V4L2_PIX_FMT_JPEG;
	default:
		return 0;
	}
}
EXPORT_SYMBOL_GPL(ipu_mbus_code_to_fourcc);

/*
 * Standard bpp from pixel format.
 */
int ipu_bits_per_pixel(u32 pixelformat)
{
	switch (pixelformat) {
	case V4L2_PIX_FMT_RGB332:
		return 8;
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
		return 12;
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
	case V4L2_PIX_FMT_RGB565:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YVYU:
	case V4L2_PIX_FMT_NV16:
	case V4L2_PIX_FMT_NV61:
		return 16;
	case V4L2_PIX_FMT_BGR24:
	case V4L2_PIX_FMT_RGB24:
		return 24;
	case V4L2_PIX_FMT_BGR32:
	case V4L2_PIX_FMT_RGB32:
		return 32;
	default:
		break;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_bits_per_pixel);

bool ipu_pixelformat_is_planar(u32 pixelformat)
{
	switch (pixelformat) {
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
	case V4L2_PIX_FMT_NV16:
	case V4L2_PIX_FMT_NV61:
		return true;
	}

	return false;
}
EXPORT_SYMBOL_GPL(ipu_pixelformat_is_planar);

bool ipu_pixelformat_has_alpha(u32 pixelformat)
{
	switch (pixelformat) {
	case V4L2_PIX_FMT_RGB32:
	case V4L2_PIX_FMT_BGR32:
		return true;
		break;
	default:
		return false;
		break;
	}
	return false;
}
EXPORT_SYMBOL_GPL(ipu_pixelformat_has_alpha);

int ipu_stride_to_bytes(u32 pixel_stride, u32 pixelformat)
{
	switch (pixelformat) {
	case V4L2_PIX_FMT_RGB332:
	case V4L2_PIX_FMT_YUV420:
	case V4L2_PIX_FMT_YVU420:
	case V4L2_PIX_FMT_YUV422P:
	case V4L2_PIX_FMT_YVU422P:
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
	case V4L2_PIX_FMT_NV16:
	case V4L2_PIX_FMT_NV61:
		/*
		 * for the planar YUV formats, the stride passed to
		 * ipu_cpmem_channel_init must be the stride in bytes
		 * of the Y plane. And all the planar YUV formats have
		 * an 8-bit Y value.
		 */
		return (8 * pixel_stride) >> 3;
	case V4L2_PIX_FMT_RGB565:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_GENERIC16:
		return (16 * pixel_stride) >> 3;
	case V4L2_PIX_FMT_BGR24:
	case V4L2_PIX_FMT_RGB24:
		return (24 * pixel_stride) >> 3;
	case V4L2_PIX_FMT_BGR32:
	case V4L2_PIX_FMT_RGB32:
		return (32 * pixel_stride) >> 3;
	default:
		break;
	}
	return 1;
}
EXPORT_SYMBOL_GPL(ipu_stride_to_bytes);

int ipu_degrees_to_rot_mode(enum ipu_rotate_mode *mode, int degrees,
			    bool hflip, bool vflip)
{
	u32 r90, vf, hf;

	switch (degrees) {
	case 0:
		vf = hf = r90 = 0;
		break;
	case 90:
		vf = hf = 0;
		r90 = 1;
		break;
	case 180:
		vf = hf = 1;
		r90 = 0;
		break;
	case 270:
		vf = hf = r90 = 1;
		break;
	default:
		return -EINVAL;
	}

	hf ^= (u32)hflip;
	vf ^= (u32)vflip;

	*mode = (enum ipu_rotate_mode)((r90 << 2) | (hf << 1) | vf);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_degrees_to_rot_mode);

int ipu_rot_mode_to_degrees(int *degrees, enum ipu_rotate_mode mode,
			    bool hflip, bool vflip)
{
	u32 r90, vf, hf;

	r90 = ((u32)mode >> 2) & 0x1;
	hf = ((u32)mode >> 1) & 0x1;
	vf = ((u32)mode >> 0) & 0x1;
	hf ^= (u32)hflip;
	vf ^= (u32)vflip;

	switch ((enum ipu_rotate_mode)((r90 << 2) | (hf << 1) | vf)) {
	case IPU_ROTATE_NONE:
		*degrees = 0;
		break;
	case IPU_ROTATE_90_RIGHT:
		*degrees = 90;
		break;
	case IPU_ROTATE_180:
		*degrees = 180;
		break;
	case IPU_ROTATE_90_LEFT:
		*degrees = 270;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_rot_mode_to_degrees);

bool ipu_idmac_inuse(struct ipuv3_channel *channel)
{
	return atomic_read(&channel->inuse) ? true : false;
}
EXPORT_SYMBOL_GPL(ipu_idmac_inuse);

/*
 * Returns true if the channel has been acquired successfully,
 * and false on contention.
 */
static inline bool ipu_idmac_try_get(struct ipuv3_channel *channel)
{
	return (atomic_cmpxchg(&channel->inuse, 0, 1) == 0);
}

struct ipuv3_channel *ipu_idmac_get(struct ipu_soc *ipu, unsigned num,
				    bool allow_wait)
{
	struct ipuv3_channel *channel;
	int ret;

	dev_dbg(ipu->dev, "%s %d\n", __func__, num);

	if (num > 63)
		return ERR_PTR(-ENODEV);

	channel = &ipu->channel[num];

	if (!ipu_idmac_try_get(channel)) {
		if (!allow_wait)
			return ERR_PTR(-EBUSY);

		ret = wait_event_interruptible(channel->inuse_wait,
					       ipu_idmac_try_get(channel));
		if (ret < 0)
			return ERR_PTR(ret);
	}

	return channel;
}
EXPORT_SYMBOL_GPL(ipu_idmac_get);

void ipu_idmac_put(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;

	dev_dbg(ipu->dev, "%s %d\n", __func__, channel->num);

	if (atomic_cmpxchg(&channel->inuse, 1, 0) == 1)
		wake_up_interruptible(&channel->inuse_wait);
}
EXPORT_SYMBOL_GPL(ipu_idmac_put);

int ipu_idmac_get_num(struct ipuv3_channel *channel)
{
	return channel->num;
}
EXPORT_SYMBOL_GPL(ipu_idmac_get_num);

void ipu_idmac_set_double_buffer(struct ipuv3_channel *channel,
		bool doublebuffer)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&ipu->lock, flags);

	reg = ipu_cm_read(ipu, IPU_CHA_DB_MODE_SEL(channel->num));
	if (doublebuffer)
		reg |= idma_mask(channel->num);
	else
		reg &= ~idma_mask(channel->num);
	ipu_cm_write(ipu, reg, IPU_CHA_DB_MODE_SEL(channel->num));

	__ipu_idmac_reset_current_buffer(channel);

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_set_double_buffer);

void ipu_idmac_set_triple_buffer(struct ipuv3_channel *channel,
				 bool triplebuffer)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&ipu->lock, flags);

	reg = ipu_cm_read(ipu, IPU_CHA_TRB_MODE_SEL(channel->num));
	if (triplebuffer)
		reg |= idma_mask(channel->num);
	else
		reg &= ~idma_mask(channel->num);
	ipu_cm_write(ipu, reg, IPU_CHA_TRB_MODE_SEL(channel->num));

	__ipu_idmac_reset_current_buffer(channel);

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_set_triple_buffer);

static const struct {
	int chnum;
	u32 reg;
	int shift;
} idmac_lock_en_info[] = {
	{ .chnum =  5, .reg = IDMAC_CH_LOCK_EN_1, .shift =  0, },
	{ .chnum = 11, .reg = IDMAC_CH_LOCK_EN_1, .shift =  2, },
	{ .chnum = 12, .reg = IDMAC_CH_LOCK_EN_1, .shift =  4, },
	{ .chnum = 14, .reg = IDMAC_CH_LOCK_EN_1, .shift =  6, },
	{ .chnum = 15, .reg = IDMAC_CH_LOCK_EN_1, .shift =  8, },
	{ .chnum = 20, .reg = IDMAC_CH_LOCK_EN_1, .shift = 10, },
	{ .chnum = 21, .reg = IDMAC_CH_LOCK_EN_1, .shift = 12, },
	{ .chnum = 22, .reg = IDMAC_CH_LOCK_EN_1, .shift = 14, },
	{ .chnum = 23, .reg = IDMAC_CH_LOCK_EN_1, .shift = 16, },
	{ .chnum = 27, .reg = IDMAC_CH_LOCK_EN_1, .shift = 18, },
	{ .chnum = 28, .reg = IDMAC_CH_LOCK_EN_1, .shift = 20, },
	{ .chnum = 45, .reg = IDMAC_CH_LOCK_EN_2, .shift =  0, },
	{ .chnum = 46, .reg = IDMAC_CH_LOCK_EN_2, .shift =  2, },
	{ .chnum = 47, .reg = IDMAC_CH_LOCK_EN_2, .shift =  4, },
	{ .chnum = 48, .reg = IDMAC_CH_LOCK_EN_2, .shift =  6, },
	{ .chnum = 49, .reg = IDMAC_CH_LOCK_EN_2, .shift =  8, },
	{ .chnum = 50, .reg = IDMAC_CH_LOCK_EN_2, .shift = 10, },
};

int __ipu_idmac_lock_enable(struct ipuv3_channel *channel, int num_bursts)
{
	struct ipu_soc *ipu = channel->ipu;
	u32 bursts, regval;
	int i;

	switch (num_bursts) {
	case 0:
	case 1:
		bursts = 0x00; /* locking disabled */
		break;
	case 2:
		bursts = 0x01;
		break;
	case 4:
		bursts = 0x02;
		break;
	case 8:
		bursts = 0x03;
		break;
	default:
		return -EINVAL;
	}

	for (i = 0; i < ARRAY_SIZE(idmac_lock_en_info); i++) {
		if (channel->num == idmac_lock_en_info[i].chnum)
			break;
	}
	if (i >= ARRAY_SIZE(idmac_lock_en_info))
		return -EINVAL;

	regval = ipu_idmac_read(ipu, idmac_lock_en_info[i].reg);
	regval &= ~(0x03 << idmac_lock_en_info[i].shift);
	regval |= (bursts << idmac_lock_en_info[i].shift);
	ipu_idmac_write(ipu, regval, idmac_lock_en_info[i].reg);

	return 0;
}
EXPORT_SYMBOL_GPL(__ipu_idmac_lock_enable);

int ipu_idmac_lock_enable(struct ipuv3_channel *channel, int num_bursts)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&ipu->lock, flags);
	ret = __ipu_idmac_lock_enable(channel, num_bursts);
	spin_unlock_irqrestore(&ipu->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_idmac_lock_enable);

int ipu_module_enable(struct ipu_soc *ipu, u32 mask)
{
	unsigned long lock_flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, lock_flags);

	val = ipu_cm_read(ipu, IPU_CONF);
	val |= mask;
	ipu_cm_write(ipu, val, IPU_CONF);

	spin_unlock_irqrestore(&ipu->lock, lock_flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_module_enable);

int ipu_module_disable(struct ipu_soc *ipu, u32 mask)
{
	unsigned long lock_flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, lock_flags);

	val = ipu_cm_read(ipu, IPU_CONF);
	val &= ~mask;
	ipu_cm_write(ipu, val, IPU_CONF);

	spin_unlock_irqrestore(&ipu->lock, lock_flags);

	return 0;
}
EXPORT_SYMBOL_GPL(ipu_module_disable);

void __ipu_idmac_clear_buffer_ready(struct ipuv3_channel *channel,
				    u32 buf_num)
{
	struct ipu_soc *ipu = channel->ipu;

	if (!idma_is_valid(channel) || buf_num > 2)
		return;

	ipu_cm_write(ipu, 0xF0300000, IPU_GPR); /* write one to clear */
	switch (buf_num) {
	case 0:
		ipu_cm_write(ipu, idma_mask(channel->num),
			     IPU_CHA_BUF0_RDY(channel->num));
		break;
	case 1:
		ipu_cm_write(ipu, idma_mask(channel->num),
				IPU_CHA_BUF1_RDY(channel->num));
		break;
	default:
		ipu_cm_write(ipu, idma_mask(channel->num),
				IPU_CHA_BUF2_RDY(channel->num));
		break;
	}
	ipu_cm_write(ipu, 0x0, IPU_GPR); /* write one to set */
}

void ipu_idmac_clear_buffer_ready(struct ipuv3_channel *channel, u32 buf_num)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;

	spin_lock_irqsave(&ipu->lock, flags);
	__ipu_idmac_clear_buffer_ready(channel, buf_num);
	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_clear_buffer_ready);

bool ipu_idmac_buffer_is_ready(struct ipuv3_channel *channel, u32 buf_num)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	u32 reg = 0;

	spin_lock_irqsave(&ipu->lock, flags);
	switch (buf_num) {
	case 0:
		reg = ipu_cm_read(ipu, IPU_CHA_BUF0_RDY(channel->num));
		break;
	case 1:
		reg = ipu_cm_read(ipu, IPU_CHA_BUF1_RDY(channel->num));
		break;
	case 2:
		reg = ipu_cm_read(ipu, IPU_CHA_BUF2_RDY(channel->num));
		break;
	}
	spin_unlock_irqrestore(&ipu->lock, flags);

	return ((reg & idma_mask(channel->num)) != 0);
}
EXPORT_SYMBOL_GPL(ipu_idmac_buffer_is_ready);

int ipu_idmac_current_buffer(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	int bufnum;
	u32 reg;

	spin_lock_irqsave(&ipu->lock, flags);

	reg = ipu_cm_read(ipu, IPU_CHA_TRB_MODE_SEL(channel->num));
	if (reg & idma_mask(channel->num)) {
		reg = ipu_cm_read(ipu, IPU_CHA_TRIPLE_CUR_BUF(channel->num));
		bufnum = (reg & tri_cur_buf_mask(channel->num)) >>
			tri_cur_buf_shift(channel->num);
	} else {
		reg = ipu_cm_read(ipu, IPU_CHA_CUR_BUF(channel->num));
		bufnum = (reg & idma_mask(channel->num)) ? 1 : 0;
	}

	spin_unlock_irqrestore(&ipu->lock, flags);

	return bufnum;
}
EXPORT_SYMBOL_GPL(ipu_idmac_current_buffer);

/*
 * According to the imx6 TRM, the IPU_CHA_CUR_BUF and IPU_CHA_TRIPLE_CUR_BUF
 * registers are read only. Some previous revision of the imx processors
 * must have used a write one to channel mask in IPU_CHA_CUR_BUF and
 * IPU_CHA_TRIPLE_CUR_BUF to reset the channel's internal current buffer
 * counter so that the IDMAC starts transferring from buffer 0 on next
 * channel enable. That is the theory anyway. Freescale's kernels still
 * do this, and according to them it is for backward compatibility. So we
 * keep this functionality as well, if it indeed does nothing, it is at
 * least harmless.
 */
void __ipu_idmac_reset_current_buffer(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	u32 reg;

	reg = ipu_cm_read(ipu, IPU_CHA_TRB_MODE_SEL(channel->num));
	if (reg & idma_mask(channel->num))
		ipu_cm_write(ipu, tri_cur_buf_mask(channel->num),
			     IPU_CHA_TRIPLE_CUR_BUF(channel->num));
	else
		ipu_cm_write(ipu, idma_mask(channel->num),
			     IPU_CHA_CUR_BUF(channel->num));
}

void ipu_idmac_reset_current_buffer(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;

	spin_lock_irqsave(&ipu->lock, flags);
	__ipu_idmac_reset_current_buffer(channel);
	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_reset_current_buffer);

void ipu_idmac_select_buffer(struct ipuv3_channel *channel, u32 buf_num)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned int chno = channel->num;
	unsigned long flags;

	spin_lock_irqsave(&ipu->lock, flags);

	/* Mark buffer as ready. */
	switch (buf_num) {
	case 0:
		ipu_cm_write(ipu, idma_mask(chno), IPU_CHA_BUF0_RDY(chno));
		break;
	case 1:
		ipu_cm_write(ipu, idma_mask(chno), IPU_CHA_BUF1_RDY(chno));
		break;
	case 2:
		ipu_cm_write(ipu, idma_mask(chno), IPU_CHA_BUF2_RDY(chno));
		break;
	default:
		break;
	}

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_select_buffer);

/*
 * Links IPUV3_CHANNEL_IC_PRP_ENC_MEM to IPUV3_CHANNEL_MEM_ROT_ENC
 */
int ipu_link_prp_enc_rot_enc(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PRPENC_ROT_SRC_SEL_MASK;
	fs_proc_flow1 |= (0x07 << FS_PRPENC_ROT_SRC_SEL_OFFSET);

	fs_proc_flow2 &= ~FS_PRPENC_DEST_SEL_MASK;
	fs_proc_flow2 |= (0x01 << FS_PRPENC_DEST_SEL_OFFSET);

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_link_prp_enc_rot_enc);

/*
 * Unlinks IPUV3_CHANNEL_IC_PRP_ENC_MEM from IPUV3_CHANNEL_MEM_ROT_ENC
 */
int ipu_unlink_prp_enc_rot_enc(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PRPENC_ROT_SRC_SEL_MASK;
	fs_proc_flow2 &= ~FS_PRPENC_DEST_SEL_MASK;

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_unlink_prp_enc_rot_enc);

/*
 * Links IPUV3_CHANNEL_IC_PRP_VF_MEM to IPUV3_CHANNEL_MEM_ROT_VF
 */
int ipu_link_prpvf_rot_prpvf(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PRPVF_ROT_SRC_SEL_MASK;
	fs_proc_flow1 |= (0x08 << FS_PRPVF_ROT_SRC_SEL_OFFSET);

	fs_proc_flow2 &= ~FS_PRPVF_DEST_SEL_MASK;
	fs_proc_flow2 |= (0x01 << FS_PRPVF_DEST_SEL_OFFSET);

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_link_prpvf_rot_prpvf);

/*
 * Unlinks IPUV3_CHANNEL_IC_PRP_VF_MEM from IPUV3_CHANNEL_MEM_ROT_VF
 */
int ipu_unlink_prpvf_rot_prpvf(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PRPVF_ROT_SRC_SEL_MASK;
	fs_proc_flow2 &= ~FS_PRPVF_DEST_SEL_MASK;

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_unlink_prpvf_rot_prpvf);

/*
 * Links IPUV3_CHANNEL_IC_PP_MEM to IPUV3_CHANNEL_MEM_ROT_PP
 */
int ipu_link_pp_rot_pp(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PP_ROT_SRC_SEL_MASK;
	fs_proc_flow1 |= (0x05 << FS_PP_ROT_SRC_SEL_OFFSET);

	fs_proc_flow2 &= ~FS_PP_DEST_SEL_MASK;
	fs_proc_flow2 |= (0x03 << FS_PP_DEST_SEL_OFFSET);

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_link_pp_rot_pp);

/*
 * Unlinks IPUV3_CHANNEL_IC_PP_MEM from IPUV3_CHANNEL_MEM_ROT_PP
 */
int ipu_unlink_pp_rot_pp(struct ipu_soc *ipu)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow2;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow2 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW2);

	fs_proc_flow1 &= ~FS_PP_ROT_SRC_SEL_MASK;
	fs_proc_flow2 &= ~FS_PP_DEST_SEL_MASK;

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow2, IPU_FS_PROC_FLOW2);

	spin_unlock_irqrestore(&ipu->lock, flags);
	return 0;
}
EXPORT_SYMBOL_GPL(ipu_unlink_pp_rot_pp);

/*
 * Links IPUV3_CHANNEL_CSI[0-3] to the IC PRP input channels
 * (IPUV3_CHANNEL_MEM_IC_PRP_VF and IPUV3_CHANNEL_MEM_VDI*).
 */
int ipu_link_csi_prpvf(struct ipu_soc *ipu, int csi_id)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow3;
	int ret = 0;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow3 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW3);

	switch (csi_id) {
	case 0:
		fs_proc_flow3 &= ~FS_SMFC0_DEST_SEL_MASK;
		fs_proc_flow3 |= (0x05 << FS_SMFC0_DEST_SEL_OFFSET);
		break;
	case 1:
		fs_proc_flow3 &= ~FS_SMFC1_DEST_SEL_MASK;
		fs_proc_flow3 |= (0x05 << FS_SMFC1_DEST_SEL_OFFSET);
		break;
	case 2:
		fs_proc_flow3 &= ~FS_SMFC2_DEST_SEL_MASK;
		fs_proc_flow3 |= (0x05 << FS_SMFC2_DEST_SEL_OFFSET);
		break;
	case 3:
		fs_proc_flow3 &= ~FS_SMFC3_DEST_SEL_MASK;
		fs_proc_flow3 |= (0x05 << FS_SMFC3_DEST_SEL_OFFSET);
		break;
	default:
		ret = -EINVAL;
		goto out;
	}

	fs_proc_flow1 &= ~FS_PRP_SRC_SEL_MASK;
	fs_proc_flow1 |= ((csi_id + 1) << FS_PRP_SRC_SEL_OFFSET);

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow3, IPU_FS_PROC_FLOW3);
out:
	spin_unlock_irqrestore(&ipu->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_link_csi_prpvf);

/*
 * Unlinks IPUV3_CHANNEL_CSI[0-3] from IC PRP channels.
 */
int ipu_unlink_csi_prpvf(struct ipu_soc *ipu, int csi_id)
{
	unsigned long flags;
	u32 fs_proc_flow1;
	u32 fs_proc_flow3;
	int ret = 0;

	spin_lock_irqsave(&ipu->lock, flags);

	fs_proc_flow1 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	fs_proc_flow3 = ipu_cm_read(ipu, IPU_FS_PROC_FLOW3);

	switch (csi_id) {
	case 0:
		fs_proc_flow3 &= ~FS_SMFC0_DEST_SEL_MASK;
		break;
	case 1:
		fs_proc_flow3 &= ~FS_SMFC1_DEST_SEL_MASK;
		break;
	case 2:
		fs_proc_flow3 &= ~FS_SMFC2_DEST_SEL_MASK;
		break;
	case 3:
		fs_proc_flow3 &= ~FS_SMFC3_DEST_SEL_MASK;
		break;
	default:
		ret = -EINVAL;
		goto out;
	}

	fs_proc_flow1 &= ~FS_PRP_SRC_SEL_MASK;

	ipu_cm_write(ipu, fs_proc_flow1, IPU_FS_PROC_FLOW1);
	ipu_cm_write(ipu, fs_proc_flow3, IPU_FS_PROC_FLOW3);
out:
	spin_unlock_irqrestore(&ipu->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_unlink_csi_prpvf);

int __ipu_idmac_enable_channel(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	u32 val;

	val = ipu_idmac_read(ipu, IDMAC_CHA_EN(channel->num));
	val |= idma_mask(channel->num);
	ipu_idmac_write(ipu, val, IDMAC_CHA_EN(channel->num));

	return 0;
}

int __ipu_idmac_disable_channel(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	u32 val;

	/* Disable DMA channel(s) */
	val = ipu_idmac_read(ipu, IDMAC_CHA_EN(channel->num));
	val &= ~idma_mask(channel->num);
	ipu_idmac_write(ipu, val, IDMAC_CHA_EN(channel->num));

	__ipu_idmac_reset_current_buffer(channel);

	/* Set channel buffers NOT to be ready */
	__ipu_idmac_clear_buffer_ready(channel, 0);
	__ipu_idmac_clear_buffer_ready(channel, 1);
	__ipu_idmac_clear_buffer_ready(channel, 2);

	/* Reset the double buffer */
	val = ipu_cm_read(ipu, IPU_CHA_DB_MODE_SEL(channel->num));
	val &= ~idma_mask(channel->num);
	ipu_cm_write(ipu, val, IPU_CHA_DB_MODE_SEL(channel->num));

	return 0;
}

int ipu_idmac_enable_channel(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&ipu->lock, flags);
	ret = __ipu_idmac_enable_channel(channel);
	spin_unlock_irqrestore(&ipu->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_idmac_enable_channel);

int ipu_idmac_disable_channel(struct ipuv3_channel *channel)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&ipu->lock, flags);
	ret = __ipu_idmac_disable_channel(channel);
	spin_unlock_irqrestore(&ipu->lock, flags);
	return ret;
}
EXPORT_SYMBOL_GPL(ipu_idmac_disable_channel);

void ipu_idmac_select_multi_vdi_buffer(struct ipu_soc *ipu, u32 buf_num)
{

	unsigned long flags;
	unsigned int chno = IPUV3_CHANNEL_MEM_VDI;
	u32 mask_bit =
		idma_mask(IPUV3_CHANNEL_MEM_VDI_P) |
		idma_mask(chno) |
		idma_mask(IPUV3_CHANNEL_MEM_VDI_N);

	spin_lock_irqsave(&ipu->lock, flags);

	/* Mark buffer as ready. */
	if (buf_num == 0)
		ipu_cm_write(ipu, mask_bit, IPU_CHA_BUF0_RDY(chno));
	else
		ipu_cm_write(ipu, mask_bit, IPU_CHA_BUF1_RDY(chno));

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_select_multi_vdi_buffer);


/*
 * The imx6 rev. D TRM says that enabling the WM feature will increase
 * the channel priority. Refer to Table 36-8 Calculated priority value.
 * The sub-module that is the sink or source for the channel must enable
 * watermark signal for this to take effect (SMFC_WM for instance).
 */
void ipu_idmac_enable_watermark(struct ipuv3_channel *channel, bool enable)
{
	struct ipu_soc *ipu = channel->ipu;
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_idmac_read(ipu, IDMAC_WM_EN(channel->num));
	if (enable)
		val |= 1 << (channel->num % 32);
	else
		val &= ~(1 << (channel->num % 32));
	ipu_idmac_write(ipu, val, IDMAC_WM_EN(channel->num));

	spin_unlock_irqrestore(&ipu->lock, flags);
}
EXPORT_SYMBOL_GPL(ipu_idmac_enable_watermark);

static int ipu_mem_reset(struct ipu_soc *ipu)
{
	int timeout = 100000;

	ipu_cm_write(ipu, 0x807FFFFF, IPU_MEM_RST);

	while (ipu_cm_read(ipu, IPU_MEM_RST) & 0x80000000) {
		usleep_range(10, 11);
		timeout -= 10;
		if (timeout <= 0) {
			dev_err(ipu->dev, "IPU mem reset timeout!\n");
			return -ETIME;
		}
	}

	return 0;
}

/*
 * Set the source mux for the given CSI. Selects either parallel or
 * MIPI CSI2 sources. This is not exported because it is used only
 * by ipu.
 */
void ipu_set_csi_src_mux(struct ipu_soc *ipu, int csi_id, bool mipi_csi2)
{
	unsigned long flags;
	u32 val, mask;

	mask = (csi_id == 1) ? IPU_CONF_CSI1_DATA_SOURCE :
		IPU_CONF_CSI0_DATA_SOURCE;

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_cm_read(ipu, IPU_CONF);
	if (mipi_csi2)
		val |= mask;
	else
		val &= ~mask;
	ipu_cm_write(ipu, val, IPU_CONF);

	spin_unlock_irqrestore(&ipu->lock, flags);
}

/*
 * Set the source mux for the IC. Selects either CSI[01] or the VDI.
 * This is not exported because it is used only by ipu.
 */
void ipu_set_ic_src_mux(struct ipu_soc *ipu, int csi_id, bool vdi)
{
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_cm_read(ipu, IPU_CONF);
	if (vdi) {
		val |= IPU_CONF_IC_INPUT;
	} else {
		val &= ~IPU_CONF_IC_INPUT;
		if (csi_id == 1)
			val |= IPU_CONF_CSI_SEL;
		else
			val &= ~IPU_CONF_CSI_SEL;
	}
	ipu_cm_write(ipu, val, IPU_CONF);

	spin_unlock_irqrestore(&ipu->lock, flags);
}

/*
 * Set the source for the VDIC. Selects either from CSI[01] or memory.
 * This is not exported because it is used only by ipu.
 */
void ipu_set_vdi_src_mux(struct ipu_soc *ipu, bool csi)
{
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_cm_read(ipu, IPU_FS_PROC_FLOW1);
	val &= ~FS_VDI_SRC_SEL_MASK;
	if (csi)
		val |= (0x01 << FS_VDI_SRC_SEL_OFFSET);
	ipu_cm_write(ipu, val, IPU_FS_PROC_FLOW1);

	spin_unlock_irqrestore(&ipu->lock, flags);
}

struct ipu_devtype {
	const char *name;
	unsigned long cm_ofs;
	unsigned long cpmem_ofs;
	unsigned long srm_ofs;
	unsigned long tpm_ofs;
	unsigned long csi0_ofs;
	unsigned long csi1_ofs;
	unsigned long smfc_ofs;
	unsigned long ic_ofs;
	unsigned long vdi_ofs;
	unsigned long disp0_ofs;
	unsigned long disp1_ofs;
	unsigned long dc_tmpl_ofs;
	enum ipuv3_type type;
};

static struct ipu_devtype ipu_type_imx51 = {
	.name = "IPUv3EX",
	.cm_ofs = 0x1e000000,
	.cpmem_ofs = 0x1f000000,
	.srm_ofs = 0x1f040000,
	.tpm_ofs = 0x1f060000,
	.disp0_ofs = 0x1e040000,
	.disp1_ofs = 0x1e048000,
	.dc_tmpl_ofs = 0x1f080000,
	.type = IPUV3EX,
};

static struct ipu_devtype ipu_type_imx53 = {
	.name = "IPUv3M",
	.cm_ofs = 0x06000000,
	.cpmem_ofs = 0x07000000,
	.srm_ofs = 0x07060000,
	.tpm_ofs = 0x07080000,
	.disp0_ofs = 0x06040000,
	.disp1_ofs = 0x06048000,
	.dc_tmpl_ofs = 0x07080000,
	.type = IPUV3M,
};

static struct ipu_devtype ipu_type_imx6q = {
	.name = "IPUv3H",
	.cm_ofs = 0x00200000,
	.cpmem_ofs = 0x00300000,
	.srm_ofs = 0x00340000,
	.tpm_ofs = 0x00360000,
	.csi0_ofs = 0x00230000,
	.csi1_ofs = 0x00238000,
	.disp0_ofs = 0x00240000,
	.disp1_ofs = 0x00248000,
	.smfc_ofs =  0x00250000,
	.ic_ofs = 0x00220000,
	.vdi_ofs = 0x00268000,
	.dc_tmpl_ofs = 0x00380000,
	.type = IPUV3H,
};

static struct ipu_devtype ipu_type_imx6dl = {
	.name = "IPUv3HDL",
	.cm_ofs = 0x00200000,
	.cpmem_ofs = 0x00300000,
	.srm_ofs = 0x00340000,
	.tpm_ofs = 0x00360000,
	.csi0_ofs = 0x00230000,
	.csi1_ofs = 0x00238000,
	.disp0_ofs = 0x00240000,
	.disp1_ofs = 0x00248000,
	.smfc_ofs =  0x00250000,
	.ic_ofs = 0x00220000,
	.vdi_ofs = 0x00268000,
	.dc_tmpl_ofs = 0x00380000,
	.type = IPUV3HDL,
};

static const struct of_device_id imx_ipu_dt_ids[] = {
	{ .compatible = "fsl,imx51-ipu", .data = &ipu_type_imx51, },
	{ .compatible = "fsl,imx53-ipu", .data = &ipu_type_imx53, },
	{ .compatible = "fsl,imx6q-ipu", .data = &ipu_type_imx6q, },
	{ .compatible = "fsl,imx6dl-ipu", .data = &ipu_type_imx6dl, },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_ipu_dt_ids);

static int ipu_submodules_init(struct ipu_soc *ipu,
		struct platform_device *pdev, unsigned long ipu_base,
		struct clk *ipu_clk)
{
	char *unit;
	int ret;
	struct device *dev = &pdev->dev;
	const struct ipu_devtype *devtype = ipu->devtype;

	ret = ipu_cpmem_init(ipu, dev, ipu_base + devtype->cpmem_ofs);
	if (ret) {
		unit = "cpmem";
		goto err_cpmem;
	}

	ret = ipu_csi_init(ipu, dev, 0, ipu_base + devtype->csi0_ofs,
			IPU_CONF_CSI0_EN, ipu_clk);
	if (ret) {
		unit = "csi0";
		goto err_csi_0;
	}

	ret = ipu_csi_init(ipu, dev, 1, ipu_base + devtype->csi1_ofs,
			IPU_CONF_CSI1_EN, ipu_clk);
	if (ret) {
		unit = "csi1";
		goto err_csi_1;
	}

	ret = ipu_smfc_init(ipu, dev, ipu_base + devtype->smfc_ofs,
			    IPU_CONF_SMFC_EN);
	if (ret) {
		unit = "smfc";
		goto err_smfc;
	}

	ret = ipu_ic_init(ipu, dev,
			  ipu_base + devtype->ic_ofs,
			  ipu_base + devtype->tpm_ofs,
			  IPU_CONF_IC_EN);
	if (ret) {
		unit = "ic";
		goto err_ic;
	}

	ret = ipu_irt_init(ipu, dev, IPU_CONF_ROT_EN);
	if (ret) {
		unit = "irt";
		goto err_irt;
	}

	ret = ipu_vdi_init(ipu, dev, ipu_base + devtype->vdi_ofs,
			   IPU_CONF_VDI_EN | IPU_CONF_ISP_EN |
			   IPU_CONF_IC_INPUT);
	if (ret) {
		unit = "vdi";
		goto err_vdi;
	}

	ret = ipu_di_init(ipu, dev, 0, ipu_base + devtype->disp0_ofs,
			IPU_CONF_DI0_EN, ipu_clk);
	if (ret) {
		unit = "di0";
		goto err_di_0;
	}

	ret = ipu_di_init(ipu, dev, 1, ipu_base + devtype->disp1_ofs,
			IPU_CONF_DI1_EN, ipu_clk);
	if (ret) {
		unit = "di1";
		goto err_di_1;
	}

	ret = ipu_dc_init(ipu, dev, ipu_base + devtype->cm_ofs +
			IPU_CM_DC_REG_OFS, ipu_base + devtype->dc_tmpl_ofs);
	if (ret) {
		unit = "dc_template";
		goto err_dc;
	}

	ret = ipu_dmfc_init(ipu, dev, ipu_base +
			devtype->cm_ofs + IPU_CM_DMFC_REG_OFS, ipu_clk);
	if (ret) {
		unit = "dmfc";
		goto err_dmfc;
	}

	ret = ipu_dp_init(ipu, dev, ipu_base + devtype->srm_ofs);
	if (ret) {
		unit = "dp";
		goto err_dp;
	}

	return 0;

err_dp:
	ipu_dmfc_exit(ipu);
err_dmfc:
	ipu_dc_exit(ipu);
err_dc:
	ipu_di_exit(ipu, 1);
err_di_1:
	ipu_di_exit(ipu, 0);
err_di_0:
	ipu_vdi_exit(ipu);
err_vdi:
	ipu_irt_exit(ipu);
err_irt:
	ipu_ic_exit(ipu);
err_ic:
	ipu_smfc_exit(ipu);
err_smfc:
	ipu_csi_exit(ipu, 1);
err_csi_1:
	ipu_csi_exit(ipu, 0);
err_csi_0:
	ipu_cpmem_exit(ipu);
err_cpmem:
	dev_err(&pdev->dev, "init %s failed with %d\n", unit, ret);
	return ret;
}

static void ipu_irq_handle(struct ipu_soc *ipu, const int *regs,
			   int num_regs, bool err_regs)
{
	unsigned long status, ctrl;
	int i, bit, irq_base;

	for (i = 0; i < num_regs; i++) {

		status = ipu_cm_read(ipu, IPU_INT_STAT(regs[i]));
		ctrl = ipu_cm_read(ipu, IPU_INT_CTRL(regs[i]));
		status &= ctrl;

		if (err_regs && status)
			dev_err(ipu->dev, "error, reg %d: %08lx\n",
				regs[i], status);

		irq_base = ipu->irq_start + regs[i] * 32;
		for_each_set_bit(bit, &status, 32)
			generic_handle_irq(irq_base + bit);
	}
}

static void ipu_irq_handler(unsigned int irq, struct irq_desc *desc)
{
	struct ipu_soc *ipu = irq_desc_get_handler_data(desc);
	const int int_reg[] = { 0, 1, 2, 3, 10, 11, 12, 13, 14};
	struct irq_chip *chip = irq_get_chip(irq);

	chained_irq_enter(chip, desc);

	ipu_irq_handle(ipu, int_reg, ARRAY_SIZE(int_reg), false);

	chained_irq_exit(chip, desc);
}

static void ipu_err_irq_handler(unsigned int irq, struct irq_desc *desc)
{
	struct ipu_soc *ipu = irq_desc_get_handler_data(desc);
	const int int_reg[] = { 4, 5, 8, 9};
	struct irq_chip *chip = irq_get_chip(irq);

	chained_irq_enter(chip, desc);

	ipu_irq_handle(ipu, int_reg, ARRAY_SIZE(int_reg), true);

	chained_irq_exit(chip, desc);
}

static void ipu_ack_irq(struct irq_data *d)
{
	struct ipu_soc *ipu = irq_data_get_irq_chip_data(d);
	unsigned int irq = d->irq - ipu->irq_start;

	ipu_cm_write(ipu, 1 << (irq % 32), IPU_INT_STAT(irq / 32));
}

static void ipu_unmask_irq(struct irq_data *d)
{
	struct ipu_soc *ipu = irq_data_get_irq_chip_data(d);
	unsigned int irq = d->irq - ipu->irq_start;
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&ipu->lock, flags);

	reg = ipu_cm_read(ipu, IPU_INT_CTRL(irq / 32));
	reg |= 1 << (irq % 32);
	ipu_cm_write(ipu, reg, IPU_INT_CTRL(irq / 32));

	spin_unlock_irqrestore(&ipu->lock, flags);
}

static void ipu_mask_irq(struct irq_data *d)
{
	struct ipu_soc *ipu = irq_data_get_irq_chip_data(d);
	unsigned int irq = d->irq - ipu->irq_start;
	unsigned long flags;
	u32 reg;

	spin_lock_irqsave(&ipu->lock, flags);

	reg = ipu_cm_read(ipu, IPU_INT_CTRL(irq / 32));
	reg &= ~(1 << (irq % 32));
	ipu_cm_write(ipu, reg, IPU_INT_CTRL(irq / 32));

	spin_unlock_irqrestore(&ipu->lock, flags);
}

static struct irq_chip ipu_irq_chip = {
	.name = "IPU",
	.irq_ack = ipu_ack_irq,
	.irq_mask = ipu_mask_irq,
	.irq_unmask = ipu_unmask_irq,
};

void ipu_dump(struct ipu_soc *ipu)
{
	int i;
	dev_dbg(ipu->dev, "IPU_CONF = \t0x%08X\n",
		ipu_cm_read(ipu, IPU_CONF));
	dev_dbg(ipu->dev, "IDMAC_CONF = \t0x%08X\n",
		ipu_idmac_read(ipu, IDMAC_CONF));
	dev_dbg(ipu->dev, "IDMAC_CHA_EN1 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_CHA_EN(0)));
	dev_dbg(ipu->dev, "IDMAC_CHA_EN2 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_CHA_EN(32)));
	dev_dbg(ipu->dev, "IDMAC_CHA_PRI1 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_CHA_PRI(0)));
	dev_dbg(ipu->dev, "IDMAC_CHA_PRI2 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_CHA_PRI(32)));
	dev_dbg(ipu->dev, "IDMAC_BAND_EN1 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_BAND_EN(0)));
	dev_dbg(ipu->dev, "IDMAC_BAND_EN2 = \t0x%08X\n",
	       ipu_idmac_read(ipu, IDMAC_BAND_EN(32)));
	dev_dbg(ipu->dev, "IPU_CHA_DB_MODE_SEL0 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_CHA_DB_MODE_SEL(0)));
	dev_dbg(ipu->dev, "IPU_CHA_DB_MODE_SEL1 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_CHA_DB_MODE_SEL(32)));
	dev_dbg(ipu->dev, "IPU_CHA_TRB_MODE_SEL0 = \t0x%08X\n",
		ipu_cm_read(ipu, IPU_CHA_TRB_MODE_SEL(0)));
	dev_dbg(ipu->dev, "IPU_CHA_TRB_MODE_SEL1 = \t0x%08X\n",
		ipu_cm_read(ipu, IPU_CHA_TRB_MODE_SEL(32)));
	dev_dbg(ipu->dev, "IPU_FS_PROC_FLOW1 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_FS_PROC_FLOW1));
	dev_dbg(ipu->dev, "IPU_FS_PROC_FLOW2 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_FS_PROC_FLOW2));
	dev_dbg(ipu->dev, "IPU_FS_PROC_FLOW3 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_FS_PROC_FLOW3));
	dev_dbg(ipu->dev, "IPU_FS_DISP_FLOW1 = \t0x%08X\n",
	       ipu_cm_read(ipu, IPU_FS_DISP_FLOW1));
	for (i = 0; i < 15; i++)
		dev_dbg(ipu->dev, "IPU_INT_CTRL(%d) = \t%08X\n", i,
			ipu_cm_read(ipu, IPU_INT_CTRL(i)));
}
EXPORT_SYMBOL_GPL(ipu_dump);

int ipu_idmac_channel_irq(struct ipu_soc *ipu, struct ipuv3_channel *channel,
		enum ipu_channel_irq irq_type)
{
	return ipu->irq_start + irq_type + channel->num;
}
EXPORT_SYMBOL_GPL(ipu_idmac_channel_irq);

int ipu_error_irq(struct ipu_soc *ipu, enum ipu_err_irq err_irq_type)
{
	return ipu->irq_start + err_irq_type;
}
EXPORT_SYMBOL_GPL(ipu_error_irq);

int ipu_gp_irq(struct ipu_soc *ipu, enum ipu_gp_irq gp_irq_type)
{
	return ipu->irq_start + gp_irq_type;
}
EXPORT_SYMBOL_GPL(ipu_gp_irq);

static void ipu_submodules_exit(struct ipu_soc *ipu)
{
	ipu_dp_exit(ipu);
	ipu_dmfc_exit(ipu);
	ipu_dc_exit(ipu);
	ipu_di_exit(ipu, 1);
	ipu_di_exit(ipu, 0);
	ipu_vdi_exit(ipu);
	ipu_irt_exit(ipu);
	ipu_ic_exit(ipu);
	ipu_smfc_exit(ipu);
	ipu_csi_exit(ipu, 1);
	ipu_csi_exit(ipu, 0);
	ipu_cpmem_exit(ipu);
}

static int ipu_irq_init(struct ipu_soc *ipu)
{
	int i;

	ipu->irq_start = irq_alloc_descs(-1, 0, IPU_NUM_IRQS, 0);
	if (ipu->irq_start < 0) {
		dev_err(ipu->dev, "irq_alloc_descs() failed with %d!\n",
			ipu->irq_start);
		return ipu->irq_start;
	}

	for (i = ipu->irq_start; i < ipu->irq_start + IPU_NUM_IRQS; i++) {
		irq_set_chip_and_handler(i, &ipu_irq_chip, handle_level_irq);
		set_irq_flags(i, IRQF_VALID);
		irq_set_chip_data(i, ipu);
	}

	irq_set_chained_handler(ipu->irq_sync, ipu_irq_handler);
	irq_set_handler_data(ipu->irq_sync, ipu);
	irq_set_chained_handler(ipu->irq_err, ipu_err_irq_handler);
	irq_set_handler_data(ipu->irq_err, ipu);

	return 0;
}

static void ipu_irq_exit(struct ipu_soc *ipu)
{
	int i;

	irq_set_chained_handler(ipu->irq_err, NULL);
	irq_set_handler_data(ipu->irq_err, NULL);
	irq_set_chained_handler(ipu->irq_sync, NULL);
	irq_set_handler_data(ipu->irq_sync, NULL);

	for (i = ipu->irq_start; i < ipu->irq_start + IPU_NUM_IRQS; i++) {
		set_irq_flags(i, 0);
		irq_set_chip(i, NULL);
		irq_set_chip_data(i, NULL);
	}

	irq_free_descs(ipu->irq_start, IPU_NUM_IRQS);
}

/*
 * The IPU has four AXI bus masters. Each bus master can generate
 * requests with settable bus priorities, or QoS value, between 0
 * (lowest priority) and 7 (higher). A special QoS value of 0xf
 * bypasses the AXI bus arbiter for highest-priority "real-time"
 * requests.
 */
static void ipu_set_axi_bus_prio(struct ipu_soc *ipu, u32 *prio)
{
	unsigned long flags;
	unsigned gpr;
	int bus_id;

	spin_lock_irqsave(&ipu->lock, flags);

	gpr = (ipu_get_num(ipu) == 0) ? IOMUXC_GPR6 : IOMUXC_GPR7;

	for (bus_id = 0; bus_id < 4; bus_id++) {
		regmap_update_bits(ipu->gp_reg, gpr,
				   IPU_GPR_QOS_W_ID_MASK(bus_id),
				   IPU_GPR_QOS_W_ID(bus_id, prio[bus_id]));
		regmap_update_bits(ipu->gp_reg, gpr,
				   IPU_GPR_QOS_R_ID_MASK(bus_id),
				   IPU_GPR_QOS_R_ID(bus_id, prio[bus_id]));
	}

	spin_unlock_irqrestore(&ipu->lock, flags);
}

/*
 * Enable the limit on the number of pending non-realtime channel
 * read/write requests in the request queues. A limit value of
 * 0 disables non-realtime read/write request limits.
 */
static void ipu_set_nonrt_req_limit(struct ipu_soc *ipu,
				    u32 limit, bool is_write)
{
	u32 limit_mask, limit_off, enable, val;
	unsigned long flags;

	if (is_write) {
		limit_mask = NONRT_REQ_W_LIM_MASK;
		limit_off = NONRT_REQ_W_LIM_OFFSET;
		enable = NONRT_REQ_W_LIM_EN;
	} else {
		limit_mask = NONRT_REQ_R_LIM_MASK;
		limit_off = NONRT_REQ_R_LIM_OFFSET;
		enable = NONRT_REQ_R_LIM_EN;
	}

	spin_lock_irqsave(&ipu->lock, flags);

	val = ipu_idmac_read(ipu, IDMAC_CONF);

	val &= ~limit_mask;
	if (limit)
		val |= (enable | (limit << limit_off));
	else
		val &= ~enable;

	ipu_idmac_write(ipu, val, IDMAC_CONF);

	spin_unlock_irqrestore(&ipu->lock, flags);
}

static int ipu_probe_counter;

static int ipu_probe(struct platform_device *pdev)
{
	const struct of_device_id *of_id =
			of_match_device(imx_ipu_dt_ids, &pdev->dev);
	struct ipuv3_platform_data *pdata = pdev->dev.platform_data;
	struct device_node *np = pdev->dev.of_node;
	struct ipu_soc *ipu;
	struct resource *res;
	unsigned long ipu_base;
	int i, ret, irq_sync, irq_err;
	const struct ipu_devtype *devtype;
	u32 axi_bus_prio[4];
	u32 nonrt_limit;

	if (ipu_probe_counter >= IPU_NUM)
		return -ENODEV;

	devtype = of_id->data;

	dev_dbg(&pdev->dev, "Initializing %s\n", devtype->name);

	irq_sync = platform_get_irq(pdev, 0);
	irq_err = platform_get_irq(pdev, 1);
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (!res || irq_sync < 0 || irq_err < 0) {
		dev_err(&pdev->dev, "failed to get platform resources!\n");
		return -ENODEV;
	}

	dev_dbg(&pdev->dev, "irq_sync: %d irq_err: %d\n",
		irq_sync, irq_err);

	ipu_base = res->start;

	ipu = devm_kzalloc(&pdev->dev, sizeof(*ipu), GFP_KERNEL);
	if (!ipu) {
		dev_err(&pdev->dev, "failed to alloc ipu_soc!\n");
		return -ENOMEM;
	}

	for (i = 0; i < 64; i++) {
		struct ipuv3_channel *channel = &ipu->channel[i];
		channel->ipu = ipu;
		channel->num = i;
		channel->alt_num = ipu_cpmem_get_third_buf_cpmem_num(channel);
		init_waitqueue_head(&channel->inuse_wait);
	}

	ipu->id = ipu_probe_counter++;
	ipu->devtype = devtype;
	ipu->ipu_type = devtype->type;

	spin_lock_init(&ipu->lock);

	dev_dbg(&pdev->dev, "cm_reg:   0x%08lx\n",
		ipu_base + devtype->cm_ofs);
	dev_dbg(&pdev->dev, "idmac:    0x%08lx\n",
		ipu_base + devtype->cm_ofs + IPU_CM_IDMAC_REG_OFS);
	dev_dbg(&pdev->dev, "cpmem:    0x%08lx\n",
		ipu_base + devtype->cpmem_ofs);
	dev_dbg(&pdev->dev, "csi0:    0x%08lx\n",
		ipu_base + devtype->csi0_ofs);
	dev_dbg(&pdev->dev, "csi1:    0x%08lx\n",
		ipu_base + devtype->csi1_ofs);
	dev_dbg(&pdev->dev, "smfc:    0x%08lx\n",
		ipu_base + devtype->smfc_ofs);
	dev_dbg(&pdev->dev, "disp0:    0x%08lx\n",
		ipu_base + devtype->disp0_ofs);
	dev_dbg(&pdev->dev, "disp1:    0x%08lx\n",
		ipu_base + devtype->disp1_ofs);
	dev_dbg(&pdev->dev, "srm:      0x%08lx\n",
		ipu_base + devtype->srm_ofs);
	dev_dbg(&pdev->dev, "tpm:      0x%08lx\n",
		ipu_base + devtype->tpm_ofs);
	dev_dbg(&pdev->dev, "dc:       0x%08lx\n",
		ipu_base + devtype->cm_ofs + IPU_CM_DC_REG_OFS);
	dev_dbg(&pdev->dev, "ic:       0x%08lx\n",
		ipu_base + devtype->cm_ofs + IPU_CM_IC_REG_OFS);
	dev_dbg(&pdev->dev, "dmfc:     0x%08lx\n",
		ipu_base + devtype->cm_ofs + IPU_CM_DMFC_REG_OFS);

	ipu->cm_reg = devm_ioremap(&pdev->dev,
			ipu_base + devtype->cm_ofs, PAGE_SIZE);
	ipu->idmac_reg = devm_ioremap(&pdev->dev,
			ipu_base + devtype->cm_ofs + IPU_CM_IDMAC_REG_OFS,
			PAGE_SIZE);

	if (!ipu->cm_reg || !ipu->idmac_reg) {
		ret = -ENOMEM;
		dev_err(&pdev->dev, "failed to map CM/IDMAC registers!\n");
		goto failed_ioremap;
	}

	ipu->gp_reg = syscon_regmap_lookup_by_compatible(
		"fsl,imx6q-iomuxc-gpr");
	if (IS_ERR(ipu->gp_reg)) {
		ret = PTR_ERR(ipu->gp_reg);
		dev_err(&pdev->dev, "failed to regmap GPR with %d\n", ret);
		goto failed_regmap;
	}

	ipu->clk = devm_clk_get(&pdev->dev, "bus");
	if (IS_ERR(ipu->clk)) {
		ret = PTR_ERR(ipu->clk);
		dev_err(&pdev->dev, "clk_get failed with %d\n", ret);
		goto failed_clk_get;
	}

	platform_set_drvdata(pdev, ipu);

	clk_prepare_enable(ipu->clk);

	ipu->dev = &pdev->dev;
	ipu->irq_sync = irq_sync;
	ipu->irq_err = irq_err;

	ret = ipu_irq_init(ipu);
	if (ret)
		goto out_failed_irq;

	if (pdata && pdata->reset)
		pdata->reset(ipu->id);
	ipu_mem_reset(ipu);

	/* Set MCU_T to divide MCU access window into 2 */
	ipu_cm_write(ipu, 0x00400000L | (IPU_MCU_T_DEFAULT << 18),
			IPU_DISP_GEN);

	ret = of_property_read_u32_array(np, "axi-bus-prio", axi_bus_prio, 4);
	if (!ret)
		ipu_set_axi_bus_prio(ipu, axi_bus_prio);

	ret = of_property_read_u32(np, "nonrt-read-req-limit", &nonrt_limit);
	if (!ret)
		ipu_set_nonrt_req_limit(ipu, nonrt_limit, false);

	ret = of_property_read_u32(np, "nonrt-write-req-limit", &nonrt_limit);
	if (!ret)
		ipu_set_nonrt_req_limit(ipu, nonrt_limit, true);

	ret = ipu_submodules_init(ipu, pdev, ipu_base, ipu->clk);
	if (ret)
		goto failed_submodules_init;

	return 0;

failed_submodules_init:
	ipu_irq_exit(ipu);
out_failed_irq:
failed_clk_get:
failed_regmap:
failed_ioremap:
	return ret;
}

static int ipu_remove(struct platform_device *pdev)
{
	struct ipu_soc *ipu = platform_get_drvdata(pdev);
	struct resource *res;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	ipu_submodules_exit(ipu);
	ipu_irq_exit(ipu);

	clk_disable_unprepare(ipu->clk);

	return 0;
}

static struct platform_driver imx_ipu_driver = {
	.driver = {
		.name = "imx-ipuv3",
		.of_match_table = imx_ipu_dt_ids,
	},
	.probe = ipu_probe,
	.remove = ipu_remove,
};

static int __init imx_ipu_init(void)
{
	return platform_driver_register(&imx_ipu_driver);
}

static void __exit imx_ipu_exit(void)
{
	platform_driver_unregister(&imx_ipu_driver);
}

subsys_initcall(imx_ipu_init);
module_exit(imx_ipu_exit);

MODULE_DESCRIPTION("i.MX IPU v3 driver");
MODULE_AUTHOR("Sascha Hauer <s.hauer@pengutronix.de>");
MODULE_LICENSE("GPL");
