/*
 * Copyright 2008-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 * Copyright (C) 2012 Mentor Graphics, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/dma-mapping.h>
#include <linux/sched.h>
#include <linux/of_platform.h>
#include <linux/errno.h>
#include <linux/mxc_asrc.h>
#include <linux/poll.h>
#include <linux/platform_data/dma-imx.h>

static dev_t asrc_devt;

static struct class *asrc_class;

/**
 * find_uid_key() - Find UID unique ID in section manager
 * @private	struct fsl_asrc_private *
 * @uid		int uid
 * @return	SECTION_INVALID on UID doesnt exist, >=0 section manager index
 *
 * Note: This function needs to be held by struct fsl_asrc_private's data_lock
 */
static int find_uid_key(struct fsl_asrc_private *private, int uid)
{
	int i, err = SECTION_INVALID;

	/* iterate over all pairs */
	for (i = 0; i < ASRC_PAIR_NUM; i++) {
		if (private->manager[i].uid == uid) {
			err = i;
			break;
		}
	}

	return err;
}

/**
 * find_empty() - Find empty section manager index
 * @private	struct fsl_asrc_private *
 * @return	SECTION_INVALID on UID doesnt exist, >=0 section manager index
 *
 * Note: This function needs to be held by struct fsl_asrc_private's data_lock
 */
static int find_empty(struct fsl_asrc_private *private)
{
	int i, ret = SECTION_INVALID;

	/* iterate over all pairs */
	for (i = 0; i < ASRC_PAIR_NUM; i++) {
		if ((private->manager[i].uid == SECTION_INVALID) &&
			(private->manager[i].pair_idx == SECTION_INVALID)) {
			ret = i;
			break;
		}
	}

	return ret;
}

/**
 * clean_manager_pair() - Make inactive a single section the manager contains
 * and invalidate the manager index if both sections are inactive
 * @private	struct fsl_asrc_private *
 * @return	Sections remaining to be cleaned
 *
 * Note: This function needs to be held by struct fsl_asrc_private's data_lock
 */
static int clean_manager_pair(struct fsl_asrc_private *private,
		int key, enum asrc_section_direction direction)
{
	private->manager[key].usage[direction] = INACTIVE;
	private->manager[key].section[direction] = NULL;

	if ((private->manager[key].usage[SECTION_INPUT] == INACTIVE) &&
			(private->manager[key].usage[SECTION_OUTPUT] ==
					INACTIVE)) {
		private->manager[key].pair_idx = SECTION_INVALID;
		private->manager[key].uid = SECTION_INVALID;
		return 0;
	}

	return 1;
}

/**
 * clean_manager_uid() - Make inactive a single section specified by the UID
 * and section direction
 * @private	struct fsl_asrc_private *
 * @uid		int
 * @direction	enum asrc_section_direction
 * @return	0 on Success
 *
 * Note: This function needs to be held by struct fsl_asrc_private's data_lock
 */
static int clean_manager_uid(struct fsl_asrc_private *private,
		int uid,
		enum asrc_section_direction direction)
{
	int key = find_uid_key(private, uid);

	dev_dbg(private->dev, "%s cleaning managed pair\n", __func__);

	if (key < 0)
		return -EINVAL;
	else
		return clean_manager_pair(private, key, direction);
}

/**
 * free_dma_buffer() - Frees an allocated DMA buffer of a section
 * @private	struct fsl_asrc_private *
 */
static void free_dma_buffer(struct asrc_section_private *section_priv)
{
	if (section_priv->section_dma.dma.dma_vaddr != NULL) {
		dma_free_coherent(section_priv->private->dev,
				section_priv->section_dma.dma.length,
				section_priv->section_dma.dma.dma_vaddr,
				section_priv->section_dma.dma.dma_paddr);
		section_priv->section_dma.dma.dma_vaddr = NULL;
		section_priv->section_dma.dma.dma_paddr = 0;
	}
	return;
}

/**
 * allocate_dma_buffer() - Allocates a DMA buffer for a section
 * @private	struct fsl_asrc_private *
 * @return	0 on Success, NOBUFS on failure
 */
static int allocate_dma_buffer(struct asrc_section_private *section_priv)
{
	struct dma_block *dma = &section_priv->section_dma.dma;

	dma->dma_vaddr =
		dma_alloc_coherent(section_priv->private->dev,
			dma->length, &dma->dma_paddr,
			GFP_KERNEL | GFP_DMA);
	if (!dma->dma_vaddr)
		goto exit;

	dev_dbg(section_priv->private->dev, "%s dma vaddr:0x%lx\n",
			__func__, (unsigned long)dma->dma_vaddr);
	dev_dbg(section_priv->private->dev, "%s dma paddr:0x%lx\n",
			__func__, (unsigned long)dma->dma_paddr);

	return 0;

exit:
	dev_err(section_priv->private->dev, "can't allocate buffer\n");
	free_dma_buffer(section_priv);
	return -ENOBUFS;
}

/**
 * configure_dma_buffer() - sets buffer config, allocates buffer and initializes
 * a sync struct
 * @private	struct fsl_asrc_private *
 * @config	struct section_buffer_config *
 * @return	0 on Success
 */
static int configure_dma_buffer(
		struct asrc_section_private *section_priv,
		struct section_buffer_config *config)
{
	int err;

	if (section_priv->section_dma.dma.dma_vaddr) {
		dev_err(section_priv->private->dev,
				"buffer for section already allocated\n");
		return -EPERM;
	}

	memcpy(&section_priv->buffer_config, config,
			sizeof(struct section_buffer_config));

	/* check incoming data */
	if ((config->buffer_bytes == 0) || (config->periods == 0))
		return -EINVAL;

	section_priv->section_dma.dma.length = config->buffer_bytes;
	section_priv->section_dma.dma.periods = config->periods;

	err = allocate_dma_buffer(section_priv);
	if (err < 0)
		return err;

	section_priv->sync.appl_pos =
		section_priv->sync.avail_min =
		section_priv->sync.stop_threshold =
		section_priv->sync.boundary =
		section_priv->sync.hw_pos = 0;
	section_priv->sync.state =
		asrc_get_section_state(section_priv->section_params);

	return 0;
}

static bool filter(struct dma_chan *chan, void *param)
{

	if (!imx_dma_is_general_purpose(chan))
		return false;

	chan->private = param;
	return true;
}

static struct dma_chan *allocate_dma_channel(u32 dma_req)
{
	dma_cap_mask_t mask;
	struct imx_dma_data dma_data = {0};

	dma_data.peripheral_type = IMX_DMATYPE_ASRC;
	dma_data.priority = DMA_PRIO_MEDIUM;
	dma_data.dma_request = dma_req;

	/* Try to grab a DMA channel */
	dma_cap_zero(mask);
	dma_cap_set(DMA_SLAVE, mask);

	return dma_request_channel(mask, filter, &dma_data);
}

/**
 * configure_dma_slave() - configure and allocate the dma slave then set the
 * dma tx descriptor
 * @section_priv	struct asrc_section_private *
 * @return	0 SUCCESS, EINVAL DMA could not be setup properly
 */
static int configure_dma_slave(struct asrc_section_private *section_priv)
{
	struct dma_slave_config slave_config = {0};
	struct fsl_asrc_private *private = section_priv->private;
	struct section_dma *section_dma = &section_priv->section_dma;
	int ret;
	int datawidth_bytes =
		asrc_get_format_data_bytewidth(section_priv->section_params);

	/* this is internally multiplied by the datawidth in the sdma */
	u32 maxburst = asrc_get_period(section_priv->section_params);

	if (datawidth_bytes < 0)
		return datawidth_bytes;

	/* direction is only modified during 'open' */
	if (section_priv->direction == SECTION_INPUT) {
		slave_config.direction = DMA_MEM_TO_DEV;
		slave_config.dst_addr =
			asrc_get_fifo_addr(section_priv->section_params);
		slave_config.dst_addr_width = datawidth_bytes;
		slave_config.dst_maxburst = maxburst;
	} else {
		slave_config.direction = DMA_DEV_TO_MEM;
		slave_config.src_addr =
			asrc_get_fifo_addr(section_priv->section_params);
		slave_config.src_addr_width = datawidth_bytes;
		slave_config.src_maxburst = maxburst;

	}

	ret = dmaengine_slave_config(section_dma->channel, &slave_config);
	if (ret) {
		dev_err(private->dev,
		"%s slave config for %s failed with %d\n",
		__func__,
		((section_priv->direction == SECTION_INPUT) ?
				"INPUT FIFO" : "OUTPUT FIFO"),
				ret);
		return ret;
	}

	dev_dbg(private->dev, "%s %s dma slave configured\n"
			"\tdst_addr:0x%08lx\n"
			"\tsrc_addr:0x%08lx\n"
			"\taddr_width:%u\n"
			"\tmaxburst:%u\n", __func__,
			(section_priv->direction == SECTION_INPUT) ?
					"DMA_MEM_TO_DEV" : "DMA_DEV_TO_MEM",
					(unsigned long)slave_config.dst_addr,
					(unsigned long)slave_config.src_addr,
					(unsigned int)datawidth_bytes,
					(unsigned int)maxburst);

	section_dma->desc =
			section_dma->channel->device->device_prep_dma_cyclic(
			section_dma->channel, section_dma->dma.dma_paddr,
			section_priv->buffer_config.buffer_bytes,
			(section_priv->buffer_config.buffer_bytes /
			section_priv->buffer_config.periods),
			(section_priv->direction == SECTION_INPUT) ?
			DMA_TO_DEVICE : DMA_FROM_DEVICE,
			0, NULL);



	if (section_dma->desc) {
		/* Add to pending queue */
		dmaengine_submit(section_priv->section_dma.desc);
		return 0;
	} else
		return -EINVAL;
}

/**
 * frames_avail() - calculate available buffer frames based on the sync struct
 * @sync	struct section_sync *
 * @dma		struct dma_block *
 * @direction	enum asrc_section_direction
 * @return	# of frames available for use
 */
static unsigned int frames_avail(struct section_sync *sync,
		struct dma_block *dma,
		enum asrc_section_direction direction)
{
	long avail;

	if (direction == SECTION_INPUT) {
		/* we are managing the buffer in a unit of frames */
		avail = (dma->periods * dma->frames_period) -
				sync->appl_pos + sync->hw_pos;
		if (avail < 0)
			avail += sync->boundary;
		else if (avail >= sync->boundary)
			avail -= sync->boundary;
	} else {
		avail = sync->hw_pos - sync->appl_pos;
		if (avail < 0)
			avail += sync->boundary;
	}

	return avail;
}

/**
 * asrc_xrun_callback() - Handles an xrun event from the core
 */
static void asrc_xrun_callback(void *data)
{
	struct asrc_section_params *params =
			(struct asrc_section_params *)data;

	if (params) {
		/* stop the current transfer */
		asrc_stop_conversion(params);
	}
}

static void asrc_statechange_callback(unsigned long data)
{
	struct asrc_section_private *section_priv =
			(struct asrc_section_private *)data;

	if (data) {
		dev_dbg(section_priv->private->dev,
			"############################# %s uid:%d %s state change\n",
			__func__,
			section_priv->uid,
			((section_priv->direction == SECTION_INPUT) ?
					"INPUT" : "OUTPUT"));

		wake_up_interruptible(&section_priv->wait_queue);
	}

	return;
}

/**
 * asrc_dma_callback() - update our poll state if necessary
 * @data	void * - struct asrc_section_private *
 */
static void asrc_dma_callback(void *data)
{
	struct asrc_section_private *section_priv =
			(struct asrc_section_private *)data;
	struct section_sync *sync = &section_priv->sync;
	struct dma_block *dma = &section_priv->section_dma.dma;

	unsigned long lock_flags;

	unsigned int hwpos;
	unsigned int avail;

	spin_lock_irqsave(&section_priv->sect_lock, lock_flags);
	hwpos = sync->hw_pos + dma->frames_period;
	if (hwpos >= sync->boundary)
		hwpos -= sync->boundary;
	sync->hw_pos = hwpos;

	avail = frames_avail(sync, dma, section_priv->direction);

	if (avail >= sync->stop_threshold) {
		dev_dbg(section_priv->private->dev,
				"%s %d avail:%u >= stop_thresh:%u\n",
				__func__,
				section_priv->direction,
				avail, sync->stop_threshold);
		asrc_xrun(section_priv->section_params);
	}

	sync->state =
			asrc_get_section_state(section_priv->section_params);

	if ((avail >= sync->avail_min) ||
			(sync->state == SECTION_SOFT_XRUN) ||
			(sync->state == SECTION_HARD_XRUN) ||
			(sync->state == SECTION_IRQ_ERR))
		wake_up_interruptible(&section_priv->wait_queue);

	spin_unlock_irqrestore(&section_priv->sect_lock, lock_flags);

	return;
}

/**
 * unwind_section() - unwind a section from the prepare state or higher
 * @section_priv	struct asrc_section_private *
 * @state	enum asrc_section_state
 * @return	0 successful unwind, EPERM attempt to unwind from invalid state
 */
static int unwind_section(struct asrc_section_private *section_priv,
		enum asrc_section_state state)
{
	struct section_dma *section_dma = &section_priv->section_dma;
	unsigned long lock_flags;
	int err = 0;

	switch (state) {
	case SECTION_PREPARE:
		asrc_setup(section_priv->section_params);
	case SECTION_SETUP:
		/* free buffer and dma resources if necessary */
		if (section_dma->channel)
			dma_release_channel(section_dma->channel);
		asrc_open_section(section_priv->section_params);
	case SECTION_OPENED:
		/* Note: unless both sides are completely closed,
		 * a section will remain in open state even if
		 * the file descriptor is closed */
		free_dma_buffer(section_priv);

		spin_lock_irqsave(&section_priv->private->data_lock,
				lock_flags);
		err = clean_manager_uid(section_priv->private,
				section_priv->uid,
				section_priv->direction);
		asrc_set_callback(section_priv->section_params,
							NULL, 0);

		if (err == 0)
			asrc_release_pair(section_priv->section_params);

		spin_unlock_irqrestore(
				&section_priv->private->data_lock,
				lock_flags);

		break;
	default:
		return -EPERM;
	};

	return 0;
}

/**
 * ioctl_set_filter_settings() - set ASRC filter setting for section
 * @section_priv	struct asrc_section_private *
 * @arg		unsigned long - enum filter_settings
 * @return	0 success, EPERM not allowed to be called currently, EINVAL on
 *		invalid argument
 */
static int ioctl_set_filter_settings(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	enum filter_settings settings;
	int err;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_SET_FILTER called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	if (copy_from_user(&settings, (void __user *)arg,
			   sizeof(enum filter_settings)))
		return -EFAULT;

	err = asrc_set_filter_settings(section_priv->section_params, settings);

	dev_dbg(section_priv->private->dev, "%s setting filter settings:%d\n",
					__func__, settings);
	return err;
}

/**
 * ioctl_get_section_capabilities() - retrieve supported format, rate, channels
 * from hardware
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - struct asrc_section_capabilities
 * @return	0 SUCCESS, EPERM not allowed to be called currently
 */
static int ioctl_get_section_capabilities(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	struct asrc_section_capabilities capabilities;
	int err = 0;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_GET_CAPABILITY called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	err = asrc_get_capabilities(section_priv->section_params,
			&capabilities);

	dev_dbg(section_priv->private->dev, "%s getting capabilities:\n"
			"formats:0x%016llx\n"
			"channels_max:%d\n"
			"channels_min:%d\n"
			"rate_max:%d\n"
			"rate_min:%d\n"
			, __func__, capabilities.formats,
			capabilities.channels_max,
			capabilities.channels_min,
			capabilities.rate_max,
			capabilities.rate_min);

	if (copy_to_user((void __user *)arg, &capabilities,
			   sizeof(struct asrc_section_capabilities)))
		return -EFAULT;

	return err;
}

/**
 * ioctl_get_buffer_capabilities() - retrieve min/max buffer size, period bytes,
 * and periods.
 * from hardware
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - struct buffer_capabilities
 * @return	0 SUCCESS, EPERM not allowed to be called currently
 */
static int ioctl_get_buffer_capabilities(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	struct buffer_capabilities capabilities;
	int err = 0;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_GET_BUF_CAPABILITY called "
				"before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	err = asrc_get_buffer_capabilities(section_priv->section_params,
			&capabilities);

	dev_dbg(section_priv->private->dev, "%s getting capabilities:\n"
			"bufbytes_min:%lu\n"
			"bufbytes_max:%lu\n"
			"periodbytes_min:%u\n"
			"periodbytes_max:%u\n"
			"periods_min:%u\n"
			"periods_max:%u\n"
			, __func__, capabilities.bufbytes_min,
			capabilities.bufbytes_max,
			capabilities.periodbytes_min,
			capabilities.periodbytes_max,
			capabilities.periods_min,
			capabilities.periods_max);

	if (copy_to_user((void __user *)arg, &capabilities,
			   sizeof(struct buffer_capabilities)))
		return -EFAULT;

	return err;
}

/**
 * ioctl_set_period() - retrieve min/max buffer size, period bytes,
 * and periods.
 * from hardware
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - unsigned char period
 * @return	0 SUCCESS, EPERM not allowed to be called currently, EINVAL
 *		invalid argument
 */
static int ioctl_set_period(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	unsigned char period;
	int err;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_SET_PERIOD called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	if (copy_from_user(&period, (void __user *)arg,
			   sizeof(unsigned char)))
		return -EFAULT;

	err = asrc_set_period(section_priv->section_params, period);

	dev_dbg(section_priv->private->dev, "%s setting period:%d\n",
					__func__, period);

	return err;
}

/**
 * ioctl_setup_section() - Sets section channels, format and rate
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - struct section_config
 * @return	0 success, EPERM not allowed to be called currently, EINVAL on
 *		invalid argument
 */
static int ioctl_setup_section(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	struct section_config config;
	int err;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_SETUP_SECTION called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	if (copy_from_user(&config, (void __user *)arg,
			   sizeof(struct section_config)))
		return -EFAULT;

	err = asrc_setup_section(section_priv->section_params, &config);

	dev_dbg(section_priv->private->dev,
			"%s setup section config\n", __func__);

	return err;
}

/**
 * ioctl_sync_section() - updates a section's internal sync structure from
 * userspace's sync structure
 * @section_priv	struct asrc_section_private *
 * @arg		unsigned long - struct section_sync
 * @return	0 success, EINVAL invalid argument
 */
static int ioctl_sync_section(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	struct fsl_asrc_private *private = section_priv->private;
	struct section_sync user_sync;
	struct section_sync *driver_sync = &section_priv->sync;
	int err = 0;
	unsigned long lock_flags;

	if (!(section_priv->section_params)) {
		dev_err(private->dev,
				"ASRC_SYNC called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	if (copy_from_user(&user_sync, (void __user *)arg,
			   sizeof(struct section_sync)))
		return -EFAULT;

	if ((user_sync.appl_pos > user_sync.boundary)) {
		dev_err(private->dev,
			"error: appl_pos: %d > boundary: %d\n",
			user_sync.appl_pos,
			user_sync.boundary);
		return -EINVAL;
	}

	spin_lock_irqsave(&section_priv->sect_lock, lock_flags);
	driver_sync->state =
			asrc_get_section_state(section_priv->section_params);
	driver_sync->appl_pos = user_sync.appl_pos;
	driver_sync->boundary = user_sync.boundary;
	driver_sync->avail_min = user_sync.avail_min;
	driver_sync->stop_threshold = user_sync.stop_threshold;
	user_sync.hw_pos = driver_sync->hw_pos;
	user_sync.state = driver_sync->state;
	spin_unlock_irqrestore(&section_priv->sect_lock, lock_flags);

	dev_dbg(private->dev, "%s:%d--%s--%d ---------- hw %u appl %u",
			__func__,
			section_priv->uid,
			((section_priv->direction == SECTION_INPUT) ?
					"INPUT" : "OUTPUT"),
			user_sync.state,
			user_sync.hw_pos,
			user_sync.appl_pos);

	err = copy_to_user((void __user *)arg, &user_sync,
			sizeof(struct section_sync));

	if (err)
		return -EFAULT;

	return 0;
}

/**
 * ioctl_release_pair() - updates the private manager to release a section in
 * use and if both sections are released, inform the asrc core to release the
 * pair that is currently in use.
 * @section_priv	struct asrc_section_private *
 * @return	0 success, EPERM not allowed to be called currently, EINVAL
 *		invalid argument
 */
static int ioctl_release_pair(struct asrc_section_private *section_priv)
{
	enum asrc_section_state state;
	int err = 0;

	if (!section_priv->section_params) {
		dev_err(section_priv->private->dev,
			"ASRC_RELEASE_PAIR called before ASRC_REQ_PAIR");
		return -EPERM;
	}

	state = asrc_get_section_state(section_priv->section_params);

	/* If we are calling release pair, it should be done while not in the
	 * run, ready, or xrun state. This is because in these other states
	 * polling or other cleanup may be necessary and it is expected the user
	 * has the best judgement to call xrun/stop and properly wind-down.
	 * If the user wants to force abort, calling close on the file
	 * descriptor will have the desired effect.
	 */

	err = unwind_section(section_priv, state);

	return err;
}

/**
 * ioctl_xrun() - synchronizes both sections of a pair to be in error state
 * @section_priv	struct asrc_section_private *
 * @return	0 success, EPERM not allowed to be called currently
 */
static int ioctl_xrun(struct asrc_section_private *section_priv)
{
	int err;

	if (!(section_priv->section_params) ||
			!(section_priv->section_dma.channel)) {
		dev_err(section_priv->private->dev,
			"ASRC_SECTION_XRUN called before ASRC_REQ_PAIR or"
				"ASRC_FREE_BUFFER\n");
		return -EPERM;
	}

	dev_dbg(section_priv->private->dev,
				"%s: ================== %s\n",
				__func__,
				(section_priv->direction == SECTION_INPUT ?
						"INPUT" : "OUTPUT"));

	err = asrc_xrun(section_priv->section_params);

	return 0;
}

/**
 * ioctl_free_buffer() - Frees an allocated section buffer
 * @section_priv struct asrc_section_private *
 * @return	0 SUCCESS, EPERM not allowed to be called currently
 */
static int ioctl_free_buffer(struct asrc_section_private *section_priv)
{

	if (!(section_priv->section_params) ||
			!(section_priv->section_dma.channel)) {
		dev_err(section_priv->private->dev,
			"ASRC_FREE_BUFFER called before ASRC_REQ_PAIR or"
				" ASRC_CONFIG_BUFFER\n");
		return -EPERM;
	}

	/* free dma resource */
	free_dma_buffer(section_priv);

	return 0;
}

/**
 * ioctl_set_clock_reference() - Set a section's ideal reference clock
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - ideal_clock_name
 * @return	0 SUCCESS, EPERM not allowed to be called currently, EINVAL
 *		invalid argument
 */
static int ioctl_set_clock_reference(
		struct asrc_section_private *section_priv,
		unsigned long arg)
{
	char ideal_clock_name[ASRC_MAX_CLKNAME_LEN];
	int err, len;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_SET_CLK_REFERENCE called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	len = strncpy_from_user(ideal_clock_name, (void __user *)arg,
			ASRC_MAX_CLKNAME_LEN);
	if (len < 0)
		return len;

	dev_dbg(section_priv->private->dev,
			"%s setting clock reference %s(len=%d)\n", __func__,
			ideal_clock_name, len);

	err = asrc_set_clock_reference(section_priv->section_params,
			ideal_clock_name);

	return err;
}

/**
 * ioctl_configure_buffer() - Set a section's buffer length, periods,
 * initializes the internal sync structure and allocates buffer memory.
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - struct section_buffer_config
 * @return	0 SUCCESS, EPERM not allowed to be called currently, EINVAL
 *		invalid argument
 */
static int
ioctl_configure_buffer(struct asrc_section_private *section_priv,
		unsigned long arg)
{
	struct section_buffer_config config;
	struct buffer_capabilities capabilities;

	int err = 0;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_CONFIG_BUFFER called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	if (copy_from_user(&config, (void __user *)arg,
			   sizeof(struct section_buffer_config)))
		return -EFAULT;

	asrc_get_buffer_capabilities(section_priv->section_params,
			&capabilities);

	if ((config.buffer_bytes < capabilities.bufbytes_min) ||
		(config.buffer_bytes > capabilities.bufbytes_max)) {
		dev_err(section_priv->private->dev,
			"requested buffer bytes out of range:(%lu>%lu>%lu)\n",
			capabilities.bufbytes_min, config.buffer_bytes,
			capabilities.bufbytes_max);
		return -EINVAL;
	}

	if ((config.periods < capabilities.periods_min) ||
		(config.periods > capabilities.periods_max)) {
		dev_err(section_priv->private->dev,
			"requested buffer periods out of range\n");
		return -EINVAL;
	}

	if (config.buffer_bytes % config.periods) {
		dev_err(section_priv->private->dev,
			"The buffer size is not an integer multiple "
				"of the soft period\n");
		return -EINVAL;
	}

	dev_dbg(section_priv->private->dev,
			"%s allocating:\n"
			"%lu buf bytes\n"
			"%u periods\n",
			__func__, config.buffer_bytes, config.periods);

	err = configure_dma_buffer(section_priv, &config);

	return err;
}

/**
 * ioctl_prepare() - reset buffer position, release and rerequest dma chan,
 * prepare new dma slave
 * @section_priv struct asrc_section_private *
 * @return	0 SUCCESS, EPERM not allowed to be called currently, EBUSY
 *		dma channel busy, EINVAL could not configure dma slave or desc
 */
static int
ioctl_prepare(struct asrc_section_private *section_priv)
{
	unsigned long lock_flags;
	int err;
	struct fsl_asrc_private *private = section_priv->private;
	struct section_dma *section_dma = &section_priv->section_dma;
	unsigned int period_bytes;
	unsigned int frame_bytes;

	if (!(section_priv->section_params) ||
			(section_dma->dma.length == 0) ||
			(section_dma->dma.periods == 0)) {
		dev_err(section_priv->private->dev,
			"ASRC_SECTION_PREPARE called before ASRC_REQ_PAIR or"
			"ASRC_CONFIG_BUFFER\n");
		return -EPERM;
	}

	spin_lock_irqsave(&section_priv->sect_lock, lock_flags);
	period_bytes = section_dma->dma.length / section_dma->dma.periods;
	frame_bytes = asrc_get_frame_bytewidth(section_priv->section_params);
	if (frame_bytes != 0) {
		section_dma->dma.frames_period = period_bytes / frame_bytes;
		section_priv->sync.appl_pos = 0;
		section_priv->sync.hw_pos = 0;
	}
	spin_unlock_irqrestore(&section_priv->sect_lock, lock_flags);

	if (frame_bytes == 0) {
		dev_err(section_priv->private->dev,
			"SECTION_SETUP must have failed, "
				"cannot prepare section\n");
		return -EPERM;
	}

	/* release DMA and request again */
	if (section_dma->channel)
		dma_release_channel(section_dma->channel);
	section_dma->channel = NULL;
	section_dma->channel =
			allocate_dma_channel(
					asrc_get_dma_request(
						section_priv->section_params));


	if (section_dma->channel == NULL) {
		dev_err(private->dev, "can't allocate dma channel on %s\n",
				(section_priv->direction == SECTION_INPUT) ?
						"INPUT" : "OUTPUT");
		return -EBUSY;
	}

	err = configure_dma_slave(section_priv);
	if (err) {
		dev_err(private->dev, "unable to configure dma slave\n");
		return -EINVAL;
	}

	section_dma->desc->callback = asrc_dma_callback;
	section_dma->desc->callback_param = section_priv;

	err = asrc_prepare(section_priv->section_params);

	dev_dbg(section_priv->private->dev,
			"%s prepared channel:0x%p dma irq:%d err:%d\n",
			__func__,
			section_dma->channel,
			asrc_get_dma_request(section_priv->section_params),
			err);

	return err;
}

/**
 * ioctl_init_section() - Initializes a section to it's default values
 * @section_priv struct asrc_section_private *
 * @return	0 SUCCESS, EPERM not allowed to be called currently
 */
static int
ioctl_init_section(struct asrc_section_private *section_priv)
{
	int err;

	if (!(section_priv->section_params)) {
		dev_err(section_priv->private->dev,
			"ASRC_INIT_SECTION called before ASRC_REQ_PAIR\n");
		return -EPERM;
	}

	err = asrc_init_section(section_priv->section_params);

	dev_dbg(section_priv->private->dev, "%s %s section initialized:%p\n",
				__func__,
				(section_priv->direction == SECTION_INPUT) ?
						"INPUT" : "OUTPUT",
				section_priv->section_params);

	return err;
}

/**
 * ioctl_req_pair() - request an asrc pair from asrc core
 * @section_priv struct asrc_section_private *
 * @arg		unsigned long - struct asrc_req
 * @return	0 SUCCESS, EPERM not allowed to be called currently, EFAULT
 *		couldn't copy from user, EINVAL invalid argument, EBUSY section
 *		or pair in use already
 *
 *		Reserves an asrc pair and/or updates the section manager for the
 *		current section on success.
 */
static int
ioctl_req_pair(struct asrc_section_private *section_priv,
				unsigned long arg)
{
	struct asrc_req req;
	int err, key;
	struct fsl_asrc_private *private = section_priv->private;
	unsigned long lock_flags;
	enum asrc_pair_index pair;

	if (section_priv->section_params) {
		dev_err(private->dev, "ASRC_REQ_PAIR previously called\n");
		return -EPERM;
	}

	if (copy_from_user(&req, (void __user *)arg,
			   sizeof(struct asrc_req)))
		return -EFAULT;

	/* make sure we have a valid id */
	if (req.uid <= 0)
		return -EINVAL;

	/* update our private data */
	section_priv->uid = req.uid;

	/* lock struct */
	spin_lock_irqsave(&private->data_lock, lock_flags);

	/* check if uid exists */
	key = find_uid_key(private, section_priv->uid);

	if (key < 0) {
		dev_dbg(private->dev, "%s key not found for uid:%d\n",
				__func__, section_priv->uid);
		key = find_empty(private);
		/* request an asrc pair from the core if not already done so*/
		if (key >= 0) {
			dev_dbg(private->dev,
					"%s private:%p mask:%x idx add:%p\n",
					__func__,
					private, req.pair_mask, &pair);
			err = asrc_req_pair(private->asrc_core_private,
					req.pair_mask, &pair);
			/* update the manager pointer */
			if (!err) {
				dev_dbg(private->dev,
						"%s key:%d,uid:%d,pair:%d\n",
						__func__, key,
						section_priv->uid,
						pair);
				private->manager[key].pair_idx = pair;
				private->manager[key].uid = section_priv->uid;
			}
		} else
			err = -EBUSY;
	} else {
		/* retrieve the pair idx */
		pair = private->manager[key].pair_idx;
		/* check if direction is used */
		if (private->manager[key].usage[section_priv->direction] ==
				ACTIVE)
			err = -EBUSY;
		else
			err = 0;
		dev_dbg(private->dev, "%s key found for uid:%d dir:%s\n",
				__func__, section_priv->uid,
				(section_priv->direction == SECTION_INPUT) ?
						"INPUT" : "OUTPUT");
	}

	/* if we arent in error update the direction and set the section ptr */
	if (err == 0) {

		private->manager[key].usage[section_priv->direction] = ACTIVE;
		asrc_get_section_from_pair(private->asrc_core_private,
				pair,
				&section_priv->section_params,
				section_priv->direction);
		private->manager[key].section[section_priv->direction] =
				section_priv;

		init_waitqueue_head(&section_priv->wait_queue);

		asrc_open_section(section_priv->section_params);

		asrc_set_callback(section_priv->section_params,
					asrc_statechange_callback,
					(unsigned long)section_priv);

		if ((private->manager[key].usage[SECTION_OUTPUT] != ACTIVE) ||
			(private->manager[key].usage[SECTION_INPUT] != ACTIVE))
			asrc_set_xrun_callback(section_priv->section_params,
					asrc_xrun_callback,
					(void *)section_priv->section_params);

		dev_dbg(private->dev, "%s section addr:%p, err:%d\n", __func__,
				section_priv->section_params, err);
	}
	/* unlock */
	spin_unlock_irqrestore(&private->data_lock, lock_flags);

	return err;
}

/**
 * ioctl_start_conversion() - start ASRC conversion
 * @section_priv	struct asrc_section_private *
 * @return	0 success, EPERM or EINVAL invalid attempt to start conversion
 */
static int
ioctl_start_conversion(struct asrc_section_private *section_priv)
{
	int err;

	if (!(section_priv->section_params) ||
			!(section_priv->section_dma.channel) ||
			(section_priv->sync.boundary == 0)) {
		dev_err(section_priv->private->dev,
			"ASRC_START_CONV called before ASRC_REQ_PAIR or"
				" ASRC_CONFIG_BUFFER\n");
		return -EPERM;
	}

	dma_async_issue_pending(section_priv->section_dma.channel);

	err = asrc_start_conversion(section_priv->section_params);

	return err;
}

/**
 * ioctl_stop_conversion() - stops an active ASRC conversion
 * @section_priv	struct asrc_section_private *
 * @return	0 success, EPERM or EINVAL invalid attempt to stop conversion
 */
static int
ioctl_stop_conversion(struct asrc_section_private *section_priv)
{
	int err;

	if (!(section_priv->section_params) ||
			!(section_priv->section_dma.channel)) {
		dev_err(section_priv->private->dev,
			"ASRC_STOP_CONV called before ASRC_REQ_PAIR or"
				" ASRC_CONFIG_BUFFER\n");
		return -EPERM;
	}

	if (section_priv->section_dma.channel) {
		dev_dbg(section_priv->private->dev,
				"%s stopping dma transfer:%p\n",
				__func__, section_priv->section_dma.channel);
		dmaengine_terminate_all(section_priv->section_dma.channel);
	}

	err = asrc_stop_conversion(section_priv->section_params);

	wake_up_interruptible(&section_priv->wait_queue);

	dev_dbg(section_priv->private->dev, "%s stop conversion:%d\n",
			__func__, err);

	return err;
}

/**
 * asrc_ioctl() - ioctl function
 * @inode	struct inode *
 * @file	struct file *
 * @cmd		unsigned int
 * @arg		unsigned long - the pointer to the userspace input data
 * @return	0 success, ENODEV for invalid device instance, EFAULT for
 *		invalid address, EBUSY for device busy, EINVAL for invalid
 *		parameter passed.
 */
static long asrc_ioctl(struct file *file,
		      unsigned int cmd, unsigned long arg)
{
	int err = 0;
	struct asrc_section_private *section_priv = file->private_data;

	switch (cmd) {
	case ASRC_REQ_PAIR:
		err = ioctl_req_pair(section_priv, arg);
		break;
	case ASRC_INIT_SECTION:
		err = ioctl_init_section(section_priv);
		break;
	case ASRC_CONFIG_BUFFER:
		err = ioctl_configure_buffer(section_priv, arg);
		break;
	case ASRC_SET_FILTER:
		err = ioctl_set_filter_settings(section_priv, arg);
		break;
	case ASRC_SET_PERIOD:
		err = ioctl_set_period(section_priv, arg);
		break;
	case ASRC_GET_CAPABILITY:
		err = ioctl_get_section_capabilities(section_priv, arg);
		break;
	case ASRC_GET_BUF_CAPABILITY:
		err = ioctl_get_buffer_capabilities(section_priv, arg);
		break;
	case ASRC_SETUP_SECTION:
		err = ioctl_setup_section(section_priv, arg);
		break;
	case ASRC_START_CONV:
		err = ioctl_start_conversion(section_priv);
		break;
	case ASRC_STOP_CONV:
		err = ioctl_stop_conversion(section_priv);
		break;
	case ASRC_SET_CLK_REFERENCE:
		err = ioctl_set_clock_reference(section_priv, arg);
		break;
	case ASRC_SYNC:
		err = ioctl_sync_section(section_priv, arg);
		break;
	case ASRC_SECTION_PREPARE:
		err = ioctl_prepare(section_priv);
		break;
	case ASRC_FREE_BUFFER:
		err = ioctl_free_buffer(section_priv);
		break;
	case ASRC_RELEASE_PAIR:
		err = ioctl_release_pair(section_priv);
		break;
	case ASRC_SECTION_XRUN:
		err = ioctl_xrun(section_priv);
		break;
	default:
		return -EINVAL;
	}

	return err;
}

/**
 * mxc_asrc_open() - Allocates section private data sets direction
 * @inode	structure inode *
 * @file	structure file *
 * @return	0 success, ENODEV invalid device instance, ENOMEM failed to
 *		allocate buffer, ERESTARTSYS interrupted by user
 *
 *		Important Note: Currently there is a limitation of 1 section per
 *		file descriptor. If you want to open another section, another
 *		file descriptor will need to be opened. The reason for this is
 *		that section related data is stored in the filp's private data
 *		until closed.
 */
static int mxc_asrc_open(struct inode *inode, struct file *file)
{
	struct fsl_asrc_private *asrc_priv  = container_of(inode->i_cdev,
			struct fsl_asrc_private, cdev);
	struct asrc_section_private *section_priv;

	enum asrc_section_direction direction;

	dev_dbg(asrc_priv->dev, "%s section open: core add: %p\n", __func__,
			asrc_priv->asrc_core_private);

	if (signal_pending(current))
		return -EINTR;

	if (file->f_mode & FMODE_WRITE) {
		dev_dbg(asrc_priv->dev, "Input opened\n");
		direction = SECTION_INPUT;
	} else if (file->f_mode & FMODE_READ) {
		dev_dbg(asrc_priv->dev, "Output opened\n");
		direction = SECTION_OUTPUT;
	} else {
		dev_err(asrc_priv->dev, "device can only be opened in "\
						"read or write only modes.\n");
		return -EINVAL;
	}

	section_priv = kzalloc(sizeof(struct asrc_section_private), GFP_KERNEL);
	if (section_priv == NULL) {
		dev_dbg(asrc_priv->dev, "Failed to allocate params\n");
		return -ENOMEM;
	}

	spin_lock_init(&section_priv->sect_lock);
	section_priv->private = asrc_priv;
	file->private_data = section_priv;
	section_priv->direction = direction;

	dev_dbg(asrc_priv->dev, "%s: section initialized:%p\n",
			__func__, section_priv);

	return 0;
}

/**
 * mxc_asrc_close() - asrc close function
 * @inode		struct inode *
 * @file		structure file *
 * @return status	0 Success, EINTR busy lock error,
 *			 ENOBUFS remap_page error
 */
static int mxc_asrc_close(struct inode *inode, struct file *file)
{
	struct asrc_section_private *section_priv = file->private_data;
	struct fsl_asrc_private *private = section_priv->private;
	struct section_dma *section_dma = &section_priv->section_dma;
	enum asrc_section_state state;

	if (section_priv->section_params) {

		state = asrc_get_section_state(section_priv->section_params);

		switch (state) {
		case SECTION_SOFT_XRUN:
		case SECTION_HARD_XRUN:
		case SECTION_IRQ_ERR:
			/* do something with the xrun if necessary */
		case SECTION_RUN:
			/* stop the asrc */
			asrc_stop_conversion(section_priv->section_params);
		case SECTION_READY:
			/* terminate dma transfer */
			dmaengine_terminate_all(section_dma->channel);
		case SECTION_PREPARE:
			state = SECTION_PREPARE;
		case SECTION_SETUP:
		case SECTION_OPENED:
			unwind_section(section_priv, state);
		case SECTION_CLOSED:
			section_priv->section_params = NULL;
			break;
		default:
			return -EINVAL;
		}
	}

	dev_dbg(private->dev, "%s pair closed: core add:%p\n", __func__,
			section_priv->private->asrc_core_private);

	section_priv->private = NULL;

	kfree(section_priv);
	file->private_data = NULL;

	return 0;
}

/**
 * mxc_asrc_mmap() - asrc mmap function
 * @file		structure file *
 * @vma			structure vm_area_struct *
 * @return status	0 Success, EINTR busy lock error,
 *			ENOBUFS remap_page error
 */
static int mxc_asrc_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct asrc_section_private *section_priv = file->private_data;
	unsigned long size, boundary_size;
	int res = 0;

	if (!section_priv->section_dma.dma.dma_paddr) {
		dev_err(section_priv->private->dev,
			"Can't map memory before allocating buffer\n");
		res = -EPERM;
		goto map_fail;
	}

	size = vma->vm_end - vma->vm_start;
	boundary_size = (((section_priv->section_dma.dma.length - 1) /
			PAGE_SIZE) + 1) * PAGE_SIZE;

	if (size != boundary_size) {
		dev_err(section_priv->private->dev,
			"mmap memory size differs from requested "\
			"buffer size %lu != %u\n",
			size, section_priv->section_dma.dma.length);
		res = -EINVAL;
		goto map_fail;
	}

	vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);
	res = remap_pfn_range(vma, vma->vm_start,
			(section_priv->section_dma.dma.dma_paddr >> PAGE_SHIFT),
			size, vma->vm_page_prot);

	if (res < 0) {
		res = -ENOBUFS;
		goto map_fail;
	}

	vma->vm_flags &= ~VM_IO;

map_fail:

	return res;
}


/**
 * mxc_asrc_poll() - asrc polling function
 * @file		structure file *
 * @wait		structure poll_table *
 * @return status	POLLIN/POLLRDNORM, POLLOUT/POLLWRNORM
 *
 * This function is meant to be used to determine if a the ASRC pair described
 * by the file descriptor is ready to be read from/written to.
 *
 * EOF support is not included because there is no clean way of determining
 * when the end of a sample stream occurs coming out of the ASRC.
 */
static unsigned int
mxc_asrc_poll(struct file *file, struct poll_table_struct *wait)
{
	struct asrc_section_private *section_priv = file->private_data;
	struct section_sync *sync = &section_priv->sync;
	struct section_dma *section_dma = &section_priv->section_dma;
	enum asrc_section_state state;
	unsigned int mask = 0;
	unsigned long lock_flags;
	unsigned int avail = 0;

	if (!(section_priv->section_params) ||
			!(section_priv->section_dma.channel)) {
		dev_err(section_priv->private->dev,
			"ASRC_START_CONV called before ASRC_REQ_PAIR or"
				" ASRC_CONFIG_BUFFER\n");
		return -EPERM;
	}

	poll_wait(file, &section_priv->wait_queue, wait);

	spin_lock_irqsave(&section_priv->sect_lock, lock_flags);

	avail = frames_avail(sync, &section_dma->dma, section_priv->direction);

	if (avail >= sync->avail_min) {
		if (section_priv->direction == SECTION_INPUT)
			mask = POLLOUT | POLLWRNORM;
		else
			mask = POLLIN | POLLRDNORM;
	}

	state = asrc_get_section_state(section_priv->section_params);
	if ((state != SECTION_READY) && (state != SECTION_RUN) &&
					(state != SECTION_PREPARE)) {
		dev_dbg(section_priv->private->dev,
					"POLLERR: %s stop_threshold = %d\n",
					(section_priv->direction ==
						SECTION_INPUT ?
							"INPUT" : "OUTPUT"),
							sync->stop_threshold);
		mask |= POLLERR;
	}

	spin_unlock_irqrestore(&section_priv->sect_lock, lock_flags);

	return mask;
}

static struct device_attribute asrc_attrs[] = {
	__ATTR_NULL,
};

static const struct file_operations asrc_fops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl	= asrc_ioctl,
	.poll = mxc_asrc_poll,
	.mmap = mxc_asrc_mmap,
	.open = mxc_asrc_open,
	.release = mxc_asrc_close,
};

static int mxc_asrc_probe(struct platform_device *pdev)
{
	int i, err = 0;
	int ret;
	struct device_node *np = pdev->dev.of_node, *asrc_core_np;
	struct platform_device *asrc_core_pdev;
	struct fsl_asrc_private *asrc_private;

	dev_info(&pdev->dev, "mxc asrc device driver probed\n");

	/* If the ASRC is not to be used
	 *      status = "disabled"
	 * property in the device tree node.
	 */
	if (!of_device_is_available(np))
		return -ENODEV;

	/* allocate our device data */
	asrc_private = kzalloc(sizeof(struct fsl_asrc_private),
			GFP_KERNEL);

	if (asrc_private == NULL) {
		dev_err(&pdev->dev,
				"Failed to allocate private data\n");
		return -ENOMEM;
	}

	/* initialize data */
	for (i = 0; i < ASRC_PAIR_NUM; i++) {
		clean_manager_pair(asrc_private, i, SECTION_INPUT);
		clean_manager_pair(asrc_private, i, SECTION_OUTPUT);
	}

	platform_set_drvdata(pdev, asrc_private);

	asrc_core_np = of_parse_phandle(pdev->dev.of_node,
			"asrc-core", 0);
	if (!asrc_core_np) {
		dev_err(&pdev->dev, "phandle missing or invalid\n");
		ret = -EINVAL;
		goto error_kmalloc;
	}

	/* retrieve asrc_core pdev handle */
	asrc_core_pdev = of_find_device_by_node(asrc_core_np);
	if (!asrc_core_pdev) {
		dev_err(&pdev->dev,
				"failed to find ASRC core platform device\n");
		ret = -EINVAL;
		goto error_kmalloc;
	}

	asrc_private->asrc_core_private = platform_get_drvdata(asrc_core_pdev);
	if (!asrc_private->asrc_core_private) {
		dev_err(&pdev->dev,
				"failed to find valid ASRC core. "
				"Did it fail to load?\n");
		ret = -EINVAL;
		goto error_kmalloc;
	}

	dev_info(&pdev->dev, "mxc asrc core found: %p.\n",
			asrc_private->asrc_core_private);

	/* initialize spinlocks */
	spin_lock_init(&asrc_private->data_lock);

	cdev_init(&asrc_private->cdev, &asrc_fops);
	asrc_private->cdev.owner = THIS_MODULE;

	ret = cdev_add(&asrc_private->cdev, MKDEV(MAJOR(asrc_devt), 0), 1);
	if (ret < 0)
		goto error_kmalloc;

	asrc_private->dev =
		device_create(asrc_class, &pdev->dev,
				asrc_private->cdev.dev,
				asrc_private, ASRC_DEVICE_NAME);
	if (IS_ERR(asrc_private->dev)) {
		dev_err(&pdev->dev, "Unable to create asrc char device\n");
		err = PTR_ERR(asrc_private->dev);
		goto err_cdev;
	}

	asrc_private->dev->coherent_dma_mask = DMA_BIT_MASK(32);

	dev_info(&pdev->dev, "initialized\n");

	return 0;

err_cdev:
	cdev_del(&asrc_private->cdev);
error_kmalloc:
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(asrc_private);
	return ret;
}

/**
 * mxc_asrc_remove() - Exit asrc device
 * @pdev	Pointer to the registered platform device
 * @return	Error code indicating success or failure
 */
static int mxc_asrc_remove(struct platform_device *pdev)
{
	struct fsl_asrc_private *asrc_private = platform_get_drvdata(pdev);

	device_destroy(asrc_class, asrc_private->cdev.dev);
	cdev_del(&asrc_private->cdev);

	dev_set_drvdata(&pdev->dev, NULL);

	kfree(asrc_private);
	asrc_private = NULL;

	return 0;
}

static const struct of_device_id asrc_ids[] = {
	{ .compatible = "fsl,imx6-asrc", },
	{}
};
MODULE_DEVICE_TABLE(of, asrc_ids);

static struct platform_driver mxc_asrc_driver = {
	.driver = {
		   .name = ASRC_DEVICE_NAME,
		   .owner = THIS_MODULE,
		   .of_match_table = asrc_ids,
		   },
	.probe = mxc_asrc_probe,
	.remove = mxc_asrc_remove,
};

static __init int asrc_init(void)
{
	int ret;

	/* register our character device */
	ret = alloc_chrdev_region(&asrc_devt, 0, 1, ASRC_DEVICE_NAME);
	if (ret)
		return ret;

	asrc_class = class_create(THIS_MODULE, ASRC_DEVICE_NAME);
	if (IS_ERR(asrc_class)) {
		ret = PTR_ERR(asrc_class);
		goto err_chrdev;
	}

	asrc_class->dev_attrs = asrc_attrs;

	ret = platform_driver_register(&mxc_asrc_driver);
	if (ret == 0)
		return ret;

	class_destroy(asrc_class);
err_chrdev:
	unregister_chrdev_region(asrc_devt, 1);

	return ret;
}

static void __exit asrc_exit(void)
{
	platform_driver_unregister(&mxc_asrc_driver);

	class_destroy(asrc_class);

	unregister_chrdev_region(asrc_devt, 1);

	return;
}

module_init(asrc_init);
module_exit(asrc_exit);
MODULE_AUTHOR("Mentor Graphics, Inc.");
MODULE_DESCRIPTION("Asynchronous Sample Rate Converter");
MODULE_LICENSE("GPL");
