/*
 * MASCA Encode/Decode Command/Message File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License ,  or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_blkdev.h"
#include "masca_sm.h"



struct message_decoder_type {
	unsigned char msg_id;
	enum masca_error (*decode_function)(
		struct masca_intrptr_msg_info_list * const p_msg_list ,
		union masca_intrptr_decode_info * const p_decode_info,
		const struct masca_intrptr_msg_info * const p_msg ,
		const enum masca_intrptr_decode_msg decoded_msg);
};


#define MAX_RECEIVE_MESSAGES		33
/*!!!!!!!!!!!!!!!!IMPORTANT!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
Dont miss to configure this parameter when you add new messages
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
#define DIAG_MSG_START_INDEX		7
#define DIAG_MSG_INIT_LASER_INDEX	25
#define DIAG_MSG_LAST_ERRS_INDEX	30
#define DIAG_MSG_USED_CD_CNT_INDEX	31

/*min measured temperature value in degree centigrade*/
#define MIN_TEMPERATURE_VALUE		((short)-40)
#define MAX_TEMPERATURE_VALUES		166

#define MSG_MODE1_DATA_ID		0x31
#define MSG_TRACK_DATA_ID		0x39
#define MSG_TRACK_TEXT_ID		0x41
#define MSG_TRACK_INFO_ID		0x45
#define MSG_TOC_DATA_ID			0x47
#define MSG_VERSION_ID			0x51
#define MSG_DIAG_LASER_CURRENT_ID	0xF1
#define MSG_DIAG_ADJUSTMENT_ID		0xF3
#define MSG_DIAG_CMD_RECEVIED_ID	0x91
#define MSG_DIAG_SWITCH_STATUS_ID	0x81
#define MSG_DIAG_C1C2_ERROR_CNT_ID	0xFB
#define MSG_DIAG_TEMPERATURE_ID		0xFD
#define MSG_DIAG_TIME_READOUT_ID	0x13
#define MSG_DIAG_EJECT_LOAD_ID		0x15
#define MSG_DIAG_JUMP_COUNTER_ID	0x17
#define MSG_DIAG_EEPROM_READOUT_ID	0x19
#define MSG_DIAG_WRITE_STICKER_ID	0x1B
#define MSG_DIAG_READ_STICKER_ID	0x1D
#define MSG_DIAG_JITTER_MEASURE_ID	0x1F
#define MSG_DIAG_VERTICAL_MEASURE_ID	0x21
#define MSG_DIAG_ECCENTRIC_MEASURE_ID	0x23
#define MSG_DIAG_REFLECTIVE_MEASURE_ID	0x25
#define MSG_DIAG_INTERNAL_SELF_TEST_ID	0x27
#define MSG_DIAG_EXTERNAL_SDRAM_TEST_ID	0x29
#define MSG_DIAG_INIT_LASER_CURRENT_ID	0x31
#define MSG_DIAG_NUM_READ_ERRS_ID	0x33
#define MSG_DIAG_DATE_PRODUCT_INSTALL_ID	0x35
#define MSG_DIAG_SERIAL_NO_READ_ID	0x41
#define MSG_MSG_ERR_HISTORY_ID		0x43
#define MSG_MSG_LAST_ERRS_ID		0x45
#define MSG_USED_CD_TYPE_CNT_ID		0x47
#define MSG_PROFIL_USED_CDS_ID		0x49

#define MODE1_DATA_MSGS			2
#define TRACK_DATA_MSGS			2
#define TRACK_TEXT_MSGS			1
#define TRACK_INFO_MSGS			2
#define TOC_DATA_MSGS			1
#define VERSION_INFO_MSGS		2
#define MSG_LENGTH_VERSION_INFO		4
#define MSG_LENGTH_TOC_DATA		8
#define MSG_LENGTH_TRACK_INFO		7
#define MSG_LENGTH_TRACK_DATA		5
#define MSG_LENGTH_MODE1_DATA		8
#define MIN_MSG_LNG_STATUS		8
#define MAX_MSG_LNG_STATUS		17
#define MSG_LEN_TEMPERATURE		3
#define MSG_LEN_SWITCH_STATUS		2
#define MSG_LEN_LASER			3
#define MSG_LEN_ADJUST			10
#define MSG_LEN_CMD_RECEIVED		2
#define MSG_LEN_C1C2_ERR_CNT		5
#define MSG_LEN_TIME_READOUT_CNT	102
/* as per diag design spec value
is 162+1 - time readout byte
size*/
#define MSG_LEN_EJT_LD_CYCLES		4
#define MSG_LEN_JUMP_CNT		6
#define MSG_LEN_EEPROM_READOUT		130
/* as per diag design spec
value is 276+1 EEPROM readout
byte size*/
#define MSG_LEN_WRITE_STICKER		3
#define MSG_LEN_READ_STICKER		19
#define MSG_LEN_JITTER_MEASURE		2
#define MSG_LEN_VERTICAL_MEASURE	3
#define MSG_LEN_ECCENTRIC_MEASURE	3
#define MSG_LEN_REFLECTIVE_MEASURE	2
#define MSG_LEN_INTERNAL_SELF_TEST	2
#define MSG_LEN_EXTERNAL_SDRAM_TEST	2
#define MSG_LEN_INIT_LASER_CURRENT	4
#define MSG_LEN_NUM_READ_ERRS		4
#define MSG_LEN_DATE_PRODUCT_INSTALL	5
#define MSG_LEN_SERIAL_NO		5
#define MSG_LEN_MSG_ERR_HISTORY		76
#define MSG_LEN_MSG_LAST_ERRS		12
#define MSG_LEN_USED_CD_TYPE_CNT	14


#define AUDIO_MODE_MASK			0x07
#define BIT_RATE_MASK			0xF8
#define SAMPLING_RATE_MASK		0x0F

#define MIN_TRACK_TEXT_LENGTH		5
#define MAX_TRACK_TEXT_LENGTH		254

#define COMMAND_MAX_BIT_SETS		4
#define COMMAND_POSITION		0
#define COMMAND_HDR_LENGTH		1

#define COMMAND_STATUS_LENGTH		6
#define MULTI_ON_BYTE_POS		1
#define MULTI_ON_BIT_POS		3
#define IIS_SW_OFF_ENABLE_BYTE_POS	1
#define COMMAND_STATUS_TRACK_POS	3
#define IIS_SW_OFF_ENABLE_BIT_POS	4

#define COMMAND_PLAY_LNG_RESUME		1
#define COMMAND_PLAY_LENGTH_LOW		2
#define COMMAND_PLAY_LENGTH_HIGH	8
#define COMMAND_PLAY_DIR		3
#define COMMAND_PLAY_TRK		4
#define COMMAND_PLAY_MIN		5
#define COMMAND_PLAY_SEC		6
#define COMMAND_PLAY_FRM		7
#define COMMAND_PLAY_BLK_CNT		8

#define COMMAND_TRACK_INFO_LENGTH	3
#define COMMAND_TRACK_INFO_ID		0x14
#define COMMAND_TRACK_POS		3
#define COMMAND_DIR_POS			2

#define COMMAND_DIAG_ID			0x16
#define DIAG_BYTE_CNT_POS		0x02
#define DIAG_CMD_INDEX			0x01
#define DIAG_SUB_CMD_POS		0x01
#define DIAG_SUB_CMD_PARA_POS		0x02


#define MASK_BYTE_NUM			0x0F
#define MASK_BIT_POS			0xF0
#define SHIFT_CNT_HIGHER_NIBBLE		0x04

/* Laser current calculation - as per masca diagnostic specification
 * Laser current(mA) = (Resolution_factor * Current consumption * U33s/65535)/R
 * U33s = 3.3V , Resistor(R) = 10 ohm , Resolution_factor = 10000 and
 * Current consumption(x) value from CD drive. */
#define GET_CURRENT_CONSUM(x) (10000 * (x) * 33/65535)

static enum masca_error
mode1_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
track_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
track_text_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
track_info_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
toc_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
version_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error
status_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static void decode_rejected_command(unsigned char * const p_buf ,
	union masca_intrptr_decode_info * const p_decode_info);

static enum masca_intrptr_decode_msg
decode_bits_after_dsp_init(unsigned char const * const p_buf,
		enum masca_intrptr_decode_msg **pp_msg_container);

static void decode_bits_after_mech_init(unsigned char const * const p_buf,
		enum masca_intrptr_decode_msg **pp_msg_container);

static void get_bit_byte_set_values(unsigned char * const p_buf ,
					const unsigned char byte ,
					unsigned char * const p_param_count ,
					const unsigned char byte_num);

static enum masca_error diag_laser_cur_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_adjument_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_cmd_recevied_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_switch_status_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_c1c2_err_cnt_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_temp_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_time_readout_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_eject_load_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_jump_counter_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_eeprom_readout_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_write_sticker_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_read_sticker_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_jitter_measure_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_vertical_measure_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_eccentric_measure_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_reflective_measure_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_internal_self_test_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_external_sdram_test_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_init_laser_current_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_number_read_errs_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_date_product_install_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_serial_number_read_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_msg_err_history_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_msg_last_errs_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_used_cd_type_cnt_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static enum masca_error diag_user_profil_used_cds_decode_func(
	struct masca_intrptr_msg_info_list * const p_msg_list ,
	union masca_intrptr_decode_info * const p_decode_info ,
	const struct masca_intrptr_msg_info * const p_msg ,
	const enum masca_intrptr_decode_msg decoded_msg);

static short convert_to_centigrade(const unsigned short adc_count);

/* Table containing all the received messages and
corresponding decode functions
*/
static const struct message_decoder_type message_decoder
	[MAX_RECEIVE_MESSAGES] = {
	{MSG_STATUS_ID     ,  status_decode_func    } ,
	{MSG_MODE1_DATA_ID ,  mode1_data_decode_func} ,
	{MSG_TRACK_DATA_ID ,  track_data_decode_func} ,
	{MSG_TRACK_TEXT_ID ,  track_text_decode_func} ,
	{MSG_TRACK_INFO_ID ,  track_info_decode_func} ,
	{MSG_TOC_DATA_ID   ,  toc_data_decode_func  } ,
	{MSG_VERSION_ID    ,  version_decode_func   } ,
	{MSG_DIAG_LASER_CURRENT_ID ,  diag_laser_cur_decode_func } ,
	{MSG_DIAG_ADJUSTMENT_ID ,  diag_adjument_decode_func } ,
	{MSG_DIAG_CMD_RECEVIED_ID ,  diag_cmd_recevied_decode_func } ,
	{MSG_DIAG_SWITCH_STATUS_ID ,  diag_switch_status_decode_func } , /*10*/
	{MSG_DIAG_C1C2_ERROR_CNT_ID ,  diag_c1c2_err_cnt_decode_func } ,
	{MSG_DIAG_TEMPERATURE_ID ,  diag_temp_decode_func} ,
	{MSG_DIAG_TIME_READOUT_ID ,  diag_time_readout_decode_func } ,
	{MSG_DIAG_EJECT_LOAD_ID ,  diag_eject_load_decode_func } ,
	{MSG_DIAG_JUMP_COUNTER_ID ,  diag_jump_counter_decode_func } ,
	{MSG_DIAG_EEPROM_READOUT_ID ,  diag_eeprom_readout_decode_func } ,
	{MSG_DIAG_WRITE_STICKER_ID ,  diag_write_sticker_decode_func } ,
	{MSG_DIAG_READ_STICKER_ID ,  diag_read_sticker_decode_func } ,
	{MSG_DIAG_JITTER_MEASURE_ID ,  diag_jitter_measure_decode_func } ,/*20*/
	{MSG_DIAG_VERTICAL_MEASURE_ID ,  diag_vertical_measure_decode_func } ,
	{MSG_DIAG_ECCENTRIC_MEASURE_ID ,  diag_eccentric_measure_decode_func } ,
	{MSG_DIAG_REFLECTIVE_MEASURE_ID ,
					diag_reflective_measure_decode_func } ,
	{MSG_DIAG_INTERNAL_SELF_TEST_ID ,
					diag_internal_self_test_decode_func } ,
	{MSG_DIAG_EXTERNAL_SDRAM_TEST_ID ,
					diag_external_sdram_test_decode_func } ,
	{MSG_DIAG_INIT_LASER_CURRENT_ID ,
					 diag_init_laser_current_decode_func } ,
	{MSG_DIAG_NUM_READ_ERRS_ID ,  diag_number_read_errs_decode_func } ,
	{MSG_DIAG_DATE_PRODUCT_INSTALL_ID ,
				 diag_date_product_install_decode_func } ,
	{MSG_DIAG_SERIAL_NO_READ_ID ,  diag_serial_number_read_decode_func } ,
	{MSG_MSG_ERR_HISTORY_ID ,  diag_msg_err_history_decode_func } ,
	{MSG_MSG_LAST_ERRS_ID ,  diag_msg_last_errs_decode_func } , /*30*/
	{MSG_USED_CD_TYPE_CNT_ID ,  diag_used_cd_type_cnt_decode_func } ,
	{MSG_PROFIL_USED_CDS_ID ,  diag_user_profil_used_cds_decode_func }
};

static bool is_iis_switch_off;
bool is_multisession_on;
bool enable_temp_limit;

/*this array holds the information to encode a command
It just specifies the bit set required.
Also currently not all commands are supported as it is not
required.
*/
static const unsigned char masca_commands[CMD_MAX][COMMAND_MAX_BIT_SETS] = {
	/*each number = "Bit number to be set in the byte"
	 *e.g 0x01 = set bit 0 in byte1 (parameter 1 according
	 *to masca specifications)
	 *interpretation for diag commands is
	 *particular diag cmd ,  length of the cmd*/
	{COMMAND_STATUS_ID ,  0x01  ,  0xFF  ,  0xFF} ,
	{COMMAND_STATUS_ID ,  0x01 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x11 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x21 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x31 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0xFF , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0xFF , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x41 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x51 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x51 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x61 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x71 , 0xFF , 0xFF} ,
	{COMMAND_STATUS_ID ,  0x32 , 0xFF , 0xFF},/*CMD_EN_IGN_AUDIO*/
	{COMMAND_STATUS_ID ,  0x71 , 0xFF , 0xFF},/*CMD_DIS_IGN_AUDIO*/
	{COMMAND_TRACK_INFO_ID ,  0x31 , 0xFF , 0xFF} ,
	{COMMAND_TRACK_INFO_ID ,  0x41 , 0xFF , 0xFF} ,
	{COMMAND_TRACK_INFO_ID ,  0x51 , 0xFF , 0xFF} ,
	{COMMAND_TRACK_INFO_ID ,  0x61 , 0xFF , 0xFF} ,
	{COMMAND_TRACK_INFO_ID ,  0x61 , 0xFF , 0xFF} ,
	{COMMAND_PLAY_ID ,  0x01 , 0x21 , 0xFF} , /* CMD_PAUSE_MUTE */
	{COMMAND_PLAY_ID ,  0x21 , 0xFF , 0xFF} , /* CMD_PAUSE */
	{COMMAND_PLAY_ID ,  0x11 , 0xFF , 0xFF} , /* CMD_STOP */
	{COMMAND_PLAY_ID ,  0x31 , 0xFF , 0xFF} , /* CMD_SEARCH_FWD */
	{COMMAND_PLAY_ID ,  0x31 , 0x51 , 0xFF} , /* CMD_SEARCH_BWD */
	{COMMAND_PLAY_ID ,  0x31 , 0x41 , 0xFF} , /* CMD_FAST_SEARCH_FWD */
	{COMMAND_PLAY_ID ,  0x31 , 0x51 , 0x41} , /* CMD_FAST_SEARCH_BWD */
	{COMMAND_PLAY_ID ,  0x61 , 0xFF , 0xFF} , /* CMD_INSERT */
	{COMMAND_PLAY_ID ,  0x71 , 0xFF , 0xFF} , /* CMD_EJECT */
	{COMMAND_PLAY_ID ,  0xFF , 0xFF , 0xFF} , /* CMD_PLAY */
	{COMMAND_PLAY_ID ,  0x02 , 0xFF , 0xFF} , /* CMD_PLAY_CONTINUOUS */
	/* CMD_PLAY_CONTINUOUS_HALFMAX */
	{COMMAND_PLAY_ID ,  0x02 , 0x32 , 0xFF} ,
	{COMMAND_PLAY_ID ,  0x02 , 0x12 , 0xFF} , /* CMD_PLAY_CONTINUOUS_MAX */
	{COMMAND_PLAY_ID ,  0x02 , 0xFF , 0xFF} , /* CMD_CONTINOUS */
	{COMMAND_PLAY_ID ,  0xFF , 0xFF , 0xFF} , /* CMD_PLAY_NON_CONTINUOUS */
	{COMMAND_PLAY_ID ,  0x12 , 0xFF , 0xFF} , /* CMD_DRIVE_HIGH_SPEED */
	{COMMAND_PLAY_ID ,  0x32 , 0xFF , 0xFF} , /* CMD_DRIVE_HALF_MAXSPEED */
	{COMMAND_PLAY_ID ,  0xFF , 0xFF , 0xFF} , /* CMD_RESUME */
	{COMMAND_DIAG_ID ,  0xFC , 0x01 , 0xFF} , /* CMD_GET_TEMPERATURE */
	{COMMAND_DIAG_ID ,  0xF0 , 0x01 , 0xFF} , /* CMD_GET_LASER_CURRENT */
	{COMMAND_DIAG_ID ,  0x07 , 0x01 , 0xFF} , /* CMD_GET_SWITCH_STATUS */
	{COMMAND_DIAG_ID ,  0xF2 , 0x01 , 0xFF} , /* CMD_ADJUSTMENT */
	{COMMAND_DIAG_ID ,  0xFA , 0x01 , 0xFF} , /* CMD_C1C2_ERR_COUNT_ON */
	{COMMAND_DIAG_ID ,  0x0A , 0x01 , 0xFF} , /* CMD_CD_TEXT_OFF */
	{COMMAND_DIAG_ID ,  0x0C , 0x01 , 0xFF} , /* CMD_CD_TEXT_MODE2 */
	{COMMAND_DIAG_ID ,  0x12 , 0x01 , 0xFF} , /* CMD_TIME_READOUT _40_*/
	{COMMAND_DIAG_ID ,  0x14 , 0x01 , 0xFF} , /* CMD_EJECT_LOAD_CYCLES */
	{COMMAND_DIAG_ID ,  0x16 , 0x01 , 0xFF} , /* CMD_JUMP_COUNTER */
	{COMMAND_DIAG_ID ,  0x18 , 0x01 , 0xFF} , /* CMD_EEPROM_READOUT */
	{COMMAND_DIAG_ID ,  0x1A , 0x08 , 0xFF} , /* CMD_WRITE_STICKER */
	{COMMAND_DIAG_ID ,  0x1C , 0x01 , 0xFF} , /* CMD_READ_STICKER */
	{COMMAND_DIAG_ID ,  0x1E , 0x01 , 0xFF} , /* CMD_JITTER_MEASURE */
	{COMMAND_DIAG_ID ,  0x20 , 0x01 , 0xFF} , /* CMD_VERTICAL_DEV_MEASURE*/
	{COMMAND_DIAG_ID ,  0x22 , 0x01 , 0xFF} , /* CMD_ECCENTRIC_MEASURE */
	{COMMAND_DIAG_ID ,  0x24 , 0x01 , 0xFF} , /* CMD_REFLECTIVE_MEASURE */
	/* CMD_INTERNAL_SELF_TEST _50_*/
	{COMMAND_DIAG_ID ,  0x26 , 0x01 , 0xFF} ,
	{COMMAND_DIAG_ID ,  0x28 , 0x01 , 0xFF} , /* CMD_EXTERNAL_SDRAM_TEST */
	{COMMAND_DIAG_ID ,  0x30 , 0x01 , 0xFF} , /* CMD_INIT_LASER_CURRENT */
	{COMMAND_DIAG_ID ,  0x32 , 0x01 , 0xFF} , /* CMD_NUM_READ_ERRS */
	{COMMAND_DIAG_ID ,  0x34 , 0x01 , 0xFF} , /* CMD_DATE_PRODUCTION_READ*/
	/* CMD_DATE_INSTALL_WRITE 05->date */
	{COMMAND_DIAG_ID ,  0x36 , 0x04 , 0xFF} ,
	{COMMAND_DIAG_ID ,  0x38 ,  0x01 , 0xFF} , /* CMD_DATE_INSTALL_READ */
	{COMMAND_DIAG_ID ,  0x40 ,  0x01 , 0xFF} , /* CMD_SERIAL_NO_READ */
	{COMMAND_DIAG_ID ,  0x42 , 0x01 , 0xFF} , /*CMD_STATUS_MSG_ERR_HISTROY*/
	{COMMAND_DIAG_ID ,  0x44 , 0x01 ,  0xFF} , /*CMD_STATUS_MSG_LAST_ERRS*/
	{COMMAND_DIAG_ID ,  0x46 , 0x01 ,  0xFF} , /* CMD_USED_CD_TYPE_COUNT */
	{COMMAND_DIAG_ID ,  0x48 ,  0x01 ,  0xFF} /*CMD_USER_PROFILE_USED_CDS*/
};

/*This table is taken from masca diagnostic specifications.
Observe the way the table is written considering only max value
column. As this is enough to find out the temperature value
corresponding to an ADC value
*/
static const unsigned short temperature_table[MAX_TEMPERATURE_VALUES] = {
	0x001B , 0x001C , 0x001E , 0x001F , 0x0021 , 0x0023 , 0x0025 , 0x0027 ,
	0x0029 , 0x002B , 0x002D , 0x002F , 0x0031 , 0x0034 , 0x0036 , 0x0039 ,
	0x003C , 0x003E , 0x0041 , 0x0044 , 0x0047 , 0x004A , 0x004E , 0x0051 ,
	0x0054 , 0x0058 , 0x005C , 0x0060 , 0x0063 , 0x0068 , 0x006C , 0x0070 ,
	0x0074 , 0x0079 , 0x007D , 0x0082 , 0x0087 , 0x008C , 0x0091 , 0x0096 ,
	0x009B , 0x00A1 , 0x00A6 , 0x00AC , 0x00B2 , 0x00B8 , 0x00BE , 0x00C4 ,
	0x00CB , 0x00D1 , 0x00D8 , 0x00DF , 0x00E6 , 0x00ED , 0x00F4 , 0x00FB ,
	0x0103 , 0x010A , 0x0112 , 0x011A , 0x0121 , 0x0129 , 0x0131 , 0x013A ,
	0x0142 , 0x014A , 0x0152 , 0x015B , 0x0163 , 0x016C , 0x0174 , 0x017D ,
	0x0186 , 0x018E , 0x0197 , 0x01A0 , 0x01A9 , 0x01B1 , 0x01BA , 0x01C3 ,
	0x01CC , 0x01D4 , 0x01DD , 0x01E6 , 0x01EE , 0x01F7 , 0x01FF , 0x0208 ,
	0x0210 , 0x0219 , 0x0221 , 0x0229 , 0x0232 , 0x023A , 0x0242 , 0x024A ,
	0x0252 , 0x025A , 0x0261 , 0x0269 , 0x0271 , 0x0278 , 0x0280 , 0x0287 ,
	0x028E , 0x0295 , 0x029C , 0x02A3 , 0x02AA , 0x02B1 , 0x02B7 , 0x02BE ,
	0x02C4 , 0x02CB , 0x02D1 , 0x02D7 , 0x02DD , 0x02E3 , 0x02E9 , 0x02EE ,
	0x02F4 , 0x02F9 , 0x02FF , 0x0304 , 0x0309 , 0x030E , 0x0313 , 0x0318 ,
	0x031D , 0x0322 , 0x0326 , 0x032B , 0x032F , 0x0333 , 0x0338 , 0x033C ,
	0x0340 , 0x0344 , 0x0348 , 0x034C , 0x034F , 0x0353 , 0x0356 , 0x035A ,
	0x035D , 0x0361 , 0x0364 , 0x0367 , 0x036A , 0x036D , 0x0370 , 0x0373 ,
	0x0376 , 0x0379 , 0x037C , 0x037E , 0x0381 , 0x0384 , 0x0386 , 0x0389 ,
	0x038B , 0x038D , 0x038F , 0x0392 , 0x0394 , 0x0395
};


/**
* \func masca_intrptr_init
*
* Initializes the interpreter.
* Only IIS switch OFF flag with respect to cmd_status needs
* to be preserved as sending of other commands in cmd_status
* resets this flag.
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*/
void masca_intrptr_init(const struct masca_configuration * const p_config)
{
	is_iis_switch_off = false;
}

/**
* \func masca_intrptr_deinit
*
* Only IIS switch OFF flag with respect to cmd_status needs
* to be preserved as sending of other commands in cmd_status
* resets this flag.
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*/
void masca_intrptr_deinit(void)
{
	is_iis_switch_off = false;
}


/*
* \func masca_intrptr_decode_msg_info
*
* Decodes the detail message information depending on the message being
* passed. Before calling this API one should get the list of
* messages using API MASCA_intrptr_decode_msg_list. This API calls
* the respective function responsible for decoding the particular message.
*
* \param decoded_msg  : Message obtained by decoding using API
*                       masca_intrptr_decode_msg_list
* \param p_decode_info: Pointer to detail information of the specfied
*                       decoded_msg. The information will be filled by
*                       this API
* \param p_msg        : Pointer to the message from MASCA drive. The buffer
*                       should just hold the data starting from message code
*                       omitting length and checksum.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : decoding is successful
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
enum masca_error
masca_intrptr_decode_msg_info(
		const enum masca_intrptr_decode_msg decoded_msg ,
		union masca_intrptr_decode_info * const p_decode_info ,
		const struct masca_intrptr_msg_info * const p_msg)
{
	enum masca_error ret_error = MASCA_PTR_NULL;
	unsigned char msg_count;
	bool is_msg_found = false;

	INTE_TD_TRACE("%s\n" ,  __func__);

	if ((p_msg == NULL) || (p_msg->p_buffer == NULL))
		return MASCA_PTR_NULL;

	if (p_msg->msg_length > 0) {
		if (decoded_msg < DIAG_TEMPERATURE) {
			msg_count = 0;
		} else if (decoded_msg < MESSAGE_MAX) {
			/*done just to jump to diag commands faster*/
			msg_count = DIAG_MSG_START_INDEX;
		} else {
			ret_error = MASCA_UNKNOWN_MSG_CMD;
			/*quit the loop ahead*/
			msg_count = MAX_RECEIVE_MESSAGES;
		}
		/*now search for the message from the configured table*/
		while ((msg_count < MAX_RECEIVE_MESSAGES)
			&& (false == is_msg_found)) {
			if (message_decoder[msg_count].msg_id ==
					p_msg->p_buffer[DRIVE_MSG_ID]) {
				ret_error =
				message_decoder[msg_count].decode_function(
						NULL, p_decode_info, p_msg,
						decoded_msg);
				is_msg_found = true;
			}
			msg_count++;
		}
		if (false == is_msg_found) {
			ret_error = MASCA_UNKNOWN_MSG_CMD;
			INTE_TE_TRACE(
			"intrptr_decode_msg_info MASCA_UNKNOWN_MSG_CMD\n");
		}
	} else {
		ret_error = MASCA_INVALID_PARAM;
		INTE_TE_TRACE("%s MASCA_UNKNOWN_MSG_CMD\n", __func__);
	}
	return ret_error;
}

static char find_conflict_drive_msg_id_index(
		const struct masca_intrptr_msg_info * const p_msg,
		unsigned char search_index)
{
	/* Check for the conflicting response id */
	if ((p_msg->p_buffer[DRIVE_MSG_ID] == MSG_MODE1_DATA_ID) ||
		(p_msg->p_buffer[DRIVE_MSG_ID] == MSG_TRACK_INFO_ID) ||
		(p_msg->p_buffer[DRIVE_MSG_ID] == MSG_TOC_DATA_ID)) {
		if ((p_msg->msg_length == MSG_LEN_INIT_LASER_CURRENT)
			&& (p_msg->p_buffer[DRIVE_MSG_ID] ==
					MSG_DIAG_INIT_LASER_CURRENT_ID)) {
			search_index = DIAG_MSG_INIT_LASER_INDEX;
		} else if ((p_msg->msg_length == MSG_LEN_MSG_LAST_ERRS)
			&& (p_msg->p_buffer[DRIVE_MSG_ID] ==
					MSG_MSG_LAST_ERRS_ID)) {
			search_index = DIAG_MSG_LAST_ERRS_INDEX;
		} else if ((p_msg->msg_length == MSG_LEN_USED_CD_TYPE_CNT)
			&& (p_msg->p_buffer[DRIVE_MSG_ID] ==
					MSG_USED_CD_TYPE_CNT_ID)) {
			search_index = DIAG_MSG_USED_CD_CNT_INDEX;
		}
	}

	return search_index;
}

static enum masca_error
masca_search_msgtable(struct masca_intrptr_msg_info_list * const p_msg_info,
		const struct masca_intrptr_msg_info * const p_msg,
		bool *is_msg_found)
{
	unsigned char search_idx = 0;
	enum masca_error ret_error = MASCA_PTR_NULL;

	/*now search for the message from the configured table*/
	while ((search_idx < MAX_RECEIVE_MESSAGES) &&
						(false == *is_msg_found)) {
		if (message_decoder[search_idx].msg_id ==
					p_msg->p_buffer[DRIVE_MSG_ID]) {
			/* Few command response id and diagnostic response id
			 * are having a same message response id */
			search_idx = find_conflict_drive_msg_id_index(p_msg,
								search_idx);

			ret_error = message_decoder[search_idx].decode_function(
					p_msg_info, NULL, p_msg, MESSAGE_MAX);
			*is_msg_found = true;
		}
		search_idx++;
	}

	return ret_error;
}

/*
* \func masca_intrptr_decode_msg_list
*
* Decodes the list of messages present in the message from MASCA drive.
* This API calls the respective function responsible for decoding the
* particular message.
*
* \param p_msg_info   : pointer to structure holding message list
*                       It will be filled by this API.
* \param p_msg        : Pointer to the message from MASCA drive. The buffer
*                       should just hold the data starting from message code
*                       omitting length and checksum.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : decoding is successful
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
enum masca_error
masca_intrptr_decode_msg_list(
			struct masca_intrptr_msg_info_list * const p_msg_info ,
			const struct masca_intrptr_msg_info * const p_msg)
{
	enum masca_error ret_error = MASCA_PTR_NULL;
	bool is_msg_found = false;

	if ((p_msg->p_buffer != NULL) && (p_msg->msg_length > 0)) {
		ret_error = masca_search_msgtable(p_msg_info, p_msg,
				&is_msg_found);
		if (false == is_msg_found) {
			ret_error = MASCA_UNKNOWN_MSG_CMD;
			INTE_TE_TRACE(
			"intrptr_decode_msg_list MASCA_UNKNOWN_MSG_CMD\n");
		}
	} else {
		if (p_msg->p_buffer == NULL) {
			INTE_TE_TRACE(
			"intrptr_decode_msg_list p_msg->p_buffer == NULL\n");
		} else {
			ret_error = MASCA_INVALID_PARAM;
			INTE_TE_TRACE(
			"intrptr_decode_msg_list MASCA_INVALID_PARAM\n");
		}
	}

	return ret_error;
}

static void
masca_intrptr_status_cmd(unsigned char * const p_encode_buffer,
			unsigned char * const p_length,
			enum masca_intrptr_cmd *masca_cmd,
			const struct masca_intrptr_cmd_info * const p_cmd_info)
{
	/*Its a status command*/
	p_encode_buffer[COMMAND_POSITION] = COMMAND_STATUS_ID;
	*p_length = COMMAND_HDR_LENGTH + COMMAND_STATUS_LENGTH;

	if (CMD_IIS_SW_OFF_ENABLE == *masca_cmd)
		is_iis_switch_off = true;
	else if (CMD_IIS_SW_OFF_DISABLE == *masca_cmd)
		is_iis_switch_off = false;
	else if (CMD_MULTISESSION_ON == *masca_cmd)
		is_multisession_on = true;
	else if (CMD_MULTISESSION_OFF == *masca_cmd)
		is_multisession_on = false;
	else if ((CMD_TRACK_AUDIO == *masca_cmd) ||
			(CMD_TRACK_START == *masca_cmd))
		p_encode_buffer[COMMAND_STATUS_TRACK_POS] =
			p_cmd_info->track_number;

	if (true == is_iis_switch_off)
		p_encode_buffer[IIS_SW_OFF_ENABLE_BYTE_POS] |=
				1 << IIS_SW_OFF_ENABLE_BIT_POS;

	if (true == is_multisession_on)
		p_encode_buffer[MULTI_ON_BYTE_POS] |= 1 << MULTI_ON_BIT_POS;
}

static void
masca_intrptr_play_cmd(unsigned char * const p_encode_buffer,
			unsigned char * const p_length,
			enum masca_intrptr_cmd *masca_cmd,
			const struct masca_intrptr_cmd_info * const p_cmd_info)
{
	/*Its a play command*/
	p_encode_buffer[COMMAND_POSITION] = COMMAND_PLAY_ID;
	if (CMD_RESUME == *masca_cmd) {
		*p_length = COMMAND_HDR_LENGTH + COMMAND_PLAY_LNG_RESUME;
	} else if ((CMD_PLAY != *masca_cmd) &&
		(CMD_PLAY_CONTINUOUS != *masca_cmd) &&
		(CMD_PLAY_CONTINUOUS_HALFMAX != *masca_cmd) &&
		(CMD_PLAY_CONTINUOUS_MAX != *masca_cmd)) {
		if ((CMD_SEARCH_FWD == *masca_cmd) ||
			(CMD_SEARCH_BWD == *masca_cmd) ||
			(CMD_FAST_SEARCH_FWD == *masca_cmd) ||
			(CMD_FAST_SEARCH_FWD == *masca_cmd)) {
			*p_length = COMMAND_HDR_LENGTH +
				COMMAND_PLAY_LNG_RESUME;
		} else {
			*p_length = COMMAND_HDR_LENGTH +
				COMMAND_PLAY_LENGTH_LOW;
		}
	} else {
		*p_length = COMMAND_HDR_LENGTH + COMMAND_PLAY_LENGTH_HIGH;
		if (p_cmd_info->track_number == 0) {
			p_encode_buffer[COMMAND_PLAY_DIR] =
						p_cmd_info->dir_number;
			p_encode_buffer[COMMAND_PLAY_TRK] =
						p_cmd_info->track_number;
			p_encode_buffer[COMMAND_PLAY_MIN] = p_cmd_info->minute;
			p_encode_buffer[COMMAND_PLAY_SEC] = p_cmd_info->second;
			p_encode_buffer[COMMAND_PLAY_FRM] = p_cmd_info->frame;
			p_encode_buffer[COMMAND_PLAY_BLK_CNT] =
					p_cmd_info->block_count;
		} else {
			p_encode_buffer[COMMAND_PLAY_DIR] = 0;
			p_encode_buffer[COMMAND_PLAY_TRK] =
						p_cmd_info->track_number;
			p_encode_buffer[COMMAND_PLAY_MIN] = 0;
			p_encode_buffer[COMMAND_PLAY_SEC] = 0;
			p_encode_buffer[COMMAND_PLAY_FRM] = 0;
			p_encode_buffer[COMMAND_PLAY_BLK_CNT] = 0;
		}
	}
}

static void
masca_intrptr_diag_cmd(unsigned char * const p_encode_buffer,
			unsigned char * const p_length,
			enum masca_intrptr_cmd *masca_cmd,
			const struct masca_intrptr_cmd_info * const p_cmd_info)
{
	/*Its a diagnostic command*/
	p_encode_buffer[COMMAND_POSITION] = COMMAND_DIAG_ID;
	p_encode_buffer[DIAG_SUB_CMD_POS] =
				masca_commands[*masca_cmd][DIAG_CMD_INDEX];
	if (*masca_cmd == CMD_WRITE_STICKER) {
		memcpy(&p_encode_buffer[DIAG_SUB_CMD_PARA_POS],
			p_cmd_info->date_write_sticker,
			DIAG_CMD_WIRTE_STICKER);
	} else if (*masca_cmd == CMD_DATE_INSTALL_WRITE) {
		memcpy(&p_encode_buffer[DIAG_SUB_CMD_PARA_POS],
			&p_cmd_info->instll_dat_write,
			MAX_DATE_PROD_INSTALL_SIZE);
	}
	*p_length =  COMMAND_HDR_LENGTH +
				masca_commands[*masca_cmd][DIAG_BYTE_CNT_POS];
}

/**
* \func masca_intrptr_encode_cmd
*
* encodes the command defined as enum masca_intrptr_cmd to the
* raw data which corresponds to command of the masca drive. This
* API does not fill the checksum and the message length field.
* These fields will be filled by msg_handler.
*
* \param p_cmd_info      :  pointer to the command informaiton structure. This
*                           should be filled by calling API with necessary
*                           parameters for the respective command.
*
* \param p_encode_buffer : Pointer to the buffer holding command to be
*                          sent to masca drive. It is filled by this API.
*
* \param p_length        : Pointer to length of the encoded command. It is
*                          filled by this API.
*
* \return <description of return value>
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown command
*  MASCA_OK             : command is successfully encoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
enum masca_error
masca_intrptr_encode_cmd(const struct masca_intrptr_cmd_info *
				const p_cmd_info ,
				unsigned char * const p_encode_buffer ,
				unsigned char * const p_length)
{
	enum masca_error ret_error = MASCA_OK;
	unsigned char bit_set_count;
	enum masca_intrptr_cmd masca_cmd;
	unsigned char bit_byte;
	unsigned char bit_pos;
	unsigned char byte_num;

	/*set encode buffer to 0*/
	memset(p_encode_buffer, 0, COMMAND_MAX_LENGTH);
	*p_length = 0;
	masca_cmd = p_cmd_info->command;

	if (masca_cmd < CMD_GET_TEMPERATURE) {
		/* Its not a diagnostic command
		 * perform bit sets for the particular command*/
		for (bit_set_count = 1;
			bit_set_count < COMMAND_MAX_BIT_SETS;
			bit_set_count++) {
			bit_byte = masca_commands[masca_cmd][bit_set_count];
			if (BIT_BYTE_INVALID != bit_byte) {
				bit_pos = (unsigned char)(
						(bit_byte & MASK_BIT_POS) >>
						SHIFT_CNT_HIGHER_NIBBLE);
				byte_num = bit_byte & MASK_BYTE_NUM;
				p_encode_buffer[byte_num] |= (1 << bit_pos);
			}
		}

		if (CMD_DIS_IGN_AUDIO >= masca_cmd) {
			masca_intrptr_status_cmd(p_encode_buffer, p_length,
							&masca_cmd, p_cmd_info);
		} else if ((masca_cmd >= CMD_TRACK_NAME) &&
				(masca_cmd <= CMD_TRACK_ATTRIBUTES)) {
			/*It is command track info */
			p_encode_buffer[COMMAND_POSITION] =
						COMMAND_TRACK_INFO_ID;
			p_encode_buffer[COMMAND_DIR_POS] =
						p_cmd_info->dir_number;
			p_encode_buffer[COMMAND_TRACK_POS] =
						p_cmd_info->track_number;
			*p_length = COMMAND_HDR_LENGTH +
						COMMAND_TRACK_INFO_LENGTH;
		} else {
			masca_intrptr_play_cmd(p_encode_buffer, p_length,
							&masca_cmd, p_cmd_info);
		}
	} else if (masca_cmd < CMD_MAX) {
		/* Its a diagnostic command */
		masca_intrptr_diag_cmd(p_encode_buffer, p_length, &masca_cmd,
							p_cmd_info);
	} else {
		ret_error = MASCA_UNKNOWN_MSG_CMD;
		INTE_TE_TRACE("intrptr_encode_cmd MASCA_UNKNOWN_MSG_CMD\n");
	}

	return ret_error;
}


/**
* \func mode1_data_decode_func
*
* This function decodes the mode1_data message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
mode1_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/

	if (NULL != p_msg_list) {
		p_msg_list->msg_group = MSG_PLAY_POSITION;
		error = MASCA_OK;
		p_msg_list->num_of_msgs = MODE1_DATA_MSGS;
		p_msg_list->decoded_messages[0] = TRACK_AUDIO;
		p_msg_list->decoded_messages[1] = PLAY_INFO;
	}

	if (MSG_LENGTH_MODE1_DATA != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (TRACK_AUDIO == decoded_msg) {
			p_decode_info->track_info.adr_control =
					p_msg->p_buffer[MODE1_ADR_CTRL];
		} else if (PLAY_INFO == decoded_msg) {
			p_decode_info->play_pos.track_num =
					p_msg->p_buffer[MODE1_TRK_NO];
			p_decode_info->play_pos.track_subindex =
					p_msg->p_buffer[MODE1_IDX_INFO];
			p_decode_info->play_pos.min_relative =
					p_msg->p_buffer[MODE1_MIN];
			p_decode_info->play_pos.sec_relative =
					p_msg->p_buffer[MODE1_SEC];
			p_decode_info->play_pos.min_absolute =
					p_msg->p_buffer[MODE1_ABS_MIN];
			p_decode_info->play_pos.sec_absolute =
					p_msg->p_buffer[MODE1_ABS_SEC];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

/**
* \func track_data_decode_func
*
* This function decodes the track_data message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
track_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/

	if (NULL != p_msg_list) {
		p_msg_list->msg_group = MSG_TRACK_DATA;
		error = MASCA_OK;
		p_msg_list->num_of_msgs = TRACK_DATA_MSGS;
		p_msg_list->decoded_messages[0] = TRACK_AUDIO;
		p_msg_list->decoded_messages[1] = TRACK_START;
	}

	if (MSG_LENGTH_TRACK_DATA != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (TRACK_AUDIO == decoded_msg) {
			p_decode_info->track_info.adr_control =
						p_msg->p_buffer[TRK_ADR_CTRL];
		} else if (TRACK_START == decoded_msg) {
			p_decode_info->track_start.track_number =
						p_msg->p_buffer[TRK_TRK_NO];
			p_decode_info->track_start.start_min =
						p_msg->p_buffer[TRK_MIN];
			p_decode_info->track_start.start_sec =
						p_msg->p_buffer[TRK_SEC];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

/**
* \func track_text_decode_func
*
* This function decodes the track_text message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
track_text_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	unsigned char *p_buf;
	/*NULL checks for p_msg is carried out ,  outside this function*/

	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_TRACK_TEXT;
		p_msg_list->num_of_msgs = TRACK_TEXT_MSGS;
		p_buf = p_msg->p_buffer;
		if (test_bit(TEXT_NAME, (unsigned long *)p_buf))
			*p_msg_list->decoded_messages = TRACK_NAME;
		else if (test_bit(TEXT_AUTHOR, (unsigned long *)p_buf))
			*p_msg_list->decoded_messages = AUTHOR_NAME;
		else if (test_bit(TEXT_ALBUM, (unsigned long *)p_buf))
			*p_msg_list->decoded_messages = ALBUM_NAME;
		else if (test_bit(TEXT_DIR_NAME, (unsigned long *)p_buf))
			*p_msg_list->decoded_messages = DIR_NAME;
		else if (test_bit(TEXT_FILE_NAME, (unsigned long *)p_buf))
			*p_msg_list->decoded_messages = FILE_NAME;
		else
			error = MASCA_UNKNOWN_MSG_CMD;
	}

	if ((MAX_TRACK_TEXT_LENGTH < p_msg->msg_length) ||
			(MIN_TRACK_TEXT_LENGTH > p_msg->msg_length)) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		p_decode_info->track_text.char_encoding =
						p_msg->p_buffer[TXT_CODE];
		p_decode_info->track_text.dir_number =
						p_msg->p_buffer[TXT_DIR_NO];
		p_decode_info->track_text.track_number =
						p_msg->p_buffer[TXT_TRK_NO];
		p_decode_info->track_text.p_name = &p_msg->p_buffer[TXT_INFO];
		p_decode_info->track_text.name_length =
						p_msg->msg_length - TXT_INFO;
	}

	return error;
}

/**
* \func track_info_decode_func
*
* This function decodes the track_info message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
track_info_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_TRACK_INFO;
		p_msg_list->num_of_msgs = TRACK_INFO_MSGS;
		p_msg_list->decoded_messages[0] = TRACK_ATTRIBUTES;
		p_msg_list->decoded_messages[1] = TRACK_DURATION;
	}

	if (MSG_LENGTH_TRACK_INFO != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (TRACK_ATTRIBUTES == decoded_msg) {
			/*Implement the full decoding when these parameter
			come into use. As of now the parameters are not fully
			decoded. Also these parameters are not needed. */
			p_decode_info->track_attr.audio_mode =
				p_msg->p_buffer[TRK_INFO_1] & AUDIO_MODE_MASK;
			p_decode_info->track_attr.bit_rate =
				(unsigned short)(p_msg->p_buffer[TRK_INFO_2]
							& BIT_RATE_MASK);
			p_decode_info->track_attr.sampling_rate =
			p_msg->p_buffer[TRK_INFO_DIR] & SAMPLING_RATE_MASK;
		} else if (TRACK_DURATION == decoded_msg) {
			p_decode_info->track_length.directory_num =
					p_msg->p_buffer[TRK_INFO_DIR];
			p_decode_info->track_length.trk_num =
					p_msg->p_buffer[TRK_INFO_TRK_NO];
			p_decode_info->track_length.duration_minutes  =
					p_msg->p_buffer[TRK_INFO_MIN];
			p_decode_info->track_length.duration_seconds  =
					p_msg->p_buffer[TRK_INFO_SEC];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

/**
* \func toc_data_decode_func
*
* This function decodes the toc_data message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
toc_data_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_TOC_DATA;
		p_msg_list->num_of_msgs = TOC_DATA_MSGS;
		*p_msg_list->decoded_messages = SYS_TOC;
	}

	if (MSG_LENGTH_TOC_DATA != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (SYS_TOC == decoded_msg) {
			p_decode_info->sys_toc.minimum_track_num =
					p_msg->p_buffer[TOC_MIN_TRK_NO];
			p_decode_info->sys_toc.maximum_track_num =
					p_msg->p_buffer[TOC_MAX_TRK_NO];
			p_decode_info->sys_toc.max_cd_min =
					p_msg->p_buffer[TOC_TOTAL_MIN];
			p_decode_info->sys_toc.max_cd_sec =
					p_msg->p_buffer[TOC_TOTAL_SEC];
			p_decode_info->sys_toc.last_toc_min =
					p_msg->p_buffer[TOC_LAST_TRK_MIN];
			p_decode_info->sys_toc.last_toc_sec =
					p_msg->p_buffer[TOC_LAST_TRK_SEC];
			p_decode_info->sys_toc.last_toc_frame =
					p_msg->p_buffer[TOC_LAST_TRK_FRM];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}


/**
* \func version_decode_func
*
* This function decodes the version_info message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
version_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	unsigned char * const p_buf = p_msg->p_buffer;

	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_VERSION_INFO;
		p_msg_list->num_of_msgs = VERSION_INFO_MSGS;
		p_msg_list->decoded_messages[0] = VERSION_INFO;
		p_msg_list->decoded_messages[1] = FEATURE_INFO;
	}

	if (MSG_LENGTH_VERSION_INFO != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (VERSION_INFO == decoded_msg) {
			p_decode_info->ver_info.sw_version =
							p_buf[VER_SOFT_NO];
			p_decode_info->ver_info.hw_version =
							p_buf[VER_HARD_NO];
			/* typecast 2 lower bits for Vendor id */
			p_decode_info->ver_info.vendors_id =
					p_buf[VER_FEATURE] & VENDOR_ID_MASK;
		} else if (FEATURE_INFO == decoded_msg) {
			p_decode_info->feature_info.has_playlist_functionality =
				test_bit(VER_FEATURE_PLAYLIST,
						(unsigned long *)p_buf);
			p_decode_info->feature_info.has_esp_memory_support =
				test_bit(VER_FEATURE_ESP,
						(unsigned long *)p_buf);
			p_decode_info->feature_info.has_aac_support =
				test_bit(VER_FEATURE_AAC,
						(unsigned long *)p_buf);
			p_decode_info->feature_info.has_wma_support =
				test_bit(VER_FEATURE_WMA,
						(unsigned long *)p_buf);
			p_decode_info->feature_info.has_MP3_support =
				test_bit(VER_FEATURE_MP3,
						(unsigned long *)p_buf);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}


/**
* \func status_decode_func
*
* This function decodes the status message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message.
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
status_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_UNKNOWN_MSG_CMD;
	enum masca_intrptr_decode_msg dsp_init_state;
	enum masca_intrptr_decode_msg mechanical_init_state =
					MECHANICAL_INIT_PENDING;
	enum masca_intrptr_decode_msg *p_msg_container;
	unsigned char *p_buf;

	INTE_TD_TRACE("%s\n" ,  __func__);
	/*NULL checks for p_msg is carried out ,  outside of this function*/
	p_buf = p_msg->p_buffer;

	if ((MIN_MSG_LNG_STATUS > p_msg->msg_length) ||
			(MAX_MSG_LNG_STATUS < p_msg->msg_length)) {
		return error;
	} else if (NULL != p_msg_list) {
		INTE_TI_TRACE("Status message from the drive");
		p_msg_list->msg_group = MSG_STATUS;
		p_msg_container = p_msg_list->decoded_messages;
		error = MASCA_OK;

		if (!(test_bit(MSG_STATUS_DSP_INIT_ERROR,
						(unsigned long *)p_buf))) {
			/*software of the drive got initialized successfully*/
			*p_msg_container++ = DSP_INIT_SUCCESSFUL;
			dsp_init_state = DSP_INIT_SUCCESSFUL;
		} else {
			*p_msg_container++ = DSP_INIT_FAILURE;
			dsp_init_state = DSP_INIT_FAILURE;
		}

		if (DSP_INIT_SUCCESSFUL == dsp_init_state)
			mechanical_init_state = decode_bits_after_dsp_init(
						p_msg->p_buffer,
						&p_msg_container);

		if (MECHANICAL_INIT_COMPLETE == mechanical_init_state)
			decode_bits_after_mech_init(p_msg->p_buffer,
						&p_msg_container);

		p_msg_list->num_of_msgs =
				p_msg_container - p_msg_list->decoded_messages;
	}

	if ((NULL != p_decode_info) && (COMMAND_REJECTED == decoded_msg))
			decode_rejected_command(p_buf ,  p_decode_info);

	return error;
}


/**
* \func decode_bits_after_dsp_init
*
* This function decodes the bits of status message which are valid
* in dsp initialized state.
*
* \param p_buf           : Pointer to the MASCA message. This should be
*                          filled by the caller.
*
* \param p_msg_container : Pointer to the array holding list of message
*                          It is filled by this API
*
* \param p_count         : Pointer to the count variable. This API increments
*                          the count by as much sub messages as it adds to the
*                          list.
*
* \return
*  MECHANICAL_INIT_COMPLETE: mechanical initialization is complete
*  MECHANICAL_INIT_PENDING : mechanical initialization is not yet complete
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_intrptr_decode_msg
decode_bits_after_dsp_init(unsigned char const * const p_buf,
			enum masca_intrptr_decode_msg **pp_msg_container)
{
	enum masca_intrptr_decode_msg mechanical_init_state;
	enum masca_intrptr_decode_msg *p_msg_que = NULL;
	p_msg_que = *pp_msg_container;

	/*The parameters to this function are already checked for NULL*/
	if (test_bit(MSG_STATUS_POWER_FAIL, (unsigned long *)p_buf))
		*p_msg_que++ = DRIVE_POWER_FAIL;
	else
		*p_msg_que++ = DRIVE_POWER_OK;

	if (test_bit(MSG_STATUS_INIT_READY, (unsigned long *)p_buf)) {
		*p_msg_que++ = MECHANICAL_INIT_COMPLETE;
		mechanical_init_state = MECHANICAL_INIT_COMPLETE;
		INTE_TI_TRACE("MECHANICAL_INIT_COMPLETE in decode_msg\n");
	} else {
		*p_msg_que++ = MECHANICAL_INIT_PENDING;
		mechanical_init_state = MECHANICAL_INIT_PENDING;
	}

	if (test_bit(MSG_STATUS_IIS_OFF_REQESTED, (unsigned long *)p_buf))
		*p_msg_que++ = IIS_OFF_REQUEST;

	if (test_bit(MSG_STATUS_GENERAL_ERROR, (unsigned long *)p_buf))
		*p_msg_que++ = DRIVE_MECHANICAL_FAILURE;

	if (test_bit(MSG_STATUS_VERSION_ERROR, (unsigned long *)p_buf))
		*p_msg_que++ = HW_SW_VERSIONS_INCOMPATIBLE;

	if (test_bit(MSG_STATUS_REJECT, (unsigned long *)p_buf))
		*p_msg_que++ = COMMAND_REJECTED;
	else
		*p_msg_que++ = COMMAND_ACCEPTED;

	if (test_bit(MSG_STATUS_POWER_OFF, (unsigned long *)p_buf))
		*p_msg_que++ = DRIVE_POWEROFF_ALLOWED;
	else
		*p_msg_que++ = DRIVE_POWEROFF_NOT_ALLOWED;

	if (test_bit(MSG_STATUS_CONT_PLAY, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_PLAYMODE_CONTINOUS;
	else
		*p_msg_que++ = DISC_PLAYMODE_NON_CONTINOUS;

	if (test_bit(MSG_STATUS_MULTISESSION_ON, (unsigned long *)p_buf))
		*p_msg_que++ = MULTISESSION_PLAYBACK_ENABLED;
	else
		*p_msg_que++ = MULTISESSION_PLAYBACK_DISABLED;

	if ((test_bit(MSG_STATUS_TEMP_ERROR, (unsigned long *)p_buf)) &&
		(true == enable_temp_limit)) {
		*p_msg_que++ = TEMPERATURE_LIMIT_REACHED;
		masca_set_async_sense(SENSE_TEMP_ERR);
	} else {
		*p_msg_que++ = TEMPERATURE_OK;
		masca_clr_async_sense(SENSE_TEMP_ERR);
	}

	masca_update_drive_temperature();
	*pp_msg_container = p_msg_que;
	return mechanical_init_state;
}

/* Extract CD disc inserted and TOC ready
 * related information from CD drive status message */
static void
masca_cd_inserted_toc_ready_status_msg(unsigned char const * const p_buf,
			enum masca_intrptr_decode_msg **pp_msg_container)
{
	enum masca_intrptr_decode_msg *p_msg_que = NULL;
	p_msg_que = *pp_msg_container;

	/*disc is inserted and TOC is ready*/
	if (test_bit(MSG_STATUS_STOP, (unsigned long *)p_buf)) {
		*p_msg_que++ = DISC_STOPPED;
	} else if (test_bit(MSG_STATUS_PLAY_MODE, (unsigned long *)p_buf)) {
		*p_msg_que++ = DISC_PLAYING;
		if (test_bit(MSG_STATUS_TRACK_END, (unsigned long *)p_buf))
			*p_msg_que++ = TRACK_END_REACHED;
		else
			*p_msg_que++ = TRACK_END_NOT_REACHED;
	} else if (test_bit(MSG_STATUS_PAUSE_MODE, (unsigned long *)p_buf)) {
		*p_msg_que++ = DISC_PAUSED;
		if (test_bit(MSG_STATUS_TRACK_END, (unsigned long *)p_buf))
			*p_msg_que++ = TRACK_END_REACHED;
		else
			*p_msg_que++ = TRACK_END_NOT_REACHED;
	} else if (test_bit(MSG_STATUS_SEARCH_MODE, (unsigned long *)p_buf)) {
		*p_msg_que++ = DISC_SEARCHING;
		if (test_bit(MSG_STATUS_TRACK_END, (unsigned long *)p_buf))
			*p_msg_que++ = TRACK_END_REACHED;
		else
			*p_msg_que++ = TRACK_END_NOT_REACHED;
	} else {
		*p_msg_que++ = DISC_INSERTED;
	}
	*p_msg_que++ = TOC_READY;

	if (test_bit(MSG_STATUS_CDROM, (unsigned long *)p_buf))
		*p_msg_que++ = TRACKS_CD_ROM_ONLY;
	else
		*p_msg_que++ = TRACKS_AUDIO_ONLY;

	if (test_bit(MSG_STATUS_IGNORE_AUDIO, (unsigned long *)p_buf))
		/* On Mixed-mode CD,Audio tracks are ignored */
		*p_msg_que++ = AUDIO_TRACKS_IGNORED;
	else
		/* On Mixed-mode CD, Data tracks are ignored */
		*p_msg_que++ = AUDIO_TRACKS_CONSIDERED;

	/*masca spec does not specify when multisession bit
	will be set. Logically it should be set when toc_ready
	bit is set */
	if (test_bit(MSG_STATUS_MULTISESSION, (unsigned long *)p_buf)) {
		*p_msg_que++ = DISC_MULTISESSION;
		if (test_bit(MSG_STATUS_LAST_SESSION, (unsigned long *)p_buf))
			*p_msg_que++ = LAST_SESSION_UNREADABLE;
		else
			*p_msg_que++ = LAST_SESSION_READABLE;
	} else {
		*p_msg_que++ = DISC_SINGLE_SESSION;
	}

	if (test_bit(MSG_STATUS_CD_TEXT_AVAIL, (unsigned long *)p_buf))
		*p_msg_que++ = CD_TEXT_AVAILABLE;
	else
		*p_msg_que++ = CD_TEXT_UNAVAILABLE;

	if (test_bit(MSG_STATUS_FRAME_ERROR, (unsigned long *)p_buf))
		*p_msg_que++ = BLOCK_UNREADABLE;
	else
		*p_msg_que++ = BLOCK_READABLE;

	if (test_bit(MSG_STATUS_MUTE_ON, (unsigned long *)p_buf))
		*p_msg_que++ = MUTE_ON;
	else
		*p_msg_que++ = MUTE_OFF;

	if (test_bit(MSG_STATUS_DURATION_AVAIL, (unsigned long *)p_buf))
		*p_msg_que++ = TRACK_DURATION_AVAILABLE;
	else
		*p_msg_que++ = TRACK_DURATION_UNAVAILABLE;

	*pp_msg_container = p_msg_que;
}

/* Extract CD disc present in slot related information from
 * CD drive status message */
static void
masca_cd_in_slot_status_msg(unsigned char const * const p_buf,
			enum masca_intrptr_decode_msg **pp_msg_container)
{
	enum masca_intrptr_decode_msg *p_msg_que = NULL;
	p_msg_que = *pp_msg_container;

	/*disc is not inserted but disc is present*/
	if (test_bit(MSG_STATUS_INSERT_BUSY, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_INSERTING;
	else if (test_bit(MSG_STATUS_EJECT_BUSY, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_EJECTING;
	else if (test_bit(MSG_STATUS_LOADING_ERROR, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_LOAD_ERROR;
	else
		/*CD is just parked at the tip of the drive*/
		*p_msg_que++ = DISC_IN_SLOT;

	*pp_msg_container = p_msg_que;
}

/**
* \func decode_bits_after_mech_init
*
* This function decodes the bits of status message which are valid
* after mechanical initialization.
*
* \param p_buf           : Pointer to the MASCA message. This should be
*                          filled by the caller.
*
* \param p_msg_container : Pointer to the array holding list of message
*                          It is filled by this API
*
* \param p_count         : Pointer to the count variable. This API increments
*                          the count by as much sub messages as it adds to the
*                          list.
*
* \return
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static void
decode_bits_after_mech_init(unsigned char const * const p_buf,
			enum masca_intrptr_decode_msg **pp_msg_container)
{
	enum masca_intrptr_decode_msg *p_msg_que = NULL;
	p_msg_que = *pp_msg_container;

	/*The parameters to this function are already checked for NULL*/
	if (test_bit(MSG_STATUS_NO_DISC, (unsigned long *)p_buf)) {
		/*disc is not present*/
		*p_msg_que++ = DISC_ABSENT;
		/*It is possible to have eject or insert mechanical simulation
		without using disc*/
		if (test_bit(MSG_STATUS_INSERT_BUSY, (unsigned long *)p_buf)) {
			*p_msg_que++ = DISC_INSERTING;
		} else if (test_bit(MSG_STATUS_EJECT_BUSY,
						(unsigned long *)p_buf)) {
			*p_msg_que++ = DISC_EJECTING;
		}
		/* extracted without NO CD related info from status message */
		goto masca_no_cd;
	}

	if (test_bit(MSG_STATUS_DISC_DEFECT, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_DEFECTIVE;
	else
		*p_msg_que++ = DISC_NOT_DEFECTIVE;

	if ((test_bit(MSG_STATUS_DISC_INSERTED, (unsigned long *)p_buf)) &&
		(test_bit(MSG_STATUS_TOC_READY, (unsigned long *)p_buf))) {
		/*disc is inserted and TOC is ready*/
		masca_cd_inserted_toc_ready_status_msg(p_buf, &p_msg_que);
	} else if ((test_bit(MSG_STATUS_DISC_INSERTED,
						(unsigned long *)p_buf)) &&
		(!(test_bit(MSG_STATUS_TOC_READY, (unsigned long *)p_buf)))) {
		/*disc is inserted and TOC is not ready yet*/
		*p_msg_que++ = DISC_INSERTED;

		if (test_bit(MSG_STATUS_READING_CD, (unsigned long *)p_buf))
			*p_msg_que++ = TOC_READING_IN_PROGRESS;

		if (test_bit(MSG_STATUS_LOADING_ERROR, (unsigned long *)p_buf))
			/* Disc inserted and loding error */
			*p_msg_que++ = DISC_LOAD_ERROR;

		if (test_bit(MSG_STATUS_TOC_ERROR, (unsigned long *)p_buf))
			*p_msg_que++ = TOC_UNREADABLE;
	} else if (!(test_bit(MSG_STATUS_DISC_INSERTED,
					(unsigned long *)p_buf))) {
		/*disc is not inserted but disc is present*/
		masca_cd_in_slot_status_msg(p_buf, &p_msg_que);
	}

	if (test_bit(MSG_STATUS_HIGH_ROTATION_ON, (unsigned long *)p_buf))
		*p_msg_que++ = DRIVE_SPEED_HIGH;
	else if (test_bit(MSG_STATUS_HALF_HIGH_ROTATION,
						(unsigned long *)p_buf))
		*p_msg_que++ = DRIVE_SPEED_HALF_MAXIMUM;
	else
		*p_msg_que++ = DRIVE_SPEED_1X;

	if (test_bit(MSG_STATUS_CM_8_REJECTED, (unsigned long *)p_buf))
		*p_msg_que++ = DISC_8CM_UNSUPPORTED;
	else
		*p_msg_que++ = DISC_8CM_SUPPORTED;

masca_no_cd:
	*pp_msg_container = p_msg_que;
}

#define FIRST_BYTE_CMD_REJ	1 /* Command rejected info in first byte */
#define SECODN_BYTE_CMD_REJ	2 /* Command rejected info in second byte */
/**
* \func decode_rejected_command
*
* This API decodes the command rejected by the drive which is coming
* as part of status message.
*
* \param p_buf         :   Pointer to the MASCA message. This should be
*                          filled by the caller.
*
* \param p_decode_info :   Pointer to the deocde structure. This will be
*                          filled by this API.
*
* \return
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static void
decode_rejected_command(unsigned char * const p_buf ,
			union masca_intrptr_decode_info * const p_decode_info)
{
	enum masca_intrptr_cmd cmd;
	unsigned char info_cmd[4] = {BIT_BYTE_INVALID ,  BIT_BYTE_INVALID ,
	BIT_BYTE_INVALID ,  BIT_BYTE_INVALID};
	unsigned char param_count = 0;
	unsigned char bytes_to_compare = 4;
	unsigned char cmd_count = 0;
	unsigned char cmd_rejected;
	unsigned char compare_count;

	/*NULL checks for params are done outside of this function*/
	/*copy the command code*/
	info_cmd[param_count] = p_buf[DRIVE_REJ_CMD_ID_POS];
	param_count++;
	cmd_rejected = info_cmd[0];
	p_decode_info->reject_cmds.num_of_rejected_cmds = 0;

	if (COMMAND_TRACK_INFO_ID == cmd_rejected) {
		/*Only first byte of this command involves bit setting*/
		get_bit_byte_set_values(info_cmd, p_buf[DRIVE_REJ_CMD_PARAM1],
					&param_count, FIRST_BYTE_CMD_REJ);
	} else if ((COMMAND_PLAY_ID == cmd_rejected)
		|| (COMMAND_STATUS_ID == cmd_rejected)) {
		/*Only first two bytes of this command involves bit setting*/
		get_bit_byte_set_values(info_cmd, p_buf[DRIVE_REJ_CMD_PARAM1],
					&param_count, FIRST_BYTE_CMD_REJ);
		get_bit_byte_set_values(info_cmd, p_buf[DRIVE_REJ_CMD_PARAM2],
					&param_count, SECODN_BYTE_CMD_REJ);
	} else if (COMMAND_DIAG_ID == cmd_rejected) {
		/*it is a diag command*/
		info_cmd[1] = p_buf[DRIVE_REJ_CMD_PARAM1];
		bytes_to_compare = 2;
	} else {
		bytes_to_compare = 0;
		INTE_TE_TRACE(
		"decode_rejected_command MASCA_INTRPTR_UNKNOWN_REJECT_CMD\n");
	}

	if ((bytes_to_compare > 0) && (bytes_to_compare <= 4)) {
		for (cmd = CMD_VERSION_INFO; cmd < CMD_MAX; cmd++) {
			/*go through masca_commands table to find out the
			command corresponding to mathcing bit sets
			*/
			for (compare_count = 0;
				compare_count < bytes_to_compare;
				compare_count++) {
				if (masca_commands[cmd][compare_count]
						== info_cmd[compare_count]) {
					if (compare_count ==
					       (bytes_to_compare - 1)) {
						p_decode_info->reject_cmds
						  .reject_cmds[cmd_count] = cmd;
						cmd_count++;
						p_decode_info->reject_cmds
							.num_of_rejected_cmds =
								      cmd_count;
					}
				} else {
					/*just move to next command*/
					compare_count = bytes_to_compare;
				}
			}
		}
	}
}

#define MAX_PARAM_COUNT 3
#define MAX_BIT_CNT	7 /* bit0 to bit7 */
/**
* \func get_bit_byte_set_values
*
* This API converts the bit sets to values as indicated in masca_commands
* array. Refer to this array to understand the bit set logic. Converse of
* it is implemented over here.
*
* \param p_buf         :   Pointer to the MASCA message. This should be
*                          filled by the caller.
*
* \param byte          :   The byte value whose bit sets needs to be analysed
*
* \param p_param_count :   Inidicates the current bit set parameters. Also
*                          this API updates the number of parameter it adds.
* \return
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static void get_bit_byte_set_values(unsigned char * const p_buf ,
					const unsigned char byte ,
					unsigned char * const p_param_count ,
					const unsigned char byte_num)
{
	unsigned char bit_count;
	unsigned char bit;
	unsigned char param_count;

	param_count = *p_param_count;
	/*now we are just doing the reverse of what we do when we encode
	a command.i.e. from the bit set logic we are obtaining the codes
	stored in the array masca_commands
	*/
	for (bit_count = 0;
		(bit_count <= MAX_BIT_CNT) && (param_count <= MAX_PARAM_COUNT);
		bit_count++) {
		bit = (unsigned char)(1 << bit_count);
		if (((unsigned char)(byte & bit)) == bit) {
			p_buf[param_count] = (unsigned char)
				(((unsigned char)(bit_count << 4)) | byte_num);
			param_count++;
		}
	}
	*p_param_count = param_count;
}




/**
* \func status_decode_func
*
* This function decodes the temperature message of MASCA drive
* The necessary parameters should be checked for NULL before
* calling this function. It serves as a common function for decoding
* the list of sub messages in a message and decoding details of a
* sub message. It returns temperature in degree centigrade format
*
* \param p_msg_list    : Pointer to list holding messages. This will be
*                        filled by this API.
*
* \param p_decode_info : Pointer to the detailed decoded information of
*                        a sub message represented by enum "decoded_msg".
*                        This structure is filled by this API.
*
* \param p_msg         : Pointer to the message from MASCA drive. The buffer
*                        should just hold the data starting from message code
*                        omitting length and checksum. The contents should
*                        be filled by caller.
*
* \param decoded_msg   : sub message whose detail needs to be decoded to
*                        parameters of p_decode_info.
*
* \return
*  MASCA_PTR_NULL       : one of the parameters is NULL
*  MASCA_UNKNOWN_MSG_CMD: unknown message
*  MASCA_OK             : message is successfully decoded
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static enum masca_error
diag_temp_decode_func(struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	unsigned short adc_count = 0;
	short conv_temp = 0;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_TEMPERATURE;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_TEMPERATURE;
	}

	if (MSG_LEN_TEMPERATURE != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_TEMPERATURE == decoded_msg) {
			adc_count = (((unsigned short)(
					p_msg->p_buffer[diag_res_param1] <<
								BYTE_2_START)) |
					(p_msg->p_buffer[diag_res_param2]));
			conv_temp = convert_to_centigrade(adc_count);

			p_decode_info->diag_info.drv_temperature[0] =
			(unsigned char)((conv_temp >> BYTE_2_START)
							& BYTE_MASK);
			p_decode_info->diag_info.drv_temperature[1] =
			(unsigned char)((conv_temp) & BYTE_MASK);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error
diag_laser_cur_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	unsigned int current_consum_hex = 0; /* in hexa-decimal number */
	unsigned int laser_current = 0;

	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_LASER_CURRENT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_LASER_CURRENT;
	}

	if (MSG_LEN_LASER != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_LASER_CURRENT == decoded_msg) {
			current_consum_hex = (((unsigned int)
				((p_msg->p_buffer[diag_res_param1])
							<< BYTE_2_START)) |
				(p_msg->p_buffer[diag_res_param2]));
			laser_current = GET_CURRENT_CONSUM(current_consum_hex);
			p_decode_info->diag_info.drv_laser_current[0] =
			      (unsigned char)((laser_current >> BYTE_4_START)
							      & BYTE_MASK);
			p_decode_info->diag_info.drv_laser_current[1] =
			      (unsigned char)((laser_current >> BYTE_3_START)
							      & BYTE_MASK);
			p_decode_info->diag_info.drv_laser_current[2] =
			      (unsigned char)((laser_current >> BYTE_2_START)
							      & BYTE_MASK);
			p_decode_info->diag_info.drv_laser_current[3] =
			      (unsigned char)((laser_current >> BYTE_1_START)
							      & BYTE_MASK);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error
diag_switch_status_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_SWITCH_STATUS;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_SWITCH_STATUS;
	}

	if (MSG_LEN_SWITCH_STATUS != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_SWITCH_STATUS == decoded_msg)
			p_decode_info->diag_info.drv_switch_status =
					p_msg->p_buffer[diag_res_param1];
		else
			error = MASCA_UNKNOWN_MSG_CMD;
	}

	return error;
}


static enum masca_error
diag_adjument_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_ADJUSTMENT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_ADJUSTMENT;
	}

	if (MSG_LEN_ADJUST != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_ADJUSTMENT == decoded_msg) {
			memcpy(&p_decode_info->diag_info.drv_diag_adjust.
								focus_gain,
				&p_msg->p_buffer[diag_res_param1],
				MAX_DIAG_ADJUST_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error
diag_cmd_recevied_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_CMD_RECEVIED;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_CMD_RECEVIED;
	}

	if (MSG_LEN_CMD_RECEIVED != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_CMD_RECEVIED == decoded_msg)
			p_decode_info->diag_info.drv_switch_status =
					p_msg->p_buffer[diag_res_param1];
		else
			error = MASCA_UNKNOWN_MSG_CMD;
	}

	return error;
}

static enum masca_error
diag_c1c2_err_cnt_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_C1C2_ERR_COUNT_ON;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_C1C2_ERR_COUNT_ON;
	}

	if (MSG_LEN_C1C2_ERR_CNT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_C1C2_ERR_COUNT_ON == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_c1c2_err_count,
				&p_msg->p_buffer[diag_res_param1],
				MAX_C1C2_ERR_CNT_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_time_readout_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_TIME_READOUT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_TIME_READOUT;
	}

	if (MSG_LEN_TIME_READOUT_CNT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_TIME_READOUT == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_time_readout,
				&p_msg->p_buffer[diag_res_param1],
				MAX_DIAG_TIME_READOUT_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_eject_load_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_EJECT_LOAD_CYCLES;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_EJECT_LOAD_CYCLES;
	}

	if (MSG_LEN_EJT_LD_CYCLES != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_EJECT_LOAD_CYCLES == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_eject_load_cycles,
				&p_msg->p_buffer[diag_res_param1],
				MAX_EJT_LD_CYC_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_jump_counter_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_JUMP_COUNTER;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_JUMP_COUNTER;
	}

	if (MSG_LEN_JUMP_CNT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_JUMP_COUNTER == decoded_msg) {
			p_decode_info->diag_info.drv_jump_cnt.eeprom_chksum =
					p_msg->p_buffer[diag_res_param1];
			memcpy(p_decode_info->diag_info.drv_jump_cnt.
					jump_counter,
				&p_msg->p_buffer[diag_res_param2],
				(MAX_JUMP_CNT_SIZE - DIAG_CHECKSUM_SIZE));
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_eeprom_readout_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_EEPROM_READOUT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_EEPROM_READOUT;
	}

	if (MSG_LEN_EEPROM_READOUT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_EEPROM_READOUT == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_eeprom_readout,
				&p_msg->p_buffer[diag_res_param1],
				MAX_EEPROM_READOUT_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_write_sticker_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_WRITE_STICKER;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_WRITE_STICKER;
	}

	if (MSG_LEN_WRITE_STICKER != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_WRITE_STICKER == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_write_sticker_no,
				&p_msg->p_buffer[diag_res_param1],
				MAX_WRITE_STICKER_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_read_sticker_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_READ_STICKER;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_READ_STICKER;
	}

	if (MSG_LEN_READ_STICKER != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_READ_STICKER == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_read_sticker_no,
				&p_msg->p_buffer[diag_res_param1],
				MAX_READ_STICKER_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}


static enum masca_error diag_jitter_measure_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_JITTER_MEASURE;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_JITTER_MEASURE;
	}

	if (MSG_LEN_JITTER_MEASURE != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_JITTER_MEASURE == decoded_msg) {
			p_decode_info->diag_info.drv_jitter_measure =
					p_msg->p_buffer[diag_res_param1];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_vertical_measure_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list,
			union masca_intrptr_decode_info * const p_decode_info,
			const struct masca_intrptr_msg_info * const p_msg,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_VERTICAL_DEV_MEASURE;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_VERTICAL_DEV_MEASURE;
	}

	if (MSG_LEN_VERTICAL_MEASURE != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_VERTICAL_DEV_MEASURE == decoded_msg) {
			memcpy(p_decode_info->diag_info.
					drv_vertical_dev_measure,
				&p_msg->p_buffer[diag_res_param1],
				MAX_VER_DEV_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_eccentric_measure_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_ECCENTRIC_MEASURE;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_ECCENTRIC_MEASURE;
	}

	if (MSG_LEN_ECCENTRIC_MEASURE != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_ECCENTRIC_MEASURE == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_eccentric_measure,
				&p_msg->p_buffer[diag_res_param1],
				MAX_ECCENTRICITY_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_reflective_measure_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_REFLECTIVE_MEASURE;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_REFLECTIVE_MEASURE;
	}

	if (MSG_LEN_REFLECTIVE_MEASURE != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_REFLECTIVE_MEASURE == decoded_msg) {
			p_decode_info->diag_info.drv_reflective_measure =
					p_msg->p_buffer[diag_res_param1];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_internal_self_test_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_INTERNAL_SELF_TEST;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_INTERNAL_SELF_TEST;
	}

	if (MSG_LEN_INTERNAL_SELF_TEST != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_INTERNAL_SELF_TEST == decoded_msg) {
			p_decode_info->diag_info.drv_internal_self_test =
					p_msg->p_buffer[diag_res_param1];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_external_sdram_test_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_EXTERNAL_SDRAM_TEST;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_EXTERNAL_SDRAM_TEST;
	}

	if (MSG_LEN_EXTERNAL_SDRAM_TEST != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_EXTERNAL_SDRAM_TEST == decoded_msg) {
			p_decode_info->diag_info.drv_external_sdram_test =
					p_msg->p_buffer[diag_res_param1];
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_init_laser_current_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	unsigned short current_consum_hex = 0; /* in hexa-decimal number */
	unsigned short laser_current = 0;

	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_INIT_LASER_CURRENT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_INIT_LASER_CURRENT;
	}

	if (MSG_LEN_INIT_LASER_CURRENT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_INIT_LASER_CURRENT == decoded_msg) {
			*p_decode_info->diag_info.drv_init_laser_current
					= p_msg->p_buffer[diag_res_param1];
			current_consum_hex = (unsigned short)
					((p_msg->p_buffer[diag_res_param2]
							  << BYTE_2_START) |
					(p_msg->p_buffer[diag_res_param3]));
			laser_current = GET_CURRENT_CONSUM(current_consum_hex);
			p_decode_info->diag_info.drv_init_laser_current[1] =
				(unsigned char)((laser_current >> BYTE_2_START)
								& BYTE_MASK);
			p_decode_info->diag_info.drv_init_laser_current[2] =
				(unsigned char)((laser_current) & BYTE_MASK);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_number_read_errs_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_NUM_READ_ERRS;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_NUM_READ_ERRS;
	}

	if (MSG_LEN_NUM_READ_ERRS != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_NUM_READ_ERRS == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_num_read_errs,
				&p_msg->p_buffer[diag_res_param1],
				MAX_NO_READ_ERRS_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_date_product_install_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_DATE_PRODUCTION_INSTALL_READ;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_DATE_PRODUCTION_READ;
	}

	if (MSG_LEN_DATE_PRODUCT_INSTALL != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_DATE_PRODUCTION_READ == decoded_msg) {
			memcpy(p_decode_info->diag_info
				.drv_date_product_install_read,
				&p_msg->p_buffer[diag_res_param1],
				MAX_DATE_PROD_INSTALL_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_serial_number_read_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_SERIAL_NO_READ;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_SERIAL_NO_READ;
	}

	if (MSG_LEN_SERIAL_NO != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_SERIAL_NO_READ == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_serial_no_read,
				&p_msg->p_buffer[diag_res_param1],
				MAX_SERIAL_NO_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_msg_err_history_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_STATUS_MSG_ERR_HISTROY;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_STATUS_MSG_ERR_HISTROY;
	}

	if (MSG_LEN_MSG_ERR_HISTORY != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_STATUS_MSG_ERR_HISTROY == decoded_msg) {
			memcpy(p_decode_info->diag_info
					.drv_status_msg_err_histroy,
				&p_msg->p_buffer[diag_res_param1],
				MAX_MSG_ERR_HIS_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_msg_last_errs_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_STATUS_MSG_LAST_ERRS;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_STATUS_MSG_LAST_ERRS;
	}

	if (MSG_LEN_MSG_LAST_ERRS != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_STATUS_MSG_LAST_ERRS == decoded_msg) {
			memcpy(p_decode_info->diag_info
					.drv_status_msg_last_errs,
				&p_msg->p_buffer[diag_res_param1],
				MAX_MSG_LAST_ERRS_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_used_cd_type_cnt_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_USED_CD_TYPE_COUNT;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_USED_CD_TYPE_COUNT;
	}

	if (MSG_LEN_USED_CD_TYPE_CNT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_USED_CD_TYPE_COUNT == decoded_msg) {
			memcpy(p_decode_info->diag_info.drv_used_cd_type_cnt,
				&p_msg->p_buffer[diag_res_param1],
				MAX_CD_TYPE_CNT_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}

static enum masca_error diag_user_profil_used_cds_decode_func(
			struct masca_intrptr_msg_info_list * const p_msg_list ,
			union masca_intrptr_decode_info * const p_decode_info ,
			const struct masca_intrptr_msg_info * const p_msg ,
			const enum masca_intrptr_decode_msg decoded_msg)
{
	enum masca_error error = MASCA_PTR_NULL;
	/*NULL checks for p_msg is carried out ,  outside this function*/
	if (NULL != p_msg_list) {
		error = MASCA_OK;
		p_msg_list->msg_group = MSG_USER_PROFILE_USED_CDS;
		p_msg_list->num_of_msgs = 1;
		*p_msg_list->decoded_messages = DIAG_USER_PROFILE_USED_CDS;
	}

	if (MSG_LEN_USED_CD_TYPE_CNT != p_msg->msg_length) {
		error = MASCA_UNKNOWN_MSG_CMD;
	} else if (NULL != p_decode_info) {
		error = MASCA_OK;
		if (DIAG_USER_PROFILE_USED_CDS == decoded_msg) {
			memcpy(p_decode_info->diag_info.
					drv_status_msg_last_errs,
				&p_msg->p_buffer[diag_res_param1],
				MAX_PROFIL_USED_CD_SIZE);
		} else {
			error = MASCA_UNKNOWN_MSG_CMD;
		}
	}

	return error;
}
/**
* \func convert_to_centigrade
*
* converts the ADC count value to degree centigrade
* by using the look up table "temperature_table". It performs
* binary search over this table to get the temperature.
*
* \param adc_count adc count value received on masca message
*
* \return
*  short temperature: temperature in degree centigrade
*
* \see MASCA drive design specification
*  P56-078-CR-A02_MASCA_DesignSpecification_CR-RB-2011-010.doc
*  MASCA specification 3.2.1
*/
static short convert_to_centigrade(const unsigned short adc_count)
{
	/*now we will perform binary search (slight variation) over the
	temperature table to find out the temperature in centigrade
	*/
	short temperature = 0;
	unsigned short min = 0;
	unsigned short max = MAX_TEMPERATURE_VALUES - 1;
	unsigned short mid = 0;
	bool is_element_found = true;

	if (adc_count < temperature_table[min])
		temperature = (short)((short)min + MIN_TEMPERATURE_VALUE);
	else if (adc_count > temperature_table[max])
		temperature = (short)((short)max + MIN_TEMPERATURE_VALUE);
	else
		is_element_found = false;

	while (false == is_element_found) {
		is_element_found = true;
		if (temperature_table[min] == adc_count) {
			temperature = (short)((short)min +
							MIN_TEMPERATURE_VALUE);
		} else if ((temperature_table[max] == adc_count) ||
				(max == (min + 1))) {
			/*observe this condition ,  if the value lies
			between two consecutive values of the temperature
			table then the index correponding to greater value
			is considered
			*/
			temperature = (short)((short)max +
							MIN_TEMPERATURE_VALUE);
		} else {
			is_element_found = false;
			mid = (unsigned short)((unsigned short)
							(max + min) >> 1);
			if (adc_count >= temperature_table[mid])
				min = mid;
			else
				max = mid;
		}
	}
	return temperature;
}

