/*
 * MASCA Helper Functions File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"

struct masca_rd_val {
	unsigned char *name;
	int *p_val;
	int max_val;
	int lower_limit;
	int upper_limit;
};

struct masca_rd_str {
	unsigned char *str_nm;
	unsigned char *p_buf;
	int max_char;
};

struct masca_alm_unit {
	unsigned int alm_time;
	void (*timer_cbk)(void);
};

static DEFINE_SPINLOCK(timer_lock);

#define DFT_MASCA_MAX_SIMULTANEOUS_REQ  10
#define DFT_MASCA_LIFETIME_TMOUT	20000
/*AUTOREINSERT TIMEOUT = 0 ; Configurable through Sysfs*/
#define DFT_MASCA_AUTOREINSERT_TMOUT	0
#define DFT_MASCA_DIAG_TMOUT		2000
#define DFT_MASCA_DIAG_RTY_CNT		2
#define DFT_MASCA_DIAG_RST_CNT		1
#define DFT_MASCA_DRVCTRL_TMOUT		2000
#define DFT_MASCA_DRVCTRL_RTY_CNT	2
#define DFT_MASCA_DRVCTRL_RST_CNT	1
#define DFT_MASCA_EJECT_TMOUT		4000
#define DFT_MASCA_EJECT_RTY_CNT		4
#define DFT_MASCA_EJECT_RST_CNT		4
#define DFT_MASCA_TOCTEXT_TMOUT		30000
#define DFT_MASCA_TOCTEXT_RTY_CNT	4
#define DFT_MASCA_TOCTEXT_RST_CNT	1
#define DFT_MASCA_BLKRD_WAIT_TMOUT	10000
#define DFT_MASCA_BLKRD_RTY_CNT		3
#define TIMER_BASE_PERIOD		50/*50ms*/

static unsigned int timer_count;
static struct timer_list masca_timer;


static struct masca_alm_unit alm_table[MASCA_MAX_ALMS] = {
	{0, masca_diag_alm_cbk},
	{0, masca_drvctrl_alm_cbk},
	{0, masca_toctexthash_alm_cbk},
	{0, masca_eject_alm_cbk},
	{0, masca_auto_stop_alm_hndlr},
	{0, masca_auto_reinsrt_alm_hndlr},
	{0, masca_mngr_sm_cbk},
	{0, masca_blkrd_alm_cbk}
};


static void masca_fire_alarm(void);

void masca_timer_handler(unsigned long data)
{
	masca_fire_alarm();
	mod_timer(&masca_timer, jiffies+msecs_to_jiffies(50));
}


void masca_helper_init(void)
{
	/* timer handler */
	init_timer(&masca_timer);
	setup_timer(&masca_timer, &masca_timer_handler, 0);
	/*50 ms timer fired*/
	mod_timer(&masca_timer, jiffies+msecs_to_jiffies(50));
}

void masca_helper_deinit(void)
{
	/* delete timer */
	del_timer(&masca_timer);
}


void masca_read_configuration(struct masca_configuration * const p_config)
{
	p_config->pllel_maxreq                  =
						DFT_MASCA_MAX_SIMULTANEOUS_REQ;
	p_config->life_time_tmout               = DFT_MASCA_LIFETIME_TMOUT;
	p_config->auto_reinsrt_tmout            = DFT_MASCA_AUTOREINSERT_TMOUT;
	p_config->diag_tmout                    = DFT_MASCA_DIAG_TMOUT;
	p_config->diag_rty_cnt                  = DFT_MASCA_DIAG_RTY_CNT;
	p_config->diag_rst_cnt                  = DFT_MASCA_DIAG_RST_CNT;
	p_config->drvctrl_tmout                 = DFT_MASCA_DRVCTRL_TMOUT;
	p_config->drvctrl_rty_cnt               = DFT_MASCA_DRVCTRL_RTY_CNT;
	p_config->drvctrl_rst_cnt               = DFT_MASCA_DRVCTRL_RST_CNT;
	p_config->eject_tmout                   = DFT_MASCA_EJECT_TMOUT;
	p_config->eject_rty_cnt                 = DFT_MASCA_EJECT_RTY_CNT;
	p_config->eject_rst_cnt                 = DFT_MASCA_EJECT_RST_CNT;
	p_config->toctext_tmout                 = DFT_MASCA_TOCTEXT_TMOUT;
	p_config->toctext_rty_cnt               = DFT_MASCA_TOCTEXT_RTY_CNT;
	p_config->toctext_rst_cnt               = DFT_MASCA_TOCTEXT_RST_CNT;
	p_config->blkrd_tmout                   = DFT_MASCA_BLKRD_WAIT_TMOUT;
	p_config->blkrd_rty_cnt                 = DFT_MASCA_BLKRD_RTY_CNT;
}

void masca_get_msf(
			const unsigned int lba,
			unsigned char * const p_min,
			unsigned char * const p_sec,
			unsigned char * const p_frame
			)
{
	if ((NULL != p_min) && (NULL != p_sec) && (NULL != p_frame)) {
		*p_frame = lba % FRAMES_PER_SEC;
		*p_min = lba / FRAMES_PER_MIN;
		*p_sec = (lba % FRAMES_PER_MIN)/FRAMES_PER_SEC;
	}
}

static void masca_fire_alarm(void)
{
	unsigned long flags;
	enum masca_alm_id count;

	spin_lock_irqsave(&timer_lock, flags);
	timer_count++;
	if (0 == timer_count)
		timer_count = 1;

	for (count = ALM_DIAG; count < MASCA_MAX_ALMS; count++) {
		if (alm_table[count].alm_time == timer_count) {
			alm_table[count].alm_time = 0;
			if (NULL != alm_table[count].timer_cbk)
				alm_table[count].timer_cbk();
		}
	}
	spin_unlock_irqrestore(&timer_lock, flags);
}

void masca_util_sta_alm(const unsigned int alm_interval,
					const enum masca_alm_id alm_id)
{
	unsigned int remainder;
	unsigned int count_increment;
	unsigned long flags;

	if (alm_id < MASCA_MAX_ALMS) {
		if (0 == alm_interval) {
			/*passing 0 tmout is equivalent to timeout forever*/
			alm_table[alm_id].alm_time = 0;
		} else {
			remainder = alm_interval % TIMER_BASE_PERIOD;
			count_increment = (alm_interval / TIMER_BASE_PERIOD);
			if (0 != remainder)
				count_increment++;

			spin_lock_irqsave(&timer_lock, flags);
			count_increment = timer_count + count_increment;
			if (0 == count_increment)
				count_increment = 1;

			alm_table[alm_id].alm_time = count_increment;
			spin_unlock_irqrestore(&timer_lock, flags);
		}
	}
}


void masca_util_stp_alm(const enum masca_alm_id alm_id)
{
	if (alm_id < MASCA_MAX_ALMS)
		alm_table[alm_id].alm_time = 0;
}
