/*
 * MASCA Hardware Abstraction Layer File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/spi/spi.h>
#include "masca_common_types.h"
#include "masca_scsi_handler.h"
#include "masca_event_config.h"
#include "masca_hal.h"

#include "masca_spidev.h"

/* Issues a pulse of >1ms duration on CD_RESET */
#define MASCA_RST_PULSE		2
#define BYTE_COUNT_POS		0x00
#define CHECKSUM_POS		0x01
#define DATA_START_POS		0x02
#define CHECKSUM_SUCCESS	0x00
#define MASCA_MAX_CMD_LENGTH	0x0b /* 11 bytes */
#define MSG_HNDLR_HDR_LNG	0x02
#define CS_LOW2TRANSFER_BEGIN	45  /* > 40us (t of cslow->clk)*/
#define CS_LOW2RCV_BEGIN	45  /* > 40us (t of cslow->clk)*/
#define TRANSFER_END2CS_OFF	25  /* > 20us (t of clk->cshigh)*/
#define REQ_OFF2CS_ON_MIN	25  /* >=20us */
#define REQ_OFF2CS_ON_MAX	30
#define CHECK_DELAY_MIN		45  /* CD_SPI_REQ reset
						for priority change*/
#define CHECK_DELAY_MAX		50
/* > 40us Intermediate time delay slave to master is greater than 100us */
#define SLAVE_TO_MASTER_CS_MAX_DELAY_US	110
/* > 20us Intermediate time delay master to slave is greater than 20us */
#define MASTER_TO_SLAVE_CS_MAX_DELAY_US	25
#define MAX_RETRY_CNT			3   /* Retry count for communication
							change priority */

/* Since lstart is not used, count is 3.
 * In future if gpio is requested for lstart line
 * this value should be incremented by 1 */
#define GPIO_COUNT	3

static struct gpio_irq {
	int irq;
	bool is_irq_alloc;
} req_irq;

static irqreturn_t req_interrupt(int irq, void *data)
{
	masca_util_set_event(DRV_MNGR_RCV_MESSAGE);

	return IRQ_HANDLED;
}

int masca_configure_irq(int gpio_num,
			irqreturn_t (*handlr)(int irq, void *data),
			const char *hndlname)
{
	int ret_err = 0;
	req_irq.is_irq_alloc = FALSE;
	req_irq.irq = gpio_to_irq(gpio_num);
	if ((req_irq.irq < 0) || request_irq(req_irq.irq, handlr,
					IRQF_TRIGGER_FALLING, hndlname, NULL))
		return -ENODEV;
	req_irq.is_irq_alloc = TRUE;
	return ret_err;
}

void masca_gpio_deinit(int gpio_num)
{
	gpio_free(gpio_num);
}

int masca_gpio_init(void)
{
	int gpio_return;
	enum masca_error ret_err = MASCA_OK;
	struct gpio_irq {
		int gpio_irq;
		char *name;
		irqreturn_t (*handlr)(int irq, void *data);
	} irq;

	/* Allocation for masca_chipselect_num */
	if (gpio_is_valid(masca_chipselect_num) < 0) {
		HAL_TA_TRACE("Not valid GPIO No.:%d\n", masca_chipselect_num);
		goto masca_gpio_init_error;
	}

	gpio_return = gpio_request(masca_chipselect_num, "MASCA_CS");
	if (gpio_return != 0) {
		HAL_TA_TRACE("MASCA_CS_NUM not allocated:%d\n", gpio_return);
		goto masca_gpio_init_error;
	}

	if (gpio_direction_output(masca_chipselect_num, 1) < 0) {
		HAL_TA_TRACE("Error setting direction of MASCA_CS_NUM\n");
		goto masca_cs_error;
	}

	/* Allocation for masca_reqline_num */
	if (gpio_is_valid(masca_reqline_num) < 0) {
		HAL_TA_TRACE("Not valid GPIO number:%d\n", masca_reqline_num);
		goto masca_cs_error;
	}

	gpio_return = gpio_request(masca_reqline_num, "MASCA_REQLINE");
	if (gpio_return != 0) {
		HAL_TA_TRACE("MASCA_REQ_NUM not allocated:%d\n", gpio_return);
		goto masca_cs_error;
	}

	if (gpio_direction_input(masca_reqline_num) < 0) {
		HAL_TA_TRACE("Error setting direction of masca_reqline_num\n");
		goto masca_req_line_error;
	}

	/* Allocation for masca_reset_num */
	if (gpio_is_valid(masca_reset_num) < 0) {
		HAL_TA_TRACE("Not valid GPIO No.:%d\n", masca_chipselect_num);
		goto masca_req_line_error;
	}

	gpio_return = gpio_request(masca_reset_num, "MASCA_RESET");
	if (gpio_return != 0) {
		HAL_TA_TRACE("MASCA_RST_NUM not allocated:%d\n", gpio_return);
		goto masca_req_line_error;
	}

	if (gpio_direction_output(masca_reset_num, 0) < 0) {
		HAL_TA_TRACE("Error setting direction of MASCA_RST_NUM\n");
		goto masca_rst_line_error;
	}

	/* Configure interrupt */
	irq.gpio_irq = masca_reqline_num;
	irq.handlr = &req_interrupt;
	irq.name = "MASCA_REQLINE";

	ret_err = masca_configure_irq(masca_reqline_num, req_interrupt,
					"MASCA_REQLINE_IRQ");
	if (ret_err < 0) {
		HAL_TA_TRACE(
		"Error allocating interrupt line for MASCA REQ pin\n");
		goto masca_rst_line_error;
		return ret_err;
	}
	return MASCA_OK;

masca_rst_line_error:
	masca_gpio_deinit(masca_reset_num);

masca_req_line_error:
	masca_gpio_deinit(masca_reqline_num);

masca_cs_error:
	masca_gpio_deinit(masca_chipselect_num);

masca_gpio_init_error:
	return -1;
}

void masca_cs_on(void)
{
	gpio_set_value(masca_chipselect_num, cs_low_active);
}

void masca_cs_off(void)
{
	gpio_set_value(masca_chipselect_num, !cs_low_active);
}

bool masca_check_req(void)
{
	int masca_req = 0;
	bool data_present = FALSE;

	masca_req = gpio_get_value(masca_reqline_num);

	if (req_low_active == 0) {
		if (masca_req == 0)
			data_present = TRUE;
	} else { /*req_low_active == 1*/
		if (masca_req > 0)
			data_present = TRUE;
	}

	return data_present;
}

void masca_intr_spireq_on(void)
{
	enable_irq(req_irq.irq);
}

void masca_intr_spireq_off(void)
{
	disable_irq(req_irq.irq);
}

void masca_drv_reset(bool rst_line_on)
{
	if (rst_line_on == TRUE)
		gpio_set_value(masca_reset_num, reset_low_active);
	else {
		mdelay(MASCA_RST_PULSE);
		gpio_set_value(masca_reset_num, !reset_low_active);
	}
}

void masca_hal_rst_pulse(void)
{
	masca_spi_lock_bus();
	/* put drive into reset */
	masca_cs_on();
	masca_intr_spireq_off();
	gpio_set_value(masca_reset_num, reset_low_active);
	mdelay(MASCA_RST_PULSE);

	/* release drive reset */
	gpio_set_value(masca_reset_num, !reset_low_active);
	masca_intr_spireq_on();
	masca_cs_off();

	/*unlock SPI Bus*/
	spi_bus_unlock(ptr->master);
}

enum masca_error masca_hal_rcv(char *rcvbuf)
{
	bool read_req = FALSE;
	short spi_ret = 0;
	short rcv_len = 0;
	enum masca_error ret_err = MASCA_PTR_NULL;

	masca_spi_lock_bus();
	/* check request line is active or not*/
	read_req = masca_check_req();
	if (TRUE == read_req) {
		masca_cs_on();
		udelay(CS_LOW2RCV_BEGIN);
		/* Read first byte of header to find length of a drive
		   message */
		spi_ret = masca_spi_read(&rcvbuf[BYTE_COUNT_POS], 1);
		if ((spi_ret > 0x00) && (rcvbuf[BYTE_COUNT_POS] >= 0x02) &&
					(rcvbuf[BYTE_COUNT_POS] != 0xFF)) {
			/* Update rcv_len for reading a drive message */
			rcv_len = rcvbuf[BYTE_COUNT_POS];
			/* Read remaining data */
			spi_ret = masca_spi_read(&rcvbuf[CHECKSUM_POS],
								rcv_len);
			if (spi_ret == rcv_len) {
				if (0xfd == rcvbuf[DATA_START_POS]) {
					/* Nothing do be done */
				} else {
					HAL_TD_TRACE(
					"msg cnt= %02x, Chksum= %02x id = %02x\n",
					rcvbuf[BYTE_COUNT_POS],
					rcvbuf[CHECKSUM_POS],
					rcvbuf[DATA_START_POS]);
				}
			}
			if (spi_ret > 0)
				ret_err =  MASCA_OK;
			else
				ret_err =  MASCA_IO_ERR;

		} else { /*if(spi_ret > 0)*/
			ret_err = MASCA_IO_ERR;
		}
		udelay(TRANSFER_END2CS_OFF);
		masca_cs_off();
		/* Intermediate time */
		udelay(SLAVE_TO_MASTER_CS_MAX_DELAY_US);
	} else { /*if(TRUE == read_req)*/
		ret_err = MASCA_IO_ERR;
	}
	HAL_TD_TRACE(
	"masca_hal: err=%d,spi_ret=%d,rcv_len=%d\n", ret_err, spi_ret, rcv_len);
	/*unlock SPI Bus*/
	spi_bus_unlock(ptr->master);
	return ret_err;
}

bool masca_priority_change(void)
{
	bool chng_prio = FALSE;
	short retry_cnt = 0;
	chng_prio = masca_check_req();
	while (retry_cnt < MAX_RETRY_CNT) {
		masca_cs_on();
		usleep_range(CHECK_DELAY_MIN, CHECK_DELAY_MAX);
		masca_cs_off();
		usleep_range(REQ_OFF2CS_ON_MIN, REQ_OFF2CS_ON_MAX);
		chng_prio = masca_check_req();
		retry_cnt++;
		usleep_range(400, 500);
	}

	if (chng_prio != TRUE)
		masca_util_clr_event(DRV_MNGR_RCV_MESSAGE);

	return chng_prio;
}


/* Send command through SPI device to the MASCA drive */
enum masca_error masca_hal_snd(char *sndbuf, int len)
{
	bool	req_ret = FALSE; /* Request line value */
	enum masca_error ret_error = MASCA_PTR_NULL;
	int spi_ret = 0;
	masca_spi_lock_bus();

	/* First check of Double check SPI chip select */
	if (masca_check_req() == TRUE) {
		HAL_TD_TRACE("priority change 1 check %s\n", __func__);
		req_ret = masca_priority_change();
	}
	/* masca chip select on */
	masca_cs_on();
	udelay(CS_LOW2TRANSFER_BEGIN);
	/* second check of Double check SPI chip select */
	if (masca_check_req() == TRUE) {
		HAL_TD_TRACE("priority change 2 check %s\n", __func__);
		req_ret = masca_priority_change();
		masca_cs_on();
		udelay(CS_LOW2TRANSFER_BEGIN);
	}
	if (masca_check_req() == FALSE) {
		spi_ret = masca_spi_write(sndbuf, len);
		udelay(TRANSFER_END2CS_OFF);
		masca_cs_off();
		/* Intermediate time */
		udelay(MASTER_TO_SLAVE_CS_MAX_DELAY_US);
		if (spi_ret != len)
			ret_error = MASCA_IO_ERR;
		else
			ret_error = MASCA_OK;

		HAL_TD_TRACE(
		"cmd byte cnt= 0x%02x, Chksum= 0x%02x cmdid= 0x%02x\n",
		sndbuf[BYTE_COUNT_POS], sndbuf[CHECKSUM_POS],
						sndbuf[DATA_START_POS]);
	} else {
		masca_cs_off();
		ret_error = MASCA_IO_ERR;
		HAL_TE_TRACE("command send failed\n");
	}

	/*unlock SPI Bus*/
	spi_bus_unlock(ptr->master);

	return ret_error;
}

enum masca_error masca_prepare_sndbuf(const char * const sndbuf, const int len)
{
	int ret_err = MASCA_OK;
	int byte_cnt = 0;
	char checksum = 0;
	char cmd[MASCA_MAX_CMD_LENGTH];
	char cmdlen = 0;

	if (len > (MASCA_MAX_CMD_LENGTH - MSG_HNDLR_HDR_LNG)) {
		ret_err = MASCA_INVALID_PARAM;
		HAL_TE_TRACE("invalid parameter\n");
	} else {
		for (byte_cnt = 0; byte_cnt < len; byte_cnt++)
			checksum ^= sndbuf[byte_cnt];

		cmd[BYTE_COUNT_POS] = (MSG_HNDLR_HDR_LNG + len) - 1;

		cmd[CHECKSUM_POS] = checksum;
		memcpy(&cmd[DATA_START_POS], sndbuf, len);
		cmdlen = MSG_HNDLR_HDR_LNG + len;
		ret_err = masca_hal_snd(cmd, cmdlen);

		if (ret_err == MASCA_IO_ERR)
			HAL_TI_TRACE("IO error prepare send buf\n");
	}

	return ret_err;
}

enum masca_error masca_prepare_recvbuf(char *rcvbuf, int *len)
{
	enum masca_error ret_err = MASCA_OK;
	char msglen = 0;
	char checksum = 0;
	int byte_cnt = 0;

	ret_err = masca_hal_rcv(rcvbuf);
	if (MASCA_OK == ret_err)  {
		msglen = rcvbuf[BYTE_COUNT_POS];
		for (byte_cnt = CHECKSUM_POS; byte_cnt <= msglen;
							byte_cnt++) {
			checksum ^= rcvbuf[byte_cnt];
		}

		if (CHECKSUM_SUCCESS == checksum) {
			*len = --msglen;
			HAL_TD_TRACE("rcvbuf %d\n", *len);
		} else {
			ret_err = MASCA_CHECKSUM_ERROR;
			HAL_TE_TRACE("checksum failed\n");
		}
	} else {
		ret_err =  MASCA_IO_ERR;
	}
	return ret_err;
}

void masca_deinitialize_hal(void)
{
	int gpio_no[GPIO_COUNT] = {	masca_chipselect_num,
					masca_reqline_num,
					masca_reset_num };
	unsigned char idx = 0;
	/* Free interrupt */
	if (req_irq.is_irq_alloc)
		free_irq(req_irq.irq, NULL);

	/* Free GPIO's */
	while (idx < GPIO_COUNT)
		masca_gpio_deinit(gpio_no[idx++]);
}

int masca_initialize_hal(void)
{
	int err = MASCA_OK;

	err = masca_gpio_init();
	if (MASCA_OK == err) {
		err = masca_spi_open();
		if (MASCA_OK != err)
			masca_deinitialize_hal();
	}

	return err;
}
