/*
 * Generates all Events except Disk end.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */


#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"
#include "masca_blkdev.h"


static void
masca_chk_voltage(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);
static void
masca_chk_temperature(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);

static void
masca_chk_disc_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);
static void masca_chk_disc_load_err_evt(
			const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);

static void
masca_chk_toc_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);

static void
masca_chk_trkend_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);
static void
masca_chk_play_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);

static void
masca_chk_dsk_defect(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info);

static struct masca_last_evt last_evt;

static enum masca_drv_event	evt_slot_temp;
static enum masca_drv_event	evt_slot_disc;
static enum masca_drv_event	evt_disc_load_err;
static enum masca_drv_event	evt_slot_toc;
static enum masca_drv_event	evt_slot_trk_end;
static enum masca_drv_event	evt_slot_play_info;
static enum masca_drv_event	evt_slot_dsk_defect;
static struct masca_drv_play_info	stored_play_pos;

void masca_evt_gen_init(void)
{
	last_evt.disc_event = MESSAGE_MAX;
	last_evt.disc_load_err_event = MESSAGE_MAX;
	last_evt.temp_event = MESSAGE_MAX;
	last_evt.volt_event = MESSAGE_MAX;
	last_evt.toc_event = MESSAGE_MAX;
	last_evt.trk_end_event = MESSAGE_MAX;
	last_evt.dsk_defect_evt = MESSAGE_MAX;
	evt_slot_temp = DRV_NO_EVENT;
	evt_slot_disc = DRV_NO_EVENT;
	evt_disc_load_err = DRV_NO_EVENT;
	evt_slot_toc = DRV_NO_EVENT;
	evt_slot_trk_end = DRV_NO_EVENT;
	evt_slot_play_info = DRV_NO_EVENT;
	evt_slot_dsk_defect = DRV_NO_EVENT;
	stored_play_pos.abs_min = 0;
	stored_play_pos.abs_sec = 0;
	stored_play_pos.rel_min = 0;
	stored_play_pos.rel_sec = 0;
	stored_play_pos.trk_playing = 0;
	stored_play_pos.index = 0;
}

void masca_handle_event_q(const unsigned int q_evt_ptn,
				struct masca_drv_event_info * const p_evt_info)
{
	if (EVENT_Q_UP == q_evt_ptn) {
		if (DRV_NO_EVENT != evt_slot_temp) {
			p_evt_info->drv_event = evt_slot_temp;
			evt_slot_temp = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_slot_disc) {
			p_evt_info->drv_event = evt_slot_disc;
			evt_slot_disc = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_disc_load_err) {
			p_evt_info->drv_event = evt_disc_load_err;
			evt_disc_load_err = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_slot_trk_end) {
			p_evt_info->drv_event = evt_slot_trk_end;
			evt_slot_trk_end = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_slot_toc) {
			p_evt_info->drv_event = evt_slot_toc;
			evt_slot_toc = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_slot_play_info) {
			p_evt_info->drv_event = evt_slot_play_info;
			memcpy(&(p_evt_info->play_position),
			&(stored_play_pos), sizeof(struct masca_drv_play_info));
			evt_slot_play_info = DRV_NO_EVENT;
		} else if (DRV_NO_EVENT != evt_slot_dsk_defect) {
			p_evt_info->drv_event = evt_slot_dsk_defect;
			evt_slot_dsk_defect = DRV_NO_EVENT;
		}

		if ((DRV_NO_EVENT != evt_slot_temp)
			|| (DRV_NO_EVENT != evt_slot_disc)
			|| (DRV_NO_EVENT != evt_disc_load_err)
			|| (DRV_NO_EVENT != evt_slot_trk_end)
			|| (DRV_NO_EVENT != evt_slot_toc)
			|| (DRV_NO_EVENT != evt_slot_play_info))
			/*Set the util event again till all events get over*/
			masca_util_set_event(EVENT_Q_UP);
	}
}

void
masca_chk_for_events(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	/*The order of processing is important here as we can return
	* only one event at a time.*/

	if (MSG_STATUS == p_list->msg_group) {
		masca_chk_voltage(p_list, p_msg, p_evt_info);
		masca_chk_temperature(p_list, p_msg, p_evt_info);
		masca_chk_toc_evt(p_list, p_msg, p_evt_info);
		masca_chk_disc_evt(p_list, p_msg, p_evt_info);
		masca_chk_disc_load_err_evt(p_list, p_msg, p_evt_info);
		masca_chk_trkend_evt(p_list, p_msg, p_evt_info);
		masca_chk_dsk_defect(p_list, p_msg, p_evt_info);
	}
	if (MSG_PLAY_POSITION == p_list->msg_group)
		masca_chk_play_evt(p_list, p_msg, p_evt_info);
}

void masca_get_last_event(struct masca_last_evt *const p_lst_evt)
{
	memcpy(p_lst_evt, &last_evt, sizeof(struct masca_last_evt));
}

void masca_get_cur_pos(struct masca_drv_play_info **const p_position)
{
	*p_position = &stored_play_pos;
}


static void
masca_chk_voltage(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = DRIVE_POWER_OK;
	search_msg.search_msg[1] = DRIVE_POWER_FAIL;
	p_evt_info->drv_event = DRV_NO_EVENT;
	/*voltage state*/
	if ((masca_is_msg_prsnt(&search_msg, p_list,
			&found_msg, MASCA_RANGE)) &&
			(last_evt.volt_event != found_msg)) {
		last_evt.volt_event = found_msg;
		/*We have a change in power state, so inform application*/
		if (DRIVE_POWER_OK == found_msg)
			p_evt_info->drv_event = DRV_NORMAL_VOLT;
		else
			p_evt_info->drv_event = DRV_UNDER_VOLT;
	}
}

static void masca_chk_temperature(
			const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;
	enum masca_drv_event             output_event;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = TEMPERATURE_OK;
	search_msg.search_msg[1] = TEMPERATURE_LIMIT_REACHED;

	/*temperature state*/
	if ((masca_is_msg_prsnt(&search_msg, p_list,
			&found_msg, MASCA_RANGE)) &&
			(last_evt.temp_event != found_msg)) {
		last_evt.temp_event = found_msg;
		/*We have a change in temperature state, so inform application*/
		if (TEMPERATURE_OK == found_msg)
			output_event = DRV_NORMAL_TEMP;
		else
			output_event = DRV_OVER_TEMP;

		if (DRV_NO_EVENT == p_evt_info->drv_event) {
			evt_slot_temp = DRV_NO_EVENT;
			p_evt_info->drv_event = output_event;
		} else {
			/*queue up the event and use util_set_event*/
			evt_slot_temp = output_event;
			masca_util_set_event(EVENT_Q_UP);
		}
	}
}

static void masca_chk_disc_load_err_evt(
			const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_drv_event             output_event;

	if ((masca_chk_msg(p_list, DISC_LOAD_ERROR)) &&
		(last_evt.disc_load_err_event != DISC_LOAD_ERROR)) {
		/*We have a change in disc load error state, so inform
		application*/
		last_evt.disc_load_err_event = DISC_LOAD_ERROR;
		output_event = DRV_MECHANIC_LOAD_ERR;

		if (DRV_NO_EVENT == p_evt_info->drv_event) {
			p_evt_info->drv_event = output_event;
			evt_disc_load_err = DRV_NO_EVENT;
		} else {
			/*queue up the event*/
			evt_disc_load_err = output_event;
			masca_util_set_event(EVENT_Q_UP);
		}
	} else {
		last_evt.disc_load_err_event = MESSAGE_MAX;
	}
}

static void
masca_chk_disc_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;
	enum masca_drv_event             output_event;
	struct masca_intrptr_cmd_info      cmd_to_drive;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = DISC_ABSENT;
	search_msg.search_msg[1] = DISC_IN_SLOT;

	if ((masca_is_msg_prsnt(&search_msg, p_list,
		&found_msg, MASCA_RANGE)) &&
		(last_evt.disc_event != found_msg)) {
		/*We have a change in disc state, so inform application*/
		last_evt.disc_event = found_msg;
		output_event = DRV_NO_EVENT;
		switch (found_msg) {
		case DISC_ABSENT:
			output_event = DRV_DISC_ABSENT;
			/*set to single session mode,
			This is very essential as copy protected media can
			crash the firmware of the drive*/
			/*Turn on the multi session play back*/
			cmd_to_drive.command = CMD_MULTISESSION_OFF;
			masca_tx_cmd(&cmd_to_drive);
			/*medium not present*/
			masca_update_global_sense_buffer(0x02, 0x3A, 0x00);
			break;
		case DISC_INSERTING:
			output_event = DRV_DISC_INSERTING;
			/*Logical Unit not ready, Operation in Progress*/
			masca_update_global_sense_buffer(0x02, 0x04, 0x07);
			break;
		case DISC_INSERTED:
			output_event = DRV_DISC_INSERTED;
			break;
		case DISC_EJECTING:
			output_event = DRV_DISC_EJECTING;
			/*medium not present*/
			masca_update_global_sense_buffer(0x02, 0x3A, 0x00);
			break;
		case DISC_IN_SLOT:
			output_event = DRV_DISC_IN_SLOT;
			break;
		case DISC_STOPPED:
			output_event = DRV_DISC_STOPPED;
			break;
		case DISC_PLAYING:
			output_event = DRV_DISC_PLAYING;
			/*Audio Play Operation in Progress*/
			masca_update_global_sense_buffer(0x05, 0x00, 0x11);
			break;
		case DISC_SEARCHING:
			output_event = DRV_DISC_PLAYING;
			break;
		case DISC_PAUSED:
			output_event = DRV_DISC_PAUSED;
			break;
		default:
			break;
		}
		if (DRV_NO_EVENT == p_evt_info->drv_event) {
			p_evt_info->drv_event = output_event;
			evt_slot_disc = DRV_NO_EVENT;
		} else {
			/*queue up the event*/
			evt_slot_disc = output_event;
			masca_util_set_event(EVENT_Q_UP);
		}
	}
}

static void
masca_chk_toc_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;
	enum masca_drv_event             output_event;
	struct masca_intrptr_cmd_info      cmd_to_drive;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = TOC_READING_IN_PROGRESS;
	search_msg.search_msg[1] = TOC_UNREADABLE;

	if ((DISC_INSERTING < last_evt.disc_event)
			&& (DISC_EJECTING > last_evt.disc_event)) {
		if ((masca_is_msg_prsnt(&search_msg, p_list,
					&found_msg, MASCA_RANGE)) &&
					(last_evt.toc_event != found_msg)) {
			last_evt.toc_event = found_msg;
			output_event = DRV_NO_EVENT;
			if (TOC_READY == found_msg) {
				/*Now check if it is a pure CDROM and then only
				enable the multisession if the disc is a
				multisession disc*/
				output_event = DRV_DISC_TOC_READY;
				search_msg.num_messages = 1;
				search_msg.search_msg[0] = TRACKS_CD_ROM_ONLY;
				search_msg.search_msg[1] = MESSAGE_MAX;

				if (masca_is_msg_prsnt(&search_msg,
						p_list, &found_msg, MASCA_OR)) {
					search_msg.num_messages = 2;
					search_msg.search_msg[0] =
						MULTISESSION_PLAYBACK_DISABLED;
					search_msg.search_msg[1] =
						DISC_MULTISESSION;

					if (masca_is_msg_prsnt(&search_msg,
							p_list, &found_msg,
							MASCA_AND)) {
						cmd_to_drive.command =
							CMD_MULTISESSION_ON;
						masca_tx_cmd(&cmd_to_drive);
						output_event = DRV_NO_EVENT;
					}
				}

			} else if (TOC_UNREADABLE == found_msg) {
				output_event = DRV_DISC_TOC_ERROR;
				masca_update_global_sense_buffer(0x03, 0x57,
								0x00);
			} else {
				output_event = DRV_DISC_READING_TOC;
			}

			if (DRV_NO_EVENT != output_event) {
				if (DRV_NO_EVENT == p_evt_info->drv_event) {
					p_evt_info->drv_event = output_event;
					evt_slot_toc = DRV_NO_EVENT;
				} else {
					/*queue up the event*/
					evt_slot_toc = output_event;
					masca_util_set_event(EVENT_Q_UP);
				}
			}
		}
	} else {
		last_evt.toc_event = MESSAGE_MAX;
	}
}

static void masca_chk_trkend_evt(
			const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;
	enum masca_drv_event             output_event;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = TRACK_END_NOT_REACHED;
	search_msg.search_msg[1] = TRACK_END_REACHED;

	if ((DISC_PLAYING <= last_evt.disc_event)
		&& (DISC_PAUSED >= last_evt.disc_event)) {
		if ((masca_is_msg_prsnt(&search_msg, p_list,
				&found_msg, MASCA_RANGE)) &&
				(last_evt.trk_end_event != found_msg)) {
			last_evt.trk_end_event = found_msg;
			output_event = DRV_NO_EVENT;
			if (TRACK_END_REACHED == found_msg)
				output_event = DRV_TRACK_END;

			if (DRV_NO_EVENT == p_evt_info->drv_event) {
				p_evt_info->drv_event = output_event;
				evt_slot_trk_end = DRV_NO_EVENT;
			} else {
				/*queue up the event*/
				evt_slot_trk_end = output_event;
				masca_util_set_event(EVENT_Q_UP);
			}
		}
	} else {
		last_evt.trk_end_event = MESSAGE_MAX;
	}
}

static void masca_chk_dsk_defect(
			const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	enum masca_intrptr_decode_msg    found_msg;
	struct masca_msg_search            search_msg;
	enum masca_drv_event             output_event;

	search_msg.num_messages = 2;
	search_msg.search_msg[0] = DISC_DEFECTIVE;
	search_msg.search_msg[1] = DISC_NOT_DEFECTIVE;

	if ((masca_is_msg_prsnt(&search_msg, p_list,
			&found_msg, MASCA_RANGE)) &&
			(last_evt.dsk_defect_evt != found_msg)) {
		last_evt.dsk_defect_evt = found_msg;
		output_event = DRV_NO_EVENT;
		if (DISC_DEFECTIVE == found_msg)
			output_event = DRV_DISC_DEFECTIVE;

		if (DRV_NO_EVENT == p_evt_info->drv_event) {
			p_evt_info->drv_event = output_event;
			evt_slot_dsk_defect = DRV_NO_EVENT;
		} else {
			/*queue up the event*/
			evt_slot_dsk_defect = output_event;
			masca_util_set_event(EVENT_Q_UP);
		}
	}
}

static void
masca_chk_play_evt(const struct masca_intrptr_msg_info_list * const p_list,
			const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_event_info * const p_evt_info)
{
	union masca_intrptr_decode_info   msg_detail;

	if (masca_chk_msg(p_list, PLAY_INFO) &&
		((DISC_PLAYING == last_evt.disc_event)
		|| (DISC_SEARCHING == last_evt.disc_event))) {
		(void)masca_intrptr_decode_msg_info(PLAY_INFO,
							&msg_detail, p_msg);
		stored_play_pos.abs_min = msg_detail.play_pos.min_absolute;
		stored_play_pos.abs_sec = msg_detail.play_pos.sec_absolute;
		stored_play_pos.rel_min = msg_detail.play_pos.min_relative;
		stored_play_pos.rel_sec = msg_detail.play_pos.sec_relative;
		stored_play_pos.trk_playing = msg_detail.play_pos.track_num;
		stored_play_pos.index = msg_detail.play_pos.track_subindex;

		if (DRV_NO_EVENT == p_evt_info->drv_event) {
			p_evt_info->drv_event = DRV_DISC_PLAY_INFO;
			memcpy(&(p_evt_info->play_position),
			       &(stored_play_pos),
			       sizeof(struct masca_drv_play_info));
			evt_slot_play_info = DRV_NO_EVENT;
		} else {
			/*queue up the event*/
			evt_slot_play_info = DRV_DISC_PLAY_INFO;
			masca_util_set_event(EVENT_Q_UP);
		}
	}
}
