/*
 * Handles all drive control commands except eject.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"
#include "masca_sm.h"
#include "masca_blkdev.h"

enum masca_play_state {
	PLAY_STOPPED,
	PLAY_STARTING,
	PLAY_ONGOING,
	PLAY_PAUSED_STARTING,
	PLAY_PAUSED_ONGOING
};

enum masca_scan_state {
	SCAN_STOPPED,
	SCAN_FORWARD,
	SCAN_BACKWARD,
	SCAN_FAST_FORWARD,
	SCAN_FAST_BACKWARD
};

struct masca_cd_info {
	unsigned int start_addr;
	unsigned int end_addr;
	unsigned char  frst_trk_num;
	unsigned char  lst_trk_num;
};

/*LBA equivalent of MSF 0xFFFFFF*/
#define MSF_CURRENT_POS		0x11CE20
#define DO_NOT_PLAY		0xFFFFFFFF
#define OPTIC_POS_UNKNOWN	0xFFFFFFFF
/*duration in seconds the drive is allowed to play outside the play range,
* if it plays beyond this limit, the play is considered to be complete*/
#define OUTOFBOUND_LIMIT_TIME	2
#define LBA_CURRENT_POS		0xFFFFFFFF

/* For Mixed mode CD, Maximum time out value to switch CDDA */
#define MIX_MODE_CDDA_TIME_OUT 10000

/* Invalid play track micro */
#define INVAILD_PLAY_TRK 0xFF

static void
masca_drvctrl_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info);

static void
masca_exec_drvctrl(struct masca_drv_response_info * const p_response_info);

static enum masca_error
masca_handle_stop_cmd(unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd);

static enum masca_error
masca_handle_pause_cmd(enum masca_intrptr_cmd * const p_nxt_cmd);

static enum masca_error
masca_handle_resume_cmd(unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd);

static enum masca_error
masca_handle_play_cmd(const unsigned int lba_start,
				const unsigned int lba_end,
				unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd);

static enum masca_error
masca_handle_scan_cmd(enum masca_cmd app_cmd,
				const unsigned int lba_start,
				unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd);

static void masca_copy_current_params(void);
static void masca_apply_transition(void);
static bool masca_get_trk_lba(const unsigned char trk_num,
					unsigned int * const p_lba_start);
static void masca_updt_cd_info(void);
static enum masca_intrptr_cmd masca_updt_current_pos(const unsigned int
					optic_head_pos,
					const unsigned int optic_relative_pos,
					const unsigned char track_no);
static unsigned char masca_get_trk_num(const unsigned int lba);
static enum masca_intrptr_cmd masca_updt_trk_end_evt(void);
static bool masca_is_lba_valid(unsigned int * const p_lba_start,
					unsigned int * const p_lba_end);
static struct masca_cmd_slot       drvctrl;

struct masca_play_ctrl {
	enum masca_play_state     play_state;
	enum masca_scan_state     scan_state;
	enum masca_play_state     new_play_state;
	enum masca_scan_state     new_scan_state;
	unsigned int                  play_start_addr;
	unsigned int                  play_end_addr;
	unsigned int                  current_pos;
	unsigned int                  new_play_start_addr;
	unsigned int                  new_play_end_addr;
	unsigned int                  new_current_pos;
	unsigned int                  play_excd_duration;
	unsigned int                  new_play_excd_duration;
	unsigned char                   track_playing;
	unsigned char                   new_track_playing;
	unsigned char			last_trk_no_req;
};

static struct masca_play_ctrl      play_ctrl;
static struct masca_cd_info        cd_info;
static enum masca_intrptr_cmd    last_cmd_to_drive;

void masca_init_drvctrl(const unsigned int retry_cnt, const unsigned int
					reset_cnt,
					const unsigned int response_tmout)
{
	drvctrl.slot_stat               = SLOT_PAUSED;
	drvctrl.app_cmd_info.command    = MAX_COMMAND;
	drvctrl.slot_tmout              = response_tmout;
	drvctrl.hw_response             = MASCA_OK;
	cd_info.end_addr                = 0;
	cd_info.start_addr              = 0;
	cd_info.frst_trk_num            = 0;
	cd_info.lst_trk_num             = 0;
	play_ctrl.play_start_addr       = 0;
	play_ctrl.play_end_addr         = 0;
	play_ctrl.track_playing         = 0;
	play_ctrl.current_pos           = OPTIC_POS_UNKNOWN;
	play_ctrl.play_state            = PLAY_STOPPED;
	play_ctrl.scan_state            = SCAN_STOPPED;
	play_ctrl.play_excd_duration    = 0;
	masca_copy_current_params();
	masca_init_tx_rx(SLOT_DRV_CTRL, retry_cnt, reset_cnt);
}

enum masca_cmd masca_get_drvctrl_cmd(void)
{
	return drvctrl.app_cmd_info.command;
}

void masca_drvctrl_alm_cbk(void)
{
	masca_util_set_event(SLOT_DRV_CTRL_TMOUT);
}


enum masca_error
masca_tx_drvctrl(const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	bool		exec_slot = FALSE;
	enum masca_error	ret_err;

	ret_err = masca_fill_slot(&drvctrl, p_cmd_info, &exec_slot);
	if ((MASCA_OK == ret_err) && (TRUE == exec_slot)) {
		/*reload the retry counters for the slot*/
		masca_reload_rty_cnt(SLOT_DRV_CTRL);
		masca_exec_drvctrl(p_response_info);
	}
	return ret_err;
}

void masca_rx_drvctrl(
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_response_info * const p_response_info)
{
	struct masca_drv_play_info *p_position;
	unsigned int                 optic_head_pos;
	unsigned int			optic_relative_pos;
	enum masca_error         ret_err;
	enum masca_intrptr_cmd   cmd_to_drive = CMD_MAX;
	struct masca_last_evt      last_evt;
	if (SLOT_PAUSED != drvctrl.slot_stat) {
		/* Finding out a response is received or not is a generic
		* functionality. Hence it is implemented in a common place*/
		ret_err = masca_rcv_response(SLOT_DRV_CTRL, p_list, p_rjt_cmds);
		if ((NULL != p_response_info) && (MASCA_OK != ret_err)) {
			drvctrl.hw_response = ret_err;
			/*Now we need to provide a reply back*/
			if (MAX_COMMAND == p_response_info->drv_cmd_response) {
				masca_drvctrl_response(p_msg, p_response_info);
			} else {
				/*We got response to the slot, so clear the
				  timeout*/
				masca_util_stp_alm(ALM_DRVCTRL);
				masca_util_clr_event(SLOT_DRV_CTRL_TMOUT);
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
			}
		}
		if (NULL != p_list) {
			if (MSG_PLAY_POSITION == p_list->msg_group) {
				/*we received optic head pos from the hardware*/
				masca_get_cur_pos(&p_position);
				if (NULL != p_position) {
					optic_head_pos =
						GET_LBA(p_position->abs_min,
							p_position->abs_sec, 0);
					optic_relative_pos =
						GET_LBA(p_position->rel_min,
							p_position->rel_sec, 0);
					cmd_to_drive =
					masca_updt_current_pos(optic_head_pos,
						optic_relative_pos,
						p_position->trk_playing);
				}
			} else if (MSG_STATUS == p_list->msg_group) {
				last_evt.trk_end_event = TRACK_END_NOT_REACHED;
				masca_get_last_event(&last_evt);
				if (TRACK_END_REACHED == last_evt.trk_end_event)
					/*We reached the end of the disc*/
					cmd_to_drive = masca_updt_trk_end_evt();
			}
		}
		if (CMD_MAX > cmd_to_drive) {
			/*We can only have a pause command over here*/
			masca_reload_rty_cnt(SLOT_DRV_CTRL);
			masca_send_cmd(SLOT_DRV_CTRL, cmd_to_drive, 0, 0, 0);
			masca_util_sta_alm(drvctrl.slot_tmout, ALM_DRVCTRL);
		}
	}
}

void masca_pause_drvctrl(void)
{
	drvctrl.slot_stat = SLOT_PAUSED;
	/*we are pausing so don't allow timeouts*/
	if (MAX_COMMAND > drvctrl.app_cmd_info.command) {
		masca_util_stp_alm(ALM_DRVCTRL);
		masca_util_clr_event(SLOT_DRV_CTRL_TMOUT);
	}
	if ((SCAN_STOPPED  != play_ctrl.scan_state)
		|| (SCAN_STOPPED  != play_ctrl.new_scan_state)
		|| (PLAY_STARTING == play_ctrl.play_state)
		|| (PLAY_ONGOING  == play_ctrl.play_state)
		|| (PLAY_STARTING == play_ctrl.new_play_state)
		|| (PLAY_ONGOING  == play_ctrl.new_play_state)) {
		masca_send_cmd(SLOT_DRV_CTRL, CMD_PAUSE, 0, 0, 0);
		/* We will not worry about responses because pause
		* and resume operations generally occur when doing silent
		* reset of the drive or while retries in case no response
		* from the drive*/
	}
	/*stop the respective unit so that no retry happens*/
	masca_abort_unit(SLOT_DRV_CTRL);
}

void masca_resume_drvctrl(
		struct masca_drv_response_info * const p_response_info)
{
	drvctrl.slot_stat = SLOT_IDLE;
	if (MAX_COMMAND > drvctrl.app_cmd_info.command) {
		masca_exec_drvctrl(p_response_info);
	} else if ((SCAN_STOPPED != play_ctrl.scan_state)
			|| (PLAY_STARTING == play_ctrl.play_state)
			|| (PLAY_ONGOING == play_ctrl.play_state)) {
		/*Now we will internally execute a command and retries
		* as well*/
		masca_reload_rty_cnt(SLOT_DRV_CTRL);
		masca_send_cmd(SLOT_DRV_CTRL, last_cmd_to_drive,
					play_ctrl.current_pos, 0, 0);
		masca_util_sta_alm(drvctrl.slot_tmout, ALM_DRVCTRL);
	} else {
		/*nothing to do*/
	}
}

void masca_abort_drvctrl(struct masca_drv_response_info * const p_response_info,
				enum masca_error	abort_reason)
{
	enum masca_cmd app_cmd = drvctrl.app_cmd_info.command;
	if (MAX_COMMAND > app_cmd) {
		drvctrl.hw_response = abort_reason;
		if (MAX_COMMAND == p_response_info->drv_cmd_response) {
			drvctrl.abort_slot = FALSE;
			masca_drvctrl_response(NULL, p_response_info);
		} else {
			/* set an abort update so that through event
			* interface of this slot we can abort the command*/
			drvctrl.abort_slot = TRUE;
			masca_util_set_event(SLOT_ABORT_UPDATE);
			masca_util_stp_alm(ALM_DRVCTRL);
			masca_util_clr_event(SLOT_DRV_CTRL_TMOUT);
		}
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_DRV_CTRL);
	}
}

bool masca_evt_drvctrl(unsigned int evt_ptn_slot,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_required = FALSE;
	enum masca_error            ret_err;
	if (SLOT_RESPONSE_UPDATE == evt_ptn_slot) {
		if (MASCA_OK != drvctrl.hw_response) {
			if (MAX_COMMAND == p_response_info->drv_cmd_response)
				masca_drvctrl_response(NULL, p_response_info);
			else
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
		}
	} else if (SLOT_ABORT_UPDATE == evt_ptn_slot) {
		if (TRUE == drvctrl.abort_slot)
			masca_abort_drvctrl(p_response_info,
							drvctrl.hw_response);
	} else if (SLOT_PAUSED != drvctrl.slot_stat) {
		if (SLOT_DRV_CTRL_TMOUT == evt_ptn_slot) {
			/*Now we need to do retries*/
			ret_err = masca_evt_tmout(SLOT_DRV_CTRL,
							&is_rst_required);
			if (MASCA_OK != ret_err) {
				drvctrl.hw_response = ret_err;
				masca_drvctrl_response(NULL, p_response_info);
			} else {
				masca_util_sta_alm(drvctrl.slot_tmout,
								ALM_DRVCTRL);
			}
		}
	} else {
		/*In paused state no other events are entertained*/
	}
	return is_rst_required;
}

void masca_reset_play_sm(void)
{
	cd_info.end_addr		= 0;
	cd_info.start_addr		= 0;
	cd_info.frst_trk_num		= 0;
	cd_info.lst_trk_num		= 0;
	play_ctrl.play_start_addr       = 0;
	play_ctrl.play_end_addr		= 0;
	play_ctrl.track_playing		= 0;
	play_ctrl.current_pos		= OPTIC_POS_UNKNOWN;
	play_ctrl.play_state		= PLAY_STOPPED;
	play_ctrl.scan_state		= SCAN_STOPPED;
	play_ctrl.play_excd_duration	= 0;
	masca_copy_current_params();
}

/*This function does not pause or stop the disc,
The reason being that it should be called in cases like
undervolt or overtemperature or Sleep of drive manager*/
void masca_stop_play_sm(void)
{
	play_ctrl.play_state            = PLAY_STOPPED;
	play_ctrl.scan_state            = SCAN_STOPPED;
	masca_copy_current_params();
}

static
void masca_drvctrl_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_cmd app_cmd;
	app_cmd = drvctrl.app_cmd_info.command;

	if (MASCA_OK != drvctrl.hw_response) {
		if (MASCA_PROCESSED == drvctrl.hw_response) {
			if ((FBWD_LBA >= app_cmd) || (STOP == app_cmd)
				|| (RESUME == app_cmd)) {
				/*These are the commands which can affect state
				* transitions So when these are successfully
				* executed, apply thenew state parameters. This
				* is very important otherwisethe retries will
				* not work */
				masca_apply_transition();
			}
		}
		p_response_info->drv_cmd_response =
						drvctrl.app_cmd_info.command;
		p_response_info->response_error = drvctrl.hw_response;
		masca_util_stp_alm(ALM_DRVCTRL);
		masca_util_clr_event(SLOT_DRV_CTRL_TMOUT);
		drvctrl.app_cmd_info.command    = MAX_COMMAND;
		if (SLOT_PAUSED != drvctrl.slot_stat)
			drvctrl.slot_stat = SLOT_IDLE;
		drvctrl.hw_response = MASCA_OK;
	}
}

static void masca_exec_drvctrl(
		struct masca_drv_response_info * const p_response_info)
{
	enum masca_cmd		 app_cmd;
	struct masca_play_info_msf	*p_msf;
	struct masca_play_info_lba	*p_lba;
	enum masca_error		ret_err = MASCA_IR_PARAM_VAL_INVALID;
	enum masca_intrptr_cmd	cmd_to_drive = CMD_MAX;
	unsigned int			lba_start;
	unsigned int			lba_end;
	unsigned int			actual_play_start;
	unsigned int			play_start_pos = 0;
	unsigned int			lba_remaining;
	unsigned char			trk_no = 0;
	unsigned char			blk_cnt = 0;
	bool			is_param_valid = FALSE;

	app_cmd = drvctrl.app_cmd_info.command;
	drvctrl.slot_stat = SLOT_EXECUTING;
	drvctrl.hw_response = MASCA_OK;
	p_msf = &(drvctrl.app_cmd_info.command_param.drv_param.play_info_msf);
	p_lba = &(drvctrl.app_cmd_info.command_param.drv_param.play_info_lba);

	masca_updt_cd_info();
	if (PLAY_MSF_CONTINUOUS >= app_cmd) {
		if ((PLAY_MSF == app_cmd) || (PLAY_MSF_CONTINUOUS == app_cmd)) {
			lba_start = GET_LBA(p_msf->start_min, p_msf->start_sec,
							p_msf->start_frame);
			lba_end = GET_LBA(p_msf->end_min, p_msf->end_sec,
							p_msf->end_frame);
			trk_no = p_msf->trk_num_playmsf;
		} else {
			lba_start = p_lba->start_sector_address;
			if (lba_start == LBA_CURRENT_POS) {
				lba_end =
					play_ctrl.current_pos + p_lba->play_lng;
			} else {
				lba_end = p_lba->start_sector_address +
								p_lba->play_lng;
			}
			trk_no = p_lba->trk_num_playlba;
		}
		if ((MSF_CURRENT_POS == lba_start)
					|| (LBA_CURRENT_POS == lba_start)) {
			is_param_valid = masca_is_lba_valid(&lba_end, &lba_end);
			lba_start = MSF_CURRENT_POS;
			actual_play_start = play_ctrl.current_pos;
		} else {
			is_param_valid = masca_is_lba_valid(&lba_start,
								&lba_end);
			actual_play_start = lba_start;
		}

		if (TRUE == is_param_valid) {
			if ((lba_end - actual_play_start) > FRAMES_PER_SEC) {
				/*We will stop just one position before*/
				lba_remaining = lba_end % FRAMES_PER_SEC;
				lba_end -= lba_remaining;
			}
			ret_err = masca_handle_play_cmd(lba_start, lba_end,
						&play_start_pos, &cmd_to_drive);

			if (trk_no != 0)
				play_ctrl.last_trk_no_req = p_msf->trk_num_last;
			else
				play_ctrl.last_trk_no_req = INVAILD_PLAY_TRK;

			if (app_cmd == PLAY_MSF)
				cmd_to_drive = CMD_PLAY;
		}
	} else if ((FFWD_MSF <= app_cmd) && (FBWD_LBA >= app_cmd)) {
		if ((FWD_LBA == app_cmd) || (BWD_LBA == app_cmd)
			|| (FFWD_LBA == app_cmd) || (FBWD_LBA == app_cmd)) {
			lba_start = p_lba->start_sector_address;
			trk_no = p_lba->trk_num_playlba;
		} else {
			lba_start = GET_LBA(p_msf->start_min, p_msf->start_sec,
							p_msf->start_frame);
			trk_no = p_msf->trk_num_playmsf;
		}
		if (0 == trk_no) {
			is_param_valid =
				      masca_is_lba_valid(&lba_start,
								&lba_start);
		} else {
			is_param_valid = masca_get_trk_lba(trk_no, &lba_start);
			/*drive rejects the command if track no and play pos
			  both are set*/
			trk_no = 0;
		}
		if (TRUE == is_param_valid) {
			ret_err = masca_handle_scan_cmd(app_cmd, lba_start,
						&play_start_pos, &cmd_to_drive);
		}
	} else if (STOP == app_cmd) {
		ret_err = masca_handle_stop_cmd(&play_start_pos, &cmd_to_drive);
	} else if (PAUSE == app_cmd) {
		ret_err = masca_handle_pause_cmd(&cmd_to_drive);
	} else if (RESUME == app_cmd) {
		ret_err = masca_handle_resume_cmd(&play_start_pos,
								&cmd_to_drive);
	} else {
		/*stop or insert command*/
		cmd_to_drive = app_cmd_mapping[app_cmd].mapped_cmd;
		ret_err = MASCA_OK;
	}

	if (MASCA_OK == ret_err) {
		last_cmd_to_drive = cmd_to_drive;
		masca_send_cmd(SLOT_DRV_CTRL, cmd_to_drive, play_start_pos,
		trk_no, blk_cnt);

		if ((DIS_IGN_AUDIO == app_cmd) ||
				(CMD_DIS_IGN_AUDIO == cmd_to_drive))
			masca_util_sta_alm(MIX_MODE_CDDA_TIME_OUT, ALM_DRVCTRL);
		else
			masca_util_sta_alm(drvctrl.slot_tmout, ALM_DRVCTRL);
	} else {
		/* No need to send any command in response to application
		* request. The request is termed serviced*/
		drvctrl.hw_response = ret_err;
		masca_drvctrl_response(NULL, p_response_info);
	}
}

static void masca_updt_cd_info(void)
{
	struct masca_cd_toc *p_toc_info = NULL;

	masca_get_cd_toc(&p_toc_info);
	if (NULL != p_toc_info) {
		cd_info.start_addr =
			p_toc_info->track_info[p_toc_info->first_track_number].
								track_start_lba;
		cd_info.end_addr = GET_LBA(p_toc_info->cd_max_min,
						p_toc_info->cd_max_sec, 0);
		cd_info.lst_trk_num  = p_toc_info->last_track_number;
		cd_info.frst_trk_num = p_toc_info->first_track_number;
	}
}

static unsigned char masca_find_lba_track_num(const unsigned int check_lba,
						const unsigned char search_trk)
{
	bool            loop_end = FALSE;
	unsigned char              trk_no = 0;
	unsigned char              trk_found = 0;
	unsigned int             trk_start_addr = 0;
	unsigned int		trk_end_addr = 0;
	struct masca_cd_toc	*p_cd_toc = &cd_toc;
	masca_sm_get_toc(&p_cd_toc);

	/* Find check_lba track number */
	for (trk_no = search_trk;
		(trk_no <= p_cd_toc->last_track_number) && (loop_end != TRUE);
		trk_no++) {
		trk_start_addr = p_cd_toc->track_info[trk_no].track_start_lba;
		if (trk_no == p_cd_toc->last_track_number) {
			trk_end_addr = GET_LBA(p_cd_toc->cd_max_min,
						     p_cd_toc->cd_max_sec, 0);
		} else {
			trk_end_addr = p_cd_toc->track_info[trk_no + 1].
							track_start_lba - 1;
		}

		if ((trk_start_addr <= check_lba) &&
			(check_lba <= trk_end_addr)) {
			trk_found = trk_no;
			loop_end = TRUE;
		}
	}
	return trk_found;
}


/**
 * \func masca_valid_lba
 *
 * This function is used to check and trim the data track
 * information in p_lba_start and p_lba_end.
 */
static bool masca_valid_lba(unsigned char start_track, unsigned char end_track,
					unsigned int * const p_lba_start,
					unsigned int * const p_lba_end)
{
	bool	loop_end = FALSE;
	bool	is_valid_lba = FALSE;
	unsigned char	trk_no = 0;
	unsigned char	trk_adr_ctrl = 0;
	unsigned char	valid_start_trk = 0;
	unsigned char	valid_end_trk = 0;
	unsigned char	data_track_bit = 0x40;/* MASCA drive specification */
	struct masca_cd_toc *p_cd_toc = NULL;
	masca_get_cd_toc(&p_cd_toc);
	/* Check for start LBA */
	for (trk_no = start_track;
		(trk_no <= end_track) && (loop_end != TRUE);
		trk_no++) {
		trk_adr_ctrl = p_cd_toc->track_info[trk_no].adr_ctrl;
		if ((unsigned char)(trk_adr_ctrl & data_track_bit) !=
							data_track_bit) {
			loop_end = TRUE;
			valid_start_trk = trk_no;
		}
	}
	/* update vaild p_lba_start address of next audio track address */
	if (valid_start_trk != 0) {
		if (valid_start_trk != start_track) {
			*p_lba_start =
			 p_cd_toc->track_info[valid_start_trk].track_start_lba;
		}
		is_valid_lba = TRUE;
	} else {
		is_valid_lba = FALSE;
	}
	loop_end = FALSE;

	/* Check for end LBA */
	for (trk_no = end_track; (trk_no >= valid_start_trk) &&
		(is_valid_lba != FALSE) && (loop_end != TRUE); trk_no--) {
		trk_adr_ctrl = p_cd_toc->track_info[trk_no].adr_ctrl;
		if ((unsigned char)(trk_adr_ctrl & data_track_bit) !=
							data_track_bit) {
			loop_end = TRUE;
			valid_end_trk = trk_no;
		}
	}
	/* update vaild p_lba_start address of pervious audio track address */
	if (is_valid_lba != FALSE) {
		if (valid_end_trk != 0) {
			if (valid_end_trk != end_track) {
				*p_lba_end =
					p_cd_toc->track_info[valid_end_trk + 1]
							.track_start_lba - 1;
			}
			is_valid_lba = TRUE;
		} else {
		is_valid_lba = FALSE;
		}
	}

	return is_valid_lba;
}

static bool masca_is_lba_valid(unsigned int * const p_lba_start,
					unsigned int * const p_lba_end)
{
	bool	is_lba_valid = FALSE;
	bool	is_trk_num_vaild = FALSE;
	unsigned char	trk_adr_ctrl = 0;
	unsigned char	data_track_bit = 0x40;/* MASCA drive specification */
	unsigned char	start_trk = 0;
	unsigned char	end_trk = 0;
	unsigned int	lba_start = *p_lba_start;
	unsigned int	lba_end = *p_lba_end;
	struct masca_cd_toc	*p_cd_toc = NULL;

	masca_get_cd_toc(&p_cd_toc);

	if ((lba_end >= lba_start) &&
		(lba_start >= cd_info.start_addr) &&
		(lba_end <= cd_info.end_addr)) {
		/* Find lba_start track number */
		start_trk = masca_find_lba_track_num(lba_start,
		p_cd_toc->first_track_number);

		/* start track number(start_trk) should be greater than
		 * TOC start track number and less than the TOC last
		 * track number */
		if ((start_trk >= p_cd_toc->first_track_number) &&
			(start_trk <= p_cd_toc->last_track_number)) {
			/* Find lba_start track number */
			end_trk =
				masca_find_lba_track_num(lba_end,
								start_trk);

			/* end track number(end_trk) should be greater than
			 * or equal to start track number(start_trk) and less
			 * than the TOC last
			 * track number*/
			if ((end_trk >= start_trk) &&
				(end_trk <= p_cd_toc->last_track_number)) {
				/* start_trk and end_trk nos are falls under
				   the TOC range */
				is_trk_num_vaild = TRUE;
			} else {
				/* end_trk number isn't under the TOC range */
				is_trk_num_vaild = FALSE;
				/*logical block address out of range*/
				masca_update_global_sense_buffer(0x05, 0x21,
									 0x00);
			}
		} else {
			/* start_trk number isn't under the TOC range */
			is_trk_num_vaild = FALSE;
		}

		if (is_trk_num_vaild != FALSE) {
			if (start_trk == end_trk) {
				/* Both lba_start and lba_end are under same
				   track */
				trk_adr_ctrl = p_cd_toc->track_info[start_trk]
								.adr_ctrl;
				if ((unsigned char)(trk_adr_ctrl &
					data_track_bit) == data_track_bit) {
					is_lba_valid = FALSE;
				} else {
					is_lba_valid = TRUE;
				}
			} else {
				is_lba_valid = masca_valid_lba(start_trk,
						end_trk, &lba_start, &lba_end);
				*p_lba_start = lba_start;
				*p_lba_end = lba_end;
			}
		} else {
			is_lba_valid = FALSE;
		}
	} else {
		is_lba_valid = FALSE;
	}
	return is_lba_valid;
}

static bool masca_get_trk_lba(const unsigned char trk_num,
					unsigned int *const p_lba_start)
{
	bool is_trk_prsnt = FALSE;
	struct masca_cd_toc	*p_toc_info;
	masca_get_cd_toc(&p_toc_info);
	if ((trk_num >= p_toc_info->first_track_number)
		&& (trk_num <= p_toc_info->last_track_number)) {
		is_trk_prsnt = TRUE;
		if (NULL != p_lba_start) {
			*p_lba_start =
				p_toc_info->track_info[trk_num].
						track_start_lba;
		}
	}
	return is_trk_prsnt;
}

static void masca_apply_transition(void)
{
	play_ctrl.play_state = play_ctrl.new_play_state;
	play_ctrl.scan_state = play_ctrl.new_scan_state;
	play_ctrl.current_pos = play_ctrl.new_current_pos;
	play_ctrl.play_start_addr = play_ctrl.new_play_start_addr;
	play_ctrl.play_end_addr	= play_ctrl.new_play_end_addr;
	play_ctrl.track_playing = play_ctrl.new_track_playing;
	play_ctrl.play_excd_duration = play_ctrl.new_play_excd_duration;
}

static void masca_copy_current_params(void)
{
	play_ctrl.new_play_state = play_ctrl.play_state;
	play_ctrl.new_scan_state = play_ctrl.scan_state;
	play_ctrl.new_current_pos = play_ctrl.current_pos;
	play_ctrl.new_play_start_addr = play_ctrl.play_start_addr;
	play_ctrl.new_play_end_addr = play_ctrl.play_end_addr;
	play_ctrl.new_track_playing = play_ctrl.track_playing;
	play_ctrl.new_play_excd_duration = play_ctrl.play_excd_duration;
}

static enum masca_error
masca_handle_stop_cmd(unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd)
{
	enum masca_intrptr_cmd   nxt_cmd = CMD_MAX;
	unsigned int                 play_position = DO_NOT_PLAY;
	enum masca_error         ret_err = MASCA_INVALID_STATE;
	struct masca_last_evt      last_evt;

	masca_copy_current_params();
	if (SCAN_STOPPED != play_ctrl.scan_state) {
		/*according to ATAPI specification, now we will start play
		* if play was running before*/
		nxt_cmd = CMD_PAUSE;
		/*SCAN is no more valid*/
		play_ctrl.new_scan_state = SCAN_STOPPED;
		ret_err = MASCA_OK;
		if (PLAY_STOPPED != play_ctrl.play_state) {
			if ((play_ctrl.current_pos >= play_ctrl.play_start_addr)
			 && (play_ctrl.current_pos < play_ctrl.play_end_addr)) {
				if (PLAY_PAUSED_STARTING ==
							play_ctrl.play_state)
					play_ctrl.new_play_state =
								PLAY_STARTING;
				else if (PLAY_PAUSED_ONGOING ==
							play_ctrl.play_state)
					play_ctrl.new_play_state = PLAY_ONGOING;

				masca_get_last_event(&last_evt);
				if (DISC_PAUSED == last_evt.disc_event)
					nxt_cmd = CMD_RESUME;
				else
					nxt_cmd = CMD_PLAY_CONTINUOUS;
				play_position = play_ctrl.current_pos;
			} else {
				/*provide an evt for play completion over here*/
				masca_util_set_event(PLAY_RANGE_COMPLETE);
				play_ctrl.new_play_state = PLAY_STOPPED;
			}
		}
	} else if ((PLAY_PAUSED_STARTING == play_ctrl.play_state)
			|| (PLAY_PAUSED_ONGOING == play_ctrl.play_state)) {
		/*We are already in paused state, no need to issue
		* any commands and also no need to provide any events
		* since play is stopped*/
		play_ctrl.new_play_state = PLAY_STOPPED;
		ret_err = MASCA_PROCESSED;
	} else if (PLAY_STOPPED != play_ctrl.play_state) {
		/*currently play is either starting or ongoing
		* so issue a pause command*/
		nxt_cmd = CMD_PAUSE;
		play_ctrl.new_play_state = PLAY_STOPPED;
		ret_err = MASCA_OK;
	} else {
		/*Both scan and play are stopped, so nothing to do*/
		ret_err = MASCA_PROCESSED;
	}
	*p_pos_to_play = play_position;
	*p_nxt_cmd = nxt_cmd;

	return ret_err;
}


static enum masca_error
masca_handle_pause_cmd(enum masca_intrptr_cmd *const p_nxt_cmd)
{
	enum masca_error ret_err = MASCA_INVALID_STATE;
	enum masca_intrptr_cmd nxt_cmd = CMD_MAX;
	masca_copy_current_params();
	if (SCAN_STOPPED != play_ctrl.scan_state) {
		nxt_cmd = CMD_PAUSE;
		play_ctrl.new_scan_state = SCAN_STOPPED;
		if (PLAY_STARTING == play_ctrl.play_state)
			/*play is paused before at-least one play position is
			* received. These states are required because we only
			* hold the approximate position of optic head.*/
			play_ctrl.new_play_state = PLAY_PAUSED_STARTING;
		else if (PLAY_ONGOING == play_ctrl.play_state)
			/*play is paused before one play position with
			 * in boundaries is received. These states are required
			 * because we only hold the approximate position
			 * of optic head.*/
			play_ctrl.new_play_state = PLAY_PAUSED_ONGOING;
		ret_err = MASCA_OK;
	} else if ((PLAY_PAUSED_STARTING == play_ctrl.play_state)
			|| (PLAY_PAUSED_ONGOING == play_ctrl.play_state)) {
		/*According to ATAPI specifications a pause when the drive is
		* already paused is not treated as an error*/
		ret_err = MASCA_PROCESSED;
	} else if (PLAY_STOPPED != play_ctrl.play_state) {
		nxt_cmd = CMD_PAUSE;
		if (PLAY_STARTING == play_ctrl.play_state)
			/*play is paused before at-least one play position is
			* received. These states are required because we
			* only hold the approximate position of optic head.*/
			play_ctrl.new_play_state = PLAY_PAUSED_STARTING;
		else
			/* play is paused before one play position with in
			* boundaries is received. These states are required
			* because we only hold the approximateposition of
			* optic head.*/
			play_ctrl.new_play_state = PLAY_PAUSED_ONGOING;
		ret_err = MASCA_OK;
	}

	if (NULL != p_nxt_cmd)
		*p_nxt_cmd = nxt_cmd;
	return ret_err;
}

/*
 * masca_play_cmd function
 * if the play range is in last track, then
 * send PLAY NON CONTINUOUS mode command otherwise
 * send PLAY CONTINUOUS mode command to CD drive */
static enum masca_intrptr_cmd masca_play_cmd(unsigned int start_msf)
{
	struct masca_cd_toc	*p_cd_toc = NULL;

	masca_get_cd_toc(&p_cd_toc);
	if (masca_get_trk_num(start_msf) == p_cd_toc->last_audio_trk_num)
		return CMD_PLAY;	/* PLAY NON CONTINUOUS */
	else
		return CMD_PLAY_CONTINUOUS;
}

static enum masca_error
masca_handle_resume_cmd(unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd)
{
	enum masca_intrptr_cmd nxt_cmd = CMD_MAX;
	unsigned int play_position = DO_NOT_PLAY;
	enum masca_error ret_err = MASCA_OK;
	struct masca_last_evt      last_evt;
	masca_get_last_event(&last_evt);

	masca_copy_current_params();
	if ((SCAN_STOPPED != play_ctrl.scan_state) &&
		(PLAY_STOPPED != play_ctrl.play_state)) {
		/*SCAN is no more valid*/
		play_ctrl.new_scan_state = SCAN_STOPPED;
		if ((play_ctrl.current_pos >= play_ctrl.play_start_addr)
			&& (play_ctrl.current_pos < play_ctrl.play_end_addr)) {
			/*resume is valid now*/
			if (PLAY_PAUSED_STARTING == play_ctrl.play_state)
				play_ctrl.new_play_state = PLAY_STARTING;
			else if (PLAY_PAUSED_ONGOING == play_ctrl.play_state)
				play_ctrl.new_play_state = PLAY_ONGOING;

			/* RESUME command in MASCA drive starts play at
			 * current position and stops play at end of current
			 * playing track. So PLAY command is issued for
			 * RESUME command.
			 * */
			play_position = play_ctrl.current_pos;
			nxt_cmd = masca_play_cmd(play_position);
		} else {
			nxt_cmd = CMD_PAUSE;
			/*provide an event for play completion over here*/
			masca_util_set_event(PLAY_RANGE_COMPLETE);
			play_ctrl.new_play_state = PLAY_STOPPED;
		}
	} else if ((PLAY_PAUSED_STARTING == play_ctrl.play_state)
			|| (PLAY_PAUSED_ONGOING == play_ctrl.play_state)) {
		/*Check whether we received at least one play position from
		* drive since we started play*/
		if (PLAY_PAUSED_STARTING == play_ctrl.play_state)
			play_ctrl.new_play_state = PLAY_STARTING;
		else
			play_ctrl.new_play_state = PLAY_ONGOING;

		/* RESUME command in MASCA drive starts play at
		 * current position and stops play at end of current
		 * playing track. So PLAY command is issued for
		 * RESUME command.
		 * */
		play_position = play_ctrl.current_pos;
		nxt_cmd = masca_play_cmd(play_position);
	} else {
		ret_err = MASCA_INVALID_STATE;
	}
	*p_pos_to_play = play_position;
	*p_nxt_cmd = nxt_cmd;

	return ret_err;
}


static enum masca_error
masca_handle_scan_cmd(enum masca_cmd app_cmd,
				const unsigned int lba_start,
				unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd)
{
	masca_copy_current_params();
	play_ctrl.new_current_pos = lba_start;
	if ((FFWD_MSF == app_cmd) || (FFWD_LBA == app_cmd))
		play_ctrl.new_scan_state = SCAN_FAST_FORWARD;
	else if ((FWD_MSF == app_cmd) || (FWD_LBA == app_cmd))
		play_ctrl.new_scan_state = SCAN_FORWARD;
	else if ((FBWD_MSF == app_cmd) || (FBWD_LBA == app_cmd))
		play_ctrl.new_scan_state = SCAN_FAST_BACKWARD;
	else
		play_ctrl.new_scan_state = SCAN_BACKWARD;

	/*MASCA hardware does not support scanning from
	a specified position, so do a play to that position
	and then do scan after receiving response to play*/
	*p_pos_to_play = lba_start;
	*p_nxt_cmd = app_cmd_mapping[drvctrl.app_cmd_info.command].mapped_cmd;

	return MASCA_OK;
}

static enum masca_error
masca_handle_play_cmd(const unsigned int lba_start,
				const unsigned int lba_end,
				unsigned int * const p_pos_to_play,
				enum masca_intrptr_cmd * const p_nxt_cmd)
{
	enum masca_error ret_err = MASCA_IR_PARAM_VAL_INVALID;
	unsigned int play_position = DO_NOT_PLAY;
	enum masca_intrptr_cmd nxt_cmd = CMD_MAX;
	struct masca_last_evt	last_evt;
	struct masca_cd_toc	*p_cd_toc = NULL;

	masca_copy_current_params();
	if (MSF_CURRENT_POS != lba_start) {
		if (lba_start == lba_end) {
			/*Stop the scan or play if it is running*/
			if ((SCAN_STOPPED != play_ctrl.scan_state) ||
				(play_ctrl.play_state != PLAY_STOPPED)) {
				ret_err = MASCA_OK;
				nxt_cmd = CMD_PAUSE;
				play_ctrl.new_scan_state = SCAN_STOPPED;
				play_ctrl.new_play_state = PLAY_STOPPED;
			} else {
				ret_err = MASCA_PROCESSED;
			}
			/*provide an event for play completion over here*/
			masca_util_set_event(PLAY_RANGE_COMPLETE);
		} else {
			ret_err = MASCA_OK;
			play_position = lba_start;
			nxt_cmd = masca_play_cmd(play_position);
			play_ctrl.new_play_state = PLAY_STARTING;
			play_ctrl.new_scan_state = SCAN_STOPPED;
			play_ctrl.new_play_start_addr = lba_start;
			play_ctrl.new_play_end_addr = lba_end;
			play_ctrl.new_play_excd_duration = 0;
			/*We need to do this as we are freshly starting from a
			* new position. Use case would be before getting a play
			* position track end is reached. In this case we need to
			* quit*/
			play_ctrl.new_track_playing =
						masca_get_trk_num(lba_start);
			/*we do not know the current  position but we still
			* set it to the requested play_start address because
			* more play commands can come before we receive the
			* play position from the drive*/
			play_ctrl.new_current_pos = lba_start;
		}
	} else if (OPTIC_POS_UNKNOWN == play_ctrl.current_pos) {
		ret_err = MASCA_INVALID_STATE;
	} else if (lba_end > play_ctrl.current_pos) {
		ret_err = MASCA_OK;
		if ((SCAN_STOPPED != play_ctrl.scan_state)
			|| (PLAY_STOPPED == play_ctrl.play_state)) {
			/*Scan will be stopped*/
			play_position = play_ctrl.current_pos;
			nxt_cmd = masca_play_cmd(play_position);
			play_ctrl.new_play_state = PLAY_STARTING;
			play_ctrl.new_scan_state = SCAN_STOPPED;
			play_ctrl.new_play_start_addr = play_ctrl.current_pos;
			play_ctrl.new_play_excd_duration = 0;
		} else if ((PLAY_PAUSED_STARTING == play_ctrl.play_state)
			|| (PLAY_PAUSED_ONGOING  == play_ctrl.play_state)) {
			masca_get_last_event(&last_evt);
			/* RESUME command in MASCA drive starts play at
			 * current position and stops play at end of current
			 * playing track. So PLAY command is issued for
			 * RESUME command.
			 * */
			play_position = play_ctrl.current_pos;
			nxt_cmd = masca_play_cmd(play_position);
			if (PLAY_PAUSED_STARTING == play_ctrl.play_state)
				play_ctrl.new_play_state = PLAY_STARTING;
			else
				play_ctrl.new_play_state = PLAY_ONGOING;
		} else if ((PLAY_ONGOING == play_ctrl.play_state) ||
				(PLAY_STARTING == play_ctrl.play_state)) {
			masca_get_cd_toc(&p_cd_toc);
			if (masca_get_trk_num(lba_end)
					== play_ctrl.track_playing)
				ret_err = MASCA_PROCESSED;
			else {
				if (PLAY_MODE_NON_CONTINUOUS
						== masca_get_play_mode())
					nxt_cmd = CMD_CONTINOUS;
				else
					ret_err = MASCA_PROCESSED;
			}
		}
		play_ctrl.new_play_end_addr = lba_end;
	} else if (lba_end == play_ctrl.current_pos) {
		if ((SCAN_STOPPED != play_ctrl.scan_state) ||
			(play_ctrl.play_state != PLAY_STOPPED)) {
			ret_err = MASCA_OK;
			nxt_cmd = CMD_PAUSE;
			play_ctrl.new_scan_state = SCAN_STOPPED;
			play_ctrl.new_play_state = PLAY_STOPPED;
		} else {
			ret_err = MASCA_PROCESSED;
		}
		/*provide an event for play completion over here*/
		masca_util_set_event(PLAY_RANGE_COMPLETE);
	} else {
		ret_err = MASCA_IR_PARAM_VAL_INVALID;
	}
	*p_pos_to_play = play_position;
	*p_nxt_cmd = nxt_cmd;

	return ret_err;
}

static enum masca_intrptr_cmd masca_updt_current_pos(const unsigned int
						optic_head_pos,
						const unsigned int
						optic_relative_pos,
						const unsigned char track_no)
{
	unsigned int play_strt_addrs = play_ctrl.play_start_addr;
	unsigned int play_end_addrs = play_ctrl.play_end_addr;
	enum masca_intrptr_cmd   cmd_to_drive = CMD_MAX;
	struct masca_cd_toc *p_cd_toc = NULL;
	masca_get_cd_toc(&p_cd_toc);

	/*This is a race condition. Carefully observe that we should execute
	* updates only if no state changing command waiting for response*/
	/*We need to issue disc end event no matter whatever the state is.
	This section cannot be combined in the above code*/
	if ((SCAN_BACKWARD == play_ctrl.scan_state)
		|| (SCAN_FAST_BACKWARD == play_ctrl.scan_state)) {
		/*We are going in backward direction*/
		if ((optic_head_pos <= cd_info.start_addr) &&
						(optic_head_pos != 0))
			masca_util_set_event(DISC_END_REACHED);
	} else {
		/*We are going in forward direction and optic position
		* is at the end of the disc*/
		if (optic_head_pos >= cd_info.end_addr)
			masca_util_set_event(DISC_END_REACHED);
	}

	if ((play_ctrl.play_state == play_ctrl.new_play_state)
		&& (play_ctrl.new_scan_state == play_ctrl.scan_state)) {

		if (SCAN_STOPPED != play_ctrl.scan_state) {
			if ((SCAN_BACKWARD == play_ctrl.scan_state)
				|| (SCAN_FAST_BACKWARD ==
							play_ctrl.scan_state)) {
				/*We are going in backward direction*/
				if ((optic_head_pos <= cd_info.start_addr) &&
					(optic_head_pos != 0)) {
					/*we need to stop scanning*/
					cmd_to_drive = CMD_PAUSE;
					play_ctrl.scan_state = SCAN_STOPPED;
				}
			} else {
				/*We are going in forward direction and optic
				 * position is at the end of the disc*/
				if (optic_head_pos >= cd_info.end_addr) {
					/*we need to stop scanning*/
					cmd_to_drive = CMD_PAUSE;
					play_ctrl.scan_state = SCAN_STOPPED;
				}
			}
		} else if (PLAY_STARTING == play_ctrl.play_state) {
			if (optic_head_pos < optic_relative_pos) {
				cmd_to_drive = CMD_PLAY_CONTINUOUS;
				masca_send_cmd(SLOT_DRV_CTRL, cmd_to_drive,
					play_strt_addrs + FRAMES_PER_SEC, 0, 0);
			}
			if ((optic_head_pos >= play_strt_addrs)
				&& (optic_head_pos < play_end_addrs)) {
				/*Now the hardware has started to play in
				  the expected range*/
				play_ctrl.play_state = PLAY_ONGOING;
			} else {
				/*The hardware is still setting up itself to
				  the position*/
				if (OUTOFBOUND_LIMIT_TIME <=
						play_ctrl.play_excd_duration) {
					/*The head is still positioned outside,
					* might be this area on the disc has
					* gone bad, so consider play to be
					* complete*/
					play_ctrl.play_state = PLAY_STOPPED;
					cmd_to_drive = CMD_PAUSE;
					masca_util_set_event(
							PLAY_RANGE_COMPLETE);
				}
				if ((play_end_addrs - play_strt_addrs) <=
							       FRAMES_PER_SEC) {
					/*This is a very small gap*/
					if (optic_head_pos >= play_end_addrs) {
						/* For such small play gap there
						* will always be an error of
						approximately 1sec in playing.*/
						play_ctrl.play_state =
								PLAY_STOPPED;
						cmd_to_drive = CMD_PAUSE;
						masca_util_set_event(
							   PLAY_RANGE_COMPLETE);
					}
				}
				play_ctrl.play_excd_duration++;
			}
		} else if (PLAY_ONGOING == play_ctrl.play_state) {
			if ((optic_head_pos >= cd_info.end_addr)
				|| (optic_head_pos >= play_end_addrs)
				|| (optic_head_pos < play_strt_addrs)) {
				/*we need to stop the play as we reached
				* end of the disc*/
				play_ctrl.play_state = PLAY_STOPPED;
				cmd_to_drive = CMD_PAUSE;
				masca_util_set_event(PLAY_RANGE_COMPLETE);
			}
		} else {
			/*Nothing to do over here*/
		}
		/*Update the new state changes if any*/
		masca_copy_current_params();
	}
	play_ctrl.current_pos = optic_head_pos;
	play_ctrl.track_playing = track_no;

	return cmd_to_drive;
}


/* During play/scan opertion, Update to audio track position
 * to skip the data track position for a mixed mode CD's  */
static void masca_updt_audio_track_pos(unsigned char data_trk)
{
	unsigned char	data_track_bit = 0x40;/* MASCA drive specification */
	bool	data_trk_skip = FALSE;
	unsigned char	trk_no = 0;
	unsigned int	play_start_pos = 0;
	unsigned int	lba_end = 0;
	unsigned int	lba_start = 0;
	enum masca_cmd	app_cmd = MAX_COMMAND;
	enum masca_intrptr_cmd cmd_to_drive = CMD_MAX;
	enum masca_error	ret_err = MASCA_PTR_NULL;
	struct masca_cd_toc *p_cd_toc = NULL;
	masca_get_cd_toc(&p_cd_toc);
	/* Special handling for mixed mode CD's
	   Update lba_start for backward scan */
	if ((SCAN_BACKWARD == play_ctrl.scan_state) ||
		(SCAN_FAST_BACKWARD == play_ctrl.scan_state)) {
		for (trk_no = data_trk;
			(trk_no >= p_cd_toc->first_track_number) &&
			((p_cd_toc->track_info[trk_no].adr_ctrl &
				data_track_bit) == data_track_bit); trk_no--) {
			lba_start = p_cd_toc->track_info[trk_no].track_start_lba
									- 1;
			lba_end = play_ctrl.play_end_addr;
			data_trk_skip = TRUE;
		}
	} else {
		/* Update lba_start for play and forward scan */
		for (trk_no = data_trk;
			(trk_no <= p_cd_toc->last_track_number) &&
			((p_cd_toc->track_info[trk_no].adr_ctrl &
					data_track_bit) == data_track_bit);
			trk_no++) {
			lba_start = p_cd_toc->track_info[trk_no + 1].
								track_start_lba;
			lba_end = play_ctrl.play_end_addr;
			data_trk_skip = TRUE;
		}
	}

	if ((PLAY_STOPPED != play_ctrl.play_state)
		&& (SCAN_STOPPED == play_ctrl.scan_state)
		&& (data_trk_skip != FALSE)) {
		ret_err = masca_handle_play_cmd(lba_start, lba_end,
					&play_start_pos, &cmd_to_drive);
	} else if (((PLAY_STOPPED == play_ctrl.play_state) ||
		(PLAY_ONGOING == play_ctrl.play_state))
		&& (SCAN_STOPPED != play_ctrl.scan_state)
		&& (data_trk_skip != FALSE)) {
		if (SCAN_FORWARD == play_ctrl.scan_state)
			app_cmd = FWD_LBA;
		else if (SCAN_BACKWARD == play_ctrl.scan_state)
			app_cmd = BWD_LBA;
		else if (SCAN_FAST_FORWARD == play_ctrl.scan_state)
			app_cmd = FFWD_LBA;
		else /* SCAN_FAST_BACKWARD */
			app_cmd = FBWD_LBA;
		ret_err = masca_handle_scan_cmd(app_cmd, lba_start,
						&play_start_pos, &cmd_to_drive);
		cmd_to_drive = app_cmd_mapping[app_cmd].mapped_cmd;
	} else {
	}

	if (MASCA_OK == ret_err) {
		last_cmd_to_drive = cmd_to_drive;
		masca_send_cmd(SLOT_DRV_CTRL, cmd_to_drive,
						play_start_pos, 0, 0);
		masca_util_sta_alm(drvctrl.slot_tmout, ALM_DRVCTRL);
		play_ctrl.play_state = play_ctrl.new_play_state;
		play_ctrl.scan_state = play_ctrl.new_scan_state;
	}
}

static enum masca_intrptr_cmd masca_updt_trk_end_evt(void)
{
	unsigned char	data_track_bit = 0x40;/* MASCA drive specification */
	unsigned char	end_play_trk_num = 0;
	enum masca_intrptr_cmd cmd_to_drive = CMD_MAX;
	unsigned int	play_start_pos = 0;
	enum masca_error	ret_err = MASCA_INVALID_STATE;
	struct masca_cd_toc	*p_cd_toc = NULL;
	masca_get_cd_toc(&p_cd_toc);

	if ((SCAN_BACKWARD == play_ctrl.scan_state)
		|| (SCAN_FAST_BACKWARD == play_ctrl.scan_state)) {
		/* Check: Previous track is data track or not? */
		if ((p_cd_toc->track_info[play_ctrl.track_playing - 1].adr_ctrl
			& data_track_bit) == data_track_bit) {
			masca_updt_audio_track_pos(play_ctrl.track_playing - 1);
		}
	}  else {/* Check: Next track is data track or not? */
		if ((p_cd_toc->track_info[play_ctrl.track_playing + 1]
			.adr_ctrl & data_track_bit) == data_track_bit) {
			end_play_trk_num =
				masca_get_trk_num(play_ctrl.play_end_addr);
			if ((end_play_trk_num == play_ctrl.track_playing) &&
				(SCAN_STOPPED == play_ctrl.scan_state)) {
				ret_err = masca_handle_stop_cmd(&play_start_pos,
								&cmd_to_drive);
				if (MASCA_OK == ret_err) {
					last_cmd_to_drive = cmd_to_drive;
					masca_send_cmd(SLOT_DRV_CTRL,
							cmd_to_drive,
							play_start_pos, 0, 0);
					masca_util_sta_alm(drvctrl.slot_tmout,
								ALM_DRVCTRL);
					play_ctrl.play_state =
						play_ctrl.new_play_state;
					play_ctrl.scan_state =
						play_ctrl.new_scan_state;
				}
				/* provide an event for play completion */
				masca_util_set_event(PLAY_RANGE_COMPLETE);
				play_ctrl.new_play_state = PLAY_STOPPED;

				if (p_cd_toc->last_audio_trk_num
							== end_play_trk_num) {
					/* If there is no audio tracks further,
					 * then provide an event for
					 * disc completion over here*/
					masca_util_set_event(DISC_END_REACHED);
				}
			} else {
				masca_updt_audio_track_pos(
						play_ctrl.track_playing + 1);
			}
		}
	}

	/*This is a race condition. Carefully observe that we should execute
	* updates only if no state changing command waiting for response*/
	if (OPTIC_POS_UNKNOWN == play_ctrl.current_pos) {
		/*We are having a track end evt even before we received current
		 * position when we play the disc for the first time*/
		/*Now compare the new applied values*/
		if ((SCAN_BACKWARD == play_ctrl.new_scan_state)
			|| (SCAN_FAST_BACKWARD == play_ctrl.new_scan_state)) {
			if (cd_info.frst_trk_num ==
						play_ctrl.new_track_playing) {
				masca_util_set_event(DISC_END_REACHED);
			}
		} else {
			if (cd_info.lst_trk_num ==
						 play_ctrl.new_track_playing) {
				masca_util_set_event(DISC_END_REACHED);
			}
		}
	} else if ((SCAN_BACKWARD == play_ctrl.scan_state)
		|| (SCAN_FAST_BACKWARD == play_ctrl.scan_state)) {
		if (cd_info.frst_trk_num == play_ctrl.track_playing)
			masca_util_set_event(DISC_END_REACHED);
	} else {
		if (cd_info.lst_trk_num == play_ctrl.track_playing)
			masca_util_set_event(DISC_END_REACHED);
	}

	if ((play_ctrl.play_state == play_ctrl.new_play_state)
		&& (play_ctrl.new_scan_state == play_ctrl.scan_state)) {
		if ((SCAN_BACKWARD == play_ctrl.scan_state)
			|| (SCAN_FAST_BACKWARD == play_ctrl.scan_state)) {
			if (play_ctrl.track_playing == cd_info.frst_trk_num) {
				/*Our hardware does not stop playback at the
				*  disk end. Now we need to stop the current
				* play back */
				cmd_to_drive = CMD_PAUSE;
				play_ctrl.play_state = PLAY_STOPPED;
				play_ctrl.scan_state = SCAN_STOPPED;
			}
		} else if ((PLAY_STOPPED != play_ctrl.play_state)
			|| (SCAN_STOPPED != play_ctrl.scan_state)) {
			if (play_ctrl.track_playing == cd_info.lst_trk_num) {
				/*Our hardware does not stop playback at the
				* disk end. Now we need to stop the current
				* play back*/
				cmd_to_drive = CMD_PAUSE;
				play_ctrl.play_state = PLAY_STOPPED;
				play_ctrl.scan_state = SCAN_STOPPED;
			}
		} else {
		}
		/*Update the new state changes if any*/
		masca_copy_current_params();

		if ((play_ctrl.track_playing == play_ctrl.last_trk_no_req) ||
			(play_ctrl.track_playing ==
					p_cd_toc->last_audio_trk_num)) {
			play_ctrl.play_state = PLAY_STOPPED;
			play_ctrl.scan_state = SCAN_STOPPED;
			masca_util_set_event(PLAY_RANGE_COMPLETE);
		}

		if (((INVAILD_PLAY_TRK ==  play_ctrl.last_trk_no_req) &&
			(play_ctrl.track_playing + 1 ==
					p_cd_toc->last_audio_trk_num)) ||
			(play_ctrl.track_playing + 1 ==
					play_ctrl.last_trk_no_req))
			cmd_to_drive = CMD_PLAY_NON_CONTINUOUS;
	}
	return cmd_to_drive;
}


static unsigned char masca_get_trk_num(const unsigned int lba)
{
	struct masca_cd_toc *p_toc_info = NULL;
	struct masca_track_info *p_trk_info;
	bool             is_element_found = FALSE;
	unsigned char trk_count = 0;
	unsigned char lst_track;
	unsigned char frst_track;

	masca_get_cd_toc(&p_toc_info);
	if (NULL != p_toc_info) {
		p_trk_info = &p_toc_info->track_info[CD_DISC_INFO];
		frst_track = p_toc_info->first_track_number;
		lst_track = p_toc_info->last_track_number;

		if (lba <= p_toc_info->track_info[frst_track].track_start_lba) {
			is_element_found = TRUE;
			trk_count = frst_track;
		} else if (lba >= p_toc_info->track_info[lst_track]
							 .track_start_lba) {
			is_element_found = TRUE;
			trk_count = lst_track;
		} else {
			trk_count = (frst_track + lst_track) >> 1;
		}
		/*Do a binary search over the TOC info*/
		while ((lst_track > frst_track) &&
						(FALSE == is_element_found)) {
			if ((p_trk_info[trk_count].track_start_lba == lba)
			|| ((p_trk_info[trk_count].track_start_lba  < lba)
			&& (p_trk_info[trk_count + 1].track_start_lba > lba)))
				is_element_found = TRUE;
			else if (p_trk_info[trk_count].track_start_lba < lba)
				frst_track = trk_count;
			else
				lst_track = trk_count;

			trk_count = (lst_track + frst_track) >> 1;
		}
	}
	return trk_count;
}

