/*
 * MASCA Diagnostic Commands and Response Handler File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"

static void
masca_diag_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info);

static void masca_exec_diag(const struct masca_drv_response_info *
							const p_response_info);

static struct masca_cmd_slot diag_slot;

void masca_init_diag(const unsigned int retry_cnt, const unsigned int reset_cnt,
			const unsigned int response_tmout)
{
	diag_slot.slot_stat = SLOT_PAUSED;
	diag_slot.app_cmd_info.command = MAX_COMMAND;
	diag_slot.hw_response = MASCA_OK;
	diag_slot.slot_tmout = response_tmout;
	masca_init_tx_rx(SLOT_DIAGNOSTIC, retry_cnt, reset_cnt);
}

enum masca_cmd masca_get_diag_cmd(void)
{
	return diag_slot.app_cmd_info.command;
}

void masca_diag_alm_cbk(void)
{
	masca_util_set_event(SLOT_DIAGNOSTIC_TMOUT);
}

enum masca_error
masca_tx_diag(const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	bool                    exec_slot = FALSE;
	enum masca_error             ret_err;

	ret_err = masca_fill_slot(&diag_slot, p_cmd_info, &exec_slot);
	if ((MASCA_OK == ret_err) && (TRUE == exec_slot)) {
		/*reload the retry counters for the slot*/
		masca_reload_rty_cnt(SLOT_DIAGNOSTIC);
		masca_exec_diag(p_response_info);
	}
	return ret_err;
}


void masca_rx_diag(
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_response_info * const p_response_info)
{
	/*In the paused state also the responses from the hardware
	* are respected*/
	enum masca_error ret_err;
	if (SLOT_PAUSED != diag_slot.slot_stat) {
		/*Finding out a response is received or not is a generic
		* functionality. Hence it is implemented in a common place*/
		ret_err = masca_rcv_response(SLOT_DIAGNOSTIC, p_list,
								p_rjt_cmds);
		if ((NULL != p_response_info) && (MASCA_OK != ret_err)) {
			diag_slot.hw_response = ret_err;
			/*Now we need to provide a reply back*/
			if (MAX_COMMAND == p_response_info->drv_cmd_response) {
				masca_diag_response(p_msg, p_response_info);
			} else {
				/*We got response to the slot, so clear the
				 *timeout. This should never happen
				 *for diagnostic slot as the messages are
				 *explicit and many messages are not combined
				 *to one as like status message of the drive*/
				masca_util_stp_alm(ALM_DIAG);
				masca_util_clr_event(SLOT_DIAGNOSTIC_TMOUT);
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
				DRV_MNGR_TA_TRACE("DRV_DIAG : rcv resp = %d\n",
					ret_err);
			}
		}
	}
}

void masca_pause_diag(void)
{
	diag_slot.slot_stat = SLOT_PAUSED;
	/*we are pausing so don't allow timeouts*/
	if (MAX_COMMAND > diag_slot.app_cmd_info.command) {
		masca_util_stp_alm(ALM_DIAG);
		masca_util_clr_event(SLOT_DIAGNOSTIC_TMOUT);
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_DIAGNOSTIC);
	}
}

void masca_resume_diag(struct masca_drv_response_info * const p_response_info)
{
	diag_slot.slot_stat = SLOT_IDLE;
	if (MAX_COMMAND > diag_slot.app_cmd_info.command)
		masca_exec_diag(p_response_info);
}

void masca_abort_diag(struct masca_drv_response_info * const p_response_info,
				enum masca_error	abort_reason)
{
	if (MAX_COMMAND > diag_slot.app_cmd_info.command) {
		diag_slot.hw_response = abort_reason;
		if (MAX_COMMAND == p_response_info->drv_cmd_response) {
			diag_slot.abort_slot = FALSE;
			masca_diag_response(NULL, p_response_info);
		} else {
			/* set an abort update so that through event
			* interface of this slot we can abort the command*/
			diag_slot.abort_slot = TRUE;
			masca_util_set_event(SLOT_ABORT_UPDATE);
			masca_util_stp_alm(ALM_DIAG);
			masca_util_clr_event(SLOT_DIAGNOSTIC_TMOUT);
		}
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_DIAGNOSTIC);
	}
}

bool masca_evt_diag(unsigned int evt_ptn_slot,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_required = FALSE;
	enum masca_error            ret_err;
	if (SLOT_RESPONSE_UPDATE == evt_ptn_slot) {
		/*We encouter this situation when we receive in a
		* single message response to two slots. Currently
		* for the diagnostic slot this situation can only
		* occur if the diagnostic command gets rejected. In case
		* of rejection, no additional message info required*/
		/*A generic handling is to buffer the message before
		* setting SLOT_RESPONSE_UPDATE event. Right now that can be
		* costly because of memory.*/
		if (MASCA_OK != diag_slot.hw_response) {
			if (MAX_COMMAND == p_response_info->drv_cmd_response)
				masca_diag_response(NULL, p_response_info);
			else
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
		}
	} else if (SLOT_ABORT_UPDATE == evt_ptn_slot) {
		if (TRUE == diag_slot.abort_slot)
			masca_abort_diag(p_response_info,
							diag_slot.hw_response);
	} else if (SLOT_PAUSED != diag_slot.slot_stat) {
		if (SLOT_DIAGNOSTIC_TMOUT == evt_ptn_slot) {
			/*Now we need to do retries*/
			ret_err = masca_evt_tmout(SLOT_DIAGNOSTIC,
							&is_rst_required);
			if (MASCA_OK != ret_err) {
				diag_slot.hw_response = ret_err;
				masca_diag_response(NULL, p_response_info);
			} else {
				masca_util_sta_alm(diag_slot.slot_tmout,
								      ALM_DIAG);
			}
		}
	} else {
		/*In paused state no other events are entertained*/
	}
	return is_rst_required;
}

/* copy diagnostic command response to p_response_info diagnostic parameter */
static void copy_diag_response_from_drv_mngr(enum masca_cmd app_cmd,
		union masca_intrptr_decode_info *p_msg_detail,
		struct masca_drv_response_info * const p_response_info)
{

	switch (app_cmd) {
	case DIAG_GET_TEMPERATURE:
		memcpy(p_response_info->response_param.cd_diag_params.
								temperature,
			p_msg_detail->diag_info.drv_temperature,
			MAX_TEMPERATURE_SIZE);
		break;
	case GET_VERSION_INFO:
		p_response_info->response_param.cd_version_info.
			hardware_version = p_msg_detail->ver_info.hw_version;
		p_response_info->response_param.cd_version_info.
			software_version = p_msg_detail->ver_info.sw_version;
		/* Vendor identification */
		p_response_info->response_param.cd_version_info.
		vendor_identification = p_msg_detail->ver_info.vendors_id;
		break;
	case DIAG_GET_LASER_CURRENT:
		memcpy(p_response_info->response_param.cd_diag_params.
								laser_current,
			p_msg_detail->diag_info.drv_laser_current,
			MAX_LASER_CUR_SIZE);
		break;
	case DIAG_GET_SWITCH_STATUS:
		p_response_info->response_param.cd_diag_params.switch_status =
				p_msg_detail->diag_info.drv_switch_status;
		break;
	case DIAG_CMD_ADJUSTMENT:    /*30*/
		p_response_info->response_param.cd_diag_params.diag_adjust =
			p_msg_detail->diag_info.drv_diag_adjust;
		break;
	case DIAG_CMD_C1C2_ERR_COUNT_ON:
		memcpy(p_response_info->response_param.cd_diag_params.
								c1c2_err_count,
			p_msg_detail->diag_info.drv_c1c2_err_count,
			MAX_C1C2_ERR_CNT_SIZE);
		break;
	case DIAG_CMD_CD_TEXT_OFF:
	case DIAG_CMD_CD_TEXT_MODE2:
		p_response_info->response_param.cd_diag_params.cmd_recevied =
			p_msg_detail->diag_info.drv_cmd_recevied;
		break;
	case DIAG_CMD_TIME_READOUT:
		memcpy(p_response_info->response_param.cd_diag_params.
								time_readout,
			p_msg_detail->diag_info.drv_time_readout,
			MAX_DIAG_TIME_READOUT_SIZE);
		break;
	case DIAG_CMD_EJECT_LOAD_CYCLES: /*35*/
		memcpy(p_response_info->response_param.cd_diag_params.
							eject_load_cycles,
			p_msg_detail->diag_info.drv_eject_load_cycles,
			MAX_EJT_LD_CYC_SIZE);
		break;
	case DIAG_CMD_JUMP_COUNTER:
		p_response_info->response_param.cd_diag_params.jump_cnt =
				p_msg_detail->diag_info.drv_jump_cnt;
		break;
	case DIAG_CMD_EEPROM_READOUT:
		memcpy(p_response_info->response_param.cd_diag_params.
								eeprom_readout,
			p_msg_detail->diag_info.drv_eeprom_readout,
			MAX_EEPROM_READOUT_SIZE);
		break;
	case DIAG_CMD_WRITE_STICKER:
		memcpy(p_response_info->response_param.cd_diag_params.
							write_sticker_no,
			p_msg_detail->diag_info.drv_write_sticker_no,
			MAX_WRITE_STICKER_SIZE);
		break;
	case DIAG_CMD_READ_STICKER:
		memcpy(p_response_info->response_param.cd_diag_params.
							read_sticker_no,
			p_msg_detail->diag_info.drv_read_sticker_no,
			MAX_READ_STICKER_SIZE);
		break;
	case DIAG_CMD_JITTER_MEASURE:    /*40*/
		p_response_info->response_param.cd_diag_params.jitter_measure =
				p_msg_detail->diag_info.drv_jitter_measure;
		break;
	case DIAG_CMD_VERTICAL_DEV_MEASURE:
		memcpy(p_response_info->response_param.cd_diag_params.
							vertical_dev_measure,
			p_msg_detail->diag_info.drv_vertical_dev_measure,
			MAX_VER_DEV_SIZE);
		break;
	case DIAG_CMD_ECCENTRIC_MEASURE:
		memcpy(p_response_info->response_param.cd_diag_params.
							eccentric_measure,
			p_msg_detail->diag_info.drv_eccentric_measure,
			MAX_ECCENTRICITY_SIZE);
		break;
	case DIAG_CMD_REFLECTIVE_MEASURE:
		p_response_info->response_param.cd_diag_params.
							reflective_measure =
			p_msg_detail->diag_info.drv_reflective_measure;
		break;
	case DIAG_CMD_INTERNAL_SELF_TEST:
		p_response_info->response_param.cd_diag_params.
							internal_self_test =
			p_msg_detail->diag_info.drv_internal_self_test;
		break;
	case DIAG_CMD_EXTERNAL_SDRAM_TEST:   /*45*/
		p_response_info->response_param.cd_diag_params.
							external_sdram_test =
			p_msg_detail->diag_info.drv_external_sdram_test;
		break;
	case DIAG_CMD_INIT_LASER_CURRENT:
		memcpy(p_response_info->response_param.cd_diag_params.
							init_laser_current,
			p_msg_detail->diag_info.drv_init_laser_current,
			MAX_INTI_LASER_CUR_SIZE);
		break;
	case DIAG_CMD_NUM_READ_ERRS:
		memcpy(p_response_info->response_param.cd_diag_params.
								num_read_errs,
			p_msg_detail->diag_info.drv_num_read_errs,
			MAX_NO_READ_ERRS_SIZE);
		break;
	case DIAG_CMD_DATE_PRODUCTION_READ:
	case DIAG_CMD_DATE_INSTALL_READ:
		memcpy(p_response_info->response_param.cd_diag_params.
						date_product_install_read,
			p_msg_detail->diag_info.drv_date_product_install_read,
			MAX_DATE_PROD_INSTALL_SIZE);
		break;
	case DIAG_CMD_DATE_INSTALL_WRITE:/*50*/
		p_response_info->response_param.cd_diag_params.cmd_recevied =
			p_msg_detail->diag_info.drv_cmd_recevied;
		break;
	case DIAG_CMD_SERIAL_NO_READ:
		memcpy(p_response_info->response_param.cd_diag_params.
							serial_no_read,
			p_msg_detail->diag_info.drv_serial_no_read,
			MAX_SERIAL_NO_SIZE);
		break;
	case DIAG_CMD_STATUS_MSG_ERR_HISTROY:
		memcpy(p_response_info->response_param.cd_diag_params.
							status_msg_err_histroy,
			p_msg_detail->diag_info.drv_status_msg_err_histroy,
			MAX_MSG_ERR_HIS_SIZE);
		break;
	case DIAG_CMD_STATUS_MSG_LAST_ERRS:
		memcpy(p_response_info->response_param.cd_diag_params.
							status_msg_last_errs,
			p_msg_detail->diag_info.drv_status_msg_last_errs,
			MAX_MSG_LAST_ERRS_SIZE);
		break;
	case DIAG_CMD_USED_CD_TYPE_COUNT:
		memcpy(p_response_info->response_param.cd_diag_params.
							used_cd_type_cnt,
			p_msg_detail->diag_info.drv_used_cd_type_cnt,
			MAX_CD_TYPE_CNT_SIZE);
		break;
	case DIAG_CMD_USER_PROFILE_USED_CDS: /*55*/
		memcpy(p_response_info->response_param.cd_diag_params.
							user_profile_used_cds,
			p_msg_detail->diag_info.drv_user_profile_used_cds,
			MAX_PROFIL_USED_CD_SIZE);
		break;
	default:
		DRV_MNGR_TE_TRACE("DrvMngDiag:Unknown cmd %d\n", app_cmd);
		break;
	}
}

static void
masca_diag_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info)
{
	union masca_intrptr_decode_info	msg_detail;
	enum masca_error		err;
	enum masca_intrptr_decode_msg	expected_response;
	enum masca_cmd			app_cmd;
	enum masca_intrptr_cmd		drv_spfc_cmd;
	if (MASCA_PROCESSED == diag_slot.hw_response) {
		app_cmd = diag_slot.app_cmd_info.command;
		drv_spfc_cmd = app_cmd_mapping[app_cmd].mapped_cmd;
		expected_response = drv_responses[drv_spfc_cmd].
							drive_message[0];
		err = masca_intrptr_decode_msg_info(expected_response,
							&msg_detail, p_msg);
		if (MASCA_OK == err)
			copy_diag_response_from_drv_mngr(app_cmd,
					&msg_detail, p_response_info);
		else
			DRV_MNGR_TE_TRACE
			("DrvMngDiag: Err frm decode msg info:%d\n", err);
	}
	p_response_info->drv_cmd_response = diag_slot.app_cmd_info.command;
	p_response_info->response_error = diag_slot.hw_response;
	masca_util_stp_alm(ALM_DIAG);
	masca_util_clr_event(SLOT_DIAGNOSTIC_TMOUT);
	diag_slot.app_cmd_info.command = MAX_COMMAND;
	diag_slot.hw_response = MASCA_OK;

	if (SLOT_PAUSED != diag_slot.slot_stat)
		diag_slot.slot_stat = SLOT_IDLE;
}

static void
masca_exec_diag(const struct masca_drv_response_info * const p_response_info)
{
	unsigned int     install_date_write = 0;
	enum masca_intrptr_cmd       cmd_to_drive;
	diag_slot.slot_stat = SLOT_EXECUTING;
	cmd_to_drive =
		app_cmd_mapping[diag_slot.app_cmd_info.command].mapped_cmd;
	diag_slot.hw_response = MASCA_OK;

	masca_util_sta_alm(diag_slot.slot_tmout, ALM_DIAG);

	if (CMD_WRITE_STICKER == cmd_to_drive) {
		masca_send_diag_cmd(SLOT_DIAGNOSTIC, cmd_to_drive, 0,
				diag_slot.app_cmd_info.command_param.drv_param
				.diag_cmd_params.write_sticker);
	} else if (CMD_DATE_INSTALL_WRITE == cmd_to_drive) {
		install_date_write = diag_slot.app_cmd_info.command_param
				.drv_param.diag_cmd_params.install_date_write;
		masca_send_diag_cmd(SLOT_DIAGNOSTIC, cmd_to_drive,
							install_date_write, 0);
	} else {
		masca_send_diag_cmd(SLOT_DIAGNOSTIC, cmd_to_drive, 0, 0);
	}
}
