/*
 * MASCA Hadware Manager File.Handles Initilization,Undervolt and Reset
 * Sequences.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_drv_mngr.h"
#include "masca_event_config.h"
#include "masca_hal.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"
#include "masca_block_reader.h"
#include "masca_blkdev.h"
#include "masca_sm.h"

enum masca_drv_mngr_stat {
	DRV_RECOGNIZING = 0,
	DRV_CONNECTED,
	DRV_DISCONNECTED
};

enum masca_recogn_stat {
	RECONG_START = 0,
	RESET_DRV,
	WAIT_FOR_DRV_INIT
};

#define DRIVE_RST_TIME		100
#define DRIVE_INIT_TMOUT	2000
#define DRIVE_POLL_TMOUT	4000
#define BLK_READ_RESET_COUNT	3
#define BLK_READ_RETRY_COUNT	3
#define BLK_READ_RESP_TMOUT	20000

#define MAX_CD_DRIVE_MSG_SZ	256
unsigned char p_raw_data[MAX_CD_DRIVE_MSG_SZ];

static void
masca_exec_mngr_sm(unsigned int sm_evt_ptn, const bool reset_drive,
			const struct masca_intrptr_msg_info_list * const p_list,
			struct masca_drv_response_info * const p_response_info);

static void
masca_hndl_slt_resp(const unsigned int updt_evt_ptn,
			struct masca_drv_response_info * const p_response_info);
static void
masca_handle_recp_events(struct masca_drv_event_info * const p_evt_info,
			struct masca_drv_response_info * const p_response_info);

static void
masca_control_slots(const enum masca_drv_mngr_stat prev_state,
			const enum masca_drv_mngr_stat current_state,
			struct masca_drv_response_info * const p_response_info);

static const struct masca_slot_class slot_funcs[SLOT_MAX_NUM] = {
	{masca_get_diag_cmd, masca_tx_diag, masca_rx_diag, masca_evt_diag,
		masca_pause_diag, masca_resume_diag, masca_abort_diag},

	{masca_get_eject_cmd, masca_tx_eject, masca_rx_eject, masca_evt_eject,
		masca_pause_eject, masca_resume_eject, masca_abort_eject},
	{masca_get_drvctrl_cmd, masca_tx_drvctrl, masca_rx_drvctrl,
	masca_evt_drvctrl, masca_pause_drvctrl, masca_resume_drvctrl,
	masca_abort_drvctrl},
	{masca_get_toctexthash_cmd, masca_tx_toctexthash, masca_rx_toctexthash,
	masca_evt_toctexthash, masca_pause_toctexthash,
	masca_resume_toctexthash, masca_abort_toctexthash},
	{masca_get_blkrd_cmd, masca_tx_blkrd, masca_rx_blkrd,
	masca_evt_blkrd, masca_pause_blkrd, masca_resume_blkrd,
	masca_abort_blkrd}
};

static enum masca_recogn_stat	recogn_state = RECONG_START;
static enum masca_drv_mngr_stat	mngr_drv_stat = DRV_RECOGNIZING;
static enum masca_intrptr_decode_msg	prev_volt_evt;

static bool resume_drvctrl = FALSE;
static bool resume_toctexthash = FALSE;
static enum routetype route_set;

void masca_drv_mngr_init(const struct masca_configuration * const p_config)
{
	struct masca_last_evt last_evt;

	/*Initialise all slots over here*/
	masca_init_diag(p_config->diag_rty_cnt,
			p_config->diag_rst_cnt,
			p_config->diag_tmout);
	masca_init_drvctrl(p_config->drvctrl_rty_cnt,
			   p_config->drvctrl_rst_cnt,
			   p_config->drvctrl_tmout);
	masca_init_eject(p_config->eject_rty_cnt,
			 p_config->eject_rst_cnt,
			 p_config->eject_tmout);
	masca_init_toc_text_hash(p_config->toctext_rty_cnt,
				p_config->toctext_rst_cnt,
				p_config->toctext_tmout);
	masca_init_blkrd(BLK_READ_RETRY_COUNT, BLK_READ_RESET_COUNT,
					BLK_READ_RESP_TMOUT);
	masca_blkrd_set_spin_speed(SPIN_SPEED_HALFMAX);
	masca_evt_gen_init();

	mngr_drv_stat = DRV_RECOGNIZING;
	recogn_state = RECONG_START;
	resume_drvctrl = FALSE;
	resume_toctexthash = FALSE;
	masca_get_last_event(&last_evt);
	prev_volt_evt = last_evt.volt_event;

	/*According to masca specifications 100ms time gap is required
	before activating CD_RESET*/
	masca_util_sta_alm(DRIVE_RST_TIME, ALM_DRV_SM);
	route_set = ROUTE_MAX;
}

void masca_drv_mngr_deinit(void)
{
	masca_util_stp_alm(ALM_DRV_SM);
	mngr_drv_stat = DRV_RECOGNIZING;
	recogn_state = RECONG_START;
}

enum masca_error
masca_drive_mngr_cmd(const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_error     ret_err = MASCA_OK;
	enum masca_slot_num  slot_no;
	enum masca_cmd       input_cmd;

	p_response_info->drv_cmd_response = MAX_COMMAND;
	input_cmd = p_cmd_info->command;
	DRV_MNGR_TD_TRACE(
	"DRV_MNG,cmd = %d,mng_drv_stat = %d", input_cmd, mngr_drv_stat);
	if (RESET_DRIVE == input_cmd) {
		/*Do the silent reset of the drive i.e the
		* slots will pe paused and started after the
		* drive comes up*/
		p_response_info->drv_cmd_response = input_cmd;
		p_response_info->response_error = MASCA_PROCESSED;
		masca_exec_mngr_sm(0, TRUE, NULL, NULL);
	} else if (DRV_DISCONNECTED == mngr_drv_stat) {
		/*In this state drive itself is not connected.
		So the command is not taken in for further processing*/
		p_response_info->drv_cmd_response = input_cmd;
		p_response_info->response_error = MASCA_HE_COMM_FAIL;
	} else {
		/* The drive is connected, so the command can be
		  processed*/
		/* The commands can wakeup up the drive here if the
		drive is in sleep. So application should not give any
		commands if it wants the drive to be in sleep state*/
		slot_no = app_cmd_mapping[input_cmd].assign_slot;
		if (NULL != slot_funcs[slot_no].exec_func) {
			ret_err = slot_funcs[slot_no].exec_func(
					p_cmd_info, p_response_info);
		}
	}

	return ret_err;
}

enum masca_error
masca_drive_mngr_event(const unsigned int event_pattern,
			struct masca_drv_event_info * const p_evt_info,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_req = FALSE;
	struct masca_sm_state		sm_state;
	masca_sm_get_state(&sm_state);

	p_evt_info->drv_event = DRV_NO_EVENT;
	p_response_info->drv_cmd_response = MAX_COMMAND;
	switch (event_pattern) {
	case DRV_MNGR_RCV_MESSAGE:
		masca_handle_recp_events(p_evt_info, p_response_info);
		break;
	case DRV_MNGR_SM_UPDATE:
		masca_exec_mngr_sm(DRV_MNGR_SM_UPDATE, FALSE,
						NULL, p_response_info);
		break;
	case SLOT_TOC_TEXT_HASH_TMOUT:
		is_rst_req = masca_evt_toctexthash(
					SLOT_TOC_TEXT_HASH_TMOUT,
					p_response_info);
		break;
	case SLOT_EJECT_TMOUT:
		is_rst_req = masca_evt_eject(SLOT_EJECT_TMOUT,
						p_response_info);
		break;
	case SLOT_DIAGNOSTIC_TMOUT:
		is_rst_req = masca_evt_diag(SLOT_DIAGNOSTIC_TMOUT,
						p_response_info);
		break;
	case SLOT_DRV_CTRL_TMOUT:
		is_rst_req = masca_evt_drvctrl(SLOT_DRV_CTRL_TMOUT,
						p_response_info);
		break;
	case EVENT_Q_UP:
		masca_handle_event_q(EVENT_Q_UP, p_evt_info);
		break;
	case SLOT_RESPONSE_UPDATE:
		masca_hndl_slt_resp(SLOT_RESPONSE_UPDATE,
						p_response_info);
		break;
	case SLOT_ABORT_UPDATE:
		masca_hndl_slt_resp(SLOT_ABORT_UPDATE, p_response_info);
		break;
	case PLAY_RANGE_COMPLETE:
		p_evt_info->drv_event = DRV_PLAY_RANGE_END;
		break;
	case DISC_END_REACHED:
		p_evt_info->drv_event = DRV_DISC_END;
		/*end of medium reached*/
		masca_update_global_sense_buffer(0x06, 0x3B, 0x0F);
		break;
	case BLOCK_READ_COMPLETE:
	case BLOCK_READ_FAILED:
	case BLOCK_READ_REQ:
	case BLOCK_READER_IDLE:
	case BLOCK_READ_RESUME:
		is_rst_req = masca_evt_blkrd(event_pattern, p_response_info);
		break;
	default:
		break;
	}
	if (DRV_DISC_EJECTING == p_evt_info->drv_event) {
		/* Additionally we need to handle interdependencies
		 * between slots,i.e. when eject command is given slots
		 * except diagnostic are invalid*/
		masca_abort_drvctrl(p_response_info, MASCA_ABORT);
		masca_abort_toctexthash(p_response_info, MASCA_ABORT);
		masca_abort_blkrd(p_response_info, MASCA_ABORT);
		/* It is essential to reset the play state machine
		 * since the play state and positions will become
		 * invalid */
		masca_reset_play_sm();
		/*flush out data in cache*/
		/* masca_blkrdr_flush(); */
	}

	if ((sm_state.sm_pwr_state != SM_PWR_SLEEP)
			&& (TRUE == is_rst_req)) {
		/*Now do a silent reset*/
		masca_exec_mngr_sm(0, TRUE, NULL, NULL);
	}

	return MASCA_OK;
}

void masca_drive_mngr_abort(const enum masca_cmd abort_command)
{
	enum masca_slot_num slot_no;
	struct masca_drv_response_info slot_response;
	slot_response.drv_cmd_response = MAX_COMMAND;
	for (slot_no = SLOT_DIAGNOSTIC; slot_no < SLOT_MAX_NUM; slot_no++) {
		if (NULL != slot_funcs[slot_no].get_cmd_func) {
			if (abort_command ==
					slot_funcs[slot_no].get_cmd_func()) {
				if (NULL != slot_funcs[slot_no].abort_func)
					slot_funcs[slot_no].abort_func(
						&slot_response, MASCA_ABORT);
			}
		}
	}
}

void masca_drive_mngr_sleep(void)
{
	/*send a power down command over here*/
	struct masca_intrptr_cmd_info cmd_to_drive;
	cmd_to_drive.command = CMD_POWER_DOWN;
	masca_tx_cmd(&cmd_to_drive);
	/*stop the play state machine. The application should
	take care of resuming the play*/
	masca_stop_play_sm();
}

void masca_drive_mngr_wakeup(void)
{
	/*send a wake up command over here*/
	struct masca_intrptr_cmd_info cmd_to_drive;
	cmd_to_drive.command = CMD_VERSION_INFO;
	masca_tx_cmd(&cmd_to_drive);
}

void masca_drv_mngr_flush_buffers(void)
{
	masca_blkrdr_flush();
}

void masca_mngr_sm_cbk(void)
{
	masca_util_set_event(DRV_MNGR_SM_UPDATE);
}


enum masca_error masca_fill_slot(struct masca_cmd_slot * const p_slot,
			const struct masca_drv_cmd_info * const p_cmd_info,
			bool * const p_to_execute)
{
	enum masca_error ret_err = MASCA_BUSY;
	bool        exec_slot = FALSE;
	enum masca_cmd   app_cmd;

	app_cmd = p_slot->app_cmd_info.command;
	if (SLOT_PAUSED == p_slot->slot_stat) {
		if (MAX_COMMAND == app_cmd) {
			memcpy((&(p_slot->app_cmd_info)), p_cmd_info,
				sizeof(struct masca_drv_cmd_info));
			ret_err = MASCA_OK;
		}
	} else if (MAX_COMMAND == app_cmd) {
		memcpy((&(p_slot->app_cmd_info)), p_cmd_info,
				sizeof(struct masca_drv_cmd_info));
		exec_slot = TRUE;
		ret_err = MASCA_OK;
	} else {
		DRV_MNGR_TD_TRACE("other than SLOT_PAUSED\n");
		/*Nothing to do*/
	}

		*p_to_execute = exec_slot;

	return ret_err;
}

static void
masca_hndl_slt_resp(const unsigned int updt_evt_ptn,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_slot_num slot_no;
	for (slot_no = SLOT_DIAGNOSTIC; slot_no < SLOT_MAX_NUM; slot_no++)
		slot_funcs[slot_no].event_func(updt_evt_ptn, p_response_info);
}

/* masca_sm_update_ign_audio_state function is used to update
* the current status of igonre audio bit for mixed mode CD from drive */
static void masca_sm_update_ign_audio_state(char *p_status_msg)
{
	/* CD drive ignore audio has been enabled with TOC reading/TOC ready */
	if ((test_bit(MSG_STATUS_DISC_INSERTED,
					(unsigned long *)p_status_msg)) &&
		(test_bit(MSG_STATUS_IGNORE_AUDIO,
					(unsigned long *)p_status_msg)) &&
		(!(test_bit(MSG_STATUS_READING_CD,
					(unsigned long *)p_status_msg))) &&
		(!(test_bit(MSG_STATUS_TOC_READY,
					(unsigned long *)p_status_msg)))) {
		masca_sm_set_ign_aud_status(IGN_AUD_TOC_NOT_READY);
	} else {
		/* Ignore audio state */
		if (test_bit(MSG_STATUS_IGNORE_AUDIO,
					(unsigned long *)p_status_msg)) {
			if (test_bit(MSG_STATUS_READING_CD,
					(unsigned long *)p_status_msg)) {
				masca_sm_set_ign_aud_status(
						IGN_AUD_ENABLE_TOC_READ);
			} else if (test_bit(MSG_STATUS_TOC_READY,
					(unsigned long *)p_status_msg)) {
				masca_sm_set_ign_aud_status(
						IGN_AUD_ENABLE_TOC_READY);
			} else {
				masca_sm_set_ign_aud_status(IGN_AUD_ENABLE);
			}
		} else {
			if (test_bit(MSG_STATUS_READING_CD,
					(unsigned long *)p_status_msg)) {
				masca_sm_set_ign_aud_status(
						IGN_AUD_DISABLE_TOC_READ);
			} else if (test_bit(MSG_STATUS_TOC_READY,
					(unsigned long *)p_status_msg)) {
				masca_sm_set_ign_aud_status(
						IGN_AUD_DISABLE_TOC_READY);
			} else {
				masca_sm_set_ign_aud_status(IGN_AUD_DISABLE);
			}
		}
	}
}

static void masca_rcv_func_slot(
		struct masca_intrptr_msg_info_list * const p_list_msgs ,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_event_info * const p_evt_info,
		struct masca_drv_response_info * const p_response_info)
{
	bool	is_msg_prsnt;
	enum masca_slot_num	slot_no;
	struct masca_intrptr_rjt_cmd_list *p_rjt_cmds = NULL;
	union masca_intrptr_decode_info	msg_detail;

	is_msg_prsnt = masca_chk_msg(p_list_msgs, COMMAND_REJECTED);
	/*execute any updates to state machine*/
	masca_exec_mngr_sm(0, FALSE, p_list_msgs, p_response_info);
	/*Now handle the event function*/
	masca_chk_for_events(p_list_msgs, p_msg, p_evt_info);

	/*Check for responses if any*/
	if (TRUE == is_msg_prsnt) {
		if ((p_msg->p_buffer[DRIVE_MSG_ID] == MSG_STATUS_ID) &&
			(p_msg->p_buffer[DRIVE_REJ_CMD_ID_POS] ==
						COMMAND_PLAY_ID) &&
			(p_msg->p_buffer[DRIVE_REJ_CMD_ID_POS+0x02] ==
						COMMAND_PALY_CONT)) {
			/* Special handling for mixed mode CD to
			 *  ignore command rejected status
			 *  msg from drive during toc reading*/
		} else {
			(void)masca_intrptr_decode_msg_info(
					COMMAND_REJECTED, &msg_detail, p_msg);
			p_rjt_cmds = &(msg_detail.reject_cmds);
		}
	}

	for (slot_no = SLOT_DIAGNOSTIC; slot_no < SLOT_MAX_NUM; slot_no++) {
		slot_funcs[slot_no].rcv_func(p_list_msgs, p_rjt_cmds,
						p_msg, p_response_info);
	}
}

static void
masca_handle_recp_events(struct masca_drv_event_info * const p_evt_info,
			struct masca_drv_response_info * const p_response_info)
{
	int data_lng = 0;
	enum masca_error err;
	struct masca_intrptr_msg_info_list	list_msgs;
	struct masca_intrptr_msg_info		raw_message;
	enum masca_media_type cd_type;

	masca_sm_get_media_type(&cd_type);

	memset(p_raw_data, 0, MAX_CD_DRIVE_MSG_SZ);
	err = masca_prepare_recvbuf(p_raw_data, &data_lng);
	if (MASCA_OK == err) {
		raw_message.msg_length = (unsigned char)data_lng;
		raw_message.p_buffer = &p_raw_data[DRIVE_MSG_ID_BYTE];

		if (MSG_STATUS_ID == p_raw_data[DRIVE_MSG_ID_BYTE]) {
			/* Power state - stop mode */
			if (test_bit(MSG_STATUS_STOP,
					(unsigned long *)raw_message.p_buffer))
				masca_sm_update_power_status(SM_PWR_STANDBY);
			else
				masca_sm_update_power_status(SM_PWR_ACTIVE);

			if (DISC_MIXED_MODE == cd_type)
				masca_sm_update_ign_audio_state(
						&p_raw_data[DRIVE_MSG_ID_BYTE]);
			/* update Last session readability status */
			if (test_bit(MSG_STATUS_LAST_SESSION,
					(unsigned long *)raw_message.p_buffer))
				masca_sm_update_last_sess_state(OPEN);
			else
				masca_sm_update_last_sess_state(CLOSED);

			/* Drive status */
			if (test_bit(MSG_STATUS_POWER_FAIL,
					(unsigned long *) raw_message.p_buffer))
				masca_update_drive_status(DRV_MSG_POWER_FAIL);
			else if (test_bit(MSG_STATUS_READING_CD,
					(unsigned long *) raw_message.p_buffer))
				masca_update_drive_status(DRV_MSG_READING_CD);
			else if (test_bit(MSG_STATUS_TOC_READY,
					(unsigned long *) raw_message.p_buffer))
				masca_update_drive_status(DRV_MSG_TOC_READY);
			else if (test_bit(MSG_STATUS_DISC_INSERTED,
					(unsigned long *) raw_message.p_buffer))
				masca_update_drive_status(DRV_MSG_DISC_INS);
			else
				masca_update_drive_status(DRV_MSG_MAX);

			/* Play mode */
			if (test_bit(MSG_STATUS_CONT_PLAY,
					(unsigned long *) raw_message.p_buffer))
				masca_update_play_mode(PLAY_MODE_CONTINUOUS);
			else
				masca_update_play_mode(
						PLAY_MODE_NON_CONTINUOUS);
		}

		err = masca_intrptr_decode_msg_list(&list_msgs, &raw_message);
		if (MASCA_OK == err)
			masca_rcv_func_slot(&list_msgs, &raw_message,
						p_evt_info, p_response_info);
	} else {
		DRV_MNGR_TI_TRACE
			("MASCA_DRV_MNGR : Err - msg handler rcvis %d\n", err);
	}
}

void masca_exec_mngr_sm_extn(enum masca_intrptr_decode_msg found_msg, bool
is_msg_prsnt, const struct masca_intrptr_msg_info_list * const p_list)
{
	if (TRUE == is_msg_prsnt) {
		if (found_msg == DSP_INIT_SUCCESSFUL) {
			is_msg_prsnt = masca_chk_msg(p_list,
						MECHANICAL_INIT_COMPLETE);
			if (TRUE == is_msg_prsnt) {
				mngr_drv_stat = DRV_CONNECTED;
				DRV_MNGR_TD_TRACE("drv_connected\n");
				masca_clr_async_sense(SENSE_DRV_DISCONNECT);
			}
		} else {
			mngr_drv_stat = DRV_DISCONNECTED;
			/*start a timeout to check whether the drive comes up
			later point of time i.e polls the drive at regular
			intervals*/
			masca_util_sta_alm(DRIVE_POLL_TMOUT, ALM_DRV_SM);
			masca_set_async_sense(SENSE_DRV_DISCONNECT);
			}
	}

}


static void
masca_exec_mngr_sm(unsigned int sm_evt_ptn,
			const bool reset_drive,
			const struct masca_intrptr_msg_info_list * const p_list,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_msg_prsnt;
	struct masca_msg_search find_msg;
	enum masca_intrptr_decode_msg found_msg;
	enum masca_drv_mngr_stat prev_state = mngr_drv_stat;
	if (TRUE == reset_drive) {
		mngr_drv_stat = DRV_RECOGNIZING;
		recogn_state = RESET_DRV;
	}
	if (DRV_RECOGNIZING == mngr_drv_stat) {
		if (RECONG_START == recogn_state) {
			/*We enter this path during startup and when moving from
			disconnected state to DRV_RECOGNIZING state. As per
			masca specifications there should 100ms time before
			pulling up CD_RESET line.*/
			if (DRV_MNGR_SM_UPDATE == sm_evt_ptn)
				recogn_state = RESET_DRV;
		}
		if (RESET_DRV == recogn_state) {
			masca_hal_rst_pulse();
			recogn_state = WAIT_FOR_DRV_INIT;
			masca_util_sta_alm(DRIVE_INIT_TMOUT, ALM_DRV_SM);
		} else if (WAIT_FOR_DRV_INIT == recogn_state) {
			if (DRV_MNGR_SM_UPDATE == sm_evt_ptn) {
				mngr_drv_stat = DRV_DISCONNECTED;
				masca_set_async_sense(SENSE_DRV_DISCONNECT);
				/*start a timeout to check whether the drive
				comes up later point of time i.e polls the drive
				at regular intervals*/
			       masca_util_sta_alm(DRIVE_POLL_TMOUT, ALM_DRV_SM);
			} else {
				find_msg.num_messages = RANGE_COUNT;
				find_msg.search_msg[0] = DSP_INIT_SUCCESSFUL;
				find_msg.search_msg[1] = DSP_INIT_FAILURE;
				is_msg_prsnt = masca_is_msg_prsnt(&find_msg,
								  p_list,
								  &found_msg,
								  MASCA_RANGE);
				masca_exec_mngr_sm_extn(found_msg, is_msg_prsnt,
				p_list);
			}
		}
	} else if (DRV_CONNECTED == mngr_drv_stat) {
		/*as of now nothing to do in connected state*/
	} else {
		find_msg.num_messages = RANGE_COUNT;
		find_msg.search_msg[0] = DSP_INIT_SUCCESSFUL;
		find_msg.search_msg[1] = DSP_INIT_FAILURE;
		is_msg_prsnt = masca_is_msg_prsnt(&find_msg, p_list, &found_msg,
						  MASCA_RANGE);
		if (TRUE == is_msg_prsnt) {
			if (found_msg == DSP_INIT_SUCCESSFUL) {
				mngr_drv_stat = DRV_RECOGNIZING;
				recogn_state = WAIT_FOR_DRV_INIT;
				masca_util_sta_alm(DRIVE_INIT_TMOUT,
								ALM_DRV_SM);
				is_msg_prsnt = masca_chk_msg(p_list,
						MECHANICAL_INIT_COMPLETE);
				if (TRUE == is_msg_prsnt) {
					mngr_drv_stat = DRV_CONNECTED;
					DRV_MNGR_TD_TRACE("Drv_connected\n");
					masca_clr_async_sense(
							SENSE_DRV_DISCONNECT);
				}
			}
		} else if (DRV_MNGR_SM_UPDATE == sm_evt_ptn) {
			/*Try searching for drive again*/
			mngr_drv_stat = DRV_RECOGNIZING;
			recogn_state = RECONG_START;
			masca_util_sta_alm(DRIVE_RST_TIME, ALM_DRV_SM);
		}
	}
	masca_control_slots(prev_state, mngr_drv_stat, p_response_info);
}

static void
masca_control_slots(const enum masca_drv_mngr_stat prev_state,
			const enum masca_drv_mngr_stat current_state,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_slot_num slot_no;
	struct masca_last_evt last_evt;
	enum masca_intrptr_decode_msg current_volt_evt;
	if (prev_state != current_state) {
		if ((DRV_RECOGNIZING == prev_state)
			&& (DRV_CONNECTED   == current_state)) {
			/*we cannot resume all slots*/
			/*The slots which can be immediately resumed are
			* diag and eject*/
			masca_resume_eject(p_response_info);
			masca_resume_diag(p_response_info);
			resume_drvctrl = TRUE;
			resume_toctexthash = TRUE;
		} else if ((DRV_CONNECTED == prev_state)
				&& (DRV_RECOGNIZING   == current_state)) {
			for (slot_no = SLOT_DIAGNOSTIC;
				slot_no < SLOT_MAX_NUM; slot_no++) {
				if (NULL != slot_funcs[slot_no].pause_func)
					slot_funcs[slot_no].pause_func();
			}
		} else if ((DRV_DISCONNECTED == prev_state)
				&& (DRV_CONNECTED   == current_state)) {
			/*we cannot resume all slots*/
			/*The slots which can be immediately resumed are
			* diag and eject*/
			masca_resume_eject(p_response_info);
			masca_resume_diag(p_response_info);
			resume_drvctrl = TRUE;
			resume_toctexthash = TRUE;
		} else if (DRV_DISCONNECTED == current_state) {
			masca_set_async_sense(SENSE_DRV_DISCONNECT);
			/*Abort all slots*/
			resume_drvctrl = FALSE;
			resume_toctexthash = FALSE;
			for (slot_no = SLOT_DIAGNOSTIC;
				slot_no < SLOT_MAX_NUM; slot_no++) {
				if (NULL != slot_funcs[slot_no].abort_func) {
					slot_funcs[slot_no].abort_func(
						p_response_info, MASCA_IO_ERR);
				}
			}
		}
	}
	masca_get_last_event(&last_evt);
	current_volt_evt = last_evt.volt_event;
	if (prev_volt_evt != current_volt_evt) {
		if ((DRIVE_POWER_FAIL == prev_volt_evt)
			&& (DRIVE_POWER_OK   == current_volt_evt)) {
			if (DRV_CONNECTED == current_state) {
				masca_resume_eject(p_response_info);
				resume_drvctrl = TRUE;
				resume_toctexthash = TRUE;
			}
		} else if ((DRIVE_POWER_OK   == prev_volt_evt)
				&& (DRIVE_POWER_FAIL == current_volt_evt)) {
			masca_pause_eject();
			masca_pause_drvctrl();
			masca_stop_play_sm();
			masca_pause_toctexthash();
			masca_pause_blkrd();
		}
	}
	prev_volt_evt = current_volt_evt;
	if (TRUE == resume_drvctrl) {
		if ((DRIVE_POWER_OK == last_evt.volt_event)
			&& (TEMPERATURE_OK == last_evt.temp_event)
			&& (TOC_READY      == last_evt.toc_event)) {
			resume_drvctrl = FALSE;
			masca_resume_drvctrl(p_response_info);
		}
	}

	if (TRUE == resume_toctexthash) {
		if ((DRIVE_POWER_OK == last_evt.volt_event)
			&& (TEMPERATURE_OK == last_evt.temp_event)
			&& (TOC_READY      == last_evt.toc_event)) {
			resume_toctexthash = FALSE;
			masca_resume_toctexthash(p_response_info);
			masca_resume_blkrd(p_response_info);
		}
	}
}

/*
 * masca_drv_mngr_disable_cache
 *
 * Bocke read cache enable/diaable
 *
 * drv_mngr_disable_cache = TRUE : Disable cache
 * drv_mngr_disable_cache = FALSE : Enable cache
 *
 */
extern void masca_drv_mngr_disable_cache(bool drv_mngr_disable_cache)
{
	masca_drv_mngr_blkrd_cache_disable(drv_mngr_disable_cache);
}
