/*
 * MASCA DMA lib interface file. Handles the DMA transfer operations.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Sundhar Ashokan <sundhar.ashokan@in.bosch.com>
 *        : Saurabh Arora <saurabh.arora@in.bosch.com>
 *        : Dhanasekaran D <dhanasekaran.d@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#include "masca_common_types.h"
#include "masca_block_reader.h"
#include "masca_helper.h"
#include "masca_dma.h"
#include "masca_ssi.h"
#include "masca_trace.h"

/* Maximum processed sectors buffer size = 8 dma period size */
#define PRO_BUF_TH (DMA_PERIOD_SZ * 8)
/* Overlap processed sectors buffer limit = 12 dma period size */
#define OVERLAP_BUF_TH (DMA_PERIOD_SZ * 12)

static struct dma_mode_config gssi3_dma_cnfg;
static struct dma_mgr gdma_mgr;

static struct dma_buffer gdma_buf;

static bool filter(struct dma_chan *chan, void *param);
static int dma_tunable_params_init(void);
static void dma_tunable_params_deinit(void);
static unsigned int get_free_buf_sz(unsigned int start_idx,
					unsigned int end_idx);

void mascassi_dma_init_buf_idx(void)
{
	atomic_set(&gdma_buf.rd_idx, 0);
	atomic_set(&gdma_buf.wr_idx, 0);
	gdma_buf.wmark.unpro_end_sct = 0;
	gdma_buf.wmark.pro_st_sct = 0;
	gdma_buf.wmark.pro_st_idx = 0;
	gdma_buf.wmark.pro_end_sct = 0;
	gdma_buf.wmark.pro_end_idx = 0;
	gdma_buf.cb_cnt = 0;
	gdma_buf.prefetch = false;
}
EXPORT_SYMBOL(mascassi_dma_init_buf_idx);

/**
 * masca_init_dma() - masca DMA init
 * @return status	0 Success, Errorcode on failure
 *
 * This grabs a DMA channel and configures it.
 */
int masca_dma_init()
{
	dma_cap_mask_t mask;
	int ret = 0;

	if (dma_tunable_params_init() != 0)
		return -ENOMEM;

	mascassi_dma_init_buf_idx();
	spin_lock_init(&gdma_buf.lock);

	/* Request for a DMA channel and grab one */
	dma_cap_zero(mask);
	dma_cap_set(DMA_SLAVE, mask);

	gdma_mgr.chan = dma_request_channel(mask, filter, &gssi3_dma_cnfg.data);

	if (NULL == gdma_mgr.chan) {
		DMA_TR_A("Failed when requesting a DMA channel\n");
		return -EBUSY;
	} else {
		DMA_TR_D("Success when requesting DMA channel\n");
	}

	ret = dmaengine_slave_config(gdma_mgr.chan, &gssi3_dma_cnfg.slave);

	if (ret)
		DMA_TR_A("dmaengine_slave_config failed\n");
	else
		gdma_mgr.init = true;
	return ret;
}

/**
 * masca_deinit_dma() - masca DMA deinit
 * @return status	0 on successful deinit
 *
 * This terminates DMA operations and releases the DMA channel
 */
int masca_dma_deinit(void)
{
	int ret = 0;
	DMA_TR_I("%s\n", __func__);
	if (NULL != gdma_mgr.chan) {
		ret = dmaengine_terminate_all(gdma_mgr.chan);
		if (0 != ret)
			DMA_TR_E("Failed in terminate all err:%d\n", ret);

		dma_release_channel(gdma_mgr.chan);
		gdma_mgr.chan = NULL;

		dma_tunable_params_deinit();
	}
	gdma_mgr.init = false;
	return ret;
}

/**
 * ssi_start_dma() - masca DMA deinit
 * @return status	0 on successful deinit
 *
 * The DMA engine is kick started
 */
int masca_dma_start(void)
{
	if (false == gdma_mgr.init) {
		DMA_TR_E("DMA not initialized\n");
		return -EINVAL;
	}

	mascassi_dma_init_buf_idx();

	/*start cyclic DMA */
	if (NULL != gdma_mgr.desc)
		dmaengine_terminate_all(gdma_mgr.chan);

	gdma_mgr.desc = gdma_mgr.chan->device->device_prep_dma_cyclic(
			gdma_mgr.chan, gssi3_dma_cnfg.cyclic.dma_paddr,
			DMA_PERIOD_SZ * DMA_NUM_PERIODS,
			DMA_PERIOD_SZ, DMA_FROM_DEVICE,
			0,
			NULL);

	if (NULL == gdma_mgr.desc) {
		DMA_TR_E("device_prep_dma_cyclic failed\n");
		return -EINVAL;
	} else {
		gdma_mgr.desc->callback = gssi3_dma_cnfg.cyclic.callback;
		gdma_mgr.desc->callback_param = NULL;
		DMA_TR_D("Callback registered\n");
	}

	/* Start the DMA engine*/
	gdma_mgr.cookie = dmaengine_submit(gdma_mgr.desc);
	if (gdma_mgr.cookie > 0) {
		DMA_TR_I("Got a valid DMA cookie: %d\n", gdma_mgr.cookie);
		dma_async_issue_pending(gdma_mgr.chan);
		return 0;

	} else {
		DMA_TR_E("Invalid DMA cookie, err: %d\n", gdma_mgr.cookie);
		return gdma_mgr.cookie;
	}
}

int masca_dma_stop(void)
{
	if (true == gdma_mgr.init) {
		dmaengine_terminate_all(gdma_mgr.chan);
		DMA_TR_I(" Terminate DMA operations\n");
	}
	return 0;
}

/**
 * filter() -
 * @return status	0 on successful deinit
 *
 * i.MX provided function to identify the type of DMA device.
 * Needed by the SDMA library
 */
static bool filter(struct dma_chan *chan, void *param)
{
	if (!imx_dma_is_general_purpose(chan))
		return false;
	chan->private = param;
	return true;
}

static unsigned int get_free_buf_sz(unsigned int start_idx,
					unsigned int end_idx)
{
	unsigned int free_sz;
	if (start_idx <= end_idx)
		free_sz = DMA_BUF_SZ - (end_idx - start_idx);
	else /*wrap around*/
		free_sz = start_idx - end_idx;
	return free_sz;
}

static unsigned int pending_data(unsigned int rd_idx, unsigned int wr_idx)
{
	return ((wr_idx >= rd_idx) ?
			(wr_idx - rd_idx) : (DMA_BUF_SZ - rd_idx + wr_idx));
}

unsigned int predict_end_sector(struct buf_water_mark const wmark)
{
	unsigned int wr_idx = atomic_read(&gdma_buf.wr_idx);
	unsigned int unpro_sz = pending_data(wmark.pro_end_idx, wr_idx);

	return (unpro_sz/FRAME_SZ) + wmark.pro_end_sct;
}

/*
 * Threshold handling: Move the valid start sector index to create space
 * for new data, when at least 48 processed sectors available. Otherwise
 * terminate the DMA processing & stop the drive until all fetched
 * sectors are processed.
 */
bool update_water_mark(struct buf_water_mark const wmark)
{
	bool is_idle = false;
	gdma_buf.wmark.unpro_end_sct = predict_end_sector(wmark);
	if (get_free_buf_sz(wmark.pro_st_idx, atomic_read(&gdma_buf.wr_idx))
			< OVERLAP_BUF_TH) {
		/*processed data*/
		if (pending_data(wmark.pro_st_idx, wmark.pro_end_idx)
				< PRO_BUF_TH) {
			is_idle = true;
			DMA_TR_I("DMA threshold reached!!!\n");
		} else {
			/*skip 2 period of data*/
			gdma_buf.wmark.pro_st_sct +=  PERIOD_SZ * 2;
			gdma_buf.wmark.pro_st_idx += DMA_PERIOD_SZ * 2;
			gdma_buf.wmark.pro_st_idx %= DMA_BUF_SZ;
			DMA_TR_I("DMA watermark changed!!!\n");
		}
	}
	return is_idle;
}

/**
 * dma_end_callback() -
 * @return void
 *
 * This function is invoked when one PERIOD size of data is received from SSI
 * FIFO to the  DMA memory. Please note, this is called in interrupt context
 * so keeping it minimal by just updating hwoff which points to the new source
 * address to be read.
 */
void masca_dma_period_callback(void)
{
	unsigned long flags;
	bool stop_dma = false;
	unsigned int wr_idx = 0;

	gdma_buf.cb_cnt++;
	wr_idx = gdma_buf.cb_cnt * DMA_PERIOD_SZ;

	if (wr_idx >= DMA_BUF_SZ) {
		wr_idx = 0;
		gdma_buf.cb_cnt = 0;
		DMA_TR_I("DMA wraparound occured!!! rd_idx:0x%x\n",
				atomic_read(&gdma_buf.rd_idx));
	}
	atomic_set(&gdma_buf.wr_idx, wr_idx);

	spin_lock_irqsave(&gdma_buf.lock, flags);
	if (gdma_buf.wmark.pro_end_sct && gdma_buf.wmark.pro_end_idx)
		stop_dma = update_water_mark(gdma_buf.wmark);
	spin_unlock_irqrestore(&gdma_buf.lock, flags);

	/* Trigger Block-Reader task about data arrival or stop */
	if (stop_dma) {
		gdma_mgr.blk_evnt_clbk(BLKRDR_IDLE);
		gdma_buf.prefetch = true;
	}

	if (gdma_buf.prefetch == false)
		gdma_mgr.blk_evnt_clbk(BLKRDR_READ_DATA);
}

int dma_tunable_params_init(void)
{
	int ret = 0;
	gssi3_dma_cnfg.data.peripheral_type = IMX_DMATYPE_SSI;
	gssi3_dma_cnfg.data.priority = DMA_PRIO_LOW;
	/*
	 * This info is obtained from i.MX ref manual. Chapter 3.
	 * Interrupt and DMA events for specific SSI used.
	 * eg. 45 SSI-3 SSI-3 receive 0 DMA request
	 */
	gssi3_dma_cnfg.data.dma_request = 45;
	/*config dma slave datas - peripheral specific infos */
	gssi3_dma_cnfg.slave.direction = DMA_DEV_TO_MEM;
	/*Source to read the data. Eg.SSI_RX_FIFO address should be fed*/
	gssi3_dma_cnfg.slave.src_addr = SSI3_BASE_ADDR + SSI_SRX0;
	/*Number of bytes/word. 32 bit not supported by SSI */
	gssi3_dma_cnfg.slave.src_addr_width = 2;
	/*depends on SSI FIFO depth.TBD 0,1,2,4,8 */
	gssi3_dma_cnfg.slave.src_maxburst = 8;
	/*PERIOD size .i.e callback received after this much bytes received*/
	gssi3_dma_cnfg.cyclic.dma_period_size = DMA_PERIOD_SZ;
	/*Number of periods after which it will wrap around */
	gssi3_dma_cnfg.cyclic.num_periods = DMA_NUM_PERIODS;
	/*TODO: Choose timeout considering the drive behaviour and data rate */

	/* Allocate DMA memory */
	gssi3_dma_cnfg.cyclic.dma_vaddr =
		dma_alloc_coherent(NULL, DMA_NUM_PERIODS * DMA_PERIOD_SZ,
				&(gssi3_dma_cnfg.cyclic.dma_paddr),
				GFP_KERNEL | GFP_DMA);

	if (gssi3_dma_cnfg.cyclic.dma_vaddr == NULL) {
		DMA_TR_A("Failed when allocating DMA memory\n");
		ret = -ENOMEM;
	} else {
		DMA_TR_I("Success when allocating DMA memory\n");
		gssi3_dma_cnfg.cyclic.callback =
			(dma_async_tx_callback)masca_dma_period_callback;
		gdma_buf.buf = gssi3_dma_cnfg.cyclic.dma_vaddr;
	}
	return ret;
}


void dma_tunable_params_deinit(void)
{
	/*Deallocate the DMA memory */
	if (NULL != gssi3_dma_cnfg.cyclic.dma_vaddr) {
		dma_free_coherent(NULL, DMA_NUM_PERIODS * DMA_PERIOD_SZ,
				gssi3_dma_cnfg.cyclic.dma_vaddr,
				gssi3_dma_cnfg.cyclic.dma_paddr);
		DMA_TR_I("released DMA buff\n");
		gssi3_dma_cnfg.cyclic.dma_vaddr = NULL;
	}

	gdma_buf.buf = NULL;
}

struct dma_buffer *masca_dma_get_buffer(void)
{
	return &gdma_buf;
}
EXPORT_SYMBOL(masca_dma_get_buffer);

void masca_dma_set_clbks(void (*blk_evnt_clbk)(const unsigned char evt_pattern))
{
	gdma_mgr.blk_evnt_clbk = blk_evnt_clbk;
}
EXPORT_SYMBOL(masca_dma_set_clbks);
