/*
 * Forward a command to drive manager or state machine depend on the type
 * of command received
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_blkdev.h"
#include "masca_sm.h"
#include "masca_event_config.h"
#include "masca_interpreter.h"
#include "masca_drv_mngr_slot.h"

static struct masca_cmd_lst_elem *p_frst_elem;
static struct masca_cmd_lst_elem *p_last_elem;
static bool	abort_executing_cmd;
static bool	abort_pending_cmd;
static bool	abort_cmd_search;
unsigned int	auto_reinsert_tmout;
unsigned int	lifetime_tmout;

static void masca_prepare_response(
			const struct masca_drv_response_info * const p_res_info,
			struct masca_output * const p_output);

static void masca_cmd_drive_mngr(
				const struct masca_cmd_params * const p_cmd,
				struct masca_output * const p_output);

static struct masca_cmd_lst_elem *
masca_get_elem_by_id(const unsigned int hash);

static struct masca_cmd_lst_elem *
masca_get_frst_elem(const enum masca_cmd_lst_stat elem_stat);

static void masca_empty_lst_elem(
		const struct masca_cmd_lst_elem * const p_lst_to_empty);

static void masca_add_to_lst(
			const struct masca_cmd_params * const p_cmd_info,
			const enum masca_cmd_lst_stat elem_stat);

static struct masca_cmd_lst_elem *
masca_get_elem_by_cmd(const enum masca_cmd cmd,
				const enum masca_cmd_lst_stat elem_stat);

static bool masca_is_cmd_allowed(const enum masca_cmd app_cmd,
					enum masca_error *err);

static enum masca_error masca_get_cd_info(enum masca_cmd,
					struct masca_output * const p_output);

/* As per above array table between command and different drive state */
static bool cmd_drive_state[DIAG_GET_TEMPERATURE][SM_MAX_DRV_STATE] = {
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{1,  1,  1,  1,  1,  1},/*Insert*/
	{1,  1,  1,  1,  1,  1},/*Eject*/
	{1,  1,  1,  1,  1,  1},/*PWR_ACTIVE*/
	{1,  0,  0,  0,  1,  0},/*PWR_STANDBY*/
	{1,  0,  0,  0,  1,  0},/*PWR_STANDBY_NO_CD*/
	{1,  1,  1,  1,  1,  1},/*PWR_SLEEP*/
	{0,  0,  0,  0,  1,  0},
	{1,  1,  1,  1,  1,  1},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{1,  1,  1,  1,  1,  1},
	{1,  1,  1,  1,  1,  1},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  0,  0},
	{0,  0,  0,  0,  1,  0},
	{1,  1,  1,  1,  1,  1},
	{1,  1,  1,  1,  1,  1},/* EN_IGN_AUDIO */
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},/* Test unit ready */
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},
	{0,  0,  0,  0,  1,  0},/* Send Diagnostics */
	{1,  1,  1,  1,  1,  1},/* Receive Diagnostics */
	{0,  0,  0,  0,  1,  0},/* READ_DISK_INFO */
	{1,  1,  0,  0,  1,  1},/* Get Event */
	{0,  0,  0,  0,  1,  0} /* READ_SUBCHANNEL */
};

/* As per above array table between command and different driver state */
static bool cmd_driver_state[DIAG_GET_TEMPERATURE][MAX_MODULE_STATE] = {
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{1,  1},/*Insert*/
	{0,  1},/*Eject*/
	{0,  1},/*PWR_ACTIVE*/
	{0,  1},/*PWR_STANDBY*/
	{0,  1},/*PWR_STANDBY_NO_CD*/
	{0,  1},/*PWR_SLEEP*/
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  1},
	{0,  0},
	{0,  1},
	{1,  1},
	{1,  1},/*EN_IGN_AUDIO*/
	{1,  1},
	{0,  1},
	{1,  1},/* Test unit ready */
	{1,  1},
	{1,  1},
	{0,  1},
	{0,  1},/* Send Diagnostics */
	{0,  1},/* Receive Diagnostics */
	{0,  1},/* READ_DISK_INFO */
	{1,  1}, /* Get Event Dummy entry, allowed always */
	{0,  1} /* READ_SUBCHANNEL */
};

static bool cmd_volt_state(const enum masca_cmd cmd,
				enum masca_sm_voltage_state volt_state)
{
	bool is_cmd_allowed = TRUE;

	if ((volt_state == UNDER_VOLTAGE)
			|| (DRV_MSG_POWER_FAIL == masca_get_drive_status())) {
		if (!(((cmd >= PWR_ACTIVE) && (cmd <= PWR_SLEEP))
				|| (CMD_SEND_DIAGNOSTIC == cmd)
				|| (CMD_RECEIVE_DIAGNOSTIC == cmd)))
			is_cmd_allowed = FALSE;
	}

	return is_cmd_allowed;
}

static bool cmd_disc_state(const enum masca_cmd cmd,
				enum masca_sm_disc_state disc_state)
{
	bool is_cmd_allowed = TRUE;

	if ((cmd == READ_SECTORS_HASH) ||
			((cmd == PAUSE) && (disc_state == SM_STOPPED)))
		is_cmd_allowed = FALSE;

	return is_cmd_allowed;
}

/**
 * \func    masca_cmd_hndlr_init
 *
 * Initialize masca command handler by means of configure and initialize
 * alarm handler for reinsert and stop commands.
 *
 */
void masca_cmd_hndlr_init(const struct masca_configuration * const p_config)
{
	struct masca_cmd_lst_elem *p_elem;
	unsigned int         elem_count;
	unsigned int         lists_to_construct;

	auto_reinsert_tmout = (unsigned int)p_config
					->auto_reinsrt_tmout;
	lifetime_tmout = (unsigned int)p_config->life_time_tmout;
	abort_executing_cmd = FALSE;
	abort_pending_cmd = FALSE;
	abort_cmd_search = FALSE;
	p_elem = p_config->p_cmd_hndlr_lst;
	/*Remember the first element of linked list*/
	p_frst_elem = p_elem;
	p_last_elem = p_elem;
	if (NULL != p_elem) {
		lists_to_construct = (unsigned int)
					(p_config->pllel_maxreq - 1);
		/*create singly linked list*/
		for (elem_count = 0; elem_count < lists_to_construct;
							elem_count++) {
			p_elem[elem_count].p_nxt =
						&p_elem[elem_count + 1];
			p_elem[elem_count].elem_stat = EMPTY;
			}
			p_elem[elem_count].p_nxt = NULL;
		p_elem[elem_count].elem_stat = EMPTY;
	}
}

static enum masca_error masca_get_cd_info(enum masca_cmd cmd,
					struct masca_output * const p_output)
{
	enum masca_error ret_err = MASCA_PTR_NULL;
	if (cmd == GET_CD_CAPACITY)
		ret_err = masca_sm_get_cd_capacity(
					&p_output->output_params.cd_capacity);
	else if (cmd == GET_TOC)
		ret_err = masca_sm_get_toc(&p_output->output_params.p_cd_toc);
	else if (cmd == GET_SESSION_INFO)
		ret_err = masca_sm_get_session_info(
				&p_output->output_params.p_cd_session_info);
	else if (cmd == GET_TITLE_ALBUM_AUTHOR)
		ret_err = masca_sm_get_cd_text(
					&p_output->output_params.p_cd_text);

	return ret_err;
}


/**
 * \func    masca_cmd_hndlr_cmd
 *
 * Forward a command to drive manager or state machine
 * depend on the type of command received.With the help of
 * command/state machine table and updated state machine state,
 * command allowed for further process.
 *
 * \param p_cmd     : pointer to the memory location holding
 *                    command information structure.
 *
 * \param p_output  : pointer to the output structure holding
 *                    command response information.
 *
 * \return
 *  MASCA_PROCESSED : when the command is successfully processed and reply is
 *                    also given out.
 *  MASCA_ACCEPTED  : when the command is accepted for processing.
 *  MASCA_REJECTED  : Command is rejected in the case the queue is full.
 *  MASCA_Q_FULL    : command is accepted but no further commands can be
 *                    accepted unless some of the commands are replied.
 *  MASCA_PTR_NULL  : If any of the parameters passed is NULL.
 *  MASCA_ABORT     : If the particular command is aborted
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_OK        : If message receiving is successful.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 *  MASCA_INVALID_STATE     : To process the command particular state is
 *                            not valid.
 *  MASCA_UNKNOWN_MSG_CMD   : If the command sent is an invalid command.
 */
extern enum masca_error masca_cmd_hndlr_cmd(
				const struct masca_cmd_params * const p_cmd,
				struct masca_output * const p_output)
{
	bool			update_resp = TRUE;
	bool			is_cmd_allowed = TRUE;
	struct masca_sm_state	sm_state;
	enum masca_error	ret_err = MASCA_PTR_NULL;
	enum masca_cmd		app_cmd = p_cmd->command;
	masca_sm_get_state(&sm_state);

	if (!((PWR_SLEEP == app_cmd) || (PWR_ACTIVE == app_cmd))) {
		ret_err = masca_check_async_sense();
		if ((ret_err == MASCA_HE_COMM_FAIL)
				|| (ret_err == MASCA_HE_VOLTAGE_FAULT))
			is_cmd_allowed = FALSE;
		else
			is_cmd_allowed = masca_is_cmd_allowed(p_cmd->command,
					&ret_err);
	}
	if (TRUE == is_cmd_allowed) {
		if (app_cmd == PWR_SLEEP) {
			masca_update_power_evt(PWRCHG_SUCCESS);
			masca_sm_update_power_status(SM_PWR_SLEEP);
			ret_err = MASCA_OK;
		} else if (((app_cmd == PWR_STANDBY)
			|| (app_cmd == PWR_STANDBY_NO_CD))
			&& (sm_state.sm_pwr_state == SM_PWR_STANDBY)) {
			masca_update_power_evt(PWRCHG_SUCCESS);
			masca_sm_update_power_status(SM_PWR_STANDBY);
			ret_err = MASCA_OK;
		} else if ((app_cmd <= READ_SECTORS)
			|| (app_cmd == GET_VERSION_INFO)
			|| (app_cmd == RESUME)) {
			/*This is a command to drive manager*/
			masca_cmd_drive_mngr(p_cmd, p_output);
			update_resp = FALSE;
		} else if (app_cmd == GET_MEDIA_STAT) {
			p_output->output_params.mda_stat =
						masca_sm_get_media_status();
		} else if (app_cmd == CMD_SET_CD_SPEED) {
			p_output->output_params.spin_speed =
						p_cmd->cmd_params.cd_speed;
			masca_blkrd_set_spin_speed(p_cmd->cmd_params.cd_speed);
		} else if ((app_cmd >= GET_TEST_UNIT_READY) &&
				(app_cmd <= READ_SUBCHANNEL)) {
			ret_err = MASCA_OK;
		} else if ((app_cmd >= GET_CD_CAPACITY) &&
				(app_cmd <= GET_TITLE_ALBUM_AUTHOR)) {
			ret_err = masca_get_cd_info(app_cmd, p_output);
		} else {
			CMD_TE_TRACE(
			"SM allowed,but %d cmd isnt executed", app_cmd);
			update_resp = FALSE;
		}
	}

	if (update_resp != FALSE) {
		p_output->replied_command = app_cmd;
		p_output->response_id = p_cmd->reply_id;
		p_output->reply_status = ret_err;
		if (app_cmd == CMD_SEND_DIAGNOSTIC)
			p_output->reply_status = MASCA_OK;
	}

	return ret_err;
}

/**
 * \func    masca_cmd_hndlr_cmd_reply
 *
 * This API is called when reply to a command is available.
 * If the CD_Drv_Manager replies to a command
 * then it is routed to cmd_handler.
 *
 * \param p_response_info   : pointer to the memory location holding
 *                            response information from CD_Drv_Manager.
 *
 * \param p_output          : pointer to the output structure holding
 *                            command response information.
 *
 * \return
 *  MASCA_PROCESSED : The command is processed successfully.
 *  MASCA_ABORT     : If the particular command is aborted.
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_PTR_NULL  : If any of the parameters are NULL.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 */
void masca_cmd_hndlr_cmd_reply(
		const struct masca_drv_response_info * const p_response_info,
		struct masca_output * const p_output)
{
	/*Check if this particular command exists in the queue
	* and fill out the response structure*/
	masca_prepare_response(p_response_info, p_output);
	/*Check if there are any commands pending*/
	masca_util_set_event(CMD_FROM_QUEUE);
}

void masca_cmd_hndlr_extn(struct masca_output * const p_output,
				struct masca_drv_cmd_info manager_cmd,
				struct masca_cmd_lst_elem *p_elem,
				struct masca_drv_response_info manager_response)
{
	if (manager_response.drv_cmd_response != p_elem->acpptd_cmd.command) {
		/*Check whether we got response to any othercommand & update*/
		masca_prepare_response(&manager_response, p_output);
	} else {/*The command is successfully processed.remove from list*/
		p_output->replied_command = manager_cmd.command;
		p_output->response_id = p_elem->acpptd_cmd.reply_id;
		p_output->reply_status = manager_response.response_error;
		p_output->output_params.cd_version_info =
				manager_response.response_param.cd_version_info;
		/*clear out the entry in our list*/
		masca_empty_lst_elem(p_elem);
	}
}

/**
 * \func    masca_cmd_hndlr_evt
 *
 * This API is called in case of an event. When cmd_handler
 * has events then it posts to GDI handler's function
 * which is ultimately routed through this API.
 *
 * \param evt_pattern   : event pattern of command handler
 *
 * \param p_output      : pointer to the output structure holding
 *                            command response information.
 *
 * \return
 *  MASCA_PROCESSED : The command is processed successfully.
 *  MASCA_ABORT     : If the particular command is aborted.
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_PTR_NULL  : If any of the parameters are NULL.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 */
enum masca_error masca_cmd_hndlr_evt(const unsigned int evt_pattern,
					struct masca_output * const p_output)
{
	struct masca_sm_state		sm_state;
	struct masca_drv_cmd_info	manager_cmd;
	struct masca_drv_response_info	manager_response;
	struct masca_cmd_lst_elem	*p_elem;
	enum masca_error		ret_err;

	/* Initialize structure to NULL */
	memset(&sm_state, 0, sizeof(sm_state));

	masca_sm_get_state(&sm_state);

	switch (evt_pattern) {
	case 0:
		/* Update state machine */
		masca_sm_get_state(&sm_state);

		if ((sm_state.sm_temp_state == OVER_TEMPERATURE)
		|| (sm_state.sm_volt_state == UNDER_VOLTAGE)
		|| (sm_state.sm_driver_state == SUSPENDED)) {
			/*setting both of these to FALSE will result in pause
			of the commands to drive manager i.e the commands will
			just be aborted in drive manager and are kept in the
			queue as pending*/
			abort_executing_cmd = TRUE;
			abort_pending_cmd = TRUE;
			masca_util_set_event(CMD_ABORT);
		} else if ((sm_state.sm_temp_state == NORMAL_TEMPERATURE) &&
				(sm_state.sm_volt_state == NORMAL_VOLTAGE) &&
				(sm_state.sm_driver_state == NORMAL)) {
			masca_util_set_event(CMD_FROM_QUEUE);
		}
		break;
	case CMD_HNDLR_DSK_PAUSE:
		masca_util_sta_alm(lifetime_tmout, ALM_STOP_DRV);
		masca_util_clr_event(CMD_HANDLER_DSK_STOP);
		break;
	case CMD_HANDLER_REINSERT:
		if (sm_state.sm_drive_state == SM_EJECTING) {
			/*Whatever the state it is we will reinsert the
			* disc keeping safety in mind*/
			manager_cmd.command = INSERT;
			/*This command is valid at this moment and also
			* it is a drive control command.*/
			manager_response.drv_cmd_response = MAX_COMMAND;
			(void)masca_drive_mngr_cmd(&manager_cmd,
						   &manager_response);
			masca_prepare_response(&manager_response, p_output);
		}
		break;
	case CMD_HANDLER_DSK_STOP:
		if (sm_state.sm_disc_state == SM_PAUSED) {
			manager_cmd.command = STOP_DISC;
			/*Whatever the state it is we should stop the disc.
			*As under suspende/undervolt/overtemperature it makes
			*sense to stop the disc*/
			manager_response.drv_cmd_response = MAX_COMMAND;
			(void)masca_drive_mngr_cmd(&manager_cmd,
			&manager_response);
			masca_prepare_response(&manager_response, p_output);
		}
		break;
	case CMD_FROM_QUEUE:
		/*Get the next pending element from the list*/
		/*Only commands sent to drive manager need to be queued*/
		p_elem = masca_get_frst_elem(PENDING);
		if (NULL != p_elem) {
			if (TRUE == masca_is_cmd_allowed(
					p_elem->acpptd_cmd.command, &ret_err)) {
				memcpy((void *)&manager_cmd.command_param
								     .drv_param,
				       (void *)&p_elem->acpptd_cmd.cmd_params,
				       sizeof(p_elem->acpptd_cmd.cmd_params));
				manager_cmd.command =
						p_elem->acpptd_cmd.command;
				manager_response.drv_cmd_response = MAX_COMMAND;

				ret_err = masca_drive_mngr_cmd(&manager_cmd,
				&manager_response);
				if (MASCA_BUSY == ret_err) {
					/*Check whether we got response
					  to any other commandand update*/
					masca_prepare_response(&manager_response
								, p_output);
				} else if ((MASCA_OK == ret_err) ||
				(MASCA_ACCEPTED == ret_err) ||
				(MASCA_PROCESSED == ret_err)) {
					/*command is accepted by drive manager*/
					p_elem->elem_stat = EXECUTING;
					masca_cmd_hndlr_extn(p_output,
					manager_cmd, p_elem, manager_response);
					/*Check for any next pending command*/
					if (NULL !=
						masca_get_frst_elem(PENDING)) {
						masca_util_set_event(
								CMD_FROM_QUEUE);
					}
				} else {
					CMD_TA_TRACE(
					"drv mngr return err = %d", ret_err);
				}
			} else {
				p_output->replied_command = p_elem->acpptd_cmd
								       .command;
				p_output->response_id = p_elem->acpptd_cmd
								      .reply_id;
				p_output->reply_status = ret_err;
				/*clear out the entry in our list*/
				masca_empty_lst_elem(p_elem);
			}
		}
		break;
	case CMD_ABORT:
		abort_cmd_search = TRUE;
		if (TRUE == abort_executing_cmd) {
			p_elem = masca_get_frst_elem(EXECUTING);
			if (NULL != p_elem) {
				masca_drive_mngr_abort(
						    p_elem->acpptd_cmd.command);
				p_output->replied_command = p_elem->acpptd_cmd
								       .command;
				p_output->response_id = p_elem->acpptd_cmd
								      .reply_id;
				p_output->reply_status = MASCA_ABORT;
				masca_empty_lst_elem(p_elem);
			} else {
				/*all executing commands aborted*/
				abort_executing_cmd = FALSE;
			}
			masca_util_set_event(CMD_ABORT);
		} else if (TRUE == abort_pending_cmd) {
			p_elem = masca_get_frst_elem(PENDING);
			if (NULL != p_elem) {
				masca_drive_mngr_abort(
						    p_elem->acpptd_cmd.command);
				p_output->replied_command = p_elem->acpptd_cmd
								       .command;
				p_output->response_id = p_elem->acpptd_cmd
								      .reply_id;
				p_output->reply_status = MASCA_ABORT;
				masca_empty_lst_elem(p_elem);
				masca_util_set_event(CMD_ABORT);
			} else {
				/*all executing commands aborted*/
				abort_pending_cmd = FALSE;
			}
		} else {
			/*Just abort the executing commands from drive manager
			* and keep them in the list*/
			p_elem = masca_get_frst_elem(EXECUTING);
			while (NULL != p_elem) {
				p_elem->elem_stat = PENDING;
				masca_drive_mngr_abort(
						    p_elem->acpptd_cmd.command);
				p_elem = masca_get_frst_elem(EXECUTING);
			}
		}
		abort_cmd_search = FALSE;
		break;
	default:
		break;
	}

	return MASCA_OK;
}

/**
 * \func    masca_cmd_hndlr_abort
 *
 * This API aborts the execution of particular command corresponding
 * to the request ID. The request is a unique ID for each of the commands.
 * It is given when the command was sent to cmd_handler.
 *
 * \param req_id    : Request/command ID which needs to be aborted
 *
 * \return
 *  MASCA_PROCESSED     : If the API is successfully executed.
 *  MASCA_INVALID_PARAM : If the particular request ID is not pending or
 *                        executing.
 *
 */
extern enum masca_error masca_cmd_hndlr_abort(const unsigned int req_id)
{
	enum masca_error ret_err = MASCA_INVALID_PARAM;
	struct masca_cmd_lst_elem *p_elem;
	p_elem = masca_get_elem_by_id(req_id);
	if (NULL != p_elem) {
		ret_err = MASCA_OK;
		if (EXECUTING == p_elem->elem_stat) {
			masca_drive_mngr_abort(
						p_elem->acpptd_cmd.command);
		}
		masca_empty_lst_elem(p_elem);
	}
	return ret_err;
}


/**
 * \func    masca_cmd_hndlr_deinit
 *
 * Deinitialize masca command handler by means of delete
 * alarm handler for reinsert and stop commands.
 *
 */
void masca_cmd_hndlr_deinit(void)
{
	masca_util_stp_alm(ALM_STOP_DRV);
	masca_util_stp_alm(ALM_AUTO_REINSERT);
}

/**
 * \func    auto_stop_alarm_hndlr
 *
 * start an alarm handler
 *
 */
extern void masca_auto_stop_alm_hndlr(void)
{
	masca_util_set_event(CMD_HANDLER_DSK_STOP);
}

/**
 * \func    auto_insert_alarm_hndlr
 *
 * start an alarm handler
 *
 */

extern void masca_auto_reinsrt_alm_hndlr(void)
{
	masca_util_set_event(CMD_HANDLER_REINSERT);
}

enum masca_error masca_get_driver_sense(enum masca_sm_module_state drvr_st,
					const enum masca_cmd app_cmd)
{
	enum masca_error err = MASCA_IR_INVALID_FLD_IN_CDB;

	/* TODO: Need to be implemented
	 * SUSPENDED and NORMAL states */

	return err;
}

enum masca_error masca_get_drive_sense(enum masca_sm_drive_state drv_state,
					const enum masca_cmd app_cmd)
{
	enum masca_error err = MASCA_IR_INVALID_FLD_IN_CDB;
	enum masca_drive_status drv_status = masca_get_drive_status();

	if ((SM_NO_CD == drv_state) || (SM_EJECTING == drv_state))
		err = MASCA_NR_NO_MEDIUM;
	else if ((SM_INSERTING == drv_state) || (SM_INSERTED == drv_state)
			|| (DRV_MSG_DISC_INS == drv_status))
		err = MASCA_NR_INPROGRESS_OF_RDY;
	else if ((SM_LOADING == drv_state)
			|| (DRV_MSG_READING_CD == drv_status))
		err = MASCA_NR_OPR_IN_PROGRESS;
	else if ((SM_ACCESSIBLE == drv_state)
			|| (DRV_MSG_TOC_READY == drv_status))
		err = MASCA_OK;
	return err;
}

enum masca_error masca_get_disc_sense(enum masca_sm_disc_state disc_st,
					const enum masca_cmd app_cmd)
{
	enum masca_error err = MASCA_IR_INVALID_FLD_IN_CDB;

	if (SM_PLAYING == disc_st)
		err = MASCA_IR_PLY_PROGRESS;
	if (SM_PAUSED == disc_st) {
		if (READ_SECTORS_HASH == app_cmd)
			err = MASCA_OK;
		else if (INSERT == app_cmd)
			err = MASCA_IR_CMD_SEQ_ERR;
	}
	if (SM_STOPPED == disc_st) {
		if (PAUSE == app_cmd)
			err = MASCA_UA_STDBY_TMR;
		else if (READ_SECTORS_HASH == app_cmd)
			err = MASCA_OK;
		else if (INSERT == app_cmd)
			err = MASCA_UA_STDBY_TMR;
	}

	return err;

}

enum masca_error masca_get_sm_sense(int sm_sen, struct masca_sm_state sm_st,
					const enum masca_cmd app_cmd)
{
	enum masca_error err = MASCA_IR_INVALID_FLD_IN_CDB;

	if (SENSE_DRIVER_ERR  & sm_sen)
		err = masca_get_driver_sense(sm_st.sm_driver_state,
							app_cmd);
	else if (SENSE_DRIVE_ERR & sm_sen)
		err = masca_get_drive_sense(sm_st.sm_drive_state,
							app_cmd);
	else if (SENSE_DISC_ERR & sm_sen)
		err = masca_get_disc_sense(sm_st.sm_disc_state,
							app_cmd);
	return err;
}

enum masca_error masca_get_sense(int sm_sen, struct masca_sm_state sm_st,
					const enum masca_cmd app_cmd)
{
	enum masca_error err = MASCA_IR_INVALID_FLD_IN_CDB;

	/* Check for async events as they are high priority,
	 * if there is no error [MASCA_OK] then check the below sense */
	err = masca_check_async_sense();
	if ((MASCA_HE_COMM_FAIL != err)
			&& (MASCA_HE_VOLTAGE_FAULT != err)) {
		/* check for media related commands */
		/* TODO: This logic need to be changed */
		if ((MASCA_HE_LOAD_EJECT_FAIL == err)
			|| (MASCA_NR_UNKNOWN_FORMAT == err)
			|| (MASCA_ME_TOC_ERROR == err)) {
			if (((PLAY_LBA <= app_cmd) && (FBWD_LBA >= app_cmd)) ||
				(STOP == app_cmd) ||
				(GET_CD_CAPACITY == app_cmd) ||
				(GET_TEST_UNIT_READY == app_cmd) ||
				(CMD_GET_EVT_STAT == app_cmd) ||
				((GET_TOC <= app_cmd) && (RESUME >= app_cmd))) {
				/* Do nothing */
			} else {
				err = masca_get_sm_sense(sm_sen, sm_st,
								app_cmd);
			}
		} else {
			err = masca_get_sm_sense(sm_sen, sm_st, app_cmd);
		}
	}

	return err;
}

/* masca_mixed_mode_change function will change
* the internal drive state with respect to the command
* of play track or fetch data on mixed mode CD */
static enum masca_error masca_mixed_mode_change(const enum masca_cmd app_cmd)
{
	enum masca_error error = MASCA_OK;
	enum masca_ign_aud_status ign_aud_status;

	if (READ_SECTORS == app_cmd)
		masca_set_last_cmd_ign_bit(TRUE);
	else if ((FBWD_LBA >= app_cmd) && (PAUSE != app_cmd))
		masca_set_last_cmd_ign_bit(FALSE);

	ign_aud_status = masca_sm_get_ign_aud_status();

	if ((ign_aud_status < IGN_AUD_ENABLE) &&
			(READ_SECTORS == app_cmd)) {
		masca_drive_send_ignore_audio_cmd(TRUE);
		error = MASCA_NR_INPROGRESS_OF_RDY;
	} else if (((ign_aud_status == IGN_AUD_ENABLE) ||
			(ign_aud_status == IGN_AUD_ENABLE_TOC_READ)) &&
			(READ_SECTORS == app_cmd)) {
		/*TOC read in progress*/
		error = MASCA_NR_OPR_IN_PROGRESS;
	} else if ((ign_aud_status == IGN_AUD_ENABLE_TOC_READY) &&
			(READ_SECTORS == app_cmd)) {
		/*TOC is ready*/
	} else {
		if ((ign_aud_status >= IGN_AUD_ENABLE) &&
				(ign_aud_status <= IGN_AUD_ENABLE_TOC_READY)) {
			if ((FALSE == masca_get_last_cmd_ign_bit() &&
					(FBWD_LBA >= app_cmd) &&
					(PAUSE != app_cmd))) {
				masca_drive_send_ignore_audio_cmd(FALSE);
			}
		 }
	}
	return error;
}

/**
 * \func masca_is_cmd_allowed
 *
 * This function is used to check the possible commands with respect
 * to SM states with the help of array table.It will update is_cmd_allowed
 * # if is_cmd_allowed = TRUE, commands is allowed with respect to SM state
 * # if is_cmd_allowed = FALSE,commands is not allowed with respect to SM state
 *
 */
static bool
masca_is_cmd_allowed(const enum masca_cmd app_cmd, enum masca_error *err)
{
	bool is_cmd_allowed = TRUE;
	struct masca_sm_state sm_state;
	enum masca_media_type cd_type;
	enum masca_drive_status drv_stat = masca_get_drive_status();
	int sm_sense = 0x00;
	*err = MASCA_OK;

	masca_sm_get_state(&sm_state);

	/* In sleep state only Power conditional commands are allowed */
	if ((sm_state.sm_pwr_state == SM_PWR_SLEEP) && (app_cmd != PWR_SLEEP)) {
		if ((app_cmd == PWR_STANDBY)
				|| (app_cmd == PWR_STANDBY_NO_CD)) {
			/* In sleep state, PWR_STANDBY command is allowed
			 * when there is no CD */
			if (sm_state.sm_drive_state != SM_NO_CD) {
				masca_update_power_evt(PWRCHG_FAIL);
				*err = MASCA_UA_LOW_PWR_ON;
				is_cmd_allowed = FALSE;
			}
		} else if (app_cmd != PWR_ACTIVE) {
			/* In sleep state, no commands are allowed except
			 * PWR_ACTIVE. Here special handling done for TUR
			 * and Get_evt_stat because they will not take an
			 * unit attention related sense keys */
			if ((GET_TEST_UNIT_READY == app_cmd)
				|| (CMD_GET_EVT_STAT == app_cmd))
				*err = MASCA_HE_VOLTAGE_FAULT;
			else
				*err = MASCA_UA_LOW_PWR_ON;
			is_cmd_allowed = FALSE;
		}

		if (is_cmd_allowed == FALSE)
			return is_cmd_allowed;
	}

	is_cmd_allowed &= (cmd_driver_state[app_cmd][sm_state.sm_driver_state]);
	if (FALSE == is_cmd_allowed)
		sm_sense |= SENSE_DRIVER_ERR;

	if (DRV_MSG_READING_CD == drv_stat)
		is_cmd_allowed &= cmd_drive_state[app_cmd][SM_LOADING];
	else if (DRV_MSG_TOC_READY == drv_stat)
		is_cmd_allowed &= cmd_drive_state[app_cmd][SM_ACCESSIBLE];
	else if (DRV_MSG_DISC_INS == drv_stat)
		is_cmd_allowed &= cmd_drive_state[app_cmd][SM_INSERTED];
	if (FALSE == is_cmd_allowed)
		sm_sense |= SENSE_DRIVE_ERR;

	if (TRUE == is_cmd_allowed) {
		is_cmd_allowed &=
				cmd_drive_state[app_cmd][sm_state.sm_drive_state];
		if (SM_ACCESSIBLE == sm_state.sm_drive_state) {
			is_cmd_allowed &= cmd_disc_state(app_cmd,
					sm_state.sm_disc_state);
			if (FALSE == is_cmd_allowed)
				sm_sense |= SENSE_DISC_ERR;
		}
		if (FALSE == is_cmd_allowed)
			sm_sense |= SENSE_DRIVE_ERR;
	}

	if (DRV_MSG_POWER_FAIL == drv_stat)
		is_cmd_allowed &= cmd_volt_state(app_cmd, UNDER_VOLTAGE);
	else {
		is_cmd_allowed &= cmd_volt_state(app_cmd,
				sm_state.sm_volt_state);
		if (FALSE == is_cmd_allowed)
			sm_sense |= SENSE_VOLT_ERR;
	}

	if ((TRUE == is_cmd_allowed)
			&& (SM_ACCESSIBLE == sm_state.sm_drive_state)) {
		masca_sm_get_media_type(&cd_type);
		if (DISC_CDROM == cd_type) {
			if ((INSERT <= app_cmd) &&
				(CMD_GET_EVT_STAT >= app_cmd) &&
				(STOP != app_cmd) &&
				(GET_TITLE_ALBUM_AUTHOR != app_cmd) &&
				(RESUME != app_cmd)) {
				/*command is allowed*/
			} else {
				is_cmd_allowed = FALSE;
				/*incompatible medium installed*/
				*err = MASCA_NR_MEDIUM_INCOMPATIBLE;
			}
		} else if (DISC_CDDA == cd_type) {
			if (READ_SECTORS == app_cmd) {
				is_cmd_allowed = FALSE;
				/*incompatible medium installed*/
				*err = MASCA_NR_MEDIUM_INCOMPATIBLE;
			}
		} else if (DISC_MIXED_MODE == cd_type) {
				/* Mixed mode CD */
			*err = masca_mixed_mode_change(app_cmd);
			if (*err == MASCA_OK)
				is_cmd_allowed = TRUE;
			else
				is_cmd_allowed = FALSE;
		}
	}

	if (TRUE == is_cmd_allowed)
		*err = MASCA_OK;
	else if (*err == MASCA_OK)
		*err = masca_get_sense(sm_sense, sm_state, app_cmd);

	return is_cmd_allowed;
}

static void masca_cmd_drive_mngr(
				const struct masca_cmd_params * const p_cmd,
				struct masca_output * const p_output)
{
	struct masca_drv_cmd_info	manager_cmd;
	struct masca_drv_response_info	manager_response;
	enum masca_error		ret_err;

	memcpy((void *)&manager_cmd.command_param.drv_param,
	(void *)&p_cmd->cmd_params, sizeof(p_cmd->cmd_params));
	manager_cmd.command = p_cmd->command;
	manager_cmd.reply_id = p_cmd->reply_id;
	manager_response.drv_cmd_response = MAX_COMMAND;
	manager_response.response_id = p_cmd->reply_id;

	ret_err = masca_drive_mngr_cmd(&manager_cmd, &manager_response);
	if (MASCA_BUSY == ret_err) {
		/*Add this command to the list*/
		masca_add_to_lst(p_cmd, PENDING);
		masca_prepare_response(&manager_response, p_output);
	} else if ((MASCA_OK == ret_err) || (MASCA_ACCEPTED == ret_err) ||
			(MASCA_PROCESSED == ret_err)) {
		/*command is accepted by drive manager*/
		if (manager_response.drv_cmd_response != p_cmd->command) {
			/*add command to list*/
			masca_add_to_lst(p_cmd, EXECUTING);
			masca_prepare_response(&manager_response, p_output);
		} else {
			/*no need to add to the list as command is already
			* processed. Just reply over here*/
			p_output->replied_command = manager_cmd.command;
			p_output->response_id = p_cmd->reply_id;
			p_output->reply_status =
						manager_response.response_error;
			p_output->output_params.cd_version_info =
				manager_response.response_param.cd_version_info;
		}
	} else {
		CMD_TE_TRACE("CMD_HND: drive mngr return error = %d", ret_err);
	}
}

static void masca_check_power_status(
		const struct masca_drv_response_info * const p_res_info,
		struct masca_output * const p_output)
{
	struct masca_sm_state	sm_state;
	enum masca_cmd	rep_cmd;

	masca_sm_get_state(&sm_state);
	rep_cmd = p_res_info->drv_cmd_response;
	if ((rep_cmd == PWR_ACTIVE)
		&& (p_res_info->response_error == MASCA_PROCESSED)) {
		masca_sm_update_power_status(SM_PWR_ACTIVE);
		masca_update_power_evt(PWRCHG_SUCCESS);
	} else if ((rep_cmd == PWR_STANDBY)
		&& (p_res_info->response_error == MASCA_PROCESSED)) {
		masca_sm_update_power_status(SM_PWR_STANDBY);
		masca_update_power_evt(PWRCHG_SUCCESS);
	} else if ((rep_cmd == PWR_STANDBY_NO_CD)
		&& ((sm_state.sm_pwr_state == SM_PWR_STANDBY)
			|| (p_res_info->response_error == MASCA_PROCESSED))) {
		masca_update_power_evt(PWRCHG_SUCCESS);
		/* Drive in standby state */
		p_output->reply_status = MASCA_PROCESSED;
	} else if ((rep_cmd >= PWR_ACTIVE) && (rep_cmd <= PWR_STANDBY_NO_CD)) {
		if ((rep_cmd == PWR_STANDBY)
				&& (sm_state.sm_pwr_state == SM_PWR_STANDBY)) {
			masca_update_power_evt(PWRCHG_SUCCESS);
			/* Drive in standby state */
			p_output->reply_status = MASCA_PROCESSED;
		} else
			masca_update_power_evt(PWRCHG_FAIL);
	}
}

static void masca_prepare_response(
			const struct masca_drv_response_info * const p_res_info,
			struct masca_output * const p_output)
{
	struct masca_cmd_lst_elem	*p_elem;
	enum masca_cmd			rep_cmd;

	p_output->replied_command = MAX_COMMAND;
	rep_cmd = p_res_info->drv_cmd_response;
	if (rep_cmd != MAX_COMMAND) {
		if (EJECT == rep_cmd) {
			/*start the auto re-insert timer*/
			masca_util_sta_alm(auto_reinsert_tmout,
						     ALM_AUTO_REINSERT);
		}
		p_elem = masca_get_elem_by_cmd(rep_cmd, EXECUTING);
		if (NULL != p_elem) {
			/*We got response to the command we sent*/
			p_output->replied_command = rep_cmd;
			p_output->response_id = p_elem->acpptd_cmd.reply_id;
			p_output->reply_status = p_res_info->response_error;
			p_output->output_params.cd_version_info =
			p_res_info->response_param.cd_version_info;
			if ((rep_cmd >= PWR_ACTIVE) && (rep_cmd <= PWR_SLEEP))
				masca_check_power_status(p_res_info, p_output);
			/*clear out the entry in our list*/
			masca_empty_lst_elem(p_elem);
		}
	}
}

static void
masca_add_to_lst(const struct masca_cmd_params * const p_cmd_info,
					const enum masca_cmd_lst_stat elem_stat)
{
	struct masca_cmd_lst_elem *p_elem = p_last_elem;

	if (EMPTY == p_elem->elem_stat) {
		memcpy((void *)&(p_elem->acpptd_cmd),
				(void *)p_cmd_info,
				sizeof(struct masca_cmd_params));
		p_elem->elem_stat = elem_stat;
	}
	if (NULL != p_elem->p_nxt) {
		/*point to next free element*/
		p_last_elem = p_elem->p_nxt;
	}
}

static void masca_empty_lst_elem(const struct masca_cmd_lst_elem *
							const p_lst_to_empty)
{
	struct masca_cmd_lst_elem *p_elem = p_frst_elem;
	struct masca_cmd_lst_elem *p_prev = NULL;
	while ((NULL != p_elem) && (NULL != p_lst_to_empty)) {
		if (p_elem == p_lst_to_empty) {
			p_elem->elem_stat = EMPTY;
			if (NULL != p_prev) {
				/*element in the middle of the list to be */
				/*emptied move this to end of the list*/
				p_prev->p_nxt = p_elem->p_nxt;
				if (NULL != p_last_elem) {
					p_elem->p_nxt = p_last_elem->p_nxt;
					p_last_elem->p_nxt = p_elem;
				}
			} else {
				/*This is the first element in the list to be
				 * emptied*/
				p_frst_elem = p_elem->p_nxt;
				if (NULL != p_last_elem) {
					p_elem->p_nxt = p_last_elem->p_nxt;
					p_last_elem->p_nxt = p_elem;
				}
			}
			/*terminate the loop*/
			p_elem = NULL;
		} else {
			p_prev = p_elem;
			p_elem = p_elem->p_nxt;
		}
	}
}

static struct masca_cmd_lst_elem *masca_get_elem_by_cmd(
			const enum masca_cmd cmd,
			const enum masca_cmd_lst_stat elem_stat)
{
	struct masca_cmd_lst_elem *p_elem = p_frst_elem;
	struct masca_cmd_lst_elem *p_found_elem = NULL;

	while (NULL != p_elem) {
		if ((elem_stat == p_elem->elem_stat)
			&& (p_elem->acpptd_cmd.command == cmd)) {
			p_found_elem = p_elem;
			p_elem = NULL;
		} else {
			p_elem = p_elem->p_nxt;
		}
	}
	return p_found_elem;
}


static struct masca_cmd_lst_elem*
masca_get_frst_elem(const enum masca_cmd_lst_stat elem_stat)
{
	bool		is_cmd_allowed = FALSE;
	struct masca_cmd_lst_elem *p_elem = p_frst_elem;
	struct masca_cmd_lst_elem *p_found_elem = NULL;
	enum masca_error ret_err = MASCA_IR_CMD_NOT_SUPP;

	while (NULL != p_elem) {
		if (abort_cmd_search != FALSE) {
			is_cmd_allowed = masca_is_cmd_allowed(
					p_elem->acpptd_cmd.command, &ret_err);
		}

		if (is_cmd_allowed == TRUE) {
			p_elem = p_elem->p_nxt;
		} else {
			if (elem_stat == p_elem->elem_stat) {
				p_found_elem = p_elem;
				p_elem = NULL;
			} else {
				p_elem = p_elem->p_nxt;
			}
		}
	}
	return p_found_elem;
}

static struct masca_cmd_lst_elem *masca_get_elem_by_id(const unsigned int hash)
{
	struct masca_cmd_lst_elem *p_elem = p_frst_elem;
	struct masca_cmd_lst_elem *p_found_elem = NULL;

	while (NULL != p_elem) {
		if ((EMPTY != p_elem->elem_stat)
			&& (hash == p_elem->acpptd_cmd.reply_id)) {
			p_found_elem = p_elem;
			p_elem = NULL;
		} else {
			p_elem = p_elem->p_nxt;
		}
	}
	return p_found_elem;
}
