/*
 * Copyright 2012 Robert Bosch Car Multimedia GmbH. All Rights Reserved.
 */
/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
/*!
 * @file tv_fpga.c
 *
 * @brief TV_FPGA video decoder functions
 *
 * @ingroup Camera
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/wait.h>
#include <linux/videodev2.h>
#include <media/v4l2-int-device.h>
#include <media/v4l2-of.h>
#include "mxc_v4l2_capture.h"

/*! List of input video formats supported. The video formats is corresponding
 * with v4l2 id in video_fmt_t
 */
enum tv_fpga_fmt_idx {
	TV_FPGA_5120_960 = 0,	/* Format 1) 5120 x 960 pixel */
	TV_FPGA_1280_3840,	/* Format 2) 1280 x 3840 pixel */
};
/*! Number of video standards supported (including 'not locked' signal). */
#define TV_FPGA_STD_MAX		(TV_FPGA_1280_3840)
/*! Video format structure. */
struct video_fmt_t {
	int v4l2_id;		/*!< Video for linux ID. */
	char name[16];		/*!< Name (e.g., "NTSC", "PAL", etc.) */
	u16 raw_width;		/*!< Raw width. */
	u16 raw_height;		/*!< Raw height. */
	u16 active_width;	/*!< Active width. */
	u16 active_height;	/*!< Active height. */
};
/*! Description of video formats supported.
 *
 *  5120_960 --> 5120 x 960  pixel active and raw
 *  1280_3840--> 1280 x 3840 pixel active and raw
 */
static struct video_fmt_t video_fmts[] = {

	{
	 .name = "5120_960",
	 .raw_width = 5120,	/* SENS_FRM_WIDTH */
	 .raw_height = 960,	/* SENS_FRM_HEIGHT */
	 .active_width = 5120,	/* ACT_FRM_WIDTH plus 1 */
	 .active_height = 960,	/* ACT_FRM_WIDTH plus 1 */
	 },
	{
	 .name = "1280_3840",
	 .raw_width = 1280,
	 .raw_height = 3840,
	 .active_width = 1280,
	 .active_height = 3840,
	 }
};

struct tv_fpga_dev {
	struct device *dev;
	struct i2c_client *i2c_client;
	struct v4l2_of_bus_parallel bus_cfg;
	struct v4l2_pix_format pix;
	struct v4l2_captureparm streamcap;
};

/*!* Standard index of TV_FPGA. */
static enum tv_fpga_fmt_idx video_idx = TV_FPGA_5120_960;
#define IF_NAME                    "tv_fpga"

/***********************************************************************
 * mxc_v4l2_capture interface.
 ***********************************************************************/

/***********************************************************************
 * IOCTL Functions from v4l2_int_ioctl_desc.
 ***********************************************************************/
/*!
 * ioctl_g_ifparm - V4L2 sensor interface handler for vidioc_int_g_ifparm_num
 * s: pointer to standard V4L2 device structure
 * p: pointer to standard V4L2 vidioc_int_g_ifparm_num ioctl structure
 *
 * Gets slave interface parameters.
 * Calculates the required xclk value to support the requested
 * clock parameters in p.  This value is returned in the p
 * parameter.
 *
 * vidioc_int_g_ifparm returns platform-specific information about the
 * interface settings used by the sensor.
 *
 * Called on open.
 */
static int ioctl_g_ifparm(struct v4l2_int_device *s, struct v4l2_ifparm *p)
{
	struct tv_fpga_dev *sensor = s->priv;

	pr_debug("In tv_fpga:ioctl_g_ifparm\n");

	/* Initialize structure to 0s then set any non-0 values. */
	memset(p, 0, sizeof(*p));
	p->if_type = V4L2_IF_TYPE_BT656;	/* This is the only pos. */

	p->u.bt656.mode = V4L2_IF_TYPE_BT656_MODE_NOBT_16BIT;
	/*

	 * 0: Frame begins when vsync is high.
	 * 1: Frame begins when vsync changes from low to high.
	 */
	p->u.bt656.frame_start_on_rising_vs = 1;
	/* Use Bt synchronisation codes for sync correction. */
	p->u.bt656.bt_sync_correct = 0;	/* Indicate external vsync */
	/* Swap every two adjacent image data elements. */
	/*p->u.bt656.swap:1; */

	/* Hs polarity. 0 is active high, 1 active low. */
	p->u.bt656.nobt_hs_inv =
		(sensor->bus_cfg.flags & V4L2_MBUS_HSYNC_ACTIVE_LOW) ? 1 : 0;
	/* Vs polarity. 0 is active high, 1 active low. */
	p->u.bt656.nobt_vs_inv =
		(sensor->bus_cfg.flags & V4L2_MBUS_VSYNC_ACTIVE_LOW) ? 1 : 0;
	/* Inverted latch clock polarity from slave. */
	p->u.bt656.latch_clk_inv =
		(sensor->bus_cfg.flags & V4L2_MBUS_PCLK_SAMPLE_FALLING) ? 1 : 0;

	/* Minimum accepted bus clock for slave (in Hz). */
	p->u.bt656.clock_min = 10000000;
	/* Maximum accepted bus clock for slave. */
	p->u.bt656.clock_max = 200000000;
	/*
	 * Current wish of the slave. May only change in response to
	 * ioctls that affect image capture.
	 */
	p->u.bt656.clock_curr = 133000000;

	/* TV_FPGA has a dedicated clock so no clock settings needed. */
	return 0;
}

/*!
 * ioctl_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int ioctl_g_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	struct tv_fpga_dev *sensor = s->priv;
	struct v4l2_captureparm *cparm = &a->parm.capture;
	pr_debug("In tv_fpga:ioctl_g_parm\n");

	switch (a->type) {
		/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		pr_debug("   type is V4L2_BUF_TYPE_VIDEO_CAPTURE\n");
		memset(a, 0, sizeof(*a));
		a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		cparm->capability = sensor->streamcap.capability;
		cparm->timeperframe = sensor->streamcap.timeperframe;
		cparm->capturemode = sensor->streamcap.capturemode;
		break;
	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;
	default:
		pr_debug("ioctl_g_parm:type is unknown %d\n", a->type);
		break;
	}
	return 0;
}

/*!
 * ioctl_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 *
 * This driver cannot change these settings.
 */
static int ioctl_s_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	pr_debug("In tv_fpga:ioctl_s_parm\n");

	switch (a->type) {
		/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;

	default:
		pr_debug("   type is unknown - %d\n", a->type);
		break;
	}

	return 0;
}

/*!
 * ioctl_g_fmt_cap - V4L2 sensor interface handler for ioctl_g_fmt_cap
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 v4l2_format structure
 *
 * Returns the sensor's current pixel format in the v4l2_format
 * parameter.
 */
static int ioctl_g_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct tv_fpga_dev *sensor = s->priv;

	pr_debug("In tv_fpga:ioctl_g_fmt_cap\n");
	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		pr_debug("   Returning size of %dx%d\n",
			 sensor->pix.width, sensor->pix.height);
		f->fmt.pix = sensor->pix;
		break;
	case V4L2_BUF_TYPE_PRIVATE:
	default:
		f->fmt.pix = sensor->pix;
		break;
	}
	return 0;
}

/*!
 * ioctl_try_fmt_cap - V4L2 sensor interface handler for VIDIOC_TRY_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * Return 0.
 */
static int ioctl_try_fmt_cap(struct v4l2_int_device *s,
			      struct v4l2_format *f)
{
	struct tv_fpga_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return 0;
}

/*!
 * ioctl_s_fmt_cap - V4L2 sensor interface handler for VIDIOC_S_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * Return 0.
 */
static int ioctl_s_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct tv_fpga_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return 0;
}

/*!
 * ioctl_init - V4L2 sensor interface handler for VIDIOC_INT_INIT
 * @s: pointer to standard V4L2 device structure
 */
static int ioctl_init(struct v4l2_int_device *s)
{
	pr_debug("In tv_fpga:ioctl_init\n");
	return 0;
}

/*!
 * ioctl_enum_framesizes - V4L2 sensor interface handler for
 *			   VIDIOC_ENUM_FRAMESIZES ioctl
 * @s: pointer to standard V4L2 device structure
 * @fsize: standard V4L2 VIDIOC_ENUM_FRAMESIZES ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_framesizes(struct v4l2_int_device *s,
				 struct v4l2_frmsizeenum *fsize)
{
	if (fsize->index > TV_FPGA_STD_MAX)
		return -EINVAL;

	fsize->pixel_format = V4L2_PIX_FMT_UYVY;
	fsize->discrete.width = video_fmts[fsize->index].raw_width;
	fsize->discrete.height = video_fmts[fsize->index].raw_height;

	pr_debug("ioctl_enum_framesizes: %d x %d\n",
		 fsize->discrete.width, fsize->discrete.height);

	return 0;
}

/*!
 * ioctl_enum_input - V4L2 sensor interface handler for VIDIOC_ENUMINPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @input: standard V4L2 VIDIOC_ENUMINPUT ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_input(struct v4l2_int_device *s,
			    struct v4l2_input *input)
{
	if (input->index != 0)
		return -EINVAL;

	input->type = V4L2_INPUT_TYPE_CAMERA;
	input->status = 0;
	input->capabilities = 0;
	strcpy(input->name, "top_view_fpga");

	return 0;
}

/*!
 * ioctl_g_input - V4L2 sensor interface handler for VIDIOC_G_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to return index number
 *
 * Returns 0.
 */
static int ioctl_g_input(struct v4l2_int_device *s, int *index)
{
	*index = 0;
	return 0;
}

/*!
 * ioctl_s_input - V4L2 sensor interface handler for VIDIOC_S_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to index number to set
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_s_input(struct v4l2_int_device *s, int *index)
{
	if (*index != 0)
		return -EINVAL;
	return 0;
}

/*!
 * ioctl_g_chip_ident - V4L2 sensor interface handler for
 *			VIDIOC_DBG_G_CHIP_IDENT ioctl
 * @s: pointer to standard V4L2 device structure
 * @id: pointer to int
 *
 * Return 0.
 */
static int ioctl_g_chip_ident(struct v4l2_int_device *s, int *id)
{
	((struct v4l2_dbg_chip_ident *)id)->match.type = V4L2_CHIP_MATCH_HOST;
	strcpy(((struct v4l2_dbg_chip_ident *)id)->match.name, "top_view_fpga");

	return 0;
}

/*!
 * ioctl_init - V4L2 sensor interface handler for VIDIOC_QUERYSTD
 * @s: pointer to standard V4L2 device structure
 * @std: v4l2_std_id *
 */
static int ioctl_querystd(struct v4l2_int_device *s, v4l2_std_id *std)
{
	*std = V4L2_STD_ALL;
	return 0;
}

/*!
 * ioctl_dev_init - V4L2 sensor interface handler for vidioc_int_dev_init_num
 * @s: pointer to standard V4L2 device structure
 *
 * Initialise the device when slave attaches to the master.
 */
static int ioctl_dev_init(struct v4l2_int_device *s)
{
	pr_debug("In tv_fpga:ioctl_dev_init\n");
	return 0;
}

static int ioctl_g_dev(struct v4l2_int_device *s, struct device **dev)
{
	struct tv_fpga_dev *sensor = s->priv;

	*dev = sensor->dev;
	return 0;
}

/*!
 * This structure defines all the ioctls for this module.
 */
static struct v4l2_int_ioctl_desc tv_fpga_ioctl_desc[] = {
	{vidioc_int_dev_init_num,
	 (v4l2_int_ioctl_func *) ioctl_dev_init},
/*!
 * Delinitialise the dev. at slave detach.
 * The complement of ioctl_dev_init.
 */
	{vidioc_int_g_ifparm_num,
	 (v4l2_int_ioctl_func *) ioctl_g_ifparm},
	{vidioc_int_init_num,
	 (v4l2_int_ioctl_func *) ioctl_init},
	{vidioc_int_g_dev_num,
	 (v4l2_int_ioctl_func *)ioctl_g_dev},
	{vidioc_int_g_fmt_cap_num,
	 (v4l2_int_ioctl_func *) ioctl_g_fmt_cap},
	{vidioc_int_try_fmt_cap_num,
	 (v4l2_int_ioctl_func *) ioctl_try_fmt_cap},
	{vidioc_int_s_fmt_cap_num,
	 (v4l2_int_ioctl_func *)ioctl_s_fmt_cap},
	{vidioc_int_g_parm_num,
	 (v4l2_int_ioctl_func *) ioctl_g_parm},
	{vidioc_int_s_parm_num,
	 (v4l2_int_ioctl_func *) ioctl_s_parm},
	{vidioc_int_enum_framesizes_num,
	 (v4l2_int_ioctl_func *) ioctl_enum_framesizes},
	{vidioc_int_enum_input_num,
	 (v4l2_int_ioctl_func *)ioctl_enum_input},
	{vidioc_int_g_input_num,
	 (v4l2_int_ioctl_func *)ioctl_g_input},
	{vidioc_int_s_input_num,
	 (v4l2_int_ioctl_func *)ioctl_s_input},
	{vidioc_int_g_chip_ident_num,
	 (v4l2_int_ioctl_func *) ioctl_g_chip_ident},
	{vidioc_int_querystd_num,
	 (v4l2_int_ioctl_func *)ioctl_querystd},
};

static struct v4l2_int_slave tv_fpga_slave = {
	.ioctls = tv_fpga_ioctl_desc,
	.num_ioctls = ARRAY_SIZE(tv_fpga_ioctl_desc),
};

static struct v4l2_int_device tv_fpga_int_device = {
	.module = THIS_MODULE,
	.name = "tv_fpga",
	.type = v4l2_int_type_slave,
	.u = {
	      .slave = &tv_fpga_slave,
	      },
};

/*!
 * TV_FPGA init function.
 * Called on insmod.
 *
 * @return    Error code indicating success or failure.
 */
static int tv_fpga_probe(struct platform_device *pdev)
{
	struct v4l2_of_endpoint bus_cfg;
	struct device_node *endpoint;
	struct tv_fpga_dev *sensor;
	u8 err = 0;

	pr_debug("In tv_fpga_probe\n");

	sensor = kzalloc(sizeof(struct tv_fpga_dev), GFP_KERNEL);
	if (!sensor)
		return -ENOMEM;

	sensor->streamcap.timeperframe.denominator = 30;
	sensor->streamcap.timeperframe.numerator = 1;
	video_idx = TV_FPGA_5120_960;
	sensor->pix.width = video_fmts[video_idx].raw_width;
	sensor->pix.height = video_fmts[video_idx].raw_height;
	sensor->dev = &pdev->dev;

	endpoint = v4l2_of_get_next_endpoint(pdev->dev.of_node, NULL);
	if (!endpoint) {
		dev_err(&pdev->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	v4l2_of_parse_endpoint(endpoint, &bus_cfg);
	if (bus_cfg.bus_type != V4L2_MBUS_PARALLEL) {
		dev_err(&pdev->dev, "invalid bus type, must be parallel\n");
		return -EINVAL;
	}
	of_node_put(endpoint);

	sensor->bus_cfg = bus_cfg.bus.parallel;

	/* --> see \include\linux\videodev2.h */
	/*
	 * the FPGA sensor outputs UYVY as one 16-bit word on a
	 * 16-bit parallel interface.
	 */
	sensor->pix.pixelformat = V4L2_MBUS_FMT_UYVY8_1X16;
	sensor->pix.field = V4L2_FIELD_NONE;

	pr_debug("   type is %d (expect %d)\n",
		 tv_fpga_int_device.type, v4l2_int_type_slave);
	pr_debug("   num ioctls is %d\n",
		 tv_fpga_int_device.u.slave->num_ioctls);

	/* This function attaches this structure to the /dev/video0 device.
	 * The pointer in priv points to the tv_fpga_dev structure here.*/
	tv_fpga_int_device.priv = sensor;
	err = v4l2_int_device_register(&tv_fpga_int_device);

	return err;
}

static int tv_fpga_remove(struct platform_device *pdev)
{
	kfree(tv_fpga_int_device.priv);
	v4l2_int_device_unregister(&tv_fpga_int_device);
	return 0;
}

static struct of_device_id tv_fpga_dt_ids[] = {
	{ .compatible = "bosch,tv-fpga" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, tv_fpga_dt_ids);

static struct platform_driver tv_fpga_driver = {
	.driver = {
		.name  = "tv-fpga-sensor",
		.owner = THIS_MODULE,
		.of_match_table = tv_fpga_dt_ids,
	},
	.probe    = tv_fpga_probe,
	.remove   = tv_fpga_remove,
};
module_platform_driver(tv_fpga_driver);

MODULE_AUTHOR("Robert Bosch Car Multimedia GmbH");
MODULE_DESCRIPTION("TV_FPGA video decoder driver");
MODULE_LICENSE("GPL");
