/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file ipu_still.c
 *
 * @brief IPU Use case for still image capture
 *
 * @ingroup IPU
 */

#include <linux/semaphore.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-int-device.h>
#include "mxc_v4l2_capture.h"
#include "ipu_prp_sw.h"

struct prp_still_priv {
	struct ipuv3_channel *csi_ch;
	struct ipu_smfc *smfc;
	struct ipu_csi *csi;

	dma_addr_t still_buf[2];
	void *still_buf_vaddr[2];
	int still_buf_size[2];

	int eof_irq;

	int callback_eof_flag;
	int buffer_num;

	wait_queue_head_t still_queue;
	int still_counter;

};

static inline struct prp_still_priv *get_priv(struct mx6_camera_dev *cam)
{
	return cam->prp_still_priv;
}

/*
 * Update the CSI whole sensor and active windows, and initialize
 * the CSI interface and muxes.
 */
static void prp_still_setup_csi(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);

	ipu_csi_set_window_size(priv->csi, cam->crop_current.width,
				cam->crop_current.height);
	ipu_csi_set_window_pos(priv->csi, cam->crop_current.left,
			       cam->crop_current.top);
	ipu_csi_init_interface(priv->csi, cam->crop_bounds.width,
			       cam->crop_bounds.height,
			       &cam->ep->csi_sig_cfg);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_csi_set_mipi_datatype(priv->csi, cam->ep->ep.id,
					  &cam->ep->csi_sig_cfg);

	/* select either parallel or MIPI-CSI2 as input to our CSI */
	ipu_csi_set_src(priv->csi, cam->ep->ep.id,
			cam->ep->ep.bus_type == V4L2_MBUS_CSI2);
	/* set CSI destination to memory via SMFC */
	ipu_csi_set_dest(priv->csi, IPU_CSI_DEST_IDMAC);
}

static void prp_still_put_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);

	if (!IS_ERR_OR_NULL(priv->smfc))
		ipu_smfc_put(priv->smfc);
	priv->smfc = NULL;

	if (!IS_ERR_OR_NULL(priv->csi_ch))
		ipu_idmac_put(priv->csi_ch);
	priv->csi_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->csi))
		ipu_csi_put(priv->csi);
	priv->csi = NULL;
}

static int prp_still_get_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);
	unsigned csi_ch_num;
	int csi_id, err;

	csi_id = cam->ep->ep.port;
	priv->csi = ipu_csi_get(cam->ipu, csi_id);
	if (IS_ERR(priv->csi)) {
		dev_err(cam->dev, "failed to get CSI %d\n", csi_id);
		return PTR_ERR(priv->csi);
	}

	priv->smfc = ipu_smfc_get(cam->ipu);
	if (IS_ERR(priv->smfc)) {
		dev_err(cam->dev, "failed to get SMFC\n");
		err = PTR_ERR(priv->smfc);
		goto out;
	}

	/*
	 * Choose the direct CSI-->SMFC-->MEM channel corresponding
	 * to the IPU and CSI IDs.
	 */
	csi_ch_num = IPUV3_CHANNEL_CSI0 + (ipu_get_num(cam->ipu) << 1) + csi_id;

	priv->csi_ch = ipu_idmac_get(cam->ipu, csi_ch_num, false);
	if (IS_ERR(priv->csi_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			csi_ch_num);
		err = PTR_ERR(priv->csi_ch);
		goto out;
	}

	return 0;
out:
	prp_still_put_ipu_resources(cam);
	return err;
}

static void prp_still_free_bufs(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);

	if (priv->still_buf_vaddr[0]) {
		dma_free_coherent(cam->dev, priv->still_buf_size[0],
				  priv->still_buf_vaddr[0],
				  priv->still_buf[0]);
	}
	if (priv->still_buf_vaddr[1]) {
		dma_free_coherent(cam->dev, priv->still_buf_size[1],
				  priv->still_buf_vaddr[1],
				  priv->still_buf[1]);
	}

	priv->still_buf_vaddr[0] = NULL;
	priv->still_buf[0] = 0;
	priv->still_buf_vaddr[1] = NULL;
	priv->still_buf[1] = 0;
}

static int prp_still_alloc_bufs(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);

	prp_still_free_bufs(cam);

	priv->still_buf_size[0] = PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
	priv->still_buf_vaddr[0] =
		dma_alloc_coherent(cam->dev, priv->still_buf_size[0],
				   &priv->still_buf[0], GFP_DMA | GFP_KERNEL);
	if (!priv->still_buf_vaddr[0]) {
		dev_err(cam->dev, "alloc still_bufs0 err\n");
		return -ENOMEM;
	}

	priv->still_buf_size[1] = PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
	priv->still_buf_vaddr[1] =
		dma_alloc_coherent(cam->dev, priv->still_buf_size[1],
				   &priv->still_buf[1], GFP_DMA | GFP_KERNEL);
	if (!priv->still_buf_vaddr[1]) {
		dev_err(cam->dev, "alloc still_bufs1 err\n");
		prp_still_free_bufs(cam);
		return -ENOMEM;
	}

	return 0;
}

/*!
 * CSI callback function.
 *
 * @param irq       int irq line
 * @param dev_id    void * device id
 *
 * @return status   IRQ_HANDLED for handled
 */
static irqreturn_t prp_still_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;
	struct prp_still_priv *priv = get_priv(cam);

	priv->callback_eof_flag++;
	if (priv->callback_eof_flag < 5) {
		priv->buffer_num ^= 1;
		ipu_idmac_select_buffer(priv->csi_ch, priv->buffer_num);
	} else {
		priv->still_counter++;
		wake_up_interruptible(&priv->still_queue);
	}

	return IRQ_HANDLED;
}

/*!
 * CSI still image channel setup
 *
 * @param cam           struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static void prp_still_setup_channel(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);
	u32 pixel_fmt = cam->user_fmt.fmt.pix.pixelformat;
	unsigned int burst_size;
	bool passthrough;
	u32 stride;

	ipu_cpmem_set_axi_id(priv->csi_ch, 1);

	/*
	 * If the sensor uses 16-bit parallel CSI bus, we must handle
	 * the data internally in the IPU as 16-bit generic, aka
	 * passthrough mode.
	 */
	passthrough = (cam->ep->ep.bus_type != V4L2_MBUS_CSI2 &&
		       cam->ep->csi_sig_cfg.data_width ==
		       IPU_CSI_DATA_WIDTH_16);

	/* convert pixel stride to bytes */
	stride = ipu_stride_to_bytes(cam->user_fmt.fmt.pix.width, pixel_fmt);

	ipu_cpmem_channel_init(priv->csi_ch, pixel_fmt,
			       cam->user_fmt.fmt.pix.width,
			       cam->user_fmt.fmt.pix.height, stride,
			       0, 0, 0,
			       priv->still_buf[0], priv->still_buf[1], 0);

	if (passthrough)
		ipu_cpmem_set_format_passthrough(priv->csi_ch, 16);

	if (ipu_csi_is_interlaced(priv->csi))
		ipu_cpmem_set_interlaced_scan(priv->csi_ch, 0);

	burst_size = ipu_cpmem_get_burst_size(priv->csi_ch);
	ipu_smfc_set_burst_size(priv->smfc, priv->csi_ch,
				burst_size, passthrough);

	ipu_idmac_set_double_buffer(priv->csi_ch, true);
}

/*!
 * start csi->mem task
 * @param cam       struct struct mx6_camera_dev *
 * @param buf       user buffer to place the captured still image
 *
 * @return  status
 */
static int prp_still_start(struct mx6_camera_dev *cam,
			   char __user *buf)
{
	struct prp_still_priv *priv = get_priv(cam);
	int csi_id, err;

	err = prp_still_get_ipu_resources(cam);
	if (err)
		return err;

	prp_still_setup_csi(cam);

	csi_id = ipu_csi_get_num(priv->csi);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_smfc_map(priv->smfc, priv->csi_ch, csi_id, cam->ep->ep.id);
	else
		ipu_smfc_map(priv->smfc, priv->csi_ch, csi_id, 0);

	err = prp_still_alloc_bufs(cam);
	if (err) {
		dev_err(cam->dev, "prp_still_alloc_bufs failed, %d\n", err);
		goto out_put_ipu;
	}

	prp_still_setup_channel(cam);

	priv->callback_eof_flag = 0;
	priv->buffer_num = 0;

	priv->eof_irq = ipu_idmac_channel_irq(cam->ipu,
					      priv->csi_ch,
					      IPU_IRQ_EOF);
	err = devm_request_threaded_irq(cam->dev, priv->eof_irq,
					NULL, prp_still_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-still", cam);
	if (err != 0) {
		dev_err(cam->dev, "Error registering irq %d\n", err);
		goto out_free_bufs;
	}

	ipu_idmac_select_buffer(priv->csi_ch, 0);
	ipu_idmac_clear_buffer_ready(priv->csi_ch, 1);

	ipu_smfc_enable(priv->smfc);
	ipu_idmac_enable_channel(priv->csi_ch);
	ipu_csi_enable(priv->csi);

	priv->still_counter = 0;
	err = wait_event_interruptible_timeout(priv->still_queue,
					       priv->still_counter != 0,
					       10 * HZ);
	if (!err) {
		dev_err(cam->dev, "%s: timeout\n", __func__);
		err = -ETIME;
		goto out_free_eof_irq;
	}

	err = copy_to_user(buf, priv->still_buf_vaddr[1],
			   cam->user_fmt.fmt.pix.sizeimage);
	if (err) {
		dev_err(cam->dev, "%s: copy_to_user error\n", __func__);
		goto out_free_eof_irq;
	}

	return 0;

out_free_eof_irq:
	devm_free_irq(cam->dev, priv->eof_irq, cam);
out_free_bufs:
	prp_still_free_bufs(cam);
out_put_ipu:
	prp_still_put_ipu_resources(cam);
	return err;
}

/*!
 * stop csi->mem encoder task
 * @param cam       struct struct mx6_camera_dev *
 *
 * @return  status
 */
static int prp_still_stop(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv = get_priv(cam);

	devm_free_irq(cam->dev, priv->eof_irq, cam);

	prp_still_free_bufs(cam);

	ipu_csi_disable(priv->csi);
	ipu_idmac_disable_channel(priv->csi_ch);
	ipu_smfc_disable(priv->smfc);

	prp_still_put_ipu_resources(cam);

	return 0;
}

/*!
 * function to select CSI_MEM as the working path
 *
 * @param cam       struct struct mx6_camera_dev *
 *
 * @return  status
 */
int prp_still_select(struct mx6_camera_dev *cam)
{
	cam->csi_start = prp_still_start;
	cam->csi_stop = prp_still_stop;

	return 0;
}

/*!
 * function to de-select CSI_MEM as the working path
 *
 * @param cam       struct struct mx6_camera_dev *
 *
 * @return  status
 */
int prp_still_deselect(struct mx6_camera_dev *cam)
{
	prp_still_stop(cam);

	cam->csi_start = NULL;
	cam->csi_stop = NULL;

	return 0;
}

int prp_still_init(struct mx6_camera_dev *cam)
{
	struct prp_still_priv *priv;

	priv = devm_kzalloc(cam->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	init_waitqueue_head(&priv->still_queue);

	cam->prp_still_priv = priv;

	return 0;
}

void prp_still_exit(struct mx6_camera_dev *cam)
{
}
