/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2011 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file ipu_prp_enc.c
 *
 * @brief IPU Use case for PRP-ENC
 *
 * @ingroup IPU
 */

#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include "mxc_v4l2_capture.h"
#include "ipu_prp_sw.h"

struct prp_enc_priv {
	struct ipuv3_channel *prpenc_ch;
	struct ipuv3_channel *prpenc_rot_in_ch;
	struct ipuv3_channel *prpenc_rot_out_ch;
	struct ipu_ic *ic_enc;
	struct ipu_irt *irt;
	struct ipu_csi *csi;

	dma_addr_t rot_enc_buf[2];
	void *rot_enc_buf_vaddr[2];
	int rot_enc_buf_size[2];

	int eof_irq;
	int nfb4eof_irq;

	bool last_eof;     /* waiting for last EOF at stream off */
	struct completion last_eof_comp;
};

static inline struct prp_enc_priv *get_priv(struct mx6_camera_dev *cam)
{
	return cam->prp_enc_priv;
}

/*
 * Function definitions
 */

/*
 * Update the CSI whole sensor and active windows, and initialize
 * the CSI interface and muxes.
 */
static void prp_enc_setup_csi(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);
	int csi_id = ipu_csi_get_num(priv->csi);

	ipu_csi_set_window_size(priv->csi, cam->crop_current.width,
				cam->crop_current.height);
	ipu_csi_set_window_pos(priv->csi, cam->crop_current.left,
			       cam->crop_current.top);
	ipu_csi_init_interface(priv->csi, cam->crop_bounds.width,
			       cam->crop_bounds.height,
			       &cam->ep->csi_sig_cfg);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_csi_set_mipi_datatype(priv->csi, cam->ep->ep.id,
					  &cam->ep->csi_sig_cfg);

	/* select either parallel or MIPI-CSI2 as input to our CSI */
	ipu_csi_set_src(priv->csi, cam->ep->ep.id,
			cam->ep->ep.bus_type == V4L2_MBUS_CSI2);
	/* set CSI destination to IC */
	ipu_csi_set_dest(priv->csi, IPU_CSI_DEST_IC);
	/* set IC to receive from CSI */
	ipu_ic_set_src(priv->ic_enc, csi_id, false);
}

static void prp_enc_put_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);

	if (!IS_ERR_OR_NULL(priv->irt))
		ipu_irt_put(priv->irt);
	priv->irt = NULL;

	if (!IS_ERR_OR_NULL(priv->ic_enc))
		ipu_ic_put(priv->ic_enc);
	priv->ic_enc = NULL;

	if (!IS_ERR_OR_NULL(priv->prpenc_ch))
		ipu_idmac_put(priv->prpenc_ch);
	priv->prpenc_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->prpenc_rot_in_ch))
		ipu_idmac_put(priv->prpenc_rot_in_ch);
	priv->prpenc_rot_in_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->prpenc_rot_out_ch))
		ipu_idmac_put(priv->prpenc_rot_out_ch);
	priv->prpenc_rot_out_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->csi))
		ipu_csi_put(priv->csi);
	priv->csi = NULL;
}

static int prp_enc_get_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);
	int csi_id, err;

	csi_id = cam->ep->ep.port;
	priv->csi = ipu_csi_get(cam->ipu, csi_id);
	if (IS_ERR(priv->csi)) {
		dev_err(cam->dev, "failed to get CSI %d\n", csi_id);
		return PTR_ERR(priv->csi);
	}

	priv->ic_enc = ipu_ic_get(cam->ipu, IC_TASK_ENCODER);
	if (IS_ERR(priv->ic_enc)) {
		dev_err(cam->dev, "failed to get IC ENC\n");
		err = PTR_ERR(priv->ic_enc);
		goto out;
	}

	priv->irt = ipu_irt_get(cam->ipu);
	if (IS_ERR(priv->irt)) {
		dev_err(cam->dev, "failed to get IRT\n");
		err = PTR_ERR(priv->irt);
		goto out;
	}

	priv->prpenc_ch = ipu_idmac_get(cam->ipu,
					IPUV3_CHANNEL_IC_PRP_ENC_MEM,
					false);
	if (IS_ERR(priv->prpenc_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_IC_PRP_ENC_MEM);
		err = PTR_ERR(priv->prpenc_ch);
		goto out;
	}

	priv->prpenc_rot_in_ch = ipu_idmac_get(cam->ipu,
					       IPUV3_CHANNEL_MEM_ROT_ENC,
					       false);
	if (IS_ERR(priv->prpenc_rot_in_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_MEM_ROT_ENC);
		err = PTR_ERR(priv->prpenc_rot_in_ch);
		goto out;
	}

	priv->prpenc_rot_out_ch = ipu_idmac_get(cam->ipu,
						IPUV3_CHANNEL_ROT_ENC_MEM,
						false);
	if (IS_ERR(priv->prpenc_rot_out_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_ROT_ENC_MEM);
		err = PTR_ERR(priv->prpenc_rot_out_ch);
		goto out;
	}

	return 0;
out:
	prp_enc_put_ipu_resources(cam);
	return err;
}

/*!
 * IPU ENC callback function.
 *
 * @param irq       int irq line
 * @param dev_id    void * device id
 *
 * @return status   IRQ_HANDLED for handled
 */
static irqreturn_t prp_enc_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;
	struct prp_enc_priv *priv = get_priv(cam);

	if (cam->enc_callback == NULL)
		return IRQ_HANDLED;

	if (priv->last_eof) {
		complete(&priv->last_eof_comp);
		priv->last_eof = false;
		return IRQ_HANDLED;
	}

	cam->enc_callback(cam, irq);

	return IRQ_HANDLED;
}

static irqreturn_t prp_enc_nfb4eof_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;

	dev_err(cam->dev, "ERROR: NFB4EOF\n");

	/*
	 * It has been discovered that with rotation, prp_enc disable
	 * creates a single NFB4EOF event which is 100% repeatable. So
	 * scheduling a restart here causes an endless NFB4EOF-->restart
	 * cycle. The error itself seems innocuous, capture is not adversely
	 * affected.
	 *
	 * So don't schedule a restart on NFB4EOF error. If the source
	 * of the NFB4EOF event on prp_enc disable is ever found, it can
	 * be re-enabled, but is probably not necessary. Detecting the
	 * interrupt (and clearing the irq status in the IPU) seems to
	 * be enough.
	 */
#if 0
	mod_timer(&cam->restart_timer,
		  jiffies + msecs_to_jiffies(MXC_RESTART_DELAY));
#endif
	return IRQ_HANDLED;
}

static void prp_enc_free_rot_bufs(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);

	if (priv->rot_enc_buf_vaddr[0]) {
		dma_free_coherent(cam->dev, priv->rot_enc_buf_size[0],
				  priv->rot_enc_buf_vaddr[0],
				  priv->rot_enc_buf[0]);
	}
	if (priv->rot_enc_buf_vaddr[1]) {
		dma_free_coherent(cam->dev, priv->rot_enc_buf_size[1],
				  priv->rot_enc_buf_vaddr[1],
				  priv->rot_enc_buf[1]);
	}

	priv->rot_enc_buf_vaddr[0] = NULL;
	priv->rot_enc_buf[0] = 0;
	priv->rot_enc_buf_vaddr[1] = NULL;
	priv->rot_enc_buf[1] = 0;
}

static int prp_enc_alloc_rot_bufs(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);

	prp_enc_free_rot_bufs(cam);

	priv->rot_enc_buf_size[0] = PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
	priv->rot_enc_buf_vaddr[0] =
		dma_alloc_coherent(cam->dev, priv->rot_enc_buf_size[0],
				   &priv->rot_enc_buf[0], GFP_DMA | GFP_KERNEL);
	if (!priv->rot_enc_buf_vaddr[0]) {
		dev_err(cam->dev, "alloc enc_bufs0 err\n");
		return -ENOMEM;
	}

	priv->rot_enc_buf_size[1] = PAGE_ALIGN(cam->user_fmt.fmt.pix.sizeimage);
	priv->rot_enc_buf_vaddr[1] =
		dma_alloc_coherent(cam->dev, priv->rot_enc_buf_size[1],
				   &priv->rot_enc_buf[1], GFP_DMA | GFP_KERNEL);
	if (!priv->rot_enc_buf_vaddr[1]) {
		dev_err(cam->dev, "alloc enc_bufs1 err\n");
		prp_enc_free_rot_bufs(cam);
		return -ENOMEM;
	}

	return 0;
}

static void prp_enc_setup_channel(struct mx6_camera_dev *cam,
				  struct ipuv3_channel *channel,
				  struct v4l2_pix_format *f,
				  enum ipu_rotate_mode rot_mode,
				  dma_addr_t addr0, dma_addr_t addr1,
				  bool rot_swap_width_height)
{
	struct prp_enc_priv *priv = get_priv(cam);
	unsigned int burst_size;
	u32 width, height, stride;

	if (rot_swap_width_height) {
		width = f->height;
		height = f->width;
		stride = f->height;
	} else {
		width = f->width;
		height = f->height;
		stride = f->width;
	}

	stride = ipu_stride_to_bytes(stride, f->pixelformat);

	ipu_cpmem_channel_init(channel, f->pixelformat, width, height, stride,
			       0, 0, 0, addr0, addr1, 0);

	if (rot_mode)
		ipu_cpmem_set_rotation(channel, rot_mode);

	if (channel == priv->prpenc_rot_in_ch ||
	    channel == priv->prpenc_rot_out_ch) {
		burst_size = 8;
		ipu_cpmem_set_block_mode(channel);
	} else
		burst_size = (width % 16) ? 8 : 16;

	ipu_cpmem_set_burst_size(channel, burst_size);

	if (ipu_csi_is_interlaced(priv->csi) && channel == priv->prpenc_ch)
		ipu_cpmem_set_interlaced_scan(channel, 0);

	ipu_ic_task_idma_init(priv->ic_enc, channel, width, height,
			      burst_size, rot_mode);

	ipu_cpmem_set_axi_id(channel, 1);

	ipu_idmac_set_double_buffer(channel, true);
	ipu_idmac_set_triple_buffer(channel, false);
}

static int prp_enc_setup_rotation(struct mx6_camera_dev *cam,
				  dma_addr_t eba0, dma_addr_t eba1,
				  struct v4l2_pix_format *inf,
				  struct v4l2_pix_format *outf)
{
	struct prp_enc_priv *priv = get_priv(cam);
	enum ipu_color_space in_cs, out_cs;
	int err;

	err = prp_enc_alloc_rot_bufs(cam);
	if (err) {
		dev_err(cam->dev, "prp_enc_alloc_rot_bufs failed, %d\n", err);
		return err;
	}

	in_cs = ipu_mbus_code_to_colorspace(inf->pixelformat);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	err = ipu_ic_task_init(priv->ic_enc,
			       inf->width, inf->height,
			       outf->height, outf->width,
			       in_cs, out_cs);
	if (err) {
		dev_err(cam->dev, "ipu_ic_task_init failed, %d\n", err);
		prp_enc_free_rot_bufs(cam);
		return err;
	}

	/* init the IC PRP-->MEM IDMAC channel */
	prp_enc_setup_channel(cam, priv->prpenc_ch, outf,
			      IPU_ROTATE_NONE,
			      priv->rot_enc_buf[0], priv->rot_enc_buf[1],
			      true);

	/* init the MEM-->IC PRP ROT IDMAC channel */
	prp_enc_setup_channel(cam, priv->prpenc_rot_in_ch, outf,
			      cam->rot_mode,
			      priv->rot_enc_buf[0], priv->rot_enc_buf[1],
			      true);

	/* init the destination IC PRP ROT-->MEM IDMAC channel */
	prp_enc_setup_channel(cam, priv->prpenc_rot_out_ch, outf,
			      IPU_ROTATE_NONE,
			      eba0, eba1,
			      false);

	/* now link IC PRP-->MEM to MEM-->IC PRP ROT */
	ipu_link_prp_enc_rot_enc(cam->ipu);

	/* enable the IC and IRT */
	ipu_ic_enable(priv->ic_enc);
	ipu_irt_enable(priv->irt);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->prpenc_ch, 0);
	ipu_idmac_select_buffer(priv->prpenc_ch, 1);
	ipu_idmac_select_buffer(priv->prpenc_rot_out_ch, 0);
	ipu_idmac_select_buffer(priv->prpenc_rot_out_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->prpenc_ch);
	ipu_idmac_enable_channel(priv->prpenc_rot_in_ch);
	ipu_idmac_enable_channel(priv->prpenc_rot_out_ch);

	/* and finally enable the IC PRPENC task */
	ipu_ic_task_enable(priv->ic_enc);

	return 0;
}

static int prp_enc_setup_norotation(struct mx6_camera_dev *cam,
				    dma_addr_t eba0, dma_addr_t eba1,
				    struct v4l2_pix_format *inf,
				    struct v4l2_pix_format *outf)
{
	struct prp_enc_priv *priv = get_priv(cam);
	enum ipu_color_space in_cs, out_cs;
	int err;

	in_cs = ipu_mbus_code_to_colorspace(inf->pixelformat);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	err = ipu_ic_task_init(priv->ic_enc,
			       inf->width, inf->height,
			       outf->width, outf->height,
			       in_cs, out_cs);
	if (err) {
		dev_err(cam->dev, "ipu_ic_task_init failed, %d\n", err);
		return err;
	}

	/* init the IC PRP-->MEM IDMAC channel */
	prp_enc_setup_channel(cam, priv->prpenc_ch, outf,
			      cam->rot_mode, eba0, eba1, false);

	ipu_ic_enable(priv->ic_enc);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->prpenc_ch, 0);
	ipu_idmac_select_buffer(priv->prpenc_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->prpenc_ch);

	/* and finally enable the IC PRPENC task */
	ipu_ic_task_enable(priv->ic_enc);

	return 0;
}

/*!
 * PrpENC enable channel setup function
 *
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_enc_setup(struct mx6_camera_dev *cam,
			 dma_addr_t eba0, dma_addr_t eba1)
{
	struct v4l2_pix_format inf, outf;

	inf = cam->sensor_fmt.fmt.pix;
	inf.width = cam->crop_current.width;
	inf.height = cam->crop_current.height;
	outf = cam->user_fmt.fmt.pix;

	/* if preview is enabled it has already setup the CSI */
	if (!cam->overlay_on)
		prp_enc_setup_csi(cam);

	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT)
		return prp_enc_setup_rotation(cam, eba0, eba1,
					      &inf, &outf);
	else
		return prp_enc_setup_norotation(cam, eba0, eba1,
						&inf, &outf);
}

/*!
 * function to update physical buffer address for encoder IDMA channel
 *
 * @param eba         physical buffer address for encoder IDMA channel
 * @param buffer_num  int buffer 0 or buffer 1
 *
 * @return  status
 */
static int prp_enc_update_eba(struct mx6_camera_dev *cam, dma_addr_t eba,
			      int buffer_num)
{
	struct prp_enc_priv *priv = get_priv(cam);
	struct ipuv3_channel *channel;

	dev_dbg(cam->dev, "eba %x\n", eba);

	channel = (cam->rot_mode >= IPU_ROTATE_90_RIGHT) ?
		priv->prpenc_rot_out_ch : priv->prpenc_ch;

	if (ipu_idmac_buffer_is_ready(channel, buffer_num))
		ipu_idmac_clear_buffer_ready(channel, buffer_num);

	ipu_cpmem_set_buffer(channel, buffer_num, eba);
	ipu_idmac_select_buffer(channel, buffer_num);

	return 0;
}

/*!
 * Enable encoder task
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_enc_enabling_tasks(struct mx6_camera_dev *cam,
				  dma_addr_t eba0, dma_addr_t eba1)
{
	struct prp_enc_priv *priv = get_priv(cam);
	int err = 0;

	dev_dbg(cam->dev, "IPU:In prp_enc_enabling_tasks\n");

	err = prp_enc_get_ipu_resources(cam);
	if (err)
		return err;

	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT)
		priv->eof_irq = ipu_idmac_channel_irq(
			cam->ipu, priv->prpenc_rot_out_ch, IPU_IRQ_EOF);
	else
		priv->eof_irq = ipu_idmac_channel_irq(
			cam->ipu, priv->prpenc_ch, IPU_IRQ_EOF);

	err = devm_request_threaded_irq(cam->dev, priv->eof_irq,
					NULL, prp_enc_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-prp-enc", cam);
	if (err) {
		dev_err(cam->dev, "Error registering prp_enc irq\n");
		goto out_put_ipu;
	}

	priv->nfb4eof_irq = ipu_idmac_channel_irq(cam->ipu,
						 priv->prpenc_ch,
						 IPU_IRQ_NFB4EOF);
	err = devm_request_threaded_irq(cam->dev, priv->nfb4eof_irq, NULL,
					prp_enc_nfb4eof_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-enc-nfb4eof", cam);
	if (err) {
		dev_err(cam->dev, "Error registering NFB4EOF irq\n");
		goto out_free_eof_irq;
	}

	err = prp_enc_setup(cam, eba0, eba1);
	if (err) {
		dev_err(cam->dev, "prp_enc_setup %d\n", err);
		goto out_free_nfb4eof_irq;
	}

	return 0;

out_free_nfb4eof_irq:
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);
out_free_eof_irq:
	devm_free_irq(cam->dev, priv->eof_irq, cam);
out_put_ipu:
	prp_enc_put_ipu_resources(cam);
	return err;
}

/*!
 * Disable encoder task
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
static int prp_enc_disabling_tasks(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);

	devm_free_irq(cam->dev, priv->eof_irq, cam);
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);

	/* disable IC tasks and the channels */
	ipu_ic_task_disable(priv->ic_enc);

	ipu_idmac_disable_channel(priv->prpenc_ch);
	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT) {
		ipu_idmac_disable_channel(priv->prpenc_rot_in_ch);
		ipu_idmac_disable_channel(priv->prpenc_rot_out_ch);
	}

	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_unlink_prp_enc_rot_enc(cam->ipu);

	ipu_ic_disable(priv->ic_enc);
	if (cam->rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_irt_disable(priv->irt);

	prp_enc_free_rot_bufs(cam);

	prp_enc_put_ipu_resources(cam);

	return 0;
}

/*!
 * Enable csi
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_enc_enable_csi(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);

	return ipu_csi_enable(priv->csi);
}

/*!
 * Disable csi
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_enc_disable_csi(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv = get_priv(cam);
	int ret;

	/*
	 * Mark next EOF interrupt as the last before streamoff,
	 * and then wait for interrupt handler to mark completion.
	 */
	init_completion(&priv->last_eof_comp);
	priv->last_eof = true;
	ret = wait_for_completion_timeout(&priv->last_eof_comp,
					  msecs_to_jiffies(MXC_EOF_TIMEOUT));
	if (ret == 0)
		dev_warn(cam->dev, "%s: wait last EOF timeout\n", __func__);

	return ipu_csi_disable(priv->csi);
}

/*!
 * function to select PRP-ENC as the working path
 *
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
int prp_enc_select(struct mx6_camera_dev *cam)
{
	cam->enc_update_eba = prp_enc_update_eba;
	cam->enc_enable = prp_enc_enabling_tasks;
	cam->enc_disable = prp_enc_disabling_tasks;
	cam->enc_enable_csi = prp_enc_enable_csi;
	cam->enc_disable_csi = prp_enc_disable_csi;

	return 0;
}

/*!
 * function to de-select PRP-ENC as the working path
 *
 * @param private       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
int prp_enc_deselect(struct mx6_camera_dev *cam)
{
	cam->enc_update_eba = NULL;
	cam->enc_enable = NULL;
	cam->enc_disable = NULL;
	cam->enc_enable_csi = NULL;
	cam->enc_disable_csi = NULL;

	return 0;
}

int prp_enc_init(struct mx6_camera_dev *cam)
{
	struct prp_enc_priv *priv;

	priv = devm_kzalloc(cam->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	cam->prp_enc_priv = priv;

	return 0;
}

void prp_enc_exit(struct mx6_camera_dev *cam)
{
}
